/***************************************************************************************\

Module Name:    MtlList.h

Description:    None.

References:     None.

    Copyright (c) 2002, Matrox Graphics Inc.
    All Rights Reserved.

\***************************************************************************************/

#ifndef INC_MTLLIST_H
#define INC_MTLLIST_H

//#define TESTWITHSTD     // This is used to test the List with the standard algorithm.
// --------------------------------------------------------------------------------------
//                        I N C L U D E S   A N D   U S I N G S
// --------------------------------------------------------------------------------------
#ifdef TESTWITHSTD
#include <xutility>
#endif
#include "MtlXMemory.h"
// --------------------------------------------------------------------------------------
//                        C O N S T A N T S   A N D   T Y P E S
// --------------------------------------------------------------------------------------

// Create the namespace for the Matrox Template Library
// Defines are required to avoid tabulation reformating by VC editor.
#define MTLBEGIN        namespace Mtl{
#define MTLEND          }

#ifdef TESTWITHSTD
#define BOOL            bool
#define TRUE            true
#define FALSE           false

#define ValueType       value_type     
#define Pointer         pointer        
#define ConstPointer    const_pointer  
#define Reference       reference      
#define ConstReference  const_reference
#define SizeType        size_type      
#define DifferenceType  difference_type

#define Iterator        iterator
#define ConstIterator   const_iterator
#define ReverseIterator reverse_iterator
#define ConstReverseIterator   const_reverse_iterator

#define Back            back
#define Begin           begin
#define Clear           clear
#define Empty           empty
#define End             end
#define Erase           erase
#define Front           front
#define Insert          insert
#define PushBack        push_back
#define PushFront       push_front
#define PopBack         pop_back
#define PopFront        pop_front
#define RBegin          rbegin
#define REnd            rend
#define Remove          remove
#define Reverse         reverse
#define Size            size
#define Splice          splice

#endif

// --------------------------------------------------------------------------------------
//                 G L O B A L   V A R I A B L E   R E F E R E N C E S
// --------------------------------------------------------------------------------------

// --------------------------------------------------------------------------------------
//               G L O B A L   F U N C T I O N   D E C L A R A T I O N S
// --------------------------------------------------------------------------------------

// --------------------------------------------------------------------------------------
//     I N L I N E S,  M A C R O S,  A N D   T E M P L A T E   D E F I N I T I O N S
// --------------------------------------------------------------------------------------
MTLBEGIN

// TEMPLATE CLASS ReverseBidirectionalIterator
/***************************************************************************************\

Class:          ReverseBidirectionalIterator


Description:    None.

Comments:       None.

\***************************************************************************************/
template<class IteratorType,
         class ValueType,
         class ReferenceType = ValueType&,
         class PointerType = ValueType *,
         class DifferenceType = unsigned long>
class ReverseBidirectionalIterator
#ifdef TESTWITHSTD
: public std::_Bidit<ValueType, DifferenceType> 
#endif
{
public:
    typedef ReverseBidirectionalIterator<IteratorType,
                                         ValueType, 
                                         ReferenceType, 
                                         PointerType, 
                                         DifferenceType>    ReverseIteratorType;
    ReverseBidirectionalIterator() {}
    explicit ReverseBidirectionalIterator(IteratorType itX):itCurrent(itX) {}
    IteratorType Base() const {return (itCurrent); }
    ReferenceType operator*() const {IteratorType itTemp = itCurrent; return (*--itTemp); }
    ReverseIteratorType& operator++() {--itCurrent; return (*this); }
    ReverseIteratorType operator++(int) {ReverseIteratorType itTemp = *this; --itCurrent; return (itTemp); }
    ReverseIteratorType& operator--() {++itCurrent; return (*this); }
    ReverseIteratorType operator--(int) {ReverseIteratorType itTemp = *this; ++itCurrent; return (itTemp); }
    BOOL operator==(const ReverseIteratorType& itX) const {return(itCurrent == itX.Base()); }
    BOOL operator!=(const ReverseIteratorType& itX) const {return(itCurrent != itX.Base()); }
protected:
    IteratorType itCurrent;
};

/***************************************************************************************\

Class:          List

Description:    None.

Comments:       None.

\***************************************************************************************/
template<class ValueType, class AllocatorType = allocator<ValueType> >
class List
{
    public:
        // ----- Class Types -----
        typedef typename AllocatorType::pointer          Pointer;
        typedef typename AllocatorType::const_pointer    ConstPointer;
        typedef typename AllocatorType::reference        Reference;
        typedef typename AllocatorType::const_reference  ConstReference;
        typedef typename AllocatorType::size_type        SizeType;
        typedef typename AllocatorType::difference_type  DifferenceType;

    protected:
        // ----- Class Types -----
        struct Node;
        struct BaseNode;
        friend struct BaseNode;
        struct BaseNode
        {
            Node*       m_poNext;
            Node*       m_poPrev;
        };  // struct BaseNode
        
        friend struct Node;
        struct Node:public BaseNode
        {
            ValueType   m_oValue;
        };  // struct Node
        
        struct Acc;
        friend struct Acc;
        struct Acc
        {
            //static Node*&       Next(Node* poNode) { return ((Node*&)poNode->m_poNext); }
            //static Node*&       Prev(Node* poNode) { return ((Node*&)poNode->m_poPrev); }
            static Reference    Value(Node* poNode) { return ((Reference)poNode->m_oValue); }
        };
        
    public:
        // ----- Class Types -----
        // Class ConstIterator
        class Iterator;
        class ConstIterator;
        friend class ConstIterator;
        class ConstIterator 
#ifdef TESTWITHSTD
            : public std::_Bidit<ValueType, DifferenceType> 
#endif
        {
        public:
            ConstIterator() {}
            ConstIterator(Node* poNode):m_poNode(poNode) {}
            ConstIterator(const Iterator& itNode):m_poNode(itNode.m_poNode) {}
            ConstReference operator*() const {return (Acc::Value(m_poNode)); }
            ConstIterator& operator++() {m_poNode = m_poNode->m_poNext; return (*this); }
            ConstIterator operator++(int) { ConstIterator itTmp = *this; ++*this; return (itTmp); }
            ConstIterator& operator--() {m_poNode = m_poNode->m_poPrev; return (*this); }
            ConstIterator operator--(int) { ConstIterator itTmp = *this; --*this; return (itTmp); }
            BOOL operator==(const ConstIterator& itNode) const {return(m_poNode == itNode.m_poNode); }
            BOOL operator!=(const ConstIterator& itNode) const {return(!(*this == itNode)); }
            Node*   MyNode() const { return (m_poNode); }
        protected:
            Node*   m_poNode;
        };
        
        // Class Iterator
        friend class Iterator;
        class Iterator:public ConstIterator
        {
        public:
            Iterator() {}
            Iterator(Node* poNode):ConstIterator(poNode) {}
            Reference operator*() const { return (Acc::Value(m_poNode)); }
            Iterator& operator++() {m_poNode = m_poNode->m_poNext; return (*this); }        // Prefix
            Iterator operator++(int) { Iterator itTmp = *this; ++*this; return (itTmp); }   // Postfix
            Iterator& operator--() {m_poNode = m_poNode->m_poPrev; return (*this); }
            Iterator operator--(int) { Iterator itTmp = *this; --*this; return (itTmp); }
            BOOL operator==(const Iterator& itNode) const {return(m_poNode == itNode.m_poNode); }
            BOOL operator!=(const Iterator& itNode) const {return(!(*this == itNode)); }
        };

        typedef ReverseBidirectionalIterator<Iterator, 
                                             ValueType, 
                                             Reference, 
                                             Pointer, 
                                             DifferenceType>    ReverseIterator;
        typedef ReverseBidirectionalIterator<ConstIterator, 
                                             ValueType, 
                                             ConstReference, 
                                             ConstPointer, 
                                             DifferenceType>    ConstReverseIterator;
        
        // ----- Attributes -----
        // ----- Methods -----
        // Creates a list of zero elements.
        explicit List(const AllocatorType& oAllocator = AllocatorType()):m_oAllocator(oAllocator), m_ulSize(0) 
        {   m_poHead = reinterpret_cast<Node*>(&m_oHead); 
            m_poHead->m_poNext = m_poHead; 
            m_poHead->m_poPrev = m_poHead;}
        // The destructor. Releases any allocated memory for this list.
        // No need to free the node m_poHead, its allocated in the class itself.
        ~List() { Erase(Begin(), End()); m_poHead = 0; m_ulSize = 0; }
        
        // Returns a bidirectional iterator that points to the first element.
        Iterator Begin() { return(Iterator(m_poHead->m_poNext)); }
        // Returns a constant bidirectional iterator that points to the first element.
        ConstIterator Begin() const {return (ConstIterator(m_poHead->m_poNext)); }
        // Returns a bidirectional iterator that points to the past-the-end value.
        Iterator End() { return(Iterator(m_poHead)); }
        // Returns a constant bidirectional iterator that points to the past-the-end value.
        ConstIterator End() const {return (ConstIterator(m_poHead)); }

        // Returns a bidirectional iterator that points to the past-the-end value.
        ReverseIterator RBegin() {return (ReverseIterator(End())); }
        // Returns a constant bidirectional iterator that points to the past-the-end value.
        ConstReverseIterator RBegin() const {return (ConstReverseIterator(End())); }
        // Returns a bidirectional iterator that points to the first element.
        ReverseIterator REnd() {return (ReverseIterator(Begin())); }
        // Returns a constant bidirectional iterator that points to the first element.
        ConstReverseIterator REnd() const {return (ConstReverseIterator(Begin())); }
        
        // Returns the number of elements.
        SizeType Size() const {return (m_ulSize); }
        // Returns true if the size is zero.
        BOOL Empty() const {return (Size() == 0); }
        // Returns a reference to the first element.
        Reference Front() {return (*Begin()); }
        // Returns a constant reference to the first element.
        ConstReference Front() const {return (*Begin()); }
        // Returns a reference to the last element.        
        Reference Back() {return (*(--End())); }
        // Returns a constant reference to the last element.
        ConstReference Back() const {return (*(--End())); }
        // Adds an element to the beginning of a list.
        BOOL PushFront(const ValueType& oValue) {return (Insert(Begin(), oValue)); }
        // Removes the first element.
        void PopFront() {Erase(Begin()); }
        // Adds an element to the end of a list.
        BOOL PushBack(const ValueType& oValue) {return (Insert(End(), oValue));}
        // Removes the last element.
        void PopBack() {Erase(--End()); }
        
        inline BOOL Insert(Iterator itNode, const ValueType& oValue = ValueType());
        inline BOOL Insert(Iterator itNode, SizeType ulSize, const ValueType& oValue);
        inline BOOL Insert(Iterator itNode, const ValueType* poValueF, const ValueType* poValueL);
        inline BOOL Insert(Iterator itNode, ConstIterator itFrom, ConstIterator itUpTo);
        inline Iterator Erase(Iterator itNode);
        inline Iterator Erase(Iterator itFirst, Iterator itLast);
        // Erases all elements from the list.
        void Clear(){Erase(Begin(), End()); }
        
        inline void Splice(Iterator itInsertHere, List<ValueType, AllocatorType>& lstListToInsert);
        inline void Splice(Iterator itInsertHere, List<ValueType, AllocatorType>& lstListToInsert, Iterator itFrom);
        inline void Splice(Iterator itInsertHere, List<ValueType, AllocatorType>& lstListToInsert, Iterator itFrom, Iterator itUpTo);
        inline BOOL Remove(const ValueType& oValue);
            
        inline void Reverse();
        
    protected:
        // ----- Class Types -----
        // ----- Attributes -----
        AllocatorType m_oAllocator;
        
        Node*       m_poHead;   // Holds onto the List: this is initialize with 
                                // "BaseNode    m_oHead;" when the list is constructed.
                                // ValueType is not allocated nor initialize.
        BaseNode    m_oHead;    // Holds onto the List: Does not contain ValueType to avoid its allocation in the class.
        SizeType    m_ulSize;   // Keep number of elements in the list.
        // ----- Methods -----
        inline Node* BuyNode(Node* poNext = 0, Node* poPrev = 0);
        inline void  FreeNode(Node* poNode);
        
        inline void  _Splice(Iterator itInsertHere, Iterator itTakingFrom, Iterator itTakingUpTo);
    private:
        // ----- Class Types -----
        // ----- Attributes -----
        // ----- Methods -----
        
};  // class List

/***************************************************************************************\

Function:       List<ValueType>::Insert

Description:    Inserts an element into the list before the specified position.

Parameters:     itNode  The position in the target list where the first element is inserted.
                oValue  The value of the element being inserted into the list.

Return Value:   BOOL    Return TRUE if allocation/insertion succed, FALSE otherwise.

Comments:       None.

\***************************************************************************************/
template <class ValueType, class AllocatorType>
BOOL List<ValueType, AllocatorType>::Insert(Iterator itNode, const ValueType& oValue)
{
    Node* poNode = itNode.MyNode();
    Node* poNewNode = BuyNode(poNode, poNode->m_poPrev);
    if (poNewNode == NULL)
        return FALSE;
    poNode->m_poPrev = poNewNode;
    poNode = poNode->m_poPrev; 
    poNode->m_poPrev->m_poNext = poNode;
    // Call the ValueType constructor here.
    //new ((void *)(&Acc::Value(poNode))) ValueType(oValue);
    m_oAllocator.construct(&Acc::Value(poNode), oValue);
    ++m_ulSize;
    return TRUE;
}

/***************************************************************************************\

Function:       List<ValueType>::Insert

Description:    Inserts ulSize copies of the element into the list before the specified position.

Parameters:     itNode  The position in the target list where the first element is inserted.
                ulSize  The number of time the element is being inserted into the list.
                oValue  The value of the element being inserted into the list.

Return Value:   BOOL    Return TRUE if allocation/insertion succed, FALSE otherwise.

Comments:       None.

\***************************************************************************************/
template <class ValueType, class AllocatorType>
BOOL List<ValueType, AllocatorType>::Insert(Iterator itNode, SizeType ulSize, const ValueType& oValue)
{
    for(; 0 < ulSize; --ulSize)
    {
        if (Insert(itNode, oValue) == FALSE)
            return FALSE;
    }
    return TRUE;
}

/***************************************************************************************\

Function:       List<ValueType>::Insert

Description:    Inserts elements in the range [poValueFirst, poValueLast) into the list 
                before the specified position.

Parameters:     itNode      The position in the target list where the first element is inserted.
                poValueF    The first element pointer in the range of elements to be copied.
                poValueL    The first element pointer beyond the range of elements to be copied.

Return Value:   BOOL        Return TRUE if allocation/insertion succed, FALSE otherwise.

Comments:       The pointer to the ValueType need to support preincrementation.

\***************************************************************************************/
template <class ValueType, class AllocatorType>
BOOL List<ValueType, AllocatorType>::Insert(Iterator itNode, const ValueType* poValueF, const ValueType* poValueL)
{
    for (; poValueF != poValueL; ++poValueF) 
    {
        if (Insert(itNode, *poValueF) == FALSE)
            return FALSE;
    }
    return TRUE;
}

/***************************************************************************************\

Function:       List<ValueType>::Insert

Description:    Inserts copies of the element in the range [itFrom, itUpTo) into the list 
                before the specified position.

Parameters:     itNode  The position in the target list where the first element is inserted.
                itFrom  The position of the first element in the range of elements in the 
                        argument list to be copied. 
                itUpTo  The position of the first element beyond the range of elements in 
                        the argument list to be copied.

Return Value:   BOOL    Return TRUE if allocation/insertion succed, FALSE otherwise.

Comments:       None.

\***************************************************************************************/
template <class ValueType, class AllocatorType>
BOOL List<ValueType, AllocatorType>::Insert(Iterator itNode, ConstIterator itFrom, ConstIterator itUpTo)
{
    for (; itFrom != itUpTo; ++itFrom) 
    {
        if (Insert(itNode, *itFrom) == FALSE)
            return FALSE;
    }
    return TRUE;
}

/***************************************************************************************\

Function:       List<ValueType>::Erase

Description:    Removes the element pointed to by itNode. Returns an iterator pointing to 
                the element following the deleted element, or End() if the deleted item 
                was the last one in this list.

Parameters:     itNode          Position of the element to be removed from the list.

Return Value:   List<ValueType>::Iterator.

Comments:       None.

\***************************************************************************************/
template <class ValueType, class AllocatorType>
List<ValueType, AllocatorType>::Iterator List<ValueType, AllocatorType>::Erase(Iterator itNode)
{
    Node* poNode = (itNode++).MyNode();
    poNode->m_poPrev->m_poNext = poNode->m_poNext;
    poNode->m_poNext->m_poPrev = poNode->m_poPrev;
    // Call the ValueType destructor here.
    //(&Acc::Value(poNode))->~ValueType();
    m_oAllocator.destroy(&Acc::Value(poNode));
    FreeNode(poNode);
    --m_ulSize;
    return (itNode);
}
       
/***************************************************************************************\

Function:       List<ValueType>::Erase

Description:    Removes the elements in the range (itFirst, itLast). Returns an iterator 
                pointing to the element following the element following the last deleted 
                element, or End() if there were no elements after the deleted range.

Parameters:     itFirst     Position of the first element removed from the list. 
                itLast      Position just beyond the last element removed from the list.

Return Value:   List<ValueType>::Iterator.

Comments:       None.

\***************************************************************************************/
template <class ValueType, class AllocatorType>
List<ValueType, AllocatorType>::Iterator List<ValueType, AllocatorType>::Erase(Iterator itFirst, Iterator itLast) 
{
    while(itFirst != itLast) 
        Erase(itFirst++); 
    return(itFirst); 
}

/***************************************************************************************\

Function:       List<ValueType>::Splice

Description:    Inserts all elements in the argument list before the element located at 
                itInsertHere in the target list. It also removes all elements from the 
                argument list.

Parameters:     itInsertHere    The position in the target list before which the elements 
                                of the argument list are to be inserted. 
                lstListToInsert The argument list that is to be inserted into the target 
                                list.

Return Value:   None.

Comments:       None.

\***************************************************************************************/
template <class ValueType, class AllocatorType>
void List<ValueType, AllocatorType>::Splice(Iterator itInsertHere, List<ValueType, AllocatorType>& lstListToInsert)
{
    if (!lstListToInsert.Empty())
    {
        _Splice(itInsertHere, lstListToInsert.Begin(), lstListToInsert.End());
        m_ulSize += lstListToInsert.m_ulSize;
        lstListToInsert.m_ulSize = 0; 
    }
}

/***************************************************************************************\

Function:       List<ValueType>::Splice

Description:    Removes the element pointed to by itFrom in the argument list and inserts
                it before the element in the target list pointed to by itInsertHere.

Parameters:     itInsertHere    The position in the target list before which the elements 
                                of the argument list are to be inserted. 
                lstListToInsert The argument list that is to be inserted into the target 
                                list.
                itTakingFrom    The first element in the range to be inserted from the 
                                argument list.

Return Value:   None.

Comments:       None.

\***************************************************************************************/
template <class ValueType, class AllocatorType>
void List<ValueType, AllocatorType>::Splice(Iterator itInsertHere, List<ValueType, AllocatorType>& lstListToInsert, Iterator itFrom)
{
    Iterator itUpTo = itFrom;
    if (itInsertHere != itFrom && itInsertHere != ++itUpTo)
    {
        _Splice(itInsertHere, itFrom, itUpTo);
        ++m_ulSize;
        --lstListToInsert.m_ulSize;     
    }
}

/***************************************************************************************\

Function:       List<ValueType>::Splice

Description:    Inserts the range designated by [itTakingFrom, itTakingUpTo) from the 
                argument list before the element in the target list pointed to by 
                itInsertHere. It also removes the range inserted from the argument list.

Parameters:     itInsertHere    The position in the target list before which the elements 
                                of the argument list are to be inserted. 
                lstListToInsert The argument list that is to be inserted into the target 
                                list.
                itTakingFrom    The first element in the range to be inserted from the 
                                argument list.
                itTakingUpTo    The first element beyond the range to be inserted from 
                                the argument list.

Return Value:   None.

Comments:       None.

\***************************************************************************************/
template <class ValueType, class AllocatorType>
void List<ValueType, AllocatorType>::Splice(Iterator itInsertHere, List<ValueType, AllocatorType>& lstListToInsert, Iterator itFrom, Iterator itUpTo)
{
    if (itFrom != itUpTo)
    {
        if (&lstListToInsert != this)
        {
            // Adjuste the new size of each list.
            DifferenceType ulDiff = 0;
            Iterator itCounter = itFrom;
            for (; itCounter != itUpTo; ++itCounter)  // Distance
                ++ulDiff;
            m_ulSize += ulDiff;
            lstListToInsert.m_ulSize -= ulDiff; 
        }
    _Splice(itInsertHere, itFrom, itUpTo); 
    }
}

/***************************************************************************************\

Function:       List<ValueType>::Remove

Description:    Erases elements in a list that match a specified value.

Parameters:     oValue  The value which, if held by an element, will result in that 
                        element's removal from the list.

Return Value:   None.

Comments:       None.

\***************************************************************************************/
template <class ValueType, class AllocatorType>
BOOL List<ValueType, AllocatorType>::Remove(const ValueType& oValue)
{
    BOOL bFlag = FALSE; 
    Iterator itEnd = End();
    for (Iterator itPosition = Begin(); itPosition != itEnd; )
    {
        if (*itPosition == oValue)
        {
            Erase(itPosition++);
            bFlag = TRUE; 
        }
        else
            ++itPosition; 
    }
    return bFlag; 
}

/***************************************************************************************\

Function:       List<ValueType>::Reverse

Description:    Reverses the order in which the elements occur in a list.

Parameters:     

Return Value:   None.

Comments:       None.

\***************************************************************************************/
template <class ValueType, class AllocatorType>
void List<ValueType, AllocatorType>::Reverse()
{
    if (2 <= Size())
    {
        Iterator itEnd = End();
        for (Iterator itPosition = ++Begin(); itPosition != itEnd; )
		{
            Iterator itFrom = itPosition;
			_Splice(Begin(), itFrom, ++itPosition); 
        }
    }
}

/***************************************************************************************\

Function:       List<ValueType>::BuyNode

Description:    Allocate a Node.

Parameters:     poNext
                poPrev

Return Value:   List<ValueType>::Node*.

Comments:       Note that we allocate the size of the Node and not the Node itself. This
                avoid the call to the ValueType constructor(Done in Insert).

\***************************************************************************************/
template <class ValueType, class AllocatorType>
List<ValueType, AllocatorType>::Node* List<ValueType, AllocatorType>::BuyNode(Node* poNext, Node* poPrev)
{
    Node* poNewNode = (Node*)m_oAllocator._Charalloc(1 * sizeof (Node)); //(Node*)operator new (sizeof (Node));
    if (poNewNode != NULL)
    {
        poNewNode->m_poNext = poNext != 0 ? poNext : poNewNode;
        poNewNode->m_poPrev = poPrev != 0 ? poPrev : poNewNode;
    }
    return (poNewNode);
}

/***************************************************************************************\

Function:       List<ValueType>::FreeNode

Description:    Delete space allocated for a Node without calling the ValueType Destructor.

Parameters:     poNode

Return Value:   None.

Comments:       The destructor of ValueType must be call before calling FreeNode(Done in Erase).

\***************************************************************************************/
template <class ValueType, class AllocatorType>
void List<ValueType, AllocatorType>::FreeNode(Node* poNode) 
{ 
    m_oAllocator.deallocate(poNode, 1); //operator delete(poNode); 
}

/***************************************************************************************\

Function:       List<ValueType>::_Splice

Description:    Inserts the range designated by [itTakingFrom, itTakingUpTo) from the 
                argument list before the element in the target list pointed to by 
                itInsertHere. It also removes the range inserted from the argument list.

Parameters:     itInsertHere    The position in the target list before which the elements 
                                of the argument list are to be inserted. 
                itTakingFrom    The first element in the range to be inserted from the 
                                argument list.
                itTakingUpTo    The first element beyond the range to be inserted from 
                                the argument list.

Return Value:   None.

Comments:       Generic function for internal use.

\***************************************************************************************/
template <class ValueType, class AllocatorType>
void List<ValueType, AllocatorType>::_Splice(Iterator itInsertHere, Iterator itTakingFrom, Iterator itTakingUpTo)
{
    itTakingUpTo.MyNode()->m_poPrev->m_poNext = itInsertHere.MyNode();
    itTakingFrom.MyNode()->m_poPrev->m_poNext = itTakingUpTo.MyNode();
    itInsertHere.MyNode()->m_poPrev->m_poNext = itTakingFrom.MyNode();
    Node* poNode = itInsertHere.MyNode()->m_poPrev;
    itInsertHere.MyNode()->m_poPrev = itTakingUpTo.MyNode()->m_poPrev;
    itTakingUpTo.MyNode()->m_poPrev = itTakingFrom.MyNode()->m_poPrev;
    itTakingFrom.MyNode()->m_poPrev = poNode;
}

#ifdef TESTWITHSTD

#undef ValueType       
#undef Pointer         
#undef ConstPointer    
#undef Reference       
#undef ConstReference  
#undef SizeType        
#undef DifferenceType  

#undef Iterator     
#undef ConstIterator
#undef ReverseIterator
#undef ConstReverseIterator

#undef Back     
#undef Begin  
#undef Clear  
#undef Empty    
#undef End      
#undef Erase    
#undef Front    
#undef Insert   
#undef PushBack 
#undef PushFront
#undef PopBack  
#undef PopFront 
#undef RBegin
#undef REnd
#undef Remove   
#undef Reverse  
#undef Size     
#undef Splice  

#endif

MTLEND
#endif  // #ifndef INC_MTLLIST_H
