/***************************************************************************************\
    
Module Name:    parhl_regs.h     

Description:    definitions of registers on a Matrox board.

Author:         Karl Lessard, 25-07-01
    
\***************************************************************************************/

#ifndef _PARHL_REGS_H_
#define _PARHL_REGS_H_

/* return register tag used by bus mastering */
#define DMA_TAG(reg)                                                        \
    (unsigned char)((reg - 0x1000) >> 2)

/* creating a command header */
#define DMA_CMD(reg1, reg2, reg3, reg4)                                     \
    ( (DMA_TAG(reg1)        & 0x000000FF) |                                 \
     ((DMA_TAG(reg2) << 8)  & 0x0000FF00) |                                 \
     ((DMA_TAG(reg3) << 16) & 0x00FF0000) |                                 \
     ((DMA_TAG(reg4) << 24) & 0xFF000000) )                 


/*****************************
 *
 * GENERAL PURPOSE REGISTERS 
 *
 *****************************/

/* Configuration registers */
#define CFG_DEVID                               0x00
#define CFG_DEVCTL                              0x04
#define CFG_CLASS_VGA                           0x08
#define CFG_HEADER                              0x0C
#define CFG_MGABASE0                            0x10
#define CFG_MGABASE1                            0x14
#define CFG_AGP_CMD                             0xF8
#define CFG_AGP_STS                             0xF4
    #define CFG_AGP_STS_RATE_CAP_MASK               0x07
 
/* HOST Interface registers */
#define HST_SURFPCI_OFFSET                      0x0000
    #define HST_SURFPCI_ENABLE                      (1 << 28)
#define HST_SURFADDR_OFFSET                     0x0004
#define HST_SURFMODE_OFFSET                     0x0008
    #define HST_SURFMODE_SRFMODE_XYLINEAR           (1 << 16)
    #define HST_SURFMODE_SRFMODE_TILEDBLK           (2 << 16)
    #define HST_SURFMODE_SRFMODE_TILEDLIN           (3 << 16)
    #define HST_SURFMODE_SRFMODE_TEXTILEDBLK        (4 << 16)
    #define HST_SURFMODE_SRFMODE_DXT                (5 << 16)
    #define HST_SURFMODE_SRFFORMAT_MONO             (0x00 << 24)
    #define HST_SURFMODE_SRFFORMAT_MONOWIN          (0x02 << 24)
    #define HST_SURFMODE_SRFFORMAT_CURSORALPHA      (0x04 << 24)
    #define HST_SURFMODE_SRFFORMAT_CURSORMONO       (0x05 << 24)
    #define HST_SURFMODE_SRFFORMAT_DXT1             (0x08 << 24)
    #define HST_SURFMODE_SRFFORMAT_DXT2             (0x09 << 24)
    #define HST_SURFMODE_SRFFORMAT_DXT3             (0x0A << 24)
    #define HST_SURFMODE_SRFFORMAT_DXT4             (0x0B << 24)
    #define HST_SURFMODE_SRFFORMAT_DXT5             (0x0C << 24)
    #define HST_SURFMODE_SRFFORMAT_W8               (0x10 << 24)
    #define HST_SURFMODE_SRFFORMAT_L8               (0x11 << 24)
    #define HST_SURFMODE_SRFFORMAT_A8               (0x12 << 24)
    #define HST_SURFMODE_SRFFORMAT_YV12             (0x14 << 24)
    #define HST_SURFMODE_SRFFORMAT_W16              (0x20 << 24)
    #define HST_SURFMODE_SRFFORMAT_L10              (0x21 << 24)
    #define HST_SURFMODE_SRFFORMAT_A8L8             (0x22 << 24)
    #define HST_SURFMODE_SRFFORMAT_A4R4G4B4         (0x24 << 24)
    #define HST_SURFMODE_SRFFORMAT_X4R4G4B4         (0x25 << 24)
    #define HST_SURFMODE_SRFFORMAT_A1R5G5B5         (0x26 << 24)
    #define HST_SURFMODE_SRFFORMAT_R5G5B5           (0x27 << 24)
    #define HST_SURFMODE_SRFFORMAT_R5G6B5           (0x28 << 24)
    #define HST_SURFMODE_SRFFORMAT_Z16S0            (0x2A << 24)
    #define HST_SURFMODE_SRFFORMAT_S1Z15            (0x2B << 24)
    #define HST_SURFMODE_SRFFORMAT_U8V8             (0x2C << 24)
    #define HST_SURFMODE_SRFFORMAT_U5V5L6           (0x2D << 24)
    #define HST_SURFMODE_SRFFORMAT_UYVY             (0x2E << 24)
    #define HST_SURFMODE_SRFFORMAT_W32              (0x30 << 24)
    #define HST_SURFMODE_SRFFORMAT_A8R8G8B8         (0x32 << 24)
    #define HST_SURFMODE_SRFFORMAT_X8R8G8B8         (0x33 << 24)
    #define HST_SURFMODE_SRFFORMAT_A2R10G10B10      (0x34 << 24)
    #define HST_SURFMODE_SRFFORMAT_S8Z24            (0x36 << 24)
    #define HST_SURFMODE_SRFFORMAT_U8V8L8X8         (0x38 << 24)
    #define HST_SURFMODE_SRFFORMAT_L1616            (0x3A << 24)
    #define HST_SURFMODE_SRFFORMAT_AYUV8888         (0x3B << 24)
#define HST_SURFDIM_OFFSET                      0x000C
    #define HST_SURFDIM_DEPTH_SHIFT                 16

#define HST_SURF_OFFSET                         0x0010

#define HST_SURFPCI(idx)                        \
    (HST_SURF_OFFSET * (idx) + HST_SURFPCI_OFFSET)
#define HST_SURFADDR(idx)                       \
    (HST_SURF_OFFSET * (idx) + HST_SURFADDR_OFFSET)
#define HST_SURFMODE(idx)                       \
    (HST_SURF_OFFSET * (idx) + HST_SURFMODE_OFFSET)
#define HST_SURFDIM(idx)                        \
    (HST_SURF_OFFSET * (idx) + HST_SURFDIM_OFFSET)

#define HST_RESET                               0x0300
#define HST_CONFIG                              0x0304
#define HST_STATUS                              0x0308
    #define HST_STATUS_DAFIFOSTATE_MASK             0x7
    #define HST_STATUS_FWDATAEMPTYH_MASK            0x10
    #define HST_STATUS_FWDATAFULLH_MASK             0x20
    #define HST_STATUS_FWEMPTYH_MASK                0x40
    #define HST_STATUS_FWFULLH_MASK                 0x80
    #define HST_STATUS_SDINVALID_ALL_MASK           0x1000
    #define HST_STATUS_SWFLAG_MASK                  0xf0000000
#define HST_ICLEAR                              0x030C
#define HST_IEN                                 0x0310

#define MEMORYTYPE_LOCAL                        0x0
#define MEMORYTYPE_AGP                          0x2
#define MEMORYTYPE_PCI                          0x3
#define MEMORYTYPE_MASK                         0x3
    
/* GEN registers */
#define GEN_STATUS                              0x08F0
    #define GEN_STATUS_SOFTRAPEN                    (1 << 16)                              
    #define GEN_STATUS_ENDPRDMASTS                  (1 << 17)                              
#define GEN_ISTATUS                             0x08F4
#define GEN_ICLEAR                              0x08F8
#define GEN_IEN                                 0x08FC


/* DMA registers */
#define DMA_PRIMPTR                             0x0400
#define DMA_PRIMADDR                            0x0404
    #define DMA_PRIMADDR_PRIMADDRESS_MASK           0xfffffffc
#define DMA_PRIMEND                             0x0408
    #define DMA_PRIMEND_PRIMSRCSEL_LOCAL            (0 << 0)
    #define DMA_PRIMEND_PRIMSRCSEL_AGP              (2 << 0)
    #define DMA_PRIMEND_PRIMSRCSEL_PCI              (3 << 0)
    #define DMA_PRIMEND_PRIMSRCSEL_MASK             0x00000003
    #define DMA_PRIMEND_PRIMEND_MASK                0xfffffffc
#define DMA_SECADDR_RD                          0x040C
#define DMA_SECEND_RD                           0x0410
#define DMA_SETUPADDR_RD                        0x0414
#define DMA_SETUPEND_RD                         0x0418
#define DMA_DWGMARK_RD                          0x041C
#define DMA_STATSINDEX                          0x0420
#define DMA_STATSDATA                           0x0424

#define DMA_STATUS_EMPTY                        0x0450
    #define DMA_STATUS_EMPTY_RSFEMPTY               (1 << 22)
#define DMA_STATUS_FULL                         0x0454

#define DMA_BASE0                               0x1000
#define DMA_BASE1                               0x1004
#define DMA_BASE2                               0x1008
#define DMA_BASE3                               0x100C
#define DMA_BASE4                               0x1010
#define DMA_BASE5                               0x1014
#define DMA_BASE6                               0x1018
#define DMA_BASE7                               0x101C

#define DMA_STRIDE01                            0x1040
#define DMA_STRIDE23                            0x1044
#define DMA_STRIDE45                            0x1048
#define DMA_STRIDE67                            0x104C

#define DMA_PAD                                 0x1080
#define DMA_STOP                                0x1084
#define DMA_MAP                                 0x1088
#define DMA_COUNT                               0x108C
#define DMA_WRITE                               0x1090
    #define DMA_WRITE_DMAWADD_MASK                  0xffffffe0
#define DMA_SYNC                                0x1094
#define DMA_SOFTRAP                             0x1098
#define DMA_PRIMJUMP                            0x109C
    #define DMA_PRIMJUMP_PJUMPADD_MASK              0xfffffffc
#define DMA_SECADDR                             0x10A0
    #define DMA_SECADDR_SECMOD_GEN_PURPOSE          (0 << 0)
    #define DMA_SECADDR_SECMOD_SHADER               (2 << 0)      
    #define DMA_SECADDR_SECMOD_MASK                 0x00000003
    #define DMA_SECADDR_SECADDRESS_MASK             0xfffffffc
#define DMA_SECEND                              0x10A4
    #define DMA_SECEND_SECSRCSEL_LOCAL              (0 << 0)
    #define DMA_SECEND_SECSRCSEL_AGP                (2 << 0)
    #define DMA_SECEND_SECSRCSEL_PCI                (3 << 0)
    #define DMA_SECEND_SECSRCSEL_MASK               0x00000003
    #define DMA_SECEND_SECEND_MASK                  0xfffffffc
#define DMA_SETUPADDR                           0x10A8
#define DMA_SETUPEND                            0x10AC
#define DMA_CTL                                 0x10B0
#define DMA_SEQINDEX                            0x10B4
#define DMA_SEQDATA                             0x10B8
#define DMA_DWGMARK                             0x12BC
    #define DMA_DWGMARK_WRTEN_ON                    (1 << 0)
    #define DMA_DWGMARK_INTEN_ON                    (1 << 1)
    #define DMA_DWGMARK_SYNC_CACHEFLUSH             (0 << 2)
    #define DMA_DWGMARK_SYNC_SRCDATAREAD            (1 << 2)
    #define DMA_DWGMARK_TAG_MASK                    0xfffffff8

#define DMA_INDEXMAX                            0x1060
#define DMA_INDEXMIN	                        0x1064

/* Graphic Registers */
#define DEPTH_INIT                              0x12B4
#define DEPTH_MIN                               0x12C0
#define DEPTH_MAX                               0x12C4
#define DEPTH_TESTS_LO                          0x12C8
#define DEPTH_TESTS_HI                          0x12CC
#define DEPTH_WRITE_MASKS                       0x12D0
#define ALPHA_TEST                              0x12E4
#define RTB_CONFIG                              0x12E8
#define RTB_AR_BLEND_FACTOR                     0x12EC
#define RTB_GB_BLEND_FACTOR                     0x12F0

/* Vertex Engine Registers */
#define VE_INTVEC0                              0x10C0
#define VE_INTVEC1                              0x10C4
#define VE_INTVEC2                              0x10C8
#define VE_INTVEC3                              0x10CC
#define VE_INTVEC4                              0x10D0
#define VE_INTVEC5                              0x10D4
#define VE_INTVEC6                              0x10D8
#define VE_INTVEC7                              0x10DC

#define VE_CONTROL                              0x10F0

#define VE_VECTOR0                              0x1100
#define VE_VECTOR1                              0x1104
#define VE_BREAK                                0x1108
#define VE_RESUME                               0x110C

#define VE_VECTOR0_NB                           0x0930
#define VE_VECTOR1_NB                           0x0934
#define VE_BREAK_NB                             0x0938
#define VE_RESUME_NB                            0x093C


#define VE_EXCVEC                               0x10F8

#define VE_PC                                   0x0904
#define VE_STATUS                               0x0908
#define VE_COUNT                                0x090C
#define VE_DBG_ADDR                             0x0910
#define VE_DBG_DATA                             0x0914
#define VE_IMEM_ADDR                            0x0918
#define VE_IMEM_DATA                            0x091C
#define VE_MISC                                 0x0920

/* Culling Engine Registers */
#define CE_FVF                                  0x1110
#define CE_FVFSOURCE0                           0x1114
#define CE_FVFSOURCE1                           0x1118
#define CE_CLIP                                 0x111C
#define CE_GBX                                  0x1120
#define CE_GBY                                  0x1124
#define CE_GBZN                                 0x1128
#define CE_GBZF                                 0x112C
#define CE_CONTROL                              0x1130
#define CE_CONTROL_LOW                          0x1134
#define CE_PRIM                                 0x1138

/* SE Direct Vertex programming */
#define VERTEX3_DEST0                           0x1160
#define VERTEX3_DEST1                           0x1164
#define VERTEX3_DEST2                           0x1168
#define VERTEX3_DEST3                           0x116C
#define VERTEX3_DEST(vtx)                       do { (VERTEX3_DEST0 + (vtx) * 0x04); } while(0);
#define VERTEX2                                 0x1174
#define VERTEX1_NOGO                            0x1178
#define VERTEX1_GO                              0x117C
#define VERTEX0_POS                             0x1180
#define VERTEX0_COL0                            0x1184
#define VERTEX0_COL1                            0x1188
#define VERTEX0_TEX0                            0x118C
#define VERTEX0_TEX1                            0x1190
#define VERTEX0_TEX2                            0x1194
#define VERTEX0_TEX3                            0x1198
#define VERTEX0_TEX(stg)                        do { (VERTEX0_TEX0 + (stg) * 0x04); } while(0);
#define VERTEX0_FOG                             0x119C
#define VERTEX0_POSD                            0x11A0
#define VERTEX0_EMPTY                           0x11BC

/* DWG registers */
#define DWG_SURF0ADDR                           0x1220
#define DWG_SURF1ADDR                           0x1228
#define DWG_SURF2ADDR                           0x1230
#define DWG_SURF3ADDR                           0x1238
    #define DWG_SURFADDR_SRFPCI_AGP                (0 << 0)
    #define DWG_SURFADDR_SRFPCI_PCI                (1 << 0)
    #define DWG_SURFADDR_SRFSYS_LOCAL              (0 << 1)
    #define DWG_SURFADDR_SRFSYS_SYSTEM             (1 << 1)
    #define DWG_SURFADDR_SRFADDR_MASK               0xfffffffc
#define DWG_SURF0MODE                           0x1224
#define DWG_SURF1MODE                           0x122C
#define DWG_SURF2MODE                           0x1234
#define DWG_SURF3MODE                           0x123C
    #define DWG_SURFMODE_SRFWIDTH_MASK              0x00003fff
    #define DWG_SURFMODE_SRFMODE_LINEAR             (0 << 16)
    #define DWG_SURFMODE_SRFMODE_XYLINEAR           HST_SURFMODE_SRFMODE_XYLINEAR
    #define DWG_SURFMODE_SRFMODE_TILEDBLK           HST_SURFMODE_SRFMODE_TILEDBLK
    #define DWG_SURFMODE_SRFMODE_TILEDLIN           HST_SURFMODE_SRFMODE_TILEDLIN
    #define DWG_SURFMODE_SRFMODE_TEXTILEDBLK        HST_SURFMODE_SRFMODE_TEXTILEDBLK
    #define DWG_SURFMODE_SRFMODE_DXT                HST_SURFMODE_SRFMODE_DXT
    #define DWG_SURFMODE_SRFLEVEL_NORMAL            (0 << 19)
    #define DWG_SURFMODE_SRFLEVEL_CUBE_TEXTURE      (1 << 19)
    #define DWG_SURFMODE_SRFLEVEL_3D_TEXTURE        (2 << 19)
    #define DWG_SURFMODE_SRFEND_LITTLE_ENDIAN       (0 << 23)
    #define DWG_SURFMODE_SRFEND_BIG_ENDIAN          (1 << 23)
    #define DWG_SURFMODE_SRFFORMAT_MONO             HST_SURFMODE_SRFFORMAT_MONO 
    #define DWG_SURFMODE_SRFFORMAT_MONOWIN          HST_SURFMODE_SRFFORMAT_MONOWIN 
    #define DWG_SURFMODE_SRFFORMAT_CURSORALPHA      HST_SURFMODE_SRFFORMAT_CURSORALPHA
    #define DWG_SURFMODE_SRFFORMAT_CURSORMONO       HST_SURFMODE_SRFFORMAT_CURSORMONO
    #define DWG_SURFMODE_SRFFORMAT_DXT1             HST_SURFMODE_SRFFORMAT_DXT1
    #define DWG_SURFMODE_SRFFORMAT_DXT2             HST_SURFMODE_SRFFORMAT_DXT2
    #define DWG_SURFMODE_SRFFORMAT_DXT3             HST_SURFMODE_SRFFORMAT_DXT3
    #define DWG_SURFMODE_SRFFORMAT_DXT4             HST_SURFMODE_SRFFORMAT_DXT4
    #define DWG_SURFMODE_SRFFORMAT_DXT5             HST_SURFMODE_SRFFORMAT_DXT5
    #define DWG_SURFMODE_SRFFORMAT_W8               HST_SURFMODE_SRFFORMAT_W8
    #define DWG_SURFMODE_SRFFORMAT_L8               HST_SURFMODE_SRFFORMAT_L8
    #define DWG_SURFMODE_SRFFORMAT_A8               HST_SURFMODE_SRFFORMAT_A8
    #define DWG_SURFMODE_SRFFORMAT_YV12             HST_SURFMODE_SRFFORMAT_YV12
    #define DWG_SURFMODE_SRFFORMAT_W16              HST_SURFMODE_SRFFORMAT_W16
    #define DWG_SURFMODE_SRFFORMAT_L10              HST_SURFMODE_SRFFORMAT_L10
    #define DWG_SURFMODE_SRFFORMAT_A8L8             HST_SURFMODE_SRFFORMAT_A8L8
    #define DWG_SURFMODE_SRFFORMAT_A4R4G4B4         HST_SURFMODE_SRFFORMAT_A4R4G4B4
    #define DWG_SURFMODE_SRFFORMAT_X4R4G4B4         HST_SURFMODE_SRFFORMAT_X4R4G4B4
    #define DWG_SURFMODE_SRFFORMAT_A1R5G5B5         HST_SURFMODE_SRFFORMAT_A1R5G5B5
    #define DWG_SURFMODE_SRFFORMAT_R5G5B5           HST_SURFMODE_SRFFORMAT_R5G5B5
    #define DWG_SURFMODE_SRFFORMAT_R5G6B5           HST_SURFMODE_SRFFORMAT_R5G6B5
    #define DWG_SURFMODE_SRFFORMAT_Z16S0            HST_SURFMODE_SRFFORMAT_Z16S0
    #define DWG_SURFMODE_SRFFORMAT_S1Z15            HST_SURFMODE_SRFFORMAT_S1Z15
    #define DWG_SURFMODE_SRFFORMAT_U8V8             HST_SURFMODE_SRFFORMAT_U8V8
    #define DWG_SURFMODE_SRFFORMAT_U5V5L6           HST_SURFMODE_SRFFORMAT_U5V5L6
    #define DWG_SURFMODE_SRFFORMAT_UYVY             HST_SURFMODE_SRFFORMAT_UYVY
    #define DWG_SURFMODE_SRFFORMAT_W32              HST_SURFMODE_SRFFORMAT_W32
    #define DWG_SURFMODE_SRFFORMAT_A8R8G8B8         HST_SURFMODE_SRFFORMAT_A8R8G8B8
    #define DWG_SURFMODE_SRFFORMAT_X8R8G8B8         HST_SURFMODE_SRFFORMAT_X8R8G8B8
    #define DWG_SURFMODE_SRFFORMAT_A2R10G10B10      HST_SURFMODE_SRFFORMAT_A2R10G10B10
    #define DWG_SURFMODE_SRFFORMAT_S8Z24            HST_SURFMODE_SRFFORMAT_S8Z24
    #define DWG_SURFMODE_SRFFORMAT_U8V8L8X8         HST_SURFMODE_SRFFORMAT_U8V8L8X8
    #define DWG_SURFMODE_SRFFORMAT_L1616            HST_SURFMODE_SRFFORMAT_L1616
    #define DWG_SURFMODE_SRFFORMAT_AYUV8888         HST_SURFMODE_SRFFORMAT_AYUV8888
    
#define DWG_MINWIDTH                            0x12F4
#define DWG_AA_THRESH                           0x12F8
#define DWG_AA_CONFIG                           0x12FC

#define DWG_CTL                                 0x1300
    /* 2d opcodes */
    #define DWG_CTL_OPCODE_SHIFT                0
    #define DWG_CTL_OPCODE_MASK                 (0x0F << DWG_CTL_OPCODE_SHIFT)
    #define DWG_CTL_OPCODE_2DRECT               (0x00 << DWG_CTL_OPCODE_SHIFT)
    #define DWG_CTL_OPCODE_2DLINE               (0x01 << DWG_CTL_OPCODE_SHIFT)
    #define DWG_CTL_OPCODE_BLIT                 (0x02 << DWG_CTL_OPCODE_SHIFT)
    #define DWG_CTL_OPCODE_BLIT_MONO            (0x03 << DWG_CTL_OPCODE_SHIFT)
    
    /* 3d opcodes */
    #define DWG_CTL_OPCODE_3DTRIANGLE           (0x04 << DWG_CTL_OPCODE_SHIFT)
    #define DWG_CTL_OPCODE_3DLINE               (0x05 << DWG_CTL_OPCODE_SHIFT)
    #define DWG_CTL_OPCODE_SPRITE               (0x0C << DWG_CTL_OPCODE_SHIFT)
    #define DWG_CTL_OPCODE_3DPOINT              (0x0D << DWG_CTL_OPCODE_SHIFT)

    /* Other opcodes */
    #define DWG_CTL_OPCODE_CACHEFLUSH           (0x06 << DWG_CTL_OPCODE_SHIFT)
    #define DWG_CTL_OPCODE_CACHEFLUSHINV        (0x07 << DWG_CTL_OPCODE_SHIFT)
    #define DWG_CTL_OPCODE_ZINIT                (0x08 << DWG_CTL_OPCODE_SHIFT)
    #define DWG_CTL_OPCODE_ZINIT_MASK           (0x0A << DWG_CTL_OPCODE_SHIFT)
    #define DWG_CTL_OPCODE_FRAGRESOLVE          (0x09 << DWG_CTL_OPCODE_SHIFT)

    #define DWG_CTL_ROP_SHIFT                   4
    #define DWG_CTL_ROP_MASK                    (0x0F << DWG_CTL_ROP_SHIFT)
    #define DWG_CTL_ROP_0                       (0x00 << DWG_CTL_ROP_SHIFT)
    #define DWG_CTL_ROP_1                       (0x01 << DWG_CTL_ROP_SHIFT)
    #define DWG_CTL_ROP_2                       (0x02 << DWG_CTL_ROP_SHIFT)
    #define DWG_CTL_ROP_3                       (0x03 << DWG_CTL_ROP_SHIFT)
    #define DWG_CTL_ROP_4                       (0x04 << DWG_CTL_ROP_SHIFT)
    #define DWG_CTL_ROP_5                       (0x05 << DWG_CTL_ROP_SHIFT)
    #define DWG_CTL_ROP_6                       (0x06 << DWG_CTL_ROP_SHIFT)
    #define DWG_CTL_ROP_7                       (0x07 << DWG_CTL_ROP_SHIFT)
    #define DWG_CTL_ROP_8                       (0x08 << DWG_CTL_ROP_SHIFT)
    #define DWG_CTL_ROP_9                       (0x09 << DWG_CTL_ROP_SHIFT)
    #define DWG_CTL_ROP_A                       (0x0A << DWG_CTL_ROP_SHIFT)
    #define DWG_CTL_ROP_B                       (0x0B << DWG_CTL_ROP_SHIFT)
    #define DWG_CTL_ROP_C                       (0x0C << DWG_CTL_ROP_SHIFT)
    #define DWG_CTL_ROP_D                       (0x0D << DWG_CTL_ROP_SHIFT)
    #define DWG_CTL_ROP_E                       (0x0E << DWG_CTL_ROP_SHIFT)
    #define DWG_CTL_ROP_F                       (0x0F << DWG_CTL_ROP_SHIFT)
    #define DWG_CTL_ROP_BLACKNESS               DWG_CTL_ROP_0
    #define DWG_CTL_ROP_INVERT                  DWG_CTL_ROP_5
    #define DWG_CTL_ROP_SRCCOPY                 DWG_CTL_ROP_C
    #define DWG_CTL_ROP_WHITENESS               DWG_CTL_ROP_F
    #define DWG_CTL_SRCIDX_SHIFT                8
    #define DWG_CTL_SRCIDX_MASK                 (0x03 << DWG_CTL_SRCIDX_SHIFT)
    #define DWG_CTL_SRCIDX_0                    (0x00 << DWG_CTL_SRCIDX_SHIFT)
    #define DWG_CTL_SRCIDX_1                    (0x01 << DWG_CTL_SRCIDX_SHIFT)
    #define DWG_CTL_SRCIDX_2                    (0x02 << DWG_CTL_SRCIDX_SHIFT)
    #define DWG_CTL_SRCIDX_3                    (0x03 << DWG_CTL_SRCIDX_SHIFT)
    #define DWG_CTL_DSTIDX_SHIFT                10
    #define DWG_CTL_DSTIDX_MASK                 (0x03 << DWG_CTL_DSTIDX_SHIFT)
    #define DWG_CTL_DSTIDX_0                    (0x00 << DWG_CTL_DSTIDX_SHIFT)
    #define DWG_CTL_DSTIDX_1                    (0x01 << DWG_CTL_DSTIDX_SHIFT)
    #define DWG_CTL_DSTIDX_2                    (0x02 << DWG_CTL_DSTIDX_SHIFT)
    #define DWG_CTL_DSTIDX_3                    (0x03 << DWG_CTL_DSTIDX_SHIFT)
    #define DWG_CTL_PATTERN_SHIFT               12
    #define DWG_CTL_PATTERN_MASK                (0x03 << DWG_CTL_PATTERN_SHIFT)
    #define DWG_CTL_PATTERN_SOLID               (0x00 << DWG_CTL_PATTERN_SHIFT)
    #define DWG_CTL_PATTERN_COLOR               (0x01 << DWG_CTL_PATTERN_SHIFT)
    #define DWG_CTL_PATTERN_MONO_OPAQUE         (0x02 << DWG_CTL_PATTERN_SHIFT)
    #define DWG_CTL_PATTERN_MONO_TRANSPARENT    (0x03 << DWG_CTL_PATTERN_SHIFT)
    #define DWG_CTL_TRANSPARENT_SHIFT           12
    #define DWG_CTL_TRANSPARENT                 (0x01 << DWG_CTL_TRANSPARENT_SHIFT)
    #define DWG_CTL_XDIR_SHIFT                  12
    #define DWG_CTL_XDIR_REVERSE                (0x01 << DWG_CTL_XDIR_SHIFT)
    #define DWG_CTL_YDIR_SHIFT                  13
    #define DWG_CTL_YDIR_REVERSE                (0x01 << DWG_CTL_YDIR_SHIFT)
    #define DWG_CTL_XOFFSET_SHIFT               14
    #define DWG_CTL_XOFFSET_MASK                (0x07 << DWG_CTL_XOFFSET_SHIFT)
    #define DWG_CTL_YOFFSET_SHIFT               17
    #define DWG_CTL_YOFFSET_MASK                (0x07 << DWG_CTL_YOFFSET_SHIFT)
    #define DWG_CTL_ADJMODE_SHIFT               20
    #define DWG_CTL_ADJMODE_MASK                (0xFF << DWG_CTL_ADJMODE_SHIFT)
    #define DWG_CTL_ADJMODE_NONE                (0x00 << DWG_CTL_ADJMODE_SHIFT)
    #define DWG_CTL_ADJMODE_ADJXY               (0x01 << DWG_CTL_ADJMODE_SHIFT)
    #define DWG_CTL_CLRLINSRC_SHIFT             21
    #define DWG_CTL_CLRLINSRC                   (0x01 << DWG_CTL_CLRLINSRC_SHIFT)
    #define DWG_CTL_CLRXYEXT_SHIFT              21
    #define DWG_CTL_CLRXYEXT                    (0x01 << DWG_CTL_CLRXYEXT_SHIFT)
    #define DWG_CTL_LINE_CLOSE_SHIFT            22
    #define DWG_CTL_LINE_CLOSE                  (0x01 << DWG_CTL_LINE_CLOSE_SHIFT)
    #define DWG_CTL_COLORKEY_ENABLE_SHIFT       22
    #define DWG_CTL_COLORKEY_ENABLE             (0x01 << DWG_CTL_COLORKEY_ENABLE_SHIFT)
    #define DWG_CTL_CLIP_SHIFT                  23
    #define DWG_CTL_CLIP_DISABLE                (0x00 << DWG_CTL_CLIP_SHIFT)
    #define DWG_CTL_CLIP_ENABLE                 (0x01 << DWG_CTL_CLIP_SHIFT)

    /* Detailed 2d opcodes */
    #define DWG_CTL_OPCODE_2DRECT_SOLID         (DWG_CTL_OPCODE_2DRECT | DWG_CTL_PATTERN_SOLID)
    #define DWG_CTL_OPCODE_2DRECT_COLOR         (DWG_CTL_OPCODE_2DRECT | DWG_CTL_PATTERN_COLOR)
    #define DWG_CTL_OPCODE_2DRECT_MONO_OPAQUE   (DWG_CTL_OPCODE_2DRECT | DWG_CTL_PATTERN_MONO_OPAQUE)
    #define DWG_CTL_OPCODE_2DRECT_MONO_TRANS    (DWG_CTL_OPCODE_2DRECT | DWG_CTL_PATTERN_MONO_TRANSPARENT)
    #define DWG_CTL_OPCODE_BLIT_MONO_OPAQUE     (DWG_CTL_OPCODE_BLIT_MONO)
    #define DWG_CTL_OPCODE_BLIT_MONO_TRANS      (DWG_CTL_OPCODE_BLIT_MONO | DWG_CTL_TRANSPARENT)
    #define DWG_CTL_OPCODE_2DLINE_OPEN          (DWG_CTL_OPCODE_2DLINE | DWG_CTL_CLRXYEXT | DWG_CTL_ADJMODE_ADJXY)
    #define DWG_CTL_OPCODE_2DLINE_CLOSE         (DWG_CTL_OPCODE_2DLINE | DWG_CTL_CLRXYEXT | DWG_CTL_LINE_CLOSE | DWG_CTL_ADJMODE_ADJXY)
    #define DWG_CTL_OPCODE_2DLINEEXT_OPEN       (DWG_CTL_OPCODE_2DLINE | DWG_CTL_ADJMODE_ADJXY)
    #define DWG_CTL_OPCODE_2DLINEEXT_CLOSE      (DWG_CTL_OPCODE_2DLINE | DWG_CTL_LINE_CLOSE | DWG_CTL_ADJMODE_ADJXY)

    /* Fake 3d opcode */
    #define DWG_CTL_OPCODE_POINT                (DWG_CTL_OPCODE_3DLINE | DWG_CTL_LINE_CLOSE)

    /* 3d opcodes bitfield */
    #define DWG_CTL_READ_Z_SHIFT                8
    #define DWG_CTL_READ_Z_FROM_VIDEO_MEMORY    (1 << DWG_CTL_READ_Z_SHIFT) 
    #define DWG_CTL_READ_Z_FROM_DEPTH_INIT      (0 << DWG_CTL_READ_Z_SHIFT)
    #define DWG_CTL_MASK_Z_SHIFT                9
    #define DWG_CTL_MASK_Z_ENABLE               (1 << DWG_CTL_MASK_Z_SHIFT)
    #define DWG_CTL_FVF_SHIFT                   8
    #define DWG_CTL_FVF_MASK                    (0xFF << DWG_CTL_FVF_SHIFT)
    #define DWG_CTL_FVF_COLOR0                  (0x02 << DWG_CTL_FVF_SHIFT)
    #define DWG_CTL_FVF_COLOR1                  (0x04 << DWG_CTL_FVF_SHIFT)
    #define DWG_CTL_FVF_FOG                     (0x08 << DWG_CTL_FVF_SHIFT)
    #define DWG_CTL_FVF_TEXCOORD0               (0x10 << DWG_CTL_FVF_SHIFT)
    #define DWG_CTL_FVF_TEXCOORD1               (0x20 << DWG_CTL_FVF_SHIFT)
    #define DWG_CTL_FVF_TEXCOORD2               (0x40 << DWG_CTL_FVF_SHIFT)
    #define DWG_CTL_FVF_TEXCOORD3               (0x80 << DWG_CTL_FVF_SHIFT)
    #define DWG_CTL_COLORCLAMPEN_SHIFT          17
    #define DWG_CTL_COLORCLAMPEN_MASK           (0xFF << DWG_CTL_COLORCLAMPEN_SHIFT)
    #define DWG_CTL_COLORCLAMPEN_DISABLED       (0x00 << DWG_CTL_COLORCLAMPEN_SHIFT)
    #define DWG_CTL_COLORCLAMPEN_ENABLED        (0x01 << DWG_CTL_COLORCLAMPEN_SHIFT)
    #define DWG_CTL_PSHADERARITH_SHIFT          18
    #define DWG_CTL_PSHADERARITH_MASK           (0xFF << DWG_CTL_PSHADERARITH_SHIFT)
    #define DWG_CTL_PSHADERARITH_NOSAT          (0x00 << DWG_CTL_PSHADERARITH_SHIFT)
    #define DWG_CTL_PSHADERARITH_SAT            (0x01 << DWG_CTL_PSHADERARITH_SHIFT)
    #define DWG_CTL_PSHADERSIZE_SHIFT           19
    #define DWG_CTL_PSHADERSIZE_MASK            (0xFF << DWG_CTL_PSHADERSIZE_SHIFT)
    #define DWG_CTL_PSHADERSIZE_6INST           (0x00 << DWG_CTL_PSHADERSIZE_SHIFT)
    #define DWG_CTL_PSHADERSIZE_12INST          (0x01 << DWG_CTL_PSHADERSIZE_SHIFT)
    #define DWG_CTL_COLRNDMODE_SHIFT            21
    #define DWG_CTL_COLRNDMODE_MASK             (0xFF << DWG_CTL_COLRNDMODE_SHIFT)
    #define DWG_CTL_COLRNDMODE_16PIXEL          (0x00 << DWG_CTL_COLRNDMODE_SHIFT)
    #define DWG_CTL_COLRNDMODE_32PIXEL          (0x01 << DWG_CTL_COLRNDMODE_SHIFT)

    /* Cache flush and cache flush invalidate opcode */
    #define DWG_CTL_ICACHESEL                   0x00000100
    #define DWG_CTL_DCACHESEL                   0x00000200
    #define DWG_CTL_FCACHESEL                   0x00000400
    #define DWG_CTL_CACHESEL_ALL                (DWG_CTL_ICACHESEL | DWG_CTL_DCACHESEL | DWG_CTL_FCACHESEL)
    #define DWG_CTL_CACHEFLUSH_ALL              (DWG_CTL_OPCODE_CACHEFLUSH | DWG_CTL_CACHESEL_ALL)
    #define DWG_CTL_CACHEFLUSHINV_ALL           (DWG_CTL_OPCODE_CACHEFLUSHINV | DWG_CTL_CACHESEL_ALL)
    /* For now: Local and System cache are set to all. */
    #define DWG_CTL_CACHEFLUSH_LOCAL            DWG_CTL_CACHEFLUSH_ALL
    #define DWG_CTL_CACHEFLUSH_SYSTEM           DWG_CTL_CACHEFLUSH_ALL
    #define DWG_CTL_CACHEFLUSHINV_LOCAL         DWG_CTL_CACHEFLUSHINV_ALL
    #define DWG_CTL_CACHEFLUSHINV_SYSTEM        DWG_CTL_CACHEFLUSHINV_ALL
#define DWG_XYSRC                               0x1304
#define DWG_XYDST                               0x1308
#define DWG_XYLEN                               0x130C
#define DWG_LINSRC                              DWG_XYSRC
#define DWG_LINDST                              DWG_XYDST
#define DWG_LINLEN                              DWG_XYLEN
#define DWG_XYSTART                             DWG_XYSRC
#define DWG_XYEND                               DWG_XYDST
#define DWG_XYSTARTEXT                          DWG_XYLEN
#define DWG_XYENDEXT                            0x1310
#define DWG_FCOL                                0x1314
#define DWG_BCOL                                0x1318
#define DWG_PAT0                                0x131C
#define DWG_PAT1                                0x1320
#define DWG_PLNWT                               0x1324
#define DWG_CLEFTTOP                            0x1328
#define DWG_CRIGHTBOTTOM                        0x132C
#define DWG_PAD                                 0x1374

#define DWG_GO                                  0x0080
#define DWG_CTL_GO                              (DWG_CTL            + DWG_GO)
#define DWG_XYSRC_GO                            (DWG_XYSRC          + DWG_GO)
#define DWG_XYDST_GO                            (DWG_XYDST          + DWG_GO)
#define DWG_XYLEN_GO                            (DWG_XYLEN          + DWG_GO)
#define DWG_LINSRC_GO                           (DWG_LINSRC         + DWG_GO)
#define DWG_LINDST_GO                           (DWG_LINDST         + DWG_GO)
#define DWG_LINLEN_GO                           (DWG_LINLEN         + DWG_GO)
#define DWG_XYSTART_GO                          (DWG_XYSTART        + DWG_GO)
#define DWG_XYEND_GO                            (DWG_XYEND          + DWG_GO)
#define DWG_XYSTARTEXT_GO                       (DWG_XYSTARTEXT     + DWG_GO)
#define DWG_XYENDEXT_GO                         (DWG_XYENDEXT       + DWG_GO)
#define DWG_FCOL_GO                             (DWG_FCOL           + DWG_GO)
#define DWG_BCOL_GO                             (DWG_BCOL           + DWG_GO)
#define DWG_PAT0_GO                             (DWG_PAT0           + DWG_GO)
#define DWG_PAT1_GO                             (DWG_PAT1           + DWG_GO)
#define DWG_PLNWT_GO                            (DWG_PLNWT          + DWG_GO)
#define DWG_CLEFTTOP_GO                         (DWG_CLEFTTOP       + DWG_GO)
#define DWG_CRIGHTBOTTOM_GO                     (DWG_CRIGHTBOTTOM   + DWG_GO)

/* Fog registers */
#define FOG_BLEND                               0x12D4
#define FOG_START                               0x12D8
#define FOG_END                                 0x12DC
#define FOG_TABLE                               0x12E0

/* Pixel Shader registers */
#define PSHADER_A_INST0                         0x1260
#define PSHADER_ARGB_INST0                      0x1264
#define PSHADER_A_INST1                         0x1268
#define PSHADER_ARGB_INST1                      0x126C
#define PSHADER_AR_LOCAL0                       0x1278
#define PSHADER_GB_LOCAL0                       0x127C
#define PSHADER_AR_LOCAL1                       0x1280
#define PSHADER_GB_LOCAL1                       0x1284

/* Texture Registers */
#define TEX0_CTL                                0x1200
#define TEX1_CTL                                0x1204
#define TEX2_CTL                                0x1208
#define TEX3_CTL                                0x120C
#define TEX_CTL(stg)                            do { (TEX0_CTL + 0x04 * (stg)); } while(0);

#define TEX_FILTER                              0x11C0
#define TEX_ADDRMODE                            0x11C4
#define TEX_BORDERCOL                           0x11C8
#define TEX_CKEY                                0x11CC
#define TEX_KEYMASK                             0x11D0
#define TEX_BUMPMAT0                            0x11D4
#define TEX_BUMPMAT1                            0x11D8
#define TEX_SURFADDR                            0x11DC
#define TEX_SURFMODE                            0x11E0
#define TEX_SURFDIM                             0x11E4


/* PRI Registers */
#define PRI_REGS_OFFSET                         (0x1A00 - 0x1800)
#define PRI_CTL(idx)         (0x1800 + PRI_REGS_OFFSET * idx)
#define PRI_HSRCST(idx)      (0x1804 + PRI_REGS_OFFSET * idx)
#define PRI_HSRCEND(idx)     (0x1808 + PRI_REGS_OFFSET * idx)
#define PRI_LUTCOMP0(idx)    (0x180C + PRI_REGS_OFFSET * idx)
#define PRI_LUTCOMP1(idx)    (0x1810 + PRI_REGS_OFFSET * idx)
#define PRI_LUTCOMP2(idx)    (0x1814 + PRI_REGS_OFFSET * idx)
#define PRI_LUTCOMP3(idx)    (0x1818 + PRI_REGS_OFFSET * idx)
#define PRI_LUTCOMP4(idx)    (0x181C + PRI_REGS_OFFSET * idx)
#define PRI_LUTCOMP5(idx)    (0x1820 + PRI_REGS_OFFSET * idx)
#define PRI_SRFADDRA1(idx)   (0x1824 + PRI_REGS_OFFSET * idx)
#define PRI_SRFADDRA2(idx)   (0x1828 + PRI_REGS_OFFSET * idx)
#define PRI_SRFADDRB1(idx)   (0x182C + PRI_REGS_OFFSET * idx)
#define PRI_SRFADDRB2(idx)   (0x1830 + PRI_REGS_OFFSET * idx)
#define PRI_SRFMODE(idx)     (0x1834 + PRI_REGS_OFFSET * idx)
#define PRI_LUTINDEX(idx)    (0x1838 + PRI_REGS_OFFSET * idx)
#define PRI_LUTRED(idx)      (0x183C + PRI_REGS_OFFSET * idx)
#define PRI_LUTGRN(idx)      (0x1840 + PRI_REGS_OFFSET * idx)
#define PRI_LUTBLU(idx)      (0x1844 + PRI_REGS_OFFSET * idx)
#define PRI_VSRCST(idx)      (0x1848 + PRI_REGS_OFFSET * idx)
#define PRI_VSRCEND(idx)     (0x184C + PRI_REGS_OFFSET * idx)
#define PRI_LUT_AUTOINC_MASK  0x80000000
#define PRI_LUT_MASK          0xffc0
#define PRI_LUT_DATA_SHIFT    8
/* VMIX Registers */
#define VMIX_REGS_OFFSET                         (0x1A80 - 0x1880)
#define VMIX_COLKEYHIGH(idx) do { (0x1880 + VMIX_REGS_OFFSET * idx); } while(0);
#define VMIX_COLKEYLOW(idx)  do { (0x1884 + VMIX_REGS_OFFSET * idx); } while(0);
#define VMIX_CTL(idx)        do { (0x1888 + VMIX_REGS_OFFSET * idx); } while(0);
#define VMIX_OVSCOL(idx)     do { (0x188C + VMIX_REGS_OFFSET * idx); } while(0);

/* Cursor Registers */
#define CURSOR_REGS_OFFSET                      (0x1AC0 - 0x18C0)
#define CURSOR_BASE(idx)                        (0x18C0 + CURSOR_REGS_OFFSET * idx) 
#define CURSOR_COLOR_OFFSET                     (0x0000)
#define CURSOR_CTL_OFFSET                       (0x0004)
#define CURSOR_POSITION_OFFSET                  (0x0008)
#define CURSOR_SRFADDR_OFFSET                   (0x000C)
#define CURSOR_SRFFORMAT_OFFSET                 (0x0010)
#define CURSOR_COLOR(idx)                       (CURSOR_BASE(idx) + CURSOR_COLOR_OFFSET)  
#define CURSOR_CTL(idx)                         (CURSOR_BASE(idx) + CURSOR_CTL_OFFSET)  
#define CURSOR_CTL_MODE_SHIFT                   (0)
#define CURSOR_CTL_MODE_DISABLED                (0 << CURSOR_CTL_MODE_SHIFT)
#define CURSOR_CTL_MODE_XWINDOW                 (0x3 << CURSOR_CTL_MODE_SHIFT)
#define CURSOR_CTL_LUT_SHIFT                    (8)
#define CURSOR_POSITION(idx)                    (CURSOR_BASE(idx) + CURSOR_POSITION_OFFSET)  
#define CURSOR_SRFADDR(idx)                     (CURSOR_BASE(idx) + CURSOR_SRFADDR_OFFSET)   
#define CURSOR_SRFFORMAT(idx)                   (CURSOR_BASE(idx) + CURSOR_SRFFORMAT_OFFSET) 
#define HARDCURSOR_NUMBER                        2
#define HARDCURSOR_A                             0
#define HARDCURSOR_B                             1

/* CRTC Registers */
#define CRTC_REGS_OFFSET                (0x1B00 - 0x1900)
#define CRTC_CTL(idx)            do { (0x1900 + CRTC_REGS_OFFSET * idx); } while(0);
#define CRTC_GRABER(idx)         do { (0x1904 + CRTC_REGS_OFFSET * idx); } while(0);
#define CRTC_HBLANK(idx)         do { (0x1908 + CRTC_REGS_OFFSET * idx); } while(0);
#define CRTC_HOVLCOORD(idx)      do { (0X190C + CRTC_REGS_OFFSET * idx); } while(0);
#define CRTC_HPARAM(idx)         do { (0x1910 + CRTC_REGS_OFFSET * idx); } while(0);
#define CRTC_HSYNC(idx)          do { (0x1914 + CRTC_REGS_OFFSET * idx); } while(0);
#define CRTC_INTERRUPT(idx)      do { (0x1918 + CRTC_REGS_OFFSET * idx); } while(0);
#define CRTC_PRELOAD(idx)        do { (0x191C + CRTC_REGS_OFFSET * idx); } while(0);
#define CRTC_VBLANK(idx)         do { (0x1920 + CRTC_REGS_OFFSET * idx); } while(0);
#define CRTC_VCOUNT(idx)         do { (0x1924 + CRTC_REGS_OFFSET * idx); } while(0);
#define CRTC_VOVLCOORD(idx)      do { (0x1928 + CRTC_REGS_OFFSET * idx); } while(0);
#define CRTC_VPARAM(idx)         do { (0x192C + CRTC_REGS_OFFSET * idx); } while(0);
#define CRTC_VSYNC(idx)          do { (0x1930 + CRTC_REGS_OFFSET * idx); } while(0);
#define CRTC_VSYNCOFFSET(idx)    do { (0x1934 + CRTC_REGS_OFFSET * idx); } while(0);

/* Overlay Registers */
#define OVL_REGS_OFFSET                          (0x1B80 - 0x1980)
#define OVL_CTL(idx)         do { (0x19A8 + OVL_REGS_OFFSET * idx); } while(0);
#define OVL_HSRCST(idx)      do { (0x19AC + OVL_REGS_OFFSET * idx); } while(0);
#define OVL_HSRCEND(idx)     do { (0x19B0 + OVL_REGS_OFFSET * idx); } while(0);
#define OVL_HSRCLST(idx)     do { (0x19B4 + OVL_REGS_OFFSET * idx); } while(0);
#define OVL_LUTINDEX(idx)    do { (0x19B8 + OVL_REGS_OFFSET * idx); } while(0);
#define OVL_LUTRED(idx)      do { (0x19BC + OVL_REGS_OFFSET * idx); } while(0);
#define OVL_LUTGRN(idx)      do { (0x19C0 + OVL_REGS_OFFSET * idx); } while(0);
#define OVL_LUTBLU(idx)      do { (0x19C4 + OVL_REGS_OFFSET * idx); } while(0);
#define OVL_SRFADDRA1(idx)   do { (0x19C8 + OVL_REGS_OFFSET * idx); } while(0);
#define OVL_SRFADDRA2(idx)   do { (0x19CC + OVL_REGS_OFFSET * idx); } while(0);
#define OVL_SRFADDRB1(idx)   do { (0x19D0 + OVL_REGS_OFFSET * idx); } while(0);
#define OVL_SRFADDRB2(idx)   do { (0x19D4 + OVL_REGS_OFFSET * idx); } while(0);
#define OVL_SRFMODE(idx)     do { (0x19D8 + OVL_REGS_OFFSET * idx); } while(0);
#define OVL_HFCLIPMIN(idx)   do { (0x19DC + OVL_REGS_OFFSET * idx); } while(0);
#define OVL_HFCLIPMAX(idx)   do { (0x19E0 + OVL_REGS_OFFSET * idx); } while(0);
#define OVL_HISCALE(idx)     do { (0x19E4 + OVL_REGS_OFFSET * idx); } while(0);
#define OVL_HCOEFINDEX(idx)  do { (0x19E8 + OVL_REGS_OFFSET * idx); } while(0);
#define OVL_HCOEFDATA(idx)   do { (0x19EC + OVL_REGS_OFFSET * idx); } while(0);
#define OVL_VSRCST(idx)      do { (0x19F0 + OVL_REGS_OFFSET * idx); } while(0);
#define OVL_VSRCEND(idx)     do { (0x19F4 + OVL_REGS_OFFSET * idx); } while(0);

/* DEM Registers */
#define DEMCLKSEL1                               0x1C00
#define DEMCLKSEL2                               0x1C04
#define DEMCTL                                   0x1C08
#define DEMREQCTLA                               0x1C0C
#define DEMREQCTLB                               0x1C10
#define DEMDACSENSE                              0x1C14
#define DEMDAC1CTL                               0x1C18
#define DEMDAC2CTL                               0x1C1C
#define DEMDVICTL1                               0x1C20
#define DEMDVICTL2                               0x1C24
#define DEMDAC1CRC                               0x1C28
#define DEMDAC1CRCMSK                            0x1C2C
#define DEMDAC2CRC                               0x1C30
#define DEMDAC2CRCMSK                            0x1C34
#define DEMDVIACRC                               0x1C38
#define DEMDVIACRCMSK                            0x1C3C
#define DEMDVIBCRC                               0x1C40
#define DEMDVIBCRCMSK                            0x1C44
#define DEMDDCIO                                 0x1C48
#define DEMMISCIO                                0x1C4C
#define DEM_BIST_CTL                             0x1C74
#define DEM_BIST_STATUS                          0x1C78
#define DEM_BIST_DEBUG                           0x1C7C

#define MEM_LEVELS                               0x12B0
#define MEM_CONTROL                              0x12B8

#endif
