 /***************************************************************************\
|*                                                                           *|
|*       Copyright 2002 Matrox Graphics Inc. All rights reserved.            *|
|*                                                                           *|
|*     NOTICE TO THE USER: This source code is the copyrighted work of       *| 
|*     Matrox. Users of this source code are hereby granted a nonexclusive,  *|
|*     royalty-free copyright license to use this code in individual and     *| 
|*     commercial software.                                                  *|
|*                                                                           *|
|*     Any use of this source code must include, in the user documenta-      *|
|*     tion and internal comments to the code, notices to the end user       *|
|*     as follows:                                                           *|
|*                                                                           *|
|*       Copyright 2002 Matrox Graphics Inc. All rights reserved.            *|
|*                                                                           *|
|*     The source code is provided to you AS IS and WITH ALL FAULTS. Matrox  *|
|*     makes no representation and gives no warranty whatsoever, whether     *|
|*     express or implied, and without limitation, with regard to the        *|
|*     quality, safety, contents, performance, merchantability, non-         *|
|*     infringement or suitability for any particular or intended purpose    *|
|*     of this source code. In no event will Matrox be liable for any        *|
|*     direct, indirect, punitive, special, incidental or consequential      *|
|*     damages however they may arise and even if Matrox has been previously *|
|*     advised of the possibility of such damages.                           *|
|*                                                                           *|
 \***************************************************************************/
/***************************************************************************************\

File Name:      MtxParhlCore.c

Description:    MTX driver interface implementation for Parhelia core.

References:     None.

Author:         Karl Lessard    <klessard@matrox.com>

    Copyright (c) 2002, Matrox Graphics Inc.
    All Rights Reserved.

\***************************************************************************************/

#define __NO_VERSION__
#include "MtxParhlDrv.h"

/******************************************
 *
 * Global variables
 *
 *****************************************/

/* parhelia core driver */
static mtx_core_driver_t
mtx_parhl_driver = {

    init:       MtxParhlInitDriver,
    cleanup:    MtxParhlCleanupDriver,
    enable:     MtxParhlEnableDevice,
    disable:    MtxParhlDisableDevice,
    open:       MtxParhlOpenDevice,
    close:      MtxParhlCloseDevice,
    get_ioctl:  MtxParhlGetIoctl,
};

/* ioctls specific to parhelia */
static mtx_ioctl_t
MtxParhlIoctls[] = {

    [_IOC_NR(MTX_PARHL_IOCTL_ALLOC_BLOCK) - MTX_PARHL_IOCTL_NROFF] 
    = { MtxParhlIoctlAllocBlock, 1 },
    [_IOC_NR(MTX_PARHL_IOCTL_FREE_BLOCK) - MTX_PARHL_IOCTL_NROFF] 
    = { MtxParhlIoctlFreeBlock, 1 },
    [_IOC_NR(MTX_PARHL_IOCTL_GET_LOCAL_MEMORY_LAYOUT) - MTX_PARHL_IOCTL_NROFF] 
    = { MtxParhlIoctlGetLocalMemoryLayout, 0 },

    /* no more that MTX_IOCTL_RANGE item */
    [MTX_IOCTL_RANGE]                       = { NULL, 0 },
};

/***************************************************************************************
 * Function:       MtxParhlInitSharedRegions
 *
 * Description:    Init a region shared between parhelia core driver and 
 *                 user-space drivers.
 *
 * Parameters:     dev          MTX device
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:       None.
 */
static int MtxParhlInitSharedRegions(mtx_device_t* dev)
{
    SharedLockedData *poSharedLockedData;
    SharedData *poSharedData;
    SHwMemoryMapData *poVideoInfo;
    SHwRegisterMapData *poRegisterInfo;
    SHwRegAccessData *poRegAccess;
    int iCount;
    
    MtxParhlDevice *poParhlDev = (MtxParhlDevice*)dev->core_data;
    
    /* Alloc shared locked region */
    if (!mtx_mem_alloc_region(dev, sizeof(SharedLockedData), MTX_MEMTYPE_PCI_LOCKED, 
                              &poParhlDev->oSharedLockedRegion)) {

        MTX_ERROR("PARHL: Fail to initialize shared locked region\n");
        return -ENOMEM;
    }

    poParhlDev->oSharedLockedRegion.flags |= MTX_REGION_SHARED;
    poParhlDev->oSharedLockedRegion.map_perms = MTX_MAP_AUTH;

    poSharedLockedData = (SharedLockedData*)poParhlDev->oSharedLockedRegion.kaddr;

    /* Init sync objects */
    poSharedLockedData->oDmaSyncData.m_dwHwTagValue     = 0;
    poSharedLockedData->oDmaSyncData.m_dwPhysAddress    = virt_to_bus(&poSharedLockedData->oDmaSyncData);
    poSharedLockedData->oDmaSyncData.m_dwLastValue      = 0;

    poSharedLockedData->oDwgMarkData.m_dwPrimAddrValue  = 0;
    poSharedLockedData->oDwgMarkData.m_dwHwTagValue     = 0;
    poSharedLockedData->oDwgMarkData.m_dwPhysAddress    = virt_to_bus(&poSharedLockedData->oDwgMarkData);
    poSharedLockedData->oDwgMarkData.m_dwLastValue      = 0;

    /* Set physical attributes */
    poSharedLockedData->ulSize = poParhlDev->oSharedLockedRegion.size;
    poSharedLockedData->dwPhysicalAddress = poParhlDev->oSharedLockedRegion.base;

    poSharedLockedData->dwLastPrimendValue = 0;
    poSharedLockedData->ulPendingSoftraps  = 0;

    /* Init hardware map data */
    poRegAccess = (SHwRegAccessData*)poSharedLockedData->oHwRegAccessData;
    poRegAccess->dwLastDwgCtlRegister = 0;
    poRegAccess->dwDirectAccessCount  = 0;

    poVideoInfo = (SHwMemoryMapData*)poSharedLockedData->oHwMemoryMapData;
    poVideoInfo->m_dwPciPhysical = dev->framebuffer.base;
    poVideoInfo->m_dwPciSize     = dev->framebuffer.size; /* aperture size */
    poVideoInfo->m_dwSizeAB      = 32 * 1024 * 1024;//dev->framebuffer.size / 2;
    poVideoInfo->m_dwSizeCD      = 32 * 1024 * 1042;//dev->framebuffer.size / 2;
    poVideoInfo->m_pvSimLinear   = (ULONG)NULL;
    poVideoInfo->m_dwSimPhysical = (ULONG)NULL;
    poVideoInfo->m_dwSimSize     = 0;

    poRegisterInfo = (SHwRegisterMapData*)poSharedLockedData->oHwRegisterMapData;
    poRegisterInfo->m_dwPhysical    = dev->registers.base;
    poRegisterInfo->m_dwSize        = dev->registers.size;
    poRegisterInfo->m_pvSimLinear   = (ULONG)NULL;
    poRegisterInfo->m_dwSimPhysical = (ULONG)NULL;
    poRegisterInfo->m_dwSimSize     = 0;

    /* Init device owner data */
    poSharedLockedData->oHwEngineOwnerData.m_eOwnerId = OWNER_NOBODY;
    poSharedLockedData->oHwEngineOwnerData.m_dwOwnerContext = 0;

    /* Init host surface descriptors */
    for (iCount = 0; iCount < MAX_NB_HOST_DESCRIPTORS; iCount++)
    {
        SHostDescData* poHostDesc = &(poSharedLockedData->aoHostDescriptors[iCount]);
        poHostDesc->m_dwState = HOSTDESC_STATE_INVALID;
        poHostDesc->m_ulRefCount = 0;
    }

    poSharedLockedData->oHostInterfaceMutex = 0; /* Default unlocked value */

    poParhlDev->poSharedLockedData = poSharedLockedData;
    
    /* increment ref count as kernel might also refer to this data */
    atomic_inc(&poParhlDev->oSharedLockedRegion.ref_count);

    /* Alloc shared region */
    if (!mtx_mem_alloc_region(dev, sizeof(SharedData), MTX_MEMTYPE_PCI_LOCKED, /* XXX This should be in normal system memory */
                              &poParhlDev->oSharedRegion)) {
        
        MTX_ERROR("PARHL: Fail to initialize shared region\n");
        return -ENOMEM;
    }
    
    poParhlDev->oSharedRegion.flags |= MTX_REGION_SHARED;
    poParhlDev->oSharedRegion.map_perms = MTX_MAP_AUTH;
    
    poSharedData = (SharedData*)poParhlDev->oSharedRegion.kaddr;

    /* Init common fields */
    poSharedData->ulSize = poParhlDev->oSharedRegion.size;
    poSharedData->dwPhysicalAddress = poParhlDev->oSharedRegion.base;

    poParhlDev->poSharedData = poSharedData;
    poParhlDev->poVideoInfo = poVideoInfo;
    poParhlDev->poRegisterInfo = poRegisterInfo;
    
    /* increment ref count as kernel might also refer to this data */
    atomic_inc(&poParhlDev->oSharedRegion.ref_count);
    
    return 0;
}


/* ~~~~~ CORE INTERFACE ~~~~~ */

/***************************************************************************************
 * Function:       MtxParhlInitDriver
 *
 * Description:    Initialize MTX parhelia driver so it can enable devices properly.
 *
 * Parameters:     None.
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:       None.
 */
int MtxParhlInitDriver(void)
{
    return 0;
}

/***************************************************************************************
 * Function:       MtxParhlCleanupDriver
 *
 * Description:    Cleanup MTX parhelia driver.
 *
 * Parameters:     None.
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:       None.
 */
int MtxParhlCleanupDriver(void)
{
    return 0;
}

/***************************************************************************************
 * Function:       MtxParhlEnableDevice
 *
 * Description:    Enable a device in parhelia core driver and attach private data
 *                 to the mtx device core_data field.
 *
 * Parameters:     dev          MTX device
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:       None.
 */
int MtxParhlEnableDevice(mtx_device_t* dev)
{
    MtxParhlDevice* poParhlDev;
    int ret;

    poParhlDev = MTXALLOC(sizeof(MtxParhlDevice));
    if (!poParhlDev)
        return -ENOMEM;
    
    dev->core_data = (mtx_core_handle_t)poParhlDev;
    
    /* Init shared regions */
    ret = MtxParhlInitSharedRegions(dev);
    if (ret < 0) 
        return ret;

    /* Init surface manager */
    ret = MtxParhlSurfInit(dev);
    if (ret < 0) {
        MTX_ERROR("Fail to initialize surface manager for Parhelia device\n");
        return ret;
    }

    /* Init aperture manager */
    ret = MtxParhlAptInit(dev);
    if (ret < 0) {
        MTX_ERROR("Fail to initialize aperture manager for Parhelia device\n");
        return ret;
    }

    /* Init board for busmastering operations */
    REG_WRITE(dev, DMA_CTL, 0x40);
    pci_write_config_dword(dev->pci_dev, PCI_COMMAND, 0x7); 

    return 0;
}

/***************************************************************************************
 * Function:       MtxParhlDisableDevice
 *
 * Description:    Disable a device from parhelia core driver
 *
 * Parameters:     dev          MTX device
 *
 * Return Value:   None.
 *
 * Comments:       None.
 */
void MtxParhlDisableDevice(mtx_device_t* dev)
{
    MtxParhlDevice *poParhlDev = (MtxParhlDevice*)dev->core_data;

    if (!poParhlDev)
        return;

    /* Cleanup surface manager */
    MtxParhlSurfCleanup(dev);

    /* Cleanup aperture manager */
    MtxParhlAptCleanup(dev);

    /* Cleanup shared regions */
    if (poParhlDev->oSharedLockedRegion.flags & MTX_REGION_ALLOCATED) {
    
        atomic_dec(&poParhlDev->oSharedLockedRegion.ref_count);
        mtx_mem_free_region(dev, &poParhlDev->oSharedLockedRegion);
    }

    if (poParhlDev->oSharedRegion.flags & MTX_REGION_ALLOCATED) {
        
        atomic_dec(&poParhlDev->oSharedRegion.ref_count);
        mtx_mem_free_region(dev, &poParhlDev->oSharedRegion);
    }

    /* free parhelia device memory */
    MTXFREE(poParhlDev, sizeof(*poParhlDev));

    return;
}

/***************************************************************************************
 * Function:       MtxParhlOpenDevice
 *
 * Description:    Called when a new context has been created on a device.
 *
 * Parameters:     dev      MTX device
 *                 ctx      MTX context opening the device
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:       None.
 */
int MtxParhlOpenDevice(mtx_device_t *dev, mtx_context_t *ctx)
{
    return 0;
}

/***************************************************************************************
 * Function:       MtxParhlCloseDevice
 *
 * Description:    Called when a device context has been deleted.
 *
 * Parameters:     dev      MTX device
 *                 ctx      MTX context opening the device
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:       None.
 */
void MtxParhlCloseDevice(mtx_device_t *dev, mtx_context_t *ctx)
{
}


/***************************************************************************************
 * Function:       MtxParhlGetIoctl
 *
 * Description:    Return a MTX ioctl data structure for a specified command
 *                 number.
 *
 * Parameters:     cmd_nr       ioctl command number
 *
 * Return Value:   mtx_ioctl_t for this command, NULL if with don't support
 *                 this command number
 *
 * Comments:       FIXME: try to incorporate cleanly this code to mtx driver, as it
 *                 it the same for all core drivers.
 */
mtx_ioctl_t*
MtxParhlGetIoctl(unsigned int cmd_nr)
{
    int nr = cmd_nr - MTX_PARHL_IOCTL_NROFF;
    
    if (nr > MTX_PARHL_IOCTL_MAXNR || cmd_nr < MTX_PARHL_IOCTL_NROFF)
        return NULL;

    return &MtxParhlIoctls[nr];
}
