 /***************************************************************************\
|*                                                                           *|
|*       Copyright 2002 Matrox Graphics Inc. All rights reserved.            *|
|*                                                                           *|
|*     NOTICE TO THE USER: This source code is the copyrighted work of       *| 
|*     Matrox. Users of this source code are hereby granted a nonexclusive,  *|
|*     royalty-free copyright license to use this code in individual and     *| 
|*     commercial software.                                                  *|
|*                                                                           *|
|*     Any use of this source code must include, in the user documenta-      *|
|*     tion and internal comments to the code, notices to the end user       *|
|*     as follows:                                                           *|
|*                                                                           *|
|*       Copyright 2002 Matrox Graphics Inc. All rights reserved.            *|
|*                                                                           *|
|*     The source code is provided to you AS IS and WITH ALL FAULTS. Matrox  *|
|*     makes no representation and gives no warranty whatsoever, whether     *|
|*     express or implied, and without limitation, with regard to the        *|
|*     quality, safety, contents, performance, merchantability, non-         *|
|*     infringement or suitability for any particular or intended purpose    *|
|*     of this source code. In no event will Matrox be liable for any        *|
|*     direct, indirect, punitive, special, incidental or consequential      *|
|*     damages however they may arise and even if Matrox has been previously *|
|*     advised of the possibility of such damages.                           *|
|*                                                                           *|
 \***************************************************************************/
/***************************************************************************************\

File Name:      MtxParhlApt.c

Description:    MTX Parhelia host aperture management.

References:     None.

Author:         Karl Lessard    <klessard@matrox.com>

    Copyright (c) 2002, Matrox Graphics Inc.
    All Rights Reserved.

\***************************************************************************************/

#define __NO_VERSION__
#include "MtxParhlDrv.h"

// --------------------------------------------------------------------------------------
//                        C O N S T A N T S   A N D   T Y P E S
// --------------------------------------------------------------------------------------

#define DESCRIPTOR_INVALID   0xFFFFFFFF

#define HOSTDESC_IS_VALID(desc)    (!(desc->m_dwState & HOSTDESC_STATE_INVALID))
    
enum IntervalState
{
    IN_FREE_INTERVAL,   // Inside a group of free descriptors
    OUT_FREE_INTERVAL,  // Outside a group of free descriptors
};

// --------------------------------------------------------------------------------------
//                                      C O D E
// --------------------------------------------------------------------------------------

/***************************************************************************************
 * Function:       MtxParhlAptLockAddressSpace
 *
 * Description:    Find free space in host aperture to access big enough to hold the
 *                 the specified size, and lock (reserve) it.
 * 
 * Parameters:     dev              MTX device
 *                 ulOffset         Offset in aperture, invalid if any address block
 *                                  will do (0 means first address of video aperture).
 *                 ulSize           Size requested.
 *
 * Return Value:   A heap block representing the address space, NULL if failed.
 *
 * Comments:       None.
 */
mtx_heap_block_t* 
MtxParhlAptLockAddressSpace(mtx_device_t *dev, ULONG ulOffset, ULONG ulSize)
{
    mtx_heap_block_t *poBlock = NULL;
  
    if (ulOffset != MTX_PARHL_INVALID_ADDRESS) {

        /* Allocate a block in address space block */
        poBlock = mtx_heap_alloc_block_at(dev, &dev->heap_groups[MTX_MEMTYPE_LOCAL], 
                                          dev->framebuffer.base + ulOffset, ulSize);
        
    } else {

        /* Allocate a block in address space block */
        poBlock = mtx_heap_alloc_block_into(dev, &dev->heap_groups[MTX_MEMTYPE_LOCAL], ulSize);
    }
    
    if (!poBlock)
    {
        MTX_ERROR("Fail to lock an address block in host aperture\n");
        return NULL;
    }
    
    return poBlock;
}

/***************************************************************************************
 * Function:       MtxParhlAptUnlockAddressSpace
 *
 * Description:    Unlock a address space previously reserved in the host
 *                 aperture.
 * 
 * Parameters:     dev              MTX device
 *                 poBlock          Address block we want to release
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:       None.
 */
int
MtxParhlAptUnlockAddressSpace(mtx_device_t *dev, mtx_heap_block_t* poBlock)
{
    int ret;
   
    /* Allocate a block in address space block */
    if ((ret = mtx_heap_free_block(dev, poBlock)) < 0)
    {
        MTX_ERROR("Fail to unlock address block in host aperture (base: 0x%08lx)\n",
                  poBlock->region->base);
        return ret;
    }
    
    return 0;
}

/***************************************************************************************
 * Function:       MtxParhlAptInit
 *
 * Description:    Init parhelia aperture manager (surface mapper).
 *
 * Parameters:     dev          MTX device
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:       None.
 */
int MtxParhlAptInit(mtx_device_t* dev)
{
    return 0;
}

/***************************************************************************************
 * Function:       MtxParhlAptCleanup
 *
 * Description:    Cleanup parhelia aperture manager.
 *
 * Parameters:     dev          MTX device
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:       None.
 */
void MtxParhlAptCleanup(mtx_device_t* dev)
{
}
