 /***************************************************************************\
|*                                                                           *|
|*       Copyright 2002 Matrox Graphics Inc. All rights reserved.            *|
|*                                                                           *|
|*     NOTICE TO THE USER: This source code is the copyrighted work of       *| 
|*     Matrox. Users of this source code are hereby granted a nonexclusive,  *|
|*     royalty-free copyright license to use this code in individual and     *| 
|*     commercial software.                                                  *|
|*                                                                           *|
|*     Any use of this source code must include, in the user documenta-      *|
|*     tion and internal comments to the code, notices to the end user       *|
|*     as follows:                                                           *|
|*                                                                           *|
|*       Copyright 2002 Matrox Graphics Inc. All rights reserved.            *|
|*                                                                           *|
|*     The source code is provided to you AS IS and WITH ALL FAULTS. Matrox  *|
|*     makes no representation and gives no warranty whatsoever, whether     *|
|*     express or implied, and without limitation, with regard to the        *|
|*     quality, safety, contents, performance, merchantability, non-         *|
|*     infringement or suitability for any particular or intended purpose    *|
|*     of this source code. In no event will Matrox be liable for any        *|
|*     direct, indirect, punitive, special, incidental or consequential      *|
|*     damages however they may arise and even if Matrox has been previously *|
|*     advised of the possibility of such damages.                           *|
|*                                                                           *|
 \***************************************************************************/
/***************************************************************************************\

File Name:      mtx_fops.c

Description:    File operations on a MTX device file.

References:     None.

Author:         Karl Lessard    <klessard@matrox.com>

    Copyright (c) 2001, Matrox Graphics Inc.
    All Rights Reserved.

\***************************************************************************************/

#define __NO_VERSION__
#include "mtx_drv.h"

/* IOCTLs for MTX driver */
static mtx_ioctl_t mtx_ioctls[] = {
    
    /* ioctl unique nbr for this driver     = { ioctl method name, root only } */
    
    [_IOC_NR(MTX_IOCTL_HARD_RESET)]         = { mtx_ioctl_hard_reset, 1 }, /* for debug */
    
    [_IOC_NR(MTX_IOCTL_GET_MEMORY_INFO)]    = { mtx_ioctl_get_memory_info, 0 },
    [_IOC_NR(MTX_IOCTL_ALLOC_BUFFER)]       = { mtx_ioctl_alloc_buffer, 1 },
    [_IOC_NR(MTX_IOCTL_FREE_BUFFER)]        = { mtx_ioctl_free_buffer, 1 },
    [_IOC_NR(MTX_IOCTL_GET_SHARED_BUFFER)]  = { mtx_ioctl_get_shared_buffer, 0 },
    [_IOC_NR(MTX_IOCTL_RELEASE_SHARED_BUFFER)]  = { mtx_ioctl_release_shared_buffer, 0 },
    [_IOC_NR(MTX_IOCTL_BUSMASTERING_TEST)]  = { mtx_ioctl_busmastering_test, 0 },
    [_IOC_NR(MTX_IOCTL_ADD_CONTEXT)]        = { mtx_ioctl_add_context, 0 },
    [_IOC_NR(MTX_IOCTL_REMOVE_CONTEXT)]     = { mtx_ioctl_remove_context, 0 },

    /* no more that MTX_IOCTL_RANGE item */
    [MTX_IOCTL_RANGE]                       = { NULL, 0 }
};

/***************************************************************************************
 * Function:       mtx_open
 *
 * Description:    Open fop. Called whenever a process open a file controlled
 *                 by the MTX driver.
 *
 * Parameters:     inode               Storing device inode for this file.
 *                 filp                pointer to the file structure.
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:       Minor numbers for a device file is a concatenation of the
 *                 PCI bus nb. and the PCI slot nb. of the hardware. For now,
 *                 minor numbers are only 8 bits values, so we can only support
 *                 up to 16 PCI buses and 16 devices. Eventually, minor numbers 
 *                 will be 16 bits long, so this inconvenience will be fixed.
 *
 *                 Once a process has opened a device, it cannot use it until
 *                 it has created a context for it, using the ioctl.
 */
int mtx_open(struct inode *inode, struct file *filp)
{
    mtx_file_t *priv;
    mtx_device_t *dev;
    int minor;

    /* XXX For now, we just support AGP devices */
    if (!agp_drv) {
        MTX_ERROR("Fail to open device: no AGP driver found\n");
        return -EINVAL;
    }

    /* try to retrieve MTX data from file ptr */
    priv = (mtx_file_t*)filp->private_data;
    if (priv != NULL) {
        MTX_ERROR("Device already opened by this file\n");
        return -EINVAL;
    }
        
    minor = inode->i_rdev;

    /* Get the MTX device from the device file minor number */
    dev = mtx_dev_find(MTX_MINOR_PCIBUS(minor), MTX_MINOR_PCISLOT(minor));
    if (!dev)
        return -EINVAL;

    /* Allocate structure for private file data */
    priv = MTXALLOC(sizeof(mtx_file_t));
    if (!priv)
        return -ENOMEM;
        
    filp->private_data = priv;

    /* Init private file data */
    priv->auth = capable(CAP_SYS_ADMIN) ? 1 : 0;
    priv->dev = dev;
    priv->ctx = NULL;
    
    /* Increment open count for this device */
    spin_lock(&dev->count_lock);
    if (!dev->open_count++) {
        
        int ret = 0;
        spin_unlock(&dev->count_lock);

        /* This is the first reference to this device, initialize it now */
        if ((ret = mtx_dev_init(dev)) < 0) {

            mtx_release(inode, filp);
            return ret;
        }

        /* Dump memory statistics */
        mtx_mem_stats_dump();
        
    } else spin_unlock(&dev->count_lock);

    return 0;
}

/***************************************************************************************
 * Function:       mtx_release
 *
 * Description:    Release fop. Called whenever a process close the last reference to a 
 *                 opened MTX device file.
 *
 * Parameters:     inode               Storing device inode for this file.
 *                 filp                pointer to the file structure.
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:
 *
 */
int mtx_release(struct inode *inode, struct file *filp)
{
    mtx_file_t *priv = (mtx_file_t*)filp->private_data;
    mtx_device_t *dev;

    if (!priv) return -EINVAL;
 
    dev = priv->dev;
    if (dev) {

        if (priv->ctx != NULL)
        {
            MTX_WARNING("Closing a device still in use\n");

            /* Delete file context */
            mtx_ctx_delete(dev, priv->ctx);
        }

        /* Decrement open count for this device */
        spin_lock(&dev->count_lock);
        if (!--dev->open_count) {

            spin_unlock(&dev->count_lock);

            /* This is the last reference to this device,
            * clean it up now */
            mtx_dev_cleanup(dev);
        
            /* Dump memory statistics */
            mtx_mem_stats_dump();

        } else spin_unlock(&dev->count_lock);
    }
        
    /* Free file private data */
    MTXFREE(priv, sizeof(*priv));
    
    return 0;
}

/***************************************************************************************
 * Function:       mtx_ioctl
 *
 * Description:    Ioctl fop. Called whenever a process sends a IOCTL request to
 *                 the device file.
 *
 * Parameters:     inode               Storing device inode for this file.
 *                 filp                pointer to the file structure.
 *                 cmd                 ioctl command id
 *                 arg                 argument passed to ioctl command
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:
 *
 */
int mtx_ioctl(struct inode *inode, struct file *filp, unsigned int cmd, unsigned long arg)
{
    mtx_file_t *priv = (mtx_file_t *)filp->private_data;
    unsigned int type = _IOC_TYPE(cmd);
    unsigned int nr = _IOC_NR(cmd);
    mtx_ioctl_t *ioctl = NULL;

    if (!priv) return -EINVAL;
    
    /* Validate ioctl */
    if ((type != MTX_IOCTL_MAGIC))
        return -ENOTTY;

    if (nr <= MTX_IOCTL_MAXNR) {
        
        /* Get mtx ioctl data from command number */
        ioctl = &mtx_ioctls[nr];
        
    } else {
    
        mtx_device_t *dev = priv->dev;

        /* This maybe a core specific ioctl, try to get its data */
        if (dev->core && dev->core->get_ioctl)
            ioctl = dev->core->get_ioctl(nr);

        if (!ioctl)
            /* invalid command number */
            return -ENOTTY;
    }

    /* Check for user access */
    if (ioctl->auth_needed && !priv->auth)               /* needs root authentication */
        return -EACCES;

    /* Call the ioctl specific function */
    return (ioctl->func) ? ioctl->func(priv, cmd, arg) : -EFAULT;
}

/***************************************************************************************
 * Function:       mtx_mmap
 *
 * Description:    Mapping a device to user-space memory, i.e. associating a range 
 *                 of user-space addresses to device memory. All regions a user
 *                 process can map has been previously initialized by the kernel
 *                 module, and can be referred in the memory_regions list for this
 *                 device.
 *
 * Parameters:     filp             pointer to the file structure
 *                 vma              pointer to the vma structure
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:       
 */
int
mtx_mmap(struct file *filp, struct vm_area_struct *vma)
{
    mtx_file_t *priv = (mtx_file_t *)filp->private_data;
    mtx_device_t *dev = priv ? priv->dev : NULL;
    unsigned long offset = vma->vm_pgoff << PAGE_SHIFT;
    mtx_region_t* region = NULL;
    int retcode = -EINVAL;
    
    if (!dev)
        return -EINVAL;
 
    /* Check if the physical address really belong to us, by matching it
     * with a memory region for this device. (Note: only device regions can be mapped, 
     * and not individual memory block) */
    region = mtx_find_region_from_base(dev, offset);

    /* Don't map if no region matched */
    if (!region)
        return -EINVAL;
    
    /* Check if user have permissions to do it */
    if (!((region->map_perms != MTX_MAP_NONE) &&   
            (((region->map_perms & MTX_MAP_AUTH) && priv->auth) ||
            ((region->map_perms & MTX_MAP_OWNER) && (region->pid == current->pid)) ||
            ((region->map_perms & MTX_MAP_OTHER)))))
        return -EPERM;
  
    switch (region->type) {

        case MTX_MEMTYPE_AGP:
            retcode = mtx_vm_map_agp(dev, region, vma); 
            break;

        case MTX_MEMTYPE_IO:
            retcode = mtx_vm_map_io(dev, region, vma);
            break;

        case MTX_MEMTYPE_PCI_LOCKED:
            if (filp->f_flags & O_SYNC) vma->vm_flags |= VM_IO;
            retcode = mtx_vm_map_pci(dev, region, vma);
            break;

        case MTX_MEMTYPE_SYS:
            retcode = mtx_vm_map_sys(dev, region, vma); 
            break;
            
        default:
            /* Memory type not supported by mmap */
            return -EINVAL;
    }

    if (!retcode)
        /* Call explicitly the open method */
        vma->vm_ops->open(vma);

    return retcode;
}
