 /***************************************************************************\
|*                                                                           *|
|*       Copyright 2002 Matrox Graphics Inc. All rights reserved.            *|
|*                                                                           *|
|*     NOTICE TO THE USER: This source code is the copyrighted work of       *| 
|*     Matrox. Users of this source code are hereby granted a nonexclusive,  *|
|*     royalty-free copyright license to use this code in individual and     *| 
|*     commercial software.                                                  *|
|*                                                                           *|
|*     Any use of this source code must include, in the user documenta-      *|
|*     tion and internal comments to the code, notices to the end user       *|
|*     as follows:                                                           *|
|*                                                                           *|
|*       Copyright 2002 Matrox Graphics Inc. All rights reserved.            *|
|*                                                                           *|
|*     The source code is provided to you AS IS and WITH ALL FAULTS. Matrox  *|
|*     makes no representation and gives no warranty whatsoever, whether     *|
|*     express or implied, and without limitation, with regard to the        *|
|*     quality, safety, contents, performance, merchantability, non-         *|
|*     infringement or suitability for any particular or intended purpose    *|
|*     of this source code. In no event will Matrox be liable for any        *|
|*     direct, indirect, punitive, special, incidental or consequential      *|
|*     damages however they may arise and even if Matrox has been previously *|
|*     advised of the possibility of such damages.                           *|
|*                                                                           *|
 \***************************************************************************/
/***************************************************************************************\

File Name:      mtx_drv.h

Description:    Main header file for MTX kernel module.

References:     None.

Author:         Karl Lessard    <klessard@matrox.com>

    Copyright (c) 2001, Matrox Graphics Inc.
    All Rights Reserved.

\***************************************************************************************/

#ifndef INC_MTXDRV_H
#define INC_MTXDRV_H

#ifndef __cplusplus

/*******************************************
 *
 * Included headers
 *
 ******************************************/

#include <linux/config.h>
#include <linux/types.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/string.h>
#include <linux/fs.h> 
#include <linux/pci.h>
#include <linux/spinlock.h>
#include <linux/agp_backend.h>
#include <asm/mtrr.h>
#include <asm/uaccess.h>

#include "mtx.h"

/******************************************
 *
 * Constants and types
 *
 *****************************************/

/* PCI ids */
#ifndef PCI_DEVICE_ID_G450
#define PCI_DEVICE_ID_G450              0x0525
#endif
#ifndef PCI_DEVICE_ID_G550
#define PCI_DEVICE_ID_G550              0x2527
#endif
#ifndef PCI_DEVICE_ID_PARHELIA
#define PCI_DEVICE_ID_PARHELIA_512      0x0527
#endif
#define MTX_PCI_DEVICE_ID(id, subid)                                \
    { PCI_VENDOR_ID_MATROX,             /* vendor id */             \
      id,                               /* device id */             \
      PCI_VENDOR_ID_MATROX,             /* vendor subid */          \
      subid,                            /* device subid */          \
      (PCI_CLASS_DISPLAY_VGA << 8),     /* device class */          \
      0xffffff00,                       /* class mask */            \
      0 }                               /* private_data */
      
#define NBR_SUPPORTED_CARDS             1

#define MAX_CONTEXTS_PER_DEVICE         (PAGE_SIZE * 8)

#if LINUX_VERSION_CODE < 0x020410
      
#define list_for_each_safe(pos, n, head) \
	for (pos = (head)->next, n = pos->next; pos != (head); \
		pos = n, n = pos->next)

#endif

/******************************************
 *
 * Data structures
 *
 *****************************************/

struct mtx_device;
typedef unsigned long mtx_core_handle_t;
      
/*
 * mtx_agp_driver_t
 *
 * Holds information about AGP driver backend
 */
typedef struct mtx_agp_driver {
    
    struct mtx_device  *dev;            /* MTX device that control AGP driver */
    
    agp_kern_info       kern_info;      /* AGP information retrieve from kernel driver */
    off_t               aperture_base;  /* Shortcut to kern_info->aper_base */
    size_t              aperture_size;  /* Size in bytes of AGP aperture */
    
    int                 rate;           /* Current transfer rate */
    int                 enabled;        /* Set to 1 when AGP controller has been enabled once */

    int                 mtrr;           /* MTRR number used for AGP aperture (-1 if none) */

} mtx_agp_driver_t;

/*
 * mtx_context_t
 *
 * MTX process context, identified by a unique id
 */
typedef struct mtx_context {

    struct mtx_device  *dev;            /* MTX device */

    int                 unique_id;      /* Unique ID for this context */
    struct list_head    region_list;    /* Memory region owned by this context */
  #ifdef MTX_WITH_MEMORY_HEAPS
    struct list_head    block_list;     /* Memory block owned by this context */
  #endif
    mtx_core_handle_t   core_data;      /* Data private to core driver */

} mtx_context_t;

/*
 * mtx_file_t
 *
 * MTX file private data
 */
typedef struct mtx_file {
    
    int                  auth;          /* Authorization received by root (boolean value)
                                           (always true for a root process */
    struct mtx_device   *dev;           /* MTX device this file refers to */
    mtx_context_t       *ctx;           /* MTX context */

} mtx_file_t;
    
/*
 * mtx_ioctl_t
 *
 * definition of an I/O control in MTX.
 * Same model used by DRM from VA linux.
 */
typedef struct mtx_ioctl {

    int (*func)(mtx_file_t*, unsigned int, unsigned long);  /* Function to call */
    
    int auth_needed;              /* True if this ioctl can only be called by a root 
                                     (or authorized) processes */

} mtx_ioctl_t;

/*
 * mtx_core_driver_t
 *
 * common interface to any core driver used to driver
 * specific boards
 */
typedef struct mtx_core_driver {

    /* Init driver so it can enable devices properly (mandatory) */
    int             (*init)(void);
    
    /* Cleanup driver (mandatory) */
    int             (*cleanup)(void);
    
    /* Enable a device in core driver and return attach a handle to it */
    int             (*enable)(struct mtx_device* dev);

    /* Disable a valid device from the core driver */
    void            (*disable)(struct mtx_device* dev);

    /* Open a new context on a device */
    int             (*open)(struct mtx_device* dev, struct mtx_context* ctx);

    /* Close a context on a device */
    void            (*close)(struct mtx_device* dev, struct mtx_context* ctx);

    /* Return an MTX ioctl structure for this command number if we own it */
    mtx_ioctl_t*    (*get_ioctl)(unsigned int cmd_nr);

} mtx_core_driver_t;

/*
 * mtx_region_t
 *
 * Physical memory space used by the driver. There is only
 * one instance of region, while there may be more than one heap
 * block referring to it (when region is shared).
 */
typedef enum {  /* region flags */

    MTX_REGION_DYNAMIC      = 0x01,     /* region has been dynamically allocated */
    MTX_REGION_SHARED       = 0x02,     /* region is a shared region */
    MTX_REGION_ALLOCATED    = 0x04,     /* memory has been allocated for this region */
        
} mtx_region_flags_t;

typedef enum {  /* map flags */

    MTX_MAP_NONE            = 0x00,     /* No process can map the region (use alone) */
    MTX_MAP_AUTH            = 0x01,     /* Authorized (and root) processes can map the region */
    MTX_MAP_OWNER           = 0x02,     /* The process owning this region can map the region */
    MTX_MAP_OTHER           = 0x04,     /* Other non-root processes can map the region */
    
} mtx_map_perms_t;

typedef struct mtx_region {

    struct list_head    list_node;      /* For device list maintenance */
    
    struct list_head    ctx_list_node;  /* For regions private to a context */
    pid_t               pid;            /* ID of owner process (0 if allocated by kernel or shared) */

    unsigned long       base;           /* base address */
    unsigned long       busaddr;        /* bus address */
    void*               kaddr;          /* kernel virtual/logical address */
    unsigned long       size;           /* size of the region (XXX desired, not real) */

    mtx_memory_type     type;           /* Type of the region */
   
    unsigned long       io_flags;       /* PCI flags (for I/O regions) */
    mtx_map_perms_t     map_perms;      /* User-mappings permissions and flags */
    u8                  flags;          /* region flags */
    
    atomic_t            ref_count;      /* number of reference to this buffer(0 if private region)*/
    int                 mtrr;           /* Memory type range register used (negative if unused) */
    
} mtx_region_t;

#ifdef MTX_WITH_MEMORY_HEAPS
/*
 * mtx_heap_block_t
 *
 * A heap block is a working structure used by heaps to 
 * manage memory regions.
 */
struct mtx_heap;
typedef struct mtx_heap_block {

    struct list_head    heap_list_node; /* for heap list maintenance */    
    struct list_head    ctx_list_node;  /* for ctx list maintenance */    

    mtx_region_t       *region;         /* the physical region */

    struct mtx_heap    *heap;           /* heap that owns this block */
    unsigned long       offset;         /* offset in heap */
#if 0
    unsigned long   aligned_size;       /* the apparent size */
    unsigned long   aligned_offset;     /* offset after alignment */
    unsigned long   aligned_base;       /* physical address after alignment */
#endif
    
} mtx_heap_block_t;

/*
 * mtx_heap_t
 *
 * A heap is responsible of managing memory allocation
 * in a specific portion of memory.
 */
typedef struct mtx_heap {
    
    struct list_head    list_node;     /* for list maintenance */    
    
    unsigned long       base;          /* The physical base address */
    unsigned long       size;          /* Size of this heap */
    mtx_memory_type     type;          /* Type of memory managed by this heap */

    struct list_head    block_list;    /* list of block allocated by this heap */ 

    union {
        /* data specific to system heaps */
        struct { 
        } system;
        
        /* data specific to local heaps */
        struct {
            struct list_head free_regions;  /* list of free regions */    
        } local;
    } data;
    
} mtx_heap_t;

/* 
 * mtx_heap_group_t
 *
 * A group of heaps is a list of all available heaps for
 * for the same memory location.
 */
typedef struct mtx_heap_group {

    struct list_head    heap_list;  /* list of heaps included in this group */
    unsigned int        heap_count; /* number of heaps in this group */

    unsigned long       base;       /* lower base for his group */
    unsigned long       size;       /* total size of this group */

} mtx_heap_group_t;
#endif

/*
 * mtx_device_t
 *
 * logical representation of a physical
 * device
 */
typedef struct mtx_device {

    struct list_head            list_node;      /* For list maintenance */
   
    struct pci_dev             *pci_dev;        /* PCI device structure */
    const struct pci_device_id *pci_id;         /* PCI device id */

    int                         agp_cap_offset; /* AGP pci capability offset, -1 if PCI board */

    spinlock_t                  count_lock;     /* spinlock used to access count value */
    int                         open_count;     /* count of opened reference to this device */

    struct semaphore            semaphore;      /* For some other data accesses to this device */

    struct list_head            memory_regions; /* Device memory regions, global to any contexts */
    mtx_region_t                framebuffer;    /* Framebuffer region */
    mtx_region_t                registers;      /* Control registers region */
    
  #ifdef MTX_WITH_MEMORY_HEAPS
    mtx_heap_group_t            heap_groups[MTX_MEMTYPE_COUNT]; /* Heap groups */
  #endif

    mtx_core_driver_t          *core;           /* Core driver interface */
    mtx_core_handle_t           core_data;      /* Core driver data handle */

    void*                       context_bitmap; /* Bitmap for context unique id's */ 
    
 } mtx_device_t;


/******************************************
 *
 * Helping macros
 *
 *****************************************/
#ifndef module_init
#define module_init(x) int init_module(void) { return x(); }
#define module_exit(x) void cleanup_module(void) { x(); }
#endif

/*** print statements ***/
#define MTX_PRINT(fmt, arg...)   printk("[mtx] "fmt, ##arg)
#define MTX_ERROR(fmt, arg...)   printk("[mtx] *ERROR* "fmt, ##arg)
#define MTX_WARNING(fmt, arg...) printk("[mtx] *WARNING* "fmt, ##arg)
#define MTX_INFO(fmt, arg...)    printk("[mtx] "fmt, ##arg)

#ifdef DEBUG
#define MTX_DEBUG(fmt, arg...)   printk("["__FUNCTION__"(%s,%u)]: "fmt,__FILE__,__LINE__,##arg)
#else
#define MTX_DEBUG(fmt, arg...)
#endif

/* XXX do not use this for string parameters! */
#define MTX_MODULE_PARM_PREFIX  "mtxp_"
#define MTX_MODULE_PARM(section, name)  mtxp_##section##name
#define MTX_MODULE_PARM_DECL(section, name, type, tag, def)   \
    type mtxp_##section##name = def;                          \
    MODULE_PARM(mtxp_##section##name, tag)
    
/*** memory allocation ***/
#ifdef MEMORY_STATS
void*   mtx_mem_stats_alloc(size_t size, const char* file, int line);
void    mtx_mem_stats_free(void* ptr);
#define MTXALLOC(size)      mtx_mem_stats_alloc(size, __FILE__, __LINE__)
#define MTXFREE(ptr, size)  mtx_mem_stats_free(ptr)
#else
void*   mtx_mem_alloc(size_t size);
void    mtx_mem_free(void* ptr, size_t size);
#define MTXALLOC(size)      mtx_mem_alloc(size)
#define MTXFREE(ptr, size)  mtx_mem_free(ptr, size)
#endif
            
/*** region manipulation ***/

    /* Read region specifications from PCI config space */
#define READ_PCI_REGION(dev, reg, bar)                                          do {\
    reg.base  = pci_resource_start(dev->pci_dev, bar);                              \
    reg.size  = pci_resource_end(dev->pci_dev, bar) - reg.base + 1;                 \
    reg.io_flags = pci_resource_flags(dev->pci_dev, bar);                           \
                                                                                    } while(0)

    /* Add region to device memory region list */
#define ADD_REGION_TO_LIST(dev_or_ctx, reg)                                         \
    list_add_tail(&((reg)->list_node), &((dev_or_ctx)->memory_regions));
#define DEL_REGION_FROM_LIST(dev_or_ctx, reg)                                       \
	list_del(&((reg)->list_node))

/* align a value with a alignment of a power of 2 */
#define MTX_ALIGN_POW_2(value, alignment)                                           \
   (((value) + (alignment) - 1) & ~((alignment) - 1))

/*** memory heaps ***/
#ifdef MTX_WITH_MEMORY_HEAPS

    /* Init a group of heaps */
#define INIT_HEAP_GROUP(group)                                                  do {\
        INIT_LIST_HEAD(&(group)->heap_list);                                        \
        (group)->heap_count = 0;                                                    \
        (group)->base = 0xFFFFFFFF;                                                 \
        (group)->size = 0;                                                          } while(0)

    /* Add a heap to a group of heaps */
#define ADD_HEAP_TO_GROUP(group, heap)                                          do {\
    list_add_tail(&((heap)->list_node), &((group)->heap_list));                     \
    (group)->heap_count++;                                                          \
    if ((heap)->base < (group)->base) (group)->base = (heap)->base;                 \
    (group)->size += (heap)->size;                                                  } while(0)

#endif
    
/*** read/write helpers ***/
    
    /* to get address of a register in reg space */
#define REG_ADDR(dev, reg)                                                      \
    (((u8*)(dev)->registers.kaddr) + reg)

    /* 32 bits registers */
#define REG_WRITE(dev, reg, value)                                      do {\
    *(volatile u32 *)REG_ADDR(dev, reg) = value;                           } while(0)
#define REG_READ(dev, reg)                                                 \
    (*(volatile u32 *)REG_ADDR(dev, reg))

    /* 16 bits registers */
#define REG_WRITE_16(dev, reg, value)                                   do {\
    *(volatile u16 *)REG_ADDR(dev, reg) = value;                           } while(0)
#define REG_READ_16(dev, reg)                                              \
    (*(volatile u16 *)REG_ADDR(dev, reg))


/**** Inline methods ****/

/* 
 * (mtx_get_pg_order)
 *
 * return the number of pages required 
 * to hold this size 
 */
#define MTX_MAX_ORDER   9   /* seems to be reasonable, and old-kernel compatible */
static __inline__ int 
mtx_get_pg_order(size_t size) {

    int order;

    for (order = 0; order <= MTX_MAX_ORDER; order++)
        if ((PAGE_SIZE << order) >= size)
            return order;

    return -1;
}

/*
 * (mtx_is_pow_of_2)
 *
 * return if the value is a power of 2 
 */
static __inline__ int
mtx_is_pow_of_2(unsigned long value) {

    int bit_count = 0;
    unsigned long tmpvalue = value;

    while (tmpvalue != 0) {
        bit_count += (tmpvalue & 0x1);
        tmpvalue = tmpvalue >> 1;
    }
  
    /* the value is a pow of 2 if only one of its bits is set to 1 */
    return (bit_count == 1);
}

/* 
 * (mtx_find_region_from_base)
 *
 * find a region in a list of regions by matching
 * its base address
 */
static __inline__ mtx_region_t*
mtx_find_region_from_base(mtx_device_t* dev, unsigned long base) {

    struct list_head* pos;
    mtx_region_t *region;

    list_for_each(pos, &dev->memory_regions) {
        region = list_entry(pos, mtx_region_t, list_node);
        if (region && (region->base == base))
            return region;
    }

    return NULL;
}

/* 
 * (mtx_find_region_from_virt)
 *
 * find a region in a list of regions by matching
 * its virtual address
 */
static __inline__ mtx_region_t*
mtx_find_region_from_virt(mtx_device_t* dev, void* kaddr) {

    struct list_head* pos;
    mtx_region_t *region;

    list_for_each(pos, &dev->memory_regions) {
        region = list_entry(pos, mtx_region_t, list_node);
        if (region && (region->kaddr == kaddr))
            return region;
    }

    return NULL;
}

#ifdef MTX_WITH_MEMORY_HEAPS
/* 
 * (mtx_find_block_from_base)
 *
 * Same as mtx_find_region_from_base, but within a context block list.
 * Block is returned.
 */
static __inline__ mtx_heap_block_t*
mtx_find_block_from_base(struct list_head* block_list, unsigned long base) {

    struct list_head* pos;
    mtx_heap_block_t *block;

    list_for_each(pos, block_list) {
        block = list_entry(pos, mtx_heap_block_t, ctx_list_node);
        if (block && block->region && block->region->base == base)
            return block;
    }

    return NULL;
}
#endif

/*
 * (mtx_split_region)
 *
 * Split region in two at n bytes. Region at left is placed in new_reg
 * and the one at right is kept in orig_reg.
 */
static __inline__ void
mtx_split_region(mtx_region_t *new_reg, mtx_region_t *orig_reg, unsigned int n) {

    /* copy everything (but the list_node) to new region */
    memcpy(((u8*)new_reg) + sizeof(struct list_head),
           ((u8*)orig_reg) + sizeof(struct list_head),
           sizeof(mtx_region_t) - sizeof(struct list_head));
    
    new_reg->size = n;
    
    orig_reg->base    += n;
    orig_reg->busaddr += n;
    orig_reg->kaddr    = (u8*)orig_reg->kaddr + n;
    orig_reg->size    -= n;
}

/*
 * (mtx_merge_regions)
 *
 * Merge two region in one. The old_reg is merged (after or before) the
 * orig_reg. Only orig_reg is modified.
 */
static __inline__ void
mtx_merge_regions(mtx_region_t *old_reg, mtx_region_t *orig_reg, unsigned int before) {
     
    /* modify size of orig region to hold old_reg space */
    orig_reg->size += old_reg->size;

    if (before) {
        
        /* we need to update orig_reg adresses too */ 
        orig_reg->base = old_reg->base;
        orig_reg->kaddr = old_reg->kaddr;
        orig_reg->busaddr = old_reg->busaddr;
    }
}


/******************************************
 *
 * Internal functions prototype
 *
 *****************************************/

/* fops */
int     mtx_open(struct inode *inode, struct file *filp);
int     mtx_release(struct inode *inode, struct file *filp);
int     mtx_ioctl(struct inode *inode, struct file *filp, unsigned int cmd, unsigned long arg);
int     mtx_mmap(struct file *filp, struct vm_area_struct *vma);

/* vm */
struct page* mtx_vm_nopage_stub(struct vm_area_struct*, unsigned long, int);
void    mtx_vm_open_stub(struct vm_area_struct*);
void    mtx_vm_close_stub(struct vm_area_struct*);
struct page* mtx_vm_nopage_agp(struct vm_area_struct*, unsigned long, int);
struct page* mtx_vm_nopage_pci(struct vm_area_struct*, unsigned long, int);
struct page* mtx_vm_nopage_sys(struct vm_area_struct*, unsigned long, int);
int     mtx_vm_map_io(mtx_device_t*, mtx_region_t*, struct vm_area_struct*);
int     mtx_vm_map_agp(mtx_device_t*, mtx_region_t*, struct vm_area_struct*);
int     mtx_vm_map_pci(mtx_device_t*, mtx_region_t*, struct vm_area_struct*);
int     mtx_vm_map_sys(mtx_device_t*, mtx_region_t*, struct vm_area_struct*);

/* devices */
int     mtx_dev_probe(struct pci_dev *pci_dev, const struct pci_device_id *pci_id);
void    mtx_dev_remove(struct pci_dev *pci_dev);
mtx_device_t* mtx_dev_find(unsigned int pci_bus, unsigned int pci_dev);
int     mtx_dev_init(mtx_device_t*);
int     mtx_dev_cleanup(mtx_device_t*);

/* contexts */
mtx_context_t*  mtx_ctx_create(mtx_device_t*);
int             mtx_ctx_delete(mtx_device_t*, mtx_context_t*);

/* memory management */
void    mtx_mem_stats_dump(void);
void    mtx_mem_stats_dump_list(int64_t);
void*   mtx_mem_ioremap(unsigned long offset, unsigned long size);
void    mtx_mem_iounmap(void* ptr);
mtx_region_t* mtx_mem_alloc_region(mtx_device_t*, size_t, mtx_memory_type, mtx_region_t*);
void    mtx_mem_free_region(mtx_device_t*, mtx_region_t*);

/* agp driver */
int     mtx_agp_init(void);
void    mtx_agp_cleanup(void);
int     mtx_agp_acquire(mtx_device_t*);
void    mtx_agp_release(mtx_device_t*);
int     mtx_agp_enable(mtx_device_t*);
void    mtx_agp_disable(mtx_device_t*);
void*   mtx_agp_alloc_page_block(unsigned int);
int     mtx_agp_bind_pages(void*, unsigned long);
int     mtx_agp_test_device(mtx_device_t*);

/* ioctls */
int     mtx_ioctl_hard_reset(mtx_file_t*, unsigned int, unsigned long);
int     mtx_ioctl_get_memory_info(mtx_file_t*, unsigned int, unsigned long);
int     mtx_ioctl_alloc_buffer(mtx_file_t*, unsigned int, unsigned long);
int     mtx_ioctl_free_buffer(mtx_file_t*, unsigned int, unsigned long);
int     mtx_ioctl_get_shared_buffer(mtx_file_t*, unsigned int, unsigned long);
int     mtx_ioctl_release_shared_buffer(mtx_file_t*, unsigned int, unsigned long);
int     mtx_ioctl_busmastering_test(mtx_file_t*, unsigned int, unsigned long);
int     mtx_ioctl_add_context(mtx_file_t*, unsigned int, unsigned long);
int     mtx_ioctl_remove_context(mtx_file_t*, unsigned int, unsigned long);

#ifdef MTX_WITH_MEMORY_HEAPS
/* heaps */
int     mtx_heap_init(mtx_device_t*);
void    mtx_heap_cleanup(mtx_device_t*);
int     mtx_heap_create(mtx_device_t *dev, mtx_memory_type memtype, unsigned long base, unsigned long size, mtx_heap_t **heap_ptr);
void    mtx_heap_destroy(mtx_device_t *dev, mtx_heap_t *heap);
mtx_heap_block_t* mtx_heap_alloc_block_into(mtx_device_t*, mtx_heap_group_t*, unsigned long);
mtx_heap_block_t* mtx_heap_alloc_block_at(mtx_device_t*, mtx_heap_group_t*, unsigned long, unsigned long);
int     mtx_heap_free_block(mtx_device_t*, mtx_heap_block_t*);
#endif

/******************************************
 *
 * Core drivers interface
 *
 *****************************************/
extern mtx_core_driver_t mtx_parhl_driver;

extern mtx_agp_driver_t* agp_drv;
extern int MTX_MODULE_PARM(,agprate);

#else /* __cplusplus */

extern "C" {
/* declare here what can be exported safely to C++ modules */
void*   mtx_mem_alloc(size_t);
void*   mtx_mem_stats_alloc(size_t, const char*, int);
void    mtx_mem_free(void*, size_t);
void    mtx_mem_stats_free(void*);
};

#endif /* __cplusplus */

#endif  /* #ifndef INC_MTXDRV_H */
