 /***************************************************************************\
|*                                                                           *|
|*       Copyright 2002 Matrox Graphics Inc. All rights reserved.            *|
|*                                                                           *|
|*     NOTICE TO THE USER: This source code is the copyrighted work of       *| 
|*     Matrox. Users of this source code are hereby granted a nonexclusive,  *|
|*     royalty-free copyright license to use this code in individual and     *| 
|*     commercial software.                                                  *|
|*                                                                           *|
|*     Any use of this source code must include, in the user documenta-      *|
|*     tion and internal comments to the code, notices to the end user       *|
|*     as follows:                                                           *|
|*                                                                           *|
|*       Copyright 2002 Matrox Graphics Inc. All rights reserved.            *|
|*                                                                           *|
|*     The source code is provided to you AS IS and WITH ALL FAULTS. Matrox  *|
|*     makes no representation and gives no warranty whatsoever, whether     *|
|*     express or implied, and without limitation, with regard to the        *|
|*     quality, safety, contents, performance, merchantability, non-         *|
|*     infringement or suitability for any particular or intended purpose    *|
|*     of this source code. In no event will Matrox be liable for any        *|
|*     direct, indirect, punitive, special, incidental or consequential      *|
|*     damages however they may arise and even if Matrox has been previously *|
|*     advised of the possibility of such damages.                           *|
|*                                                                           *|
 \***************************************************************************/
/***************************************************************************************\

File Name:      mtx_ctx.c

Description:    MTX context manipulation. A context is a representation of each
                reference to a device. 

References:     None.

Author:         Karl Lessard    <klessard@matrox.com>

    Copyright (c) 2001, Matrox Graphics Inc.
    All Rights Reserved.

\***************************************************************************************/

#define __NO_VERSION__
#include "mtx_drv.h"


/***************************************************************************************
 * Function:       mtx_ctx_register
 *
 * Description:    Register context for use of this device. 
 *
 * Parameters:     dev           MTX device to init
 *                 ctx           MTX context to this device
 *  
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:       None.
 */
static int mtx_ctx_register(mtx_device_t *dev, mtx_context_t *ctx)
{
    int bit;

    if (!ctx || (ctx->unique_id != -1)) return -EINVAL;
    
    /* Look for a free bit in our context bitmap */
    down(&dev->semaphore);
    bit = find_first_zero_bit(dev->context_bitmap, MAX_CONTEXTS_PER_DEVICE);
    
    if (bit >= MAX_CONTEXTS_PER_DEVICE) {
        up(&dev->semaphore);
        return -EAGAIN;
    }

    /* Set this bit as used */
    set_bit(bit, dev->context_bitmap);
    up(&dev->semaphore);

    ctx->unique_id = bit;

    return 0;
}

/***************************************************************************************
 * Function:       mtx_ctx_unregister
 *
 * Description:    Unregister context from use of this device. 
 * 
 * Parameters:     dev           MTX device to init
 *                 ctx           MTX context to this device
 *  
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:       None.
 */
static int mtx_ctx_unregister(mtx_device_t *dev, mtx_context_t *ctx)
{
    if (!ctx || (ctx->unique_id >= MAX_CONTEXTS_PER_DEVICE)) return -EINVAL;
    
    /* Free bit in our context bitmap */
    down(&dev->semaphore);
    clear_bit(ctx->unique_id, dev->context_bitmap);
    up(&dev->semaphore);

    ctx->unique_id = -1;

    return 0;
}

/***************************************************************************************
 * Function:       mtx_ctx_create
 *
 * Description:    Create a new context, and register it to the device. 
 *
 * Parameters:     dev           MTX device to init
 *
 * Return Value:   A context for this device, NULL if failed.
 *
 * Comments:       None.
 */
mtx_context_t* mtx_ctx_create(mtx_device_t *dev)
{
    mtx_context_t *ctx;
    int ret;

    /* Contexts are allocated dynamically here */
    ctx = MTXALLOC(sizeof(mtx_context_t));
    if (!ctx) 
        return NULL;
    
    ctx->unique_id = -1;

    /* Register context to device */
    if ((ret = mtx_ctx_register(dev, ctx)) < 0) {
        return NULL;
    }

    /* Init region list */
    INIT_LIST_HEAD(&ctx->region_list);
  #ifdef MTX_WITH_MEMORY_HEAPS
    INIT_LIST_HEAD(&ctx->block_list);
  #endif

    if (dev->core && dev->core->open) {

        int ret;

        /* init context for core driver */
        if ((ret = dev->core->open(dev, ctx)) < 0) {

            MTX_ERROR("Fail to open new context from core driver\n");
            mtx_ctx_unregister(dev, ctx);
            return NULL;
        }
    }
    
    return ctx;
}

/***************************************************************************************
 * Function:       mtx_ctx_delete
 *
 * Description:    Delete a context, and unregister it from the device. 
 *
 * Parameters:     dev           MTX device to init
 *                 ctx           Context to delete
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:       None.
 */
int mtx_ctx_delete(mtx_device_t *dev, mtx_context_t *ctx)
{
    struct list_head *pos, *tmp;
    mtx_region_t *region = NULL;
  #ifdef MTX_WITH_MEMORY_HEAPS
    mtx_heap_block_t *block = NULL;
  #endif
    int ret;

    /* remove context from core driver */
    if (dev->core && dev->core->close)
        dev->core->close(dev, ctx);

    /* Free ressources private to this context */
    list_for_each_safe(pos, tmp, &ctx->region_list) {

        region = list_entry(pos, mtx_region_t, ctx_list_node);
        if (region) {
            mtx_mem_free_region(dev, region);
        }
    }
  #ifdef MTX_WITH_MEMORY_HEAPS
    list_for_each_safe(pos, tmp, &ctx->block_list) {

        block = list_entry(pos, mtx_heap_block_t, ctx_list_node);
        if (block) {
            mtx_heap_free_block(dev, block);
        }
    }

    /* Empty block list */
    INIT_LIST_HEAD(&ctx->block_list);
  #endif
    
    /* Empty region list */
    INIT_LIST_HEAD(&ctx->region_list);
    
    /* Unregister context from device */
    ret = mtx_ctx_unregister(dev, ctx);
    if (ret < 0) {
        return ret;
    }

    MTXFREE(ctx, sizeof(mtx_context_t));

    return 0;
}
