/***************************************************************************************\

Module Name:    global.c

Description:    None.

References:     None.

    Copyright (c) 2002, Matrox Graphics Inc.
    All Rights Reserved.

\***************************************************************************************/

// --------------------------------------------------------------------------------------
//                        I N C L U D E S   A N D   U S I N G S
// --------------------------------------------------------------------------------------

#include "mtxc++.h"
#include "mtx_drv.h"

// --------------------------------------------------------------------------------------
//                        C O N S T A N T S   A N D   T Y P E S
// --------------------------------------------------------------------------------------

typedef void (*PFN)(void);

typedef struct mtx_atexit_block {

    struct mtx_atexit_block *next;  // Next block in list
    PFN pfn;                        // Function registered
    
} mtx_atexit_block_t;

// --------------------------------------------------------------------------------------
//                 G L O B A L   V A R I A B L E   R E F E R E N C E S
// --------------------------------------------------------------------------------------

// --------------------------------------------------------------------------------------
//               G L O B A L   F U N C T I O N   D E C L A R A T I O N S
// --------------------------------------------------------------------------------------

// --------------------------------------------------------------------------------------
//     I N L I N E S,  M A C R O S,  A N D   T E M P L A T E   D E F I N I T I O N S
// --------------------------------------------------------------------------------------

// The problem:
//
//   Normally, when loading an ELF file as an executable or a shared library, the loader
//   checks for the sections .ctors and .dtors which contains all pointers to global
//   constructors/desctructors respectively. Each function pointers included in those
//   sections is called before executing the data code. 
//
//   As Linux kernel do not support C++ modules, the module loader do not check those 
//   sections, and therefore global constructors/descructors are not executed on module
//   insmod/rmmod.
//
// The solution:
//
//   We had to find a way to retrieve pointers in .ctors and .dtors sections. To do that,
//   we insert a global variable __[CD]TORS_BEGIN__ at the beginning of each sections, and
//   another __[CD]TORS_END__ at the end. To ensure that the BEGIN/END variables are really
//   at the begin/end of the module sections, this file is compiled twice, giving 2 object
//   files (one declaring the begins, and one declaring the ends). Those files MUST be the
//   first/last file to be linked when producing the kernel module.
//
//   We then check for function pointers included between a BEGIN/END pair, and can them
//   one after the other.
//
//   LD seems to mess up .ctors/.dtors sections sometimes, so we had to find a workaround for
//   this to ensure the position of the BEGIN/END pointers. I've found that patch somewhere
//   on the net.


// *** Start of code compiled on first pass
#ifdef __GLOBAL_BEGIN__

asm(".section .ctors,\"aw\",@progbits\n"
    ".globl __CTORS_BEGIN__\n"
    "__CTORS_BEGIN__:\n"
    ".previous");
// XXX ld bug
asm(".section .ctors0,\"aw\",@progbits\n"
    ".globl __CTORS_BEGIN_0__\n"
    "__CTORS_BEGIN_0__:\n"
    ".previous");

asm(".section .dtors,\"aw\",@progbits\n"
    ".globl __DTORS_BEGIN__\n"
    "__DTORS_BEGIN__:\n"
    ".previous");
// XXX ld bug
asm(".section .dtors0,\"aw\",@progbits\n"
    ".globl __DTORS_BEGIN_0__\n"
    "__DTORS_BEGIN_0__:\n"
    ".previous");

#endif 
// *** End of code for first pass


// *** Start of code for second pass
#ifdef __GLOBAL_END__

static mtx_atexit_block_t* mtx_atexit_block_list = NULL;    // List of atexit blocks

extern PFN __CTORS_BEGIN__[];
extern PFN __DTORS_BEGIN__[];
extern PFN __CTORS_END__[];
extern PFN __DTORS_END__[];

// XXX ld bug
extern PFN __CTORS_BEGIN_0__[];
extern PFN __DTORS_BEGIN_0__[];
extern PFN __CTORS_END_0__[];
extern PFN __DTORS_END_0__[];

asm(".section .ctors\n"
    ".globl __CTORS_END__\n"
    "__CTORS_END__:\n"
    ".previous");
// XXX ld bug
asm(".section .ctors0\n"
    ".globl __CTORS_END_0__\n"
    "__CTORS_END_0__:\n"
    ".previous");

asm(".section .dtors\n"
    ".globl __DTORS_END__\n"
    "__DTORS_END__:\n"
    ".previous");
// XXX ld bug
asm(".section .dtors0\n"
    ".globl __DTORS_END_0__\n"
    "__DTORS_END_0__:\n"
    ".previous");

/****************************************************************************************\

Function:       mtx_call_global_ctors

Description:    Call all global constructors collected by the compiler.

Parameters:     None.

Return Value:   void* 

Comments:       None.

\****************************************************************************************/
void __cdecl mtx_call_global_ctors()
{   
    PFN* pfnctor = __CTORS_BEGIN__;
    int count = __CTORS_END__ - __CTORS_BEGIN__, i;

    // XXX ld bug workaround
    if (((__CTORS_BEGIN__ > __CTORS_END__) &&
         (__CTORS_BEGIN_0__ > __CTORS_END_0__))
        &&
        ((__CTORS_END__ == __CTORS_END_0__) &&
         (__CTORS_END__ == __DTORS_END_0__)))
    {
        MTX_DEBUG("Using ld bug workaround\n");
        count = __CTORS_BEGIN_0__ - __CTORS_BEGIN__;
    }
    else if ((__CTORS_BEGIN__ > __CTORS_END__) ||
             (__CTORS_BEGIN_0__ > __CTORS_END_0__))
    {
        MTX_ERROR("No workaround for ld bug, failed to call global constructors\n");
        return;
    }
    
    // Call each of the global constructor
    for (i = 0; (i < count) && ((*pfnctor) != NULL); i++, pfnctor++) {
        
        MTX_DEBUG("Calling global constructor at 0x%08lx\n", (unsigned long)*pfnctor);
        (*pfnctor)();
    }
}

/****************************************************************************************\

Function:       mtx_call_global_dtors

Description:    Call all global descructors collected by the compiler, or registered
                via a atexit call.

Parameters:     sz  :   size to allocate

Return Value:   void* 

Comments:       None.

\****************************************************************************************/
void __cdecl mtx_call_global_dtors()
{
    PFN* pfndtor = __DTORS_END__ - 1; // Skip last pointer
    int count = __DTORS_END__ - __DTORS_BEGIN__, i;
    mtx_atexit_block_t *atexit_block = mtx_atexit_block_list;

    // XXX ld bug workaround
    if (((__DTORS_BEGIN__ > __DTORS_END__) &&
         (__DTORS_BEGIN_0__ > __DTORS_END_0__))
        &&
        ((__DTORS_END__ == __DTORS_END_0__) &&
         (__CTORS_END__ == __DTORS_END_0__)))
    {
        MTX_DEBUG("Using ld bug workaround\n");
        count = __DTORS_BEGIN_0__ - __DTORS_BEGIN__;

        pfndtor = __DTORS_BEGIN_0__ - 1;
    } 
    else if ((__DTORS_BEGIN__ > __DTORS_END__) ||
             (__DTORS_BEGIN_0__ > __DTORS_END_0__))
    {
        MTX_ERROR("No workaround for ld bug, failed to call global destructors\n");
        return;
    }

    // Call each of the global destructor
    for (i = 0; (i < count) && ((*pfndtor) != NULL); i++, pfndtor--) {
        
        MTX_DEBUG("Calling global destructor at 0x%08lx\n", (unsigned long)*pfndtor);
        (*pfndtor)();
    }

    // Call each of the destructor registered by atexit (local static objects)
    while (atexit_block != NULL) {

        mtx_atexit_block_t* atexit_tmp = atexit_block;

        MTX_DEBUG("Calling registered destructor at 0x%08lx\n", (unsigned long)atexit_block->pfn);
        (atexit_block->pfn)();

        // Free record
        atexit_block = atexit_block->next;
        MTXFREE((void*)atexit_tmp, sizeof(mtx_atexit_block_t));
    }

    mtx_atexit_block_list = NULL;
}

/****************************************************************************************\

Function:       atexit

Description:    Register a function to be called on call_global_dtors.

Parameters:     pfn     Function to register

Return Value:   void* 

Comments:       None.

\****************************************************************************************/
void atexit(PFN pfn)
{
    mtx_atexit_block_t **atexit_block_ptr = &mtx_atexit_block_list;

    // Check for last atexit block in the list
    while ((*atexit_block_ptr) != NULL) {

        atexit_block_ptr = &(*atexit_block_ptr)->next;        
    }

    // Allocate record
    (*atexit_block_ptr) = (mtx_atexit_block_t*) MTXALLOC(sizeof(mtx_atexit_block_t));        
    if (!(*atexit_block_ptr)) {
         
        MTX_ERROR("Failed to register function in atexit, out of memory\n");
        return;
    }

    // Set record data
    (*atexit_block_ptr)->next = NULL;
    (*atexit_block_ptr)->pfn = pfn;
}

#endif 
// *** End of code for second pass
