 /***************************************************************************\
|*                                                                           *|
|*       Copyright 2002 Matrox Graphics Inc. All rights reserved.            *|
|*                                                                           *|
|*     NOTICE TO THE USER: This source code is the copyrighted work of       *| 
|*     Matrox. Users of this source code are hereby granted a nonexclusive,  *|
|*     royalty-free copyright license to use this code in individual and     *| 
|*     commercial software.                                                  *|
|*                                                                           *|
|*     Any use of this source code must include, in the user documenta-      *|
|*     tion and internal comments to the code, notices to the end user       *|
|*     as follows:                                                           *|
|*                                                                           *|
|*       Copyright 2002 Matrox Graphics Inc. All rights reserved.            *|
|*                                                                           *|
|*     The source code is provided to you AS IS and WITH ALL FAULTS. Matrox  *|
|*     makes no representation and gives no warranty whatsoever, whether     *|
|*     express or implied, and without limitation, with regard to the        *|
|*     quality, safety, contents, performance, merchantability, non-         *|
|*     infringement or suitability for any particular or intended purpose    *|
|*     of this source code. In no event will Matrox be liable for any        *|
|*     direct, indirect, punitive, special, incidental or consequential      *|
|*     damages however they may arise and even if Matrox has been previously *|
|*     advised of the possibility of such damages.                           *|
|*                                                                           *|
 \***************************************************************************/
/***************************************************************************************\

File Name:      MtxParhlSurf.c

Description:    MTX Parhelia surfaces allocation management.

References:     None.

Author:         Karl Lessard    <klessard@matrox.com>

    Copyright (c) 2002, Matrox Graphics Inc.
    All Rights Reserved.

\***************************************************************************************/

#define __NO_VERSION__
#include "MtxParhlDrv.h"


/***************************************************************************************
 * Function:       MtxParhlSurfGeneralAlloc
 *
 * Description:    General allocation strategy used by many memory locations.
 *
 * Parameters:     dev              MTX device
 *                 poHeapGroup      Heap group for this memory location
 *                 ulSize           Size requested for new block
 *
 * Return Value:   A heap block available for a surface, NULL if failed.
 *
 * Comments:       None.
 */
static mtx_heap_block_t*
MtxParhlSurfGeneralAlloc(mtx_device_t *dev, mtx_heap_group_t *poHeapGroup, ULONG ulSize)
{
    /* TODO: Handle 'no room' with priorities */
    
    return mtx_heap_alloc_block_into(dev, poHeapGroup, ulSize); 
}

/***************************************************************************************
 * Function:       MtxParhlSurfAlternateAlloc
 *
 * Description:    General allocation strategy used by LOCAL and LOCAL COMMON surfaces.
 *
 * Parameters:     dev              MTX device
 *                 eMemType         Type of surface memory
 *                 eMemCtrl         First memory location to try (INVALID if no preferences)
 *                 ulSize           Size requested for new block
 *
 * Return Value:   A heap block available for a surface, NULL if failed.
 *
 * Comments:       None.
 */
static mtx_heap_block_t*
MtxParhlSurfAlternateAlloc(mtx_device_t *dev, 
                           MmSurfaceMemType eMemType, 
                           MmMemoryController eMemCtrl, 
                           ULONG ulSize)
{
    MtxParhlDevice *poParhlDev = (MtxParhlDevice*)dev->core_data;
    mtx_heap_block_t* poBlock = NULL;
    MmLocalMemType eMemLocAB;
    MmLocalMemType eMemLocCD;
    MmMemoryController *peMemCtrlNext;
    int iCount = 0;
    
    switch (eMemType) {
        
        case MMSURF_MEM_LOCAL:
            
            eMemLocAB = MM_LOCAL_MEMTYPE_AB;
            eMemLocCD = MM_LOCAL_MEMTYPE_CD;
            peMemCtrlNext = &poParhlDev->eNextAllocLocal;
            break;

        case MMSURF_MEM_LOCAL_COMMON:
            
            eMemLocAB = MM_LOCAL_MEMTYPE_AB_COMMON;
            eMemLocCD = MM_LOCAL_MEMTYPE_CD_COMMON;
            peMemCtrlNext = &poParhlDev->eNextAllocLocalCommon;
            break;

        default:
            return NULL;   
    }

    /* Override next controller with argument value */
    if (eMemCtrl != MM_MEMCTRL_INVALID)
        *peMemCtrlNext = eMemCtrl;
   
    while ((iCount < 2) && (poBlock == NULL))
    {
        if (*peMemCtrlNext == MM_MEMCTRL_AB) {

            poBlock = MtxParhlSurfGeneralAlloc(dev, &poParhlDev->aoLocalSurfaceHeaps[eMemLocAB], ulSize);
            *peMemCtrlNext = MM_MEMCTRL_CD;

        } else { /* MM_MEMCTRL_CD */

            poBlock = MtxParhlSurfGeneralAlloc(dev, &poParhlDev->aoLocalSurfaceHeaps[eMemLocCD], ulSize);
            *peMemCtrlNext = MM_MEMCTRL_AB;
        }

        iCount++;
    }
    
    return poBlock; 
}
    
/***************************************************************************************
 * Function:       MtxParhlSurfAllocBlock
 *
 * Description:    Alloc a block for a surface depending on the specified memory
 *                 and data types.
 *
 * Parameters:     dev              MTX device
 *                 eMemType         Surface memory type
 *                 eDataType        Surface data type
 *                 ulSize           Size requested for new block
 *
 * Return Value:   A heap block available for this surface, NULL if failed.
 *
 * Comments:       None.
 */
mtx_heap_block_t* 
MtxParhlSurfAllocBlock(mtx_device_t *dev, 
                       MmSurfaceMemType eMemType, 
                       MmSurfaceDataType eDataType, 
                       ULONG ulSize)
{
    MtxParhlDevice *poParhlDev = (MtxParhlDevice*)dev->core_data;
    mtx_heap_block_t *poBlock = NULL;
    
    /* Get surface allocation strategy from types */
    switch (eMemType) {
        
        case MMSURF_MEM_GEN_AGP:
            poBlock = MtxParhlSurfGeneralAlloc(dev, &dev->heap_groups[MTX_MEMTYPE_AGP], ulSize);
            break;

        case MMSURF_MEM_PCI_LOCKED:
            poBlock = MtxParhlSurfGeneralAlloc(dev, &dev->heap_groups[MTX_MEMTYPE_PCI_LOCKED], ulSize);
            break;

        case MMSURF_MEM_LOCAL_LIN:
            poBlock = MtxParhlSurfGeneralAlloc(dev, 
                            &poParhlDev->aoLocalSurfaceHeaps[MM_LOCAL_MEMTYPE_LINEAR], ulSize);
            break;

        case MMSURF_MEM_LOCAL_COMMON:
        case MMSURF_MEM_LOCAL:
            
            switch (eDataType) {
                
                case MMSURF_DATA_DISPLAY_PRIMARY:
                    poBlock = MtxParhlSurfGeneralAlloc(dev, 
                                    &poParhlDev->aoLocalSurfaceHeaps[MM_LOCAL_MEMTYPE_AB], ulSize);
                    break;

                case MMSURF_DATA_2D_DEVICE_BITMAP:
                    poBlock = MtxParhlSurfGeneralAlloc(dev, 
                                    &poParhlDev->aoLocalSurfaceHeaps[MM_LOCAL_MEMTYPE_AB_COMMON], ulSize);
                    break;

                case MMSURF_DATA_2D_SURFACE:
                    poBlock = MtxParhlSurfAlternateAlloc(dev, eMemType, MM_MEMCTRL_CD, ulSize);
                    break;
                    
                case MMSURF_DATA_2D_MONO_SURFACE:
                    poBlock = MtxParhlSurfGeneralAlloc(dev, 
                                    &poParhlDev->aoLocalSurfaceHeaps[MM_LOCAL_MEMTYPE_CD], ulSize);
                    break;
                    
                case MMSURF_DATA_2D_CURSOR:
                    poBlock = MtxParhlSurfGeneralAlloc(dev, 
                                    &poParhlDev->aoLocalSurfaceHeaps[MM_LOCAL_MEMTYPE_AB_COMMON], ulSize);
                    break;

                default:
                    MTX_ERROR("PARHL: Unsupported surface data type (%u)\n", eDataType);
                    return NULL;
            }
            break;

        default:
            MTX_ERROR("PARHL: Unsupported surface memory type (%u)\n", eMemType);
            return NULL;
    }

    return poBlock;
}
    
/***************************************************************************************
 * Function:       MtxParhlSurfInitLocalHeaps
 *
 * Description:    Init parhelia local surface heaps.
 *
 * Parameters:     dev          MTX device
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:       None.
 */
static int MtxParhlSurfInitLocalHeaps(mtx_device_t *dev)
{
    MtxParhlDevice *poParhlDev = (MtxParhlDevice*)dev->core_data;
    mtx_heap_t *poHeap;
    ULONG Mb = 1024L * 1024L;
    ULONG ulSize, ulLinearSize, ulCommonSize;
    ULONG ulStart;
    int ret;
    
    /* FIXME only single head supported for now */
    
    /* Diagram for local memory in single head configuration:
     *
     *  0                       16                      31.5 32
     * ------------------------------------------------------
     * A|                                                 |*|
     * B|                                                 | |
     * ------------------------------------------------------
     * ------------------------------------------------------
     * C|LIN|                                             |*|   *= common heaps
     * D|   |                                             | |
     * ------------------------------------------------------
     *  0   2                   16                      31.5 32
     *
     *  | ----- Bit 24 : 0 ----- | ------ Bit 24 : 1 ------ |
     *
     *  Local heaps just manages those offsets in video memory,
     *  and not real physical addresses. 
     */
    
    ulLinearSize = 2 * Mb;
    ulCommonSize = (Mb / 2);
    
    /* Create local linear heap */
    ulStart = MM_LOCAL_CONTROLLER_LIMIT;
    ret = mtx_heap_create(dev, MTX_MEMTYPE_LOCAL, ulStart, ulLinearSize, &poHeap);
    if (ret < 0) {
        MTX_ERROR("PARHL: Fail to initialize linear local heap\n");
        return ret;
    }
    
    ADD_HEAP_TO_GROUP(&poParhlDev->aoLocalSurfaceHeaps[MM_LOCAL_MEMTYPE_LINEAR], poHeap);
    
    /* Create AB common local poHeap */
    ulStart = 0 + poParhlDev->poVideoInfo->m_dwSizeAB - ulCommonSize;
    ret = mtx_heap_create(dev, MTX_MEMTYPE_LOCAL, ulStart, ulCommonSize, &poHeap);
    if (ret < 0) {
        MTX_ERROR("PARHL: Fail to initialize AB local common poHeap\n");
        return ret;
    }
    
    ADD_HEAP_TO_GROUP(&poParhlDev->aoLocalSurfaceHeaps[MM_LOCAL_MEMTYPE_AB_COMMON], poHeap);
                                
    /* Create CD common local poHeap */
    ulStart = MM_LOCAL_CONTROLLER_LIMIT + poParhlDev->poVideoInfo->m_dwSizeCD - ulCommonSize;
    ret = mtx_heap_create(dev, MTX_MEMTYPE_LOCAL, ulStart, ulCommonSize, &poHeap);
    if (ret < 0) {
        MTX_ERROR("PARHL: Fail to initialize CD local common poHeap\n");
        return ret;
    }
    
    ADD_HEAP_TO_GROUP(&poParhlDev->aoLocalSurfaceHeaps[MM_LOCAL_MEMTYPE_CD_COMMON], poHeap);

    /* Create AB local poHeap */
    ulStart = 0;
    ulSize = poParhlDev->poVideoInfo->m_dwSizeAB - ulCommonSize;
    ret = mtx_heap_create(dev, MTX_MEMTYPE_LOCAL, ulStart, ulSize, &poHeap);
    if (ret < 0) {
        MTX_ERROR("PARHL: Fail to initialize AB local poHeap\n");
        return ret;
    }
    
    ADD_HEAP_TO_GROUP(&poParhlDev->aoLocalSurfaceHeaps[MM_LOCAL_MEMTYPE_AB], poHeap);

    /* Create CD local poHeap */
    ulStart = MM_LOCAL_CONTROLLER_LIMIT + ulLinearSize;
    ulSize = poParhlDev->poVideoInfo->m_dwSizeCD - ulCommonSize - ulLinearSize;
    ret = mtx_heap_create(dev, MTX_MEMTYPE_LOCAL, ulStart, ulSize, &poHeap);
    if (ret < 0) {
        MTX_ERROR("PARHL: Fail to initialize CD local poHeap\n");
        return ret;
    }
    
    ADD_HEAP_TO_GROUP(&poParhlDev->aoLocalSurfaceHeaps[MM_LOCAL_MEMTYPE_CD], poHeap);

    return 0;
}

/***************************************************************************************
 * Function:       MtxParhlSurfInit
 *
 * Description:    Init parhelia surface manager.
 *
 * Parameters:     dev          MTX device
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:       None.
 */
int MtxParhlSurfInit(mtx_device_t* dev)
{
    MtxParhlDevice *poParhlDev = (MtxParhlDevice*)dev->core_data;
    int ret, iCount;

    /* Init heap groups */
    for (iCount = 0; iCount < MM_LOCAL_MEMTYPE_COUNT; iCount++) {

        mtx_heap_group_t *poHeapGroup = &poParhlDev->aoLocalSurfaceHeaps[iCount];
        
        INIT_LIST_HEAD(&poHeapGroup->heap_list);
        poHeapGroup->heap_count = 0;
        poHeapGroup->base = 0xFFFFFFFF;
        poHeapGroup->size = 0;
    }
    
    /* Init local memory heaps */
    ret = MtxParhlSurfInitLocalHeaps(dev);
    if (ret < 0) return ret;

    poParhlDev->eNextAllocLocal = MM_MEMCTRL_AB;
    poParhlDev->eNextAllocLocalCommon = MM_MEMCTRL_AB;
    
    return 0;
}

/***************************************************************************************
 * Function:       MtxParhlSurfCleanup
 *
 * Description:    Cleanup parhelia surface manager.
 *
 * Parameters:     dev          MTX device
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:       None.
 */
void MtxParhlSurfCleanup(mtx_device_t* dev)
{
    MtxParhlDevice *poParhlDev = (MtxParhlDevice*)dev->core_data;
    struct list_head *pos, *tmppos;
    int iCount;

    /* cleanup surface heaps */
    for (iCount = 0; iCount < MM_LOCAL_MEMTYPE_COUNT; iCount++) {

        mtx_heap_group_t *poHeapGroup = &poParhlDev->aoLocalSurfaceHeaps[iCount];
        
        /* cleanup each heaps in this group */
        list_for_each_safe(pos, tmppos, &poHeapGroup->heap_list) {

            mtx_heap_t* poHeap =  list_entry(pos, mtx_heap_t, list_node);

            if (poHeap) {

                /* destroy heap and decrement heap count in this group */
                mtx_heap_destroy(dev, poHeap);
                poHeapGroup->heap_count--;
            }
        }
        
        if (poHeapGroup->heap_count != 0) {
            MTX_ERROR("Not all heaps were cleaned for memory group\n");
        }
        
        /* Empty heap list */
        INIT_LIST_HEAD(&poHeapGroup->heap_list);
    }
}
