 /***************************************************************************\
|*                                                                           *|
|*       Copyright 2002 Matrox Graphics Inc. All rights reserved.            *|
|*                                                                           *|
|*     NOTICE TO THE USER: This source code is the copyrighted work of       *| 
|*     Matrox. Users of this source code are hereby granted a nonexclusive,  *|
|*     royalty-free copyright license to use this code in individual and     *| 
|*     commercial software.                                                  *|
|*                                                                           *|
|*     Any use of this source code must include, in the user documenta-      *|
|*     tion and internal comments to the code, notices to the end user       *|
|*     as follows:                                                           *|
|*                                                                           *|
|*       Copyright 2002 Matrox Graphics Inc. All rights reserved.            *|
|*                                                                           *|
|*     The source code is provided to you AS IS and WITH ALL FAULTS. Matrox  *|
|*     makes no representation and gives no warranty whatsoever, whether     *|
|*     express or implied, and without limitation, with regard to the        *|
|*     quality, safety, contents, performance, merchantability, non-         *|
|*     infringement or suitability for any particular or intended purpose    *|
|*     of this source code. In no event will Matrox be liable for any        *|
|*     direct, indirect, punitive, special, incidental or consequential      *|
|*     damages however they may arise and even if Matrox has been previously *|
|*     advised of the possibility of such damages.                           *|
|*                                                                           *|
 \***************************************************************************/
/***************************************************************************************\

File Name:      MtxParhlIoctl.c

Description:    Parhelia specific IO controls implementation for MTX driver.

References:     None.

Author:         Karl Lessard    <klessard@matrox.com>

    Copyright (c) 2002, Matrox Graphics Inc.
    All Rights Reserved.

\***************************************************************************************/

#define __NO_VERSION__
#include "MtxParhlDrv.h"

/***************************************************************************************
 * Function:       mtx_parhl_ioctl_alloc_block
 *
 * Description:    Allocate a new memory 'block' in function of the desired options
 *                 selected, as the memory type, size, and further more. The description
 *                 of the block is returned in ioctl argument.
 *
 * Parameters:     priv                MTX private file data
 *                 cmd                 ioctl command
 *                 arg                 ioctl argument
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:
 */
int 
MtxParhlIoctlAllocBlock(mtx_file_t *priv, unsigned int cmd, unsigned long arg)
{
    mtx_device_t *dev = priv->dev;
    MtxParhlIoctlBufferAllocBlock oBlockInfo;
    mtx_heap_block_t *poBlock;

    if (!dev || !priv->ctx) return -EINVAL;

    if (copy_from_user(&oBlockInfo, (void*)arg, sizeof(oBlockInfo)))
        return -EFAULT;

    /* Allocate new block for context */
    switch (oBlockInfo.ulType & MTX_PARHL_BLOCKTYPE_TYPE_MASK) {

        case MTX_PARHL_BLOCKTYPE_APERTURE:

            if (oBlockInfo.Options.Apt.ePriority == APT_PRIO_DISCARDABLE) {

                /* No support for discardable apertures from user space */
                MTX_DEBUG("WARNING: Discardable apertures (size: %lu) not supported yet, "
                            "permanent allocation scheme will used\n", oBlockInfo.ulSize);

                oBlockInfo.Options.Apt.ePriority = APT_PRIO_PERMANENT;
            }
            
            /* Get a memory space block */
            poBlock = MtxParhlAptLockAddressSpace(dev, 
                                                  oBlockInfo.Options.Apt.ulStartAddress, 
                                                  oBlockInfo.ulSize);
            break;

        case MTX_PARHL_BLOCKTYPE_SURFACE:
            
            if (oBlockInfo.Options.Surf.ePriority == MMSURF_PRIO_DISCARDABLE) {

                /* No support for discardable surfaces from user space */
                MTX_DEBUG("WARNING: Discardable surfaces (size: %lu, data: %u) "
                            "not supported yet, permanent allocation scheme will be used\n",
                            oBlockInfo.ulSize, oBlockInfo.Options.Surf.eDataType);
                
                oBlockInfo.Options.Surf.ePriority = MMSURF_PRIO_PERMANENT;
            }
        
            /* Allocate our memory block depending on surface memory and data types */
            poBlock = MtxParhlSurfAllocBlock(dev,
                                             oBlockInfo.Options.Surf.eMemType,
                                             oBlockInfo.Options.Surf.eDataType, 
                                             oBlockInfo.ulSize);
            break;

        default:
            return -EINVAL;
    }
    
    if (!poBlock || !poBlock->region) {

        MTX_DEBUG("Fail to allocate Parhelia memory block to process %u (type: %lu)\n",
                   current->pid, oBlockInfo.ulType);
        return -ENOMEM;
    }

    if ((oBlockInfo.ulType & MTX_PARHL_BLOCKTYPE_SHARED) && 
        !(poBlock->region->flags & MTX_REGION_SHARED))
    {
        /* Set block as shareable, and increment its usage count */
        poBlock->region->flags |= MTX_REGION_SHARED;
        poBlock->region->map_perms = MTX_MAP_AUTH;
        atomic_inc(&poBlock->region->ref_count);
    }

    list_add_tail(&poBlock->ctx_list_node, &priv->ctx->block_list);

    /* Copy block information to user */
    oBlockInfo.ulBase   = poBlock->region->busaddr;
    oBlockInfo.ulSize   = poBlock->region->size;
    oBlockInfo.ulOffset = poBlock->offset;

    if (copy_to_user((void*)arg, &oBlockInfo, sizeof(oBlockInfo)))
        return -EFAULT;

    return 0;
}

/***************************************************************************************
 * Function:       MtxParhlIoctlFreeBlock
 *
 * Description:    Free a block of memory previously allocated for user space by 
 *                 parhelia core driver.
 *
 * Parameters:     priv                MTX private file data
 *                 cmd                 ioctl command
 *                 arg                 ioctl argument
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:
 */
int 
MtxParhlIoctlFreeBlock(mtx_file_t *priv, unsigned int cmd, unsigned long arg)
{
    mtx_device_t *dev = priv->dev;
    MtxParhlIoctlBufferAllocBlock oBlockInfo;
    mtx_heap_block_t *poBlock = NULL;
    int ret;

    if (!dev || !priv->ctx) return -EINVAL;

    if (copy_from_user(&oBlockInfo, (void*)arg, sizeof(oBlockInfo)))
        return -EFAULT;
    
    poBlock = mtx_find_block_from_base(&priv->ctx->block_list, oBlockInfo.ulBase);
    if (!poBlock)
        return -EINVAL;

    /* perform some additionnal checks */
    if ((poBlock->region->size != oBlockInfo.ulSize) ||
        (poBlock->offset != oBlockInfo.ulOffset)) 
    {
        MTX_ERROR("Try to free an invalid memory block\n"
                  "\t\tour block: (offset: %lu, size: %lu)\n"
                  "\t\theap block: (offset: %lu, size: %lu)\n",
                   oBlockInfo.ulOffset, oBlockInfo.ulSize, 
                   poBlock->offset, poBlock->region->size);

        return -EINVAL;
    }
    
    list_del(&poBlock->ctx_list_node);
    
    switch (oBlockInfo.ulType & MTX_PARHL_BLOCKTYPE_TYPE_MASK) {
        
        case MTX_PARHL_BLOCKTYPE_APERTURE:

            /* Retrive heap block from local memory space heap and free it */
            if ((ret = MtxParhlAptUnlockAddressSpace(dev, poBlock)) < 0) {
                MTX_ERROR("Failed to release aperture block (base: 0x%08lx, size: %lu)\n",
                           poBlock->region->base, poBlock->region->size);
                return ret;
            }

            break;

        case MTX_PARHL_BLOCKTYPE_SURFACE:

            /* Retrieve heap block from block and surface descriptions */
            ret = mtx_heap_free_block(dev, poBlock);
            if (ret < 0) {
                MTX_ERROR("Failed to release surface block (base: 0x%08lx, size: %lu)\n",
                           poBlock->region->base, poBlock->region->size);
                return ret;
            }

            break;

        default:
            return -EINVAL;
    }

    return 0;
}

/***************************************************************************************
 * Function:       MtxParhlIoctlGetLocalMemoryLayout
 *
 * Description:    Return information about layout of local memory as setup
 *                 by our memory manager.
 *
 * Parameters:     priv                MTX private file data
 *                 cmd                 ioctl command
 *                 arg                 ioctl argument
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:
 */
int 
MtxParhlIoctlGetLocalMemoryLayout(mtx_file_t *priv, unsigned int cmd, unsigned long arg)
{
    mtx_device_t *dev = priv->dev;
    MtxParhlIoctlBufferLocalMemoryLayout oLocalLayoutInfo;
    MtxParhlDevice *poParhlDev = (MtxParhlDevice*)dev->core_data;

    oLocalLayoutInfo.ulABStart = poParhlDev->aoLocalSurfaceHeaps[MM_LOCAL_MEMTYPE_AB].base;
    oLocalLayoutInfo.ulABSize = poParhlDev->aoLocalSurfaceHeaps[MM_LOCAL_MEMTYPE_AB].size;
    oLocalLayoutInfo.ulCDStart = poParhlDev->aoLocalSurfaceHeaps[MM_LOCAL_MEMTYPE_CD].base;
    oLocalLayoutInfo.ulCDSize = poParhlDev->aoLocalSurfaceHeaps[MM_LOCAL_MEMTYPE_CD].size;
    oLocalLayoutInfo.ulCommonABStart = poParhlDev->aoLocalSurfaceHeaps[MM_LOCAL_MEMTYPE_AB_COMMON].base;
    oLocalLayoutInfo.ulCommonABSize = poParhlDev->aoLocalSurfaceHeaps[MM_LOCAL_MEMTYPE_AB_COMMON].size;
    oLocalLayoutInfo.ulCommonCDStart = poParhlDev->aoLocalSurfaceHeaps[MM_LOCAL_MEMTYPE_CD_COMMON].base;
    oLocalLayoutInfo.ulCommonCDSize = poParhlDev->aoLocalSurfaceHeaps[MM_LOCAL_MEMTYPE_CD_COMMON].size;
    oLocalLayoutInfo.ulLinearStart = poParhlDev->aoLocalSurfaceHeaps[MM_LOCAL_MEMTYPE_LINEAR].base;
    oLocalLayoutInfo.ulLinearSize  = poParhlDev->aoLocalSurfaceHeaps[MM_LOCAL_MEMTYPE_LINEAR].size;

    if (copy_to_user((void*)arg, &oLocalLayoutInfo, sizeof(oLocalLayoutInfo)))
        return -EFAULT;

    return 0;
}
