 /***************************************************************************\
|*                                                                           *|
|*       Copyright 2002 Matrox Graphics Inc. All rights reserved.            *|
|*                                                                           *|
|*     NOTICE TO THE USER: This source code is the copyrighted work of       *| 
|*     Matrox. Users of this source code are hereby granted a nonexclusive,  *|
|*     royalty-free copyright license to use this code in individual and     *| 
|*     commercial software.                                                  *|
|*                                                                           *|
|*     Any use of this source code must include, in the user documenta-      *|
|*     tion and internal comments to the code, notices to the end user       *|
|*     as follows:                                                           *|
|*                                                                           *|
|*       Copyright 2002 Matrox Graphics Inc. All rights reserved.            *|
|*                                                                           *|
|*     The source code is provided to you AS IS and WITH ALL FAULTS. Matrox  *|
|*     makes no representation and gives no warranty whatsoever, whether     *|
|*     express or implied, and without limitation, with regard to the        *|
|*     quality, safety, contents, performance, merchantability, non-         *|
|*     infringement or suitability for any particular or intended purpose    *|
|*     of this source code. In no event will Matrox be liable for any        *|
|*     direct, indirect, punitive, special, incidental or consequential      *|
|*     damages however they may arise and even if Matrox has been previously *|
|*     advised of the possibility of such damages.                           *|
|*                                                                           *|
 \***************************************************************************/
/***************************************************************************************\

File Name:      mtx_heap.c

Description:    MTX memory heaps. This file implements memory manipulation 
                procedures for any type of heaps. 

References:     None.

Author:         Karl Lessard    <klessard@matrox.com>

    Copyright (c) 2002, Matrox Graphics Inc.
    All Rights Reserved.

\***************************************************************************************/

#define __NO_VERSION__
#include "mtx_drv.h"

#if 0
/***  SYSTEM HEAPS  ***/


/***************************************************************************************
 * Function:       mtx_heap_system_alloc
 *
 * Description:    Allocate a block of memory for a system heap
 *
 * Parameters:     heap         heap
 *                 size         amount of memory desired
 *
 * Return Value:   A heap memory block, NULL if failed.
 *
 * Comments:
 */
static mtx_heap_block_t*
mtx_heap_system_alloc_region(mtx_heap_t *heap, unsigned long size)
{
    mtx_heap_memory_t *block;
    mtx_region_t *region;

    block = MTXALLOC(sizeof(mtx_heap_block_t));
    if (!block) {
        MTX_ERROR("Out of memory while trying to allocate heap block\n");
        return NULL;
    }

    /* alloc memory region */
    if (mtx_mem_alloc_region(&block->region) < 0) {
            
        MTX_ERROR("Fail to allocate memory for block (size: %lu, type: %u)\n", 
                  block->region.size, block->region.type);
        return NULL;
    }

    return region;
}

/***************************************************************************************
 * Function:       mtx_heap_system_free
 *
 * Description:    Free memory from a system heap
 *
 * Parameters:     heap         memory heap
 *                 region       memory region
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:
 */
static int
mtx_heap_system_free(mtx_heap_t *heap, mtx_region_t *region)
{
    /* free memory pointed by this region */
    mtx_mem_free_region(region);
    
    /* remove region from heap list */
    list_del(&region->heap_list_node);

    /* delete region structure */
    MTXFREE((void*)region, sizeof(mtx_region_t));
    
    return 0;
}
#endif
/***************************************************************************************
 * Function:       mtx_heap_get_block_from_offset
 *
 * Description:    Retrieve a block in a heap from its offset in this heap
 *
 * Parameters:     heap     memory heap
 *                 offset   offset in memory heap
 *
 * Return Value:   block at this offset, NULL if not found
 *
 * Comments:
 */
static mtx_heap_block_t*
mtx_heap_get_block_from_offset(mtx_heap_t* heap, unsigned long offset)
{
    mtx_heap_block_t* block = NULL;
    struct list_head* pos;
    
    if (!heap)
        return NULL;

    /* parse block list and match offset */
    list_for_each(pos, &heap->block_list) {

        block = list_entry(pos, mtx_heap_block_t, heap_list_node);
#if 0
        MTX_DEBUG("block offset: %u(0x%08x), offset: %u(0x%08x)\n", 
                  block->offset, block->offset, offset, offset);
#endif
        if (block && (block->offset == offset))
            return block;   /* block found */
    }

    /* no block at this offset found */
    return NULL;
}

/***************************************************************************************
 * Function:       mtx_heap_get_local_region
 *
 * Description:    Allocate memory for a local heap
 *
 * Parameters:     heap         memory heap
 *                 size         amount of memory desired
 *                 start        start address desired for block, ignored if -1
 *
 * Return Value:   a memory region free to be used
 *
 * Comments:
 */
static mtx_region_t*
mtx_heap_get_local_region(mtx_heap_t *heap, long start, unsigned long size)
{
    mtx_region_t *free_region = NULL, *new_region;
    struct list_head *pos;

    if (list_empty(&heap->data.local.free_regions))
        return NULL;

    /* Parse all free space block in this heap XXX we might sort this list */
    list_for_each(pos, &heap->data.local.free_regions) {

        mtx_region_t* heap_region = list_entry(pos, mtx_region_t, list_node);

        if (start != -1) {
            
            /* Check if our block fits in free space. */
            if ((start >= heap_region->base) && (start < (heap_region->base + heap_region->size))) {

                mtx_region_t* new_free_region = MTXALLOC(sizeof(mtx_region_t));
    
                /* Split the free region from start and add left region to free list */
                mtx_split_region(new_free_region, heap_region, (start - heap_region->base));
                list_add_tail(&new_free_region->list_node, &heap->data.local.free_regions);

                free_region = heap_region;
                break;
            }
            
        } else {
           
            /* Use a worse fit approach */
            if (!free_region || 
                (heap_region && (heap_region->size > size) && (free_region->size < heap_region->size))) {
            
                free_region = heap_region;
            }
        }
    }

    if (!free_region)
        return NULL; 

    if (free_region->size == size)
    {
        /* Free region match perfect in size, 
         * remove it from free list and return it as is */
        list_del(&free_region->list_node);
        return free_region;
    }

    /* We need to split our free region in two parts */
    new_region = (mtx_region_t*) MTXALLOC(sizeof(mtx_region_t));
    if (!new_region)
        return NULL;

    mtx_split_region(new_region, free_region, size);

    return new_region;
}

/***************************************************************************************
 * Function:       mtx_heap_release_local_region
 *
 * Description:    Release a region of local memory.
 *
 * Parameters:     heap         memory heap
 *                 region       memory region
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:
 */
static int
mtx_heap_release_local_region(mtx_heap_t *heap, mtx_region_t *region)
{
    struct list_head *pos;
    mtx_region_t *last_merged_region = NULL;

    /* Try to merge region with another free one */
    list_for_each(pos, &heap->data.local.free_regions) {

        mtx_region_t* heap_region = list_entry(pos, mtx_region_t, list_node);

        if (heap_region) {

            if ((heap_region->base + heap_region->size) == region->base) {

                if (last_merged_region != NULL) {
                    
                    /* we already merge with a free region at the left,
                     * merge with result and remove old free region */
                    mtx_merge_regions(last_merged_region, heap_region, 0);
                    list_del(&last_merged_region->list_node);
                    mtx_mem_free_region(NULL, last_merged_region);
                    last_merged_region = heap_region;

                    break; /* all possible merges are done */
                    
                } else {
                    
                    /* merge region after this one */
                    mtx_merge_regions(region, heap_region, 0);
                    mtx_mem_free_region(NULL, region);
                    last_merged_region = heap_region;
                }
                
            } else if (region->base + region->size == heap_region->base) {

                if (last_merged_region != NULL) {
                    
                    /* we already merge with a free region at the right,
                     * merge with result and remove old free region */
                    mtx_merge_regions(last_merged_region, heap_region, 1);
                    list_del(&last_merged_region->list_node);
                    mtx_mem_free_region(NULL, last_merged_region);
                    last_merged_region = heap_region;
                    
                    break; /* all possible merges are done */
                    
                } else {

                    /* merge region before this one */
                    mtx_merge_regions(region, heap_region, 1);
                    mtx_mem_free_region(NULL, region);
                    last_merged_region = heap_region;
                }
            } 
        }
    }

    if (!last_merged_region)
    {
        /* We failed to merge, simply add this region to our free list */
        list_add_tail(&region->list_node, &heap->data.local.free_regions);
    }

    return 0;
}

/***************************************************************************************
 * Function:       mtx_heap_create
 *
 * Description:    Allocate and initialize a new heap depending of the type of memory 
 *                 it will manage. 
 *
 * Parameters:     dev          MTX device
 *                 memtype      Type of memory managed by this heap
 *                 base         Physical start address of this heap        
 *                 size         Amount of memory managed by this heap (if it is 0, full
 *                              aperture size is assumed).
 *
 * Output Params:  heap_ptr     New heap initialized
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:
 */
int mtx_heap_create(mtx_device_t *dev, mtx_memory_type memtype, unsigned long base, unsigned long size, mtx_heap_t **heap_ptr)
{
    mtx_heap_t* heap;

    heap = (mtx_heap_t*) MTXALLOC(sizeof(mtx_heap_t));
    if (!heap)
        return -ENOMEM;
    
    heap->base = base;
    heap->size = size;
    heap->type = memtype;    
        
    /* Init system heap data */
    INIT_LIST_HEAD(&heap->block_list);

    switch (heap->type) {

        case MTX_MEMTYPE_LOCAL: {

            mtx_region_t *total_region = (mtx_region_t*) MTXALLOC(sizeof(mtx_region_t));
            if (!total_region)
                return -ENOMEM;

            /* Init free region lists */
            INIT_LIST_HEAD(&heap->data.local.free_regions);

            /* Create initial region and add it to free region list */
            total_region->base     = heap->base;
            total_region->kaddr    = NULL;
            total_region->busaddr  = total_region->base; /* bus 'offset' instead */
            total_region->size     = size;
            total_region->type     = MTX_MEMTYPE_LOCAL;
            total_region->flags    = MTX_REGION_DYNAMIC;
            total_region->map_perms = MTX_MAP_NONE;
            total_region->mtrr     = -1;
            list_add_tail(&total_region->list_node, &heap->data.local.free_regions);
            
            break;
        }
        
        case MTX_MEMTYPE_AGP:
        case MTX_MEMTYPE_PCI_LOCKED:
            break;

        default:
            return -EINVAL; /* unsupported type */
    }

    *heap_ptr = heap;
    return 0;
}

/***************************************************************************************
 * Function:       mtx_heap_destroy
 *
 * Description:    Cleanup a heap and free all its block
 *
 * Parameters:     dev          MTX device
 *                 heap         Heap to clean
 *
 * Return Value:   None.
     *
 * Comments:
 */
void mtx_heap_destroy(mtx_device_t *dev, mtx_heap_t *heap)
{
    struct list_head *pos, *tmp_pos;
    mtx_heap_block_t *block = NULL;
    
    if (!heap)
        return; /* Invalid heap */

    list_for_each_safe(pos, tmp_pos, &heap->block_list)
    {
        block = list_entry(pos, mtx_heap_block_t, heap_list_node);

        if (block) {
            /* Free heap block */
            mtx_heap_free_block(dev, block);
        } 
    }

    if (heap->type == MTX_MEMTYPE_LOCAL) {

        mtx_region_t *region;
    
        /* delete free regions (if present) */
        list_for_each_safe(pos, tmp_pos, &heap->data.local.free_regions) {
             
            region = list_entry(pos, mtx_region_t, list_node);
            MTXFREE(region, sizeof(mtx_region_t));
        }
    }

    /* Free heap memory */
    MTXFREE(heap, sizeof(*heap));
}

/***************************************************************************************
 * Function:       mtx_heap_alloc_block_into
 *
 * Description:    Alloc a block of memory in a specific location in memory.
 *
 * Parameters:     dev              MTX device
 *                 heap_group       Heap group
 *                 size             Size requested for new block
 *
 * Return Value:   A heap block available for this surface, NULL if failed.
 *
 * Comments:       None.
 */
mtx_heap_block_t* 
mtx_heap_alloc_block_into(mtx_device_t *dev,
                          mtx_heap_group_t *heap_group,
                          unsigned long size)
{
    mtx_heap_t *heap = NULL;
    mtx_heap_block_t *block = NULL;
    struct list_head *pos;

    if (!heap_group) return NULL;

    /* try to allocate block in first heap. If failed, try second heap and so on... */
    list_for_each(pos, &(heap_group->heap_list)) {

        heap = list_entry(pos, mtx_heap_t, list_node);
    
        if (heap) {

            block = MTXALLOC(sizeof(mtx_heap_block_t));
            if (!block) {
                MTX_ERROR("Out of memory while trying to allocate heap block\n");
                return NULL;
            }    

            if (heap->type == MTX_MEMTYPE_LOCAL) {
        
                /* Get free region in local memory aperture */
                block->region = mtx_heap_get_local_region(heap, -1, size);
                if (!block->region) 
                    goto failed;

            #if 0//DEBUG
                /* dump free list */
                {
                    struct list_head *pos;
                    mtx_region_t *dbg_region;

                    MTX_DEBUG("Get local region:\n");
                    MTX_DEBUG("\tGiving region at 0x%08lx, %lu bytes\n", 
                            block->region->base, block->region->size);
                    MTX_DEBUG("\tFree list is:\n");
                    list_for_each(pos, &heap->data.local.free_regions) {
                        dbg_region = list_entry(pos, mtx_region_t, list_node);
                        MTX_DEBUG("\t\t0x%08lx, %lu bytes\n", dbg_region->base, dbg_region->size);
                    }
                }
            #else
                MTX_DEBUG("Get local region at 0x%08lx (size: %lu)\n", 
                           block->region->base, block->region->size);
            #endif
        
            } else {
    
                /* Alloc block region in system memory */
                block->region = mtx_mem_alloc_region(dev, size, heap->type, NULL);
                if (!block->region)
                    goto failed;
                
                block->region->pid = current->pid;
                block->region->map_perms = MTX_MAP_OWNER | MTX_MAP_AUTH;
            }
    
            /* add block to heap */
            list_add_tail(&block->heap_list_node, &heap->block_list);
            block->heap = heap;
            block->offset = block->region->base - heap->base;
            
            return block;
        }
    }
    
failed:
    
    MTX_ERROR("Fail to allocate memory for block (size: %lu, type: %u)\n", 
              size, heap->type);
    
    MTXFREE(block, sizeof(mtx_heap_block_t));
    return NULL;
}

/***************************************************************************************
 * Function:       mtx_heap_alloc_block_at
 *
 * Description:    Alloc a block of memory at the specific address in heap.
 *
 * Parameters:     dev              MTX device
 *                 heap_group       Heap group
 *                 address          Address in memory location
 *                 size             Size requested for new block
 *
 * Return Value:   A heap block available for this surface, NULL if failed.
 *
 * Comments:       None.
 */
mtx_heap_block_t* 
mtx_heap_alloc_block_at(mtx_device_t *dev, 
                        mtx_heap_group_t *heap_group,
                        unsigned long address,
                        unsigned long size)
{
    mtx_heap_t *heap = NULL;
    mtx_heap_block_t *block = NULL, *heap_block = NULL;
    struct list_head *pos;
    
    MTX_DEBUG("Allocating block at: 0x%08lx, size: %lu\n", address, size);

    /* Allocate block record */
    block = MTXALLOC(sizeof(mtx_heap_block_t));
    if (!block) {
        MTX_ERROR("PARHL: Out of memory while trying to allocate heap block\n");
        return NULL;
    }
    
    /* try to allocate block in first heap. If failed, try second heap and so on... */
    list_for_each(pos, &(heap_group->heap_list)) {

        heap = list_entry(pos, mtx_heap_t, list_node);

        /* Check if the offset resides in this heap */
        if (heap && (address >= heap->base) && (address < (heap->base + heap->size))) {
            
            heap_block = mtx_heap_get_block_from_offset(heap, (address - heap->base));

            if (heap_block != NULL) {

                mtx_region_t* region = heap_block->region;

                /* A block is already present at this offset, shared its region
                 * if possible */
                if (!(region->flags & MTX_REGION_SHARED) || (region->size != size))
                    goto failed;

                block->region = region;
                atomic_inc(&region->ref_count);
                
            } else {

                /* Try to get free region in memory aperture at this offset 
                 * (only for local for now) */
                if (heap->type != MTX_MEMTYPE_LOCAL) {
                    MTX_ERROR("PARHL: Unsupported memory location for block at a specific address\n");
                    goto failed;
                }
            
                block->region = mtx_heap_get_local_region(heap, address, size);
                if (!block->region)
                    goto failed;
            }
            
            /* add block to heap */
            list_add_tail(&block->heap_list_node, &heap->block_list);
            
            block->heap = heap;
            block->offset = address - heap->base;
        #if 0//DEBUG
            /* dump free list */
            {
                struct list_head *pos;
                mtx_region_t *dbg_region;

                MTX_DEBUG("Get local region:\n");
                MTX_DEBUG("\tGiving region at 0x%08lx, %lu bytes (req address: 0x%08lx)\n", 
                        block->region->base, block->region->size, address);
                MTX_DEBUG("\tFree list is:\n");
                list_for_each(pos, &heap->data.local.free_regions) {
                    dbg_region = list_entry(pos, mtx_region_t, list_node);
                    MTX_DEBUG("\t\t0x%08lx, %lu bytes\n", dbg_region->base, dbg_region->size);
                }
            }
        #else
            MTX_DEBUG("Get local region at 0x%08lx (size: %lu, req_address: 0x%08lx)\n", 
                       block->region->base, block->region->size, address);
        #endif
            return block;
        }
    }

failed:
    
    MTX_ERROR("Fail to allocate memory for block (size: %lu, type: %u) at address 0x%08lx\n", 
              size, heap->type, address);
    
    MTXFREE(block, sizeof(mtx_heap_block_t));
    return NULL;
}

/***************************************************************************************
 * Function:       mtx_heap_free_block
 *
 * Description:    Free a block of memory previously allocated by a heap
 *
 * Parameters:     dev          MTX device
 *                 heap         memory heap
 *                 block        heap block to free
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:
 */
int
mtx_heap_free_block(mtx_device_t *dev, mtx_heap_block_t *block)
{
    mtx_heap_t *heap;
    mtx_region_t *region;
    
    if (!block || !block->heap) {
        MTX_ERROR("Try to free a block with invalid block/heap\n");
        return -EINVAL;
    }

    heap = block->heap;
    region = block->region;
    
    /* remove block from heap list and delete it */
    list_del(&block->heap_list_node);
    MTXFREE((void*)block, sizeof(mtx_heap_block_t));

    /* Decrement region count if block is shared */
    if (region->flags & MTX_REGION_SHARED) {

        MTX_DEBUG("Freeing shared memory block (base: 0x%08lx, size: %lu, ref count: %u)\n", 
                  region->base, region->size, atomic_read(&region->ref_count));
        
        if (!atomic_dec_and_test(&region->ref_count))
            return 0;   /* Block still in use, nothing else to do */
    }
    
    if (heap->type == MTX_MEMTYPE_LOCAL) {
        
        /* Local heaps */   
        int ret = mtx_heap_release_local_region(heap, region);
        if (ret < 0)
            return ret;
      #if 0//DEBUG
        /* dump free list */
        {
            struct list_head *pos;
            mtx_region_t *dbg_region;

            MTX_DEBUG("Free local region:\n");
            MTX_DEBUG("\treleasing region at 0x%08lx, %lu bytes\n", 
                       block->region->base, block->region->size);
            MTX_DEBUG("\tFree list is:\n");
            list_for_each(pos, &heap->data.local.free_regions) {
                dbg_region = list_entry(pos, mtx_region_t, list_node);
                MTX_DEBUG("\t\t0x%08lx, %lu bytes\n", dbg_region->base, dbg_region->size);
            }
        }
      #else
        MTX_DEBUG("Freeing local region at 0x%08lx (size: %lu)\n", 
                   region->base, region->size);
      #endif
        
    } else {
        
        /* free system memory pointed by block region */
        mtx_mem_free_region(dev, region);
    }

    return 0;
}

/***************************************************************************************
 * Function:       mtx_heap_init
 *
 * Description:    Init heap memory allocation mechanisms.
 * 
 * Parameters:     dev          MTX device
 *
 * Return Value:   0 if succeed, errno otherwise
 *
 * Comments:       None.
 */
int mtx_heap_init(mtx_device_t* dev)
{
    mtx_heap_t *heap;
    int i, ret;

    /* Init heap groups */
    for (i = 0; i < MTX_MEMTYPE_COUNT; i++) {
        INIT_HEAP_GROUP(&dev->heap_groups[i]);
    }

    /* Create heap for framebuffer aperture */
    ret = mtx_heap_create(dev, MTX_MEMTYPE_IO, dev->framebuffer.base, dev->framebuffer.size, &heap);
    if (ret < 0) {
        MTX_ERROR("Failed to create a memory heap for framebuffer aperture\n");
        return ret;
    }

    ADD_HEAP_TO_GROUP(&dev->heap_groups[MTX_MEMTYPE_LOCAL], heap);

    if (agp_drv && (dev->agp_cap_offset != -1))
    {
        /* Create heap for AGP memory for this device */
        ret = mtx_heap_create(dev, MTX_MEMTYPE_AGP, agp_drv->aperture_base, agp_drv->aperture_size, 
                              &heap);
        if (ret < 0) {
            MTX_ERROR("Failed to create a memory heap for AGP aperture\n");
            return ret;
        }

        ADD_HEAP_TO_GROUP(&dev->heap_groups[MTX_MEMTYPE_AGP], heap);
    }

    /* Create heap for PCI locked system memory XXX use false base and size values */
    ret = mtx_heap_create(dev, MTX_MEMTYPE_PCI_LOCKED, 0, 0, &heap);
    if (ret < 0) {
        MTX_ERROR("Failed to create a memory heap for PCI locked system memory\n");
        return ret;
    }

    ADD_HEAP_TO_GROUP(&dev->heap_groups[MTX_MEMTYPE_PCI_LOCKED], heap);

    return 0;
}

/***************************************************************************************
 * Function:       mtx_heap_cleanup
 *
 * Description:    Cleanup heaps
 *
 * Parameters:     dev          MTX device
 *
 * Return Value:   None.
 *
 * Comments:       None.
 */
void mtx_heap_cleanup(mtx_device_t* dev)
{
    struct list_head *pos, *tmppos;
    int i;
    
    for (i = 0; i < MTX_MEMTYPE_COUNT; i++) {

        mtx_heap_group_t *heap_group = &dev->heap_groups[i];
        
        /* cleanup each heaps in this group */
        list_for_each_safe(pos, tmppos, &heap_group->heap_list) {

            mtx_heap_t* heap =  list_entry(pos, mtx_heap_t, list_node);

            if (heap) {

                /* destroy heap and decrement heap count in this group */
                mtx_heap_destroy(dev, heap);
                heap_group->heap_count--;
            }
        }
        
        if (heap_group->heap_count != 0) {
            MTX_ERROR("Not all heaps were cleaned for memory group\n");
        }
        
        /* Empty heap list */
        INIT_LIST_HEAD(&heap_group->heap_list);
    }

    return;
}
