 /***************************************************************************\
|*                                                                           *|
|*       Copyright 2002 Matrox Graphics Inc. All rights reserved.            *|
|*                                                                           *|
|*     NOTICE TO THE USER: This source code is the copyrighted work of       *| 
|*     Matrox. Users of this source code are hereby granted a nonexclusive,  *|
|*     royalty-free copyright license to use this code in individual and     *| 
|*     commercial software.                                                  *|
|*                                                                           *|
|*     Any use of this source code must include, in the user documenta-      *|
|*     tion and internal comments to the code, notices to the end user       *|
|*     as follows:                                                           *|
|*                                                                           *|
|*       Copyright 2002 Matrox Graphics Inc. All rights reserved.            *|
|*                                                                           *|
|*     The source code is provided to you AS IS and WITH ALL FAULTS. Matrox  *|
|*     makes no representation and gives no warranty whatsoever, whether     *|
|*     express or implied, and without limitation, with regard to the        *|
|*     quality, safety, contents, performance, merchantability, non-         *|
|*     infringement or suitability for any particular or intended purpose    *|
|*     of this source code. In no event will Matrox be liable for any        *|
|*     direct, indirect, punitive, special, incidental or consequential      *|
|*     damages however they may arise and even if Matrox has been previously *|
|*     advised of the possibility of such damages.                           *|
|*                                                                           *|
 \***************************************************************************/
/***************************************************************************************\

File Name:      mtx_drv.c

Description:    Main file of MTX kernel module. 
                Implements module initialization and cleanup calls.

References:     None.

Author:         Karl Lessard    <klessard@matrox.com>

    Copyright (c) 2001, Matrox Graphics Inc.
    All Rights Reserved.

\***************************************************************************************/

#include "mtx_drv.h"

/* Supported boards */
static struct pci_device_id 
mtx_supported_cards[NBR_SUPPORTED_CARDS] = {
   
    MTX_PCI_DEVICE_ID( PCI_DEVICE_ID_PARHELIA_512, PCI_ANY_ID ), /* Parhelia 512 */
};

/* Driver structure used by PCI driver interface */
static struct pci_driver 
mtx_pci_driver = {
    
    name:     (char*)MTX_DRIVER_NAME,       
    id_table: mtx_supported_cards, 
    probe:    mtx_dev_probe,        
    remove:   mtx_dev_remove,
};

/* MTX file operations */
static struct file_operations 
mtx_fops = {
    owner:      THIS_MODULE,
    open:       mtx_open,
    release:    mtx_release,
    ioctl:      mtx_ioctl,
    mmap:       mtx_mmap,
};

/* MTX device major number */
static int mtx_major_nbr = MTX_MAJOR_NUMBER;

/* Module description */
MODULE_AUTHOR("Matrox Graphics Inc.");
MODULE_DESCRIPTION("Generic linux driver for Matrox graphics devices");
MODULE_SUPPORTED_DEVICE("Parhelia 512");

/* Module parameters */
MTX_MODULE_PARM_DECL(,agprate, int, "i", 1);

#ifdef MODULE_LICENSE
MODULE_LICENSE("Copyright (c) 2002, Matrox Graphics Inc.");
#endif

/***************************************************************************************
 * Function:       mtx_init 
 *
 * Description:    Called via init_module at module load time.    
 *
 * Parameters:     None.
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:
 *
 */
static int __init 
mtx_init(void)
{
    int result;

    MTX_INFO("MTX Driver version %u.%u.%u\n",
              MTX_VERSION_MAJOR, MTX_VERSION_MINOR, MTX_VERSION_PATCH);
   
    /* Registering device driver */
    result = register_chrdev(mtx_major_nbr, MTX_DRIVER_NAME, &mtx_fops);
    
    if (result < 0) {
        MTX_ERROR("Failed to register device driver using %u as major number.\n", 
                   MTX_MAJOR_NUMBER);
        return result;
    }

    if (mtx_major_nbr == 0) {

        /* Major number allocated dynamically */
        mtx_major_nbr = result;
    }

    /* Check if we can support AGP boards */
    if (mtx_agp_init() < 0) { 
        MTX_WARNING("Fail to initialize AGP transfers. Please check that agpgart is loaded\n");
    }

    /* init all core drivers */
    mtx_parhl_driver.init();

    /* Registering PCI driver */
    return pci_module_init(&mtx_pci_driver);
}

/***************************************************************************************
 * Function:       mtx_cleanup
 *
 * Description:    Called via cleanup_module at module unload time.
 *
 * Parameters:     None.
 *
 * Return Value:   None.
 *
 * Comments:
 *
 */
static void __exit 
mtx_cleanup(void)
{
    int result;

    /* Unregistering PCI driver */
    pci_unregister_driver(&mtx_pci_driver);
    
    /* cleanup all core drivers */
    mtx_parhl_driver.cleanup();

    /* Cleanup AGP driver */
    mtx_agp_cleanup();
    
    /* Unregistering device driver */
    result = unregister_chrdev(mtx_major_nbr, MTX_DRIVER_NAME);

    if (result < 0) {
        MTX_ERROR("Failed to unregister device driver using %u as major number.\n", 
                  MTX_MAJOR_NUMBER);
        return;
    }
    
    /* Dump memory statistics. It must be all zeros */
    mtx_mem_stats_dump_list(0);
}

/* Declaration of init and cleanup functions */
module_init(mtx_init);
module_exit(mtx_cleanup);
