 /***************************************************************************\
|*                                                                           *|
|*       Copyright 2002 Matrox Graphics Inc. All rights reserved.            *|
|*                                                                           *|
|*     NOTICE TO THE USER: This source code is the copyrighted work of       *| 
|*     Matrox. Users of this source code are hereby granted a nonexclusive,  *|
|*     royalty-free copyright license to use this code in individual and     *| 
|*     commercial software.                                                  *|
|*                                                                           *|
|*     Any use of this source code must include, in the user documenta-      *|
|*     tion and internal comments to the code, notices to the end user       *|
|*     as follows:                                                           *|
|*                                                                           *|
|*       Copyright 2002 Matrox Graphics Inc. All rights reserved.            *|
|*                                                                           *|
|*     The source code is provided to you AS IS and WITH ALL FAULTS. Matrox  *|
|*     makes no representation and gives no warranty whatsoever, whether     *|
|*     express or implied, and without limitation, with regard to the        *|
|*     quality, safety, contents, performance, merchantability, non-         *|
|*     infringement or suitability for any particular or intended purpose    *|
|*     of this source code. In no event will Matrox be liable for any        *|
|*     direct, indirect, punitive, special, incidental or consequential      *|
|*     damages however they may arise and even if Matrox has been previously *|
|*     advised of the possibility of such damages.                           *|
|*                                                                           *|
 \***************************************************************************/
/***************************************************************************************\

File Name:      mtx_vm.c

Description:    Virtual memory operation on a MTX device file.

References:     None.

Author:         Karl Lessard    <klessard@matrox.com>

    Copyright (c) 2001, Matrox Graphics Inc.
    All Rights Reserved.

\***************************************************************************************/

#define __NO_VERSION__
#include "mtx_drv.h"

/* I/O regions vm operations */
static struct vm_operations_struct 
mtx_vmops_stub = {
    nopage:     mtx_vm_nopage_stub,
    open:       mtx_vm_open_stub,
    close:      mtx_vm_close_stub,
};

/* AGP regions vm operations */
static struct vm_operations_struct 
mtx_vmops_agp = {
    nopage:     mtx_vm_nopage_agp,
    open:       mtx_vm_open_stub,
    close:      mtx_vm_close_stub,
};

/* PCI regions vm operations */
static struct vm_operations_struct
mtx_vmops_pci = {
    nopage:     mtx_vm_nopage_pci,
    open:       mtx_vm_open_stub,
    close:      mtx_vm_close_stub,
};

/* System regions vm operations */
static struct vm_operations_struct
mtx_vmops_sys = {
    nopage:     mtx_vm_nopage_sys,
    open:       mtx_vm_open_stub,
    close:      mtx_vm_close_stub,
};


/***************************************************************************************
 * Function:       mtx_vm_nopage_stub
 *
 * Description:    
 * 
 * Parameters:    
 * 
 * Return Value:   NOPAGE_SIGBUS
 * 
 * Comments:       
 */
struct page*
mtx_vm_nopage_stub(struct vm_area_struct *vma, unsigned long address, int write_access)
{
    return NOPAGE_SIGBUS;   /* mremap is not allowed */
}

/***************************************************************************************
 * Function:       mtx_vm_open_stub
 *
 * Description:    Called everytime a reference to this VMA is opened by a process 
 *                 (first process to request the VMA, or its child processes by forking).  
 *
 * Parameters:     vma              pointer to the vma structure
 *
 * Return Value:   None.
 *
 * Comments:       
 */
void
mtx_vm_open_stub(struct vm_area_struct *vma)
{
    /* Nothing needs to be done */
}

/***************************************************************************************
 * Function:       mtx_vm_close_stub
 *
 * Description:    Called everytime a process close a reference to the VMA.
 *
 * Parameters:     vma              pointer to the vma structure
 *
 * Return Value:   None.
 *
 * Comments:       
 */
void
mtx_vm_close_stub(struct vm_area_struct *vma)
{
    /* Nothing needs to be done */
}

/***************************************************************************************
 * Function:       mtx_vm_nopage_agp
 *
 * Description:    This should only be called when trying to mremap this region, since
 *                 all page tables has been created during the remap_page_range
 *                 procedure.
 *
 * Parameters:     vma              pointer to the vma structure
 *                 address          first address of faulting page
 *                 write_access     permissions
 *
 * Return Value:   Page structure for the faulting address, or NOPAGE
 *
 * Comments:       
 */
struct page*
mtx_vm_nopage_agp(struct vm_area_struct *vma, unsigned long address, int write_access)
{
    return NOPAGE_SIGBUS;
}

/***************************************************************************************
 * Function:       mtx_vm_nopage_pci
 *
 * Description:    Return the missing page for PCI locked memory mappings.
 *
 * Parameters:     vma              pointer to the vma structure
 *                 address          first address of faulting page
 *                 write_access     permissions
 *
 * Return Value:   Page structure for the faulting address, or NOPAGE
 *
 * Comments:       
 */
struct page*
mtx_vm_nopage_pci(struct vm_area_struct *vma, unsigned long address, int write_access)
{
    struct page *page;
    unsigned long paddr;

    /* retrieve physical address corresponding to the fault address */
    paddr = (address - vma->vm_start) + (vma->vm_pgoff << PAGE_SHIFT);
    
    /* convert physical address to virtual, and retrieve page (and lock it) */
    page = virt_to_page(__va(paddr));
    get_page(page);
    
    return page;
}

/***************************************************************************************
 * Function:       mtx_vm_nopage_sys
 *
 * Description:    Return the missing page for system memory mappings.
 *
 * Parameters:     vma              pointer to the vma structure
 *                 address          first address of faulting page
 *                 write_access     permissions
 *
 * Return Value:   Page structure for the faulting address, or NOPAGE
 *
 * Comments:       
 */
struct page*
mtx_vm_nopage_sys(struct vm_area_struct *vma, unsigned long address, int write_access)
{
    struct page *page;
    mtx_region_t *region;
    unsigned long offset;
    unsigned long kaddr;
    pgd_t *pgd; 
    pmd_t *pmd; 
    pte_t *pte;

    /* retrieve the region for this vma */
    region = (mtx_region_t*)vma->vm_private_data;
    if (!region) return NOPAGE_SIGBUS;
        
    /* retrieve offset in this region we want to map */
    offset = address - vma->vm_start;
        
    /* retrieve kernel virtual address at this offset */
    kaddr = VMALLOC_VMADDR((unsigned long)((u8*)region->kaddr + offset));
    if (!kaddr)
        return NOPAGE_SIGBUS;

    /* Get page by going through the page table */
    pgd = pgd_offset_k(kaddr); if (pgd_none(*pgd)) return NULL;
    pmd = pmd_offset(pgd, kaddr); if (pmd_none(*pmd)) return NULL;
    pte = pte_offset(pmd, kaddr); if (!pte_present(*pte)) return NULL;
    page = pte_page(*pte); if (!page) return NOPAGE_SIGBUS;

    /* lock page and return it */
    get_page(page);
    return page;
}


/***************************************************************************************
 * Function:       mtx_vm_map_io
 *
 * Description:    Mapping I/O memory to user virtual space, from a vm_area structure
 *                 filled by the kernel. The only thing we need to do here is to 
 *                 create a table of pages that points to physical memory for every 
 *                 virtual addresses in the vm_area_struct.
 *
 * Parameters:     dev              MTX device
 *                 region           pointer to our memory region
 *                 vma              pointer to the vma structure
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:       
 */
int
mtx_vm_map_io(mtx_device_t *dev, mtx_region_t* region, struct vm_area_struct *vma)
{
    size_t size = vma->vm_end - vma->vm_start;
    
    /* address mapped must be in I/O memory */    
    if (region->base < __pa(high_memory))
        return -EINVAL;
            
    /* Size of mapping contain the entire memory region */
    if (region->size > (vma->vm_end - vma->vm_start))
        return -EINVAL;
    
    /* this vma points to I/O memory and should never be swapped out */
    vma->vm_flags |= VM_IO | VM_RESERVED;

    if (remap_page_range(vma->vm_start, region->base, size, vma->vm_page_prot))
        return -EAGAIN;
    
    /* Use IO vm operations */
    vma->vm_ops = &mtx_vmops_stub;
    
    return 0;
}

/***************************************************************************************
 * Function:       mtx_vm_map_agp
 *
 * Description:    Mapping AGP memory to user virtual space, from a vm_area structure
 *                 filled by the kernel. The only thing we need to do here is to 
 *                 create a table of pages that points to physical memory for every 
 *                 virtual addresses in the vm_area_struct.
 *
 * Parameters:     dev              MTX device
 *                 region           pointer to our memory region
 *                 vma              pointer to the vma structure
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:       
 */
int
mtx_vm_map_agp(mtx_device_t *dev, mtx_region_t *region, struct vm_area_struct *vma)
{
    size_t size = vma->vm_end - vma->vm_start;
        
    /* address mapped must be in I/O memory */    
    if (region->base < __pa(high_memory))
        return -EINVAL;
    
    /* Size of mapping must contain the whole memory region */
    if (region->size > (vma->vm_end - vma->vm_start))
        return -EINVAL;
    
    /* should never be swapped out */
    vma->vm_flags |= VM_IO | VM_RESERVED;
    
    if (remap_page_range(vma->vm_start, region->base, size, vma->vm_page_prot))
        return -EAGAIN;
    
    /* Use AGP vm operations */
    vma->vm_ops = &mtx_vmops_agp;

    return 0;
}

/***************************************************************************************
 * Function:       mtx_vm_map_pci
 *
 * Description:    Mapping PCI locked memory to user virtual space, from a vm_area 
 *                 structure filled by the kernel. In this case, we can be sure that
 *                 the memory space is page-aligned, contiguous, and cannot be swapped
 *                 out by the kernel.
 *                 
 * Parameters:     dev              MTX device
 *                 region           pointer to our memory region
 *                 vma              pointer to the vma structure
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:       
 */
int
mtx_vm_map_pci(mtx_device_t *dev, mtx_region_t *region, struct vm_area_struct *vma)
{
    /* Size of mapping must contain the whole memory region */
    if (region->size > (vma->vm_end - vma->vm_start))
        return -EINVAL;

    vma->vm_flags |= VM_RESERVED;
    
    /* Use PCI vm operations */
    vma->vm_ops = &mtx_vmops_pci;

    return 0;
}

/***************************************************************************************
 * Function:       mtx_vm_map_sys
 *
 * Description:    Map an area allocated in memory previously allocated by the kernel
 *                 with the standard kmalloc (or vmalloc) allocation scheme.
 *                 
 * Parameters:     dev              MTX device
 *                 region           pointer to our memory region
 *                 vma              pointer to the vma structure
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:       
 */
int
mtx_vm_map_sys(mtx_device_t *dev, mtx_region_t *region, struct vm_area_struct *vma)
{
    size_t size = vma->vm_end - vma->vm_start;

    /* Size of mapping must contain the whole memory region */
    if (region->size > (vma->vm_end - vma->vm_start))
        return -EINVAL;
    
    vma->vm_flags |= VM_LOCKED;

    if (region->size > (32 * PAGE_SIZE)) {
        
        /* Use system vm operations */
        vma->vm_ops = &mtx_vmops_sys;
        
    } else {
    
        if (remap_page_range(vma->vm_start, region->base, size, vma->vm_page_prot))
        {
            MTX_ERROR("Failed to map region at 0x%08lx, size: %lu\n", region->base, region->size);
            return -EAGAIN;
        }

        vma->vm_ops = &mtx_vmops_stub;
    }

    /* attach our region to the vma private field */
    vma->vm_private_data = (void*)region;
    
    return 0;
}
