 /***************************************************************************\
|*                                                                           *|
|*       Copyright 2002 Matrox Graphics Inc. All rights reserved.            *|
|*                                                                           *|
|*     NOTICE TO THE USER: This source code is the copyrighted work of       *| 
|*     Matrox. Users of this source code are hereby granted a nonexclusive,  *|
|*     royalty-free copyright license to use this code in individual and     *| 
|*     commercial software.                                                  *|
|*                                                                           *|
|*     Any use of this source code must include, in the user documenta-      *|
|*     tion and internal comments to the code, notices to the end user       *|
|*     as follows:                                                           *|
|*                                                                           *|
|*       Copyright 2002 Matrox Graphics Inc. All rights reserved.            *|
|*                                                                           *|
|*     The source code is provided to you AS IS and WITH ALL FAULTS. Matrox  *|
|*     makes no representation and gives no warranty whatsoever, whether     *|
|*     express or implied, and without limitation, with regard to the        *|
|*     quality, safety, contents, performance, merchantability, non-         *|
|*     infringement or suitability for any particular or intended purpose    *|
|*     of this source code. In no event will Matrox be liable for any        *|
|*     direct, indirect, punitive, special, incidental or consequential      *|
|*     damages however they may arise and even if Matrox has been previously *|
|*     advised of the possibility of such damages.                           *|
|*                                                                           *|
 \***************************************************************************/
/***************************************************************************************\

File Name:      mtx_mem.c

Description:    Memory management for MTX devices.

References:     None.

Author:         Karl Lessard    <klessard@matrox.com>

    Copyright (c) 2001, Matrox Graphics Inc.
    All Rights Reserved.

\***************************************************************************************/

#define __NO_VERSION__
#include "mtx_drv.h"

#include <linux/slab.h>
#include <linux/wrapper.h>

#ifndef KMALLOC_MAXSIZE
#define KMALLOC_MAXSIZE (PAGE_SIZE * 32)
#endif

/* import agp backend interface */
extern drm_agp_t *agp_backend;

/* list of agp memory blocks (first item use as sentinel) */
static agp_memory agp_blocks = { 0, NULL, NULL, 0, 0, NULL, 0, 0, 0, 0, 0 };

#if MEMORY_STATS
void* mtx_mem_alloc(size_t);
void  mtx_mem_free(void*, size_t);

typedef struct mtx_mem_info
{
    unsigned long        size;      /* Size of memory chunk */

    const char*          filename;  /* Name of file where allocation was called */
    int                  line;      /* Line number where allocation was called */
    struct mtx_mem_info* prev;      /* Previous memory info record */
    struct mtx_mem_info* next;      /* Next memory info record */
    int64_t              allocation_mark; /* ID gived to allocation */

} mtx_mem_info_t;

static int alloc_count = 0;
static unsigned long alloc_mem = 0;
static int alloc_pci_count = 0;
static unsigned long alloc_pci_mem = 0;
static int alloc_agp_count = 0;
static unsigned long alloc_agp_mem = 0;
static int ioremap_count = 0;

static mtx_mem_info_t mem_info_head = {0, NULL, 0, NULL, NULL, 0};
static int64_t next_allocation_mark = 0;

/***************************************************************************************
 * Function:       mtx_mem_stats_alloc
 *
 * Description:    Allocate memory in kernel address space and attach a mem_info
 *                 record to it so we can keep track of memory statistics.
 *
 * Parameters:     size             Size in bytes of memory space to allocate
 *                 filename         Name of file calling this allocation
 *                 line             Line number in file where allocation was called
 *
 * Return Value:   Ptr to allocated memory, NULL if failed
 *
 * Comments:        
 */
void* 
mtx_mem_stats_alloc(size_t size, const char* filename, int line)
{
    void *ptr = mtx_mem_alloc(size + sizeof(mtx_mem_info_t));
    
    if (!ptr) {
        
        MTX_ERROR("Failed to alloc chunk at %s:%d of size %u\n", filename, line, size);
        mtx_mem_stats_dump();
        
    } else {

        mtx_mem_info_t *mem_info = (mtx_mem_info_t*)ptr;

        if (mem_info_head.size == 0) {

            /* size of mem_info head is used as a flag for first allocation */
            mem_info_head.next = &mem_info_head;
            mem_info_head.prev = &mem_info_head;
            mem_info_head.size = ~0;
        }

        /* Fill mem_info record at the beginning of chunk */
        mem_info->size = size;
        mem_info->filename = filename;
        mem_info->line = line;
        mem_info->allocation_mark = next_allocation_mark++;

        /* Add record to list */
        mem_info->next = mem_info_head.next;
        mem_info->prev = &mem_info_head;
        mem_info_head.next->prev = mem_info;
        mem_info_head.next = mem_info;
    }

    return (void*)((char*)ptr + sizeof(mtx_mem_info_t));
}

/***************************************************************************************
 * Function:       mtx_mem_stats_free
 *
 * Description:    Free a memory chunk and the mem_info record attached to it.
 *
 * Parameters:     ptr          Pointer to memory chunk
 *
 * Return Value:   Ptr to allocated memory, NULL if failed
 *
 * Comments:        
 */
void
mtx_mem_stats_free(void* ptr)
{
    if (ptr != NULL) {
        
        mtx_mem_info_t *mem_info = (mtx_mem_info_t*)((char*)ptr - sizeof(mtx_mem_info_t));

        /* Remove record from list */
        mem_info->next->prev = mem_info->prev;
        mem_info->prev->next = mem_info->next;
        
        /* Free chunk of memory */
        mtx_mem_free((void*)mem_info, mem_info->size + sizeof(mtx_mem_info_t));
    }
}
#endif

/***************************************************************************************
 * Function:       mtx_mem_stats_dump
 *
 * Description:    Dump statistics about memory allocation.
 *
 * Parameters:     None.
 *
 * Return Value:   None.
 * 
 * Comments:        
 */
void mtx_mem_stats_dump(void)
{
#if MEMORY_STATS
    int stats_mem = alloc_count * sizeof(mtx_mem_info_t);
    int total_mem = stats_mem + alloc_mem;
    int kb = 0;

    if (total_mem > 1024) {

        /* Print stats in kb */
        stats_mem /= 1024;
        total_mem /= 1024;
        kb = 1;
    }
    
    MTX_PRINT("\n");
    MTX_PRINT("Memory statistics:\n");
    MTX_PRINT("\tSystem alloc count:     %d\n", alloc_count);
    MTX_PRINT("\tAGP alloc count:        %d\n", alloc_agp_count);
    MTX_PRINT("\tPCI locked alloc count: %d\n", alloc_pci_count);
    MTX_PRINT("\t--------------------------\n");
    MTX_PRINT("\tTotal alloc count       %d\n", alloc_count + alloc_agp_count + alloc_pci_count);
    MTX_PRINT("\tI/O map count:          %d\n", ioremap_count);
    MTX_PRINT("\n");
    MTX_PRINT("System memory statistics:\n");
    MTX_PRINT("\tAverage size (bytes):     %ld\n", alloc_count ? alloc_mem/alloc_count : 0);
    MTX_PRINT("\tStats memory %s     %d, %d%%\n", kb ? "(kb):   " : "(bytes):", 
               stats_mem, total_mem ? (stats_mem * 100) / total_mem : 0);
    MTX_PRINT("\tTotal memory %s     %d\n", kb ? "(kb):   " : "(bytes):", total_mem);
    MTX_PRINT("\n");
#endif
}

/***************************************************************************************
 * Function:       mtx_mem_stats_dump_list
 *
 * Description:    Dump statistics about memory allocation + the list of allocation
 *                 info record starting at the allocation mark specified.
 *
 * Parameters:     start_mark       Allocation mark where to start list dump
 *
 * Return Value:   None.
 * 
 * Comments:        
 */
void mtx_mem_stats_dump_list(int64_t start_mark)
{
#if MEMORY_STATS
    mtx_mem_info_t *mem_info = mem_info_head.next;
    unsigned long total_dumped = 0, block_count = 0;

    /* Dump memory statistics */
    mtx_mem_stats_dump();
    
    MTX_PRINT("\tDump allocation list from mark %lld\n", start_mark);

    while (mem_info != &mem_info_head) {

        if (mem_info->allocation_mark >= start_mark) {

            MTX_PRINT("\t\tMemory: mark %lld, file \"%s\", line %d, size %ld\n",
                       mem_info->allocation_mark, mem_info->filename, mem_info->line, mem_info->size);

            total_dumped += mem_info->size;
            block_count++;
        }

        mem_info = mem_info->next;
    }

    MTX_PRINT("\t%lu block dumped, total memory (kb): %lu\n", block_count, total_dumped);
    MTX_PRINT("\n");
#endif
}


/***************************************************************************************
 * Function:       mtx_mem_alloc
 *
 * Description:    Allocate memory in kernel address space. The memory chunk is
 *                 insure to be contiguous in virtual address space, but not
 *                 physically. It then cannot be used for DMA by the device.
 *
 * Parameters:     size             Size in bytes of memory space to allocate
 *
 * Return Value:   Ptr to allocated memory, NULL if failed
 *
 * Comments:        
 */
void* 
mtx_mem_alloc(size_t size)
{
    void *ptr;
    u32 align = sizeof(void*);

    /* align size depending on architecture */
    size = (size + align) & ~align;

    /* TODO: Check if we are inside interrupt handler. If we do, use GFP_ATOMIC! */
    
    if (size <= KMALLOC_MAXSIZE) {
        
        /* Allocate memory using kmalloc */
        ptr = kmalloc(size, GFP_KERNEL);
        
    } else {

        /* Use vmalloc to allocate the memory chunk */
        ptr = vmalloc(size);
    }

    if (!ptr) {
        MTX_ERROR("Failed to allocate memory\n");
        return NULL;
    }
    
    /* Reset memory to zeros */
    memset(ptr, 0, size);

  #if MEMORY_STATS
    ++alloc_count;
    alloc_mem += size;
  #endif

    return ptr;
}

/***************************************************************************************
 * Function:       mtx_mem_free
 *
 * Description:    Free a memory chunk previously allocated by mtx_mem_alloc.
 *
 * Parameters:     ptr      Pointer to memory to free.
 *                 size     Size in bytes of memory to free.
 *
 * Return Value:   None.
 *
 * Comments:
 *
 */
void 
mtx_mem_free(void* ptr, size_t size)
{
    u32 align = sizeof(void*);

    /* align size depending on architecture */
    size = (size + align) & ~align;

    if (!ptr || !size) {
        MTX_ERROR("Attempt to free NULL pointer\n");
        return;
    }
    
    if (size <= KMALLOC_MAXSIZE) {

        /* This memory chunk has been allocated by kmalloc */
        kfree(ptr);
        
    } else {

        /* This memory chunk has been alloc by vmalloc */
        vfree(ptr);
    }
    
  #if MEMORY_STATS
    --alloc_count;
    alloc_mem -= size;
  #endif
}

/***************************************************************************************
 * Function:       mtx_mem_alloc_pci
 *
 * Description:    Allocate memory in kernel address space and that can be
 *                 accessed directly by a PCI device. It then needs to be
 *                 contiguous in physical memory, and in a memory range
 *                 accessible by DMA.
 *
 * Parameters:     size             Size in bytes of memory space to allocate
 *
 * Return Value:   Ptr to allocated memory, NULL if failed
 *
 * Comments:       We'll use a page oriented scheme. For now, the size must not be
 *                 higher than 2^9 pages. If it do, the allocation failed.
 *
 */
static void* 
mtx_mem_alloc_pci(size_t size)
{
    unsigned long ptr, kaddr;
    int pg_order, i;

    pg_order = mtx_get_pg_order(size);

    if (pg_order < 0)
        return NULL;

    /* TODO: Check if we are inside interrupt handler. If we do, use GFP_ATOMIC! */

    /* Get the needed number of pages */
    ptr = __get_free_pages(GFP_KERNEL, pg_order);

    if (!ptr) {
        MTX_ERROR("Failed to allocate PCI memory\n");
        return 0;
    }

    for (i = 0, kaddr = ptr; i <= pg_order; i++, kaddr += PAGE_SIZE) {

        /* Zero the pages */
        memset((void*)kaddr, 0, PAGE_SIZE);

        /* Reserved each pages so they will not be managed by the kernel */
        mem_map_reserve(virt_to_page(kaddr));
    }
    
  #if MEMORY_STATS
    ++alloc_pci_count;
    alloc_pci_mem += size;
  #endif
    
    return (void*)ptr;
}

/***************************************************************************************
 * Function:       mtx_mem_free_pci
 *
 * Description:    Free a memory chunk previously allocated by mtx_mem_alloc_pci.
 *
 * Parameters:     ptr      Pointer to memory to free.
 *
 * Return Value:   None.
 *
 * Comments:
 *
 */
static void 
mtx_mem_free_pci(void* ptr, size_t size)
{
    unsigned long kaddr;
    int pg_order, i;

    if (!ptr || !size) {
        MTX_ERROR("Attempt to free NULL pointer\n");
        return;
    }

    if (((unsigned long)ptr & (PAGE_SIZE - 1)) != 0) {
        MTX_ERROR("Attempt to free non paged-aligned PCI pointer\n");
        return;
    }

    pg_order = mtx_get_pg_order(size);
    
    for (i = 0, kaddr = (unsigned long)ptr; i <= pg_order; i++, kaddr += PAGE_SIZE) {

        /* Give back pages to the kernel memory manager */
        mem_map_unreserve(virt_to_page(kaddr));

        /* Free page */
        free_page(kaddr);
    }
    
  #if MEMORY_STATS
    --alloc_pci_count;
    alloc_pci_mem -= size;
  #endif
}

/***************************************************************************************
 * Function:       mtx_mem_alloc_agp
 *
 * Description:    Allocate memory in kernel address space and that can be
 *                 accessed directly by a AGP device. It then needs to be
 *                 allocated by the AGP chipset, which take cares of the
 *                 GART table.
 *                 
 * Parameters:     size             Size in bytes of memory space to allocate
 *
 * Return Value:   Base of allocated memory, NULL if failed
 *
 * Comments:
 *
 */
static unsigned long
mtx_mem_alloc_agp(size_t size)
{
    agp_memory *info_block, *ptr_block;
    unsigned long pg_offset = 0, pg_count = 0;
    int ret;
    
    if (!agp_drv || !agp_backend || !agp_drv->enabled)
        return 0;

    pg_count = ((size + (PAGE_SIZE - 1)) & PAGE_MASK) / PAGE_SIZE;
    if (!pg_count)
        return 0;

    /* Alloc APG memory block */
    info_block = agp_backend->allocate_memory(pg_count, AGP_NORMAL_MEMORY);

    if (!info_block) {
        MTX_ERROR("Failed to allocate AGP memory block\n");
        return 0;
    }

    /* Find free space in AGP aperture for our block */
    for (ptr_block = &agp_blocks; ptr_block != NULL; ptr_block = ptr_block->next) {

        if (ptr_block->next == NULL) {

            /* End of list, add block here and bind it after last block in aperture */
            info_block->prev = ptr_block;
            info_block->next = NULL;
            ptr_block->next = info_block;
        
            pg_offset = ptr_block->pg_start + ptr_block->page_count;
                
            break;
                
        } else if (ptr_block->next->pg_start 
                    - (ptr_block->pg_start + ptr_block->page_count) >= pg_count) {

            /* We can fit our block between those two items */
            info_block->prev = ptr_block;
            info_block->next = ptr_block->next;
            ptr_block->next = ptr_block->next->prev = info_block;

            pg_offset = ptr_block->pg_start + ptr_block->page_count;

            break;
        }
    }

    if (!ptr_block) {
        
        MTX_ERROR("Failed to find place for AGP block in aperture\n");
        return 0;
    }

    /* Bind memory block to GATT */
    ret = agp_backend->bind_memory(info_block, pg_offset);
    
    if (ret < 0) {
        MTX_ERROR("Failed to bind physical memory to AGP memory block\n");
        return 0;
    }
    
  #if MEMORY_STATS
    ++alloc_agp_count;
    alloc_agp_mem += size;
  #endif

    return (agp_drv->kern_info.aper_base + (pg_offset << PAGE_SHIFT));
}

/***************************************************************************************
 * Function:       mtx_mem_free_agp
 *
 * Description:    Free a memory chunk previously allocated by mtx_mem_alloc_agp.
 *
 * Parameters:     ptr      Pointer to memory to free.
 *
 * Return Value:   None.
 *
 * Comments:
 *
 */
static void 
mtx_mem_free_agp(unsigned long ptr, size_t size)
{
    agp_memory *ptr_block = NULL;
    unsigned int pg_start, pg_count;
    int ret;
    
    if (!agp_drv || !agp_backend || !agp_drv->enabled)
        return;

    if (((unsigned long)ptr & (PAGE_SIZE - 1)) != 0) {
        MTX_ERROR("Attempt to free non paged-aligned AGP pointer\n");
        return;
    }

    pg_start = ((off_t)ptr - agp_drv->kern_info.aper_base) >> PAGE_SHIFT;
    pg_count = ((size + (PAGE_SIZE - 1)) & PAGE_MASK) / PAGE_SIZE;

    /* Find which block are we trying to release (after sentinel) FIXME: Attach block to region? */
    for (ptr_block = agp_blocks.next; ptr_block != NULL; ptr_block = ptr_block->next) {

        if (ptr_block->pg_start == pg_start) {

            /* We found it, check if size matches */
            if (ptr_block->page_count != pg_count) {
                MTX_ERROR("Page count doesn't match with AGP block\n");
                return;
            }
            
            /* Unlink block from list */
            if (ptr_block->next)
                ptr_block->next->prev = ptr_block->prev;
                
            if (ptr_block->prev)
                ptr_block->prev->next = ptr_block->next;
            
            break;
        }
    }

    if (!ptr_block) {
        MTX_ERROR("Pointer doesn't match with AGP blocks\n");
        return;
    }

    /* Unbind memory from AGP driver */
    ret = agp_backend->unbind_memory(ptr_block);

    if (ret < 0) {
        MTX_ERROR("Failed to unbind memory block (returned %d)\n", ret);
        return;
    }

    /* Free memory block */
    agp_backend->free_memory(ptr_block);
    
  #if MEMORY_STATS
    --alloc_agp_count;
    alloc_agp_mem -= size;
  #endif
}

/***************************************************************************************
 * Function:       mtx_mem_ioremap
 *
 * Description:    Map any memory to kernel memory space.
 *
 * Parameters:     offset           Start address of the mapping
 *                 size             Size of the mapping
 *
 * Return Value:   Ptr to mapped memory, NULL if failed
 *
 * Comments:
 *
 */
void*
mtx_mem_ioremap(unsigned long offset, unsigned long size)
{
    void *ptr = ioremap(offset, size);

    if (!ptr) return NULL;
    
  #if MEMORY_STATS
    ++ioremap_count;
  #endif

    return ptr;
}

/***************************************************************************************
 * Function:       mtx_mem_iounmap
 *
 * Description:    Free kernel memory map.
 *
 * Parameters:     ptr              Pointer to mapped memory
 *
 * Return Value:   None
 *
 * Comments:
 *
 */
void
mtx_mem_iounmap(void* ptr)
{
    if (!ptr) return;

    iounmap(ptr);
    
  #if MEMORY_STATS
    --ioremap_count;
  #endif
}

/***************************************************************************************
 * Function:       mtx_mem_alloc_region
 *
 * Description:    Its job is to allocate memory for a device region with
 *                 the specified size and type.
 *
 * Parameters:     dev              MTX device
 *                 size             amount of memory desired
 *                 type             type of memory
 *                 reg              if not NULL, init this region instead of allocating 
 *                                  a new one.
 *
 * Return Value:   a MTX memory region, NULL if failed
 *
 * Comments:       Caller is responsible of incrementing accordingly the region 
 *                 reference count. It is initialized 0, and must be 0 when
 *                 calling mtx_mem_free_region.
 */
mtx_region_t*
mtx_mem_alloc_region(mtx_device_t *dev, size_t size, mtx_memory_type type, mtx_region_t* reg) 
{
    mtx_region_t *region;
    
    if (!size)
    {
        MTX_ERROR("Allocating a zero sized memory region (type: %u))\n", type);
        return NULL;
    }

    if (!reg) { 
     
        /* Allocate a new region */
        region = MTXALLOC(sizeof(mtx_region_t));
        if (!region)
            return NULL;

        region->flags = MTX_REGION_DYNAMIC;
        
    } else {
        
        /* Init a already existing region */
        region = reg;
        region->flags = 0;
    }
    
    /* init fields */
    region->map_perms = MTX_MAP_NONE;
    region->io_flags  = 0;
    region->type      = type;
    region->size      = size;
    region->mtrr      = -1;
    region->pid       = 0;
    atomic_set(&region->ref_count, 0);

    switch (region->type) {

        case MTX_MEMTYPE_PCI_LOCKED:
                
            /* Allocate shared region in locked memory */
            region->kaddr = mtx_mem_alloc_pci(region->size);
            if (!region->kaddr) goto FAILED;

            /* Get physical and bus address */
            region->base    = __pa(region->kaddr);
            region->busaddr = virt_to_bus(region->kaddr);
            break;

        case MTX_MEMTYPE_SYS:

            /* Allocate standard system to this region */
            region->kaddr = mtx_mem_alloc(region->size);
            if (!region->kaddr) goto FAILED;
            
            /* Get physical and bus address */
            region->base    = __pa(region->kaddr);
            region->busaddr = 0; /* no PCI accesses for a system region */
            break;

        case MTX_MEMTYPE_AGP:

            /* Allocate AGP memory to this region */
            region->base = mtx_mem_alloc_agp(region->size);
            if (!region->base) goto FAILED;
            
            /* invalidate virtual addresses */
            region->kaddr   = NULL;
            region->busaddr = region->base;
            break;

        default:
            goto FAILED;
    }

    region->flags |= MTX_REGION_ALLOCATED;

    ADD_REGION_TO_LIST(dev, region);
    
    MTX_DEBUG("Created device buffer at 0x%08lx (size: %lu, type: %u)\n", 
              region->base, region->size, region->type);

    return region;

FAILED:

    MTX_ERROR("Failed to create device buffer (size: %u, type: %u)\n", size, type);
    mtx_mem_free_region(dev, region);

    return NULL;
}

/***************************************************************************************
 * Function:       mtx_mem_free_region
 *
 * Description:    This free memory allocated for a region.
 *
 * Parameters:     dev              MTX device
 *                 region           Region to free
 *
 * Return Value:   None.
 *
 * Comments:
 */
void
mtx_mem_free_region(mtx_device_t *dev, mtx_region_t *region)
{
    // Sanity check
    if (atomic_read(&region->ref_count) != 0) {
        
        MTX_DEBUG("WARNING: Freeing a buffer still in use (addr:0x%08lx, size: %lu, type: %u, ref: %i))\n", 
                  region->base, region->size, region->type, atomic_read(&region->ref_count));
    }

    if (region->mtrr >= 0) {
      #if defined(__i386__)
        /* release memory range */
        mtrr_del(region->mtrr, region->base, region->size);
      #endif
    }

    if (region->flags & MTX_REGION_ALLOCATED) {
        
        MTX_DEBUG("Freeing device buffer at 0x%08lx (size: %lu, type: %u)\n", 
                  region->base, region->size, region->type);

        DEL_REGION_FROM_LIST(dev, region);
        
        switch (region->type) {

            case MTX_MEMTYPE_PCI_LOCKED:

                /* Free pci locked memory */
                mtx_mem_free_pci(region->kaddr, region->size);
                break;

            case MTX_MEMTYPE_AGP:

                /* Free agp memory */
                if (region->kaddr)
                    mtx_mem_iounmap(region->kaddr);

                mtx_mem_free_agp(region->base, region->size);
                break;

            case MTX_MEMTYPE_SYS:

                /* Free kernel memory */
                mtx_mem_free(region->kaddr, region->size);
                break;

            case MTX_MEMTYPE_IO:

                /* unmap IO memory region */
                if (region->kaddr)
                    mtx_mem_iounmap(region->kaddr);
                break;

            default:
                break;
        }
    }
   
    if (region->flags & MTX_REGION_DYNAMIC) {

        /* Free region memory that has been allocated dynamically */
        MTXFREE(region, sizeof(mtx_region_t));

    } else {

        region->kaddr = NULL;
        region->base = 0L;
        region->size = 0L;
        region->pid = 0;
        region->flags &= ~MTX_REGION_ALLOCATED;
    }
    
    return;
}
