 /***************************************************************************\
|*                                                                           *|
|*       Copyright 2002 Matrox Graphics Inc. All rights reserved.            *|
|*                                                                           *|
|*     NOTICE TO THE USER: This source code is the copyrighted work of       *| 
|*     Matrox. Users of this source code are hereby granted a nonexclusive,  *|
|*     royalty-free copyright license to use this code in individual and     *| 
|*     commercial software.                                                  *|
|*                                                                           *|
|*     Any use of this source code must include, in the user documenta-      *|
|*     tion and internal comments to the code, notices to the end user       *|
|*     as follows:                                                           *|
|*                                                                           *|
|*       Copyright 2002 Matrox Graphics Inc. All rights reserved.            *|
|*                                                                           *|
|*     The source code is provided to you AS IS and WITH ALL FAULTS. Matrox  *|
|*     makes no representation and gives no warranty whatsoever, whether     *|
|*     express or implied, and without limitation, with regard to the        *|
|*     quality, safety, contents, performance, merchantability, non-         *|
|*     infringement or suitability for any particular or intended purpose    *|
|*     of this source code. In no event will Matrox be liable for any        *|
|*     direct, indirect, punitive, special, incidental or consequential      *|
|*     damages however they may arise and even if Matrox has been previously *|
|*     advised of the possibility of such damages.                           *|
|*                                                                           *|
 \***************************************************************************/
/***************************************************************************************\

File Name:      mtx_ioctl.c

Description:    IOCTLs implementation for MTX driver.

References:     None.

Author:         Karl Lessard    <klessard@matrox.com>

    Copyright (c) 2001, Matrox Graphics Inc.
    All Rights Reserved.

\***************************************************************************************/

#define __NO_VERSION__
#include "mtx_drv.h"

/***************************************************************************************
 * Function:       mtx_ioctl_get_memory_info
 *
 * Description:    Return important information about PCI resources and other
 *                 memory regions.
 *
 * Parameters:     priv                MTX private file data
 *                 cmd                 ioctl command
 *                 arg                 ioctl argument
 *                 
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:
 */
int mtx_ioctl_get_memory_info(mtx_file_t *priv, unsigned int cmd, unsigned long arg)
{
    mtx_device_t *dev = priv->dev;
    mtx_ioctl_get_memory_info_t mem_info;

    if (!dev) return -EINVAL;
    
    /* Fill up the ioctl structure with device information */
    mem_info.fb_base  = dev->framebuffer.base;
    mem_info.fb_size  = dev->framebuffer.size;
    mem_info.reg_base = dev->registers.base;
    mem_info.reg_size = dev->registers.size;

    if (agp_drv != NULL) {
        
        mem_info.agp_base = agp_drv->aperture_base;
        mem_info.agp_size = agp_drv->aperture_size;
        
    } else {
        
        mem_info.agp_base = 0;
        mem_info.agp_size = 0;
    }

    if ( copy_to_user((mtx_ioctl_get_memory_info_t *)arg, &mem_info, sizeof(mem_info)) )
        return -EFAULT;
        
    return 0;
}   

/***************************************************************************************
 * Function:       mtx_ioctl_get_shared_buffer
 *
 * Description:    Return the shared memory segment used internally by a user-space
 *                 driver for this device.  If no segment are allocated yet, allocate
 *                 them with the size information found in the ioctl argument. In
 *                 addition, a user-space driver can share data with the core kernel
 *                 driver by setting KERNEL_SHARE in the ioctl flags.
 *
 * Parameters:     priv                MTX private file data
 *                 cmd                 ioctl command
 *                 arg                 ioctl argument
 *                 
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:       Allocation of shared buffers can only be done by a root process.
 *                 NOTE: This IOCTL is now used as an helper for MTX_IOCTL_GET_BUFFER
 *                 method with a share type different than NONE.
 *
 *                 XXX: Since we can only match a buffer with its size, we cannot use
 *                      heap mechanisms for buffer allocation. Buffer will not be 
 *                      freed until process release it explicitely or on a dev cleanup.
 */
int mtx_ioctl_get_shared_buffer(mtx_file_t *priv, unsigned int cmd, unsigned long arg)
{
    mtx_ioctl_buffer_t buffer_info;
    mtx_device_t *dev = priv->dev;
    mtx_region_t *region = NULL;
    struct list_head *pos;

    if (!dev) return -EINVAL;

    if (copy_from_user(&buffer_info, (void*)arg, sizeof(buffer_info)))
        return -EFAULT;

    buffer_info.base = 0; /* be sure that base is zero */

    /* Check if an existing region match the requested one */
    list_for_each(pos, &dev->memory_regions)
    {
        region = list_entry(pos, mtx_region_t, list_node);

        if ((region) &&
            (region->flags & MTX_REGION_SHARED) &&
            (region->size == buffer_info.size)) {
                
            buffer_info.base = region->base;
            break;
        }
    }
    
    if (!buffer_info.base) {

        /* We'll need to allocate the region, check if we have permissions */
        if (!capable(CAP_SYS_ADMIN))
            return -EPERM;

        if ((buffer_info.type != MTX_MEMTYPE_SYS) && (buffer_info.type != MTX_MEMTYPE_PCI_LOCKED))
            return -EINVAL; /* Only those two types can be added by a process */
        
        region = mtx_mem_alloc_region(dev, buffer_info.size, buffer_info.type, NULL);
        if (!region)
            return -ENOMEM;
        
        region->flags |= MTX_REGION_SHARED;
        region->map_perms = MTX_MAP_AUTH;

        /* copy memory base for user */
        buffer_info.base = region->base;
    }

    /* Increment reference count of this buffer */
    atomic_inc(&region->ref_count);

    if (copy_to_user((void*)arg, &buffer_info, sizeof(buffer_info)))
        return -EFAULT;
    
    return 0;
}

/***************************************************************************************
 * Function:       mtx_ioctl_release_shared_buffer
 *
 * Description:    Release a reference to a shared buffer.
 *
 * Parameters:     priv                MTX private file data
 *                 cmd                 ioctl command
 *                 arg                 ioctl argument
 *                 
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:       None.
 */
int mtx_ioctl_release_shared_buffer(mtx_file_t *priv, unsigned int cmd, unsigned long arg)
{
    mtx_ioctl_buffer_t buffer_info;
    mtx_device_t *dev = priv->dev;
    mtx_region_t *region;

    if (!dev) return -EINVAL;

    if (copy_from_user(&buffer_info, (void*)arg, sizeof(buffer_info)))
        return -EFAULT;

    /* find region from base address */
    region = mtx_find_region_from_base(dev, buffer_info.base);
    if (!region)
        return -EINVAL;

    /* perform some checks */
    if ((region->size != buffer_info.size) ||
        /*(region->type != buffer_info.type) || FIXME since we don't support system shared buffer */
        !(region->flags & MTX_REGION_SHARED)) {
        
        MTX_ERROR("Releasing invalid shared buffer at 0x%08lx\n", region->base);
        return -EINVAL;
    }

    /* decrement and read new ref_count */
    if (atomic_dec_and_test(&region->ref_count)) {
        
        /* last reference to this region, free it */
        mtx_mem_free_region(dev, region);
    }    

    return 0;
}

/***************************************************************************************
 * Function:       mtx_ioctl_alloc_buffer
 *
 * Description:    Return a buffer of the specified type to user-space. This ioctl
 *                 handles all types of system memory, and should be used by any
 *                 process who wants to manage itself its device buffers (we call
 *                 them 'private' buffers)
 *
 * Parameters:     priv                MTX private file data
 *                 cmd                 ioctl command
 *                 arg                 ioctl argument
 *                 
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:       Only for authenticated processes.
 */
int mtx_ioctl_alloc_buffer(mtx_file_t *priv, unsigned int cmd, unsigned long arg)
{
    mtx_device_t *dev = priv->dev;
    mtx_ioctl_buffer_t buffer_info;
    mtx_region_t *region;

    if (!dev) return -EINVAL;

    if (copy_from_user(&buffer_info, (void*)arg, sizeof(buffer_info)))
        return -EFAULT;

    if ((buffer_info.type < 0) || (buffer_info.type > MTX_MEMTYPE_COUNT))
        return -EINVAL;

    /* Allocate new context region */
    region = mtx_mem_alloc_region(dev, buffer_info.size, buffer_info.type, NULL);
    if (!region) {
        MTX_ERROR("Fail to allocate buffer of type %u for process %u\n", 
                   buffer_info.type, current->pid);
        return -ENOMEM;
    }

    region->pid = current->pid;
    region->map_perms = MTX_MAP_OWNER;
    
    /* Add this region to the context list */
    list_add_tail(&region->ctx_list_node, &priv->ctx->region_list);
    
    /* copy memory base for user */
    buffer_info.base = region->base;

    if (copy_to_user((void*)arg, &buffer_info, sizeof(buffer_info)))
        return -EFAULT;
    
    return 0;
}

/***************************************************************************************
 * Function:       mtx_ioctl_free_buffer
 *
 * Description:    Free a buffer previously allocated. If the buffer is a shared
 *                 buffer, and this was the last reference it, the buffer is
 *                 deleted.
 *
 * Parameters:     priv                MTX private file data
 *                 cmd                 ioctl command
 *                 arg                 ioctl argument
 *                 
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:       
 */
int mtx_ioctl_free_buffer(mtx_file_t *priv, unsigned int cmd, unsigned long arg)
{
    mtx_device_t *dev = priv->dev;
    mtx_ioctl_buffer_t buffer_info;
    mtx_region_t *region;

    if (!dev) return -EINVAL;

    if (copy_from_user(&buffer_info, (void*)arg, sizeof(buffer_info)))
        return -EFAULT;

    /* find region from base address */
    region = mtx_find_region_from_base(dev, buffer_info.base);
    if (!region)
        return -EINVAL;
    
    MTX_DEBUG("Releasing buffer at 0x%08lx\n", region->base);

    /* perform some additionnal checks */
    if ((region->size != buffer_info.size) ||
        (region->type != buffer_info.type) ||
        (region->flags & MTX_REGION_SHARED))
        return -EINVAL;

    list_del(&region->ctx_list_node);
    mtx_mem_free_region(dev, region);
    
    return 0;
}

/***************************************************************************************
 * Function:       mtx_ioctl_busmastering_test
 *
 * Description:    Test busmastering on current system configuration and return
 *                 a set of flags of what is supported.
 *
 * Parameters:     priv                MTX private file data
 *                 cmd                 ioctl command
 *                 arg                 ioctl argument
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:
 */
int mtx_ioctl_busmastering_test(mtx_file_t *priv, unsigned int cmd, unsigned long arg)
{
    mtx_device_t *dev = priv->dev;
    unsigned long flags = 0;

    if (!dev) return -EINVAL;

    if ((dev->pci_dev->bus->number == 0) && (PCI_FUNC(dev->pci_dev->devfn) == 0)) {

        if (PCI_SLOT(dev->pci_dev->devfn) == 4)
            flags |= AGPINFO_FLAG_FOUND_CHIPSET_ISA;
        else
            flags |= AGPINFO_FLAG_FOUND_CHIPSET;
    }

    if (dev->agp_cap_offset != -1) {

        u32 agp_command;

        // We have an AGP board
        flags |= AGPINFO_FLAG_AGPBOARD | AGPINFO_FLAG_FOUND_CHIPSET_AGP;
        
        pci_read_config_dword(dev->pci_dev, dev->agp_cap_offset + PCI_AGP_COMMAND, &agp_command);
        
        if ((agp_command & PCI_AGP_COMMAND_FW) &&
            (agp_command & (PCI_AGP_COMMAND_RATE2 | PCI_AGP_COMMAND_RATE4))) {

            // Enable fast write
            flags |= AGPINFO_FLAG_FASTWRITE;
        }

        if (agp_drv && !mtx_agp_test_device(dev)) {
        
            u16 agp_vendor;
            
            /* AGP test passed, set flags accordingly */
            flags |= AGPINFO_FLAG_HYBRID_CAP;
       
            /* Set flags specific to AGP chipset vendor */
            pci_read_config_word(agp_drv->kern_info.device, PCI_VENDOR_ID, &agp_vendor);

            switch (agp_vendor) {
            
                case PCI_VENDOR_ID_SERVERWORKS:
                    flags &= ~AGPINFO_FLAG_HYBRID_CAP;
                    flags |= AGPINFO_FLAG_AGPDELAY;
                    break;

                case PCI_VENDOR_ID_VIA:

                    flags |= AGPINFO_FLAG_AGPSERIALIZE;

                    switch (agp_drv->kern_info.chipset) {

                        case VIA_APOLLO_KX133:
                        case VIA_APOLLO_KT133:
                            flags |= AGPINFO_FLAG_1XAFTERRESET;
                            break;

                        default:
                            break;
                
                    } /* fall through... */

                case PCI_VENDOR_ID_SI:
                    flags |= AGPINFO_FLAG_AGPSERIALIZE;
                    break;

                case PCI_VENDOR_ID_INTEL:
                    if (agp_drv->kern_info.chipset == INTEL_I840) 
                        flags |= AGPINFO_FLAG_PCIALIGN64;
                    break;

                default:
                    break;
            }

            /* AGP works fine for this device, set flags accordingly */
            flags |= AGPINFO_FLAG_AGPMEMORY | AGPINFO_FLAG_AGPXFER;
        }
        
    } else {

        /* We have a PCI board */
        flags |= AGPINFO_FLAG_PCIBOARD;
    }

    /* Set default PCI flags */
    flags |= AGPINFO_FLAG_PCIXFER | AGPINFO_FLAG_PCIMEMORY;
    
    if (copy_to_user((void*)arg, &flags, sizeof(unsigned long)))
        return -EFAULT;
    
    return 0;
}

/***************************************************************************************
 * Function:       mtx_ioctl_add_context
 *
 * Description:    Create and register a new device context for this process, and 
 *                 return an unique identification number for it.
 *
 * Parameters:     priv                MTX private file data
 *                 cmd                 ioctl command
 *                 arg                 ioctl argument
 *                 
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:       
 */
int mtx_ioctl_add_context(mtx_file_t *priv, unsigned int cmd, unsigned long arg)
{
    mtx_device_t *dev = priv->dev;
    mtx_context_t *ctx = NULL;

    /* Only one context per file */
    if (priv->ctx != NULL)
        return -EBUSY;

    /* Create a new context record for this file */
    ctx = mtx_ctx_create(dev);
    if (!ctx) {
        MTX_ERROR("Failed to add new context for device %u:%u\n", 
                  dev->pci_dev->bus->number, dev->pci_dev->devfn);
        return -EINVAL;
    }

    /* Add context to our file */
    priv->ctx = ctx;

    /* Copy context id to ioctl argument */
    if (copy_to_user((void*)arg, &(ctx->unique_id), sizeof(ctx->unique_id)))
        return -EFAULT;

    return 0;
}

/***************************************************************************************
 * Function:       mtx_ioctl_remove_context
 *
 * Description:    Unregister and delete a device context previously registered for 
 *                 this process.
 *
 * Parameters:     priv                MTX private file data
 *                 cmd                 ioctl command
 *                 arg                 ioctl argument
 *                 
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:       
 */
int mtx_ioctl_remove_context(mtx_file_t *priv, unsigned int cmd, unsigned long arg)
{
    mtx_device_t *dev = priv->dev;
    mtx_context_t *ctx;
    int ctx_id;

    if (copy_from_user(&ctx_id, (void*)arg, sizeof(ctx_id)))
        return -EFAULT;

    /* Get context record */
    ctx = priv->ctx;
    if (!ctx || (ctx->unique_id != ctx_id))
        return -EINVAL;

    /* Remove context from file */
    priv->ctx = NULL;

    /* Unregister and free context */
    return mtx_ctx_delete(dev, ctx);
}

/********************** FOR DEBUGGING PURPOSE ONLY ***************************
 * 
 * Function:       mtx_ioctl_hard_reset
 *
 * Description:    Do a hard reset of the module use count, so we can unload
 *                 the module if something bad happened.
 *
 * Parameters:     priv                MTX private file data
 *                 cmd                 ioctl command
 *                 arg                 ioctl argument
 *                 
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:       
 */
int mtx_ioctl_hard_reset(mtx_file_t *priv, unsigned int cmd, unsigned long arg)
{
#if DEBUG
    MTX_DEBUG("DOING HARD RESET! (current_mod_use = %u)\n", MOD_IN_USE);
    
    while (MOD_IN_USE > 1) 
        MOD_DEC_USE_COUNT; 
#endif
    return 0;
}
 
