 /***************************************************************************\
|*                                                                           *|
|*       Copyright 2002 Matrox Graphics Inc. All rights reserved.            *|
|*                                                                           *|
|*     NOTICE TO THE USER: This source code is the copyrighted work of       *| 
|*     Matrox. Users of this source code are hereby granted a nonexclusive,  *|
|*     royalty-free copyright license to use this code in individual and     *| 
|*     commercial software.                                                  *|
|*                                                                           *|
|*     Any use of this source code must include, in the user documenta-      *|
|*     tion and internal comments to the code, notices to the end user       *|
|*     as follows:                                                           *|
|*                                                                           *|
|*       Copyright 2002 Matrox Graphics Inc. All rights reserved.            *|
|*                                                                           *|
|*     The source code is provided to you AS IS and WITH ALL FAULTS. Matrox  *|
|*     makes no representation and gives no warranty whatsoever, whether     *|
|*     express or implied, and without limitation, with regard to the        *|
|*     quality, safety, contents, performance, merchantability, non-         *|
|*     infringement or suitability for any particular or intended purpose    *|
|*     of this source code. In no event will Matrox be liable for any        *|
|*     direct, indirect, punitive, special, incidental or consequential      *|
|*     damages however they may arise and even if Matrox has been previously *|
|*     advised of the possibility of such damages.                           *|
|*                                                                           *|
 \***************************************************************************/
/***************************************************************************************\

File Name:      mtx_dev.c

Description:    MTX devices manipulation. Supported devices detected on the PCI buses
                are stored in a global list of mtx_dev.

References:     None.

Author:         Karl Lessard    <klessard@matrox.com>

    Copyright (c) 2001, Matrox Graphics Inc.
    All Rights Reserved.

\***************************************************************************************/

#define __NO_VERSION__
#include "mtx_drv.h"


/******************************************
 *
 * Global variables
 *
 *****************************************/

static LIST_HEAD(mtx_devices);

#if DEBUG
/***************************************************************************************
 * Function:       mtx_dev_dump_regions
 *
 * Description:    Dump list of allocated regions for this device
 *
 * Parameters:     dev           MTX device to init
 *  
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:       None.
 */
static void
mtx_dev_dump_regions(mtx_device_t *dev)
{
    unsigned int total_dumped = 0;
    struct list_head *pos;
    mtx_region_t *region;
    
    MTX_PRINT("Dump device regions\n");
        
    list_for_each(pos, &dev->memory_regions)
    {
        region = list_entry(pos, mtx_region_t, list_node);
       
        if (region) {

            /* Dump region to screen */
            MTX_PRINT("\tRegions: base 0x%08lx, kaddr %p, size %lu, type %u\n",
                       region->base, region->kaddr, region->size, region->type);

            total_dumped++;
        }
    }

    MTX_PRINT("%u regions dumped\n", total_dumped);
}
#endif

/***************************************************************************************
 * Function:       mtx_dev_probe
 *
 * Description:    Add an MTX device from a PCI device previously detected and
 *                 supported by this driver.
 *
 * Parameters:     pci_dev          Supported device probed by the PCI driver.
 *                 pci_id           PCI id for this device. 
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:       This method is called via the (*probe) ptr in pci_driver 
 *                 structure, and is called when the PCI driver detected a 
 *                 new device supported by our hardware.
 */
int 
mtx_dev_probe(struct pci_dev *pci_dev, const struct pci_device_id *pci_id)
{
    mtx_device_t *dev;
    
    /* Allocate a new device structure */
    dev = (mtx_device_t *) MTXALLOC(sizeof(mtx_device_t));
    if (!dev)
        return -ENOMEM;

    /* Initialize some device data */
    dev->pci_dev  = pci_dev;
    dev->pci_id   = pci_id;
    spin_lock_init(&dev->count_lock);
    
    /* Add reference to this device in global device list */
    list_add_tail(&dev->list_node, &mtx_devices);
  
    /* Print some useful information */
    MTX_INFO("0x%04x(sub:0x%04x) board found at %02x:%02x.%1x\n", 
              dev->pci_id->device, 
              dev->pci_id->subdevice, 
              dev->pci_dev->bus->number, 
              PCI_SLOT(dev->pci_dev->devfn), 
              PCI_FUNC(dev->pci_dev->devfn));

    /* If this device is connected to AGP port, acquire control of AGP backend */
    if ((dev->agp_cap_offset = pci_find_capability(pci_dev, PCI_CAP_ID_AGP)) != 0) {

        if (agp_drv && (mtx_agp_acquire(dev) < 0)) {
            
            MTX_WARNING("Fail to acquire control of AGP driver. "
                        "AGP tranfers are therefore disabled\n");
        }
        
    } else {

        /* invalidate cap offset */
        dev->agp_cap_offset = -1;
    }

    return 0;
}

/***************************************************************************************
 * Function:       mtx_dev_remove
 *
 * Description:    Cleanup/Remove a device previously initialized.
 *
 * Parameters:     None.
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:
 */
void 
mtx_dev_remove(struct pci_dev *pci_dev)
{
    mtx_device_t *dev;

    /* Get MTX device from PCI bus id */
    dev = mtx_dev_find(pci_dev->bus->number, PCI_SLOT(pci_dev->devfn));
    if (!dev) return;
    
    MTX_DEBUG("removing 0x%04x at %02x:%02x.%x\n", 
               dev->pci_id->device, 
               dev->pci_dev->bus->number, 
               PCI_SLOT(dev->pci_dev->devfn), 
               PCI_FUNC(dev->pci_dev->devfn));
        
    /* Release AGP backend if we are connected to AGP port */
    mtx_agp_release(dev);
    
    /* Remove from device list */
    list_del(&dev->list_node);
    
    /* Free device memory */
    MTXFREE(dev, sizeof(*dev));
}

/***************************************************************************************
 * Function:       mtx_dev_find
 *
 * Description:    Find a MTX device from its PCI ID.
 *
 * Parameters:     pci_bus          PCI bus number
 *                 pci_slot         PCI slot number
 *
 * Return Value:   Pointer to an MTX device, NULL if there's no matching device
 *
 * Comments:       None.
 */
mtx_device_t* 
mtx_dev_find(unsigned int pci_bus, unsigned int pci_slot)
{
    mtx_device_t* dev;
    struct list_head *pos;

    /* Check for a matching device in the device table */ 
    list_for_each(pos, &mtx_devices)
    {
        dev = list_entry(pos, mtx_device_t, list_node);
        
        if (dev->pci_dev->bus->number == pci_bus && 
            PCI_SLOT(dev->pci_dev->devfn) == pci_slot)
        {
            /* We've found a matching device, return it! */
            return dev;
        }
    }

    /* No matches found */
    return NULL;
}

/***************************************************************************************
 * Function:       mtx_dev_init
 *
 * Description:    Init a MTX device properly to be used by the driver. PCI memory
 *                 mapped regions are initialized in this method.
 *
 * Parameters:     dev            MTX device to init
 *
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:       None.
 */
int
mtx_dev_init(mtx_device_t *dev)
{
    int ret;
    
    /* Init our private data */
    sema_init(&dev->semaphore, 1);

    INIT_LIST_HEAD(&dev->memory_regions);
    
    /* Init PCI registers region */
    READ_PCI_REGION(dev, dev->registers, 1);
    
    dev->registers.kaddr     = mtx_mem_ioremap(dev->registers.base, dev->registers.size);
    dev->registers.flags     = MTX_REGION_ALLOCATED;
    dev->registers.map_perms = MTX_MAP_AUTH;
    dev->registers.type      = MTX_MEMTYPE_IO;
    dev->registers.mtrr      = -1;
    
    ADD_REGION_TO_LIST(dev, &dev->registers);

    MTX_DEBUG("Registers at 0x%08lx, size: %luK, flags: %lu, knl_addr: 0x%08x\n", 
               dev->registers.base, dev->registers.size/1024, 
               dev->registers.io_flags, (u32)dev->registers.kaddr);
    
    /* Init PCI framebuffer region */
    READ_PCI_REGION(dev, dev->framebuffer, 0);
    
    dev->framebuffer.kaddr      = mtx_mem_ioremap(dev->framebuffer.base, dev->framebuffer.size);
    dev->framebuffer.flags      = MTX_REGION_ALLOCATED;
    dev->framebuffer.map_perms  = MTX_MAP_AUTH;
    dev->framebuffer.type       = MTX_MEMTYPE_IO;
    dev->framebuffer.mtrr       = -1;
        
  #if defined(__i386__)
    /* try to set write-combining transfer type to framebuffer */
    dev->framebuffer.mtrr = 
        mtrr_add(dev->framebuffer.base, dev->framebuffer.size, MTRR_TYPE_WRCOMB, 1);
  #endif
    
    ADD_REGION_TO_LIST(dev, &dev->framebuffer);
    
    MTX_DEBUG("Framebuffer at 0x%08lx, size: %luM, flags: %lu, knl_addr: 0x%08x, "
              "write-combining: %s\n", dev->framebuffer.base, dev->framebuffer.size/1024/1024, 
               dev->framebuffer.io_flags, (u32)dev->framebuffer.kaddr,
               ((dev->framebuffer.mtrr < 0) ? "NO" : "YES"));
    
    /* If this device is the AGP one, enable AGP point-to-point connection */
    if (mtx_agp_enable(dev) == 0) {

        MTX_DEBUG("AGP aperture at 0x%08lx, size: %luK, rate: %dX\n, write-combining: %s\n", 
                  agp_drv->aperture_base, agp_drv->aperture_size / 1024L, agp_drv->rate,
                  ((agp_drv->mtrr < 0) ? "NO" : "YES"));
        
    } else if (dev->agp_cap_offset != -1) {

        MTX_ERROR("Cannot initialized AGP device\n");
        return -EINVAL;
    }

    /* Allocate bitmap for context id's */
    dev->context_bitmap = MTXALLOC(MAX_CONTEXTS_PER_DEVICE/8);
    if (!dev->context_bitmap)
        return -ENOMEM;
    
    /* Check which board we are driving and attach its core driver */
    switch (dev->pci_dev->device) {

        case PCI_DEVICE_ID_PARHELIA_512:
            dev->core = &mtx_parhl_driver;
            break;

        default:
            /* No core driver for this device */
            break;
    }
    
  #ifdef MTX_WITH_MEMORY_HEAPS
    /* Initialize memory heaps */
    ret = mtx_heap_init(dev);

    if (ret < 0) {
        MTX_ERROR("Failed to initialize memory heaps\n");
        return ret;
    }
  #endif

    if (dev->core && dev->core->enable)
    {
        /* Enable device from core driver */
        ret = dev->core->enable(dev);

        if (ret < 0) {
            MTX_ERROR("Core driver fail to initialize device 0x%04x\n", dev->pci_dev->device);
            return ret;
        }
    }

    return 0;
}

/***************************************************************************************
 * Function:       mtx_dev_cleanup
 *
 * Description:    Cleanup data MTX device created during initialization.
 *
 * Parameters:     dev           MTX device to init
 *  
 * Return Value:   0 if succeed, errno otherwise.
 *
 * Comments:       None.
 */
int
mtx_dev_cleanup(mtx_device_t *dev)
{
    struct list_head *pos, *tmppos;
    mtx_region_t *region;

    /* Disable device in core driver */
    if (dev->core && dev->core->disable)
        dev->core->disable(dev);
    
  #ifdef MTX_WITH_MEMORY_HEAPS
    /* Cleanup memory heaps */
    mtx_heap_cleanup(dev);
  #endif
    
    /* Delete context id bitmap */
    if (dev->context_bitmap)
        MTXFREE(dev->context_bitmap, (MAX_CONTEXTS_PER_DEVICE/8));

  #if 0 //DEBUG
    /* Dump memory regions list (should be empty) */
    mtx_dev_dump_regions(dev);
  #endif
    
    /* Free any remaining memory regions */
    list_for_each_safe(pos, tmppos, &dev->memory_regions)
    {
        region = list_entry(pos, mtx_region_t, list_node);
       
        if (region) {

            /* Free memory attached to this region */
            mtx_mem_free_region(dev, region);
        }
    }

    /* Empty region list */
    INIT_LIST_HEAD(&dev->memory_regions);

    /* If this device is the AGP one, disable AGP point-to-point connection */
    mtx_agp_disable(dev);

    return 0;
}
