/********************************************************
*  mgaXconfig.c
*
*  Source file for Matrox video parameters tuning utility.
*  Functions accessing/modifying XF86ConfigRec struct.
*
*  Copyright (C) 2001, Matrox Graphics Inc.
*
*  Author : Luugi Marsan and Stephane Duguay
*  Last modified : April 2001.
*********************************************************/


#include "io.h"
#include "mgaXconfig.h"


static XF86ConfScreenPtr getScreenFromIndex(XF86ConfigPtr conf, int screenIndex);
static int getIndexFromScreen(XF86ConfigPtr conf, char* screen);
static BOOL replaceString(char** dest, char* src);
static XF86OptionPtr removeOption(XF86OptionPtr pOptionsList, char* OptionType);
void MsgBox(char* title, char* msg, int exit);


static BOOL replaceString(char** dest, char* src)
{


	free(*dest); /* even if NULL, because free don't do nothing if parameter is NULL */


	if(!src)
	{
		*dest = NULL;
		return TRUE;
	}
	else
	{


		*dest = (char*) calloc(strlen(src) + 1, sizeof(char));


		if(!*dest) return FALSE;


		strcpy(*dest, src);


		return TRUE;
	}
}


static XF86OptionPtr removeOption(XF86OptionPtr pOptionsList, char* OptionType)
{
	XF86OptionPtr pPrevOption;
	XF86OptionPtr pOption;
	XF86OptionPtr pRetour;

	if (!pOptionsList) return NULL;

	pOption = pOptionsList;
	pPrevOption = pOption;


	while (pOption)
	{
		if (!(strcmp(pOption->opt_name, OptionType)))
		{
			/* Found option to be deleted */
			/* Re-link the list around it */
			if ( pOption == pOptionsList)
				pRetour = pOption->list.next; /* We're deleting the head */
			else
			{
				pPrevOption->list.next = pOption->list.next;
				pRetour = pOptionsList;
			}

			free(pOption);
			return pRetour;
		}

		pPrevOption = pOption;
		pOption = pOption->list.next;


	}


	return pOptionsList;


}


/* This routine outputs the index of the screen string supplied by the
 * calling routine. It traveses the Screen structure and searches for
 * the string*/


static int getIndexFromScreen(XF86ConfigPtr conf, char* screen){


	int index;
	XF86ConfAdjacencyPtr pAdj;

	pAdj = conf->conf_layout_lst->lay_adjacency_lst;

	index = 0;

	/* Loop until we find the screen*/ 
	while (pAdj)
	{
#ifdef DEBUG
		printf("Comparing %s with %s\n",pAdj->adj_screen_str,screen);
#endif

		if (strcmp(pAdj->adj_screen_str, screen)== 0)
		{
#ifdef DEBUG
			printf("Found the screen. Return index %d\n",index);
#endif
			return index;
		}

		pAdj = pAdj->list.next;     
		index++; /* Increment the index*/
	}

	/* Returns -1 if the screen is not found */
#ifdef DEBUG
	printf("getIndexFromScreen: \"%s\" was not found\n", screen);
#endif


	return -1;
}


static XF86ConfScreenPtr getScreenFromIndex(XF86ConfigPtr conf, int screenIndex)
{
	int i = 0;
	XF86ConfScreenPtr pConfScreen;
	XF86ConfAdjacencyPtr pConfAdj = conf->conf_layout_lst->lay_adjacency_lst;


#ifdef DEBUG
	printf("getScreenFromIndex(%d)",screenIndex); 
#endif

	if((screenIndex < 0) || !(conf)) return NULL;


	pConfScreen = pConfAdj->adj_screen;


	while(i != screenIndex)
	{
		pConfAdj = pConfAdj->list.next;


		if(!pConfAdj) return NULL;

		pConfScreen = pConfAdj->adj_screen;
		i++;
	}


	return pConfScreen;


}



/* Opens the XF86Config file and returns the pointer to the structure
 * of the config file
 */ 
XF86ConfigPtr getConfigPtr(const char* filename)
{


	XF86ConfigPtr conf;

	if (!filename) return NULL;


	filename = xf86openConfigFile ("%A", filename, NULL);


	if (filename)
	{
		printf("Matrox PowerDesk: Configuration file (%s) opened.\n", filename);
	}
	else
	{
		printf ("Matrox PowerDesk: Couldn't open configuration file.\n");
		return NULL;
	}


	if ((conf = xf86readConfigFile ()) == NULL)
	{
		fprintf (stderr, "Matrox PowerDesk: Couldn't parse the configuration file.\n");
		xf86closeConfigFile();
		return NULL;
	}
	else
	{
		fprintf (stderr, "Matrox PowerDesk: Configuration file parsed.\n");
	}


	/* Close Config file */
	xf86closeConfigFile();

	replaceString(&(conf->conf_layout_lst->lay_identifier), "Matrox PowerDesk configured.");


	return conf;
}


/* Write Conf file */
BOOL confWriteConfigFile( char* filename,XF86ConfigPtr conf)
{
	if (!conf) return FALSE;


	if(xf86writeConfigFile (filename, conf))
	{
		/* free the config struct */
		xf86freeConfig(conf);
		return TRUE;
	}
	else
	{
		xf86freeConfig(conf);
		return FALSE;
	}


}


/* Inverts the Screens Layout. It assumes only 2 monitors */
XF86ConfigPtr confSetLayout (XF86ConfigPtr conf, int where)
{
	XF86ConfAdjacencyPtr aptr;

	if (!conf) return NULL;


	aptr = conf->conf_layout_lst->lay_adjacency_lst;
	aptr->adj_where = where;


	return conf;
}


/* Enable Dual Head
 * Adds a second Screen and Device section if it's not done already
 *  
 * 1. it will check to see how many devices that you have
 * 2. add another device Screen section and a Second device section
 * 3. Create a Screen struct and link it the rest
 */


XF86ConfigPtr confEnableDualHead ( XF86ConfigPtr conf, int where)
{
	XF86ConfScreenPtr pScrn, pNewScrn;
	XF86ConfDevicePtr pDevice, pNewDevice;
	XF86ConfMonitorPtr pMonitor, pNewMonitor;
	XF86ConfLayoutPtr pLayout;
	XF86ConfAdjacencyPtr pAdj, pNewAdj;
	int i;


	if (!conf)
	{
		printf("Matrox PowerDesk: Configuration problem encountered.\n");
		return NULL;
	}


	if(confDualHeadIsActive(conf)) return conf;


	pScrn = conf->conf_screen_lst;
	pDevice = conf->conf_device_lst;
	pMonitor = conf->conf_monitor_lst;


	pLayout = conf->conf_layout_lst;
	pAdj = pLayout->lay_adjacency_lst;


	pNewScrn = pScrn->list.next;
	pNewDevice = pDevice->list.next;
	pNewMonitor = pMonitor->list.next;


	if (!pNewMonitor) /* Only one monitor */
	{
		/* Rename first monitor */
		replaceString(&pMonitor->mon_identifier,"Display 1");

		/* add a new monitor */
		pNewMonitor = calloc(1, sizeof(XF86ConfMonitorRec));
		pNewMonitor->list.next = NULL;


		replaceString(&pNewMonitor->mon_identifier,"Display 2");
		replaceString(&pNewMonitor->mon_vendor, pMonitor->mon_vendor);
		replaceString(&pNewMonitor->mon_modelname, pMonitor->mon_modelname);
		pNewMonitor->mon_width = pMonitor->mon_width;
		pNewMonitor->mon_height = pMonitor->mon_height;
		pNewMonitor->mon_modeline_lst = NULL;


		pNewMonitor->mon_n_hsync = pMonitor->mon_n_hsync;
		for(i = 0; i < pMonitor->mon_n_hsync; i++) pNewMonitor->mon_hsync[i] = pMonitor->mon_hsync[i];

		pNewMonitor->mon_n_vrefresh = pMonitor->mon_n_vrefresh;
		for(i = 0; i < pMonitor->mon_n_vrefresh; i++) pNewMonitor->mon_vrefresh[i] = pMonitor->mon_vrefresh[i];

		pNewMonitor->mon_gamma_red = pMonitor->mon_gamma_red;
		pNewMonitor->mon_gamma_green = pMonitor->mon_gamma_green;
		pNewMonitor->mon_gamma_blue = pMonitor->mon_gamma_blue;
		pNewMonitor->mon_option_lst = NULL;
		pNewMonitor->mon_modes_sect_lst = NULL;
		pNewMonitor->mon_comment = NULL;

		/* Add it the list */
		conf->conf_monitor_lst = (XF86ConfMonitorPtr) xf86addListItem ( (GenericListPtr) \
				pMonitor, (GenericListPtr) pNewMonitor);
	}
	else
	{
		/* There is at least 2 monitors */
		/* Rename first monitor */
		replaceString(&pMonitor->mon_identifier ,"Display 1");

		/* second monitor : Call it "Display 2" */
		replaceString(&pNewMonitor->mon_identifier ,"Display 2");
	}


	if (!pNewDevice) /* Only one device */
	{
		/* For the device */
		/* Rename first device*/
		replaceString(&pDevice->dev_identifier,"MATROX CARD 1");
		replaceString(&pDevice->dev_driver, "mga");

		/* add a new device */
		pNewDevice = calloc(1, sizeof(XF86ConfDeviceRec));

		pNewDevice->list.next = NULL;
		replaceString(&pNewDevice->dev_identifier,"MATROX CARD 2");
		replaceString(&pNewDevice->dev_vendor, pDevice->dev_vendor);
		replaceString(&pNewDevice->dev_board, pDevice->dev_board);
		replaceString(&pNewDevice->dev_chipset, pDevice->dev_chipset);
		replaceString(&pNewDevice->dev_busid, pDevice->dev_busid);
		replaceString(&pNewDevice->dev_card, pDevice->dev_card);
		replaceString(&pNewDevice->dev_driver, pDevice->dev_driver);
		replaceString(&pNewDevice->dev_ramdac, pDevice->dev_ramdac);
		replaceString(&pNewDevice->dev_clockchip, pDevice->dev_clockchip);
		for(i = 0; i < 4; i++) pNewDevice->dev_dacSpeeds[i] = pDevice->dev_dacSpeeds[i];
		pNewDevice->dev_videoram = pDevice->dev_videoram;
		pNewDevice->dev_textclockfreq = pDevice->dev_textclockfreq;
		pNewDevice->dev_videoram = pDevice->dev_videoram;
		pNewDevice->dev_bios_base = pDevice->dev_bios_base;
		pNewDevice->dev_mem_base = pDevice->dev_mem_base;
		pNewDevice->dev_io_base = pDevice->dev_io_base;
		pNewDevice->dev_clocks = pDevice->dev_clocks;
		for(i = 0; i < pDevice->dev_clocks; i++) pNewDevice->dev_clock[i] = pDevice->dev_clock[i];
		pNewDevice->dev_chipid = pDevice->dev_chipid;
		pNewDevice->dev_chiprev = pDevice->dev_chiprev;
		pNewDevice->dev_irq = pDevice->dev_irq;


		pNewDevice->dev_comment = NULL;
		pNewDevice->dev_option_lst = NULL;
		pNewDevice->dev_screen = 1;

		/* Add it the list */
		conf->conf_device_lst = (XF86ConfDevicePtr) xf86addListItem ( (GenericListPtr) \
				pDevice, (GenericListPtr) pNewDevice);
	}
	else
	{
		/* There is at least 2 devices */
		/* Rename first device*/
		replaceString(&pDevice->dev_identifier ,"MATROX CARD 1");
		replaceString(&pDevice->dev_driver, "mga");

		/* second device : Call it "MATROX CARD 2" */
		pNewDevice->dev_screen = 1;
		replaceString(&pNewDevice->dev_identifier ,"MATROX CARD 2");
		replaceString(&pNewDevice->dev_driver, "mga");
	}


	pDevice->dev_option_lst = removeOption(pDevice->dev_option_lst, "NoHal");
	pNewDevice->dev_option_lst = removeOption(pNewDevice->dev_option_lst, "NoHal");

	if (!pNewScrn) /* only one screen */
	{
		XF86ConfDisplayPtr pDisplay;
		XF86ModePtr pConfMode;

		/* Rename the first identifier of Screens*/
		replaceString(&pScrn->scrn_identifier , "Display 1");
		replaceString(&pScrn->scrn_device_str, pDevice->dev_identifier);
		replaceString(&pScrn->scrn_monitor_str, pMonitor->mon_identifier);

		/* Add a new screen */
		pNewScrn = calloc(1, sizeof(XF86ConfScreenRec));


		pNewScrn->list.next = NULL;


		replaceString(&pNewScrn->scrn_identifier, "Display 2");
		replaceString(&pNewScrn->scrn_obso_driver, pScrn->scrn_obso_driver);

		pNewScrn->scrn_defaultdepth = pScrn->scrn_defaultdepth;
		pNewScrn->scrn_defaultbpp = pScrn->scrn_defaultbpp;
		pNewScrn->scrn_defaultfbbpp = pScrn->scrn_defaultfbbpp;

		replaceString(&pNewScrn->scrn_monitor_str, pNewMonitor->mon_identifier);
		pNewScrn->scrn_monitor = pNewMonitor;

		replaceString(&pNewScrn->scrn_device_str, pNewDevice->dev_identifier);
		pNewScrn->scrn_device = pNewDevice;

		pNewScrn->scrn_adaptor_lst = NULL;
		pNewScrn->scrn_option_lst = NULL;
		pNewScrn->scrn_comment = NULL;

		pDisplay = calloc(1, sizeof(XF86ConfDisplayRec));
		pDisplay->disp_option_lst = NULL;
		pDisplay->disp_mode_lst = NULL;
		pDisplay->disp_comment = NULL;
		pDisplay->disp_weight.red = 0;
		pDisplay->disp_weight.blue = 0;
		pDisplay->disp_weight.green = 0;
		pDisplay->disp_white.red = -1;
		pDisplay->disp_white.blue = -1;
		pDisplay->disp_white.green = -1;
		pDisplay->disp_black.red = -1;
		pDisplay->disp_black.blue = -1;
		pDisplay->disp_black.green = -1;
		pDisplay->list.next = NULL;


		/* add new screen at 640x480 */
		pDisplay->disp_depth = pNewScrn->scrn_defaultdepth;
		pConfMode = calloc(1, sizeof(XF86ModeRec));
		pConfMode->mode_name = calloc(strlen("640x480") + 1, sizeof(char));
		strcpy(pConfMode->mode_name, "640x480");
		pConfMode->list.next = NULL;
		pDisplay->disp_mode_lst = (XF86ModePtr)xf86addListItem(     (GenericListPtr) pDisplay->disp_mode_lst, \
				(GenericListPtr) pConfMode);

		pNewScrn->scrn_display_lst = (XF86ConfDisplayPtr) xf86addListItem(      (GenericListPtr) pNewScrn->scrn_display_lst, \
				(GenericListPtr) pDisplay);


		/* Add it the list */
		conf->conf_screen_lst = (XF86ConfScreenPtr) xf86addListItem ( (GenericListPtr) \
				pScrn, (GenericListPtr) pNewScrn);


		MsgBox( "Matrox PowerDesk",
				"Your secondary display has been enabled. Its resolution has been set to 640 x 480.\n\nTo change its resolution, click the button for the secondary display (display 2).",
				0);


	}
	else
	{
		/* rename first screen */
		replaceString(&pScrn->scrn_identifier ,"Display 1");
		replaceString(&pScrn->scrn_device_str, pDevice->dev_identifier);
		replaceString(&pScrn->scrn_monitor_str, pMonitor->mon_identifier);

		/* Call the Second one  "Display 2" */
		replaceString(&pNewScrn->scrn_identifier ,"Display 2");
		replaceString(&pNewScrn->scrn_device_str, pNewDevice->dev_identifier);
		replaceString(&pNewScrn->scrn_monitor_str, pNewMonitor->mon_identifier);

	}   


	/* Server Layout Section */
	replaceString(&pAdj->adj_refscreen, "Display 2");


	replaceString(&pAdj->adj_screen_str, pScrn->scrn_identifier);


	pAdj->adj_where= where;


	if (pAdj->list.next == NULL)
	{
		pNewAdj = calloc(1, sizeof(XF86ConfAdjacencyRec));
		replaceString(&pNewAdj->adj_screen_str ,"Display 2");
		pNewAdj->adj_screen = pNewScrn;
		pNewAdj->adj_scrnum = -1;
		pLayout->lay_adjacency_lst= (XF86ConfAdjacencyPtr) xf86addListItem ( (GenericListPtr) pLayout->lay_adjacency_lst, (GenericListPtr) pNewAdj);
	}
	else
	{
		pNewAdj = pAdj->list.next;
		pNewAdj->adj_screen = pNewScrn;
		pNewAdj->adj_scrnum = -1;
		replaceString(&pNewAdj->adj_screen_str, "Display 2");
	}


	return conf;


}


/* Disables dual head */
XF86ConfigPtr confDisableDualHead ( XF86ConfigPtr conf)
{
	XF86ConfScreenPtr pScrn;
	XF86ConfDevicePtr pDevice;
	XF86ConfLayoutPtr pLayout;
	XF86ConfAdjacencyPtr pAdj;


	if (!conf)
	{
		printf("Matrox PowerDesk: Configuration problem encountered.\n");
		return NULL;
	}


	if(!(confDualHeadIsActive(conf))) return conf;


	pScrn = conf->conf_screen_lst;
	pDevice = conf->conf_device_lst;
	pLayout = conf->conf_layout_lst;
	pAdj = pLayout->lay_adjacency_lst;


	free(pAdj->adj_refscreen);
	pAdj->adj_refscreen = NULL;
	pAdj->adj_where = 0;
	free((XF86ConfAdjacencyPtr)pAdj->list.next);
	pAdj->list.next = NULL;

	return conf;


}


/* I will check how many screns there is and how many devices */
BOOL confDualHeadIsActive(XF86ConfigPtr conf)
{
	XF86ConfScreenPtr pScrn, pTempScrn;
	XF86ConfDevicePtr pDevice, pTempDevice;
	XF86ConfLayoutPtr pLayout;
	XF86ConfAdjacencyPtr pAdj,pNextAdj;
	int numScreens;
	int numDevices;


	if (!conf)
	{
		printf("Matrox PowerDesk: Configuration problem encountered.\n");
		return FALSE;
	}   

	pScrn = conf->conf_screen_lst;
	pDevice = conf->conf_device_lst;

	pLayout = conf->conf_layout_lst;
	pAdj = pLayout->lay_adjacency_lst;


	if (pAdj->list.next)
		pNextAdj= pAdj->list.next;
	else
		pNextAdj = NULL;


	/* Count how many Screen Sections */
	numScreens = 1 ;
	for (pTempScrn = pScrn; pTempScrn->list.next != NULL; pTempScrn=pTempScrn->list.next)
		numScreens++;


	/* Count how many Device Sections */
	numDevices = 1;
	for (pTempDevice = pDevice; pTempDevice->list.next != NULL; pTempDevice=pTempDevice->list.next)
		if (!(strcmp(pTempDevice->dev_driver, "mga")))
			numDevices++;


	if(!confValidateLayout(conf)) return FALSE;


	/*Check if one of the AdjencyRec has a screen that it references
	 * to. If so Dual is active*/


	if  (pAdj->adj_refscreen)
		return TRUE;


	if (pNextAdj && pNextAdj->adj_refscreen)
		return TRUE;


	/* Dual head is not active */
	return FALSE;


}



/* Clone */
XF86ConfigPtr confEnableClone ( XF86ConfigPtr conf)
{
	XF86ConfScreenPtr pScrn, pNewScrn;
	XF86ConfDevicePtr pDevice, pNewDevice;
	XF86ConfMonitorPtr pMonitor, pNewMonitor;
	XF86ConfLayoutPtr pLayout;
	XF86ConfAdjacencyPtr pAdj, pNewAdj;
	int i;

	if (!conf)
	{
		printf("Matrox PowerDesk: Configuration problem encountered.\n");
		return NULL;
	}


	if(confCloneIsActive(conf)) return conf;


	pScrn = conf->conf_screen_lst;
	pDevice = conf->conf_device_lst;
	pMonitor = conf->conf_monitor_lst;


	pLayout = conf->conf_layout_lst;
	pAdj = pLayout->lay_adjacency_lst;


	pNewScrn = pScrn->list.next;
	pNewDevice = pDevice->list.next;
	pNewMonitor = pMonitor->list.next;


	if (!pNewMonitor) /* Only one monitor */
	{
		/* Rename first monitor */
		replaceString(&pMonitor->mon_identifier,"Display 1");

		/* add a new monitor */
		pNewMonitor = calloc(1, sizeof(XF86ConfMonitorRec));
		pNewMonitor->list.next = NULL;


		replaceString(&pNewMonitor->mon_identifier,"Display 2");
		replaceString(&pNewMonitor->mon_vendor, pMonitor->mon_vendor);
		replaceString(&pNewMonitor->mon_modelname, pMonitor->mon_modelname);
		pNewMonitor->mon_width = pMonitor->mon_width;
		pNewMonitor->mon_height = pMonitor->mon_height;
		pNewMonitor->mon_modeline_lst = NULL;


		pNewMonitor->mon_n_hsync = pMonitor->mon_n_hsync;
		for(i = 0; i < pMonitor->mon_n_hsync; i++) pNewMonitor->mon_hsync[i] = pMonitor->mon_hsync[i];

		pNewMonitor->mon_n_vrefresh = pMonitor->mon_n_vrefresh;
		for(i = 0; i < pMonitor->mon_n_vrefresh; i++) pNewMonitor->mon_vrefresh[i] = pMonitor->mon_vrefresh[i];

		pNewMonitor->mon_gamma_red = pMonitor->mon_gamma_red;
		pNewMonitor->mon_gamma_green = pMonitor->mon_gamma_green;
		pNewMonitor->mon_gamma_blue = pMonitor->mon_gamma_blue;
		pNewMonitor->mon_option_lst = NULL;
		pNewMonitor->mon_modes_sect_lst = NULL;
		pNewMonitor->mon_comment = NULL;

		/* Add it the list */
		conf->conf_monitor_lst = (XF86ConfMonitorPtr) xf86addListItem ( (GenericListPtr) \
				pMonitor, (GenericListPtr) pNewMonitor);
	}
	else
	{
		/* There is at least 2 monitors */
		/* Rename first monitor */
		replaceString(&pMonitor->mon_identifier ,"Display 1");

		/* second monitor : Call it "Display 2" */
		replaceString(&pNewMonitor->mon_identifier ,"Display 2");
	}


	if (!pNewDevice) /* Only one device */
	{
		/* For the device */
		/* Rename first device*/
		replaceString(&pDevice->dev_identifier,"MATROX CARD 1");
		replaceString(&pDevice->dev_driver, "mga");


		/* add a new device */
		pNewDevice = calloc(1, sizeof(XF86ConfDeviceRec));

		pNewDevice->list.next = NULL;
		replaceString(&pNewDevice->dev_identifier,"MATROX CARD 2");
		replaceString(&pNewDevice->dev_vendor, pDevice->dev_vendor);
		replaceString(&pNewDevice->dev_board, pDevice->dev_board);
		replaceString(&pNewDevice->dev_chipset, pDevice->dev_chipset);
		replaceString(&pNewDevice->dev_busid, pDevice->dev_busid);
		replaceString(&pNewDevice->dev_card, pDevice->dev_card);
		replaceString(&pNewDevice->dev_driver, pDevice->dev_driver);
		replaceString(&pNewDevice->dev_ramdac, pDevice->dev_ramdac);
		replaceString(&pNewDevice->dev_clockchip, pDevice->dev_clockchip);
		for(i = 0; i < 4; i++) pNewDevice->dev_dacSpeeds[i] = pDevice->dev_dacSpeeds[i];
		pNewDevice->dev_videoram = pDevice->dev_videoram;
		pNewDevice->dev_textclockfreq = pDevice->dev_textclockfreq;
		pNewDevice->dev_videoram = pDevice->dev_videoram;
		pNewDevice->dev_bios_base = pDevice->dev_bios_base;
		pNewDevice->dev_mem_base = pDevice->dev_mem_base;
		pNewDevice->dev_io_base = pDevice->dev_io_base;
		pNewDevice->dev_clocks = pDevice->dev_clocks;
		for(i = 0; i < pDevice->dev_clocks; i++) pNewDevice->dev_clock[i] = pDevice->dev_clock[i];
		pNewDevice->dev_chipid = pDevice->dev_chipid;
		pNewDevice->dev_chiprev = pDevice->dev_chiprev;
		pNewDevice->dev_irq = pDevice->dev_irq;


		pNewDevice->dev_comment = NULL;
		pNewDevice->dev_option_lst = NULL;
		pNewDevice->dev_screen = 1;

		/* Add it the list */
		conf->conf_device_lst = (XF86ConfDevicePtr) xf86addListItem ( (GenericListPtr) \
				pDevice, (GenericListPtr) pNewDevice);
	}
	else
	{
		/* There is at least 2 devices */
		/* Rename first device*/
		replaceString(&pDevice->dev_identifier ,"MATROX CARD 1");
		replaceString(&pDevice->dev_driver, "mga");


		/* second device : Call it "MATROX CARD 2" */
		pNewDevice->dev_screen = 1;
		replaceString(&pNewDevice->dev_identifier ,"MATROX CARD 2");
		replaceString(&pNewDevice->dev_driver, "mga");


	}


	pDevice->dev_option_lst = removeOption(pDevice->dev_option_lst, "NoHal");
	pNewDevice->dev_option_lst = removeOption(pNewDevice->dev_option_lst, "NoHal");


	if (!pNewScrn) /* only one screen */
	{
		XF86ConfDisplayPtr pDisplay;
		XF86ModePtr pConfMode;

		/* Rename the first identifier of Screens*/
		replaceString(&pScrn->scrn_identifier , "Display 1");
		replaceString(&pScrn->scrn_device_str, pDevice->dev_identifier);
		replaceString(&pScrn->scrn_monitor_str, pMonitor->mon_identifier);

		/* Add a new screen */
		pNewScrn = calloc(1, sizeof(XF86ConfScreenRec));


		pNewScrn->list.next = NULL;


		replaceString(&pNewScrn->scrn_identifier, "Display 2");
		replaceString(&pNewScrn->scrn_obso_driver, pScrn->scrn_obso_driver);

		pNewScrn->scrn_defaultdepth = pScrn->scrn_defaultdepth;
		pNewScrn->scrn_defaultbpp = pScrn->scrn_defaultbpp;
		pNewScrn->scrn_defaultfbbpp = pScrn->scrn_defaultfbbpp;

		replaceString(&pNewScrn->scrn_monitor_str, pNewMonitor->mon_identifier);
		pNewScrn->scrn_monitor = pNewMonitor;

		replaceString(&pNewScrn->scrn_device_str, pNewDevice->dev_identifier);
		pNewScrn->scrn_device = pNewDevice;

		pNewScrn->scrn_adaptor_lst = NULL;
		pNewScrn->scrn_option_lst = NULL;
		pNewScrn->scrn_comment = NULL;

		pDisplay = calloc(1, sizeof(XF86ConfDisplayRec));
		pDisplay->disp_option_lst = NULL;
		pDisplay->disp_mode_lst = NULL;
		pDisplay->disp_comment = NULL;
		pDisplay->disp_weight.red = 0;
		pDisplay->disp_weight.blue = 0;
		pDisplay->disp_weight.green = 0;
		pDisplay->disp_white.red = -1;
		pDisplay->disp_white.blue = -1;
		pDisplay->disp_white.green = -1;
		pDisplay->disp_black.red = -1;
		pDisplay->disp_black.blue = -1;
		pDisplay->disp_black.green = -1;
		pDisplay->list.next = NULL;


		/* add new screen at 640x480 */
		pDisplay->disp_depth = pNewScrn->scrn_defaultdepth;
		pConfMode = calloc(1, sizeof(XF86ModeRec));
		pConfMode->mode_name = calloc(strlen("640x 480") + 1, sizeof(char));
		strcpy(pConfMode->mode_name, "640x480");
		pConfMode->list.next = NULL;
		pDisplay->disp_mode_lst = (XF86ModePtr)xf86addListItem(     (GenericListPtr) pDisplay->disp_mode_lst, \
				(GenericListPtr) pConfMode);

		pNewScrn->scrn_display_lst = (XF86ConfDisplayPtr) xf86addListItem(      (GenericListPtr) pNewScrn->scrn_display_lst, \
				(GenericListPtr) pDisplay);


		/* Add it the list */
		conf->conf_screen_lst = (XF86ConfScreenPtr) xf86addListItem ( (GenericListPtr) \
				pScrn, (GenericListPtr) pNewScrn);


		MsgBox( "Matrox PowerDesk",
				"Your secondary display has been enabled. Its resolution has been set to 640 x 480.\n\nTo change its resolution, click the button for the secondary display (display 2).",
				0);


	}
	else
	{
		/* rename first screen */
		replaceString(&pScrn->scrn_identifier ,"Display 1");
		replaceString(&pScrn->scrn_device_str, pDevice->dev_identifier);
		replaceString(&pScrn->scrn_monitor_str, pMonitor->mon_identifier);

		/* Call the Second one  "Display 2" */
		replaceString(&pNewScrn->scrn_identifier ,"Display 2");
		replaceString(&pNewScrn->scrn_device_str, pNewDevice->dev_identifier);
		replaceString(&pNewScrn->scrn_monitor_str, pNewMonitor->mon_identifier);

	}   


	/*****************************************
	  Server Layout Section - Enable clone
	  Check if the there's a second adjancy list
	  if so: make sure there's not refscreen
	  if not: create a new adjancy list with the Screen 2
	 ********************************************/

	replaceString(&pAdj->adj_screen_str, "Display 1");


	pAdj->adj_screen = pScrn;
	pAdj->adj_scrnum = -1;


	if (pAdj->list.next == NULL)
	{
		pNewAdj = calloc (1, sizeof(XF86ConfAdjacencyRec));

		replaceString(&pNewAdj->adj_screen_str, "Display 2");

		pNewAdj->adj_where = 0;
		pNewAdj->adj_screen = pNewScrn;
		pNewAdj->adj_scrnum = -1;
		pLayout->lay_adjacency_lst = (XF86ConfAdjacencyPtr) xf86addListItem ( (GenericListPtr)  \
				pAdj, (GenericListPtr) pNewAdj);
	}
	else
	{
		pNewAdj = pAdj->list.next;

		replaceString(&pNewAdj->adj_screen_str, "Display 2");

		pNewAdj->adj_where = 0;
		pNewAdj->adj_screen = pNewScrn;
		pNewAdj->adj_scrnum = -1;
	}


	pAdj->adj_where = 0;


	return conf;


}


XF86ConfigPtr confDisableClone ( XF86ConfigPtr conf)
{
	XF86ConfScreenPtr pScrn;
	XF86ConfDevicePtr pDevice;
	XF86ConfLayoutPtr pLayout;
	XF86ConfAdjacencyPtr pAdj;


	if (!conf)
	{
		printf("Matrox PowerDesk: Configuration problem encountered.\n");
		return NULL;
	}


	pScrn = conf->conf_screen_lst;
	pDevice = conf->conf_device_lst;
	pLayout = conf->conf_layout_lst;
	pAdj = pLayout->lay_adjacency_lst;


	pAdj->adj_refscreen = NULL;
	pAdj->adj_where = 0;
	/* Make the second monitor NULL */
	free(pAdj->list.next);
	pAdj->list.next = NULL;

	return conf;


}


BOOL confCloneIsActive(XF86ConfigPtr conf)
{
	XF86ConfScreenPtr pScrn, pTempScrn;
	XF86ConfDevicePtr pDevice, pTempDevice;
	XF86ConfLayoutPtr pLayout;
	XF86ConfAdjacencyPtr pAdj,pNextAdj;
	int numScreens;
	int numDevices;


	if (!conf)
	{
		printf("Matrox PowerDesk: Configuration problem encountered.\n");
		return FALSE;
	}   


	pScrn = conf->conf_screen_lst;
	pDevice = conf->conf_device_lst;


	pLayout = conf->conf_layout_lst;
	pAdj = pLayout->lay_adjacency_lst;


	if (pAdj->list.next)
		pNextAdj= pAdj->list.next;
	else
		pNextAdj = NULL;


	/* Count how many Screen Sections */
	numScreens = 1 ;
	for (pTempScrn = pScrn; pTempScrn->list.next != NULL; pTempScrn=pTempScrn->list.next)
		numScreens++;


	/* Count how many Device Sections */
	numDevices = 1;
	for (pTempDevice = pDevice; pTempDevice->list.next != NULL; pTempDevice=pTempDevice->list.next)
		if (!(strcmp(pTempDevice->dev_driver, "mga")))
			numDevices++;


	if(!confValidateLayout(conf)) return FALSE;


	/* Checks to see that both both adj structs have to ref_screen */
	if  (!pAdj->adj_refscreen )
	{
		if (pNextAdj && !(pNextAdj->adj_refscreen))
			return TRUE;
	}


	return FALSE;


}


/* Direct Rendering Layer (DRI) */
/* Adds module at end of list */
XF86ConfigPtr confLoadModule (XF86ConfigPtr conf, char* load_name)
{
	XF86ConfModulePtr pModule;
	XF86LoadPtr pLoad, pTempLoad;
	XF86LoadPtr pNewLoad;

	pModule = conf->conf_modules;
	pLoad = pModule-> mod_load_lst;


	/* Search for DRI in the Module List*/
	for (pTempLoad = pModule-> mod_load_lst; pTempLoad; pTempLoad= pTempLoad->list.next)
		if (!strcmp(pTempLoad->load_name, load_name)) return conf;


#ifdef DEBUG
	printf("Did not find the module %s",load_name);
	printf(" Adding new module\n");
#endif

	/* If it doesn't find DRI */
	/* Add DRI section */
	pNewLoad = calloc (1, sizeof(XF86LoadRec));


	replaceString(&pNewLoad->load_name, load_name);


#ifdef DEBUG
	printf("pNewLoad->load_name = %s\n",pNewLoad->load_name);
#endif

	pNewLoad->load_type = 0; /* LOAD_MODULE */
	pNewLoad->load_opt = NULL;
	pNewLoad->list.next = NULL;


	pModule-> mod_load_lst = (XF86LoadPtr) xf86addListItem ( (GenericListPtr) \
			pLoad, (GenericListPtr) pNewLoad);

	return conf;


}


XF86ConfigPtr confUnloadModule (XF86ConfigPtr conf, char* load_name)
{
	XF86LoadPtr pPrevLoad;
	XF86LoadPtr pLoad;
	XF86ConfModulePtr pModule;


	if (!conf)
	{
		printf("Matrox PowerDesk: Configuration problem encountered.\n");
		return NULL;
	}

	pModule = conf->conf_modules;
	pLoad = pModule->mod_load_lst;
	pPrevLoad = pLoad; 


	while (pLoad)
	{
		if (!(strcmp(pLoad->load_name, load_name)))
		{
			/* Found module to be deleted */
			/* Re-link the list around it */
			if ( pLoad == pModule->mod_load_lst)
				pModule->mod_load_lst = pLoad->list.next; /* We're deleting the head */
			else
				pPrevLoad->list.next = pLoad->list.next;

			free(pLoad);
			break;
		}

		pPrevLoad = pLoad;
		pLoad = pLoad->list.next;


	}


	return conf;


}



BOOL confIsModuleActive(XF86ConfigPtr conf, char* load_name)
{
	XF86LoadPtr pLoad;
	XF86ConfModulePtr pModule;


	pModule = conf->conf_modules;
	pLoad = pModule->mod_load_lst;


	while (pLoad != NULL)
	{
		if (!(strcmp(pLoad->load_name,load_name)))
			return TRUE;

		pLoad= pLoad->list.next;
	}


	return FALSE;


}


/* Set the default Depth */
XF86ConfigPtr confSetDefaultDepth(XF86ConfigPtr conf, int screen, int depth)
{
	XF86ConfScreenPtr pScrn ;

	pScrn= getScreenFromIndex(conf, screen);


	if (!pScrn)
	{
		printf("getScreenFromIndex failed\n");
		return NULL;
	}


    /* For 32 bit */
	if (depth ==32) {
	pScrn->scrn_defaultdepth = 24;
	pScrn->scrn_defaultfbbpp = 32;
	}
	else 
	{
	pScrn->scrn_defaultdepth = depth;
	pScrn->scrn_defaultfbbpp = 0;
    }

	return conf;
}

/* Get the default Depth */
int confGetDefaultDepth(XF86ConfigPtr conf, int screen)
{
        XF86ConfScreenPtr pScrn ;

        pScrn= getScreenFromIndex(conf, screen);


        if (!pScrn)
        {
                printf("getScreenFromIndex failed\n");
                return 0;
        }


    /* For 32 bit */
        if(	pScrn->scrn_defaultdepth == 24 &&
	        pScrn->scrn_defaultfbbpp == 32 ) 
	{
		return 32;
        }
        else
        {
        	return pScrn->scrn_defaultdepth;
	}

}



/* This function sets the different resolutions and depth */
XF86ConfigPtr confSetResolutions(XF86ConfigPtr conf, int screenIndex, int nbRes, char** Resolutions, int colorDepth)
{
	XF86ConfScreenPtr pScrn;
	XF86ConfDisplayPtr pDisplay;
	XF86ModePtr pConfMode;
	int i;
	BOOL bFoundDisplay = FALSE;


	if (!conf) return NULL;


	pScrn = getScreenFromIndex(conf, screenIndex);


	if (!pScrn)
	{
		printf("getScreenFromIndex failed\n");
		return NULL;
	}



	pDisplay = pScrn->scrn_display_lst;

	/* Search for a display subsection with same bpp */
	while(pDisplay)
	{
		if((pDisplay->disp_depth == colorDepth) || ((pDisplay->disp_depth == 24) &&  (colorDepth == 32)))
		{       /* if found, delete previous resolution */
			bFoundDisplay = TRUE;
			free(pDisplay->disp_mode_lst);
			pDisplay->disp_mode_lst = NULL;
			break;
		}


		pDisplay = pDisplay->list.next;
	}

	if(!bFoundDisplay)
	{       /* if not found same bpp, add a new display subsection */
		pDisplay = calloc(1, sizeof(XF86ConfDisplayRec));
		pDisplay->disp_option_lst = NULL;
		pDisplay->disp_mode_lst = NULL;
		pDisplay->disp_comment = NULL;
		pDisplay->disp_weight.red = 0;
		pDisplay->disp_weight.blue = 0;
		pDisplay->disp_weight.green = 0;
		pDisplay->disp_white.red = -1;
		pDisplay->disp_white.blue = -1;
		pDisplay->disp_white.green = -1;
		pDisplay->disp_black.red = -1;
		pDisplay->disp_black.blue = -1;
		pDisplay->disp_black.green = -1;
		pDisplay->list.next = NULL;
		pScrn->scrn_display_lst = (XF86ConfDisplayPtr) xf86addListItem( (GenericListPtr) pScrn->scrn_display_lst, \
				(GenericListPtr) pDisplay);
	}

	/* set value for the pointer (found or allocated) */
	if ((colorDepth == 8 ) || (colorDepth == 16) || (colorDepth == 24)
	|| (colorDepth == 32)) {
		if (colorDepth == 32) 
		{
			pDisplay->disp_depth = 24;
		}
		else
			pDisplay->disp_depth = colorDepth;
		}
	else
		return NULL;


	/* Write the resolutions in Display section */
	for (i= 0; i < nbRes;i ++)
	{
		pConfMode = calloc(1, sizeof(XF86ModeRec));
		pConfMode->mode_name = calloc(strlen(Resolutions[i]) + 1, sizeof(char));
		strcpy(pConfMode->mode_name, Resolutions[i]);
		pConfMode->list.next = NULL;
		pDisplay->disp_mode_lst = (XF86ModePtr)xf86addListItem(     (GenericListPtr) pDisplay->disp_mode_lst, \
				(GenericListPtr) pConfMode);
	}

	return conf;


}

/* This function gets the different resolutions at a specified depth */
int confGetResolutions(XF86ConfigPtr conf, int screenIndex, int colorDepth, char** Resolutions, int maxnbRes)
{
	XF86ConfScreenPtr pScrn;
	XF86ConfDisplayPtr pDisplay;
	XF86ModePtr pConfMode;
	int ResIndex;


	if (!conf) return 0;


	pScrn = getScreenFromIndex(conf, screenIndex);


	if (!pScrn)
	{
		printf("getScreenFromIndex failed\n");
		return 0;
	}



	pDisplay = pScrn->scrn_display_lst;

	/* Search for a display subsection with same bpp */
	while(pDisplay)
	{
		if((pDisplay->disp_depth == colorDepth) || ((pDisplay->disp_depth == 24) &&  (colorDepth == 32)))
		{      /* we've found our pDisplay, break.*/ 
			break;
		}


		pDisplay = pDisplay->list.next;
	}

	if(!pDisplay) return 0;

	/* Read the resolutions in Display section */
	pConfMode = pDisplay->disp_mode_lst;
	ResIndex = 0;
	while((pConfMode != NULL) && (ResIndex < maxnbRes)) {
		Resolutions[ResIndex] = pConfMode->mode_name;
		ResIndex++;
		pConfMode = pConfMode->list.next;
		}


	return ResIndex;
}

/* Xinerama */
XF86ConfigPtr confDisableXinerama(XF86ConfigPtr conf)
{
	XF86OptionPtr pConfOptions, pConfOptionPrev, pConfXineramaOption;

	if(!conf) return NULL;

	if((!(confXineramaIsActive(conf))) || (!(conf->conf_flags)) || (!(conf->conf_flags->flg_option_lst)))
		return conf;

	pConfOptions = conf->conf_flags->flg_option_lst;

	if(!(pConfXineramaOption = xf86findOption(pConfOptions,
					"Xinerama"))) return conf;

	pConfOptionPrev = pConfOptions;

	while (pConfOptions)
	{
		if (pConfOptions  == pConfXineramaOption)
		{
			/* Found module to be deleted */
			/* Re-link the list around it */
			if (conf->conf_flags->flg_option_lst ==
					pConfXineramaOption)
				/* We're deleting the head */
				conf->conf_flags->flg_option_lst =
					pConfXineramaOption->list.next;
			else
				pConfOptionPrev->list.next =
					pConfOptions->list.next;

			free(pConfXineramaOption);
			break;
		}

		pConfOptionPrev = pConfOptions;


		pConfOptions = pConfOptions->list.next;
	}



	/* Load dri as Xinerama got off */
	confLoadModule(conf, "dri");

	return conf;


}


XF86ConfigPtr confEnableXinerama(XF86ConfigPtr conf)
{
	char* optXinerama;

	if(!conf) return NULL;

	if(confXineramaIsActive(conf)) return conf;

	if(!(conf->conf_flags))
		conf->conf_flags = calloc(1, sizeof(XF86ConfFlagsRec));

	optXinerama = calloc(strlen("Xinerama") + 1, sizeof(char));
	strcpy(optXinerama, "Xinerama");

	conf->conf_flags->flg_option_lst =
		xf86addNewOption(conf->conf_flags->flg_option_lst, optXinerama, NULL);


	/* Unload dri because Xinerama just got ON. */
	confUnloadModule(conf, "dri");

	return conf;
}


BOOL confXineramaIsActive(XF86ConfigPtr conf)
{
	XF86ConfFlagsPtr pConfFlags = conf->conf_flags;

	if(!conf) return FALSE;

	if((!(pConfFlags)) || (!(pConfFlags->flg_option_lst)))
		return FALSE;


	else if(xf86findOption(pConfFlags->flg_option_lst, "Xinerama") !=
			NULL)
		return TRUE;
	else
		return FALSE;


}


/* add a Modeline */
XF86ConfigPtr confAddModeline(XF86ConfigPtr conf, int screenIndex, XF86VidModeModeLine* pModeLine, int dotClock)
{
	XF86ConfScreenPtr pConfScreen = getScreenFromIndex(conf, screenIndex);
	XF86ConfModeLinePtr pModeLineList = pConfScreen->scrn_monitor->mon_modeline_lst;
	XF86ConfModeLinePtr pNewConfModeLine;
	int refreshRate, hits = 0;
	char newRes[255];

	refreshRate = (int)(((((dotClock * 1000) / (float)(pModeLine->vtotal * pModeLine->htotal)) * 100.0) + 50) / 100);
	
        snprintf(newRes, 255, "%dx%d, %d Hz", pModeLine->hdisplay, pModeLine->vdisplay, refreshRate);
	
	/* find a unique name if this one is already used by another mode */
	
	while (xf86findModeLine(newRes, pModeLineList))
	    snprintf(newRes, 255, "%dx%d, %d Hz (%d)", pModeLine->hdisplay, pModeLine->vdisplay, refreshRate, ++hits);

	pNewConfModeLine = calloc(1, sizeof(XF86ConfModeLineRec));

	pNewConfModeLine->ml_identifier = calloc(strlen(newRes) + 1, sizeof(char));
	strcpy(pNewConfModeLine->ml_identifier, newRes);

	pNewConfModeLine->ml_clock = dotClock;
	pNewConfModeLine->ml_hdisplay = pModeLine->hdisplay;
	pNewConfModeLine->ml_hsyncstart = pModeLine->hsyncstart;
	pNewConfModeLine->ml_hsyncend = pModeLine->hsyncend;
	pNewConfModeLine->ml_htotal = pModeLine->htotal;
	pNewConfModeLine->ml_vdisplay = pModeLine->vdisplay;
	pNewConfModeLine->ml_vsyncstart = pModeLine->vsyncstart;
	pNewConfModeLine->ml_vsyncend = pModeLine->vsyncend;
	pNewConfModeLine->ml_vtotal = pModeLine->vtotal;
	pNewConfModeLine->ml_vscan = 0;
	pNewConfModeLine->ml_flags = pModeLine->flags;
	pNewConfModeLine->ml_hskew = pModeLine->hskew;


	pConfScreen->scrn_monitor->mon_modeline_lst=
	    (XF86ConfModeLinePtr)xf86addListItem((GenericListPtr) pModeLineList, (GenericListPtr) pNewConfModeLine);
	return conf;
}


/* retrieve user-defined Modelines */
XF86ConfModeLinePtr confGetUserModeLines(XF86ConfigPtr conf, int screenIndex)
{
	XF86ConfScreenPtr pConfScreen = getScreenFromIndex(conf, screenIndex);

	if(!(pConfScreen)) return NULL;

	if(!(pConfScreen->scrn_monitor)) return NULL;

	return pConfScreen->scrn_monitor->mon_modeline_lst;
}


/* HW Cursor */
XF86ConfigPtr confEnableHWCursor (XF86ConfigPtr conf)
{

	XF86ConfDevicePtr pConfDevice, pTmpConfDevice;
	XF86OptionPtr pConfOption, pTmpConfOption;

	if(!conf) return NULL;

	pConfDevice = conf->conf_device_lst;

	/* for each device */
	for (pTmpConfDevice = pConfDevice; pTmpConfDevice; pTmpConfDevice = pTmpConfDevice->list.next)
	{
		/* if not mga, skip it. */
		if(strcmp(pTmpConfDevice->dev_driver, "mga")) continue;


		pConfOption = pTmpConfDevice->dev_option_lst;

		/* If the "hw cursor" option exist */           
		if((pTmpConfOption = xf86findOption(pTmpConfDevice->dev_option_lst, "hw cursor")))
			if(strcmp(xf86findOptionValue(pTmpConfDevice->dev_option_lst, "hw cursor"), "on"))
			{
				free(pTmpConfOption->opt_val);
				pTmpConfOption->opt_val = calloc(3, sizeof(char));
				strcpy(pTmpConfOption->opt_val,"on");
			}


		/* If the "hw cursor" do not exist, let it like that because it's ON by default */
	}


	return conf;


}


XF86ConfigPtr confDisableHWCursor (XF86ConfigPtr conf)
{


	XF86ConfDevicePtr pConfDevice, pTmpConfDevice;
	XF86OptionPtr pConfOption, pTmpConfOption;


	if(!conf) return NULL;

	pConfDevice = conf->conf_device_lst;

	for (pTmpConfDevice = pConfDevice; pTmpConfDevice; pTmpConfDevice = pTmpConfDevice->list.next)
	{
		/* if not mga, skip it. */
		if(strcmp(pTmpConfDevice->dev_driver, "mga")) continue;


		pConfOption = pTmpConfDevice->dev_option_lst;

		/* If the "hw cursor" option exist */           
		if((pTmpConfOption = xf86findOption(pTmpConfDevice->dev_option_lst, "hw cursor")))
		{
			if(strcmp(xf86findOptionValue(pTmpConfDevice->dev_option_lst, "hw cursor"), "off"))
			{
				/*free(pTmpConfOption->opt_val);*/
				pTmpConfOption->opt_val = calloc(4, sizeof(char));
				strcpy(pTmpConfOption->opt_val,"off");
			}
		}
		else /* if not then add it */
		{
			/* add the option "hw cursor" "off" */
			char* optHwCursor = calloc(strlen("hw cursor") + 1, sizeof(char));
			char* optOff = calloc(strlen("off") + 1, sizeof(char));

			strcpy(optHwCursor, "hw cursor");
			strcpy(optOff, "off");
			pTmpConfDevice->dev_option_lst = xf86addNewOption(pConfOption, optHwCursor, optOff);
		}


	}
	return NULL;
}


BOOL confHWCursorIsActive(XF86ConfigPtr conf)
{


	XF86ConfDevicePtr pConfDevice, pTmpConfDevice;
	XF86OptionPtr pConfOption, pTmpConfOption;

	if(!conf) return FALSE;

	pConfDevice = conf->conf_device_lst;

	for (pTmpConfDevice = pConfDevice; pTmpConfDevice; pTmpConfDevice = pTmpConfDevice->list.next)
	{


		if(strcmp(pTmpConfDevice->dev_driver, "mga")) continue;


		pConfOption = pTmpConfDevice->dev_option_lst;

		for(pTmpConfOption = pConfOption; pTmpConfOption; pTmpConfOption = pTmpConfOption->list.next)
		{
			if(!strcmp(pTmpConfOption->opt_name, "hw cursor"))
			{
				if(!strcmp(pTmpConfOption->opt_val, "off"))
					return FALSE;
				else if(!strcmp(pTmpConfOption->opt_val, "on"))
					return TRUE;
			}
		}
	}


	/* By default : it's ON */
	return TRUE;

}


/* TV */
XF86ConfigPtr confEnableTVout( XF86ConfigPtr conf, int screenIndex, char* tvstandard , char* cabletype)
{       

	XF86ConfScreenPtr pConfScreen;
	XF86ConfDevicePtr pConfDevice;
	XF86OptionPtr pConfOption;

	if(!conf) return NULL;

	pConfScreen = getScreenFromIndex(conf, screenIndex);


	pConfDevice = pConfScreen->scrn_device;

	/* If the "TV" option exist */          
	if((pConfOption = xf86findOption(pConfDevice->dev_option_lst, "TV")))
	{
		if(strcmp(xf86findOptionValue(pConfDevice->dev_option_lst, "TV"), "yes"))
		{
			free(pConfOption->opt_val);
			pConfOption->opt_val = calloc(4, sizeof(char));
			strcpy(pConfOption->opt_val,"yes");
		}
	}
	else /* if not then add it */
	{
		char* optTV = calloc(strlen("TV") + 1, sizeof(char));
		char* optValYes = calloc(strlen("yes") + 1, sizeof(char));


		strcpy(optTV, "TV");
		strcpy(optValYes, "yes");
		pConfDevice->dev_option_lst = xf86addNewOption(pConfDevice->dev_option_lst, optTV, optValYes);
	}

	/* If the "CableType" option exist */           
	if((pConfOption = xf86findOption(pConfDevice->dev_option_lst, "CableType")))
	{
		free(pConfOption->opt_val);
		pConfOption->opt_val = calloc(strlen(cabletype)+1, sizeof(char));
		strcpy(pConfOption->opt_val,cabletype);
	}
	else /* if not then add it */
	{
		char* optValCable = calloc(strlen(cabletype) + 1, sizeof(char));
		char* optCableType = calloc(strlen("CableType") + 1, sizeof(char));


		strcpy(optCableType, "CableType");
		strcpy(optValCable, cabletype);
		pConfDevice->dev_option_lst = xf86addNewOption(pConfDevice->dev_option_lst, optCableType, optValCable);
	}


	/* If the "TVStandard" option exist */
	if((pConfOption = xf86findOption(pConfDevice->dev_option_lst, "TVStandard")))
	{
		free(pConfOption->opt_val);
		pConfOption->opt_val = calloc(strlen(tvstandard) + 1, sizeof(char));
		strcpy(pConfOption->opt_val,tvstandard);
	}
	else /* if not then add it */
	{
		char* optValStandard = calloc(strlen(tvstandard) + 1, sizeof(char));
		char* optStandard = calloc(strlen("TVStandard") + 1, sizeof(char));


		strcpy(optStandard, "TVStandard");
		strcpy(optValStandard, tvstandard);
		pConfDevice->dev_option_lst = xf86addNewOption(pConfDevice->dev_option_lst, optStandard, optValStandard);
	}


	return conf;


}


XF86ConfigPtr confDisableTVout (XF86ConfigPtr conf, int screenIndex)
{

	XF86ConfDevicePtr pConfDevice;
	XF86ConfScreenPtr pConfScreen;
	XF86OptionPtr pConfOption;

	if(!conf) return NULL;

	if(!confTVIsActive(conf, screenIndex)) return conf;

	pConfScreen = getScreenFromIndex(conf, screenIndex);

	pConfDevice = pConfScreen->scrn_device;

	/* If the "TV" option exist */          
	if((pConfOption = xf86findOption(pConfDevice->dev_option_lst, "TV")))
	{
		if(strcmp(xf86findOptionValue(pConfDevice->dev_option_lst, "TV"), "no"))
		{
			free(pConfOption->opt_val);
			pConfOption->opt_val = calloc(3, sizeof(char));
			strcpy(pConfOption->opt_val,"no");
		}
	}


	return conf;


}


BOOL confTVIsActive(XF86ConfigPtr conf, int screenIndex)
{

	XF86ConfDevicePtr pConfDevice;
	XF86ConfScreenPtr pConfScreen;
	XF86OptionPtr pConfOption;

	if(!conf) return FALSE;

	pConfScreen = getScreenFromIndex(conf, screenIndex);
	pConfDevice = pConfScreen->scrn_device;

	/* If the "TV" option exist */      
	if((pConfOption = xf86findOption(pConfDevice->dev_option_lst, "TV")))
	{
		if(!strcmp(xf86findOptionValue(pConfDevice->dev_option_lst, "TV"), "yes"))
		{
			return TRUE;
		}
	}


	return FALSE;


}


char* confGetTVStandard(XF86ConfigPtr conf, int screenIndex)
{
	XF86ConfScreenPtr pConfScreen;
	XF86ConfDevicePtr pConfDevice;
	XF86OptionPtr pConfOption;


	if(!conf) return (char*)NULL;


	pConfScreen = getScreenFromIndex(conf, screenIndex);
	pConfDevice = pConfScreen->scrn_device;


	if((pConfOption = xf86findOption(pConfDevice->dev_option_lst, "TVStandard")))
	{
		return xf86findOptionValue(pConfDevice->dev_option_lst, "TVStandard");
	}
	else
		return (char*)NULL;


}


char* confGetCableType(XF86ConfigPtr conf, int screenIndex)
{
	XF86ConfScreenPtr pConfScreen;
	XF86ConfDevicePtr pConfDevice;
	XF86OptionPtr pConfOption;


	if(!conf) return (char*)NULL;


	pConfScreen = getScreenFromIndex(conf, screenIndex);
	pConfDevice = pConfScreen->scrn_device;


	if((pConfOption = xf86findOption(pConfDevice->dev_option_lst, "CableType")))
	{
		return xf86findOptionValue(pConfDevice->dev_option_lst, "CableType");
	}
	else
		return (char*)NULL;


}


/* DigitalScreen */
XF86ConfigPtr confEnableDigitalScreen( XF86ConfigPtr conf, int screenIndex)
{       
	XF86ConfScreenPtr pConfScreen;
	XF86ConfDevicePtr pConfDevice;
	XF86OptionPtr pConfOption;

	if(!conf) return NULL;

	pConfScreen = getScreenFromIndex(conf, screenIndex);


	pConfDevice = pConfScreen->scrn_device;

	/* If the "DigitalScreen" option exist */       
	if((pConfOption = xf86findOption(pConfDevice->dev_option_lst, "DigitalScreen")))
	{
		if(strcmp(xf86findOptionValue(pConfDevice->dev_option_lst, "DigitalScreen"), "yes"))
		{
			free(pConfOption->opt_val);
			pConfOption->opt_val = calloc(4, sizeof(char));
			strcpy(pConfOption->opt_val,"yes");
		}
	}
	else /* if not then add it */
	{
		char* optValYes = calloc(strlen("yes") + 1, sizeof(char));
		char* optDigital = calloc(strlen("DigitalScreen") + 1, sizeof(char));


		strcpy(optDigital, "DigitalScreen");
		strcpy(optValYes, "yes");
		pConfDevice->dev_option_lst = xf86addNewOption(pConfDevice->dev_option_lst, optDigital, optValYes);
	}


	return conf;
}


XF86ConfigPtr confDisableDigitalScreen(XF86ConfigPtr conf, int screenIndex)
{
	XF86ConfDevicePtr pConfDevice;
	XF86ConfScreenPtr pConfScreen;
	XF86OptionPtr pConfOption;

	if(!conf) return NULL;

	if(!confDigitalScreenIsActive(conf, screenIndex)) return conf;

	pConfScreen = getScreenFromIndex(conf, screenIndex);

	pConfDevice = pConfScreen->scrn_device;

	/* If the "DigitalScreen" option exist */       
	if((pConfOption = xf86findOption(pConfDevice->dev_option_lst, "DigitalScreen")))
	{
		if(strcmp(xf86findOptionValue(pConfDevice->dev_option_lst, "DigitalScreen"), "no"))
		{
			/*free(pConfOption->opt_val); */
			pConfOption->opt_val = calloc(3, sizeof(char));
			strcpy(pConfOption->opt_val,"no");
		}
	}


	return conf;
}


BOOL confDigitalScreenIsActive(XF86ConfigPtr conf, int screenIndex)
{
	XF86ConfDevicePtr pConfDevice;
	XF86ConfScreenPtr pConfScreen;
	XF86OptionPtr pConfOption;

	if(!conf) return FALSE;

	pConfScreen = getScreenFromIndex(conf, screenIndex);
	pConfDevice = pConfScreen->scrn_device;

	/* If the "DigitalScreen" option exist */
	if((pConfOption = xf86findOption(pConfDevice->dev_option_lst, "DigitalScreen")))
	{
		if(!strcmp(xf86findOptionValue(pConfDevice->dev_option_lst, "DigitalScreen"), "yes"))
		{
			return TRUE;
		}
	}


	return FALSE;
}


BOOL confValidateLayout (XF86ConfigPtr p)
{
	XF86ConfLayoutPtr layout = p->conf_layout_lst;
	XF86ConfAdjacencyPtr adj;
	XF86ConfInactivePtr iptr;
	XF86ConfScreenPtr screen;
	XF86ConfDevicePtr device;


	while (layout)
	{
		adj = layout->lay_adjacency_lst;
		while (adj)
		{
			/* the first one can't be "" but all others can */
			screen = xf86findScreen (adj->adj_screen_str, p->conf_screen_lst);
			if (!screen)
			{
				printf("Invalid Layout\n");
				return (FALSE);
			}
			else
				adj->adj_screen = screen;


			adj = adj->list.next;
		}

		iptr = layout->lay_inactive_lst;

		while (iptr)
		{
			device = xf86findDevice (iptr->inactive_device_str, p->conf_device_lst);
			if (!device)
			{
				printf("Bad Layout. Something wrong with device\n");
				return (FALSE);
			}
			else
				iptr->inactive_device = device;


			iptr = iptr->list.next;

		}

		layout = layout->list.next;

	}


#ifdef DEBUG
	printf("Layout is valid\n");
#endif


	return (TRUE);
}


/* Gives the ServerLayout */
/* This routine sets the where variable easily
 * But uses the getIndexFromScreen function to return the proper index
 * */
XF86ConfigPtr confGetServerLayout(XF86ConfigPtr conf, int* scrn1, int* where, int* scrn2 ){


	XF86ConfLayoutPtr pLayout; 
	XF86ConfAdjacencyPtr pAdj;
	XF86ConfScreenPtr pScreen;

	pLayout = conf->conf_layout_lst;
	pAdj = pLayout->lay_adjacency_lst;
	pScreen = pAdj->adj_screen;


#ifdef DEBUG
	printf("Entering: confGetServerLayout\n");
#endif


	/* Loops trought the Adjacency list. If there's a ref screen
	 * return that line*/
	while(pAdj)
	{

		/* Checks to see if theire is a refscreen */
		if (pAdj->adj_refscreen != NULL)
		{
			*where = pAdj->adj_where;
			*scrn1 = getIndexFromScreen(conf, pAdj->adj_screen_str);
			*scrn2 = getIndexFromScreen(conf, pAdj->adj_refscreen);
			break;
		}

		pAdj = pAdj->list.next;

	}

	return conf;


}


/* This routine renames all the screens in the config file */


BOOL confRenameScreen(XF86ConfigPtr conf, XF86ConfScreenPtr pScrn, char *name)
{
	XF86ConfLayoutPtr pLayout;
	XF86ConfAdjacencyPtr pAdj; 
	pLayout= conf->conf_layout_lst;


	if (conf == NULL || pScrn == NULL) 
		return (False);


	while (pLayout != NULL) {
		pAdj = pLayout->lay_adjacency_lst;


		while (pAdj != NULL) {
			if (pAdj->adj_screen == pScrn) {
				replaceString(&pAdj->adj_screen_str, name);
			}
			else if (pAdj->adj_top == pScrn) {
				replaceString(&pAdj->adj_top_str, name);
			}
			else if (pAdj->adj_bottom == pScrn) {
				replaceString(&pAdj->adj_bottom_str, name);
			}
			else if (pAdj->adj_left == pScrn) {
				replaceString(&pAdj->adj_left_str, name);
			}
			else if (pAdj->adj_right == pScrn) {
				replaceString(&pAdj->adj_right_str, name);
			}
			else if (pAdj->adj_refscreen != NULL &&
					strcasecmp(pAdj->adj_refscreen, name) == 0) {
				replaceString(&pAdj->adj_refscreen, name);
			}


			pAdj = (XF86ConfAdjacencyPtr)(pAdj->list.next);
		}
		pLayout = (XF86ConfLayoutPtr)(pLayout->list.next);
	}


	replaceString(&pScrn->scrn_identifier, name);


	return (True);
}


/* This routine renames all the devices in the config file */
BOOL confRenameDevice(XF86ConfigPtr conf, XF86ConfDevicePtr pDevice, char *name)
{
	XF86ConfScreenPtr pScrn;
	pScrn= conf->conf_screen_lst;


	if (conf == NULL || pDevice == NULL) 
		return (False);


	while (pScrn != NULL) {
		if (pScrn->scrn_device == pDevice) {
			replaceString(&pScrn->scrn_device_str , name);
		}


		pScrn = (XF86ConfScreenPtr)(pScrn->list.next);
	}


	replaceString(&pDevice->dev_identifier , name);


	return (True);
}
