/*
 * GLX Hardware Device Driver for S3 ViRGE DX/GX/GX2
 * Copyright (C) 1999 Jim Duchek
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Based on Mach64 driver: mach64glx.c 1.2
 *
 *    Jim Duchek <jimduchek@ou.edu>
 */

#include <sys/time.h>
#include <stdlib.h>

#include "context.h"
#include "depth.h"
#include "macros.h"
#include "texstate.h"
#include "triangle.h"
#include "vb.h"
#include "types.h"

#include "xsmesaP.h"
#include "glx_log.h"
#include "mesaglx/context.h"
#include "mesaglx/matrix.h"
#include "mesaglx/types.h"
#include <sys/time.h>


#define GC XXGC
#include "gcstruct.h"
#include "pixmapstr.h"
#include "servermd.h" /* PixmapBytePad */
#include "scrnintstr.h"
#include "regionstr.h"
#include "windowstr.h"
#undef GC

#include "s3virgeglx.h"
#include "glx_symbols.h"

/* our collected globals */
s3virgeGlx_t		s3virgeglx;
s3virgeBufferPtr	s3virgeDB = NULL;
s3virgeContextPtr	s3virgeCtx = NULL;


void s3virgeGLXCreateDepthBuffer( GLcontext* ctx )
{
	XSMesaContext	xsmesa = (XSMesaContext) ctx->DriverCtx;
	s3virgeBufferPtr	buf;

	s3virgeMsg( 1, "s3virgeGLXCreateDepthBuffer() %08x\n", xsmesa->xsm_buffer);

	if (xsmesa->xsm_buffer->backimage == NULL) {
		s3virgeError("Backimage is null!!\n");
		return;
	}

	buf = (s3virgeBufferPtr)xsmesa->xsm_buffer->backimage->devPriv;

	/* An S3 ViRGE z-buffer is 16 bit... */

 	buf->depthBufferBlock = mmAllocMem( cardHeap, buf->pitch * buf->height * 2, 7, 0 );
 	
	if ( buf->depthBufferBlock == 0 ) {
		/* put it in main memory and don't hw accel */
		s3virgeMsg( 1, "Depth buffer forced to system memory.\n" );
		buf->depthBuffer = malloc( buf->pitch * buf->height * 2 );
		if ( !buf->depthBuffer ) {
			FatalError( "Malloc for depth buffer failed" );
		}
	} else {    
		buf->depthBuffer = (unsigned char *) s3virgeglx.linearBase +
			buf->depthBufferBlock->ofs;
		if ( s3virgeglx.logLevel >= 1 )  {
			mmDumpMemInfo( cardHeap );
		}
	}

#if 0    
   /* deallocate current depth buffer if present */
    if ( ctx->Buffer->Depth )
    {
	free( ctx->Buffer->Depth );
	ctx->Buffer->Depth = NULL;
    }
    /* allocate new depth buffer, but don't initialize it */
    /* FIXME: we shouldn't need to do this if we got a hardware buffer */
    ctx->Buffer->Depth = (GLdepth *)
	malloc( ctx->Buffer->Width * ctx->Buffer->Height * sizeof(GLdepth) );
    if ( !ctx->Buffer->Depth ) {
	    FatalError( "Couldn't allocate depth buffer\n" );
    }
#endif    
}

/*
 * s3virgeGLXDestroyImage
 */
void s3virgeGLXDestroyImage( GLXImage* image )
{
	s3virgeBufferPtr	buf;

    if ( image->devPriv ) {
	buf = (s3virgeBufferPtr)image->devPriv;
	
	if ( buf->lightmapBufferBlock ) {
		mmFreeMem( buf->lightmapBufferBlock );
	}
	
	/* free memory either in card space or VM */
	if ( buf->backBufferBlock ) {
		mmFreeMem( buf->backBufferBlock );
	} else {
		free( buf->backBuffer );
	}	
		
	if ( buf->depthBufferBlock ) {
		mmFreeMem( buf->depthBufferBlock );
	} else {
		free( buf->depthBuffer );
	}			
	
	s3virgeMsg( 1, "s3virgeGLXDestroyImage()\n" );
	if ( s3virgeglx.logLevel >= 10 )  {
	    mmDumpMemInfo( cardHeap );
	}
    }
    else if ( image->data )
    {
	free( image->data );
    }

    xfree( image );
}

/*
 * s3virgeGLXCreateImage
 *
 * This will always succeed, but if the buffer couldn't be allocated in
 * card memory, HW accel won't be usable.
 */
GLXImage *s3virgeGLXCreateImage( WindowPtr pwindow, int depth,
				int width, int height, GLXImage *old_image ){
	s3virgeBufferPtr	buf;
	GLXImage		*image;

	s3virgeMsg( 1, "s3virgeGLXCreateImage( %i, %i )\n", width, height );

	buf = (s3virgeBufferPtr)calloc(1,sizeof(s3virgeBuffer));
	if ( !buf ) {
		FatalError( "Malloc for buf failed\n" );
	}
	
	buf->magic = s3virgeBufferMagic;
	
	/* we may want to support 32 bit buffers in the future */
	
	buf->width = width;
	buf->height = height;
	buf->pitch = ( width + 31 ) & ~31;

	buf->backBufferBlock = mmAllocMem( cardHeap, buf->pitch * height * 2, 7, 0 );
	
	if (s3virgeglx.lightmapHack) {
		buf->lightmapBufferBlock = mmAllocMem( cardHeap, buf->pitch * height * 2, 7, 0 );
		buf->lightmapBuffer = (unsigned char *) s3virgeglx.linearBase + buf->lightmapBufferBlock->ofs;
	} 
	
	if ( buf->backBufferBlock == 0 ) {
		/* put it in main memory and don't hw accel */
		s3virgeMsg( 1, "Back buffer forced to system memory.\n" );
		buf->backBuffer = malloc( buf->pitch * height * 2 );
		if ( !buf->backBuffer ) {
			FatalError( "Malloc for back buffer failed" );
		}
	} else {    
		buf->backBuffer = (unsigned char *) s3virgeglx.linearBase +
			buf->backBufferBlock->ofs;
	}
	
	image = (GLXImage *) xalloc( sizeof(GLXImage) );
	if ( !image ) {
		FatalError( "Malloc for back ximage failed" );
	}
	image->pwin = pwindow;
	image->width = width;
	image->height = height;
	image->bits_per_pixel = depth;
	image->data = buf->backBuffer;
	image->bytes_per_line = buf->pitch * 2;
	image->devPriv = buf;

	/* How can I draw XImage with any line pad? */
	/* e.g. bytes_per_line != PixmapBytePad */
	/* My work around: adjust width instead */
	image->width = buf->pitch;
	
	if ( s3virgeglx.logLevel >= 1 )  {
	    mmDumpMemInfo( cardHeap );
	}
	
	return image;
}


/*
 * DoMakeCurrent
 * Changing either the context or the buffer will force this.
 * Redone 3/16/2000 by Justin A. Kolodziej to match what is done in mach64 
 * driver.  
 */
static int DoMakeCurrent( XSMesaContext c, XSMesaBuffer b )
{
        s3virgeContextPtr        ctx;
        s3virgeBufferPtr         buf;

        hwMsg( 10, "DoMakeCurrent( %p, %p )\n", c, b );

        s3virgeCtx = NULL;
        s3virgeDB = NULL;

        if ( !c || !b ) {
                return 0;
        }

        ctx = c->hw_ctx;
        if ( !ctx || ctx->magic != s3virgeContextMagic ) {
                FatalError( "DoMakeCurrent: ctx->magic != s3virgeContextMagic" );
        }

        /* if we don't have a backimage, create one now.  This happens
         * because of the way software GLX doesn't allocate a
         * buffer when front buffer rendering, but we need one
         * anyway
         */
        if ( !b->backimage ) {
                GLvisual        visual;

                memset( &visual, 0, sizeof( visual ) );
                visual.RedBits = 5;
                visual.GreenBits = 5;
                visual.BlueBits = 5;
                visual.DepthBits = 16;

                b->backimage = s3virgeGLXCreateImage( (WindowPtr)b->frontbuffer,
                                                     &visual, b->width, b->height, NULL );
        }

        buf = (s3virgeBufferPtr) b->backimage->devPriv;
        if ( buf->magic != s3virgeBufferMagic ) {
                FatalError( "DoMakeCurrent: != s3virgeBufferMagic" );
        }

        if ( !ctx || !buf ) {
                return 0;
        }

        /* for development, we can turn off hardware accel on a per-context basis
         * the most global way to prevent any hardware acceleration is to force the 
         * back buffer into main memory instead of card memory.
         *
	 *  Currently disabled for s3... Justin A. Kolodziej
        if ( c->no_accel && ( buf->backBufferBlock || buf->depthBufferBlock ) )
{
                ForceSoftwareBuffers( buf );  do we have this? 
        }
	 */

        /* valid conditions to render with */
	/* buf->xsbuf = b; should/can this be implemented for s3? */
        ctx->DB = buf;
        s3virgeCtx = ctx;
        s3virgeDB = buf;
	return 0;
}


/*
 * s3virgeGLXMakeCurrent
 * Note that GLX calls this every batch of commands,
 * so it will show up several times a frame even if
 * the client application isn't issuing makeCurrent calls.
 */
GLboolean s3virgeGLXMakeCurrent( XSMesaContext c )
{
    if ( c == XSMesa ) {
	return ( GL_TRUE );
    }

    /* mesa 3.1 will sometimes leave unflushed vertexes */
    if ( XSMesa ) {
	glFlush();
    }

    s3virgeMsg( 10, "s3virgeGLXMakeCurrent( %p )\n", c );
    XSMesa = c;
    if ( c )
    {
	DoMakeCurrent( c, c->xsm_buffer );
	gl_make_current( c->gl_ctx,
			 ( c->xsm_buffer ) ? c->xsm_buffer->gl_buffer : NULL );
    }
    else
    {
	/* detach */
	DoMakeCurrent( NULL, NULL );
	gl_make_current( NULL, NULL );
    }

    return GL_TRUE;
}

/*
 * Bind buffer b to context c and make c the current rendering context.
 */
GLboolean s3virgeGLXBindBuffer( XSMesaContext c, XSMesaBuffer b )
{
    s3virgeMsg( 10, "s3virgeGLXBindBuffer( %p, %p )\n", c, b );

    DoMakeCurrent( c, b );
    return XSMesaBindBuffer( c, b );
}


s3virgeContextPtr s3virgeCreateContext( GLcontext *ctx )
{
    s3virgeContextPtr	c;

    c = (s3virgeContextPtr) calloc( 1, sizeof(s3virgeContext) );
    if ( !c ) {
	return NULL;
    }

    c->magic = s3virgeContextMagic;
    c->gl_ctx = ctx;
#if 0 /* FIXME: GH */
    c->regDWGCTL = ( DC_clipdis_disable | (0xC << DC_bop_SHIFT) |
		     DC_shftzero_enable | DC_zmode_nozcmp |
		     DC_atype_zi );

    c->regALPHACTRL = AC_src_one | AC_dst_zero | AC_amode_FCOL |
	AC_astipple_disable | AC_aten_disable | AC_atmode_noacmp |
	AC_alphasel_fromtex;
#endif

    c->renderindex = -1;		/* impossible value */
    c->new_state = ~0;
    c->reg_dirty = ~0;

    /* XXX Fix me: callback not registered when main VB is created.
     */
    if ( ctx->VB ) {
#if 0 /* FIXME: GH */
	s3virgeDDRegisterVB( ctx->VB );
#endif
    } else {
	fprintf(stderr, "**** Didn't create vb driver data\n");
    }

    if ( ctx->NrPipelineStages ) {
#if 0 /* FIXME: GH */
	ctx->NrPipelineStages =
	    s3virgeDDRegisterPipelineStages( ctx->PipelineStage,
					    ctx->PipelineStage,
					    ctx->NrPipelineStages );
#endif
    }

    ctx->Const.MaxTextureLevels = 9;
    ctx->Const.MaxTextureUnits = 1;
    ctx->Const.MaxTextureSize = 512;
    

#if 0 /* FIXME: GH */
    s3virgeDDRenderInit();
    s3virgeDDTrifuncInit();
    s3virgeDDFastPathInit();
    s3virgeDDSetupInit();
    s3virgeInitFogAttrib( &c->Fog );
#endif

    s3virgeDDExtensionsInit( ctx );


    s3virgeMsg( 2, "s3virgeCreateContext(): successful.\n" );
    return c;
}



/*
 * Create a new XSMesaContext.
 * Input:  v - XSMesaVisual
 *         share_list - another XSMesaContext with which to share display
 *                      lists or NULL if no sharing is wanted.
 * Return:  an XSMesaContext or NULL if error.
 */
XSMesaContext s3virgeGLXCreateContext( XSMesaVisual v,
				      XSMesaContext share_list )
{
    XSMesaContext	c;

    s3virgeMsg( 0, "### Creating new xsmesa context for S3 ViRGE...\n" );

    c = (XSMesaContext) calloc( 1, sizeof(struct xsmesa_context) );
    
    if ( ! c ) {
        return NULL;
    }

    c->gl_ctx = gl_create_context( v->gl_visual,
				   ( share_list ) ? share_list->gl_ctx : NULL,
				   (void *) c,
				   GL_TRUE /* direct rendering */ );
	   
    if ( !c->gl_ctx ) {
        free( c );
        return NULL;
    }
        
    c->xsm_visual = v;
    c->xsm_buffer = NULL;       /* set later by XSMesaMakeCurrent */
    c->pixelformat = v->dithered_pf;    /* Dithering is enabled by default */
    c->hw_ctx = (void *) s3virgeCreateContext( c->gl_ctx );
    if ( !c->hw_ctx ) {
	s3virgeError( "Cannot create hardware specific context.\n" );
	return NULL;
    }
    ((s3virgeContextPtr)(c->hw_ctx))->refcount++;
    c->gl_ctx->Driver.UpdateState = s3virge_setup_DD_pointers;

    s3virgeMsg( 1, "s3virgeGLXCreateContext succeeded: %p\n", c );

    return c;
}


int s3virgeDestroyContext( s3virgeContextPtr ctx )
{
    if ( !ctx ) {
	return 0;
    }
    
    if ( ctx->magic != s3virgeContextMagic ) {
	return -1;
    }

    if ( --(ctx->refcount) > 0 ) {
	return 0;
    }

#if 0 /* FIXME: GH */
    s3virgeDestroyContextTextures( ctx );
#endif

    if ( ctx == s3virgeCtx ) {
    	s3virgeCtx = NULL;
    }
    ctx->magic = 0;
    free( ctx );
    s3virgeMsg( 2, "s3virgeDestroyContext(): successfully destroyed.\n" );
    return 0;
}


void s3virgeGLXDestroyContext( XSMesaContext c )
{
    s3virgeMsg( 1, "s3virgeGLXDestroyContext( %p )\n", c );

    /* make sure all drawing is completed */
    s3virgeDmaFinish();

    if ( s3virgeDestroyContext( (s3virgeContextPtr) c->hw_ctx ) != 0 ) {
	s3virgeError( "s3virgeGLXDestroyContext(): s3virgeDestroyContext() failed!\n" );
    }

    XSMesaDestroyContext( c );
    if ( XSMesa == c ) {
	XSMesa = 0;
    }
}


/*
 * Local Variables:
 * mode: c
 * c-basic-offset: 4
 * End:
 */
