

/*
 * Mesa 3-D graphics library
 * Version:  2.4
 * Copyright (C) 1995-1996  Brian Paul
 *
 * GLX Server Extension
 * Copyright (C) 1996  Steven G. Parker  (sparker@cs.utah.edu)
 * Copyright (C) 1998, 1999 Terence Ripperda (ripperda@sgi.com)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * STEVEN PARKER, TERENCE RIPPERDA, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR 
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, 
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE 
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#ifndef XSMESAP_H
#define XSMESAP_H

#include "xsmesa.h"

/*
 * XImage buffer.
 */
struct glx_image
{
    WindowPtr pwin;
    int       width, height;
    int       bytes_per_line;
    int       bits_per_pixel;
    char     *data;
    void     *devPriv;
};

/*
 * Mesa wrapper for XVisualInfo
 */
struct xsmesa_visual
{
    GLvisual *gl_visual;        /* the core visual */
    ScreenPtr pScreen;      /* the X11 screen */
    VisualPtr pVisual;      /* the X11 visual */

    GLint level;            /* 0=normal, 1=overlay, etc */

    GLboolean ximage_flag;  /* use XImage for back buffer (not pixmap)? */

    GLuint dithered_pf; /* Pixel format when dithering */
    GLuint undithered_pf;   /* Pixel format when not dithering */

    GLfloat RedGamma;   /* Gamma values, 1.0 is default */
    GLfloat GreenGamma;
    GLfloat BlueGamma;

    GLint rmult, gmult, bmult;  /* Range of color values */
    GLint index_bits;       /* Bits per pixel in CI mode */

    /* For PF_TRUECOLOR */
    GLint rshift, gshift, bshift;   /* Pixel color component shifts */
    GLubyte Kernel[16];     /* Dither kernel */
    unsigned long RtoPixel[512];    /* RGB to pixel conversion */
    unsigned long GtoPixel[512];
    unsigned long BtoPixel[512];
    GLubyte PixelToR[256];      /* Pixel to RGB conversion */
    GLubyte PixelToG[256];
    GLubyte PixelToB[256];

    /* For PF_1BIT */
    int bitFlip;
};

/*
 * Mesa wrapper for core rendering context
 */
struct xsmesa_context
{
    GLcontext *gl_ctx;      /* the core library context */
    void      *hw_ctx;      /* HW specific context      */

    XSMesaVisual xsm_visual;        /* Describes the buffers */
    XSMesaBuffer xsm_buffer;        /* current framebuffer */

    ScreenPtr pScreen;  /* == xm_visual->display */
    GLboolean swapbytes;    /* Host byte order != display byte order? */

    GLuint pixelformat;     /* Current pixel format */

    GLubyte red, green, blue, alpha;/* current drawing color */
    unsigned long pixel;        /* current drawing pixel value */

    GLubyte clearcolor[4];      /* current clearing color */
    unsigned long clearpixel;   /* current clearing pixel value */

    /* For the protocol dispatcher */
    unsigned int large_len;
    unsigned int large_op;
    int last_large_request;
    char* large_buf;
    int large_bufp;

   /* Direct contexts */
    int try_direct;
    
    /* disable hw rendering just for this context */
    int	no_accel;
};

/*
 * Mesa wrapper for X window or Pixmap
 */
struct xsmesa_buffer
{
    struct xsmesa_buffer *Next;  /* Linked list pointer: */

    GLboolean wasCurrent;    /* was ever the current buffer? */
    GLframebuffer *gl_buffer;    /* depth, stencil, accum, etc buffers */
    XSMesaVisual xsm_visual; /* the X/Mesa visual */

    GLboolean pixmap_flag;   /* is the buffer a Pixmap? */
    DrawablePtr frontbuffer; /* either a window or pixmap */
    PixmapPtr backpixmap;    /* back buffer Pixmap */
    GLXImage *backimage;         /* back buffer XImage */

    DrawablePtr buffer;  /* the current buffer, either equal to */
                         /* frontbuffer, backpixmap or XIMAGE (None) */

    ColormapPtr cmap;        /* the X colormap */

    GLint db_state;      /* 0 = single buffered */
    /* BACK_PIXMAP = use Pixmap for back buffer */
    /* BACK_XIMAGE = use XImage for back buffer */

    GLuint width, height;    /* size of buffer */

    GLint bottom;        /* used for FLIP macro below */
    GLubyte *ximage_origin1; /* used for PIXELADDR1 macro */
    GLint ximage_width1;
    GLushort *ximage_origin2;    /* used for PIXELADDR2 macro */
    GLint ximage_width2;
    GLuint *ximage_origin4;  /* used for PIXELADDR4 macro */
    GLint ximage_width4;

    PixmapPtr stipple_pixmap;    /* For polygon stippling */
    GCPtr stipple_gc;        /* For polygon stippling */

    GCPtr gc1;           /* GC for infrequent color changes */
    GCPtr gc2;           /* GC for frequent color changes */
    GCPtr cleargc;       /* GC for clearing the color buffer */

    char scandata[4*MAX_WIDTH];

    /* The following are here instead of in the XSMesaVisual
     * because they depend on the window's colormap.
     */

    /* For PF_DITHER, PF_LOOKUP, PF_GRAYSCALE */
    unsigned long color_table[576];      /* RGB -> pixel value */

    /* For PF_DITHER, PF_LOOKUP, PF_GRAYSCALE */
    GLubyte pixel_to_r[65536];       /* pixel value -> red */
    GLubyte pixel_to_g[65536];       /* pixel value -> green */
    GLubyte pixel_to_b[65536];       /* pixel value -> blue */

    /* Used to do XAllocColor/XFreeColors accounting: */
    int num_alloced;
    unsigned long alloced_colors[256];
};

/*
 * GLX proc pointers.
 */
struct _glx_procs
{
    XSMesaVisual  (*CreateVisual)       (ScreenPtr pScreen,VisualPtr pVisual,GLboolean rgb_flag,GLboolean alpha_flag,GLboolean db_flag,GLboolean ximage_flag,GLint depth_size,GLint stencil_size,GLint accum_size,GLint level);
    void          (*DestroyVisual)      (XSMesaVisual v);
    XSMesaContext (*CreateContext)      (XSMesaVisual v,XSMesaContext share_list);
    void          (*DestroyContext)     (XSMesaContext c);
    GLboolean     (*MakeCurrent)        (XSMesaContext c);
    void          (*CopyContext)        (XSMesaContext src,XSMesaContext dst,unsigned int mask);
    XSMesaBuffer  (*CreateWindowBuffer) (XSMesaVisual v,ClientPtr client,WindowPtr w);
    XSMesaBuffer  (*CreatePixmapBuffer) (XSMesaVisual v,ClientPtr client,PixmapPtr p,ColormapPtr c);
    void          (*DestroyBuffer)      (XSMesaBuffer b);
    GLboolean     (*BindBuffer)         (XSMesaContext c, XSMesaBuffer b);
    XSMesaBuffer  (*FindBuffer)         (ScreenPtr pScreen, DrawablePtr d);
    void          (*SwapBuffers)        (XSMesaBuffer b);
    GLboolean     (*GetBackBuffer)      (XSMesaBuffer b,PixmapPtr *pixmap,XImage **ximage);
    GLboolean     (*GetDepthBuffer)     (XSMesaBuffer b,GLint *width,GLint *height,GLint *bytesPerValue,void **buffer);
    XSMesaContext (*GetCurrentContext)  (void);
    XSMesaBuffer  (*GetCurrentBuffer)   (void);
    char*         (*FeedbackBuffer)     (XSMesaContext ctx);
    char*         (*SelectionBuffer)    (XSMesaContext ctx);
    unsigned long (*GetPixel)           (GLXImage* image, int x, int y);
    void          (*PutPixel)           (GLXImage* image, int x, int y, unsigned long pixel);
    void          (*DestroyImage)       (GLXImage* image);
    GLXImage*     (*CreateImage)        (WindowPtr pwindow, GLvisual *visual, int x, int y, GLXImage* old_image);
    GLdepth       (*GetDepth)           (XSMesaContext c, int x, int y);
    void          (*PutDepth)           (XSMesaContext c, int x, int y, GLdepth depth);
    void          (*CreateDepthBuffer)  (GLcontext* ctx);

    int           (*VendorPrivate)      ( ClientPtr client,
					  XSMesaContext ctx, 
					  xGLXVendorPrivateReq *stuff );

    int           (*AllowDirect)        ( ClientPtr client );
    void          (*ValidateFrontBuffer) ( DrawablePtr d );
};
typedef struct _glx_procs GLX_PROCS;
extern GLX_PROCS GLXProcs;


/* Values for xsmesa->dest: */
#define FRONT_PIXMAP	1
#define BACK_PIXMAP	2
#define BACK_XIMAGE	4

/* Values for xsmesa->pixelformat: */
#define PF_INDEX	1	/* Color Index mode */
#define PF_TRUECOLOR	2	/* TrueColor or DirectColor, any depth */
#define PF_TRUEDITHER	3	/* TrueColor with dithering */
#define PF_8A8B8G8R	4	/* 32-bit TrueColor:  8-A, 8-B, 8-G, 8-R */
#define PF_8R8G8B	5	/* 32-bit TrueColor:  8-R, 8-G, 8-B bits */
#define PF_5R6G5B	6	/* 16-bit TrueColor:  5-R, 6-G, 5-B bits */
#define PF_DITHER	7	/* Color-mapped RGB with dither */
#define PF_LOOKUP	8	/* Color-mapped RGB without dither */
#define PF_HPCR		9	/* HP Color Recovery (ad@lms.be 30/08/95) */
#define PF_1BIT		10	/* monochrome dithering of RGB */
#define PF_GRAYSCALE	11	/* Grayscale or StaticGray */

/*
 * If pixelformat==PF_TRUECOLOR:
 */
#define PACK_TRUECOLOR( PIXEL, R, G, B )	\
   PIXEL = xsmesa->xsm_visual->RtoPixel[R]	\
         | xsmesa->xsm_visual->GtoPixel[G]	\
         | xsmesa->xsm_visual->BtoPixel[B];	\

/*
 * If pixelformat==PF_TRUEDITHER:
 */
#define PACK_TRUEDITHER( PIXEL, X, Y, R, G, B )			\
{								\
   int d = xsmesa->xsm_visual->Kernel[((X)&3) | (((Y)&3)<<2)];	\
   PIXEL = xsmesa->xsm_visual->RtoPixel[(R)+d]			\
         | xsmesa->xsm_visual->GtoPixel[(G)+d]			\
         | xsmesa->xsm_visual->BtoPixel[(B)+d];			\
}

/*
 * If pixelformat==PF_8A8B8G8R:
 */
#define PACK_8A8B8G8R( R, G, B, A )	\
	( ((A) << 24) | ((B) << 16) | ((G) << 8) | (R) )

/*
 * Like PACK_8A8B8G8R() but don't use alpha.  This is usually an acceptable
 * shortcut.
 */
#define PACK_8B8G8R( R, G, B )   ( ((B) << 16) | ((G) << 8) | (R) )

/*
 * If pixelformat==PF_8R8G8B:
 */
#define PACK_8R8G8B( R, G, B)	 ( ((R) << 16) | ((G) << 8) | (B) )

/*
 * If pixelformat==PF_5R6G5B:
 */
#define PACK_5R6G5B( R, G, B)	 ( (((R) & 0xf8) << 8) | (((G) & 0xfc) << 3) | ((B) >> 3) )

/*
 * If pixelformat==PF_DITHER:
 *
 * Improved 8-bit RGB dithering code contributed by Bob Mercier
 * (mercier@hollywood.cinenet.net).  Thanks Bob!
 */
#undef _R
#undef _G
#undef _B
#ifdef DITHER666
#define _R   6
#define _G   6
#define _B   6
#define _MIX(r,g,b)  (((r)*_G+(g))*_B+(b))
#else
#define _R	5
#define _G	9
#define _B	5
#define _MIX(r,g,b)	( ((g)<<6) | ((b)<<3) | (r) )
#endif
#define _DX	4
#define _DY	4
#define _D	(_DX*_DY)

/*
 * If pixelformat==PF_GRAYSCALE:
 */
#define GRAY_RGB( R, G, B )   xsmesa->xsm_buffer->color_table[(R) + (G) + (B)]

#define XIMAGE None

/*
 * Converts a GL window Y coord to an X window Y coord:
 */
#define FLIP(Y)  (xsmesa->xsm_buffer->bottom-(Y))

/*
 * Return the address of a 1, 2 or 4-byte pixel in the back XImage:
 * X==0 is left, Y==0 is bottom.
 */
#define PIXELADDR1( X, Y )  \
      ( xsmesa->xsm_buffer->ximage_origin1 - (Y) * xsmesa->xsm_buffer->ximage_width1 + (X) )

#define PIXELADDR2( X, Y )  \
      ( xsmesa->xsm_buffer->ximage_origin2 - (Y) * xsmesa->xsm_buffer->ximage_width2 + (X) )

#define PIXELADDR4( X, Y )  \
      ( xsmesa->xsm_buffer->ximage_origin4 - (Y) * xsmesa->xsm_buffer->ximage_width4 + (X) )

/*
 * External functions:
 */

extern unsigned long xsmesa_color_to_pixel( XSMesaContext xsmesa,
                                            GLubyte r, GLubyte g, GLubyte b, GLubyte a );

extern void xsmesa_alloc_back_buffer( XSMesaBuffer b );

extern void xsmesa_static_DD_pointers( GLcontext *ctx );
extern void xsmesa_setup_DD_pointers( GLcontext *ctx );

extern points_func xsmesa_get_points_func( GLcontext *ctx );

extern line_func xsmesa_get_line_func( GLcontext *ctx );

extern triangle_func xsmesa_get_triangle_func( GLcontext *ctx );



#endif /* XSMESAP_H */
