/*
 * GLX Server Extension
 * Copyright (C) 1998, 1999 Terence Ripperda (ripperda@sgi.com)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * TERENCE RIPPERDA, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, 
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 * DEALINGS IN THE SOFTWARE.
 */

#include <stdlib.h>
#include <stdio.h>
#include "glx_clients.h"
#include "glx_log.h"
#include "glx_symbols.h"



/** client data **/
static unsigned int num_clients = 0;
static __glxClient *__glXClients[GLX_MAX_CLIENTS];


void
__glXInitClients() {
   unsigned int i;

   for (i = 0; i < (GLX_MAX_CLIENTS + 1); i++) {
      __glXClients[i] = NULL;
   }
}


static int
glXFindClientInArray(int client) {
   unsigned int i;

   /** hope that the client modulo'd cleanly into the array **/
   i = client % GLX_MAX_CLIENTS;
   if ( (__glXClients[i]) && (__glXClients[i]->index == client) ) {
      return i;
   }

   /** nope, do slow search **/
   for (i = 0; i < GLX_MAX_CLIENTS; i++) {
      if ( (__glXClients[i]) && (__glXClients[i]->index == client) ) {
         return i;
      }
   }

   return -1;
}

static int
glXFindEmptyArraySlot(int client) {
   unsigned int i;

   /** hope that the client modulo'd cleanly into the array **/
   i = client % GLX_MAX_CLIENTS;
   if (!__glXClients[i]) {
      return i;
   }

   /** nope, do slow search **/
   for (i = 0; i < GLX_MAX_CLIENTS; i++) {
      if (!__glXClients[i]) {
         return i;
      }
   }

   return -1;
}


__glxClient *
__glXFindClient(int client) {
   int index;

   index = glXFindClientInArray(client);
   if (index < 0) return NULL;

   return __glXClients[index];
}


ClientPtr direct_client = 0;

__glxClient *
__glXAddClient(ClientPtr client) {
   __glxClient *tmp = NULL;
   XID client_id;
   int index;

   fprintf(stderr, "Creating new client: %d!\n", client->index);
   log_print(LOG_TRACE,"creating new client!");
   
   if (direct_client && client != direct_client) {
      log_print(LOG_ALWAYS, 
		"Don't allow indirect clients if there is a direct one");
      return NULL;
   }

   if (num_clients >= GLX_MAX_CLIENTS) {
      log_print(LOG_ALWAYS, "Too many clients!");
      fprintf(stderr, "Too many clients!\n");
      if (num_clients > GLX_MAX_CLIENTS) 
         log_print(LOG_ALWAYS, "How the hell'd we get so many clients?!");
      return NULL;
   }

   tmp = (__glxClient *) xalloc( sizeof(struct glx_client));
   if (tmp == NULL)
      return NULL;

   index = glXFindEmptyArraySlot(client->index);
   if (index < 0) {
      /** this should never happen! **/
      log_print(LOG_ALWAYS, "Client Array corruption: no empty slots!");
      return NULL;
   }

   __glXClients[index] = tmp;
   memset(tmp, 0, sizeof(struct glx_client));
   tmp->client = client;
   tmp->index = client->index;

   client_id = FakeClientID(client->index);
   fprintf(stderr, "Adding client as resource: %ld!\n", client_id);
   if ( !AddResource(client_id, glxClients, (pointer) client->index) )
      return NULL;

   tmp->inUse = 1;
   tmp->major = 1;
   tmp->minor = 0;

   num_clients++;
   return tmp;
}


int
__glXRemoveClient(int client) {
   int index;
   __glxClient *tmp = NULL;

   index = glXFindClientInArray(client);

   if (index < 0) return 1;

   tmp = __glXClients[index];
   __glXClients[index] = NULL;

   if (tmp->client == direct_client) {
      fprintf(stderr, "Direct client signing off\n");
      direct_client = 0;
   }

   /** currently leaking memory!!? **/
   if (tmp->exts) xfree(tmp->exts);
   xfree(tmp);

   num_clients--;
   return 0;
}


int 
__glXNumClients() {
   return num_clients;
}






