/* 
 * GLX Hardware Device Driver for s3virge
 * Copyright (C) 1999 Jim Duchek
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * KEITH WHITWELL, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR 
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE 
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * s3virgedirect.c   Jim Duchek <jimduchek@ou.edu>
 * based on mach64direct.c by Keith Whitwell <keithw@precisioninsight.com>
 */

#include <stdlib.h>
#include <signal.h>

#include "xsmesaP.h"
#include "glx_log.h"
#include "mesaglx/context.h"
#include "mesaglx/matrix.h"
#include "mesaglx/types.h"

#define GC XXGC
#include "gcstruct.h"
#include "pixmapstr.h"
#include "servermd.h" /* PixmapBytePad */
#include "scrnintstr.h"
#include "regionstr.h"
#include "windowstr.h"
#undef GC

#include "s3virgeglx.h"

#include "glx_clients.h"
#include "glx_symbols.h"
#include "glx_init.h"

#include "xf86_OSproc.h"


/*

To perform direct rendering, we need to get all configuration information
from the server and mmap the framebuffer and dma buffers, 

Direct rendering will only be allowed if there are no other contexts active.

Because it is only possible on the same machine, we don't have to worry about
byte swapping any of our communication structures, we can just copy them
into message buffers.

shouldn't we have a glxReleaseDirect instead of void (**release_hook)( void )?

*/


/* 
 */
#define GLX_START                X_GLXDirectPrivateStart
// X_GLXDirectGoDirect:
// X_GLXDirectRelease:
#define X_GLXDirectSwapBuffers   (GLX_START+2)
#define X_GLXDirectDmaFlush      (GLX_START+3)


extern int __glXNumClients();


static int (*send_vendor_private)( int opcode,
				   char *vp, 
				   size_t vp_sz,
				   xReply *reply,
				   char *data,
				   size_t data_sz ) = 0;


/* =======================================================================
 * Initialization
 */


/* Nice thing about direct rendering is that the server *is* the
 * client, in the sense that they should both be the same executable
 * image 'glx.so'.  This means that most of the protocol structs can
 * be private to this file.
 */
typedef struct {   
	char init_fn[20];		/* The first element is always the
					 * name of the function for the client
					 * to call to initialize glx.so.  */

	s3virgeGlx_t	s3virgeglx;
	int glx_first_visual;
} s3virgeDirectHWInfo;


static void s3virgeClientReleaseHardware( void )
{
	if ( s3virgeglx.linearBase == (pointer) -1) {
 	  	return;
  	} 	
	xf86UnMapVidMem(0,
		      LINEAR_REGION,
		      (pointer) s3virgeglx.linearBase,
		      s3virgeglx.videoRam*1024 );
	xf86UnMapVidMem(0,
		      MMIO_REGION,
		      (pointer) s3virgeglx.MMIOBase,
		      0x10000 );

	s3virgeglx.linearBase = (pointer) -1;
	s3virgeglx.MMIOBase = (pointer) -1;
}



ClientPtr s3virgeGLXClientInit( s3virgeDirectHWInfo *hw, 
			       void *display,
			       int screen,
			       int (*send_func)(),
			       void (**release_hook)( void ) ) {


	if (!glxInstallLocalSVGASymbols()) 
		return 0;

	// this is how we send messages to the X server 
	send_vendor_private = send_func;

	// this is all the configuration info from the server,
	// including screen res, dma buffer location, etc
	s3virgeglx = hw->s3virgeglx;

	// clean up pointers in the struct
	s3virgeglx.logFile = NULL;	
	
	// memory map the framebuffer
   	s3virgeglx.linearBase = xf86MapVidMem( 0 /* screenNum */,
				 LINEAR_REGION,
				 (pointer) s3virgeglx.linearPhysical,
				 s3virgeglx.videoRam*1024 );


	if ( s3virgeglx.linearBase == (pointer) -1) {
		s3virgeError("failed to map vga linear region");
		s3virgeClientReleaseHardware();
		return 0;
	}
	
	s3virgeglx.MMIOBase = xf86MapVidMem( 0,  /* screenNum */
					MMIO_REGION,
					(pointer)s3virgeglx.linearPhysical + 0x1000000,
					0x10000);
					
	if ( s3virgeglx.MMIOBase == (pointer) -1) {
		s3virgeError("failed to map MMIO region");
		s3virgeClientReleaseHardware();
		return 0;
	}
	

	// GLX will call this when the client drops.
	// I think this should be a separate entry point...
	*release_hook = s3virgeClientReleaseHardware;
	
        return glxInitDirectClient( display, screen, hw->glx_first_visual,
				    XF_UNKNOWN, S3_VIRGE);
}

static int s3virgeGLXGoDirect( ClientPtr client )
{
   xGLXGetStringReply reply;
   s3virgeDirectHWInfo *hw;
   int l;

   /* Allow only one direct client, which must be the only active
    * client, and must be on the local server.  
    */
   if (direct_client || !__glx_is_server || __glXNumClients() != 1 || 
       !LocalClient(client)) 
      return BadAccess;

   s3virgeDmaFlush();

   direct_client = client;
   hw = (s3virgeDirectHWInfo *)malloc( sizeof(*hw) );

   /* Name of function to call to revive the struct.  Client will
    * receive buf in reply to its go_direct request.  Client will map
    * glx.so, and use buf as a string argument to dlsym(), to retreive
    * a function to call, namely s3virgeGLXClientInit.  It will then call
    * that function passing hw as its argument.
    *
    * hw should contain all the information necessary to get glx.so
    * talking to hardware from the client.
    */
   strcpy(hw->init_fn, "s3virgeGLXClientInit");   
	hw->s3virgeglx = s3virgeglx;
	
   hw->glx_first_visual = __glx_first_visual;

   reply.type=X_Reply;
   reply.sequenceNumber = client->sequence;
   reply.length=0;
   reply.n=l=(sizeof(*hw)+3)/4;

   WriteToClient(client, sizeof(xGLXGetStringReply), (char*)&reply);
   WriteToClient(client, sizeof(int)*l, (char*)hw);

   return client->noClientException;
}



static int s3virgeGLXReleaseDirect( ClientPtr client )
{
   if (__glx_is_server && direct_client && direct_client == client) {
      direct_client = 0;
      return Success;
   }

   return BadAccess;
}



/* =======================================================================
 * Dma flush
 */
typedef struct  {
   CARD32 dma_bufferDwords;
   CARD32 dma_sync;
   CARD32 pal_upd;
   CARD8 new_pal[768];
} DmaFlushReq;


typedef struct {
    BYTE type;
    CARD8 xpad;
    CARD16 sequenceNumber;
    CARD32 length;
} DmaFlushReply;

static int s3virgeGLXDirectDMAFlush( ClientPtr client, 
				 xGLXVendorPrivateReq *stuff )
{
   DmaFlushReq *vp = (DmaFlushReq *)(stuff + 1);
   xReply reply;
   DmaFlushReply *rep = (DmaFlushReply *)&reply;	
   extern void s3virgeServerDmaFlush( int );

   if (client != direct_client) 
      return BadAccess;

   if (vp->pal_upd) {
	   s3virgeLoadTexturePalette(vp->new_pal);
   }

   s3virgeServerDmaFlush( vp->dma_sync );

   rep->type = X_Reply;
   rep->length = 0;	
   rep->sequenceNumber = client->sequence;
   
   WriteToClient( client, sizeof(xReply), (char *) &reply );
	
   return client->noClientException;
}


/*
 * s3virgeClientDMAFlush
 * When direct rendering, this is called instead of starting
 * the dma from user space.  This only happens for overflows,
 * explicit flush/finish, and software rendering.  The swapbuffers
 * interface handles flushing in normal cases.
 */
void s3virgeDirectClientDMAFlush( int wait ) 
{
   xReply reply;
   DmaFlushReq vp;
   int i;

#if 0   
   vp.pal_upd = s3virgeglx.updatedPalette;

   if (vp.pal_upd) {
   	for (i=0; i<768; i++) {
   		vp.new_pal[i] = s3virgeglx.currentTexturePalette[i];
   	}
   } 
#endif

   if (send_vendor_private( X_GLXDirectDmaFlush,
			    (char *)&vp, sizeof(vp), 
			    &reply, 0, 0 ))
   {
   	s3virgeglx.updatedPalette = 0;
   }

   s3virgeDmaResetBuffer();
}

/* ===========================================================================
 * Buffer swap
 *   - Combines dma flush and swapbuffers
 */

/* Need to issue the swap commands from the server as the client
 * doesn't have the clip rects.  This isn't so bad as we would want to
 * do a dma flush immediately afterwards anyhow.
 */
typedef struct {
	s3virgeBuffer	backBuffer;
	TMemBlock		backBufferBlock;
	TMemBlock		lightmapBufferBlock;
	CARD32 front_drawable;
	CARD32 numDwords;
	CARD32 attrib;
	CARD32 flag;
} SwapBufferReq;

typedef struct {
   BYTE type;
   CARD8 xpad;
   CARD16 sequenceNumber;
   CARD32 length;
   CARD32 newDwords;
   CARD16 width;
   CARD16 height;
} SwapBufferReply;

#define FLG_TEX_SWAP 0x1

/*
 * this happens on the X server as a result of a direct rendering
 * client attempting to swapbuffers
 */ 
static int s3virgeGLXDirectSwapBuffers( ClientPtr client, 
				    xGLXVendorPrivateReq *stuff )
{
   SwapBufferReq *vp = (SwapBufferReq *)(stuff+1);
   xReply reply;
   SwapBufferReply *rep = (SwapBufferReply *)&reply;	
   DrawablePtr frontbuf;


   if (client != direct_client) 
      return BadAccess;

   // find the window we are going to blit to
   frontbuf = LookupIDByClass( vp->front_drawable, RC_DRAWABLE );   

   if (!frontbuf) {
      return GLXBadDrawable;
   }

	// update for performance boxes
	if (vp->flag & FLG_TEX_SWAP) {
 	  	s3virgeglx.c_textureSwaps++;
	}	

	s3virgeDB = &vp->backBuffer;
	s3virgeDB->backBufferBlock = &vp->backBufferBlock;
	if (s3virgeglx.lightmapHack)
		s3virgeDB->lightmapBufferBlock = &vp->lightmapBufferBlock;
	else
		s3virgeDB->lightmapBufferBlock = NULL;

	dma_buffer->numDwords = vp->numDwords;
	
	/* diagnostic drawing tools */
	s3virgePerformanceBoxes( 1 );

	/* hardware accelerated back to front blit */
	s3virgeBackToFront( frontbuf, &vp->backBuffer );

	/* make sure all dma is going to get executed */
	/* if everything has gone well, this will be the only
	   flush each frame */
	s3virgeDmaFlush();

   rep->type = X_Reply;
   rep->length = 0;	
   rep->sequenceNumber = client->sequence;
   rep->width = frontbuf->width;
   rep->height = frontbuf->height;
   rep->newDwords = dma_buffer->numDwords;

   WriteToClient( client, sizeof(xReply), (char *) &reply );

   return client->noClientException;
}

/*
 * This will be called instead of performing the buffer swap
 * if we are a direct rendering client.
 */
void s3virgeDirectClientSwapBuffers( XSMesaBuffer b ) 
{
   SwapBufferReq vp;
   xReply reply;
   s3virgeBufferPtr buf;

   if (!b->db_state || 
       !b->backimage ||
       !(buf = (s3virgeBufferPtr)b->backimage->devPriv))
   {
      fprintf(stderr, "client swap buffers: wtf???\n");
      return;
   }
   
   if (s3virgeCtx && s3virgeCtx->gl_ctx) {
      FLUSH_VB(s3virgeCtx->gl_ctx, "s3virge client swap buffers");
   }

   s3virgeglx.swapBuffersCount++;

   vp.front_drawable = b->frontbuffer->id;
   vp.backBuffer = *buf;
   vp.backBufferBlock = *buf->backBufferBlock;
  
   if (s3virgeglx.lightmapHack)   
	vp.lightmapBufferBlock = *buf->lightmapBufferBlock;
 
   vp.flag = 0;

   if (s3virgeglx.c_textureSwaps) {
      vp.flag |= FLG_TEX_SWAP;
      s3virgeglx.c_textureSwaps = 0;
   }
   
   vp.numDwords = dma_buffer->numDwords;

   if (send_vendor_private( X_GLXDirectSwapBuffers,
			    (char *)&vp, sizeof(vp), &reply, 0, 0 ))
   {
      SwapBufferReply *rep = (SwapBufferReply *)&reply;

      b->frontbuffer->width = rep->width;
      b->frontbuffer->height = rep->height;
      dma_buffer->numDwords = rep->newDwords;

      s3virgeDmaResetBuffer();
   }
   else
      FatalError("clientSwapBuffers failed");
}




/* =======================================================================
 * GLX procs
 */


int s3virgeGLXVendorPrivate( ClientPtr client,
			 XSMesaContext ctx, 
			 xGLXVendorPrivateReq *stuff ) {

   if (!__glx_is_server) { 
	return GLXUnsupportedPrivateRequest;
   }
	
   switch (stuff->opcode) {
   case X_GLXDirectGoDirect:
      return s3virgeGLXGoDirect( client );
   case X_GLXDirectRelease:
      return s3virgeGLXReleaseDirect( client );
   case X_GLXDirectSwapBuffers:
      return s3virgeGLXDirectSwapBuffers( client, stuff );
   case X_GLXDirectDmaFlush:
      return s3virgeGLXDirectDMAFlush( client, stuff );
   default:
      s3virgeError("not-handled case in s3virgeGLXVendorPrivate");
      return GLXUnsupportedPrivateRequest;
   }
}


int s3virgeGLXAllowDirect( ClientPtr client ){
   return 1;
}


