/* $Id: mach64swap.c,v 1.22 2000/03/27 12:55:00 gareth Exp $ */

/*
 * GLX Hardware Device Driver for ATI Rage Pro
 * Copyright (C) 1999 John Carmack, Gareth Hughes
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 *    John Carmack <johnc@idsoftware.com>
 *    Gareth Hughes <gareth@precisioninsight.com>
 */

#include <stdlib.h>

#include "context.h"
#include "depth.h"
#include "macros.h"
#include "texstate.h"
#include "triangle.h"
#include "vb.h"
#include "types.h"

#include "xsmesaP.h"
#include "glx_log.h"
#include "mesaglx/context.h"
#include "mesaglx/matrix.h"
#include "mesaglx/types.h"

#define GC XXGC
#include "gcstruct.h"
#include "pixmapstr.h"
#include "servermd.h" /* PixmapBytePad */
#include "scrnintstr.h"
#include "regionstr.h"
#include "windowstr.h"
#undef GC

#include "glx_symbols.h"

#include "mach64glx.h"

/*
 * mach64BackToFront
 * Blit the visible rectangles from the back buffer to the screen
 */
void mach64BackToFront( DrawablePtr drawable, mach64BufferPtr buf ) {
	RegionPtr	prgnClip;
	BoxPtr		pbox;
	int 		i, nbox;
	int 		xorg, yorg, pitch;
	int		screenFormat;
	DMALOCALS;

	prgnClip = &((WindowPtr)drawable)->clipList;
	pbox = REGION_RECTS( prgnClip );
	nbox = REGION_NUM_RECTS( prgnClip );
	if( !nbox ) {
		/* window is completely covered */
		return;
	}

	xorg = drawable->x;
	yorg = drawable->y;
	pitch = buf->pitch;

	switch( mach64glx.depth ) {
	case 15: screenFormat = 3; break;
	case 16: screenFormat = 4; break;
	case 32: screenFormat = 6; break;
	}

	/* add the blit commands to the dma buffer */
	DMAGETPTR( 10 + 10 * nbox );

	DMAOUTREG( MACH64_Z_CNTL, 0 );				/* no z buffering */
	DMAOUTREG( MACH64_SCALE_3D_CNTL, 0 );

	DMAOUTREG( MACH64_SC_LEFT_RIGHT, 0 | ( 8191 << 16 ) );	/* no scissor */
	DMAOUTREG( MACH64_SC_TOP_BOTTOM, 0 | ( 16383 << 16 )  );

	DMAOUTREG( MACH64_CLR_CMP_CNTL, 0 );			/* disable color compare */
	DMAOUTREG( MACH64_GUI_TRAJ_CNTL, DST_X_LEFT_TO_RIGHT | DST_Y_TOP_TO_BOTTOM );

	DMAOUTREG( MACH64_DP_PIX_WIDTH,
		   ( screenFormat << 0 )
		   | ( screenFormat << 4 )
		   | ( screenFormat << 8 )
		   | ( screenFormat << 16 )
		   | ( screenFormat << 28 )
		   );

	DMAOUTREG( MACH64_DP_WRITE_MASK, 0xffffffff );		/* write to all */
	DMAOUTREG( MACH64_DP_MIX, BKGD_MIX_D | FRGD_MIX_S );
	DMAOUTREG( MACH64_DP_SRC, BKGD_SRC_BKGD_CLR | FRGD_SRC_BLIT | MONO_SRC_ONE );

	DMAOUTREG( MACH64_DST_OFF_PITCH, mach64glx.registers[MACH64_DST_OFF_PITCH>>2] );
	DMAOUTREG( MACH64_SRC_OFF_PITCH, ( (buf->pitch/8) << 22 ) | (buf->backBufferBlock->ofs>>3) );

	/* xy bitblt each visible rectangle */
	for ( i = nbox ; i > 0 ; i-- ) {
		int	x = pbox->x1 - xorg;
		int	y = pbox->y1 - yorg;
		int	w = pbox->x2 - pbox->x1;
		int	h = pbox->y2 - pbox->y1;

		/* in case the window was just resized and the
		 * app doesn't know about it yet, restrict the
		 * blit source to the buffer size
		 */
		if ( x + w > buf->width ) {
			w = buf->width - x;
		}
		if ( y + h > buf->height ) {
			h = buf->height - y;
		}

		if ( w > 0 && h > 0 ) {
			DMAOUTREG( MACH64_SRC_WIDTH1, w );
			DMAOUTREG( MACH64_SRC_Y_X, (x << 16) | y );
			DMAOUTREG( MACH64_DST_X_Y, (pbox->y1 << 16) | pbox->x1 );
			DMAOUTREG( MACH64_DST_WIDTH_HEIGHT, (h << 16) | w );
		}

		pbox++;
	}

	DMAADVANCE();
}


/*
 * ClearBox
 * Add hardware commands to draw a filled box for the
 * debugging display.
 */
static void ClearBox( int x, int y, int w, int h, int r, int g, int b )
{
	int		color;
    	DMALOCALS;

    	DMAGETPTR( 32 );

	DMAOUTREG( MACH64_Z_CNTL, 0 );				/* no z buffering */
	DMAOUTREG( MACH64_SCALE_3D_CNTL, 0 );

	DMAOUTREG( MACH64_CLR_CMP_CNTL, 0 );			/* disable color compare */
	DMAOUTREG( MACH64_GUI_TRAJ_CNTL, DST_X_LEFT_TO_RIGHT | DST_Y_TOP_TO_BOTTOM );

	color = mach64PackColor( r, g, b, 0 );

	DMAOUTREG( MACH64_DP_FRGD_CLR, color );
	DMAOUTREG( MACH64_DP_WRITE_MASK, 0xffffffff );		/* write to all */
	DMAOUTREG( MACH64_DP_MIX, BKGD_MIX_D | FRGD_MIX_S );
	DMAOUTREG( MACH64_DP_SRC, BKGD_SRC_FRGD_CLR | FRGD_SRC_FRGD_CLR | MONO_SRC_ONE );

	DMAOUTREG( MACH64_DST_OFF_PITCH, ( (mach64DB->pitch/8) << 22 ) | (mach64DB->backBufferBlock->ofs>>3) ) ;
	DMAOUTREG( MACH64_DST_X_Y, (y << 16) | x );
	DMAOUTREG( MACH64_DST_WIDTH_HEIGHT, (h << 16) | w );

    	DMAADVANCE();
}


/*
 * performanceBoxes
 * Draw some small boxesin the corner of the buffer
 * based on some performance information
 */
void mach64PerformanceBoxes( void )
{
	int		w, t;

	if ( !mach64glx.boxes || !mach64DB ) {
		return;
	}

	/* draw a box to show we are active, so if it is't seen
	 * it means that it is completely software based rendering
	 */
	if ( !__glx_is_server ) {		/* purple = direct (client dma) rendering */
		ClearBox( 4, 4, 8, 8, 255, 0, 255 );
	} else if ( mach64glx.dmaDriver ) {	/* white = server dma rendering */
		ClearBox( 4, 4, 8, 8, 255, 255, 255 );
	} else {				/* grey = server PDMA */
		ClearBox( 4, 4, 8, 8, 128, 128, 128 );
	}

	/* draw a red box if we had to wait for drawing to complete
	   (software render or texture swap) */
	if ( mach64glx.c_drawWaits ) {
		ClearBox( 16, 4, 8, 8, 255, 0, 0 );
		mach64glx.c_drawWaits = 0;
	}

	/* draw a yellow box if textures were swapped */
	if ( mach64glx.c_textureSwaps ) {
		ClearBox( 40, 4, 8, 8, 255, 255, 0 );
	}

	/* draw a green box if we had to wait for dma to complete (full
	   utilization) on the previous frame */
	if ( !mach64glx.hardwareWentIdle ) {
		ClearBox( 64, 4, 8, 8, 0, 255, 0 );
	}

	/* show buffer utilization */
	if ( mach64glx.c_dmaFlush > 1 ) {
		/* draw a solid bar if we flushed more than one buffer */
		ClearBox( 4, 16, 252, 4, 255, 32, 32 );
	} else {
		/* draw bars to represent the utilization of primary buffer */
		ClearBox( 4, 16, 252, 4, 32, 32, 32 );
		t = mach64glx.dma_buffer->maxBufferDwords;
		w = 252 * mach64glx.dma_buffer->bufferDwords / t;
		if ( w < 1 ) {
			w = 1;
		}
		ClearBox( 4, 16, w, 4, 196, 128, 128 );
	}
	mach64glx.c_dmaFlush = 0;
}

/*
 * mach64PerformanceCounters
 */
void mach64PerformanceCounters( void )
{
	/* report performance counters */
	hwMsg( 9, "swapBuffers: c_triangles:%i c_lines:%i c_points:%i c_setup:%i\n",
	       mach64glx.c_triangles, mach64glx.c_lines, mach64glx.c_points, mach64glx.c_setupPointers );

	mach64glx.c_triangles = 0;
	mach64glx.c_lines = 0;
	mach64glx.c_points = 0;
	mach64glx.c_setupPointers = 0;

	if ( mach64glx.c_textureSwaps || mach64glx.c_textureDwords ) {
		hwMsg( 9, "    c_textureSwaps:%i  c_textureDwords:%i\n",
		       mach64glx.c_textureSwaps, mach64glx.c_textureDwords );
		mach64glx.c_textureSwaps = 0;
		mach64glx.c_textureDwords = 0;
	}

	mach64glx.c_drawWaits = 0;
	mach64glx.c_clears = 0;

	hwMsg( 9, "---------------------------------------------------------\n" );
}


/*
 * Copy the back buffer to the front buffer.
 */
void mach64GLXSwapBuffers( XSMesaBuffer b ) {
	mach64BufferPtr	buf;

	mach64glx.swapBuffersCount++;

	ValidateGC( b->frontbuffer, b->cleargc );

	if ( !b->backimage ) {
		return;	/* when is this ever hit? */
	}

	buf = (mach64BufferPtr)b->backimage->devPriv;
	if ( !VALID_MACH64_BUFFER( buf ) ) {
		hwError( "BackToFront(): invalid back buffer\n" );
		return;
	}

	if ( buf->backBufferBlock ) {
		/* diagnostic drawing tools */
		mach64PerformanceBoxes();

		if ( !__glx_is_server ) {
			/* direct rendering case */
			/* send a message to the server to flush and swap */
			mach64DirectClientSwapBuffers( b );
		} else {
			/* hardware accelerated back to front blit */
			mach64BackToFront( (DrawablePtr)b->frontbuffer, buf );

			/* make sure all dma is going to get executed */
			/* if everything has gone well, this will be the only
			   flush each frame */
			mach64DmaFlush();
		}
	} else {
		/* software blit ... */
		(*b->cleargc->ops->PutImage)( (DrawablePtr)b->frontbuffer,
					      b->cleargc,
					      b->frontbuffer->depth,
					      0, 0,
					      buf->pitch,
					      buf->height,
					      0, ZPixmap, buf->backBuffer );
	}

	mach64PerformanceCounters();
}


/*
 * Local Variables:
 * mode: c
 * tab-width: 8
 * c-basic-offset: 8
 * End:
 */
