/* $Id: mach64dd.c,v 1.26 2000/03/27 07:27:02 gareth Exp $ */

/*
 * GLX Hardware Device Driver for ATI Rage Pro
 * Copyright (C) 1999 Gareth Hughes
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Based on MGA driver: mgadd.c 1.13
 *
 *    Gareth Hughes <gareth@precisioninsight.com>
 */

#include "glx_config.h"
#include "glx_symbols.h"
#include "mach64glx.h"
#include "pixmapstr.h"	/* X: for _Drawable */
#include "pixmap.h"		/* X: for window DrawablePtr */
#include "hw_buffer.h"

/*
 * Mesa's Driver Functions
 *
 * all "mach64DD*" functions are directly called through
 * mesa's driver interface structure
 */

void mach64DDExtensionsInit( GLcontext *ctx )
{
	/* disable the extensions we can't hardware accelerate.
	 * Note that when a client is running indirectly, libGL
	 * determines the extension string (none supported), not this.
	 */

	/* we can support paletted textures later if we really care */
	gl_extensions_disable( ctx, "GL_EXT_shared_texture_palette" );
	gl_extensions_disable( ctx, "GL_EXT_paletted_texture" );

	gl_extensions_disable( ctx, "GL_EXT_point_parameters" );
	gl_extensions_disable( ctx, "ARB_imaging" );
	gl_extensions_disable( ctx, "GL_EXT_blend_minmax" );
	gl_extensions_disable( ctx, "GL_EXT_blend_logic_op" );
	gl_extensions_disable( ctx, "GL_EXT_blend_subtract" );
	gl_extensions_disable( ctx, "GL_EXT_blend_color" );
	gl_extensions_disable( ctx, "GL_INGR_blend_func_separate" );
	gl_extensions_disable( ctx, "GL_EXT_texture3D" );
	gl_extensions_disable( ctx, "GL_MESA_window_pos" );
	gl_extensions_disable( ctx, "GL_MESA_resize_buffers" );
	gl_extensions_disable( ctx, "GL_SGIS_texture_edge_clamp" );
	gl_extensions_disable( ctx, "GL_MESA_window_pos" );

}


const GLubyte *mach64DDGetString( GLcontext *ctx, GLenum name )
{
	switch ( name )
	{
	case GL_VENDOR:
		return "Utah GLX";
	case GL_RENDERER:
		/* Quake3 needs to see "rage pro" to cut out some broken features */
		return "GLX-RAGE PRO";
	default:
		return 0;
	}
}


static GLint mach64DDGetParameteri( const GLcontext *ctx, GLint param )
{
	switch ( param )
	{
	case DD_HAVE_HARDWARE_FOG:
		return 1;
#if 0	// these are documented in dd.h, but not defined
    	case DD_MAX_TEXTURE_SIZE:
    		return 1024;
	case DD_MAX_TEXTURES:
		return 1;		/* hopefully not for long! */
#endif
	default:
		hwError( "mach64GetParameteri(): unknown parameter!\n" );
		return 0; /* Should I really return 0? */
	}
}


/*
 * We allways have a single back buffer, but
 * if we are in "front buffer" mode, we will automatically
 * generate flush events
 */
static GLboolean mach64DDSetBuffer( GLcontext *ctx, GLenum mode )
{
	switch ( mode ) {

	case GL_FRONT:
	case GL_FRONT_LEFT:
	case GL_FRONT_RIGHT:
	 	mach64Ctx->frontBufferRendering = 1;
	        return GL_TRUE;

	case GL_BACK:
	case GL_BACK_LEFT:
	case GL_BACK_RIGHT:
	 	mach64Ctx->frontBufferRendering = 0;
	        return GL_TRUE;

	default:
		return GL_FALSE;
	}
}

static void FlushFrontBufferRendering( void )
{
	/* if we haven't drawn anything since the last swapbuffers, no need to swap */
	if ( mach64glx.c_triangles || mach64glx.c_lines || mach64glx.c_points ||
	     mach64glx.c_clears || mach64glx.c_drawWaits ) {
		mach64GLXSwapBuffers( mach64DB->xsBuf );
	}
}

static void mach64DDRenderFinish( GLcontext *ctx )
{
	/* if we are front buffer rendering, fake a swap to screen to get everything
	   we have drawn visible. */
	if ( mach64Ctx->frontBufferRendering ) {
		FlushFrontBufferRendering();
	}
}

static void mach64DDFlush( GLcontext *ctx )
{
	/* with only two dma buffers, explicit flushing of a double buffered
	 * context is almost always a bad idea.  Because a swapbuffers does
	 * generate a flush, we are still compliant with the spec by ignoring
	 * it here.
	 */
	if ( mach64Ctx->frontBufferRendering ) {
		FlushFrontBufferRendering();
		return;
	}
}

static void mach64DDFinish( GLcontext *ctx )
{
	if ( mach64Ctx->frontBufferRendering ) {
		glFlush();
		return;
	}
	/*
	 * Enforcing OpenGL finish semantics can cause
	 * a large performance overhead on applications that use it.
	 * Our two buffer dma system can never get more than a single
	 * frame behind the application, so the only case glFinish()
	 * is truly needed is for benchmarking.
	 */
	if ( mach64glx.enforceFinish ) {
		mach64DmaFinish();
	}
}


/*
 * mach64DDUpdateStateFunctions
 * If the current state allows hardware rendering, set
 * ctx->Driver.TriangleFunc to our accelerated triangle function
 */
void mach64DDUpdateStateFunctions( GLcontext *ctx )
{
	if ( !mach64DB ) {
		return;
	}

	/* no acceleration if our back buffer isn't on card */
	if ( !mach64DB->backBufferBlock ) {
		return;
	}

	/* decide if we want a software fallback */
	if ( !mach64glx.noFallback ) {
		/* no acceleration if any stencil operations are enabled */
		if ( ctx->Stencil.Enabled ) {
			return;
		}

		/* no acceleration if stipple is on */
		if ( ctx->Polygon.StippleFlag ) {
			return;
		}

		/* no acceleration if smoothing is on */
		if ( ctx->Polygon.SmoothFlag ) {
			return;
		}

		/* no acceleration if 3D texturing is on */
		if ( ctx->Texture.Enabled & (TEXTURE0_3D | TEXTURE1_3D) ) {
			return;
		}
	}

	/* use our accelerated functions */
	mach64DDChooseRenderState( ctx );

	/* we do all of our actual state updates in BeginRender */
}

/*
 * Return the size of the current color buffer, checking to see if the
 * window has been resized.
 * There isn't an explicit "size has changed" entry point, so this serves
 * double duty.
 */
static void mach64DDGetBufferSize( GLcontext *ctx, GLuint *width, GLuint *height )
{
	XSMesaContext		xsmesa;
	XSMesaBuffer		b;
	DrawablePtr		window;
	unsigned int		winwidth, winheight;

	xsmesa = (XSMesaContext) ctx->DriverCtx;
	b = xsmesa->xsm_buffer;
	window = b->frontbuffer;

	winwidth = window->width;
	winheight = window->height;
	*width = winwidth;
	*height = winheight;

	/* see if it has changed size */
	if ( winwidth != b->width || winheight != b->height ) {
	        b->width = winwidth;
	        b->height = winheight;

	        /* Allocate new back buffer :
		 * (also deallocate the old backimage, if any) */
	        b->backimage = mach64GLXCreateImage( (WindowPtr)b->frontbuffer,
						     b->xsm_visual->gl_visual,
						     b->width,
						     b->height,
						     b->backimage );

		/* make it current */
		XSMesa = NULL;	/* force it to reset things */
		mach64GLXBindBuffer( xsmesa, b );
	}
}

static void mach64DDClearColor( GLcontext *ctx, GLubyte red, GLubyte green, GLubyte blue, GLubyte alpha )
{
	/* we just look in the context when we clear, but mesa
	 * requires this function to exist
	 */
}

static void mach64DDAllocDepthBuffer( GLcontext *ctx )
{
	/* we always allocate the depth buffer with the color buffer */
}


void mach64_setup_DD_pointers( GLcontext *ctx )
{
	cbFormat_t	cbFormat;
	dbFormat_t	dbFormat;

	ctx->Driver.GetBufferSize = mach64DDGetBufferSize;
	ctx->Driver.AllocDepthBuffer = mach64DDAllocDepthBuffer;
	ctx->Driver.SetBuffer = mach64DDSetBuffer;
	ctx->Driver.GetString = mach64DDGetString;
	ctx->Driver.UpdateState = mach64DDUpdateState;
	ctx->Driver.RenderFinish = mach64DDRenderFinish;
	ctx->Driver.Clear = mach64Clear;
	ctx->Driver.GetParameteri = mach64DDGetParameteri;
	ctx->Driver.Flush = mach64DDFlush;
	ctx->Driver.Finish = mach64DDFinish;
	ctx->Driver.UpdateState = mach64DDUpdateStateFunctions;
	ctx->Driver.RenderStart = mach64DDUpdateState;
	ctx->Driver.TexImage = mach64TexImage;
	ctx->Driver.TexSubImage = mach64TexSubImage;
	ctx->Driver.DeleteTexture = mach64DeleteTexture;
	ctx->Driver.IsTextureResident = mach64IsTextureResident;
	ctx->Driver.ClearColor = mach64DDClearColor;
	ctx->Driver.DrawPixels = mach64DDDrawPixels;
	ctx->Driver.Bitmap = mach64DDBitmap;

	ctx->Driver.TriangleCaps = 0;		/* no special handling yet */

	/* set all the buffer access functions based on bit depth */
	switch ( mach64glx.depth ) {
	case 15:
		cbFormat = CB_15BIT;
		break;
	case 16:
		cbFormat = CB_16BIT;
		break;
	case 24:
	case 32:
		cbFormat = CB_32BIT_BGRA;
		break;
	}

	dbFormat = DB_16BIT;	// no other options on RagePro

	SetDriverBufferFunctions( ctx, mach64DmaFinish,
		mach64DB->backBuffer, mach64DB->pitch, mach64DB->height, cbFormat,
		mach64DB->depthBuffer, mach64DB->pitch, mach64DB->height, dbFormat );
}


/*
 * Local Variables:
 * mode: c
 * tab-width: 8
 * c-basic-offset: 8
 * End:
 */
