/*
 * GLX Hardware Device Driver for Intel i810
 * Copyright (C) 1999 Keith Whitwell
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * KEITH WHITWELL, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR 
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE 
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 */

/* $Id: i810glx.c,v 1.3 2000/02/05 20:49:11 keithw Exp $ */

#include <stdlib.h>

#include "context.h"
#include "depth.h"
#include "macros.h"
#include "texstate.h"
#include "triangle.h"
#include "vb.h"
#include "types.h"

#include "xsmesaP.h"
#include "glx_log.h"
#include "mesaglx/context.h"
#include "mesaglx/matrix.h"
#include "mesaglx/types.h"

#define GC XXGC
#include "gcstruct.h"
#include "pixmapstr.h"
#include "servermd.h" /* PixmapBytePad */
#include "scrnintstr.h"
#include "regionstr.h"
#include "windowstr.h"
#undef GC

#include "mm.h"
#include "i810buf.h"
#include "i810dd.h"
#include "i810lib.h"
#include "i810tris.h"
#include "i810log.h"
#include "i810direct.h"

#include "i810dma.h"


extern XSMesaContext XSMesa;

/* our collected globals */
i810Glx_t	i810glx;

static void i810ServerSwapBuffers( XSMesaBuffer b );

void (*i810GLXSwapBuffers)(XSMesaBuffer b) = i810ServerSwapBuffers;



void i810DumpDB( struct i810_dest_buffer *buf )
{
  i810Msg(1,"\nCard Heap:");
  mmDumpMemInfo( i810glx.cardHeap );
  i810Msg(1,"\nSystem Heap:");
  mmDumpMemInfo( i810glx.sysmemHeap );
  i810Msg(1,"\nDump DB:\n");
  i810Msg(1,"  %s %s\n",(buf->Drawable) ? "Drawable" : "-",
         (buf->ZBuffer) ? "HasZORG" : "-");
  i810Msg(1,"  w,h,p = %d,%d,%d\n",buf->Width,buf->Height,buf->Pitch);
  i810Msg(1,"End Dump DB\n");
}


void i810GLXCreateDepthBuffer(GLcontext* ctx)
{
   XSMesaContext xsmesa = (XSMesaContext) ctx->DriverCtx;
   struct i810_dest_buffer *buf;
   struct i810_z_buffer *zb=0;

   i810Msg( 1, "i810GLXCreateDepthBuffer\n" );
	
   /* deallocate current depth buffer if present */
   if (ctx->Buffer->Depth)
   {
      free(ctx->Buffer->Depth);
      ctx->Buffer->Depth = NULL;
   }

   if (xsmesa->xsm_buffer->db_state == BACK_XIMAGE &&
       xsmesa->xsm_buffer->backimage->devPriv != 0) 
   {
      buf = (struct i810_dest_buffer *) xsmesa->xsm_buffer->backimage->devPriv;
      zb = i810CreateZBuffer( buf );

      if (!zb) {
	 /* We need a depth buffer, but cannot allocate it. So, disable hardware
	    acceleration. */
	 i810_setup_DD_pointers_no_accel( ctx );
      }

      if ( i810glx.logLevel >= 1 )  {
	 i810DumpDB(buf);
      }
   }

   /* allocate new depth buffer, but don't initialize it */
   /* FIXME: we shouldn't need to do this if we got a hardware buffer */
   ctx->Buffer->Depth = (GLdepth *) malloc(ctx->Buffer->Width * 
					   ctx->Buffer->Height * 
					   sizeof(GLdepth));
   if (!ctx->Buffer->Depth)
   {
      /* out of memory */
      ctx->Depth.Test = GL_FALSE;
      ErrorF("Couldn't allocate depth buffer\n" );
   }

}

/*
 * i810GLXDestroyImage
 */
void i810GLXDestroyImage(GLXImage* image)
{
   if (image->devPriv) {
      i810DestroyDestBuffer((struct i810_dest_buffer *)image->devPriv);
      if ( i810glx.logLevel >= 10 )  {
	 i810Msg(1, "\nAfter destroy image\nCard heap:\n");
	 mmDumpMemInfo( i810glx.cardHeap );
	 i810Msg(1, "System heap:\n");
	 mmDumpMemInfo( i810glx.sysmemHeap );
      }
   } else if (image->data)
      free(image->data);
   xfree(image);
}

/*
 * i810GLXCreateImage
 * We can't accelerate rendering to images
 */
GLXImage* i810GLXCreateImage(WindowPtr pwindow, int depth, 
			     int width, int height)
{
   int Attrib;
   struct i810_dest_buffer *buf;
   GLXImage* image=(GLXImage*)xalloc(sizeof(GLXImage));

   if (!image)
      return (NULL);
   image->pwin = pwindow;
   image->width=width;
   image->height=height;
   image->bits_per_pixel=depth;
   image->data=0;
   image->devPriv=NULL;

   Attrib = -1;
   switch (depth)
   {
   case 8:
      Attrib = DV_PF_INDEX;
      break;
   case 15:
      Attrib = DV_PF_555;
      break;
   case 16:
      Attrib = DV_PF_565;
      break;
   default:
      i810Error("Unknown width in GLXCreateImage\n");
      break;
   }
   if (Attrib != -1)
      buf = i810CreateDestBuffer(Attrib,width,height);
   else
      buf = NULL;
   image->devPriv = (void *) buf;
   if (!image->devPriv) {
      image->bytes_per_line=PixmapBytePad(width, depth);
      image->data = (char *) malloc( image->height * image->bytes_per_line );
      if (!image->data)
      {
	 i810Error("i810GLXCreateImage: malloc failed.");
	 xfree(image);
	 image = NULL;
      }
   } else {
      buf->refcount++;
      image->bytes_per_line=buf->Pitch;
      image->width = buf->Pitch / buf->BytesPerPixel;
      image->data = buf->BufAddr;
   }
   if ( i810glx.logLevel >= 1 )  {
      fprintf(stderr, "After i810GLXCreateImage\nCard heap:\n");
      mmDumpMemInfo( i810glx.cardHeap );
      fprintf(stderr, "System heap:\n");
      mmDumpMemInfo( i810glx.sysmemHeap );
   }
   return (image);
}


/*
 * i810BackToFront
 *
 * Blit the visible rectangles from the back buffer to the screen
 * Can't really do this from a direct context - don't have the clip
 * info, and getting it would be a lot slower than just sending a
 * request to the server to do the blit there.
 *
 * drawable -- front buffer 
 * buf      -- back buffer
 */
int i810BackToFront(DrawablePtr drawable, 
		    struct i810_dest_buffer *buf) 
{
   RegionPtr		prgnClip;
   BoxPtr		pbox;
   int 			i,nbox;
   int 			ofs, xorg, yorg, pitch;

   if ( !GLXSYM(xf86VTSema) ) {
      i810Error("BackToFront(): !xf86VTSema\n");
      return BadMatch;	/* Is this really an error? */
   }

   if ( drawable->width != buf->Width || 
	drawable->height != buf->Height || 
	drawable->type != DRAWABLE_WINDOW ) {
      i810Error("BackToFront(): bad drawable\n");
      return BadMatch;
   }

   prgnClip = &((WindowPtr)drawable)->clipList;
   pbox = REGION_RECTS(prgnClip);
   nbox = REGION_NUM_RECTS(prgnClip);
   if( !nbox ) return Success;

   xorg = drawable->x;
   yorg = drawable->y;
   pitch = buf->Pitch;
   ofs = buf->MemBlock->ofs; 

   {
      unsigned int BR13 = ((GLXSYM(vga256InfoRec).displayWidth * 
			    GLXSYM(vgaBytesPerPixel)) |
			   (0xCC << 16));


      for (i=0; i < nbox; i++, pbox++) {
	 int x = pbox->x1 - xorg;
	 int y = pbox->y1 - yorg;
	 int w = pbox->x2 - pbox->x1;
	 int h = pbox->y2 - pbox->y1;

	 int start = ofs + x*GLXSYM(vgaBytesPerPixel) + y*pitch;
	 int dst = ((pbox->x1 + pbox->y1 * GLXSYM(vga256InfoRec).displayWidth) * 
		    GLXSYM(vgaBytesPerPixel));

	 {
	    BEGIN_BATCH( 6 );
	    OUT_BATCH( BR00_BITBLT_CLIENT | BR00_OP_SRC_COPY_BLT | 0x4 );
	    OUT_BATCH( BR13 );	/* dest pitch, rop */

	    OUT_BATCH( (h << 16) | (w * GLXSYM(vgaBytesPerPixel)));
	    OUT_BATCH( dst );

	    OUT_BATCH( pitch );	/* src pitch */
	    OUT_BATCH( start );
	    ADVANCE_BATCH();
	 }
      }
   }
  
   return Success;   
}


/*
 * ClearBox
 * Add hardware commands to draw a filled box for the
 * debugging display.
 */
static void ClearBox( int x, int y, int w, int h, int r, int g, int b ) 
{
   int start = (i810DB->MemBlock->ofs + 
		y * i810DB->Pitch + 
		x * GLXSYM(vgaBytesPerPixel));

   BEGIN_BATCH(6);
   
   OUT_BATCH( BR00_BITBLT_CLIENT | BR00_OP_COLOR_BLT | 0x3 );
   OUT_BATCH( BR13_SOLID_PATTERN | 
	      (0xF0 << 16) |
	      i810DB->Pitch );
   OUT_BATCH( (h << 16) | (w * GLXSYM(vgaBytesPerPixel)));
   OUT_BATCH( start );
   OUT_BATCH( i810PackColor( r, g, b, 0 ) );
   OUT_BATCH( 0 );

   if (0) 
      fprintf(stderr, "box %d,%x %dx%d col %x (%d,%d,%d)\n", x,y,w,h,
	      i810PackColor( r, g, b, 0 ), r, g, b);

   ADVANCE_BATCH();

   
}


/*
 * performanceBoxes
 * Draw some small boxesin the corner of the buffer
 * based on some performance information
 */
void i810PerformanceBoxes( int is_direct ) {
   int		w, t;
	
   if ( !i810glx.boxes || !i810DB ) {
      return;
   }

   /* draw a box to show we are active, so if it is't seen
      it means that it is completely software based rendering  */
   /* draw a purple box if we are direct rendering */
   if ( is_direct ) {			/* purple = direct (client dma) rendering */
      ClearBox( 4, 4, 8, 8, 255, 0, 255 );
   } else if ( i810glx.dmaDriver ) {	/* white = server dma rendering */
      ClearBox( 4, 4, 8, 8, 255, 255, 255 );
   } else {				/* grey = servery PDMA */
      ClearBox( 4, 4, 8, 8, 128, 128, 128 );
   }
	
   /* draw a red box if we had to wait for drawing to complete 
      (software render or texture swap) */
   if ( i810glx.c_drawWaits ) {
      ClearBox( 16, 4, 8, 8, 255, 0, 0 );
      i810glx.c_drawWaits = 0;
   }
	
   /* draw a blue box if the register protection signal was hit */
   if ( i810glx.c_signals ) {
      ClearBox( 28, 4, 8, 8, 0, 0, 255 );
      i810glx.c_signals = 0;
   }
	
   /* draw a yellow box if textures were swapped */
   if ( i810glx.c_textureSwaps ) {
      ClearBox( 40, 4, 8, 8, 255, 255, 0 );
      i810glx.c_textureSwaps = 0;
   }
	
	
   /* draw a green box if we had to wait for dma to complete (full utilization) 
      on the previous frame */
   if ( !i810glx.hardwareWentIdle ) {
      ClearBox( 64, 4, 8, 8, 0, 255, 0 );
   }
   i810glx.hardwareWentIdle = 0;
	
   /* show buffer utilization */
   if ( i810glx.c_dmaFlush > 1 ) {
      /* draw a solid bar if we flushed more than one buffer */
      ClearBox( 4, 16, 252, 4, 255, 32, 32 );
   } else {
      /* draw bars to represent the utilization of primary and secondary buffers */
      ClearBox( 4, 16, 252, 4, 32, 32, 32 );
      t = i810glx.dma_buffer->mem.Size;
      w = 252 * i810glx.dma_buffer->head / t;
      if ( w < 1 ) {
	 w = 1;
      }
      ClearBox( 4, 16, w, 4, 196, 128, 128 );
   }
   i810glx.c_dmaFlush = 0;
}

static void i810SendDmaFlush( void )
{
   if (!I810_NO_FLUSH) {
      BEGIN_BATCH(2);
      OUT_BATCH( INST_PARSER_CLIENT | INST_OP_FLUSH );
      OUT_BATCH( 0 );
      ADVANCE_BATCH();
   }
}


/*
 * Copy the back buffer to the front buffer.  If there's no back buffer
 * this is a no-op.  Only called in indirect contexts.
 */
static void i810ServerSwapBuffers( XSMesaBuffer b )
{
   /* make sure mesa gives us everything */
   if (i810Ctx && i810Ctx->gl_ctx)
      glFlush();

   if ( !b->db_state ) {
      /* not double buffered, so do nothing */
   } else {
      ValidateGC(b->frontbuffer, b->cleargc);
      if ( b->backimage ) {

	 struct i810_dest_buffer *buf = 
	    (struct i810_dest_buffer *)b->backimage->devPriv;

	 if ( buf && buf->Drawable ) {

	    /* flush the last primitive */
	    i810FinishPrimitive();

	    /* Drop a flush op into the dma stream */
	    i810SendDmaFlush();

	    /* diagnostic drawing tools */
	    i810PerformanceBoxes( 0 );

	    /* hardware accelerated back to front blit */	
	    i810BackToFront( (DrawablePtr)b->frontbuffer,buf );

	    /* make sure all dma is going to get executed if
	     * everything has gone well, this will be the only flush
	     * each frame 
	     */
	    i810DmaFlush();

	 } else {
	    /* Use backimage's dimension (not buffer's) */
	    (*b->cleargc->ops->PutImage)((DrawablePtr)b->frontbuffer,
					 b->cleargc,
					 b->frontbuffer->depth,
					 0, 0,
					 b->backimage->width, 
					 b->backimage->height,
					 0, ZPixmap, b->backimage->data);
	 }
      } else {
	 /* Copy pixmap to window on server */
	 (*b->cleargc->ops->CopyArea)((DrawablePtr)b->backpixmap,
				      b->frontbuffer,
				      b->cleargc,
				      0, 0, b->width, b->height,
				      0, 0);
      }
   }
  
   /* report performance counters */
   i810Msg(10, "swapBuffers: c_triangles:%i  c_lines:%i c_points:%i c_setup:%i c_textures:%i\n", 
	   i810glx.c_triangles, i810glx.c_lines, i810glx.c_points, i810glx.c_setupPointers,
	   i810glx.c_textureSwaps );
   i810glx.c_triangles = 0;
   i810glx.c_lines = 0;
   i810glx.c_points = 0;
   i810glx.c_setupPointers = 0;
   i810Msg(10, "---------------------------------------------------------\n" );
}


/*
 * DoMakeCurrent
 * Changing either the context or the buffer will force this.
 */
static int DoMakeCurrent(XSMesaContext c, XSMesaBuffer b) {
   i810ContextPtr ctx, oldctx = i810Ctx;
   struct i810_dest_buffer *buf, *oldbuf = i810DB;
  
   i810Msg( 10, "DoMakeCurrent( %p, %p )\n", c, b );

   i810Ctx = NULL;
   i810DB = NULL;
	
   if ( c ) {
      ctx = c->hw_ctx;
      if (!ctx) {
	 return -1;
      }
   } else {
      ctx = NULL;
   }
	
   if ( b && b->buffer == XIMAGE ) {
      buf = (struct i810_dest_buffer *) b->backimage->devPriv;
      if (!buf->Drawable) {
	 buf = NULL;
      }
   } else {
      buf = NULL;
   }

   if ( !ctx && buf ) {
      return -1;
   }
   if (!ctx) {
      return 0;
   }
	
   ctx->DB = buf;

   /* valid conditions to render with */
   i810Ctx = ctx;
   i810DB = ctx->DB;	

   /* Even this is overkill: 
    */
   if (oldctx != ctx || oldbuf != i810DB) {

      if (i810DB) {
	 i810DmaExecute( i810DB->Setup, I810_DEST_SETUP_SIZE );
	 i810Ctx->Setup[I810_CTXREG_SCI2] = (i810DB->Width | 
					     (i810DB->Height << 16));
      }

      i810Ctx->new_state |= I810_NEW_CONTEXT;

      if (MESA_VERBOSE&VERBOSE_DRIVER)
	 fprintf(stderr, "needEnter3D in DoMakeCurrent\n");
   }


   return 0;
}


/*
 * i810GLXMakeCurrent
 * Note that GLX calls this every batch of commands,
 * so it will show up several times a frame even if
 * the client application isn't issuing makeCurrent calls.
 */ 
GLboolean i810GLXMakeCurrent( XSMesaContext c ) {
   if (c == XSMesa) {
      return (GL_TRUE);
   }
	
   /* mesa 3.1 will sometimes leave unflushed vertexes */
   if ( XSMesa ) {
      glFlush();
   }

   i810Msg( 10, "i810GLXMakeCurrent( %p )\n", c );
   XSMesa = c;                                                                 
   if (c) {
      DoMakeCurrent(c,c->xsm_buffer);
      gl_make_current(c->gl_ctx, 
		      c->xsm_buffer ? c->xsm_buffer->gl_buffer : NULL);
   } else {
      /* detach */
      DoMakeCurrent(NULL,NULL);
      gl_make_current(NULL, NULL);
   }
   return(GL_TRUE); 
}

/*
 * Bind buffer b to context c and make c the current rendering context.
 */
GLboolean i810GLXBindBuffer( XSMesaContext c, XSMesaBuffer b ) {
   i810Msg( 10, "i810GLXBindBuffer( %p, %p )\n", c, b );
   DoMakeCurrent(c,b);
   return(XSMesaBindBuffer(c,b));
}


/*
 * Create a new XSMesaContext.
 * Input:  v - XSMesaVisual
 *         share_list - another XSMesaContext with which to share display
 *                      lists or NULL if no sharing is wanted.
 * Return:  an XSMesaContext or NULL if error.
 */
XSMesaContext i810GLXCreateContext( XSMesaVisual v,  
				    XSMesaContext share_list )
{
   XSMesaContext c;

   i810Msg (0, "### Creating new xsmesa context for i810 ...\n" );

   c = (XSMesaContext) calloc( 1, sizeof(struct xsmesa_context) );
   if (!c)
   {
      return(NULL);
   }

   c->gl_ctx = gl_create_context(v->gl_visual,    
				 share_list ? share_list->gl_ctx : NULL,
				 (void *) c, GL_TRUE /* direct rendering */);
   if (!c->gl_ctx)
   {
      free(c);
      return(NULL);
   }

   c->xsm_visual = v;
   c->xsm_buffer = NULL;       /* set later by XSMesaMakeCurrent */
   c->pixelformat = v->dithered_pf;    /* Dithering is enabled by default */
   c->hw_ctx = (void *)i810CreateContext(c->gl_ctx);
   if (!c->hw_ctx) {
      i810Error("Cannot create hardware specific context.\n");
      return (NULL);
   }
   ((i810ContextPtr)(c->hw_ctx))->refcount++;
   c->gl_ctx->Driver.UpdateState = i810_setup_DD_pointers;

   i810Msg( 1, "i810GLXCreateContext succeeded: %p\n", c );

   return(c);
}

void i810GLXDestroyContext( XSMesaContext c ) {
   i810Msg( 1, "i810GLXDestroyContext( %p )\n", c );
	
   /* make sure all drawing is completed */
   i810WaitDrawingEngine();

   if (i810DestroyContext((i810ContextPtr) c->hw_ctx) != 0) {
      i810Error("i810GLXDestroyContext(): i810DestroyContext() failed!\n");
   }
   XSMesaDestroyContext(c);
   if ( XSMesa == c ) {
      XSMesa = 0;
   }
}




