/*
 * GLX Hardware Device Driver for Matrox Millenium G200
 * Copyright (C) 1999 Wittawat Yamwong
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR 
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE 
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 *
 *    Wittawat Yamwong <Wittawat.Yamwong@stud.uni-hannover.de>
 */

/* $Id: mgalib.h,v 1.27 2000/04/16 17:39:10 keithw Exp $ */

#ifndef MGALIB_INC
#define MGALIB_INC

#include <stdio.h>

#include "mesaglx/types.h"

#include "glx_symbols.h"
#include "mga.h"


#include "mgacommon.h"
#include "mm.h"
#include "g200_mac.h"
#include "hwlog.h"
#include "mgadma.h"
#include "mgatex.h"
#include "mgavb.h"


#ifndef MESA31
#error "The MGA driver now requires Mesa 3.1 or higher"
#endif


/*
 * buffer definitions
 */
 
#define mgaBufferMagic 0x65e813a1
#define VALID_MGA_BUFFER(b) ((b) && (b)->magic == mgaBufferMagic)

/* mga_buffer_t.Attrib */
#define MGA_TYPE_MASK 0xf
#define MGA_COLORBUFFER 1
#define MGA_ZBUFFER16   2
#define MGA_MISCBUFFER  3
#define MGA_ZBUFFER32   4
#define MGA_TEXBUFFER   5

/* if type == MGA_COLORBUFFER */
#define MGA_PF_MASK  0xf0
#define MGA_PF_INDEX 0
#define MGA_PF_565   (1 << 4)
#define MGA_PF_555   (9 << 4)
#define MGA_PF_888   (3 << 4)
#define MGA_PF_8888  (10 << 4)
#define MGA_PF_HASALPHA (8 << 4)

#define	MGA_FORCE_SYSMEM	(1<<16)


/* mga_buffer_t.Setup */
#define MGA_SETUP_0        0x27262023
#define MGA_SETUP_PITCH    1
#define MGA_SETUP_CXBNDRY  2
#define MGA_SETUP_YTOP     3
#define MGA_SETUP_YBOT     4
#define MGA_SETUP_5        0x030701ae
#define MGA_SETUP_DSTORG   6
#define MGA_SETUP_MACCESS  7
#define MGA_SETUP_PLNWT    8
#define MGA_SETUP_ZORG     9
#define MGA_SETUP_SIZE     10

/* Some phony ones for the reg_dirty bitfield:
 */
#define MGA_SETUP_DWGCTL    11
#define MGA_SETUP_ALPHACTRL 12
#define MGA_SETUP_FOGCOLOR  13
#define MGA_SETUP_TDUAL0    14
#define MGA_SETUP_TDUAL1    15
#define MGA_SETUP_WFLAG     16


struct mga_buffer_t {
	hwUI32		magic;
	struct mga_buffer_t *next;

	int		SystemMemory:1;		/* in PCI/AGP space instead of framebuffer */

	hwUI32 Setup[MGA_SETUP_SIZE];
	int		SetupSize;

	WindowPtr	pwindow;
	GLvisual	visual;
	
	void		*backBuffer;		/* pointer to buffer in VM */
	PMemBlock	backBufferBlock;	/* location in card's space, NULL if software only */

	void		*depthBuffer;
	PMemBlock	depthBufferBlock;

	int		colorPixelFormat;	/* MGA_PF_555, MGA_PF_565, MGA_PF_888 */
	
	int		width,height;		/* in pixels */
	int		pitch;				/* in pixels */
	
	int		bytesPerPixel;
	int		bytesPerDepth;		/* may include embedded stencil buffer */
	int		hasStencil;
	
	XSMesaBuffer xsBuf;				/* needed for front buffer auto-swap */
};

typedef struct mga_buffer_t mgaBuffer;
typedef struct mga_buffer_t *mgaBufferPtr;


#define mgaContextMagic 0x47323030
#define VALID_MGA_CONTEXT(ctx) ((ctx) && (ctx)->magic == mgaContextMagic)

/* SoftwareFallback 
 *    - texture env GL_BLEND -- can be fixed 
 *    - 1D and 3D textures
 *    - incomplete textures
 */
#define FALLBACK_TEXTURE   1


/*
 * context definitions
 */
 
 
/* For mgaCtx->new_state.
 */
#define MGA_NEW_DEPTH   0x1
#define MGA_NEW_ALPHA   0x2
#define MGA_NEW_FOG     0x4
#define MGA_NEW_CLIP    0x8
#define MGA_NEW_MASK    0x10
#define MGA_NEW_TEXTURE 0x20
#define MGA_NEW_CULL    0x40
#define MGA_NEW_WARP    0x80
#define MGA_NEW_CONTEXT 0x100
#define MGA_NEW_STENCIL 0x200


typedef void (*mga_interp_func)( GLfloat t, 
				 GLfloat *result,
				 const GLfloat *in,
				 const GLfloat *out );
				 

struct mga_context_t {
   hwUI32 magic;

   mgaBufferPtr DB;
   GLcontext *gl_ctx;

   hwUI32 regDWGCTL;
   hwUI32 regALPHACTRL;
   hwUI32 regSTENCIL;
   hwUI32 regSTENCILCTL;   
   hwUI32 regFOGCOLOR;
   hwUI32 regWFLAG;
   hwUI32 regTDUAL[2];		/* G400 dual texturing */

   /* Variable sized vertices
    */
   hwUI32 vertsize;

   /* Map GL texture units onto hardware.
    */
   hwUI32 multitex;
   hwUI32 tmu_source[2];
   hwUI32 tex_dest[2];
   hwUI16 GlobalPalette[256];
   	
   unsigned int SoftwareFallback;  /* or'ed values of FALLBACK_* */
   unsigned int IndirectTriangles;

   /* Support for CVA and the fast paths */
   unsigned int setupdone;
   unsigned int setupindex;
   unsigned int renderindex;
   unsigned int using_fast_path;
   unsigned int using_immediate_fast_path;
   mga_interp_func interp;

   /* Shortcircuit some state changes */
   points_func   PointsFunc;
   line_func     LineFunc;
   triangle_func TriangleFunc;
   quad_func     QuadFunc;

   /* Manage our own state */
   GLuint        new_state; 
   GLuint        reg_dirty;

   GLfloat       map[16];
};


extern GLuint mgaDDRegisterPipelineStages( struct gl_pipeline_stage *out,
					   const struct gl_pipeline_stage *in,
					   GLuint nr );

extern GLboolean mgaDDBuildPrecalcPipeline( GLcontext *ctx );


/*
 * Driver functions that uses mgaCtx or mgaDB and can be called
 * directly from API such as glClear() should use CHECK_CONTEXT
 * before doing anything with mgaCtx and mgaDB.
 *
 * Drawing functions such as Triangle are protected by CHECK_CONTEXT
 * in mgaRenderVB, so that they don't need it.
 *
 * mgaDB will be invalid when the frame buffer have to be reallocate
 * but there was not enough video memory avialable. 
 * TODO: fix this.
 */
#define CHECK_CONTEXT(x) do {					\
  if (!VALID_MGA_CONTEXT(mgaCtx) || !VALID_MGA_BUFFER(mgaDB)) {	\
    x								\
  }								\
} while(0)

typedef struct {
	mgaBufferPtr	bufferList;

	/* dma stuff */
	hwUI32		dmaDriver;	
        hwUI32         dmaSize;
        hwUI32         dmaAdr;
        hwUI32         cmdSize;
        hwUI32         cardCmds;
        hwUI32         systemTexture;
        hwUI32         noSetupDma;

        int             use_agp;
        int             isSdram;
        int			isG200;
	int			isG400;			/* a G200 will have 0 */

        /* bookkeeping for textureing */
        struct mga_texture_object_s *TexObjList;
        struct mga_texture_object_s *currentTextureObject[2];
        hwUI32 			currentTexturePalette;
        hwUI32         default32BitTextures;
        hwUI32         swapBuffersCount;
        
	/* warp stuff */
	hwUI32	       *warp_seriesStart;		/* NULL if not between
											 RenderStart/RenderFinish */ 
        hwUI32         warp_index;             /* current active pipe */
        hwUI32         warp_no_pipe_swaps;
        hwUI32         warp_residual_pipe;

	/* options */
	hwUI32		nullprims;  /* skip all primitive generation */
	hwUI32		boxes;	    /* draw performance boxes */
	hwUI32		noFallback; /* don't fall back to software, do
                                       best-effort rendering */
	hwUI32		nosgram;    /* don't use block clears */
 	hwUI32		skipDma;    /* don't send anything to the hardware */
	hwUI32		waitVSync;  /* Wait for vertical sync before swapping buffers */
 	
	/* performance counters */	
	hwUI32		c_textureUtilization;
	hwUI32		c_textureSwaps;		/* number of swap outs due to low memory */
	hwUI32		c_textureDwords;	/* image and palette secondary dwords */
	hwUI32		c_setupPointers;
	hwUI32		c_triangles;
	hwUI32		c_points;
	hwUI32		c_lines;
	hwUI32		c_drawWaits;
	hwUI32		c_signals;
	hwUI32		c_dmaFlush;
	hwUI32		c_overflows;
	hwUI32		c_paletteLoads;		/* number of times texture palettes were updated */
	hwUI32		c_paletteBlits;		/* updates forced by bindTexture */
		
	hwUI32		hardwareWentIdle; /* cleared each swapbuffers,
					     set if a
					     waitfordmacompletion ever
					     exited without having to
					     wait */
} mgaGlx_t;

extern mgaGlx_t	mgaglx;
extern mgaBufferPtr mgaDB;
extern mgaBufferPtr mgaFrontBuffer;
extern mgaContextPtr mgaCtx;


int	usec( void );

int mgaMakeCurrent(mgaContextPtr ctx, mgaBufferPtr buf);

#define MGAPACKCOLOR555(r,g,b,a) \
  ((((r) & 0xf8) << 7) | (((g) & 0xf8) << 2) | (((b) & 0xf8) >> 3) | \
    ((a) ? 0x8000 : 0))

#define MGAPACKCOLOR565(r,g,b) \
  ((((r) & 0xf8) << 8) | (((g) & 0xfc) << 3) | (((b) & 0xf8) >> 3))

#define MGAPACKCOLOR888(r,g,b) \
  (((r) << 16) | ((g) << 8) | (b))

#define MGAPACKCOLOR8888(r,g,b,a) \
  (((a) << 24) | ((r) << 16) | ((g) << 8) | (b))

#define MGAPACKCOLOR4444(r,g,b,a) \
  ((((a) & 0xf0) << 8) | (((r) & 0xf0) << 4) | ((g) & 0xf0) | ((b) >> 4))


#define MGA_DEBUG 0   
#ifndef MGA_DEBUG
extern int MGA_DEBUG;
#endif

#define MGA_DEBUG_ALWAYS_SYNC    0x1
#define MGA_DEBUG_VERBOSE_MSG    0x2
#define MGA_DEBUG_OUTREG         0x4


static __inline__ hwUI32 mgaPackColor(hwUI8 r, hwUI8 g, 
				   hwUI8 b, hwUI8 a)
{
  switch (mgaDB->colorPixelFormat) {
  case MGA_PF_555:
    return MGAPACKCOLOR555(r,g,b,a);
  case MGA_PF_565:
    return MGAPACKCOLOR565(r,g,b);
  case MGA_PF_888:
    return MGAPACKCOLOR888(r,g,b);
  case MGA_PF_8888:
    return MGAPACKCOLOR8888(r,g,b,a);
  default:
    return 0;
  }
}

#endif
