#include <X11/Xlib.h>
#include <stdio.h>
#include <stdlib.h>
#include <dlfcn.h>

#include "glxlib.h"

#undef Bool

#include "dixstruct.h"
#include "glxcommon.h"

#include "direct_protocol.h"
#include "extensions.h"
#include "pointers.h"

/* All interactions with this protocol will be atomic.  Threading is
 * not considered.
 */
static ClientRec *directClient;


/* It is also a lot easier if there is just a single direct display.
 */
Display *directDisplay;
int directScreenNum = 0;

/* The server->client communication chanel.
 */
char *glx_buf;
int glx_buf_tail, glx_buf_head;
static int glx_buf_size;

static int fake_XFlush( Display *dpy );
static int fake_XSync( Display *dpy, Bool discard );
static void fake_GLXRenderFlush( void );
static int fake_XRead( Display *dpy, char *data, long num_bytes);
static int fake_XReply( Display *display, xReply *reply, int, Bool );
static void flush_request( xReq *pending );

static void *handle;
static struct gl_api_table direct_api_tab;
static void (*__glx_release_hook)( void ) = 0;


Proto directProto = {
   fake_XReply,
   fake_XRead,
   0,				/* XSend */
   fake_GLXRenderFlush,
   fake_GLXRenderFlush,
   fake_XFlush,		
   fake_XSync,		
   flush_request,
   0,				/* pending request */
   &direct_api_tab,		/* filled via dlsym */
   0,                           /* glx table - points to __glx_dispatch_tab */
   True,			/* isDirect */
};

extern void GLXLargeRenderFlush();
extern void GLXRenderFlush();

Proto xProto = {
   _XReply,
   _XRead,
   _XSend,
   GLXRenderFlush,
   GLXLargeRenderFlush,
   XFlush,
   XSync,
   0,
   0,				/* pending */
   0,				/* api table -- fill me? */
   0,				/* glx table - not used */
   False			/* is_direct */
};



/* Call the appropriate function in glx.so.  Need to find out the
 * current x error handler & call it on failure.
 */
static void flush_request( xReq *pending ) 
{
   int opcode = pending->data;	/* glx_opcode */
   int len = pending->length; 
   int retval;
   int (*func)( ClientPtr );


   if (glx_buf_tail - glx_buf_head > 0) 
      fprintf(stderr, 
	      "fake_protocol - discarding %d uncollected bytes in buffer\n", 
	      glx_buf_tail - glx_buf_head);

   glx_buf_tail = glx_buf_head = 0;
   directClient->requestBuffer = directProto.pending;
   directClient->req_len = len;
   directProto.pending = 0;

   func = ((int (**)( ClientPtr ))directProto.dispatch_table)[opcode];
   retval = func( directClient );
   
   if (retval != Success) {
      fprintf(stderr, 
	      "Proper X error handling not implemented.\n"
	      "X Error %d for GLX request opcode %d\n",
	      retval, opcode );
      
      /* Clean up hardware and exit.
       */
      exit(1);
   }
}


static int fake_XReply( Display *display, xReply *reply, int who, Bool knows )
{
   if (directProto.pending) flush_request( directProto.pending );

   if (glx_buf_tail - glx_buf_head < sizeof(xReply)) {
      fprintf(stderr, "not enough bytes for an xReply in buffer!\n");
      return 0;
   }

   *reply = *(xReply *)(glx_buf + glx_buf_head);
   glx_buf_head += sizeof(xReply);
   return 1;
}


static int fake_XRead( Display *dpy, char *data, long num_bytes )
{
   if (directProto.pending) flush_request( directProto.pending );

   num_bytes = (num_bytes + 3) & ~3;
   memcpy(data, glx_buf + glx_buf_head, num_bytes);
   glx_buf_head += num_bytes;
   return 1;
}

static void fake_GLXRenderFlush( void )
{
}
   
static int fake_XSync( Display *dpy, Bool discard )
{
   return 1;		       
}

static int fake_XFlush( Display *dpy )
{
   return 1;		
}



/* The server side of the fake protocol.  No need to use indirection
 * as glx.so will only ever be direct or indirect - it doesn't have to
 * switch at runtime.  
 */
int WriteToClient(ClientPtr who, int count, char* server_buf)
{
   count = (count + 3) & ~3;

   if (glx_buf_tail + count > glx_buf_size) 
   {
      while ((glx_buf_size *= 2) < count);
      glx_buf = (char *) realloc( glx_buf, glx_buf_size );
   }
   
   memcpy( glx_buf + glx_buf_tail, server_buf, count );
   glx_buf_tail += count;
   return count;
}

/* Flush the communications to the client.
 */
void FlushAllOutput(void)
{
}




Bool __glx_init_direct( Display *dpy, char *reply )
{
   ClientPtr (*init_func)( char *, Display *, int, int (*)(), void (**)(void) );

   fprintf(stderr, "got reply %s\n", reply);

   if (handle) 
      return 0;			/* should never happen */

   fprintf(stderr, "try to open /usr/X11R6/lib/modules/glx.so:\n");
 
   /* FIXME: this path shouldn't be hardwired! */
   if ((handle = dlopen("/usr/X11R6/lib/modules/glx.so", RTLD_LAZY)) == 0) {
      fputs (dlerror(), stderr);
      fputs ("\n", stderr);
      goto error_case;
   }

   fprintf(stderr, "opened /usr/X11R6/lib/modules/glx.so\n");

   /* Call initialization functions in glx.so.
    */
   if ((init_func = dlsym(handle, reply)) == 0)
      goto error_case;

   fprintf(stderr, "found init func\n");

   if (!__glx_init_direct_execution_pointers( handle, &direct_api_tab ))
      goto error_case;

   fprintf(stderr, "got api pointers\n");

   if (!(directProto.dispatch_table = dlsym(handle, "__glx_dispatch_table")))
      goto error_case;

   fprintf(stderr, "got dispatch table\n");
   fprintf(stderr, "calling init func:\n");

   directClient = init_func(reply, dpy, 0,
			    (int (*)()) __glx_send_vendor_private_with_reply,
			    (void (**)()) &__glx_release_hook);
   
   if (!directClient || !__glx_release_hook)
      goto error_case;

   directDisplay = dpy;
   glx_buf_size = 512;
   glx_buf = (char *) malloc( glx_buf_size );
   glx_buf_head = glx_buf_tail = 0;
   return 1;

 error_case:
   __glx_unload_direct();
   fprintf(stderr, "failed to load glx.so\n");
   return 0;
}

void __glx_unload_direct( void )
{
   if (handle) {
      if (__glx_release_hook)
	 __glx_release_hook();
      dlclose(handle);
      handle = 0;
      __glx_release_hook = 0;
      directDisplay = 0;
   }
}


static void __attribute__ ((destructor))
unload_direct( void )
{
   __glx_unload_direct();
}







