/*
 * GLX Hardware Device Driver for SiS 6326
 * Copyright (C) 2000 Jim Duchek
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Based on Mach64 driver: mach64state.c
 *
 *    Jim Duchek <jim@linuxpimps.com>
 */

#include <stdio.h>

#include "xsmesaP.h"
#include "sis6326glx.h"
#include "types.h"
#include "pb.h"
#include "dd.h"

#include "glx_symbols.h"

/*
 * sis6326IsTexturingEnabled
 */
struct gl_texture_object *sis6326IsTexturingEnabled( const GLcontext *ctx ) {
	struct gl_texture_object	*tObj;

	if ( !( ctx->Texture.Enabled & (TEXTURE0_1D|TEXTURE0_2D|TEXTURE1_1D|TEXTURE1_2D) ) ) {
		return 0;
	}

	tObj = ctx->Texture.Unit[0].Current;
	if ( !tObj ) {
		return 0;
	}

	/* check for weird mesa state where it looks like we
	are enabled but the texture object's image is an invalid pointer.
	This should probably be fixed in mesa someplace */
	if ( ( tObj != ctx->Texture.Unit[0].CurrentD[2] ) ) {
		hwMsg( 5, "sis6326IsTexturingEnabled: wtf???\n" );
		return 0;
	}

	return tObj;
}

static int sis6326CalcTEX_CNTL( const GLcontext *ctx ) {
	struct gl_texture_object	*tObj;
	sis6326TextureObject_t *t;
	int ret;
	CARD16 *tmp;
	int i, j;
	int out = 0;
	int con;
	int pitch1, pitch2;
	int lev;
	int w2;
	int tset1 = 0;


	if ( !(tObj = sis6326IsTexturingEnabled( ctx )) ) {
		return 0;
	}

	t = tObj->DriverData;
	
	ret = S_ENABLE_TEXMAP |	S_ENABLE_TEXCACHE | S_ENABLE_LARGECACHE;
		
	ret |= S_ENABLE_TEXPERSP;

	ret |= S_ENABLE_SPECULAR;

	if (t->hasAlpha && (ctx->Texture.Unit[0].EnvMode == GL_REPLACE))
		ret |= S_ENABLE_BLENDING;

					       	
	return ret;
}

static int sis6326CalcSCALE_3D_CNTL( const GLcontext *ctx ) {
	struct gl_texture_object	*tObj;
}



static int sis6326CalcZ_CNTL( const GLcontext *ctx ) {
	int		zcntl;
	int 		zset = 0;

	if ( !ctx->Depth.Test ) {
		return 0;
	}

	zcntl = S_ENABLE_ZTEST;

	switch( ctx->Depth.Func )
	{
	case GL_NEVER:
	    zset = S_ZSET_PASS_NEVER; break;
	case GL_LESS:
	    zset = S_ZSET_PASS_NLD; break;
	case GL_LEQUAL:
	    zset = S_ZSET_PASS_NLED; break;
	case GL_EQUAL:
	    zset = S_ZSET_PASS_NED; break;
	case GL_GEQUAL:
	    zset = S_ZSET_PASS_NGED; break;
	case GL_GREATER:
	    zset = S_ZSET_PASS_NGD; break;
	case GL_NOTEQUAL:
	    zset = S_ZSET_PASS_NNED; break;
	case GL_ALWAYS:
	    zset = S_ZSET_PASS_ALWAYS; break;
	default:
	    break;
	}

	zset |= (sis6326DB->pitch * 2);
	zset |= S_ZSET_FORMAT_16;
	OUTREG(SIS6326_TRI_ZSETTING, zset);
	OUTREG(SIS6326_TRI_ZBASE, sis6326DB->depthBufferBlock->ofs);
	
	
	if ( ctx->Depth.Mask ) {
		zcntl |= S_ENABLE_ZWRITE;	/* enable writes to depth buffer */
	} 


	return zcntl;
}


static int sis6326CalcALPHA_TST_CNTL( GLcontext *ctx ) 
{
}


/* =============================================================
 */


/*
 * sis6326UpdateTextureState
 */
void sis6326UpdateTextureState( GLcontext *ctx )
{
	struct gl_texture_object	*tObj;
	sis6326TextureObject_t		*t;
	int tset1 = 0, tset2 = 0, pitch1, pitch2, lev, w2, i;
	
	/* init the first texture */
	tObj = ctx->Texture.Unit[0].Current;

	/* move a texture to the card if needed */
	if ( !tObj->DriverData ) {
		sis6326CreateTexObj( sis6326Ctx, ctx->Texture.Unit[0].Current );
	}
	
	t = (sis6326TextureObject_t *)tObj->DriverData;

	/* note it as recently used to keep from swapping out */
	t->age = sis6326glx.swapBuffersCount;


	WAITFIFOEMPTY(20);

	if (sis6326glx.texChange) {
		tset1 |= S_TSET1_TEX_CLRCACHE;
		sis6326glx.texChange = 0;
	}

	switch (tObj->MagFilter) {
	case GL_NEAREST:
		tset1 |= S_TSET1_TEX_MAG_NEAREST; break;
	case GL_LINEAR:
		tset1 |= S_TSET1_TEX_MAG_LINEAR; break;
	default:
		break;
	}

	switch (tObj->MinFilter) {
	case GL_NEAREST:
		tset1 |= S_TSET1_TEX_MIN_NEAREST; break;
	case GL_LINEAR:
		tset1 |= S_TSET1_TEX_MIN_LINEAR; break;
	case GL_NEAREST_MIPMAP_NEAREST:
		tset1 |= S_TSET1_TEX_MIN_NEAREST_MIP_NEAREST; break;
	case GL_NEAREST_MIPMAP_LINEAR:
		tset1 |= S_TSET1_TEX_MIN_NEAREST_MIP_LINEAR; break;
	case GL_LINEAR_MIPMAP_NEAREST:
		tset1 |= S_TSET1_TEX_MIN_LINEAR_MIP_NEAREST; break;
	case GL_LINEAR_MIPMAP_LINEAR:
		tset1 |= S_TSET1_TEX_MIN_LINEAR_MIP_LINEAR; break;
	default:
		break;
	}


	switch(tObj->WrapS) {
	case GL_CLAMP:
		tset1 |= S_TSET1_CLAMP_U;
		break;
	case GL_REPEAT:
		tset1 |= S_TSET1_WRAP_U;
		break;
	default:
		break;
	}

	switch(tObj->WrapT) {
	case GL_CLAMP:
		tset1 |= S_TSET1_CLAMP_V;
		break;
	case GL_REPEAT:
		tset1 |= S_TSET1_WRAP_V;
		break;
	default:
		break;
	}
	
	
	tset1 |= S_TSET1_USE_CTB | 0x8000;
	
	tset1 |= (t->maxLog2 << 8);
		
	OUTREG( SIS6326_TRI_TEXSETTING1, tset1 | t->textureFormat);
			
	switch (ctx->Texture.Unit[0].EnvMode) {
	case GL_REPLACE:
		tset2 |= S_TSET2_REPLACE | S_TSET2_ATEX;
		break;
	case GL_MODULATE:
		tset2 |= S_TSET2_MODULATE | S_TSET2_ATEXPIX;
		break;
	case GL_DECAL:
		tset2 |= S_TSET2_DECAL | S_TSET2_APIX;
		break;			
	case GL_BLEND:
		tset2 |= S_TSET2_BLEND;  // We don't REALLY support this mode. 
		tset2 |= S_TSET2_ATEXPIX;
		break; 
	default:
		break;
	}
	
	OUTREG( SIS6326_TRI_TEXSETTING2, tset2);
     
	lev = 0;
	for (i=SIS6326_TRI_TEXBASE0; i<=SIS6326_TRI_TEXBASE9; i+=4) {
		if (t->offsets[lev] == -1) {
			break;
		} else {
			OUTREG( i, (t->memBlock->ofs + t->offsets[lev]));
		}
		lev++;
	}

	w2 = t->widthLog2;
	lev = 0;

	for (i=SIS6326_TRI_TEXPITCH_0_1; i<=SIS6326_TRI_TEXPITCH_8_9; i+=4) {
		/*
		 * Allright.  Just don't ask about the formulas below.
		 * I discovered them by trial and error.  If you could
		 * possibly shed some sort of light on WHY they work,
		 * do let me know, as I haven't a clue. 
		 * -jrd
		 */
		if (t->texelBytes == 1) {
			pitch1 = 128 * (w2 - 2);
			pitch2 = 128 * (w2 - 3);
		} else {
			pitch1 = 128 * (w2 - 1);
			pitch2 = 128 * (w2 - 2);
		}
		w2 -= 2;
		if (t->offsets[lev] == -1) {
			break;
		} else {
			OUTREG( i, ((pitch1 << 16) | pitch2));
		}
		lev += 2;
	}

	OUTREG( SIS6326_TRI_TEX_W_H, ((t->widthLog2 << 28) | (t->heightLog2) << 24));

	OUTREG( SIS6326_TRI_TEX_BORDER_COLOR, ((tObj->BorderColor[3] << 24) |
					       (tObj->BorderColor[0] << 16) |
					       (tObj->BorderColor[1] << 8) |
					       (tObj->BorderColor[2])));


}

/* =============================================================
 */


/*
 * sis6326DDUpdateState
 * This will be called before any beginrender if state has changed
 */
void sis6326DDUpdateState( GLcontext *ctx ) {
	int		screenFormat;
	int		enable = 0;
	int 		i;
	int 		timeout = 0;

	hwMsg(10, "In state.\n");

	/* use our triangle function */
	if (ctx->Driver.TriangleFunc != sis6326Triangle)
		return;


	WAITFIFOEMPTY(32);

	sis6326glx.sendSpec = sis6326glx.sendTex = 0;
	
	sis6326glx.c_setupPointers++;
	
	/* Clear the vertex reuse buffer */
	sis6326glx.setupVertices[0] =
		sis6326glx.setupVertices[1] =
		sis6326glx.setupVertices[2] = (GLuint) -1;
		
	sis6326glx.setupMax = ( ctx->CompileCVAFlag ) ? ctx->VB->FirstFree : VB_MAX;

 	enable |= sis6326CalcZ_CNTL( ctx );

	if (ctx->Fog.Enabled) {
		int r, g, b;
		enable |= S_ENABLE_FOG;
		
		r = 255 * ctx->Fog.Color[0];
		g = 255 * ctx->Fog.Color[1];
		b = 255 * ctx->Fog.Color[2];
		
		sis6326glx.sendSpec = 1;
	
		OUTREG( SIS6326_TRI_FOGSETTING, (0x01000000 | 
				(r << 16) | (g << 8) | b));
	}




	if (ctx->Color.BlendEnabled) {
		int blend = 0;
		enable |= S_ENABLE_BLENDING;


		
		switch (ctx->Color.BlendSrcRGB) {
		case GL_ZERO:
			blend |= S_MISC2_SBLEND_ZERO; break;
		case GL_ONE:
			blend |= S_MISC2_SBLEND_ONE; break;
		case GL_DST_COLOR:
			blend |= S_MISC2_SBLEND_DST_COLOR; break;
		case GL_ONE_MINUS_DST_COLOR:
			blend |= S_MISC2_SBLEND_INV_DST_COLOR; break;
		case GL_SRC_ALPHA:
			blend |= S_MISC2_SBLEND_SRC_ALPHA; break;
		case GL_ONE_MINUS_SRC_ALPHA:
			blend |= S_MISC2_SBLEND_INV_SRC_ALPHA; break;				
		case GL_DST_ALPHA:
			blend |= S_MISC2_SBLEND_DST_ALPHA; break; // FIXME check for alpha buffer
		case GL_ONE_MINUS_DST_ALPHA:
			blend |= S_MISC2_SBLEND_INV_DST_ALPHA; break;				
		case GL_SRC_ALPHA_SATURATE:
			blend |= S_MISC2_SBLEND_SRC_ALPHA_SAT; break;
		default:
			// Card has more blending modes... hrm..
			break;
		}			

		switch (ctx->Color.BlendDstRGB) {
		case GL_ZERO:
			blend |= S_MISC2_DBLEND_ZERO; break;
		case GL_ONE:
			blend |= S_MISC2_DBLEND_ONE; break;
		case GL_SRC_COLOR:
			blend |= S_MISC2_DBLEND_SRC_COLOR; break;
		case GL_ONE_MINUS_SRC_COLOR:
			blend |= S_MISC2_DBLEND_INV_SRC_COLOR; break;
		case GL_SRC_ALPHA:
			blend |= S_MISC2_DBLEND_SRC_ALPHA; break;
		case GL_ONE_MINUS_SRC_ALPHA:
			blend |= S_MISC2_DBLEND_INV_SRC_ALPHA; break;
		case GL_DST_ALPHA:
			blend |= S_MISC2_DBLEND_DST_ALPHA; break; // FIXME check for alpha buffer
		case GL_ONE_MINUS_DST_ALPHA:
			blend |= S_MISC2_DBLEND_INV_DST_ALPHA; break;				
		default:
			break;
		}
	
		OUTREG(SIS6326_TRI_MISCSETTING2, (blend));
	} else {
		OUTREG(SIS6326_TRI_MISCSETTING2, (S_MISC2_SBLEND_SRC_ALPHA |
				S_MISC2_DBLEND_INV_SRC_ALPHA));
	}


	if (sis6326IsTexturingEnabled(ctx)) {
		sis6326glx.sendSpec = 1;
		sis6326glx.sendTex = 1;

		sis6326UpdateTextureState( ctx );
		enable |= sis6326CalcTEX_CNTL(ctx);
	}


	OUTREG(SIS6326_TRI_ENABLEREG, (enable));
	
	OUTREG(SIS6326_TRI_DSETTING, (S_DSET_ROP_COPY_PEN |
				      S_DSET_16BPP |
				      S_DSET_16_565 |
				      (sis6326DB->pitch * sis6326glx.bytesPerPixel)));
	OUTREG(SIS6326_TRI_DBASE, sis6326DB->backBufferBlock->ofs);
	OUTREG((SIS6326_TRI_CLIP_T_B), ((0 << 13) | (sis6326DB->height-1)));
	OUTREG((SIS6326_TRI_CLIP_L_R), ((0 << 13) | (sis6326DB->width-1)));

	
	sis6326glx.c_setupPointers++;
}

/*
 * Local Variables:
 * mode: c
 * tab-width: 8
 * c-basic-offset: 8
 * End:
 */
