/*
 * GLX Hardware Device Driver for S3 ViRGE DX/GX/GX2
 * Copyright (C) 1999 Jim Duchek
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Based on Mach64 driver: mach64dmainit.c
 *
 *    Jim Duchek <jimduchek@ou.edu>
 */


#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <sys/mman.h>
#include <stdio.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <signal.h>

#include "context.h"
#include "depth.h"
#include "macros.h"
#include "texstate.h"
#include "triangle.h"
#include "vb.h"
#include "types.h"

#include "glx_log.h"
#include "glx_config.h"
#include "mesaglx/context.h"
#include "mesaglx/matrix.h"
#include "mesaglx/types.h"

#define GC XXGC
#include "gcstruct.h"
#include "pixmapstr.h"
#include "servermd.h" /* PixmapBytePad */
#include "scrnintstr.h"
#include "regionstr.h"
#include "windowstr.h"
#undef GC

#include "glx_symbols.h"
#include "s3virgeglx.h"


s3virgeUI32             *wholeDmaHeap;

/* According to the S3 specs, the GX2 supports AGP.  I've never seen
 * one, it's really just not worth the time to hack GART in :) */

/*

GLX_S3VIRGE_DMA=0	: MMIO
GLX_S3VIRGE_DMA=1	: dma, 4k buffer (doesn't need a mem=)
GLX_S3VIRGE_DMA=2	: full 64k buffer, textures too, needs mem=
GLX_S3VIRGE_DMA=3	: same as 2 but async

GLX_S3VIRGE_DMAADR=124	: put the physical buffer at offset 92 * 0x100000
			: use GLX_S3VIRGE_DMAADR=AGP to use agp memory
GLX_S3VIRGE_DMASIZE=4	: use 4 * 0x100000 bytes of memory for the buffers
*/

memHeap_t	*cardHeap;
memHeap_t 	*sysmemHeap;

unsigned char *sysmemVirtual;

s3virgeUI32	cardPhysical;


void s3virgeDmaResetBuffer( void );
void s3virgeSetSyncBusy( void );



static const char *getenvSafe( const char *name )
{
    const char	*r;

    r = getenv( name );
    if ( !r ) {
	return "";
    }
    return r;
}

/*=============================================*/


/*
 * AllocateScatteredMemory
 * Create a file of the needed size that will be memory mapped
 * into both the client and server address spaces.
 */
static void AllocateScatteredMemory( void ) {

	if ( __glx_is_server ) {
		char	*name;
		char	junk;

		//name = tempnam( NULL, "glx_" );
		name = "/tmp/glxmemory";
		hwMsg( 1, "Using memory file: %s\n", name );
		strcpy( s3virgeglx.memoryFileName, name );
		//free( name );

		/* create a file of the needed size */
		s3virgeglx.dmaMemoryFd = open( s3virgeglx.memoryFileName,
					      O_RDWR | O_CREAT , 0600 );
		if ( s3virgeglx.dmaMemoryFd == -1 ) {
			FatalError( "Couldn't open %s", s3virgeglx.memoryFileName );
		}

		/* fill the file */
		junk = 0;
		lseek( s3virgeglx.dmaMemoryFd, 4096 - 1, SEEK_SET );
		write( s3virgeglx.dmaMemoryFd, &junk, 1 );
	} else {
		/* open the file, which was created by the server */
		hwMsg( 1, "Client - Using memory file: %s\n", s3virgeglx.memoryFileName );
		s3virgeglx.dmaMemoryFd = open( s3virgeglx.memoryFileName, O_RDWR, 0600 );
		if ( s3virgeglx.dmaMemoryFd == -1 ) {
			FatalError( "Couldn't open %s", s3virgeglx.memoryFileName );
		}
	}

	/* memory map the file, which we will use for our dma memory */
	s3virgeglx.dmaMemory = mmap( NULL, 4096,
				    PROT_READ | PROT_WRITE,  MAP_SHARED, s3virgeglx.dmaMemoryFd, 0 );

	if ( s3virgeglx.dmaMemory == MAP_FAILED ) {
		FatalError( "mmap of glx memory buffer failed" );
	}

	/* lock it down */
	if ( mlock( s3virgeglx.dmaMemory, 4096 ) == -1 ) {
		/* we could fall back to pseudo dma if we wanted to */
		FatalError( "Couldn't mlock scattered memory" );
	}

	hwMsg( 1, "4k of scattered memory at virtual 0x%x\n",
	       s3virgeglx.dmaMemory );
}

/*
 * LocateScatteredMemory
 * Do evil things to find the physical memory address of a
 * block of virtual memory.
 * This probably won't work properly on alpha.
 * It may not work properly on ppc.
 */
#define	PAGE_SIZE			4096		// to ragepro, ok to be < system
static void LocateScatteredMemory( void ) {
	int	numPages;
	int	i;
	int	*vm;
	int	magic1, magic2;
	int	remaining;
	int	memFd;
	int	test[2];
	int	page;

	vm = (int *)s3virgeglx.dmaMemory;

	// create two different numbers that will be different each
	// time the process is run
	magic1 = usec();
	magic2 = magic1 ^ 0x12345678;

	vm[0] = magic1;
	vm[1] = 42;

	// scan /dev/mem looking for our markers
	memFd = open( "/dev/mem", O_RDONLY );
	if ( memFd == -1 ) {
		FatalError( "Couldn't open /dev/mem" );
	}
	
	s3virgeglx.memoryRemapping = 0;

	for ( i = 1 ; i < 0x100000 ; i++ ) {
		lseek( memFd, i * PAGE_SIZE, SEEK_SET );
		if ( read( memFd, &test, sizeof( test ) ) != sizeof( test ) ) {
#ifdef __sun__
			continue;
#else
			hwMsg( 1, "read of /dev/mem failed at 0x%x\n", i * PAGE_SIZE );
			break;
#endif
		}
		if ( (test[0] != magic1) || (test[1] != 42)) {
			continue;
		}
		
		// now change the value and see if it changes in the mmap
		// to make extra sure we are talking about the same page
		vm[0] = magic2;
		lseek( memFd, i * PAGE_SIZE, SEEK_SET );
		if ( read( memFd, &test, sizeof( test ) ) != sizeof( test ) ) {
			hwMsg( 1, "reread of /dev/mem failed at 0x%x\n", i * PAGE_SIZE );
#ifdef __sun__
	/* some platforms (like solaris) don't allow one to read random
	   pages of memory; skipping the break here appears to let dma=1 work */
			continue;
#else
			break;
#endif /* __sun__ */
		}
		if ( test[0] != magic2 ) {
			hwMsg( 1, "magic2 failed\n" );
			continue;
		}
		
		hwMsg( 19, "virtual page 0x%x found at physical page 0x%x\n", page, i );
		s3virgeglx.memoryRemapping = i * PAGE_SIZE;
		break;
	}
	
	close( memFd );

	if ( !s3virgeglx.memoryRemapping ) {
		FatalError( "Didn't find the page\n");
	}
}




/*
 * AllocateCommandBuffers
 * The dma command buffers can be either virtual or in the sysmemHeap
 */

static void AllocateCommandBuffer( void )
{
     s3virgeUI32 *tmp;
     int i, tmp2, tmp3;
     int fd;
     int raddr;
     int rdiff, diff;
     

    dma_buffer = malloc( sizeof(s3virgeDma_buffer) );
    dma_buffer->dmaBuf = s3virgeglx.dmaMemory;
    s3virgeMsg(1, "Pointer for DMA is %08x\n", dma_buffer->dmaBuf);
    dma_buffer->numDwords = 0;
    dma_buffer->lastRead = 0;
    dma_buffer->physAddr = s3virgeglx.memoryRemapping;

    if (s3virgeglx.dmaDriver == 1) {
    	dma_buffer->size = 1024;
    } else if (s3virgeglx.dmaDriver > 1) {
    	dma_buffer->size = 16384;
    }

    s3virgeDmaStart();
    
    s3virgeDmaResetBuffer();
}

/*
 * setupSystemMem
 *
 */
void setupSystemMem(void)
{
	sysmemHeap = mmInit(0, s3virgeglx.dmaSize * 1024 * 1024);
	s3virgeglx.dmaMemoryFd = open( "/dev/mem", O_RDWR);
	sysmemVirtual = (unsigned char *)mmap(NULL, s3virgeglx.dmaSize * 1024 * 1024,
		PROT_READ | PROT_WRITE, MAP_SHARED, s3virgeglx.dmaMemoryFd, s3virgeglx.dmaAdr * 1024 * 1024);
	s3virgeglx.memoryRemapping = s3virgeglx.dmaAdr * 1024 * 1024;
	s3virgeglx.dmaMemory = sysmemVirtual;
        mmReserveMem( sysmemHeap, 0, 64 * 1024 );
}

/*
 * s3virgeDmaInit
 *
 */
void s3virgeDmaInit(void)
{
	if ( __glx_is_server )
	{
		if ( !glx_getvar_secure( "s3virge_dma" ) ) {
			s3virgeglx.dmaDriver = 0; /* default is no dma */
		} else {
			s3virgeglx.dmaDriver = glx_getint_secure( "s3virge_dma" );
		}
	}

	sysmemHeap = NULL;
	
	if (s3virgeglx.dmaDriver == 1) {
	 	/* allocate a block of scattered system memory */
		AllocateScatteredMemory();
		LocateScatteredMemory();
	} else if (s3virgeglx.dmaDriver > 1) {
		s3virgeglx.dmaAdr = glx_getint_secure( "s3virge_dmaadr" );
		s3virgeglx.dmaSize = glx_getint_secure( "s3virge_dmasize" );
		setupSystemMem();
	}

	/* setup the command buffer in the apropriate memory space */
	AllocateCommandBuffer();
	
	if ( __glx_is_server)
		s3virgeOn();

	/* verify that real dma actually works, otherwise fall back to pseudo dma */
	if ( __glx_is_server && s3virgeglx.dmaDriver > 0 ) {
//		ProbeDmaSystem();
	}

}

/* FIXME -- detect differences between ViRGE DX/GX/GX2*/
static GLboolean det_hwGfx()
{
   if( ( s3virgeglx.depth != 15 ) && ( s3virgeglx.depth != 16 ) )
    {
	s3virgeError( "Unsupported depth: %d, only 16 bpp supported right now\n",
		     s3virgeglx.depth );
	return GL_FALSE;
    }

    return GL_TRUE;
}

/*
 * s3virgeInitLogging
 *
 */
void s3virgeInitLogging( void )
{
    char	*logName;

    /* open the logfile and set loglevel */
    logName = glx_getvar( "hw_logfile" );
    if ( __glx_is_server)
    {
	s3virgeOpenLog( logName );
    }
    else
    {
	/* direct rendering clients use a different file
	   so they don't stomp on the server's log */
	if ( logName )
	{
	    char	newName[1024];

	    strcpy( newName, logName );
	    strcat( newName, "_direct" );
	    s3virgeOpenLog( newName );
	}
    }

    if ( glx_getvar( "hw_loglevel" ) ) {
	s3virgeSetLogLevel( glx_getint( "hw_loglevel" ) );
    } else {
	s3virgeSetLogLevel( DBG_LEVEL_BASE );
    }
}


/*
 *
 * GetXServerInfo
 */
static int GetXServerInfo( void ) {
    int i;
    unsigned char tmp, tmp1;
    
    s3virgeglx.MMIOBase = GLXSYM(s3vMmioMem);
    
    s3virgeglx.depth = GLXSYM(vga256InfoRec).depth;

    s3virgeglx.virtualX = GLXSYM(vga256InfoRec).virtualX;
    s3virgeglx.virtualY = GLXSYM(vga256InfoRec).virtualY;
    s3virgeglx.displayWidth = GLXSYM(vga256InfoRec).displayWidth;
    s3virgeglx.videoRam = GLXSYM(vga256InfoRec).videoRam;
    s3virgeglx.bytesPerPixel = ( GLXSYM(vga256InfoRec).bitsPerPixel + 7 ) / 8;

    s3virgeglx.linearPhysical = (pointer)GLXSYM(vga256InfoRec).MemBase;
    s3virgeglx.linearBase = xf86MapVidMem( 0, 
					   LINEAR_REGION, 
					   (pointer)GLXSYM(vga256InfoRec).MemBase,
					   s3virgeglx.videoRam * 1024);
					   
    s3virgeglx.updatedPalette = 0;

    s3virgeMsg( 1, "width: %d\n", s3virgeglx.virtualX );
    s3virgeMsg( 1, "height: %d\n", s3virgeglx.virtualY );
    s3virgeMsg( 1, "pitch: %d\n", s3virgeglx.displayWidth );
    s3virgeMsg( 1, "depth: %d\n", s3virgeglx.depth );
    s3virgeMsg( 1, "bytesPerPixel: %d\n", s3virgeglx.bytesPerPixel );
    s3virgeMsg( 1, "videoRam: %dk\n", s3virgeglx.videoRam );
    s3virgeMsg( 1, "memBase: 0x%08x\n", s3virgeglx.linearBase );
    s3virgeMsg( 1, "ioBase: 0x%08x\n", s3virgeglx.MMIOBase );

    return 1;
}

/*
 * s3virgeInitGLX
 * This is the initial entry point for the s3virge hardware driver,
 * called at X server module load time, or libGL direct rendering
 * init time.
 */
GLboolean s3virgeInitGLX( void )
{
    /*
     * Begin usual GLX module initialization.
     */
     
    s3virgeInitLogging();

    if ( __glx_is_server ) {
	    
	    if ( !GetXServerInfo() ) {
		s3virgeMsg( 1, "not a ViRGE!\n" );
		return GL_FALSE;
	    }
    
	    /* check to make sure that we are on an apropriate chip and not
       	running in 8bpp mode */
	    if ( !det_hwGfx() ) {
		return GL_FALSE;
	    }
    }
    
    /* start up our card memory manager */
    cardHeap = mmInit( 0, s3virgeglx.videoRam * 1024 );
    
    if ( !cardHeap ) {
	s3virgeMsg( 1, "cardHeap creation failed, exiting!\n" );
	return GL_FALSE;	/* really shouldn't happen */
    }

    /* reserve memory for the front buffer */
    mmReserveMem( cardHeap, 0, s3virgeglx.displayWidth * s3virgeglx.virtualY * s3virgeglx.bytesPerPixel );

    /* Reserve the block that the HW cursor uses */
    mmReserveMem( cardHeap, (s3virgeglx.videoRam - 1) * 1024, 1024);

    /* the remaining memory is available for back buffers, depth
       buffers, and textures */
    mmDumpMemInfo( cardHeap );

    /* init the dma system */
    s3virgeDmaInit();

    /* FIXME: what other GLXProcs pointers should we change? */
    GLXProcs.CreateContext = s3virgeGLXCreateContext;
    GLXProcs.DestroyContext = s3virgeGLXDestroyContext;
    GLXProcs.SwapBuffers = s3virgeGLXSwapBuffers;
    GLXProcs.CreateImage = s3virgeGLXCreateImage;
    GLXProcs.DestroyImage = s3virgeGLXDestroyImage;
    GLXProcs.CreateDepthBuffer = s3virgeGLXCreateDepthBuffer;
    GLXProcs.MakeCurrent = s3virgeGLXMakeCurrent;
    GLXProcs.BindBuffer = s3virgeGLXBindBuffer;
    GLXProcs.VendorPrivate = s3virgeGLXVendorPrivate;
    GLXProcs.AllowDirect = s3virgeGLXAllowDirect;

    /* these vars can be changed between invocations of direct clients */
    if ( glx_getint("s3virge_lightmap_hack") ) {
	s3virgeMsg( 0, "enabling Lightmap hack\n" );
	s3virgeglx.lightmapHack = 1;
    } else {
    	s3virgeglx.lightmapHack = 0;
    }

    s3virgeError( "s3virgeInitGLX completed\n" );
    return GL_TRUE;
}


/*
 * Local Variables:
 * mode: c
 * c-basic-offset: 4
 * End:
 */
