/* $Id: mach64glx.h,v 1.39 2000/03/27 12:55:00 gareth Exp $ */

/*
 * GLX Hardware Device Driver for ATI Rage Pro
 * Copyright (C) 1999 Gareth Hughes
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Based on MGA driver: mgalib.h 1.7
 *
 *    Gareth Hughes <gareth@precisioninsight.com>
 */

#ifndef __MACH64_LIB_H__
#define __MACH64_LIB_H__

#include <stdio.h>

#include "types.h"		/* MESA types */
#ifndef MESA31
#error "The Mach64 driver now requires Mesa 3.1 or higher"
#endif

#include "xsmesaP.h"	/* GLX header */
#include "hwlog.h"
#include "hwtypes.h"
#include "hw_agp.h"
#include "mm.h"

#include "rage_reg.h"
#include "mach64regs.h"


/*
 * main buffer structure for drawing destination
 */
#define mach64BufferMagic		0x65e813a1
#define VALID_MACH64_BUFFER( b )	( (b) && ( (b)->magic == mach64BufferMagic ) )

struct mach64_buffer_t {
	hwUI32		magic;
	struct mach64_buffer_t *next;

	WindowPtr	pwindow;
	GLvisual	visual;

	void		*backBuffer;		/* pointer to buffer in VM */
	PMemBlock	backBufferBlock;	/* location in card's space, NULL if software only */

	void		*depthBuffer;
	PMemBlock	depthBufferBlock;

	int		width, height;		/* in pixels */
	int		pitch;			/* in pixels */

	XSMesaBuffer	xsBuf;			/* needed to get window for front buffer swaps */
};
typedef struct mach64_buffer_t mach64Buffer;
typedef struct mach64_buffer_t *mach64BufferPtr;

/*
 * main hardware context structure
 */
#define mach64ContextMagic		0x47323030

struct mach64_context_t {
	hwUI32		magic;

	struct mach64_buffer_t *DB;
	GLcontext	*gl_ctx;

	/* map GL texture units onto hardware. */
	int		multitex;
	int		tmu_source[2];

	int		frontBufferRendering;	/* generate auto-swapbuffers */
};

#define MACH64_CONTEXT(ctx)	((mach64ContextPtr)(((XSMesaContext)(ctx)->DriverCtx)->hw_ctx))

typedef struct mach64_context_t mach64Context;
typedef struct mach64_context_t *mach64ContextPtr;


/*
 * texture object
 */
#define MACH64_TEXTURE_OBJECT_MAGIC 0x9a0b3fd2

typedef struct mach64_texture_object_s {
	hwUI32		magic;
	struct mach64_texture_object_s *next;	/* driver global chain of textures */
	struct gl_texture_object *tObj;		/* mesa texture */
	mach64ContextPtr ctx;

	PMemBlock	memBlock;		/* buffer for all miplevels */
	agp_mem_t	agpBlocks[2];		/* double buffer for agp texturing */
	int		activeBlock;

	hwUI32		totalSize;		/* for adding to c_textureUtilization */
	int		texelBytes;

	int		widthLog2;
	int		heightLog2;
	int		maxLog2;

	int		hasAlpha;		/* for SCALE_3D_CNTL */
	int		textureFormat;		/* see page 5-47 in docs */

	hwUI32 		age;   			/* set to mgaglx.swapBufferCount when bound */
} mach64TextureObject_t;

typedef mach64TextureObject_t *mach64TextureObjectPtr;



/*
 * main driver variables structure
 */
typedef struct {
	mach64Buffer		*bufferList;
	mach64TextureObject_t	*textureList;
	mach64TextureObject_t	*currentTexture[2];	/* so we never free the current */

	hwUI32		dmaActive;

	struct _dma_buffer	*dma_buffer;	/* dmaBuffers[ activeDmaBuffer ] */

	hwUI32		activeDmaBuffer;

	/* vertex reuse stuff */
	GLuint		setupVertices[3];	/* vertices currently in setup engine */
	int		setupMax;		/* max non-clipped vertex index */

	/* logging stuff */
	hwUI32		logLevel;
	FILE		*logFile;

	/* dma stuff */
	hwUI32		dmaDriver;
	hwUI32		dmaSize;		/* in megs */
	char		*dmaMemory;		/* virtual memory location */
	char		memoryFileName[256];	/* shared file for memory mapping */
	int		dmaMemoryFd;		/* file descriptor for mapped memory */
	int		*memoryRemapping;	/* physical address of each page in dmaMemory */

	int		maxDescriptors;		/* holds enough pages for half dmaSize */
	char		*descriptorMemory;
	int		descriptorPhysical;	/* with proper alignment */

	/* options */
	hwUI32		nullprims;		/* skip all primitive generation */
	hwUI32		boxes;			/* draw performance boxes */
	hwUI32		noFallback;		/* don't fall back to software, do best-effort rendering */
	hwUI32		skipDma;		/* don't send anything to the hardware */
	hwUI32		enforceFinish;		/* force glFinish to work by spec, hurting many
						   application's performance */

	/* performance counters */
	hwUI32		swapBuffersCount;	/* also used for texture thrash checking */

	hwUI32		c_textureSwaps;		/* number of swap outs due to low memory */
	hwUI32		c_textureDwords;	/* texture image dwords */
	hwUI32		c_setupPointers;
	hwUI32		c_clears;
	hwUI32		c_triangles;		/* also used to see if a front buffer swap is needed */
	hwUI32		c_lines;
	hwUI32		c_points;
	hwUI32		c_drawWaits;
	hwUI32		c_dmaFlush;
	hwUI32		c_overflows;

	hwUI32		hardwareWentIdle;	/* cleared each swapbuffers, set if
						   a waitfordmacompletion ever exited
						   without having to wait */

	hwUI32		timeTemp;		/* for time counting in logfiles */

	/* stuff we get from the apropriate fbdev/mach64info struct */
	int		depth;			/* should be 15, 16, or 24 */
	int		virtualX;
	int		virtualY;
	int		displayWidth;
	int		videoRam;		/* in K */
	char		*linearBase;		/* in virtual memory */
	char		*MMIOBase;		/* in virtual memory */
	hwUI32		linearPhysical;		/* physical location of framebuffer */
	int		bytesPerPixel;		/* should be 2 or 4 */

	hwUI32		registers[0x200];	/* read at startup */
} mach64Glx_t;


extern XSMesaContext	XSMesa;
extern mach64Glx_t	mach64glx;
extern mach64BufferPtr	mach64DB;
extern mach64ContextPtr	mach64Ctx;


#define MACH64PACKCOLOR555( r, g, b, a )				\
    ((((r) & 0xf8) << 7) | (((g) & 0xf8) << 2) | (((b) & 0xf8) >> 3) |	\
     ((a) ? 0x8000 : 0))

#define MACH64PACKCOLOR565( r, g, b )					\
    ((((r) & 0xf8) << 8) | (((g) & 0xfc) << 3) | (((b) & 0xf8) >> 3))

#define MACH64PACKCOLOR888( r, g, b )					\
    (((r) << 16) | ((g) << 8) | (b))

#define MACH64PACKCOLOR8888( r, g, b, a )				\
    (((a) << 24) | ((r) << 16) | ((g) << 8) | (b))

#define MACH64PACKCOLOR4444( r, g, b, a )				\
    ((((a) & 0xf0) << 8) | (((r) & 0xf0) << 4) | ((g) & 0xf0) | ((b) >> 4))

static __inline__ hwUI32 mach64PackColor( hwUI8 r, hwUI8 g,
					  hwUI8 b, hwUI8 a )
{
	switch ( mach64glx.depth )
	{
	case 15:
		return MACH64PACKCOLOR555( r, g, b, a );
	case 16:
		return MACH64PACKCOLOR565( r, g, b );
	case 24:
		return MACH64PACKCOLOR888( r, g, b );
	case 32:
		return MACH64PACKCOLOR8888( r, g, b, a );
	default:
		return 0;
	}
}


/* ===== mach64glx.c ===== */

void mach64GLXCreateDepthBuffer( GLcontext* ctx );
void mach64GLXDestroyImage( GLXImage* image );
GLXImage* mach64GLXCreateImage( WindowPtr pwindow, GLvisual *visual,
				int width, int height, GLXImage *old_image );
GLboolean mach64GLXMakeCurrent( XSMesaContext c );
GLboolean mach64GLXBindBuffer( XSMesaContext c, XSMesaBuffer b );
XSMesaContext mach64GLXCreateContext( XSMesaVisual v,
				      XSMesaContext share_list );
void mach64GLXDestroyContext( XSMesaContext c );

/* ===== mach64clear.c ===== */

GLbitfield mach64Clear( GLcontext *ctx, GLbitfield mask, GLboolean all,
			GLint x, GLint y, GLint width, GLint height );

/* ===== mach64swap.c ===== */

void mach64GLXSwapBuffers( XSMesaBuffer b );
void mach64GLXSwapBuffersWithoutFlush( XSMesaBuffer b );

void mach64PerformanceBoxes( void );
void mach64BackToFront( DrawablePtr drawable, mach64BufferPtr buf );

/* ===== mach64dd.c ===== */

void mach64_setup_DD_pointers( GLcontext *ctx );
void mach64DDExtensionsInit( GLcontext *ctx );

/* ===== mach64state.c ===== */

void mach64DDUpdateState( GLcontext *ctx );

/* ===== mach64tri.c ===== */

void mach64Triangle( GLcontext *ctx, GLuint v0, GLuint v1, GLuint v2, GLuint pv );
void mach64DDChooseRenderState( GLcontext *ctx );

/* ===== mach64tex.c ===== */

void mach64CreateTexObj( mach64ContextPtr ctx, struct gl_texture_object *tObj );
GLboolean mach64IsTextureResident( GLcontext *ctx, struct gl_texture_object *tObj );
void mach64DeleteTexture( GLcontext *ctx, struct gl_texture_object *tObj );
void mach64TexSubImage( GLcontext *ctx, GLenum target,
			struct gl_texture_object *tObj, GLint level,
			GLint xoffset, GLint yoffset,
			GLsizei width, GLsizei height,
			GLint internalFormat,
			const struct gl_texture_image *image );
void mach64TexImage( GLcontext *ctx, GLenum target,
		     struct gl_texture_object *tObj, GLint level,
		     GLint internalFormat,
		     const struct gl_texture_image *image );

/* allow us to dump everything to free memory for a back buffer allocation */
void mach64FlushAllTextures( void );

/* ===== mach64dma.c / mach64dmainit.c ===== */

#include "mach64dma.h"

/* ===== mach64direct.c ===== */

int mach64GLXVendorPrivate( ClientPtr client,
			    XSMesaContext ctx,
			    xGLXVendorPrivateReq *stuff );

int mach64GLXAllowDirect( ClientPtr client );

void mach64DirectClientSwapBuffers( XSMesaBuffer b );

void mach64DirectClientDMAFlush( int wait );

/* ===== mach64pipeline.c ===== */

GLuint mach64DDRegisterPipelineStages( struct gl_pipeline_stage *out,
				       const struct gl_pipeline_stage *in,
				       GLuint nr );

/* ===== mach64pixel.c ===== */

GLboolean mach64DDDrawPixels( GLcontext *ctx,
			      GLint x, GLint y, GLsizei width, GLsizei height,
			      GLenum format, GLenum type,
			      const struct gl_pixelstore_attrib *unpack,
			      const GLvoid *pixels );
GLboolean mach64DDBitmap( GLcontext *ctx,
			  GLint x, GLint y, GLsizei width, GLsizei height,
			  const struct gl_pixelstore_attrib *unpack,
			  const GLubyte *bitmap );

#endif


/*
 * Local Variables:
 * mode: c
 * tab-width: 8
 * c-basic-offset: 8
 * End:
 */
