/* xsmesa2.c,v 1.2 1998/09/07 03:33:29 ripperda Exp */


/*
 * Mesa 3-D graphics library
 * Version:  2.3
 * Copyright (C) 1995-1996  Brian Paul
 *
 * GLX Server Extension
 * Copyright (C) 1996  Steven G. Parker  (sparker@cs.utah.edu)
 * Copyright (C) 1998, 1999 Terence Ripperda (ripperda@sgi.com)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * STEVEN PARKER, TERENCE RIPPERDA, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR 
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, 
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE 
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */


/*
 * xsmesa2.c,v
 * Revision 1.2  1998/09/07 03:33:29  ripperda
 * First Update
 *
 * Revision 1.11  1997/05/26 20:34:40  brianp
 * added PF_TRUEDITHER pixel format
 *
 * Revision 1.10  1997/04/20 20:29:11  brianp
 * replaced abort() with gl_problem()
 *
 * Revision 1.9  1997/04/12 16:24:07  brianp
 * removed several xmesa_setup_DD_pointers calls, removed die()
 *
 * Revision 1.8  1997/04/04 05:23:57  brianp
 * fixed problem with black/white being reversed on monochrome displays
 *
 * Revision 1.7  1997/03/21 01:58:32  brianp
 * added RendererString() function
 *
 * Revision 1.6  1996/10/22 02:46:29  brianp
 * now use DITHER_SETUP and XDITHER macros
 * use array indexing instead of pointer dereferencing in inner loops
 *
 * Revision 1.5  1996/10/11 00:23:58  brianp
 * fixed dithering bug in write_span_DITHER_pixmap() again!
 *
 * Revision 1.4  1996/10/09 23:09:56  brianp
 * fixed dithering bug in write_span_DITHER_pixmap()
 *
 * Revision 1.3  1996/09/27 01:31:42  brianp
 * removed unused variables
 *
 * Revision 1.2  1996/09/19 03:16:04  brianp
 * new X/Mesa interface with XMesaContext, XMesaVisual, and XMesaBuffer types
 *
 * Revision 1.1  1996/09/13 01:38:16  brianp
 * Initial revision
 *
 */

/*
 * Mesa/X11 interface, part 2.
 *
 * This file contains the implementations of all the device driver functions.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#undef MINSHORT
#undef MAXSHORT
#include "mesaglx/macros.h"
#include "mesaglx/types.h"
#include "mesaglx/lines.h"
#include "mesaglx/points.h"
#include "mesaglx/triangle.h"
#include "mesaglx/context.h"
#include "xsmesaP.h"
#include "xsmdither.h"


#include "glx_log.h"

#define GC XXGC
#include "gcstruct.h"
#include "pixmapstr.h"
#undef GC

#ifdef XFree86LOADER
#include "xf86_libc.h"
#endif

/*
 * XImage macros.
 */
#define GLX_READ_PIXEL(x,y)     GLXProcs.GetPixel(img,x,y)
#define GLX_WRITE_PIXEL(x,y,p)  GLXProcs.PutPixel(img,x,y,p)
#define GLX_READ_DEPTH(x,y)     GLXProcs.GetDepth(xsmesa,x,y)
#define GLX_WRITE_DEPTH(x,y,d)  GLXProcs.PutDepth(xsmesa,x,y,d)

/*
 * Abort with an error message.
 */
static void die( char *s )
{
    ErrorF("%s\n", s );
    abort();
}

/*
 * Read a pixel from an X drawable.
 */
static unsigned long read_pixel( XSMesaContext xsmesa,
                                 XSMesaBuffer b, int x, int y )
{
    unsigned long p = 0;
    DDXPointRec pt;
    char* data=b->scandata;
    int w;
    DrawablePtr d=b->buffer;

    w=1;
    pt.x=x+d->x;
    pt.y=FLIP(y)+d->y;
    if (pt.y<0 || pt.y >= b->xsm_visual->pScreen->height)
        return (0);
    if (pt.x < 0 || pt.x >= b->xsm_visual->pScreen->width)
        return (0);

    (*d->pScreen->GetSpans)(d, w, &pt, &w, 1, data);
    switch (d->bitsPerPixel)
    {
        case 8:
            p=*(GLubyte*)data;
            break;
        case 15:
        case 16:
            p=*(GLushort*)data;
            break;
        case 24:
        case 32:
            p=*(GLuint*)data;
            break;
        default:
            die("problem in read_pixel");
    }

    return (p);
}

/*
 * Return the size (width,height of the current color buffer.
 * This function should be called by the glViewport function because
 * glViewport is often called when the window gets resized.  We need to
 * update some X/Mesa stuff when that happens.
 * Output:  width - width of buffer in pixels.
 *          height - height of buffer in pixels.
 */
static void get_buffer_size( GLcontext *ctx, GLuint *width, GLuint *height )
{
    XSMesaContext xsmesa = (XSMesaContext) ctx->DriverCtx;
    unsigned int winwidth, winheight;

    GLXProcs.ValidateFrontBuffer( xsmesa->xsm_buffer->frontbuffer );

    winwidth = xsmesa->xsm_buffer->frontbuffer->width;
    winheight = xsmesa->xsm_buffer->frontbuffer->height;
    *width = winwidth;
    *height = winheight;

    if (   winwidth!=xsmesa->xsm_buffer->width
           || winheight!=xsmesa->xsm_buffer->height)
    {
        xsmesa->xsm_buffer->width = winwidth;
        xsmesa->xsm_buffer->height = winheight;
        xsmesa_alloc_back_buffer( xsmesa->xsm_buffer );
    }

    /* Needed by FLIP macro */
    xsmesa->xsm_buffer->bottom = (int) winheight - 1;

    if (xsmesa->xsm_buffer->backimage)
    {
        /* Needed by PIXELADDR1 macro */
        xsmesa->xsm_buffer->ximage_width1
        = xsmesa->xsm_buffer->backimage->bytes_per_line;
        xsmesa->xsm_buffer->ximage_origin1
        = (GLubyte *) xsmesa->xsm_buffer->backimage->data
          + xsmesa->xsm_buffer->ximage_width1 * (winheight-1);

        /* Needed by PIXELADDR2 macro */
        xsmesa->xsm_buffer->ximage_width2
        = xsmesa->xsm_buffer->backimage->bytes_per_line / 2;
        xsmesa->xsm_buffer->ximage_origin2
        = (GLushort *) xsmesa->xsm_buffer->backimage->data
          + xsmesa->xsm_buffer->ximage_width2 * (winheight-1);

        /* Needed by PIXELADDR4 macro */
        xsmesa->xsm_buffer->ximage_width4 = xsmesa->xsm_buffer->backimage->width;
        xsmesa->xsm_buffer->ximage_origin4
        = (GLuint *) xsmesa->xsm_buffer->backimage->data
          + xsmesa->xsm_buffer->ximage_width4 * (winheight-1);
    }
}

static void finish( GLcontext *ctx )
{
}

static void flush( GLcontext *ctx )
{
}

static GLboolean set_buffer( GLcontext *ctx, GLenum mode )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    if (mode==GL_FRONT || mode==GL_FRONT_LEFT || mode==GL_FRONT_RIGHT)
    {
        /* read/write front buffer */
        xsmesa->xsm_buffer->buffer = xsmesa->xsm_buffer->frontbuffer;
        ctx->NewState |= NEW_RASTER_OPS;
        gl_update_state(ctx);
        /*xmesa_setup_DD_pointers( ctx );*/
        return (GL_TRUE);
    }
    else if ( (mode==GL_BACK || mode==GL_BACK_LEFT || mode==GL_BACK_RIGHT) 
               && xsmesa->xsm_buffer->db_state)
    {
        /* read/write back buffer */
        if (xsmesa->xsm_buffer->backpixmap)
        {
            xsmesa->xsm_buffer->buffer = (DrawablePtr)xsmesa->xsm_buffer->backpixmap;
        }
        else if (xsmesa->xsm_buffer->backimage)
        {
            xsmesa->xsm_buffer->buffer = None;
        }
        else
        {
            /* just in case there wasn't enough memory for back buffer */
            xsmesa->xsm_buffer->buffer = xsmesa->xsm_buffer->frontbuffer;
        }
        ctx->NewState |= NEW_RASTER_OPS;
        gl_update_state(ctx);
        /*xmesa_setup_DD_pointers( ctx );*/
        return (GL_TRUE);
    }
    else
    {
        return (GL_FALSE);
    }
}

static void clear_index( GLcontext *ctx, GLuint index )
{
    XSMesaContext xsmesa = (XSMesaContext) ctx->DriverCtx;
    XID value;
    xsmesa->clearpixel = (unsigned long) index;
    value = (XID)xsmesa->clearpixel;
    DoChangeGC(xsmesa->xsm_buffer->cleargc, GCForeground, &value, 0);
}

static void clear_color( GLcontext *ctx,
                         GLubyte r, GLubyte g, GLubyte b, GLubyte a )
{
    XSMesaContext xsmesa = (XSMesaContext) ctx->DriverCtx;
    XID value;
    xsmesa->clearcolor[0] = r;
    xsmesa->clearcolor[1] = g;
    xsmesa->clearcolor[2] = b;
    xsmesa->clearcolor[3] = a;
    xsmesa->clearpixel = xsmesa_color_to_pixel( xsmesa, r, g, b, a );
    value = (XID)xsmesa->clearpixel;
    DoChangeGC(xsmesa->xsm_buffer->cleargc, GCForeground, &value, 0);
}

/* Set current color index */
static void set_index( GLcontext *ctx, GLuint index )
{
    XSMesaContext xsmesa = (XSMesaContext) ctx->DriverCtx;
    XID value;
    unsigned long p = (unsigned long) index;
    xsmesa->pixel = p;
    value = (XID)p;
    DoChangeGC(xsmesa->xsm_buffer->gc1, GCForeground, &value, 0);
}

/* Set current drawing color */
static void set_color( GLcontext *ctx,
                       GLubyte r, GLubyte g, GLubyte b, GLubyte a )
{
    XSMesaContext xsmesa = (XSMesaContext) ctx->DriverCtx;
    XID value;
    xsmesa->red   = r;
    xsmesa->green = g;
    xsmesa->blue  = b;
    xsmesa->alpha = a;
    xsmesa->pixel = xsmesa_color_to_pixel( xsmesa, r, g, b, a );;
    value = (XID)xsmesa->pixel;
    DoChangeGC(xsmesa->xsm_buffer->gc1, GCForeground, &value, 0);
}

/* Set index mask ala glIndexMask */
static GLboolean index_mask( GLcontext *ctx, GLuint mask )
{
    XID value;
    XSMesaContext xsmesa = (XSMesaContext) ctx->DriverCtx;
    if (xsmesa->xsm_buffer->buffer==XIMAGE)
    {
        return (GL_FALSE);
    }
    else
    {
        unsigned long m;
        if (mask==0xffffffff)
        {
            m = AllPlanes;
        }
        else
        {
            m = (unsigned long) mask;
        }
        value = (XID)m;
        DoChangeGC(xsmesa->xsm_buffer->gc1, GCPlaneMask, &value, 0);
        DoChangeGC(xsmesa->xsm_buffer->gc2, GCPlaneMask, &value, 0);
        DoChangeGC(xsmesa->xsm_buffer->cleargc, GCPlaneMask, &value, 0);
        return (GL_TRUE);
    }
}

/* Implements glColorMask() */
static GLboolean color_mask( GLcontext *ctx,
                             GLboolean rmask, GLboolean gmask,
                             GLboolean bmask, GLboolean amask )
{
    XID value;
    XSMesaContext xsmesa = (XSMesaContext) ctx->DriverCtx;
#if defined(__cplusplus) || defined(c_plusplus)
    int xclass = xsmesa->xsm_visual->pVisual->c_class;
#else
    int xclass = xsmesa->xsm_visual->pVisual->class;
#endif
    if (xsmesa->xsm_buffer->buffer!=XIMAGE
        && (xclass==TrueColor || xclass==DirectColor))
    {
        unsigned long m;
        if (rmask && gmask && bmask)
        {
            m = AllPlanes;
        }
        else
        {
            m = 0;
            if (rmask)   m |= xsmesa->xsm_visual->pVisual->redMask;
            if (gmask)   m |= xsmesa->xsm_visual->pVisual->greenMask;
            if (bmask)   m |= xsmesa->xsm_visual->pVisual->blueMask;
        }
        value = (XID)m;
        DoChangeGC(xsmesa->xsm_buffer->gc1, GCPlaneMask, &value, 0);
        DoChangeGC(xsmesa->xsm_buffer->gc2, GCPlaneMask, &value, 0);
        DoChangeGC(xsmesa->xsm_buffer->cleargc, GCPlaneMask, &value, 0);
        return (GL_TRUE);
    }
    else
    {
        return (GL_FALSE);
    }
}

/*
 * Set the pixel logic operation.  Return GL_TRUE if the device driver
 * can perform the operation, otherwise return GL_FALSE.  GL_COPY _must_
 * be operational, obviously.
 */
static GLboolean logicop( GLcontext *ctx, GLenum op )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    XID value;
    int func;
    if ((xsmesa->xsm_buffer->buffer==XIMAGE) && op!=GL_COPY)
    {
        /* X can't do logic ops in Ximages, except for GL_COPY */
        return (GL_FALSE);
    }
    switch (op)
    {
        case GL_CLEAR:        func = GXclear;     break;
        case GL_SET:      func = GXset;       break;
        case GL_COPY:     func = GXcopy;      break;
        case GL_COPY_INVERTED:    func = GXcopyInverted;  break;
        case GL_NOOP:     func = GXnoop;      break;
        case GL_INVERT:       func = GXinvert;    break;
        case GL_AND:      func = GXand;       break;
        case GL_NAND:     func = GXnand;      break;
        case GL_OR:       func = GXor;        break;
        case GL_NOR:      func = GXnor;       break;
        case GL_XOR:      func = GXxor;       break;
        case GL_EQUIV:        func = GXequiv;     break;
        case GL_AND_REVERSE:  func = GXandReverse;    break;
        case GL_AND_INVERTED: func = GXandInverted;   break;
        case GL_OR_REVERSE:   func = GXorReverse; break;
        case GL_OR_INVERTED:  func = GXorInverted;    break;
        default:  return (GL_FALSE);
    }
    value=(XID)func;
    DoChangeGC(xsmesa->xsm_buffer->gc1, GCFunction, &value, 0);
    DoChangeGC(xsmesa->xsm_buffer->gc2, GCFunction, &value, 0);
    return (GL_TRUE);
}

/*
 * Enable/disable dithering
 */
static void dither( GLcontext *ctx, GLboolean enable )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    if (enable)
    {
        xsmesa->pixelformat = xsmesa->xsm_visual->dithered_pf;
    }
    else
    {
        xsmesa->pixelformat = xsmesa->xsm_visual->undithered_pf;
    }
}

/**********************************************************************/
/*** glClear implementations                                        ***/
/**********************************************************************/

static GLbitfield clear_pixmap( GLcontext *ctx, GLbitfield mask, GLboolean all,
                                GLint x, GLint y, GLint width, GLint height )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    xRectangle clearRect;
    GCPtr gc;

    if (all)
    {
        clearRect.x=0; clearRect.y=0;
        clearRect.width=xsmesa->xsm_buffer->width;
        clearRect.height=xsmesa->xsm_buffer->height;
    }
    else
    {
        clearRect.x=x;
        clearRect.y=xsmesa->xsm_buffer->height-y-height;
        clearRect.width=width;
        clearRect.height=height;
    }

    gc=xsmesa->xsm_buffer->cleargc;
    ValidateGC(xsmesa->xsm_buffer->buffer, gc);
    (*gc->ops->PolyFillRect)(xsmesa->xsm_buffer->buffer, gc,
                             1, &clearRect);

    /** this should return the original mask, minus the color buffer bit **/
    return(mask ^ GL_COLOR_BUFFER_BIT);
}

static GLbitfield clear_8bit_ximage( GLcontext *ctx, GLbitfield mask, 
                                     GLboolean all, GLint x, GLint y, GLint width, GLint height ) 
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    if (all)
    {
        size_t n = xsmesa->xsm_buffer->backimage->bytes_per_line
                   * xsmesa->xsm_buffer->backimage->height;
        MEMSET( xsmesa->xsm_buffer->backimage->data, xsmesa->clearpixel, n );
    }
    else
    {
        GLint i;
        for (i=0;i<height;i++)
        {
            GLubyte *ptr = PIXELADDR1( x, y+i );
            MEMSET( ptr, xsmesa->clearpixel, width );
        }
    }

    /** this should return the original mask, minus the color buffer bit **/
    return(mask ^ GL_COLOR_BUFFER_BIT);
}

static GLbitfield clear_16bit_ximage( GLcontext *ctx, GLbitfield mask,
                                      GLboolean all, GLint x, GLint y, GLint width, GLint height ) 
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    if (all)
    {
        register GLuint n;
        register GLushort *ptr2 =(GLushort *) xsmesa->xsm_buffer->backimage->data;
        register GLushort pixel = (GLushort) xsmesa->clearpixel;
        if (xsmesa->swapbytes)
        {
            pixel = ((pixel >> 8) & 0x00ff)
                    | ((pixel << 8) & 0xff00);
        }
        if ((pixel & 0xff) == (pixel >> 8))
        {
            /* low and high bytes are equal so use memset() */
            n = xsmesa->xsm_buffer->backimage->bytes_per_line
                * xsmesa->xsm_buffer->height;
            MEMSET( ptr2, pixel & 0xff, n );
        }
        else
        {
            n = xsmesa->xsm_buffer->backimage->bytes_per_line / 2
                * xsmesa->xsm_buffer->height;
            do
            {
                *ptr2++ = pixel;
                n--;
            } while (n!=0);
        }
    }
    else
    {
        register int i, j;
        register GLushort pixel = (GLushort) xsmesa->clearpixel;
        for (j=0;j<height;j++)
        {
            register GLushort *ptr2 = PIXELADDR2( x, y+j );
            for (i=0;i<width;i++)
            {
                *ptr2++ = pixel;
            }
        }
    }

    /** this should return the original mask, minus the color buffer bit **/
    return(mask ^ GL_COLOR_BUFFER_BIT);
}

static GLbitfield clear_32bit_ximage( GLcontext *ctx, GLbitfield mask, 
                                      GLboolean all, GLint x, GLint y, GLint width, GLint height ) 
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    if (all)
    {
        register GLint n = xsmesa->xsm_buffer->width * xsmesa->xsm_buffer->height;
        register GLuint *ptr4 = (GLuint *) xsmesa->xsm_buffer->backimage->data;
        register GLuint pixel = (GLuint) xsmesa->clearpixel;
        if (xsmesa->swapbytes)
        {
            pixel = ((pixel >> 24) & 0x000000ff)
                    | ((pixel >> 8)  & 0x0000ff00)
                    | ((pixel << 8)  & 0x00ff0000)
                    | ((pixel << 24) & 0xff000000);
        }
        if (pixel==0)
        {
            MEMSET( ptr4, pixel, 4*n );
        }
        else
        {
            do
            {
                *ptr4++ = pixel;
                n--;
            } while (n!=0);
        }
    }
    else
    {
        register int i, j;
        register GLuint pixel = (GLuint) xsmesa->clearpixel;
        for (j=0;j<height;j++)
        {
            register GLuint *ptr4 = PIXELADDR4( x, y+j );
            for (i=0;i<width;i++)
            {
                *ptr4++ = pixel;
            }
        }
    }

    /** this should return the original mask, minus the color buffer bit **/
    return(mask ^ GL_COLOR_BUFFER_BIT);
}

static GLbitfield clear_nbit_ximage( GLcontext *ctx, GLbitfield mask, 
                                     GLboolean all, GLint x, GLint y, GLint width, GLint height ) 
{
    XSMesaContext xsmesa = (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    if (all)
    {
        register int i, j;
        width = xsmesa->xsm_buffer->width;
        height = xsmesa->xsm_buffer->height;
        for (j=0;j<height;j++)
        {
            for (i=0;i<width;i++)
            {
                GLX_WRITE_PIXEL(i, j, xsmesa->clearpixel );
            }
        }
    }
    else
    {
        /* TODO: optimize this */
        register int i, j;
        y = FLIP(y);
        for (j=0;j<height;j++)
        {
            for (i=0;i<width;i++)
            {
                GLX_WRITE_PIXEL(x+i, y-j, xsmesa->clearpixel );
            }
        }
    }

    /** this should return the original mask, minus the color buffer bit **/
    return(mask ^ GL_COLOR_BUFFER_BIT);
}

/*
 * The Mesa library needs to be able to draw pixels in a number of ways:
 *   1. RGB vs Color Index
 *   2. as horizontal spans (polygons, images) vs random locations (points,
 *      lines)
 *   3. different color per-pixel or same color for all pixels
 *
 * Furthermore, the X driver needs to support rendering to 3 possible
 * "buffers", usually one, but sometimes two at a time:
 *   1. The front buffer as an X window
 *   2. The back buffer as a Pixmap
 *   3. The back buffer as an XImage
 *
 * Finally, if the back buffer is an XImage, we can avoid using XPutPixel and
 * optimize common cases such as 24-bit and 8-bit modes.
 *
 * By multiplication, there's at least 48 possible combinations of the above.
 *
 * Below are implementations of the most commonly used combinations.  They are
 * accessed through function pointers which get initialized here and are used
 * directly from the Mesa library.  The 8 function pointers directly correspond
 * to the first 3 cases listed above.
 *
 *
 * The function naming convention is:
 *
 *   write_[span|pixels]_[mono]_[format]_[pixmap|ximage]
 *
 * New functions optimized for specific cases can be added without too much
 * trouble.  An example might be the 24-bit TrueColor mode 8A8R8G8B which is
 * found on IBM RS/6000 X servers.
 */

/**********************************************************************/
/*** Write COLOR SPAN functions                                     ***/
/**********************************************************************/

#define RGB_SPAN_ARGS	const GLcontext *ctx,					\
			GLuint n, GLint x, GLint y,			\
			CONST GLubyte rgb[][3], const GLubyte mask[]

#define COLOR_SPAN_ARGS	const GLcontext *ctx,					\
			GLuint n, GLint x, GLint y,			\
			CONST GLubyte rgba[][4], const GLubyte mask[]

/* NOTE: if mask==NULL, draw all pixels */

/*
 * Write a span of PF_TRUECOLOR pixels to a pixmap.
 */
static void write_span_RGBA_TRUECOLOR_pixmap( COLOR_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    XID value;
    DDXPointRec pt;
    DrawablePtr buffer = xsmesa->xsm_buffer->buffer;
    GCPtr gc = xsmesa->xsm_buffer->gc2;
    register GLuint i;

    pt.y = FLIP(y);
    if (mask)
    {
        for (i=0;i<n;i++,x++)
        {
            if (mask[i])
            {
                pt.x=x;
                PACK_TRUECOLOR( value, rgba[i][0], rgba[i][1], rgba[i][2] );
                DoChangeGC(gc, GCForeground, &value, 0);
                ValidateGC(buffer, gc);
                (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
            }
        }
    }
    else
    {
        /* draw all pixels */
        for (i=0;i<n;i++)
        {
            pt.x=x;
            PACK_TRUECOLOR( value, rgba[i][0], rgba[i][1], rgba[i][2] );
            DoChangeGC(gc, GCForeground, &value, 0);
            ValidateGC(buffer, gc);
            (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
        }
    }
}

/*
 * Write a span of PF_TRUECOLOR pixels to a pixmap.
 */
static void write_span_TRUECOLOR_pixmap( RGB_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    XID value;
    DDXPointRec pt;
    DrawablePtr buffer = xsmesa->xsm_buffer->buffer;
    GCPtr gc = xsmesa->xsm_buffer->gc2;
    register GLuint i;

    pt.y = FLIP(y);
    if (mask)
    {
        for (i=0;i<n;i++,x++)
        {
            if (mask[i])
            {
                pt.x=x;
                PACK_TRUECOLOR( value, rgb[i][0], rgb[i][1], rgb[i][2] );
                DoChangeGC(gc, GCForeground, &value, 0);
                ValidateGC(buffer, gc);
                (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
            }
        }
    }
    else
    {
        /* draw all pixels */
        for (i=0;i<n;i++)
        {
            pt.x=x;
            PACK_TRUECOLOR( value, rgb[i][0], rgb[i][1], rgb[i][2] );
            DoChangeGC(gc, GCForeground, &value, 0);
            ValidateGC(buffer, gc);
            (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
        }
    }
}

/*
 * Write a span of PF_TRUEDITHER pixels to a pixmap.
 */
static void write_span_RGBA_TRUEDITHER_pixmap( COLOR_SPAN_ARGS )
{
    XSMesaContext xsmesa = (XSMesaContext) ctx->DriverCtx;
    XID value;
    DDXPointRec pt;
    DrawablePtr buffer = xsmesa->xsm_buffer->buffer;
    GCPtr gc = xsmesa->xsm_buffer->gc2;
    register GLuint i;

    y = FLIP( y );
    pt.y = y;
    if (mask)
    {
        for (i=0;i<n;i++,x++)
        {
            if (mask[i])
            {
                pt.x = x;
                PACK_TRUEDITHER(value, x, y, rgba[i][0], rgba[i][1], rgba[i][2]);
                DoChangeGC(gc, GCForeground, &value, 0);
                ValidateGC(buffer, gc);
                (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
            }
        }
    }
    else
    {
        for (i=0;i<n;i++)
        {
            pt.x = x + i;
            PACK_TRUEDITHER(value, pt.x, y, rgba[i][0], rgba[i][1], rgba[i][2]);
            DoChangeGC(gc, GCForeground, &value, 0);
            ValidateGC(buffer, gc);
            (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
        }
    }
}

/*
 * Write a span of PF_TRUEDITHER pixels to a pixmap.
 */
static void write_span_TRUEDITHER_pixmap( RGB_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    XID value;
    DDXPointRec pt;
    DrawablePtr buffer = xsmesa->xsm_buffer->buffer;
    GCPtr gc = xsmesa->xsm_buffer->gc2;
    register GLuint i;

    y = FLIP( y );
    pt.y = y;
    if (mask)
    {
        for (i=0;i<n;i++,x++)
        {
            if (mask[i])
            {
                pt.x = x;
                PACK_TRUEDITHER(value, x, y, rgb[i][0], rgb[i][1], rgb[i][2]);
                DoChangeGC(gc, GCForeground, &value, 0);
                ValidateGC(buffer, gc);
                (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
            }
        }
    }
    else
    {
        for (i=0;i<n;i++)
        {
            pt.x = x + i;
            PACK_TRUEDITHER(value, pt.x, y, rgb[i][0], rgb[i][1], rgb[i][2]);
            DoChangeGC(gc, GCForeground, &value, 0);
            ValidateGC(buffer, gc);
            (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
        }
    }
}

/*
 * Write a span of PF_8A8B8G8R pixels to a pixmap.
 */
static void write_span_8A8B8G8R_pixmap( COLOR_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    XID value;
    DDXPointRec pt;
    DrawablePtr buffer = xsmesa->xsm_buffer->buffer;
    GCPtr gc = xsmesa->xsm_buffer->gc2;
    register GLuint i;
    y = FLIP( y );
    pt.y=y;
    if (mask)
    {
        for (i=0;i<n;i++,x++)
        {
            if (mask[i])
            {
                pt.x=x;
                value = (XID)
                        PACK_8A8B8G8R( rgba[i][0], rgba[i][1], rgba[i][2], rgba[i][3] );
                DoChangeGC(gc, GCForeground, &value, 0);
                ValidateGC(buffer, gc);
                (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
            }
        }
    }
    else
    {
        /* draw all pixels */
        for (i=0; i<n; i++, x++)
        {
            /* draw pixel x,y using color red[i]/green[i]/blue[i]/alpha[i] */
            pt.x=x;
            value = (XID) 
                    PACK_8A8B8G8R( rgba[i][0], rgba[i][1], rgba[i][2], rgba[i][3] );
            DoChangeGC(gc, GCForeground, &value, 0);
            ValidateGC(buffer, gc);
            (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
        }
    }
}

/*
 * Write a span of PF_8R8G8B pixels to a pixmap.
 */
static void write_span_RGBA8_to_RGB8_pixmap( COLOR_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    DrawablePtr buffer = xsmesa->xsm_buffer->buffer;
    GCPtr gc = xsmesa->xsm_buffer->gc2;
    XID value;
    DDXPointRec pt;
    register GLuint i;
    y = FLIP( y );
    pt.y = y;
    if (mask)
    {
        for (i=0;i<n;i++,x++)
        {
            if (mask[i])
            {
                pt.x = x;
                value = (XID)PACK_8R8G8B( rgba[i][0], rgba[i][1], rgba[i][2] );
                DoChangeGC(gc, GCForeground, &value, 0);
                ValidateGC(buffer, gc);
                (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
            }
        }
    }
    else
    {
        /* draw all pixels */
        for (i=0; i<n; i++, x++)
        {
            /* draw pixel x,y using color red[i]/green[i]/blue[i]/alpha[i] */
            pt.x=x;
            value = (XID) PACK_8R8G8B( rgba[i][0], rgba[i][1], rgba[i][2]  );
            DoChangeGC(gc, GCForeground, &value, 0);
            ValidateGC(buffer, gc);
            (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
        }
    }
}

/*
 * Write a span of PF_8R8G8B pixels to a pixmap.
 */
static void write_span_8R8G8B_pixmap( RGB_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    DrawablePtr buffer = xsmesa->xsm_buffer->buffer;
    GCPtr gc = xsmesa->xsm_buffer->gc2;
    XID value;
    DDXPointRec pt;
    register GLuint i;
    y = FLIP( y );
    pt.y = y;
    if (mask)
    {
        for (i=0;i<n;i++,x++)
        {
            if (mask[i])
            {
                pt.x = x;
                value = (XID)PACK_8R8G8B( rgb[i][0], rgb[i][1], rgb[i][2] );
                DoChangeGC(gc, GCForeground, &value, 0);
                ValidateGC(buffer, gc);
                (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
            }
        }
    }
    else
    {
        /* draw all pixels */
        for (i=0; i<n; i++, x++)
        {
            /* draw pixel x,y using color red[i]/green[i]/blue[i]/alpha[i] */
            pt.x=x;
            value = (XID) PACK_8R8G8B( rgb[i][0], rgb[i][1], rgb[i][2]  );
            DoChangeGC(gc, GCForeground, &value, 0);
            ValidateGC(buffer, gc);
            (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
        }
    }
}

/*
 * Write a span of PF_5R6G5B pixels to a pixmap.
 */
static void write_span_RGBA_5R6G5B_pixmap( COLOR_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    DrawablePtr buffer = xsmesa->xsm_buffer->buffer;
    GCPtr gc = xsmesa->xsm_buffer->gc2;
    XID value;
    DDXPointRec pt;
    register GLuint i;
    y = FLIP( y );
    pt.y = y;
    if (mask)
    {
        for (i=0;i<n;i++,x++)
        {
            if (mask[i])
            {
                pt.x=x;
                value = (XID)PACK_5R6G5B( rgba[i][0], rgba[i][1], rgba[i][2] );
                DoChangeGC(gc, GCForeground, &value, 0);
                ValidateGC(buffer, gc);
                (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
            }
        }
    }
    else
    {
        /* draw all pixels */
        for (i=0; i<n; i++, x++)
        {
            /* draw pixel x,y using color red[i]/green[i]/blue[i]/alpha[i] */
            pt.x=x;
            value = (XID) PACK_5R6G5B( rgba[i][0], rgba[i][1], rgba[i][2]  );
            DoChangeGC(gc, GCForeground, &value, 0);
            ValidateGC(buffer, gc);
            (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
        }
    }
}

/*
 * Write a span of PF_5R6G5B pixels to a pixmap.
 */
static void write_span_5R6G5B_pixmap( RGB_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    DrawablePtr buffer = xsmesa->xsm_buffer->buffer;
    GCPtr gc = xsmesa->xsm_buffer->gc2;
    XID value;
    DDXPointRec pt;
    register GLuint i;
    y = FLIP( y );
    pt.y = y;
    if (mask)
    {
        for (i=0;i<n;i++,x++)
        {
            if (mask[i])
            {
                pt.x=x;
                value = (XID)PACK_5R6G5B( rgb[i][0], rgb[i][1], rgb[i][2] );
                DoChangeGC(gc, GCForeground, &value, 0);
                ValidateGC(buffer, gc);
                (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
            }
        }
    }
    else
    {
        /* draw all pixels */
        for (i=0; i<n; i++, x++)
        {
            /* draw pixel x,y using color red[i]/green[i]/blue[i]/alpha[i] */
            pt.x=x;
            value = (XID) PACK_5R6G5B( rgb[i][0], rgb[i][1], rgb[i][2]  );
            DoChangeGC(gc, GCForeground, &value, 0);
            ValidateGC(buffer, gc);
            (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
        }
    }
}

/*
 * Write a span of PF_DITHER pixels to a pixmap.
 */
static void write_span_DITHER_pixmap( RGB_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    DrawablePtr buffer = xsmesa->xsm_buffer->buffer;
    GCPtr gc = xsmesa->xsm_buffer->gc2;
    XID value;
    DDXPointRec pt;
    register GLuint i;
    XDITHER_SETUP(y);
    y = FLIP( y );
    pt.y = y;
    if (mask)
    {
        for (i=0;i<n;i++,x++)
        {
            if (mask[i])
            {
                pt.x = x;
                value = (XID) XDITHER( x, rgb[i][0], rgb[i][1], rgb[i][2] );
                DoChangeGC(gc, GCForeground, &value, 0);
                ValidateGC(buffer, gc);
                (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
            }
        }
    }
    else
    {
        /* draw all pixels */
        for (i=0; i<n; i++, x++)
        {
            /* draw pixel x,y using color red[i]/green[i]/blue[i]/alpha[i] */
            pt.x=x;
            value = (XID) XDITHER( x, rgb[i][0], rgb[i][1], rgb[i][2] );
            DoChangeGC(gc, GCForeground, &value, 0);
            ValidateGC(buffer, gc);
            (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
        }
    }
}

/*
 * Write a span of PF_1BIT pixels to a pixmap.
 */
static void write_span_1BIT_pixmap( RGB_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    DrawablePtr buffer = xsmesa->xsm_buffer->buffer;
    GCPtr gc = xsmesa->xsm_buffer->gc2;
    XID value;
    DDXPointRec pt;
    register GLuint i;
    SETUP_1BIT;
    y = FLIP( y );
    pt.y = y;
    if (mask)
    {
        for (i=0;i<n;i++,x++)
        {
            if (mask[i])
            {
                pt.x=x;
                value = (XID)DITHER_1BIT( x, y, rgb[i][0], rgb[i][1], rgb[i][2] );
                DoChangeGC(gc, GCForeground, &value, 0);
                ValidateGC(buffer, gc);
                (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
            }
        }
    }
    else
    {
        /* draw all pixels */
        for (i=0; i<n; i++, x++)
        {
            /* draw pixel x,y using color red[i]/green[i]/blue[i]/alpha[i] */
            pt.x=x;
            value = (XID) DITHER_1BIT( x, y, rgb[i][0], rgb[i][1], rgb[i][2] );
            DoChangeGC(gc, GCForeground, &value, 0);
            ValidateGC(buffer, gc);
            (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
        }
    }
}

/*
 * Write a span of PF_LOOKUP pixels to a pixmap.
 */
static void write_span_LOOKUP_pixmap( RGB_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    DrawablePtr buffer = xsmesa->xsm_buffer->buffer;
    GCPtr gc = xsmesa->xsm_buffer->gc2;
    XID value;
    DDXPointRec pt;
    register GLuint i;
    LOOKUP_SETUP;
    y = FLIP( y );
    pt.y = y;
    if (mask)
    {
        for (i=0;i<n;i++,x++)
        {
            if (mask[i])
            {
                pt.x=x;
                value = (XID)LOOKUP( rgb[i][0], rgb[i][1], rgb[i][2] );
                DoChangeGC(gc, GCForeground, &value, 0);
                ValidateGC(buffer, gc);
                (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
            }
        }
    }
    else
    {
        for (i=0;i<n;i++,x++)
        {
            /* draw pixel x,y using color red[i]/green[i]/blue[i]/alpha[i] */
            pt.x=x;
            value = (XID) LOOKUP( rgb[i][0], rgb[i][1], rgb[i][2] );
            DoChangeGC(gc, GCForeground, &value, 0);
            ValidateGC(buffer, gc);
            (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
        }
    }
}

/*
 * Write a span of PF_GRAYSCALE pixels to a pixmap.
 */
static void write_span_GRAYSCALE_pixmap( RGB_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    DrawablePtr buffer = xsmesa->xsm_buffer->buffer;
    GCPtr gc = xsmesa->xsm_buffer->gc2;
    XID value;
    DDXPointRec pt;
    register GLuint i;
    y = FLIP( y );
    pt.y = y;
    if (mask)
    {
        for (i=0;i<n;i++,x++)
        {
            if (mask[i])
            {
                pt.x=x;
                value = (XID)GRAY_RGB( rgb[i][0], rgb[i][1], rgb[i][2] );
                DoChangeGC(gc, GCForeground, &value, 0);
                ValidateGC(buffer, gc);
                (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
            }
        }
    }
    else
    {
        /* draw all pixels */
        for (i=0; i<n; i++, x++)
        {
            /* draw pixel x,y using color red[i]/green[i]/blue[i]/alpha[i] */
            pt.x=x;
            value = (XID) GRAY_RGB( rgb[i][0], rgb[i][1], rgb[i][2] );
            DoChangeGC(gc, GCForeground, &value, 0);
            ValidateGC(buffer, gc);
            (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
        }
    }
}

/*
 * Write a span of PF_RGBA_TRUECOLOR pixels to an XImage.
 */
static void write_span_RGBA_TRUECOLOR_ximage( COLOR_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    y = FLIP(y);
    if (mask)
    {
        for (i=0;i<n;i++,x++)
        {
            if (mask[i])
            {
                unsigned long p;
                PACK_TRUECOLOR( p, rgba[i][0], rgba[i][1], rgba[i][2] );
                GLX_WRITE_PIXEL( x, y, p );
            }
        }
    }
    else
    {
        /* draw all pixels */
        for (i=0;i<n;i++,x++)
        {
            unsigned long p;
            PACK_TRUECOLOR( p, rgba[i][0], rgba[i][1], rgba[i][2] );
            GLX_WRITE_PIXEL( x, y, p );
        }
    }
}

/*
 * Write a span of PF_TRUECOLOR pixels to an XImage.
 */
static void write_span_TRUECOLOR_ximage( RGB_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    y = FLIP(y);
    if (mask)
    {
        for (i=0;i<n;i++,x++)
        {
            if (mask[i])
            {
                unsigned long p;
                PACK_TRUECOLOR( p, rgb[i][0], rgb[i][1], rgb[i][2] );
                GLX_WRITE_PIXEL( x, y, p );
            }
        }
    }
    else
    {
        /* draw all pixels */
        for (i=0;i<n;i++,x++)
        {
            unsigned long p;
            PACK_TRUECOLOR( p, rgb[i][0], rgb[i][1], rgb[i][2] );
            GLX_WRITE_PIXEL( x, y, p );
        }
    }
}

/*
 * Write a span of PF_TRUEDITHER pixels to an XImage.
 */
static void write_span_RGBA_TRUEDITHER_ximage( COLOR_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    y = FLIP(y);
    if (mask)
    {
        for (i=0;i<n;i++,x++)
        {
            if (mask[i])
            {
                unsigned long p;
                PACK_TRUEDITHER(p, x, y, rgba[i][0], rgba[i][1], rgba[i][2]);
                GLX_WRITE_PIXEL( x, y, p );
            }
        }
    }
    else
    {
        /* draw all pixels */
        for (i=0;i<n;i++,x++)
        {
            unsigned long p;
            PACK_TRUEDITHER(p, x, y, rgba[i][0], rgba[i][1], rgba[i][2]);
            GLX_WRITE_PIXEL( x, y, p );
        }
    }
}

/*
 * Write a span of PF_TRUEDITHER pixels to an XImage.
 */
static void write_span_TRUEDITHER_ximage( RGB_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    y = FLIP(y);
    if (mask)
    {
        for (i=0;i<n;i++,x++)
        {
            if (mask[i])
            {
                unsigned long p;
                PACK_TRUEDITHER(p, x, y, rgb[i][0], rgb[i][1], rgb[i][2]);
                GLX_WRITE_PIXEL( x, y, p );
            }
        }
    }
    else
    {
        /* draw all pixels */
        for (i=0;i<n;i++,x++)
        {
            unsigned long p;
            PACK_TRUEDITHER(p, x, y, rgb[i][0], rgb[i][1], rgb[i][2]);
            GLX_WRITE_PIXEL( x, y, p );
        }
    }
}

/*
 * Write a span of PF_8R8G8B8A-format pixels to an ximage.
 */
static void write_span_RGBA8_to_RGB8_ximage( COLOR_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    y = FLIP(y);
    if (mask)
    {
        for (i=0;i<n;i++,x++)
        {
            if (mask[i])
                GLX_WRITE_PIXEL( x, y, PACK_8R8G8B( rgba[i][0], rgba[i][1], rgba[i][2] ));
        }
    }
    else
    {
        /* draw all pixels */
        for (i=0;i<n;i++,x++)
            GLX_WRITE_PIXEL( x, y, PACK_8R8G8B( rgba[i][0], rgba[i][1], rgba[i][2] ));
    }
}

/*
 * Write a span of PF_8A8B8G8R-format pixels to an ximage.
 */
static void write_span_8A8B8G8R_ximage( COLOR_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    y = FLIP(y);
    if (mask)
    {
        for (i=0;i<n;i++,x++)
        {
            if (mask[i])
                GLX_WRITE_PIXEL( x, y, PACK_8A8B8G8R( rgba[i][0], rgba[i][1], rgba[i][2], rgba[i][3] ));
        }
    }
    else
    {
        /* draw all pixels */
        for (i=0;i<n;i++,x++)
            GLX_WRITE_PIXEL( x, y, PACK_8A8B8G8R( rgba[i][0], rgba[i][1], rgba[i][2], rgba[i][3] ));
    }
}

/*
 * Write a span of PF_8R8G8B-format pixels to an ximage.
 */
static void write_span_8R8G8B_ximage( RGB_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    y = FLIP(y);
    if (mask)
    {
        for (i=0;i<n;i++,x++)
        {
            if (mask[i])
                GLX_WRITE_PIXEL( x, y, PACK_8R8G8B( rgb[i][0], rgb[i][1], rgb[i][2] ));
        }
    }
    else
    {
        /* draw all pixels */
        for (i=0;i<n;i++,x++)
            GLX_WRITE_PIXEL( x, y, PACK_8R8G8B( rgb[i][0], rgb[i][1], rgb[i][2] ));
    }
}

/*
 * Write a span of PF_5R6G5B-format pixels to an ximage.
 */
static void write_span_RGBA_5R6G5B_ximage( COLOR_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    y = FLIP(y);
    if (mask)
    {
        for (i=0;i<n;i++,x++)
        {
            if (mask[i])
                GLX_WRITE_PIXEL( x, y, PACK_5R6G5B( rgba[i][0], rgba[i][1], rgba[i][2] ));
        }
    }
    else
    {
        /* draw all pixels */
        for (i=0;i<n;i++,x++)
            GLX_WRITE_PIXEL( x, y, PACK_5R6G5B( rgba[i][0], rgba[i][1], rgba[i][2] ));
    }
}

/*
 * Write a span of PF_5R6G5B-format pixels to an ximage.
 */
static void write_span_5R6G5B_ximage( RGB_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    y = FLIP(y);
    if (mask)
    {
        for (i=0;i<n;i++,x++)
        {
            if (mask[i])
                GLX_WRITE_PIXEL( x, y, PACK_5R6G5B( rgb[i][0], rgb[i][1], rgb[i][2] ));
        }
    }
    else
    {
        /* draw all pixels */
        for (i=0;i<n;i++,x++)
            GLX_WRITE_PIXEL( x, y, PACK_5R6G5B( rgb[i][0], rgb[i][1], rgb[i][2] ));
    }
}

/*
 * Write a span of PF_DITHER pixels to an XImage.
 */
static void write_span_DITHER_ximage( RGB_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    int yy = FLIP(y);
    XDITHER_SETUP(yy);
    if (mask)
    {
        for (i=0;i<n;i++,x++)
        {
            if (mask[i])
                GLX_WRITE_PIXEL( x, yy, XDITHER( x, rgb[i][0], rgb[i][1], rgb[i][2] ) );
        }
    }
    else
    {
        /* draw all pixels */
        for (i=0;i<n;i++,x++)
            GLX_WRITE_PIXEL( x, yy, XDITHER( x, rgb[i][0], rgb[i][1], rgb[i][2] ) );
    }
}

/*
 * Write a span of 8-bit PF_DITHER pixels to an XImage.
 */
static void write_span_DITHER8_ximage( RGB_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    int yy = FLIP(y);
    XDITHER_SETUP(yy);
    if (mask)
    {
        for (i=0;i<n;i++,x++)
        {
            if (mask[i])
                GLX_WRITE_PIXEL( x, yy, XDITHER( x, rgb[i][0], rgb[i][1], rgb[i][2] ));
        }
    }
    else
    {
        for (i=0;i<n;i++,x++)
            GLX_WRITE_PIXEL( x, yy, XDITHER( x, rgb[i][0], rgb[i][1], rgb[i][2] ));
    }
}

/*
 * Write a span of PF_1BIT pixels to an XImage.
 */
static void write_span_1BIT_ximage( RGB_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    SETUP_1BIT;
    y = FLIP(y);
    if (mask)
    {
        for (i=0;i<n;i++,x++)
        {
            if (mask[i])
                GLX_WRITE_PIXEL( x, y, DITHER_1BIT(x, y, rgb[i][0], rgb[i][1], rgb[i][2]));
        }
    }
    else
    {
        for (i=0;i<n;i++,x++)
            GLX_WRITE_PIXEL( x, y, DITHER_1BIT(x, y, rgb[i][0], rgb[i][1], rgb[i][2]) );
    }
}

/*
 * Write a span of PF_LOOKUP pixels to an XImage.
 */
static void write_span_LOOKUP_ximage( RGB_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    LOOKUP_SETUP;
    y = FLIP(y);
    if (mask)
    {
        for (i=0;i<n;i++,x++)
        {
            if (mask[i])
                GLX_WRITE_PIXEL( x, y, LOOKUP( rgb[i][0], rgb[i][1], rgb[i][2] ) );
        }
    }
    else
    {
        /* draw all pixels */
        for (i=0;i<n;i++,x++)
            GLX_WRITE_PIXEL( x, y, LOOKUP( rgb[i][0], rgb[i][1], rgb[i][2] ) );
    }
}

/*
 * Write a span of 8-bit PF_LOOKUP pixels to an XImage.
 */
static void write_span_LOOKUP8_ximage( RGB_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    LOOKUP_SETUP;
    y = FLIP(y);
    if (mask)
    {
        for (i=0;i<n;i++,x++)
        {
            if (mask[i])
                GLX_WRITE_PIXEL( x, y, LOOKUP( rgb[i][0], rgb[i][1], rgb[i][2] ));
        }
    }
    else
    {
        /* draw all pixels */
        for (i=0;i<n;i++,x++)
            GLX_WRITE_PIXEL( x, y, LOOKUP( rgb[i][0], rgb[i][1], rgb[i][2] ));
    }
}

/*
 * Write a span of PF_GRAYSCALE pixels to an XImage.
 */
static void write_span_GRAYSCALE_ximage( RGB_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    y = FLIP(y);
    if (mask)
    {
        for (i=0;i<n;i++,x++)
        {
            if (mask[i])
                GLX_WRITE_PIXEL( x, y, GRAY_RGB( rgb[i][0], rgb[i][1], rgb[i][2] ) );
        }
    }
    else
    {
        /* draw all pixels */
        for (i=0;i<n;i++,x++)
            GLX_WRITE_PIXEL( x, y, GRAY_RGB( rgb[i][0], rgb[i][1], rgb[i][2] ) );
    }
}

/*
 * Write a span of 8-bit PF_GRAYSCALE pixels to an XImage.
 */
static void write_span_GRAYSCALE8_ximage( RGB_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    y = FLIP(y);
    if (mask)
    {
        for (i=0;i<n;i++,x++)
        {
            if (mask[i])
                GLX_WRITE_PIXEL( x, y, GRAY_RGB( rgb[i][0], rgb[i][1], rgb[i][2] ));
        }
    }
    else
    {
        /* draw all pixels */
        for (i=0;i<n;i++,x++)
            GLX_WRITE_PIXEL( x, y, GRAY_RGB( rgb[i][0], rgb[i][1], rgb[i][2] ));
    }
}

/**********************************************************************/
/*** Write COLOR PIXEL functions                                    ***/
/**********************************************************************/

#define COLOR_PIXEL_ARGS   const GLcontext *ctx,				\
			   GLuint n, const GLint x[], const GLint y[],	\
			   CONST GLubyte rgba[][4], const GLubyte mask[]


/*
 * Write an array of PF_TRUECOLOR pixels to a pixmap.
 */
static void write_pixels_TRUECOLOR_pixmap( COLOR_PIXEL_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    XID value;
    DDXPointRec pt;
    DrawablePtr buffer = xsmesa->xsm_buffer->buffer;
    GCPtr gc = xsmesa->xsm_buffer->gc2;
    register GLuint i;
    for (i=0;i<n;i++)
    {
        if (mask[i])
        {
            PACK_TRUECOLOR( value, rgba[i][0], rgba[i][1], rgba[i][2] );
            pt.x=x[i];
            pt.y=FLIP(y[i]);
            DoChangeGC(gc, GCForeground, &value, 0);
            ValidateGC(buffer, gc);
            (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
        }
    }
}

/*
 * Write an array of PF_TRUEDITHER pixels to a pixmap.
 */
static void write_pixels_TRUEDITHER_pixmap( COLOR_PIXEL_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    XID value;
    DDXPointRec pt;
    DrawablePtr buffer = xsmesa->xsm_buffer->buffer;
    GCPtr gc = xsmesa->xsm_buffer->gc2;
    register GLuint i;

    for (i=0;i<n;i++)
    {
        if (mask[i])
        {
            PACK_TRUEDITHER(value, x[i], y[i], rgba[i][0], rgba[i][1], rgba[i][2]);
            pt.x=x[i];
            pt.y=FLIP(y[i]);
            DoChangeGC(gc, GCForeground, &value, 0);
            ValidateGC(buffer, gc);
            (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
        }
    }
}

/*
 * Write an array of PF_8A8B8G8R pixels to a pixmap.
 */
static void write_pixels_8A8B8G8R_pixmap( COLOR_PIXEL_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    XID value;
    DDXPointRec pt;
    DrawablePtr buffer = xsmesa->xsm_buffer->buffer;
    GCPtr gc = xsmesa->xsm_buffer->gc2;
    register GLuint i;
    for (i=0;i<n;i++)
    {
        if (mask[i])
        {
            value = (XID)PACK_8A8B8G8R( rgba[i][0], rgba[i][1], rgba[i][2], rgba[i][3] );
            pt.x=x[i];
            pt.y=FLIP(y[i]);
            DoChangeGC(gc, GCForeground, &value, 0);
            ValidateGC(buffer, gc);
            (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
        }
    }
}

/*
 * Write an array of PF_8R8G8B pixels to a pixmap.
 */
static void write_pixels_8R8G8B_pixmap( COLOR_PIXEL_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    XID value;
    DDXPointRec pt;
    DrawablePtr buffer = xsmesa->xsm_buffer->buffer;
    GCPtr gc = xsmesa->xsm_buffer->gc2;
    register GLuint i;
    for (i=0;i<n;i++)
    {
        if (mask[i])
        {
            value = (XID)PACK_8R8G8B( rgba[i][0], rgba[i][1], rgba[i][2] );
            pt.x=x[i];
            pt.y=FLIP(y[i]);
            DoChangeGC(gc, GCForeground, &value, 0);
            ValidateGC(buffer, gc);
            (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
        }
    }
}

/*
 * Write an array of PF_5R6G5B pixels to a pixmap.
 */
static void write_pixels_5R6G5B_pixmap( COLOR_PIXEL_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    XID value;
    DDXPointRec pt;
    DrawablePtr buffer = xsmesa->xsm_buffer->buffer;
    GCPtr gc = xsmesa->xsm_buffer->gc2;
    register GLuint i;
    for (i=0;i<n;i++)
    {
        if (mask[i])
        {
            value = (XID)PACK_5R6G5B( rgba[i][0], rgba[i][1], rgba[i][2] );
            pt.x=x[i];
            pt.y=FLIP(y[i]);
            DoChangeGC(gc, GCForeground, &value, 0);
            ValidateGC(buffer, gc);
            (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
        }
    }
}

/*
 * Write an array of PF_DITHER pixels to a pixmap.
 */
static void write_pixels_DITHER_pixmap( COLOR_PIXEL_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    XID value;
    DDXPointRec pt;
    DrawablePtr buffer = xsmesa->xsm_buffer->buffer;
    GCPtr gc = xsmesa->xsm_buffer->gc2;
    register GLuint i;
    DITHER_SETUP;
    for (i=0;i<n;i++)
    {
        if (mask[i])
        {
            value = (XID)DITHER( x[i], y[i], rgba[i][0], rgba[i][1], rgba[i][2] );
            pt.x=x[i];
            pt.y=FLIP(y[i]);
            DoChangeGC(gc, GCForeground, &value, 0);
            ValidateGC(buffer, gc);
            (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
        }
    }
}

/*
 * Write an array of PF_1BIT pixels to a pixmap.
 */
static void write_pixels_1BIT_pixmap( COLOR_PIXEL_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    XID value;
    DDXPointRec pt;
    DrawablePtr buffer = xsmesa->xsm_buffer->buffer;
    GCPtr gc = xsmesa->xsm_buffer->gc2;
    register GLuint i;
    SETUP_1BIT;
    for (i=0;i<n;i++)
    {
        if (mask[i])
        {
            value = (XID)DITHER_1BIT( x[i], y[i], rgba[i][0], rgba[i][1], rgba[i][2] );
            pt.x=x[i];
            pt.y=FLIP(y[i]);
            DoChangeGC(gc, GCForeground, &value, 0);
            ValidateGC(buffer, gc);
            (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
        }
    }
}

/*
 * Write an array of PF_LOOKUP pixels to a pixmap.
 */
static void write_pixels_LOOKUP_pixmap( COLOR_PIXEL_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    XID value;
    DDXPointRec pt;
    DrawablePtr buffer = xsmesa->xsm_buffer->buffer;
    GCPtr gc = xsmesa->xsm_buffer->gc2;
    register GLuint i;
    LOOKUP_SETUP;
    for (i=0;i<n;i++)
    {
        if (mask[i])
        {
            value = (XID)LOOKUP( rgba[i][0], rgba[i][1], rgba[i][2] );
            pt.x=x[i];
            pt.y=FLIP(y[i]);
            DoChangeGC(gc, GCForeground, &value, 0);
            ValidateGC(buffer, gc);
            (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
        }
    }
}

/*
 * Write an array of PF_GRAYSCALE pixels to a pixmap.
 */
static void write_pixels_GRAYSCALE_pixmap( COLOR_PIXEL_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    XID value;
    DDXPointRec pt;
    DrawablePtr buffer = xsmesa->xsm_buffer->buffer;
    GCPtr gc = xsmesa->xsm_buffer->gc2;
    register GLuint i;
    for (i=0;i<n;i++)
    {
        if (mask[i])
        {
            value = (XID)GRAY_RGB( rgba[i][0], rgba[i][1], rgba[i][2] );
            pt.x=x[i];
            pt.y=FLIP(y[i]);
            DoChangeGC(gc, GCForeground, &value, 0);
            ValidateGC(buffer, gc);
            (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
        }
    }
}

/*
 * Write an array of PF_TRUECOLOR pixels to an ximage.
 */
static void write_pixels_TRUECOLOR_ximage( COLOR_PIXEL_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    for (i=0;i<n;i++)
    {
        if (mask[i])
        {
            unsigned long p;
            PACK_TRUECOLOR( p, rgba[i][0], rgba[i][1], rgba[i][2] );
            GLX_WRITE_PIXEL( x[i], FLIP(y[i]), p );
        }
    }
}

/*
 * Write an array of PF_TRUEDITHER pixels to an XImage.
 */
static void write_pixels_TRUEDITHER_ximage( COLOR_PIXEL_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    for (i=0;i<n;i++)
    {
        if (mask[i])
        {
            unsigned long p;
            PACK_TRUEDITHER(p, x[i], FLIP(y[i]), rgba[i][0], rgba[i][1], rgba[i][2]);
            GLX_WRITE_PIXEL( x[i], FLIP(y[i]), p );
        }
    }
}

/*
 * Write an array of PF_8A8B8G8R pixels to an ximage.
 */
static void write_pixels_8A8B8G8R_ximage( COLOR_PIXEL_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    for (i=0;i<n;i++)
    {
        if (mask[i])
            GLX_WRITE_PIXEL( x[i], FLIP(y[i]), PACK_8A8B8G8R( rgba[i][0], rgba[i][1], rgba[i][2], rgba[i][3] ));
    }
}

/*
 * Write an array of PF_8R8G8B pixels to an ximage.
 */
static void write_pixels_8R8G8B_ximage( COLOR_PIXEL_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    for (i=0;i<n;i++)
    {
        if (mask[i])
            GLX_WRITE_PIXEL( x[i], FLIP(y[i]), PACK_8R8G8B( rgba[i][0], rgba[i][1], rgba[i][2] ));
    }
}

/*
 * Write an array of PF_5R6G5B pixels to an ximage.
 */
static void write_pixels_5R6G5B_ximage( COLOR_PIXEL_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    for (i=0;i<n;i++)
    {
        if (mask[i])
            GLX_WRITE_PIXEL( x[i], FLIP(y[i]), PACK_5R6G5B( rgba[i][0], rgba[i][1], rgba[i][2] ));
    }
}

/*
 * Write an array of PF_DITHER pixels to an XImage.
 */
static void write_pixels_DITHER_ximage( COLOR_PIXEL_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    DITHER_SETUP;
    for (i=0;i<n;i++)
    {
        if (mask[i])
            GLX_WRITE_PIXEL( x[i], FLIP(y[i]), DITHER( x[i], y[i], rgba[i][0], rgba[i][1], rgba[i][2] ) );
    }
}

/*
 * Write an array of 8-bit PF_DITHER pixels to an XImage.
 */
static void write_pixels_DITHER8_ximage( COLOR_PIXEL_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    DITHER_SETUP;
    for (i=0;i<n;i++)
    {
        if (mask[i])
            GLX_WRITE_PIXEL( x[i], FLIP(y[i]), DITHER( x[i], y[i], rgba[i][0], rgba[i][1], rgba[i][2] ));
    }
}

/*
 * Write an array of PF_1BIT pixels to an XImage.
 */
static void write_pixels_1BIT_ximage( COLOR_PIXEL_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    SETUP_1BIT;
    for (i=0;i<n;i++)
    {
        if (mask[i])
            GLX_WRITE_PIXEL( x[i], FLIP(y[i]), DITHER_1BIT( x[i], y[i], rgba[i][0], rgba[i][1], rgba[i][2] ));
    }
}

/*
 * Write an array of PF_LOOKUP pixels to an XImage.
 */
static void write_pixels_LOOKUP_ximage( COLOR_PIXEL_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    LOOKUP_SETUP;
    for (i=0;i<n;i++)
    {
        if (mask[i])
            GLX_WRITE_PIXEL( x[i], FLIP(y[i]), LOOKUP(rgba[i][0], rgba[i][1], rgba[i][2]) );
    }
}

/*
 * Write an array of 8-bit PF_LOOKUP pixels to an XImage.
 */
static void write_pixels_LOOKUP8_ximage( COLOR_PIXEL_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    LOOKUP_SETUP;
    for (i=0;i<n;i++)
    {
        if (mask[i])
            GLX_WRITE_PIXEL( x[i], FLIP(y[i]), LOOKUP( rgba[i][0], rgba[i][1], rgba[i][2] ));
    }
}

/*
 * Write an array of PF_GRAYSCALE pixels to an XImage.
 */
static void write_pixels_GRAYSCALE_ximage( COLOR_PIXEL_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    for (i=0;i<n;i++)
    {
        if (mask[i])
            GLX_WRITE_PIXEL( x[i], FLIP(y[i]), GRAY_RGB( rgba[i][0], rgba[i][1], rgba[i][2] ) );
    }
}

/*
 * Write an array of 8-bit PF_GRAYSCALE pixels to an XImage.
 */
static void write_pixels_GRAYSCALE8_ximage( COLOR_PIXEL_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    for (i=0;i<n;i++)
    {
        if (mask[i])
            GLX_WRITE_PIXEL( x[i], FLIP(y[i]), GRAY_RGB( rgba[i][0], rgba[i][1], rgba[i][2] ));
    }
}

/**********************************************************************/
/*** Write MONO COLOR SPAN functions                                ***/
/**********************************************************************/

#define MONO_SPAN_ARGS	const GLcontext *ctx,	\
		 	GLuint n, GLint x, GLint y, const GLubyte mask[]


/*
 * Write a span of identical pixels to a pixmap.  The pixel value is
 * the one set by DD.color() or DD.index().
 */
static void write_span_mono_pixmap( MONO_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    DrawablePtr buffer = xsmesa->xsm_buffer->buffer;
    GCPtr gc = xsmesa->xsm_buffer->gc1;
    register GLuint i;
    register GLboolean write_all;
    y = FLIP( y );
    write_all = GL_TRUE;
    for (i=0;i<n;i++)
    {
        if (!mask[i])
        {
            write_all = GL_FALSE;
            break;
        }
    }
    ValidateGC(buffer, gc);
    if (write_all)
    {
        xRectangle rect;
        rect.x=x;
        rect.y=y;
        rect.width=n;
        rect.height=1;
        (*gc->ops->PolyFillRect)(buffer, gc, 1, &rect);
    }
    else
    {
        DDXPointRec pt;
        pt.y = y;
        for (i=0;i<n;i++,x++)
        {
            if (mask[i])
            {
                pt.x=x;
                (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
            }
        }
    }
}

/*
 * Write a span of PF_TRUEDITHER pixels to a pixmap.  The pixel value is
 * the one set by DD.color() or DD.index().
 */
static void write_span_mono_TRUEDITHER_pixmap( MONO_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    XID value;
    DDXPointRec pt;
    DrawablePtr buffer = xsmesa->xsm_buffer->buffer;
    GCPtr gc = xsmesa->xsm_buffer->gc2;
    register GLuint i;
    register GLubyte r, g, b;
    pt.y = FLIP( y );
    r = xsmesa->red;
    g = xsmesa->green;
    b = xsmesa->blue;

    for (i=0;i<n;i++,x++)
    {
        if (mask[i])
        {
            PACK_TRUEDITHER(value, x, pt.y, r, g, b);
            pt.x=x;
            DoChangeGC(gc, GCForeground, &value, 0);
            ValidateGC(buffer, gc);
            (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
        }
    }
}

/*
 * Write a span of PF_DITHER pixels to a pixmap.  The pixel value is
 * the one set by DD.color() or DD.index().
 */
static void write_span_mono_DITHER_pixmap( MONO_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    XID value;
    DDXPointRec pt;
    DrawablePtr buffer = xsmesa->xsm_buffer->buffer;
    GCPtr gc = xsmesa->xsm_buffer->gc2;
    register GLuint i;
    register GLubyte r, g, b;
    int yy = FLIP( y );
    XDITHER_SETUP(yy);
    pt.y = y;
    r = xsmesa->red;
    g = xsmesa->green;
    b = xsmesa->blue;
    for (i=0;i<n;i++,x++)
    {
        if (mask[i])
        {
            value = (XID)XDITHER( x, r, g, b );
            pt.x=x;
            DoChangeGC(gc, GCForeground, &value, 0);
            ValidateGC(buffer, gc);
            (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
        }
    }
}

/*
 * Write a span of PF_1BIT pixels to a pixmap.  The pixel value is
 * the one set by DD.color() or DD.index().
 */
static void write_span_mono_1BIT_pixmap( MONO_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    XID value;
    DDXPointRec pt;
    DrawablePtr buffer = xsmesa->xsm_buffer->buffer;
    GCPtr gc = xsmesa->xsm_buffer->gc2;
    register GLuint i;
    register GLubyte r, g, b;
    SETUP_1BIT;
    r = xsmesa->red;
    g = xsmesa->green;
    b = xsmesa->blue;
    pt.y = FLIP( y );
    for (i=0;i<n;i++,x++)
    {
        if (mask[i])
        {
            value = (XID)DITHER_1BIT( x, pt.y, r, g, b );
            pt.x=x;
            DoChangeGC(gc, GCForeground, &value, 0);
            ValidateGC(buffer, gc);
            (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
        }
    }
}

/*
 * Write a span of identical pixels to an XImage.  The pixel value is
 * the one set by DD.color() or DD.index().
 */
static void write_span_mono_ximage( MONO_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    register unsigned long p = xsmesa->pixel;
    y = FLIP( y );
    for (i=0;i<n;i++,x++)
    {
        if (mask[i])
            GLX_WRITE_PIXEL( x, y, p );
    }
}

/*
 * Write a span of identical PF_TRUEDITHER pixels to an XImage.  The pixel
 * value is the one set by DD.color().
 */
static void write_span_mono_TRUEDITHER_ximage( MONO_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    GLuint i;
    GLint r = xsmesa->red;
    GLint g = xsmesa->green;
    GLint b = xsmesa->blue;
    y = FLIP( y );
    for (i=0;i<n;i++)
    {
        if (mask[i])
        {
            unsigned long p;
            PACK_TRUEDITHER( p, x+i, y, r, g, b);
            GLX_WRITE_PIXEL( x+i, y, p );
        }
    }
}

/*
 * Write a span of identical 8A8B8G8R pixels to an XImage.  The pixel
 * value is the one set by DD.color().
 */
static void write_span_mono_8A8B8G8R_ximage( MONO_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    GLuint i, p;
    y = FLIP( y );
    p = (GLuint) xsmesa->pixel;
    for (i=0;i<n;i++)
    {
        if (mask[i])
            GLX_WRITE_PIXEL( x+i, y, p );
    }
}

/*
 * Write a span of identical 8R8G8B pixels to an XImage.  The pixel
 * value is the one set by DD.color().
 */
static void write_span_mono_8R8G8B_ximage( MONO_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    GLuint i, p;
    y = FLIP( y );
    p = (GLuint) xsmesa->pixel;
    for (i=0;i<n;i++)
    {
        if (mask[i])
            GLX_WRITE_PIXEL( x+i, y, p );
    }
}

/*
 * Write a span of identical DITHER pixels to an XImage.  The pixel
 * value is the one set by DD.color().
 */
static void write_span_mono_DITHER_ximage( MONO_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    register GLubyte r, g, b;
    int yy = FLIP(y);
    XDITHER_SETUP(yy);
    r = xsmesa->red;
    g = xsmesa->green;
    b = xsmesa->blue;
    for (i=0;i<n;i++,x++)
    {
        if (mask[i])
            GLX_WRITE_PIXEL( x, yy, XDITHER( x, r, g, b ) );
    }
}

/*
 * Write a span of identical 8-bit DITHER pixels to an XImage.  The pixel
 * value is the one set by DD.color().
 */
static void write_span_mono_DITHER8_ximage( MONO_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    register GLubyte r, g, b;
    int yy = FLIP(y);
    XDITHER_SETUP(yy);
    r = xsmesa->red;
    g = xsmesa->green;
    b = xsmesa->blue;
    for (i=0;i<n;i++,x++)
    {
        if (mask[i])
            GLX_WRITE_PIXEL( x, yy, XDITHER( x, r, g, b ));
    }
}

/*
 * Write a span of identical 8-bit LOOKUP pixels to an XImage.  The pixel
 * value is the one set by DD.color().
 */
static void write_span_mono_LOOKUP8_ximage( MONO_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    register GLubyte pixel = xsmesa->pixel;
    y = FLIP(y);
    for (i=0;i<n;i++)
    {
        if (mask[i])
            GLX_WRITE_PIXEL( x+i, y, pixel );
    }
}

/*
 * Write a span of identical PF_1BIT pixels to an XImage.  The pixel
 * value is the one set by DD.color().
 */
static void write_span_mono_1BIT_ximage( MONO_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    register GLubyte r, g, b;
    SETUP_1BIT;
    r = xsmesa->red;
    g = xsmesa->green;
    b = xsmesa->blue;
    y = FLIP(y);
    for (i=0;i<n;i++,x++)
    {
        if (mask[i])
            GLX_WRITE_PIXEL( x, y, DITHER_1BIT( x, y, r, g, b ) );
    }
}

/*
 * Write a span of identical 8-bit GRAYSCALE pixels to an XImage.  The pixel
 * value is the one set by DD.color().
 */
static void write_span_mono_GRAYSCALE8_ximage( MONO_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    GLuint i;
    unsigned long p = xsmesa->pixel;
    y = FLIP(y);
    for (i=0;i<n;i++)
    {
        if (mask[i])
            GLX_WRITE_PIXEL( x+i, y, p );
    }
}

/**********************************************************************/
/*** Write MONO COLOR PIXELS functions                              ***/
/**********************************************************************/

#define MONO_PIXEL_ARGS	const GLcontext *ctx,					\
			GLuint n, const GLint x[], const GLint y[],	\
			const GLubyte mask[]

/*
 * Write an array of identical pixels to a pixmap.  The pixel value is
 * the one set by DD.color() or DD.index.
 */
static void write_pixels_mono_pixmap( MONO_PIXEL_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    DrawablePtr buffer = xsmesa->xsm_buffer->buffer;
    GCPtr gc = xsmesa->xsm_buffer->gc1;
    register GLuint i;
    DDXPointRec pt;

    ValidateGC(buffer, gc);
    for (i=0;i<n;i++)
    {
        if (mask[i])
        {
            pt.x=x[i];
            pt.y=FLIP(y[i]);
            (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
        }
    }
}

/*
 * Write an array of PF_TRUEDITHER pixels to a pixmap.  The pixel value is
 * the one set by DD.color() or DD.index.
 */
static void write_pixels_mono_TRUEDITHER_pixmap( MONO_PIXEL_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    XID value;
    DDXPointRec pt;
    DrawablePtr buffer = xsmesa->xsm_buffer->buffer;
    GCPtr gc = xsmesa->xsm_buffer->gc2;
    register GLuint i;
    register GLubyte r, g, b;
    r = xsmesa->red;
    g = xsmesa->green;
    b = xsmesa->blue;

    for (i=0;i<n;i++)
    {
        if (mask[i])
        {
            PACK_TRUEDITHER(value, x[i], y[i], r, g, b);
            pt.x=x[i];
            pt.y=FLIP(y[i]);
            DoChangeGC(gc, GCForeground, &value, 0);
            ValidateGC(buffer, gc);
            (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
        }
    }
}

/*
 * Write an array of PF_DITHER pixels to a pixmap.  The pixel value is
 * the one set by DD.color() or DD.index.
 */
static void write_pixels_mono_DITHER_pixmap( MONO_PIXEL_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    XID value;
    DDXPointRec pt;
    DrawablePtr buffer = xsmesa->xsm_buffer->buffer;
    GCPtr gc = xsmesa->xsm_buffer->gc2;
    register GLuint i;
    register GLubyte r, g, b;
    DITHER_SETUP;
    r = xsmesa->red;
    g = xsmesa->green;
    b = xsmesa->blue;
    for (i=0;i<n;i++)
    {
        if (mask[i])
        {
            value = (XID)DITHER( x[i], y[i], r, g, b );
            pt.x=x[i];
            pt.y=FLIP(y[i]);
            DoChangeGC(gc, GCForeground, &value, 0);
            ValidateGC(buffer, gc);
            (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
        }
    }
}

/*
 * Write an array of PF_1BIT pixels to a pixmap.  The pixel value is
 * the one set by DD.color() or DD.index.
 */
static void write_pixels_mono_1BIT_pixmap( MONO_PIXEL_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    XID value;
    DDXPointRec pt;
    DrawablePtr buffer = xsmesa->xsm_buffer->buffer;
    GCPtr gc = xsmesa->xsm_buffer->gc2;
    register GLuint i;
    register GLubyte r, g, b;
    SETUP_1BIT;
    r = xsmesa->red;
    g = xsmesa->green;
    b = xsmesa->blue;
    for (i=0;i<n;i++)
    {
        if (mask[i])
        {
            value = (XID)DITHER_1BIT( x[i], y[i], r, g, b );
            pt.x=x[i];
            pt.y=FLIP(y[i]);
            DoChangeGC(gc, GCForeground, &value, 0);
            ValidateGC(buffer, gc);
            (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
        }
    }
}

/*
 * Write an array of identical pixels to an XImage.  The pixel value is
 * the one set by DD.color() or DD.index.
 */
static void write_pixels_mono_ximage( MONO_PIXEL_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    register unsigned long p = xsmesa->pixel;
    for (i=0;i<n;i++)
    {
        if (mask[i])
            GLX_WRITE_PIXEL( x[i], FLIP(y[i]), p );
    }
}

/*
 * Write an array of identical TRUEDITHER pixels to an XImage.
 * The pixel value is the one set by DD.color() or DD.index.
 */
static void write_pixels_mono_TRUEDITHER_ximage( MONO_PIXEL_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    int r = xsmesa->red;
    int g = xsmesa->green;
    int b = xsmesa->blue;
    for (i=0;i<n;i++)
    {
        if (mask[i])
        {
            unsigned long p;
            PACK_TRUEDITHER(p, x[i], FLIP(y[i]), r, g, b);
            GLX_WRITE_PIXEL( x[i], FLIP(y[i]), p );
        }
    }
}

/*
 * Write an array of identical 8A8B8G8R pixels to an XImage.  The pixel value
 * is the one set by DD.color().
 */
static void write_pixels_mono_8A8B8G8R_ximage( MONO_PIXEL_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    register GLuint p = (GLuint) xsmesa->pixel;
    for (i=0;i<n;i++)
    {
        if (mask[i])
            GLX_WRITE_PIXEL( x[i], FLIP(y[i]), p );
    }
}

/*
 * Write an array of identical 8R8G8B pixels to an XImage.  The pixel value
 * is the one set by DD.color().
 */
static void write_pixels_mono_8R8G8B_ximage( MONO_PIXEL_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    register GLuint p = (GLuint) xsmesa->pixel;
    for (i=0;i<n;i++)
    {
        if (mask[i])
            GLX_WRITE_PIXEL( x[i], FLIP(y[i]), p);
    }
}

/*
 * Write an array of identical PF_DITHER pixels to an XImage.  The pixel
 * value is the one set by DD.color().
 */
static void write_pixels_mono_DITHER_ximage( MONO_PIXEL_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    register GLubyte r, g, b;
    DITHER_SETUP;
    r = xsmesa->red;
    g = xsmesa->green;
    b = xsmesa->blue;
    for (i=0;i<n;i++)
    {
        if (mask[i])
            GLX_WRITE_PIXEL( x[i], FLIP(y[i]), DITHER( x[i], y[i], r, g, b ) );
    }
}

/*
 * Write an array of identical 8-bit PF_DITHER pixels to an XImage.  The
 * pixel value is the one set by DD.color().
 */
static void write_pixels_mono_DITHER8_ximage( MONO_PIXEL_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    register GLubyte r, g, b;
    DITHER_SETUP;
    r = xsmesa->red;
    g = xsmesa->green;
    b = xsmesa->blue;
    for (i=0;i<n;i++)
    {
        if (mask[i])
            GLX_WRITE_PIXEL( x[i], FLIP(y[i]), DITHER( x[i], y[i], r, g, b ));
    }
}

/*
 * Write an array of identical 8-bit PF_LOOKUP pixels to an XImage.  The
 * pixel value is the one set by DD.color().
 */
static void write_pixels_mono_LOOKUP8_ximage( MONO_PIXEL_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    register GLubyte pixel = xsmesa->pixel;
    for (i=0;i<n;i++)
    {
        if (mask[i])
            GLX_WRITE_PIXEL( x[i], FLIP(y[i]), pixel );
    }
}

/*
 * Write an array of identical PF_1BIT pixels to an XImage.  The pixel
 * value is the one set by DD.color().
 */
static void write_pixels_mono_1BIT_ximage( MONO_PIXEL_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    register GLubyte r, g, b;
    SETUP_1BIT;
    r = xsmesa->red;
    g = xsmesa->green;
    b = xsmesa->blue;
    for (i=0;i<n;i++)
    {
        if (mask[i])
            GLX_WRITE_PIXEL( x[i], FLIP(y[i]), DITHER_1BIT( x[i], y[i], r, g, b ));
    }
}

/*
 * Write an array of identical 8-bit PF_GRAYSCALE pixels to an XImage.  The
 * pixel value is the one set by DD.color().
 */
static void write_pixels_mono_GRAYSCALE8_ximage( MONO_PIXEL_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    register unsigned long p = xsmesa->pixel;
    for (i=0;i<n;i++)
    {
        if (mask[i])
            GLX_WRITE_PIXEL( x[i], FLIP(y[i]), p );
    }
}

/**********************************************************************/
/*** Write INDEX SPAN functions                                     ***/
/**********************************************************************/

#define INDEX_SPAN_ARGS	const GLcontext *ctx,					\
			GLuint n, GLint x, GLint y, const GLuint index[], \
			const GLubyte mask[]


/*
 * Write a span of CI pixels to a Pixmap.
 */
static void write_span_index_pixmap( INDEX_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    XID value;
    DDXPointRec pt;
    DrawablePtr buffer = xsmesa->xsm_buffer->buffer;
    GCPtr gc = xsmesa->xsm_buffer->gc2;
    register GLuint i;
    pt.y=FLIP(y);
    for (i=0;i<n;i++,x++)
    {
        if (mask[i])
        {
            pt.x=x;
            value = (XID)index[i];
            DoChangeGC(gc, GCForeground, &value, 0);
            ValidateGC(buffer, gc);
            (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
        }
    }
}

/*
 * Write a span of CI pixels to an XImage.
 */
static void write_span_index_ximage( INDEX_SPAN_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    y = FLIP(y);
    for (i=0;i<n;i++,x++)
    {
        if (mask[i])
            GLX_WRITE_PIXEL( x, y, (unsigned long) index[i] );
    }
}

/**********************************************************************/
/*** Write INDEX PIXELS functions                                   ***/
/**********************************************************************/

#define INDEX_PIXELS_ARGS	const GLcontext *ctx,				\
				GLuint n, const GLint x[], const GLint y[], \
				const GLuint index[], const GLubyte mask[]


/*
 * Write an array of CI pixels to a Pixmap.
 */
static void write_pixels_index_pixmap( INDEX_PIXELS_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    XID value;
    DDXPointRec pt;
    DrawablePtr buffer = xsmesa->xsm_buffer->buffer;
    GCPtr gc = xsmesa->xsm_buffer->gc2;
    register GLuint i;
    for (i=0;i<n;i++)
    {
        if (mask[i])
        {
            pt.x=x[i];
            pt.y=FLIP(y[i]);
            value = (XID)index[i];
            DoChangeGC(gc, GCForeground, &value, 0);
            ValidateGC(buffer, gc);
            (*gc->ops->PolyPoint)(buffer, gc, CoordModeOrigin, 1, &pt);
        }
    }
}

/*
 * Write an array of CI pixels to an XImage.
 */
static void write_pixels_index_ximage( INDEX_PIXELS_ARGS )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    for (i=0;i<n;i++)
    {
        if (mask[i])
            GLX_WRITE_PIXEL( x[i], FLIP(y[i]), (unsigned long) index[i] );
    }
}

/**********************************************************************/
/*****                      Pixel reading                         *****/
/**********************************************************************/

/*
 * Read a horizontal span of color-index pixels.
 */
static void read_index_span( const GLcontext *ctx,
                             GLuint n, GLint x, GLint y, GLuint index[] )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    int i;
    XSMesaBuffer b=xsmesa->xsm_buffer;
    ScreenPtr pScreen = b->xsm_visual->pScreen;

    y = FLIP(y);

    if (b->buffer)
    {
        DDXPointRec pt;
        char* data=b->scandata;
        int w;
        int i;

        w=n;
        pt.x=x+b->buffer->x;
        pt.y=FLIP(y)+b->buffer->y;
        if (pt.y<0 || pt.y >= pScreen->height)
            return;
        if (pt.x < 0)
        {
            w-=pt.x;
            pt.x=0;
        }
        if (pt.x+w > pScreen->width)
            w-=pt.x+w-pScreen->width;
        if (w<=0)
            return;
        (*pScreen->GetSpans)(b->buffer, w, &pt,
                             &w, 1, data);
        switch (b->buffer->bitsPerPixel)
        {
            case 8:
                {
                    GLubyte* ptr1=(GLubyte*)data;
                    for (i=0;i<n;i++)
                    {
                        index[i]=*ptr1++;
                    }
                }
                break;
            case 15:
            case 16:
                {
                    GLushort* ptr2=(GLushort*)data;
                    for (i=0;i<n;i++)
                    {
                        index[i]=*ptr2++;
                    }
                }
                break;
        }
    }
    else if (b->backimage)
    {
        for (i=0;i<n;i++,x++)
        {
            index[i] = (GLuint) GLX_READ_PIXEL( x, y );
        }
    }
}

/*
 * Read a horizontal span of color pixels.
 */
static void read_color_span( const GLcontext *ctx,
                             GLuint n, GLint x, GLint y,
                             GLubyte rgba[][4])
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    XSMesaBuffer b=xsmesa->xsm_buffer;
    XSMesaVisual v = b->xsm_visual;

    if (b->buffer)
    {
        DDXPointRec pt;
        char* data=b->scandata;
        int w;

        w=n;
        pt.x=x+b->buffer->x;
        pt.y=FLIP(y)+b->buffer->y;
        if (pt.y<0 || pt.y >= v->pScreen->height)
            return;
        if (pt.x < 0)
        {
            w-=pt.x;
            pt.x=0;
        }
        if (pt.x+w > v->pScreen->width)
            w-=pt.x+w-v->pScreen->width;
        if (w<=0)
            return;
        (*v->pScreen->GetSpans)(b->buffer, w, &pt,
                                &w, 1, data);

        switch (xsmesa->pixelformat)
        {
            case PF_TRUECOLOR:
            case PF_TRUEDITHER:
            case PF_5R6G5B:
                {
                    unsigned long rMask = xsmesa->xsm_visual->pVisual->redMask;
                    unsigned long gMask = xsmesa->xsm_visual->pVisual->greenMask;
                    unsigned long bMask = xsmesa->xsm_visual->pVisual->blueMask;
                    GLubyte *pixelToR = xsmesa->xsm_visual->PixelToR;
                    GLubyte *pixelToG = xsmesa->xsm_visual->PixelToG;
                    GLubyte *pixelToB = xsmesa->xsm_visual->PixelToB;
                    GLint rShift = xsmesa->xsm_visual->rshift;
                    GLint gShift = xsmesa->xsm_visual->gshift;
                    GLint bShift = xsmesa->xsm_visual->bshift;
                    switch (b->buffer->bitsPerPixel)
                    {
                        case 8:
                            {
                                GLubyte* ptr1=(GLubyte*)data;
                                for (i=0;i<n;i++)
                                {
                                    GLubyte p = *ptr1++;
                                    rgba[i][0] = pixelToR[(p & rMask) >> rShift];
                                    rgba[i][1] = pixelToG[(p & gMask) >> gShift];
                                    rgba[i][2] = pixelToB[(p & bMask) >> bShift];
                                    rgba[i][3] = 255;
                                }
                            }
                            break;
                        case 15:
                        case 16:
                            {
                                GLushort* ptr2=(GLushort*)data;
                                for (i=0;i<n;i++)
                                {
                                    GLushort p = *ptr2++;
                                    rgba[i][0] = pixelToR[(p & rMask) >> rShift];
                                    rgba[i][1] = pixelToG[(p & gMask) >> gShift];
                                    rgba[i][2] = pixelToB[(p & bMask) >> bShift];
                                    rgba[i][3] = 255;
                                }
                            }
                            break;
                        case 24:
                        case 32:
                            {
                                GLuint* ptr4=(GLuint*)data;
                                for (i=0;i<n;i++)
                                {
                                    GLuint p = *ptr4++;
                                    rgba[i][0] = pixelToR[(p & rMask) >> rShift];
                                    rgba[i][1] = pixelToG[(p & gMask) >> gShift];
                                    rgba[i][2] = pixelToB[(p & bMask) >> bShift];
                                    rgba[i][3] = 255;
                                }
                            }
                            break;
                        default:
                            die("Problem in DD.read_color_span (1)");
                    }
                }
                break;
            case PF_8A8B8G8R:
                {
                    GLuint *ptr4 = (GLuint *)data;
                    for (i=0;i<n;i++)
                    {
                        GLuint p4 = *ptr4++;
                        rgba[i][0] = (GLubyte) ( p4        & 0xff);
                        rgba[i][1] = (GLubyte) ((p4 >> 8)  & 0xff);
                        rgba[i][2] = (GLubyte) ((p4 >> 16) & 0xff);
                        rgba[i][3] = (GLubyte) ((p4 >> 24) & 0xff);
                    }
                }
                break;
            case PF_8R8G8B:
                {
                    GLuint *ptr4 = (GLuint *) data;
                    for (i=0;i<n;i++)
                    {
                        GLuint p4 = *ptr4++;
                        rgba[i][0] = (GLubyte) ((p4 >> 16) & 0xff);
                        rgba[i][1] = (GLubyte) ((p4 >> 8)  & 0xff);
                        rgba[i][2] = (GLubyte) ( p4        & 0xff);
                        rgba[i][3] = 255;
                    }
                }
                break;
            case PF_DITHER:
            case PF_LOOKUP:
            case PF_GRAYSCALE:
                {
                    GLubyte *red_table   = xsmesa->xsm_buffer->pixel_to_r;
                    GLubyte *green_table = xsmesa->xsm_buffer->pixel_to_g;
                    GLubyte *blue_table  = xsmesa->xsm_buffer->pixel_to_b;
                    if (xsmesa->xsm_visual->pVisual->nplanes==8)
                    {
                        GLubyte *ptr1 = (GLubyte *) data;
                        for (i=0;i<n;i++)
                        {
                            unsigned long p = *ptr1++;
                            rgba[i][0] = red_table[p];
                            rgba[i][1] = green_table[p];
                            rgba[i][2] = blue_table[p];
                            rgba[i][3] = 255;
                        }
                    }
                    else
                    {
                        /*
                        for (i=0;i<n;i++) {
                          unsigned long p = GLXProcs.GetPixel( span, i, 0 );
                          red[i]   = red_table[p];
                          green[i] = green_table[p];
                          blue[i]  = blue_table[p];
                          alpha[i] = 255;
                        }*/
                        gl_problem(NULL,"Problem in DD.read_color_span (1)");       
                    }
                }
                break;
            case PF_1BIT:
                {
                    /*
                    int bitFlip = xsmesa->xsm_visual->bitFlip;
                    y = FLIP(y);
                    for (i=0;i<n;i++,x++)
                    {
                        unsigned long p = GLX_READ_PIXEL( x, y ) ^ bitFlip;
                        rgba[i][0] = (GLubyte) (p * 255);
                        rgba[i][1] = (GLubyte) (p * 255);
                        rgba[i][2] = (GLubyte) (p * 255);
                        rgba[i][3] = 255;
                    }
                    */
                }
                break;
            case PF_HPCR:
            default:
                gl_problem(NULL,"Problem in DD.read_color_span (1)");
                return;
        }
    }
    else if (b->backimage)
    {
        y = FLIP(y);
        switch (xsmesa->pixelformat)
        {
            case PF_TRUECOLOR:
            case PF_TRUEDITHER:
            case PF_5R6G5B:
                {
                    unsigned long rMask = xsmesa->xsm_visual->pVisual->redMask;
                    unsigned long gMask = xsmesa->xsm_visual->pVisual->greenMask;
                    unsigned long bMask = xsmesa->xsm_visual->pVisual->blueMask;
                    GLubyte *pixelToR = xsmesa->xsm_visual->PixelToR;
                    GLubyte *pixelToG = xsmesa->xsm_visual->PixelToG;
                    GLubyte *pixelToB = xsmesa->xsm_visual->PixelToB;
                    GLint rShift = xsmesa->xsm_visual->rshift;
                    GLint gShift = xsmesa->xsm_visual->gshift;
                    GLint bShift = xsmesa->xsm_visual->bshift;
                    for (i=0;i<n;i++)
                    {
                        unsigned long p = GLX_READ_PIXEL( x+i, y );
                        rgba[i][0] = pixelToR[(p & rMask) >> rShift];
                        rgba[i][1] = pixelToG[(p & gMask) >> gShift];
                        rgba[i][2] = pixelToB[(p & bMask) >> bShift];
                        rgba[i][3] = 255;
                    }
                }
                break;
            case PF_8A8B8G8R:
                {
                    for (i=0;i<n;i++)
                    {
                        GLuint p4 = GLX_READ_PIXEL( x+i, y );
                        rgba[i][0] = (GLubyte) ( p4        & 0xff);
                        rgba[i][1] = (GLubyte) ((p4 >> 8)  & 0xff);
                        rgba[i][2] = (GLubyte) ((p4 >> 16) & 0xff);
                        rgba[i][3] = (GLint)   ((p4 >> 24) & 0xff);
                    }
                }
                break;
            case PF_8R8G8B:
                {
                    for (i=0;i<n;i++)
                    {
                        GLuint p4 = GLX_READ_PIXEL( x+i, y );
                        rgba[i][0] = (GLubyte) ((p4 >> 16) & 0xff);
                        rgba[i][1] = (GLubyte) ((p4 >> 8)  & 0xff);
                        rgba[i][2] = (GLubyte) ( p4        & 0xff);
                        rgba[i][3] = 255;
                    }
                }
                break;
            case PF_HPCR:
                {
                    for (i=0;i<n;i++)
                    {
                        GLubyte p = GLX_READ_PIXEL( x+i, y );
                        rgba[i][0] =  p & 0xE0;
                        rgba[i][1] = (p & 0x1C) << 3;
                        rgba[i][2] = (p & 0x03) << 6;
                        rgba[i][3] = 255;
                    }
                }
                break;
            case PF_DITHER:
            case PF_LOOKUP:
            case PF_GRAYSCALE:
                {
                    GLubyte *red_table   = xsmesa->xsm_buffer->pixel_to_r;
                    GLubyte *green_table = xsmesa->xsm_buffer->pixel_to_g;
                    GLubyte *blue_table  = xsmesa->xsm_buffer->pixel_to_b;
                    if (xsmesa->xsm_visual->pVisual->nplanes==8)
                    {
                        for (i=0;i<n;i++)
                        {
                            GLubyte p = GLX_READ_PIXEL( x+i, y );
                            rgba[i][0] = red_table[p];
                            rgba[i][1] = green_table[p];
                            rgba[i][2] = blue_table[p];
                            rgba[i][3] = 255;
                        }
                    }
                    else
                    {
                        for (i=0;i<n;i++)
                        {
                            GLubyte p = GLX_READ_PIXEL( x+i, y );
                            rgba[i][0] = red_table[p];
                            rgba[i][1] = green_table[p];
                            rgba[i][2] = blue_table[p];
                            rgba[i][3] = 255;
                        }
                    }
                }
                break;
            case PF_1BIT:
                {
                    int bitFlip = xsmesa->xsm_visual->bitFlip;
                    for (i=0;i<n;i++)
                    {
                        unsigned long p = GLX_READ_PIXEL( x, y ) ^ bitFlip;
                        rgba[i][0] = (GLubyte) (p * 255);
                        rgba[i][1] = (GLubyte) (p * 255);
                        rgba[i][2] = (GLubyte) (p * 255);
                        rgba[i][3] = 255;
                    }
                }
                break;
            default:
                gl_problem(NULL,"Problem in DD.read_color_span (2)");
                return;
        }
    }
}

/*
 * Read an array of color index pixels.
 */
static void read_index_pixels( const GLcontext *ctx,
                               GLuint n, const GLint x[], const GLint y[],
                               GLuint indx[], const GLubyte mask[] )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    XSMesaBuffer b= xsmesa->xsm_buffer;
    if (b->buffer)
    {
        for (i=0;i<n;i++)
        {
            if (mask[i])
            {
                indx[i] = (GLuint) read_pixel( xsmesa, b,
                                               x[i], FLIP(y[i]) );
            }
        }
    }
    else if (b->backimage)
    {
        for (i=0;i<n;i++)
        {
            if (mask[i])
            {
                indx[i] = (GLuint) GLX_READ_PIXEL( x[i], FLIP(y[i]) );
            }
        }
    }
}

static void read_color_pixels( const GLcontext *ctx,
                               GLuint n, const GLint x[], const GLint y[],
                               GLubyte rgba[][4], const GLubyte mask[] )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;
    GLXImage *img = xsmesa->xsm_buffer->backimage;
    register GLuint i;
    XSMesaBuffer b = xsmesa->xsm_buffer;

    if (b->buffer)
    {
        switch (xsmesa->pixelformat)
        {
            case PF_TRUECOLOR:
            case PF_TRUEDITHER:
            case PF_5R6G5B:
                {
                    unsigned long rMask = xsmesa->xsm_visual->pVisual->redMask;
                    unsigned long gMask = xsmesa->xsm_visual->pVisual->greenMask;
                    unsigned long bMask = xsmesa->xsm_visual->pVisual->blueMask;
                    GLubyte *pixelToR = xsmesa->xsm_visual->PixelToR;
                    GLubyte *pixelToG = xsmesa->xsm_visual->PixelToG;
                    GLubyte *pixelToB = xsmesa->xsm_visual->PixelToB;
                    GLint rShift = xsmesa->xsm_visual->rshift;
                    GLint gShift = xsmesa->xsm_visual->gshift;
                    GLint bShift = xsmesa->xsm_visual->bshift;
                    for (i=0;i<n;i++)
                    {
                        if (mask[i])
                        {
                            unsigned long p = read_pixel( xsmesa, b,
                                                          x[i], FLIP(y[i]) );
                            rgba[i][0] = pixelToR[(p & rMask) >> rShift];
                            rgba[i][1] = pixelToG[(p & gMask) >> gShift];
                            rgba[i][2] = pixelToB[(p & bMask) >> bShift];
                            rgba[i][3] = 255;
                        }
                    }
                }
                break;
            case PF_8A8B8G8R:
                for (i=0;i<n;i++)
                {
                    if (mask[i])
                    {
                        unsigned long p = read_pixel( xsmesa, b,
                                                      x[i], FLIP(y[i]) );
                        rgba[i][0] = (GLubyte) ( p        & 0xff);
                        rgba[i][1] = (GLubyte) ((p >> 8)  & 0xff);
                        rgba[i][2] = (GLubyte) ((p >> 16) & 0xff);
                        rgba[i][3] = (GLubyte) ((p >> 24) & 0xff);
                    }
                }
                break;
            case PF_8R8G8B:
                for (i=0;i<n;i++)
                {
                    if (mask[i])
                    {
                        unsigned long p = read_pixel( xsmesa, b,
                                                      x[i], FLIP(y[i]) );
                        rgba[i][0] = (GLubyte) ((p >> 16) & 0xff);
                        rgba[i][1] = (GLubyte) ((p >> 8)  & 0xff);
                        rgba[i][2] = (GLubyte) ( p        & 0xff);
                        rgba[i][3] = 255;
                    }
                }
                break;
            case PF_DITHER:
            case PF_LOOKUP:
            case PF_GRAYSCALE:
                {
                    GLubyte *red_table   = xsmesa->xsm_buffer->pixel_to_r;
                    GLubyte *green_table = xsmesa->xsm_buffer->pixel_to_g;
                    GLubyte *blue_table  = xsmesa->xsm_buffer->pixel_to_b;
                    for (i=0;i<n;i++)
                    {
                        if (mask[i])
                        {
                            unsigned long p = read_pixel( xsmesa, b,
                                                          x[i], FLIP(y[i]) );
                            rgba[i][0] = red_table[p];
                            rgba[i][1] = green_table[p];
                            rgba[i][2] = blue_table[p];
                            rgba[i][3] = 255;
                        }
                    }
                }
                break;
            case PF_1BIT:
                {
                    int bitFlip = xsmesa->xsm_visual->bitFlip;
                    for (i=0;i<n;i++)
                    {
                        if (mask[i])
                        {
                            unsigned long p = read_pixel( xsmesa, b,
                                                          x[i], FLIP(y[i])) ^ bitFlip;
                            rgba[i][0] = (GLubyte) (p * 255);
                            rgba[i][1] = (GLubyte) (p * 255);
                            rgba[i][2] = (GLubyte) (p * 255);
                            rgba[i][3] = 255;
                        }
                    }
                }
                break;
            case PF_HPCR:
            default:
                gl_problem(NULL,"Problem in DD.read_color_pixels (1)");
                return;
        }
    }
    else if (b->backimage)
    {
        switch (xsmesa->pixelformat)
        {
            case PF_TRUECOLOR:
            case PF_TRUEDITHER:
            case PF_5R6G5B:
                {
                    unsigned long rMask = xsmesa->xsm_visual->pVisual->redMask;
                    unsigned long gMask = xsmesa->xsm_visual->pVisual->greenMask;
                    unsigned long bMask = xsmesa->xsm_visual->pVisual->blueMask;
                    GLubyte *pixelToR = xsmesa->xsm_visual->PixelToR;
                    GLubyte *pixelToG = xsmesa->xsm_visual->PixelToG;
                    GLubyte *pixelToB = xsmesa->xsm_visual->PixelToB;
                    GLint rShift = xsmesa->xsm_visual->rshift;
                    GLint gShift = xsmesa->xsm_visual->gshift;
                    GLint bShift = xsmesa->xsm_visual->bshift;
                    for (i=0;i<n;i++)
                    {
                        if (mask[i])
                        {
                            unsigned long p = GLX_READ_PIXEL( x[i], FLIP(y[i]) );
                            rgba[i][0] = pixelToR[(p & rMask) >> rShift];
                            rgba[i][1] = pixelToG[(p & gMask) >> gShift];
                            rgba[i][2] = pixelToB[(p & bMask) >> bShift];
                            rgba[i][3] = 255;
                        }
                    }
                }
                break;
            case PF_8A8B8G8R:
                for (i=0;i<n;i++)
                {
                    if (mask[i])
                    {
                        GLuint p4 = GLX_READ_PIXEL( x[i], FLIP(y[i]) );
                        rgba[i][0] = (GLubyte) ( p4        & 0xff);
                        rgba[i][1] = (GLubyte) ((p4 >> 8)  & 0xff);
                        rgba[i][2] = (GLubyte) ((p4 >> 16) & 0xff);
                        rgba[i][3] = (GLubyte) ((p4 >> 24) & 0xff);
                    }
                }
                break;
            case PF_8R8G8B:
                for (i=0;i<n;i++)
                {
                    if (mask[i])
                    {
                        GLuint p4 = GLX_READ_PIXEL( x[i], FLIP(y[i]) );
                        rgba[i][0] = (GLubyte) ((p4 >> 16) & 0xff);
                        rgba[i][1] = (GLubyte) ((p4 >> 8)  & 0xff);
                        rgba[i][2] = (GLubyte) ( p4        & 0xff);
                        rgba[i][3] = 255;
                    }
                }
                break;
            case PF_HPCR:
                for (i=0;i<n;i++)
                {
                    if (mask[i])
                    {
                        GLubyte p = GLX_READ_PIXEL( x[i], FLIP(y[i]) );
                        rgba[i][0] =  p & 0xE0;
                        rgba[i][1] = (p & 0x1C) << 3;
                        rgba[i][2] = (p & 0x03) << 6;
                        rgba[i][3] = 255;
                    }
                }
                break;
            case PF_DITHER:
            case PF_LOOKUP:
            case PF_GRAYSCALE:
                {
                    GLubyte *red_table   = xsmesa->xsm_buffer->pixel_to_r;
                    GLubyte *green_table = xsmesa->xsm_buffer->pixel_to_g;
                    GLubyte *blue_table  = xsmesa->xsm_buffer->pixel_to_b;
                    for (i=0;i<n;i++)
                    {
                        if (mask[i])
                        {
                            unsigned long p = GLX_READ_PIXEL( x[i], FLIP(y[i]) );
                            rgba[i][0] = red_table[p];
                            rgba[i][1] = green_table[p];
                            rgba[i][2] = blue_table[p];
                            rgba[i][3] = 255;
                        }
                    }
                }
                break;
            case PF_1BIT:
                {
                    int bitFlip = xsmesa->xsm_visual->bitFlip;
                    for (i=0;i<n;i++)
                    {
                        if (mask[i])
                        {
                            unsigned long p = GLX_READ_PIXEL( x[i], FLIP(y[i]) ) ^ bitFlip;
                            rgba[i][0] = (GLubyte) (p * 255);
                            rgba[i][1] = (GLubyte) (p * 255);
                            rgba[i][2] = (GLubyte) (p * 255);
                            rgba[i][3] = 255;
                        }
                    }
                }
                break;
            default:
                gl_problem(NULL,"Problem in DD.read_color_pixels (1)");
                return;
        }
    }
}

/**********************************************************************/
/*****                     Clear Depth Buffer                     *****/
/**********************************************************************/

/*
 * Clear the depth buffer.  If the depth buffer doesn't exist yet we'll
 * allocate it now.
 * This function is only called through Driver.clear_depth_buffer.
 */
static void clear_depth( GLcontext* ctx )
{
    GLdepth clear_value = (GLdepth) (ctx->Depth.Clear * DEPTH_SCALE);

    if (ctx->Visual->DepthBits == 0 || !ctx->Buffer->Depth || !ctx->Depth.Mask)
    {
        /* no depth buffer, or writing to it is disabled */
        return;
    }

    /* The loops in this function have been written so the IRIX 5.3
     * C compiler can unroll them.  Hopefully other compilers can too!
     */
    if (ctx->Scissor.Enabled)
    {
        /* only clear scissor region */
        GLint y;
        for (y=ctx->Buffer->Ymin; y<=ctx->Buffer->Ymax; y++)
        {
            GLdepth *d = ctx->Buffer->Depth + ctx->Buffer->Width * y + ctx->Buffer->Xmin;
            GLint    n = ctx->Buffer->Xmax - ctx->Buffer->Xmin + 1;
            do
            {
                *d++ = clear_value;
                n--;
            } while (n);
        }
    }
    else
    {
        /* clear whole buffer */
        if (sizeof(GLdepth)==2 && (clear_value&0xff)==(clear_value>>8))
        {
            /* lower and upper bytes of clear_value are same, use MEMSET */
            MEMSET( ctx->Buffer->Depth, clear_value&0xff,
                    2*ctx->Buffer->Width*ctx->Buffer->Height);
        }
        else
        {
            GLdepth *d = ctx->Buffer->Depth;
            GLint n = ctx->Buffer->Width * ctx->Buffer->Height;
            while (n>=16)
            {
                d[0] = clear_value;    d[1] = clear_value;
                d[2] = clear_value;    d[3] = clear_value;
                d[4] = clear_value;    d[5] = clear_value;
                d[6] = clear_value;    d[7] = clear_value;
                d[8] = clear_value;    d[9] = clear_value;
                d[10] = clear_value;   d[11] = clear_value;
                d[12] = clear_value;   d[13] = clear_value;
                d[14] = clear_value;   d[15] = clear_value;
                d += 16;
                n -= 16;
            }
            while (n>0)
            {
                *d++ = clear_value;
                n--;
            }
        }
    }
}

/**********************************************************************/
/*****                   Depth Testing Functions                  *****/
/**********************************************************************/

/*
 * Depth test horizontal spans of fragments.  These functions are called
 * via ctx->Driver.depth_test_span only.
 *
 * Input:  n - number of pixels in the span
 *         x, y - location of leftmost pixel in span in window coords
 *         z - array [n] of integer depth values
 * In/Out:  mask - array [n] of flags (1=draw pixel, 0=don't draw) 
 * Return:  number of pixels which passed depth test
 */

/*
 * glDepthFunc( any ) and glDepthMask( GL_TRUE or GL_FALSE ).
 */
static GLuint depth_test_span( GLcontext* ctx, GLuint n, GLint x, GLint y,
                               const GLdepth z[], GLubyte mask[] )
{
    XSMesaContext xsmesa = (XSMesaContext) ctx->DriverCtx;
    GLubyte *m = mask;
    GLuint i;
    GLuint passed = 0;

    /* Flip Y */
    y = ctx->Buffer->Height - y - 1;
    /* switch cases ordered from most frequent to less frequent */
    switch (ctx->Depth.Func)
    {
        case GL_LESS:
            if (ctx->Depth.Mask)
            {
                /* Update Z buffer */
                for (i=0; i<n; i++,x++,m++)
                {
                    if (*m)
                    {
                        if (z[i] < GLX_READ_DEPTH(x,y))
                        {
                            /* pass */
                            GLX_WRITE_DEPTH(x,y,z[i]);
                            passed++;
                        }
                        else
                        {
                            /* fail */
                            *m = 0;
                        }
                    }
                }
            }
            else
            {
                /* Don't update Z buffer */
                for (i=0; i<n; i++,x++,m++)
                {
                    if (*m)
                    {
                        if (z[i] < GLX_READ_DEPTH(x,y))
                        {
                            /* pass */
                            passed++;
                        }
                        else
                        {
                            *m = 0;
                        }
                    }
                }
            }
            break;
        case GL_LEQUAL:
            if (ctx->Depth.Mask)
            {
                /* Update Z buffer */
                for (i=0;i<n;i++,x++,m++)
                {
                    if (*m)
                    {
                        if (z[i] <= GLX_READ_DEPTH(x,y))
                        {
                            GLX_WRITE_DEPTH(x,y,z[i]);
                            passed++;
                        }
                        else
                        {
                            *m = 0;
                        }
                    }
                }
            }
            else
            {
                /* Don't update Z buffer */
                for (i=0;i<n;i++,x++,m++)
                {
                    if (*m)
                    {
                        if (z[i] <= GLX_READ_DEPTH(x,y))
                        {
                            /* pass */
                            passed++;
                        }
                        else
                        {
                            *m = 0;
                        }
                    }
                }
            }
            break;
        case GL_GEQUAL:
            if (ctx->Depth.Mask)
            {
                /* Update Z buffer */
                for (i=0;i<n;i++,x++,m++)
                {
                    if (*m)
                    {
                        if (z[i] >= GLX_READ_DEPTH(x,y))
                        {
                            GLX_WRITE_DEPTH(x,y,z[i]);
                            passed++;
                        }
                        else
                        {
                            *m = 0;
                        }
                    }
                }
            }
            else
            {
                /* Don't update Z buffer */
                for (i=0;i<n;i++,x++,m++)
                {
                    if (*m)
                    {
                        if (z[i] >= GLX_READ_DEPTH(x,y))
                        {
                            /* pass */
                            passed++;
                        }
                        else
                        {
                            *m = 0;
                        }
                    }
                }
            }
            break;
        case GL_GREATER:
            if (ctx->Depth.Mask)
            {
                /* Update Z buffer */
                for (i=0;i<n;i++,x++,m++)
                {
                    if (*m)
                    {
                        if (z[i] > GLX_READ_DEPTH(x,y))
                        {
                            GLX_WRITE_DEPTH(x,y,z[i]);
                            passed++;
                        }
                        else
                        {
                            *m = 0;
                        }
                    }
                }
            }
            else
            {
                /* Don't update Z buffer */
                for (i=0;i<n;i++,x++,m++)
                {
                    if (*m)
                    {
                        if (z[i] > GLX_READ_DEPTH(x,y))
                        {
                            /* pass */
                            passed++;
                        }
                        else
                        {
                            *m = 0;
                        }
                    }
                }
            }
            break;
        case GL_NOTEQUAL:
            if (ctx->Depth.Mask)
            {
                /* Update Z buffer */
                for (i=0;i<n;i++,x++,m++)
                {
                    if (*m)
                    {
                        if (z[i] != GLX_READ_DEPTH(x,y))
                        {
                            GLX_WRITE_DEPTH(x,y,z[i]);
                            passed++;
                        }
                        else
                        {
                            *m = 0;
                        }
                    }
                }
            }
            else
            {
                /* Don't update Z buffer */
                for (i=0;i<n;i++,x++,m++)
                {
                    if (*m)
                    {
                        if (z[i] != GLX_READ_DEPTH(x,y))
                        {
                            /* pass */
                            passed++;
                        }
                        else
                        {
                            *m = 0;
                        }
                    }
                }
            }
            break;
        case GL_EQUAL:
            if (ctx->Depth.Mask)
            {
                /* Update Z buffer */
                for (i=0;i<n;i++,x++,m++)
                {
                    if (*m)
                    {
                        if (z[i] == GLX_READ_DEPTH(x,y))
                        {
                            GLX_WRITE_DEPTH(x,y,z[i]);
                            passed++;
                        }
                        else
                        {
                            *m =0;
                        }
                    }
                }
            }
            else
            {
                /* Don't update Z buffer */
                for (i=0;i<n;i++,x++,m++)
                {
                    if (*m)
                    {
                        if (z[i] == GLX_READ_DEPTH(x,y))
                        {
                            /* pass */
                            passed++;
                        }
                        else
                        {
                            *m =0;
                        }
                    }
                }
            }
            break;
        case GL_ALWAYS:
            if (ctx->Depth.Mask)
            {
                /* Update Z buffer */
                for (i=0;i<n;i++,x++,m++)
                {
                    if (*m)
                    {
                        GLX_WRITE_DEPTH(x,y,z[i]);
                        passed++;
                    }
                }
            }
            else
            {
                /* Don't update Z buffer or mask */
                passed = n;
            }
            break;
        case GL_NEVER:
            for (i=0;i<n;i++)
            {
                mask[i] = 0;
            }
            break;
        default:
            gl_problem(ctx, "Bad depth func in gl_depth_test_span_generic");
    } /*switch*/

    return (passed);
}

/*
 * glDepthFunc( any ) and glDepthMask( GL_TRUE or GL_FALSE ).
 */
static void depth_test_pixels( GLcontext* ctx, GLuint n, const GLint x[], const GLint y[],
                               const GLdepth z[], GLubyte mask[] )
{
    XSMesaContext xsmesa = (XSMesaContext) ctx->DriverCtx;
    GLuint i;

    /* switch cases ordered from most frequent to less frequent */
    switch (ctx->Depth.Func)
    {
        case GL_LESS:
            if (ctx->Depth.Mask)
            {
                /* Update Z buffer */
                for (i=0; i<n; i++)
                {
                    if (mask[i])
                    {
                        if (z[i] < GLX_READ_DEPTH(x[i],ctx->Buffer->Height - y[i]-1))
                        {
                            /* pass */
                            GLX_WRITE_DEPTH(x[i],ctx->Buffer->Height - y[i]-1,z[i]);
                        }
                        else
                        {
                            /* fail */
                            mask[i] = 0;
                        }
                    }
                }
            }
            else
            {
                /* Don't update Z buffer */
                for (i=0; i<n; i++)
                {
                    if (mask[i])
                    {
                        if (z[i] < GLX_READ_DEPTH(x[i],ctx->Buffer->Height - y[i]-1))
                        {
                            /* pass */
                        }
                        else
                        {
                            /* fail */
                            mask[i] = 0;
                        }
                    }
                }
            }
            break;
        case GL_LEQUAL:
            if (ctx->Depth.Mask)
            {
                /* Update Z buffer */
                for (i=0; i<n; i++)
                {
                    if (mask[i])
                    {
                        if (z[i] <= GLX_READ_DEPTH(x[i],ctx->Buffer->Height - y[i]-1))
                        {
                            /* pass */
                            GLX_WRITE_DEPTH(x[i],ctx->Buffer->Height - y[i]-1,z[i]);
                        }
                        else
                        {
                            /* fail */
                            mask[i] = 0;
                        }
                    }
                }
            }
            else
            {
                /* Don't update Z buffer */
                for (i=0; i<n; i++)
                {
                    if (mask[i])
                    {
                        if (z[i] <= GLX_READ_DEPTH(x[i],ctx->Buffer->Height - y[i]-1))
                        {
                            /* pass */
                        }
                        else
                        {
                            /* fail */
                            mask[i] = 0;
                        }
                    }
                }
            }
            break;
        case GL_GEQUAL:
            if (ctx->Depth.Mask)
            {
                /* Update Z buffer */
                for (i=0; i<n; i++)
                {
                    if (mask[i])
                    {
                        if (z[i] >= GLX_READ_DEPTH(x[i],ctx->Buffer->Height - y[i]-1))
                        {
                            /* pass */
                            GLX_WRITE_DEPTH(x[i],ctx->Buffer->Height - y[i]-1,z[i]);
                        }
                        else
                        {
                            /* fail */
                            mask[i] = 0;
                        }
                    }
                }
            }
            else
            {
                /* Don't update Z buffer */
                for (i=0; i<n; i++)
                {
                    if (mask[i])
                    {
                        if (z[i] >= GLX_READ_DEPTH(x[i],ctx->Buffer->Height - y[i]-1))
                        {
                            /* pass */
                        }
                        else
                        {
                            /* fail */
                            mask[i] = 0;
                        }
                    }
                }
            }
            break;
        case GL_GREATER:
            if (ctx->Depth.Mask)
            {
                /* Update Z buffer */
                for (i=0; i<n; i++)
                {
                    if (mask[i])
                    {
                        if (z[i] > GLX_READ_DEPTH(x[i],ctx->Buffer->Height - y[i]-1))
                        {
                            /* pass */
                            GLX_WRITE_DEPTH(x[i],ctx->Buffer->Height - y[i]-1,z[i]);
                        }
                        else
                        {
                            /* fail */
                            mask[i] = 0;
                        }
                    }
                }
            }
            else
            {
                /* Don't update Z buffer */
                for (i=0; i<n; i++)
                {
                    if (mask[i])
                    {
                        if (z[i] > GLX_READ_DEPTH(x[i],ctx->Buffer->Height - y[i]-1))
                        {
                            /* pass */
                        }
                        else
                        {
                            /* fail */
                            mask[i] = 0;
                        }
                    }
                }
            }
            break;
        case GL_NOTEQUAL:
            if (ctx->Depth.Mask)
            {
                /* Update Z buffer */
                for (i=0; i<n; i++)
                {
                    if (mask[i])
                    {
                        if (z[i] != GLX_READ_DEPTH(x[i],ctx->Buffer->Height - y[i]-1))
                        {
                            /* pass */
                            GLX_WRITE_DEPTH(x[i],ctx->Buffer->Height - y[i]-1,z[i]);
                        }
                        else
                        {
                            /* fail */
                            mask[i] = 0;
                        }
                    }
                }
            }
            else
            {
                /* Don't update Z buffer */
                for (i=0; i<n; i++)
                {
                    if (mask[i])
                    {
                        if (z[i] != GLX_READ_DEPTH(x[i],ctx->Buffer->Height - y[i]-1))
                        {
                            /* pass */
                        }
                        else
                        {
                            /* fail */
                            mask[i] = 0;
                        }
                    }
                }
            }
            break;
        case GL_EQUAL:
            if (ctx->Depth.Mask)
            {
                /* Update Z buffer */
                for (i=0; i<n; i++)
                {
                    if (mask[i])
                    {
                        if (z[i] == GLX_READ_DEPTH(x[i],ctx->Buffer->Height - y[i]-1))
                        {
                            /* pass */
                            GLX_WRITE_DEPTH(x[i],ctx->Buffer->Height - y[i]-1,z[i]);
                        }
                        else
                        {
                            /* fail */
                            mask[i] = 0;
                        }
                    }
                }
            }
            else
            {
                /* Don't update Z buffer */
                for (i=0; i<n; i++)
                {
                    if (mask[i])
                    {
                        if (z[i] == GLX_READ_DEPTH(x[i],ctx->Buffer->Height - y[i]-1))
                        {
                            /* pass */
                        }
                        else
                        {
                            /* fail */
                            mask[i] = 0;
                        }
                    }
                }
            }
            break;
        case GL_ALWAYS:
            if (ctx->Depth.Mask)
            {
                /* Update Z buffer */
                for (i=0; i<n; i++)
                {
                    if (mask[i])
                    {
                        GLX_WRITE_DEPTH(x[i],ctx->Buffer->Height - y[i]-1,z[i]);
                    }
                }
            }
            else
            {
                /* Don't update Z buffer or mask */
            }
            break;
        case GL_NEVER:
            /* depth test never passes */
            for (i=0;i<n;i++)
            {
                mask[i] = 0;
            }
            break;
        default:
            gl_problem(ctx, "Bad depth func in gl_depth_test_pixels_generic");
    } /*switch*/
}

/**********************************************************************/
/*****                      Read Depth Buffer                     *****/
/**********************************************************************/

/*
 * Return a span of depth values from the depth buffer as floats in [0,1].
 * This function is only called through Driver.read_depth_span_float()
 * Input:  n - how many pixels
 *         x,y - location of first pixel
 * Output:  depth - the array of depth values
 */
void read_span_float_depth( GLcontext* ctx, GLuint n, GLint x, GLint y,
                            GLfloat depth[] )
{
    XSMesaContext xsmesa = (XSMesaContext) ctx->DriverCtx;
    GLfloat scale;
    GLuint i;

    scale = 1.0F / DEPTH_SCALE;

    if (ctx->Buffer->Depth)
    {
        /* Flip Y */
        y = ctx->Buffer->Height - y - 1;
        for (i=0;i<n;i++,x++)
        {
            depth[i] = (GLfloat) GLX_READ_DEPTH(x,y) * scale;
        }
    }
    else
    {
        for (i=0;i<n;i++)
        {
            depth[i] = 0.0F;
        }
    }
}

/*
 * Return a span of depth values from the depth buffer as integers in
 * [0,MAX_DEPTH].
 * This function is only called through Driver.read_depth_span_int()
 * Input:  n - how many pixels
 *         x,y - location of first pixel
 * Output:  depth - the array of depth values
 */
static void read_span_int_depth( GLcontext* ctx, GLuint n, GLint x, GLint y,
                                 GLdepth depth[] )
{
    XSMesaContext xsmesa = (XSMesaContext) ctx->DriverCtx;
    GLuint i;

    if (ctx->Buffer->Depth)
    {
        /* Flip Y */
        y = ctx->Buffer->Height - y - 1;
        for (i=0;i<n;i++,x++)
        {
            depth[i] = GLX_READ_DEPTH(x,y);
        }
    }
    else
    {
        for (i=0;i<n;i++)
        {
            depth[i] = 0;
        }
    }
}

/**********************************************************************/
/*****                  Set Device Driver state                   *****/
/**********************************************************************/

static const char *renderer_string(void)
{
    return ("X11");
}

/*
 * Initialize all the DD.* function pointers.
 * Always the same:
 */
void xsmesa_static_DD_pointers( GLcontext *ctx )
{
    ctx->Driver.UpdateState    = xsmesa_setup_DD_pointers;
    ctx->Driver.RendererString = renderer_string;
    ctx->Driver.GetBufferSize  = get_buffer_size;
    ctx->Driver.Flush          = flush;
    ctx->Driver.Finish         = finish;
    ctx->Driver.SetBuffer      = set_buffer;
    ctx->Driver.Index          = set_index;
    ctx->Driver.Color          = set_color;
    ctx->Driver.ClearIndex     = clear_index;
    ctx->Driver.ClearColor     = clear_color;
    ctx->Driver.IndexMask      = index_mask;
    ctx->Driver.ColorMask      = color_mask;
    ctx->Driver.LogicOp        = logicop;
    ctx->Driver.Dither         = dither;
    /* Depth test */
    ctx->Driver.AllocDepthBuffer   = GLXProcs.CreateDepthBuffer;
    ctx->Driver.DepthTestSpan      = depth_test_span;
    ctx->Driver.DepthTestPixels    = depth_test_pixels;
    ctx->Driver.ReadDepthSpanFloat = read_span_float_depth;
    ctx->Driver.ReadDepthSpanInt   = read_span_int_depth;
    /* Pixel/span reading functions: */
    ctx->Driver.ReadCI32Span   = read_index_span;
    ctx->Driver.ReadRGBASpan   = read_color_span;
    ctx->Driver.ReadCI32Pixels = read_index_pixels;
    ctx->Driver.ReadRGBAPixels = read_color_pixels;
}
/*
 * Calc dynamic function pointers depending on the color
 * buffer configuration.  This is mainly called by XSMesaMakeCurrent.
 */
void xsmesa_setup_DD_pointers( GLcontext *ctx )
{
    XSMesaContext xsmesa= (XSMesaContext) ctx->DriverCtx;

    ctx->Driver.UpdateState    = xsmesa_setup_DD_pointers;
    ctx->Driver.RendererString = renderer_string;
    ctx->Driver.GetBufferSize  = get_buffer_size;
    ctx->Driver.Flush          = flush;
    ctx->Driver.Finish         = finish;
    ctx->Driver.SetBuffer      = set_buffer;
    ctx->Driver.Index          = set_index;
    ctx->Driver.Color          = set_color;
    ctx->Driver.ClearIndex     = clear_index;
    ctx->Driver.ClearColor     = clear_color;
    ctx->Driver.IndexMask      = index_mask;
    ctx->Driver.ColorMask      = color_mask;
    ctx->Driver.LogicOp        = logicop;
    ctx->Driver.Dither         = dither;
    ctx->Driver.Viewport       = NULL;
    ctx->Driver.DepthRange     = NULL;
    /* Depth test */
    ctx->Driver.AllocDepthBuffer   = GLXProcs.CreateDepthBuffer;
    ctx->Driver.DepthTestSpan      = depth_test_span;
    ctx->Driver.DepthTestPixels    = depth_test_pixels;
    ctx->Driver.ReadDepthSpanFloat = read_span_float_depth;
    ctx->Driver.ReadDepthSpanInt   = read_span_int_depth;
    /* Pixel/span reading functions: */
    ctx->Driver.ReadCI32Span   = read_index_span;
    ctx->Driver.ReadRGBASpan   = read_color_span;
    ctx->Driver.ReadCI32Pixels = read_index_pixels;
    ctx->Driver.ReadRGBAPixels = read_color_pixels;
    /* just in case these are missed later, set to NULL */
    ctx->Driver.WriteRGBASpan = NULL;
    ctx->Driver.WriteRGBSpan = NULL;
    ctx->Driver.WriteMonoRGBASpan = NULL;
    ctx->Driver.WriteCI32Span = NULL;
    ctx->Driver.WriteCI8Span = NULL;
    ctx->Driver.WriteMonoCISpan = NULL;
    ctx->Driver.WriteCI32Pixels = NULL;
    ctx->Driver.WriteMonoCIPixels = NULL;

#if 0
    if (!ctx->Driver.PointsFunc)   ctx->Driver.PointsFunc   = xsmesa_get_points_func(   ctx );
    if (!ctx->Driver.LineFunc)     ctx->Driver.LineFunc     = xsmesa_get_line_func(     ctx );
    if (!ctx->Driver.TriangleFunc) ctx->Driver.TriangleFunc = xsmesa_get_triangle_func( ctx );
#else
    if (!ctx->Driver.PointsFunc)   gl_set_point_function(ctx);
    if (!ctx->Driver.LineFunc)     gl_set_line_function(ctx);
    if (!ctx->Driver.TriangleFunc) gl_set_triangle_function(ctx);

    /*
     * These drawing functions depend on color buffer config:
     */
    if (xsmesa->xsm_buffer->buffer!=XIMAGE)
    {
        /* Writing to window or back pixmap */
        ctx->Driver.Clear = clear_pixmap;
        switch (xsmesa->pixelformat)
        {
            case PF_INDEX:
                ctx->Driver.WriteCI32Span       = write_span_index_pixmap;
                ctx->Driver.WriteMonoCISpan     = write_span_mono_pixmap;
                ctx->Driver.WriteCI32Pixels     = write_pixels_index_pixmap;
                ctx->Driver.WriteMonoCIPixels   = write_pixels_mono_pixmap;
                break;
            case PF_TRUECOLOR:
                ctx->Driver.WriteRGBSpan        = write_span_TRUECOLOR_pixmap;
                ctx->Driver.WriteRGBASpan       = write_span_RGBA_TRUECOLOR_pixmap;
                ctx->Driver.WriteMonoRGBASpan   = write_span_mono_pixmap;
                ctx->Driver.WriteRGBAPixels     = write_pixels_TRUECOLOR_pixmap;
                ctx->Driver.WriteMonoRGBAPixels = write_pixels_mono_pixmap;
                break;
            case PF_TRUEDITHER:
                ctx->Driver.WriteRGBSpan        = write_span_TRUEDITHER_pixmap;
                ctx->Driver.WriteRGBASpan       = write_span_RGBA_TRUEDITHER_pixmap;
                ctx->Driver.WriteMonoRGBASpan   = write_span_mono_TRUEDITHER_pixmap;
                ctx->Driver.WriteRGBAPixels     = write_pixels_TRUEDITHER_pixmap;
                ctx->Driver.WriteMonoRGBAPixels = write_pixels_mono_TRUEDITHER_pixmap;
                break;
            case PF_8A8B8G8R:
                ctx->Driver.WriteRGBSpan        = write_span_8R8G8B_pixmap;
                ctx->Driver.WriteRGBASpan       = write_span_8A8B8G8R_pixmap;
                ctx->Driver.WriteMonoRGBASpan   = write_span_mono_pixmap;
                ctx->Driver.WriteRGBAPixels     = write_pixels_8A8B8G8R_pixmap;
                ctx->Driver.WriteMonoRGBAPixels = write_pixels_mono_pixmap;
                break;
            case PF_8R8G8B:
                ctx->Driver.WriteRGBSpan        = write_span_8R8G8B_pixmap;
                ctx->Driver.WriteRGBASpan       = write_span_RGBA8_to_RGB8_pixmap;
                ctx->Driver.WriteMonoRGBASpan   = write_span_mono_pixmap;
                ctx->Driver.WriteRGBAPixels     = write_pixels_8R8G8B_pixmap;
                ctx->Driver.WriteMonoRGBAPixels = write_pixels_mono_pixmap;
                break;
            case PF_5R6G5B:
                ctx->Driver.WriteRGBSpan        = write_span_5R6G5B_pixmap;
                ctx->Driver.WriteRGBASpan        = write_span_RGBA_5R6G5B_pixmap;
                ctx->Driver.WriteMonoRGBASpan   = write_span_mono_pixmap;
                ctx->Driver.WriteRGBAPixels     = write_pixels_5R6G5B_pixmap;
                ctx->Driver.WriteMonoRGBAPixels = write_pixels_mono_pixmap;
                break;
            case PF_DITHER:
                ctx->Driver.WriteRGBSpan        = write_span_DITHER_pixmap;
                ctx->Driver.WriteMonoRGBASpan   = write_span_mono_DITHER_pixmap;
                ctx->Driver.WriteRGBAPixels     = write_pixels_DITHER_pixmap;
                ctx->Driver.WriteMonoRGBAPixels = write_pixels_mono_DITHER_pixmap;
                break;
            case PF_1BIT:
                ctx->Driver.WriteRGBSpan        = write_span_1BIT_pixmap;
                ctx->Driver.WriteMonoRGBASpan   = write_span_mono_1BIT_pixmap;
                ctx->Driver.WriteRGBAPixels     = write_pixels_1BIT_pixmap;
                ctx->Driver.WriteMonoRGBAPixels = write_pixels_mono_1BIT_pixmap;
                break;
            case PF_LOOKUP:
                ctx->Driver.WriteRGBSpan       = write_span_LOOKUP_pixmap;
                ctx->Driver.WriteMonoRGBASpan   = write_span_mono_pixmap;
                ctx->Driver.WriteRGBAPixels     = write_pixels_LOOKUP_pixmap;
                ctx->Driver.WriteMonoRGBAPixels = write_pixels_mono_pixmap;
                break;
            case PF_GRAYSCALE:
                ctx->Driver.WriteRGBSpan       = write_span_GRAYSCALE_pixmap;
                ctx->Driver.WriteMonoRGBASpan   = write_span_mono_pixmap;
                ctx->Driver.WriteRGBAPixels     = write_pixels_GRAYSCALE_pixmap;
                ctx->Driver.WriteMonoRGBAPixels = write_pixels_mono_pixmap;
                break;
            case PF_HPCR:
            default:
                gl_problem(NULL,"Bad pixel format in xsmesa_setup_DD_pointers (1)");
                return;
        }
    }
    else if (xsmesa->xsm_buffer->buffer==XIMAGE)
    {
        /* back XImage */
        if (sizeof(GLushort)!=2 || sizeof(GLuint)!=4)
        {
            /* Do this on Crays */
            ctx->Driver.Clear = (GLbitfield (*)
                                 (GLcontext *, GLbitfield, GLboolean, GLint, GLint, GLint, GLint))
                                (clear_nbit_ximage);
        }
        else
        {
            /* Do this on most machines */
            switch (xsmesa->xsm_buffer->backimage->bits_per_pixel)
            {
                case 8:
                    ctx->Driver.Clear = (GLbitfield (*)
                                         (GLcontext *, GLbitfield, GLboolean, GLint, GLint, GLint, 
                                          GLint))(clear_8bit_ximage);
                    break;
                case 16:
                    ctx->Driver.Clear = (GLbitfield (*)
                                         (GLcontext *, GLbitfield, GLboolean, GLint, GLint, GLint, 
                                          GLint))(clear_16bit_ximage);
                    break;
                case 32:
                    ctx->Driver.Clear = (GLbitfield (*)
                                         (GLcontext *, GLbitfield, GLboolean, GLint, GLint, GLint, 
                                          GLint))(clear_32bit_ximage);
                    break;
                default:
                    ctx->Driver.Clear = (GLbitfield (*)
                                         (GLcontext *, GLbitfield, GLboolean, GLint, GLint, GLint, 
                                          GLint))(clear_nbit_ximage);
                    break;
            }
        }
        switch (xsmesa->pixelformat)
        {
            case PF_INDEX:
                ctx->Driver.WriteCI32Span      = write_span_index_ximage;
                ctx->Driver.WriteMonoCISpan    = write_span_mono_ximage;
                ctx->Driver.WriteCI32Pixels    = write_pixels_index_ximage;
                ctx->Driver.WriteMonoCIPixels  = write_pixels_mono_ximage;
                break;
            case PF_TRUECOLOR:
                /* Generic RGB */
                ctx->Driver.WriteRGBSpan        = write_span_TRUECOLOR_ximage;
                ctx->Driver.WriteRGBASpan       = write_span_RGBA_TRUECOLOR_ximage;
                ctx->Driver.WriteMonoRGBASpan   = write_span_mono_ximage;
                ctx->Driver.WriteRGBAPixels     = write_pixels_TRUECOLOR_ximage;
                ctx->Driver.WriteMonoRGBAPixels = write_pixels_mono_ximage;
                break;
            case PF_TRUEDITHER:
                ctx->Driver.WriteRGBSpan        = write_span_TRUEDITHER_ximage;
                ctx->Driver.WriteRGBASpan       = write_span_RGBA_TRUEDITHER_ximage;
                ctx->Driver.WriteMonoRGBASpan   = write_span_mono_TRUEDITHER_ximage;
                ctx->Driver.WriteRGBAPixels     = write_pixels_TRUEDITHER_ximage;
                ctx->Driver.WriteMonoRGBAPixels = write_pixels_mono_TRUEDITHER_ximage;
                break;
            case PF_8A8B8G8R:
                ctx->Driver.WriteRGBSpan        = write_span_8R8G8B_ximage;
                ctx->Driver.WriteRGBASpan       = write_span_8A8B8G8R_ximage;
                ctx->Driver.WriteMonoRGBASpan   = write_span_mono_8A8B8G8R_ximage;
                ctx->Driver.WriteRGBAPixels     = write_pixels_8A8B8G8R_ximage;
                ctx->Driver.WriteMonoRGBAPixels = write_pixels_mono_8A8B8G8R_ximage;
                break;
            case PF_8R8G8B:
                ctx->Driver.WriteRGBSpan        = write_span_8R8G8B_ximage;
                ctx->Driver.WriteRGBASpan       = write_span_RGBA8_to_RGB8_ximage;
                ctx->Driver.WriteMonoRGBASpan   = write_span_mono_8R8G8B_ximage;
                ctx->Driver.WriteRGBAPixels     = write_pixels_8R8G8B_ximage;
                ctx->Driver.WriteMonoRGBAPixels = write_pixels_mono_8R8G8B_ximage;
                break;
            case PF_5R6G5B:
                ctx->Driver.WriteRGBSpan        = write_span_5R6G5B_ximage;
                ctx->Driver.WriteRGBASpan       = write_span_RGBA_5R6G5B_ximage;
                ctx->Driver.WriteMonoRGBASpan   = write_span_mono_ximage;
                ctx->Driver.WriteRGBAPixels     = write_pixels_5R6G5B_ximage;
                ctx->Driver.WriteMonoRGBAPixels = write_pixels_mono_ximage;
                break;
            case PF_DITHER:
                if (xsmesa->xsm_visual->pVisual->nplanes==8)
                {
                    ctx->Driver.WriteRGBSpan        = write_span_DITHER8_ximage;
                    ctx->Driver.WriteMonoRGBASpan   = write_span_mono_DITHER8_ximage;
                    ctx->Driver.WriteRGBAPixels     = write_pixels_DITHER8_ximage;
                    ctx->Driver.WriteMonoRGBAPixels = write_pixels_mono_DITHER8_ximage;
                }
                else
                {
                    ctx->Driver.WriteRGBSpan        = write_span_DITHER_ximage;
                    ctx->Driver.WriteMonoRGBASpan   = write_span_mono_DITHER_ximage;
                    ctx->Driver.WriteRGBAPixels     = write_pixels_DITHER_ximage;
                    ctx->Driver.WriteMonoRGBAPixels = write_pixels_mono_DITHER_ximage;
                }
                break;
            case PF_1BIT:
                ctx->Driver.WriteRGBSpan        = write_span_1BIT_ximage;
                ctx->Driver.WriteMonoRGBASpan   = write_span_mono_1BIT_ximage;
                ctx->Driver.WriteRGBAPixels     = write_pixels_1BIT_ximage;
                ctx->Driver.WriteMonoRGBAPixels = write_pixels_mono_1BIT_ximage;
                break;
            case PF_LOOKUP:
                if (xsmesa->xsm_visual->pVisual->nplanes==8)
                {
                    ctx->Driver.WriteRGBSpan        = write_span_LOOKUP8_ximage;
                    ctx->Driver.WriteMonoRGBASpan   = write_span_mono_LOOKUP8_ximage;
                    ctx->Driver.WriteRGBAPixels     = write_pixels_LOOKUP8_ximage;
                    ctx->Driver.WriteMonoRGBAPixels = write_pixels_mono_LOOKUP8_ximage;
                }
                else
                {
                    ctx->Driver.WriteRGBSpan        = write_span_LOOKUP_ximage;
                    ctx->Driver.WriteMonoRGBASpan   = write_span_mono_ximage;
                    ctx->Driver.WriteRGBAPixels     = write_pixels_LOOKUP_ximage;
                    ctx->Driver.WriteMonoRGBAPixels = write_pixels_mono_ximage;
                }
                break;
            case PF_GRAYSCALE:
                if (xsmesa->xsm_visual->pVisual->nplanes==8)
                {
                    ctx->Driver.WriteRGBSpan        = write_span_GRAYSCALE8_ximage;
                    ctx->Driver.WriteMonoRGBASpan   = write_span_mono_GRAYSCALE8_ximage;
                    ctx->Driver.WriteRGBAPixels     = write_pixels_GRAYSCALE8_ximage;
                    ctx->Driver.WriteMonoRGBAPixels = write_pixels_mono_GRAYSCALE8_ximage;
                }
                else
                {
                    ctx->Driver.WriteRGBSpan        = write_span_GRAYSCALE_ximage;
                    ctx->Driver.WriteMonoRGBASpan   = write_span_mono_ximage;
                    ctx->Driver.WriteRGBAPixels     = write_pixels_GRAYSCALE_ximage;
                    ctx->Driver.WriteMonoRGBAPixels = write_pixels_mono_ximage;
                }
                break;
            case PF_HPCR:
            default:
                gl_problem(NULL,"Bad pixel format in xsmesa_setup_DD_pointers (2)");
                return;
        }
    }
}
#endif
