
/*
 * GLX Server Extension
 * Copyright (C) 1998, 1999 Terence Ripperda (ripperda@sgi.com)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * TERENCE RIPPERDA, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM, DAMAGES
 * OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, 
 * ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 * DEALINGS IN THE SOFTWARE.
 */

#include "glx_log.h"
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdarg.h>


static char *log = NULL;
static FILE *fPtr = NULL;
GLint logging = 0;
static char *glx_init_string = "debug";

/*
 * currently, just return. But I may need to change this behavior
 * later, so I want to make that easy to do.
 */

#define GLX_CHECK_LOGGING						\
   if (!log) {								\
      return;								\
   }

#define GLX_CHECK_FILE

#if 0
#define GLX_START_LOG							\
   fPtr = fopen(log, "a");

#define GLX_FINISH_LOG							\
   fclose(fPtr);							\
   fPtr = NULL;
#else
#define GLX_START_LOG

#define GLX_FINISH_LOG
#endif

int
glx_set_log_file(char *new_log) {
   if (new_log == 0) return 1;
   if (log) free(log);
   log = strdup(new_log);
   return glx_clear_log();
}

void
glx_log_initialize() {
   char *str = glx_getvar(glx_init_string);
   if (str) logging = atoi(str);
   if (glx_clear_log()) {
      fprintf(stderr, "Couldn't open log file!!\n");
      logging = -1;
      return;
   } 
   
   /* by default dont log anything */
   if (!logging)
     {
	logging = -1;
	return;
     }
   
   glx_log_print("Logging set to %d\n", logging);
}


void
glx_log_finish() {
   if (fPtr) { 
      fflush(fPtr); 
      fclose(fPtr);
      fPtr=NULL;
   }
}

void
glx_log_flush() {
   if (fPtr) fflush(fPtr);
}

int
glx_clear_log() {
   if (!log) return 1;

   if (fPtr) fclose(fPtr);

   fPtr = fopen(log, "w");
   if (fPtr == NULL) {
      fprintf(stderr, "GLX: Couldn't get file pointer!\n");
      return 1;
   }

   GLX_FINISH_LOG;
   return 0;
}

void
glx_log_print(const char *fmt, ...) {
   va_list ap;
   char *sptr = (char *) fmt;
   GLX_CHECK_LOGGING;
   GLX_CHECK_FILE;

   GLX_START_LOG;
   va_start(ap, fmt);
   while (*sptr) {
      if (*sptr!= '%') {
         fprintf(fPtr, "%c", *sptr);
      } else { 
         sptr++; /* advance past '%' */
         switch (*sptr) {
            case 'f':		/* float */
               {
                  float f = va_arg(ap, float);
                  fprintf(fPtr, "%1.2f", f);
                  break;
               }
            case 'd':		/* int */
               {
                  int d = va_arg(ap, int);
                  fprintf(fPtr, "%d", d);
                  break;
               }
            case 'x':		/* hex int */
               {
                  int x = va_arg(ap, int);
                  fprintf(fPtr, "0x%x", x);
                  break;
               }
            case 'e':		/* enum int */
               {
                  int x = va_arg(ap, int);
                  glx_log_enum((GLenum) x);
                  break;
               }
            case '@':		/* glx opcode */
               {
                  int x = va_arg(ap, int);
                  log_glx_request(x);
                  break;
               }
            case '#':		/* opengl opcode */
               {
                  int x = va_arg(ap, int);
                  log_op_request(x);
                  break;
               }
            case 'g':		/* double */
               {
                  double g = va_arg(ap, double);
                  fprintf(fPtr, "%1.2g", g);
                  break;
               }
            case 'c':		/* char */
               {
                  char c = va_arg(ap, char);
                  fprintf(fPtr, "%c", c);
                  break;
               }
            case 's':		/* string */
               {
                  char *s = va_arg(ap, char *);
                  fprintf(fPtr, "%s", s);
                  break;
               }
            default:
         }
      }
      sptr++;
   }
   va_end(ap);
   GLX_FINISH_LOG;
}

void
glx_log_op_request(int op) {
   GLX_CHECK_LOGGING;
   GLX_CHECK_FILE;

   GLX_START_LOG;
   switch (op) {
      case 0:
         fprintf(fPtr, "Invalid (0)");
         break;
      case 1:
         fprintf(fPtr, "CallList");
         break;
      case 2:
         fprintf(fPtr, "CallLists");
         break;
      case 3:
         fprintf(fPtr, "ListBase");
         break;
      case 4:
         fprintf(fPtr, "Begin");
         break;
      case 5:
         fprintf(fPtr, "Bitmap");
         break;
      case 6:
         fprintf(fPtr, "Color3bv");
         break;
      case 7:
         fprintf(fPtr, "Color3dv");
         break;
      case 8:
         fprintf(fPtr, "Color3fv");
         break;
      case 9:
         fprintf(fPtr, "Color3iv");
         break;
      case 10:
         fprintf(fPtr, "Color3sv");
         break;
      case 11:
         fprintf(fPtr, "Color3ubv");
         break;
      case 12:
         fprintf(fPtr, "Color3uiv");
         break;
      case 13:
         fprintf(fPtr, "Color3usv");
         break;
      case 14:
         fprintf(fPtr, "Color4bv");
         break;
      case 15:
         fprintf(fPtr, "Color4dv");
         break;
      case 16:
         fprintf(fPtr, "Color4fv");
         break;
      case 17:
         fprintf(fPtr, "Color4fi");
         break;
      case 18:
         fprintf(fPtr, "Color4sv");
         break;
      case 19:
         fprintf(fPtr, "Color4ubv");
         break;
      case 20:
         fprintf(fPtr, "Color4uiv");
         break;
      case 21:
         fprintf(fPtr, "Color4usv");
         break;
      case 22:
         fprintf(fPtr, "EdgeFlag");
         break;
      case 23:
         fprintf(fPtr, "End");
         break;
      case 24:
         fprintf(fPtr, "Indexdv");
         break;
      case 25:
         fprintf(fPtr, "Indexfv");
         break;
      case 26:
         fprintf(fPtr, "Indexiv");
         break;
      case 27:
         fprintf(fPtr, "Indexsv");
         break;
      case 28:
         fprintf(fPtr, "Normal3bv");
         break;
      case 29:
         fprintf(fPtr, "Normal3dv");
         break;
      case 30:
         fprintf(fPtr, "Normal3fv");
         break;
      case 31:
         fprintf(fPtr, "Normal3iv");
         break;
      case 32:
         fprintf(fPtr, "Normal3sv");
         break;
      case 33:
         fprintf(fPtr, "RasterPos2dv");
         break;
      case 34:
         fprintf(fPtr, "RasterPos2fv");
         break;
      case 35:
         fprintf(fPtr, "RasterPos2iv");
         break;
      case 36:
         fprintf(fPtr, "RasterPos2sv");
         break;
      case 37:
         fprintf(fPtr, "RasterPos3dv");
         break;
      case 38:
         fprintf(fPtr, "RasterPos3fv");
         break;
      case 39:
         fprintf(fPtr, "RasterPos3iv");
         break;
      case 40:
         fprintf(fPtr, "RasterPos3sv");
         break;
      case 41:
         fprintf(fPtr, "RasterPos4dv");
         break;
      case 42:
         fprintf(fPtr, "RasterPos4fv");
         break;
      case 43:
         fprintf(fPtr, "RasterPos4iv");
         break;
      case 44:
         fprintf(fPtr, "RasterPos4sv");
         break;
      case 45:
         fprintf(fPtr, "Rectdv");
         break;
      case 46:
         fprintf(fPtr, "Rectfv");
         break;
      case 47:
         fprintf(fPtr, "Rectiv");
         break;
      case 48:
         fprintf(fPtr, "Rectsv");
         break;
      case 49:
         fprintf(fPtr, "TexCoord1dv");
         break;
      case 50:
         fprintf(fPtr, "TexCoord1fv");
         break;
      case 51:
         fprintf(fPtr, "TexCoord1iv");
         break;
      case 52:
         fprintf(fPtr, "TexCoord1sv");
         break;
      case 53:
         fprintf(fPtr, "TexCoord2dv");
         break;
      case 54:
         fprintf(fPtr, "TexCoord2fv");
         break;
      case 55:
         fprintf(fPtr, "TexCoord2iv");
         break;
      case 56:
         fprintf(fPtr, "TexCoord2sv");
         break;
      case 57:
         fprintf(fPtr, "TexCoord3dv");
         break;
      case 58:
         fprintf(fPtr, "TexCoord3fv");
         break;
      case 59:
         fprintf(fPtr, "TexCoord3iv");
         break;
      case 60:
         fprintf(fPtr, "TexCoord3sv");
         break;
      case 61:
         fprintf(fPtr, "TexCoord4dv");
         break;
      case 62:
         fprintf(fPtr, "TexCoord4fv");
         break;
      case 63:
         fprintf(fPtr, "TexCoord4iv");
         break;
      case 64:
         fprintf(fPtr, "TexCoord4sv");
         break;
      case 65:
         fprintf(fPtr, "Vertex2dv");
         break;
      case 66:
         fprintf(fPtr, "Vertex2fv");
         break;
      case 67:
         fprintf(fPtr, "Vertex2iv");
         break;
      case 68:
         fprintf(fPtr, "Vertex2sv");
         break;
      case 69:
         fprintf(fPtr, "Vertex3dv");
         break;
      case 70:
         fprintf(fPtr, "Vertex3fv");
         break;
      case 71:
         fprintf(fPtr, "Vertex3iv");
         break;
      case 72:
         fprintf(fPtr, "Vertex3sv");
         break;
      case 73:
         fprintf(fPtr, "Vertex4dv");
         break;
      case 74:
         fprintf(fPtr, "Vertex4fv");
         break;
      case 75:
         fprintf(fPtr, "Vertex4iv");
         break;
      case 76:
         fprintf(fPtr, "Vertex4sv");
         break;
      case 77:
         fprintf(fPtr, "ClipPlane");
         break;
      case 78:
         fprintf(fPtr, "ColorMaterial");
         break;
      case 79:
         fprintf(fPtr, "CullFace");
         break;
      case 80:
         fprintf(fPtr, "Fogf");
         break;
      case 81:
         fprintf(fPtr, "Fogfv");
         break;
      case 82:
         fprintf(fPtr, "Fogi");
         break;
      case 83:
         fprintf(fPtr, "Fogiv");
         break;
      case 84:
         fprintf(fPtr, "FrontFace");
         break;
      case 85:
         fprintf(fPtr, "Hint");
         break;
      case 86:
         fprintf(fPtr, "Lightf");
         break;
      case 87:
         fprintf(fPtr, "Lightfv");
         break;
      case 88:
         fprintf(fPtr, "Lighti");
         break;
      case 89:
         fprintf(fPtr, "Lightiv");
         break;
      case 90:
         fprintf(fPtr, "LightModelf");
         break;
      case 91:
         fprintf(fPtr, "LightModelfv");
         break;
      case 92:
         fprintf(fPtr, "LightModeli");
         break;
      case 93:
         fprintf(fPtr, "LightModeliv");
         break;
      case 94:
         fprintf(fPtr, "LineStipple");
         break;
      case 95:
         fprintf(fPtr, "LineWidth");
         break;
      case 96:
         fprintf(fPtr, "Materialf");
         break;
      case 97:
         fprintf(fPtr, "Materialfv");
         break;
      case 98:
         fprintf(fPtr, "Materiali");
         break;
      case 99:
         fprintf(fPtr, "Materialiv");
         break;
      case 100:
         fprintf(fPtr, "PointSize");
         break;
      case 101:
         fprintf(fPtr, "PolygonMode");
         break;
      case 102:
         fprintf(fPtr, "PolygonStipple");
         break;
      case 103:
         fprintf(fPtr, "Scissor");
         break;
      case 104:
         fprintf(fPtr, "ShadeModel");
         break;
      case 105:
         fprintf(fPtr, "TexParameterf");
         break;
      case 106:
         fprintf(fPtr, "TexParameterfv");
         break;
      case 107:
         fprintf(fPtr, "TexParameteri");
         break;
      case 108:
         fprintf(fPtr, "TexParameteriv");
         break;
      case 109:
         fprintf(fPtr, "TexImage1D");
         break;
      case 110:
         fprintf(fPtr, "TexImage2D");
         break;
      case 111:
         fprintf(fPtr, "TexEnvf");
         break;
      case 112:
         fprintf(fPtr, "TexEnvfv");
         break;
      case 113:
         fprintf(fPtr, "TexEnvi");
         break;
      case 114:
         fprintf(fPtr, "TexEnviv");
         break;
      case 115:
         fprintf(fPtr, "TexGend");
         break;
      case 116:
         fprintf(fPtr, "TexGendv");
         break;
      case 117:
         fprintf(fPtr, "TexGenf");
         break;
      case 118:
         fprintf(fPtr, "TexGenfv");
         break;
      case 119:
         fprintf(fPtr, "TexGeni");
         break;
      case 120:
         fprintf(fPtr, "TexGeniv");
         break;
      case 121:
         fprintf(fPtr, "InitNames");
         break;
      case 122:
         fprintf(fPtr, "LoadName");
         break;
      case 123:
         fprintf(fPtr, "PassThrough");
         break;
      case 124:
         fprintf(fPtr, "PopName");
         break;
      case 125:
         fprintf(fPtr, "PushName");
         break;
      case 126:
         fprintf(fPtr, "DrawBuffer");
         break;
      case 127:
         fprintf(fPtr, "Clear ");
         break;
      case 128:
         fprintf(fPtr, "ClearAccum");
         break;
      case 129:
         fprintf(fPtr, "ClearIndex");
         break;
      case 130:
         fprintf(fPtr, "ClearColor");
         break;
      case 131:
         fprintf(fPtr, "ClearStencil");
         break;
      case 132:
         fprintf(fPtr, "ClearDepth");
         break;
      case 133:
         fprintf(fPtr, "StencilMask");
         break;
      case 134:
         fprintf(fPtr, "ColorMask");
         break;
      case 135:
         fprintf(fPtr, "DepthMask");
         break;
      case 136:
         fprintf(fPtr, "IndexMask");
         break;
      case 137:
         fprintf(fPtr, "Accum");
         break;
      case 138:
         fprintf(fPtr, "Disable");
         break;
      case 139:
         fprintf(fPtr, "Enable ");
         break;
      case 140:
         fprintf(fPtr, "Invalid (140)");
         break;
      case 141:
         fprintf(fPtr, "PopAttrib");
         break;
      case 142:
         fprintf(fPtr, "PushAttrib");
         break;
      case 143:
         fprintf(fPtr, "Map1d");
         break;
      case 144:
         fprintf(fPtr, "Map1f");
         break;
      case 145:
         fprintf(fPtr, "Map2d");
         break;
      case 146:
         fprintf(fPtr, "Map2f");
         break;
      case 147:
         fprintf(fPtr, "MapGrid1d");
         break;
      case 148:
         fprintf(fPtr, "MapGrid1f");
         break;
      case 149:
         fprintf(fPtr, "MapGrid2d");
         break;
      case 150:
         fprintf(fPtr, "MapGrid2f");
         break;
      case 151:
         fprintf(fPtr, "EvalCoord1dv");
         break;
      case 152:
         fprintf(fPtr, "EvalCoord1fv");
         break;
      case 153:
         fprintf(fPtr, "EvalCoord2dv");
         break;
      case 154:
         fprintf(fPtr, "EvalCoord2fv");
         break;
      case 155:
         fprintf(fPtr, "EvalMesh1");
         break;
      case 156:
         fprintf(fPtr, "EvalPoint1");
         break;
      case 157:
         fprintf(fPtr, "EvalMesh2");
         break;
      case 158:
         fprintf(fPtr, "EvalPoint2");
         break;
      case 159:
         fprintf(fPtr, "AlphaFunc");
         break;
      case 160:
         fprintf(fPtr, "BlendFunc");
         break;
      case 161:
         fprintf(fPtr, "LogicOp");
         break;
      case 162:
         fprintf(fPtr, "StencilFunc");
         break;
      case 163:
         fprintf(fPtr, "StencilOp");
         break;
      case 164:
         fprintf(fPtr, "DepthFunc");
         break;
      case 165:
         fprintf(fPtr, "PixelZoom");
         break;
      case 166:
         fprintf(fPtr, "PixelTransferf");
         break;
      case 167:
         fprintf(fPtr, "PixelTransferi");
         break;
      case 168:
         fprintf(fPtr, "PixelMapf");
         break;
      case 169:
         fprintf(fPtr, "PixelMapuiv");
         break;
      case 170:
         fprintf(fPtr, "PixelMapusv");
         break;
      case 171:
         fprintf(fPtr, "ReadBuffer");
         break;
      case 172:
         fprintf(fPtr, "CopyPixels");
         break;
      case 173:
         fprintf(fPtr, "DrawPixels");
         break;
      case 174:
         fprintf(fPtr, "DepthRange");
         break;
      case 175:
         fprintf(fPtr, "Frustum");
         break;
      case 176:
         fprintf(fPtr, "LoadIdentity");
         break;
      case 177:
         fprintf(fPtr, "LoadMatrixf");
         break;
      case 178:
         fprintf(fPtr, "LoadMatrixd");
         break;
      case 179:
         fprintf(fPtr, "MatrixMode");
         break;
      case 180:
         fprintf(fPtr, "MultMatrixf");
         break;
      case 181:
         fprintf(fPtr, "MultMatrixd");
         break;
      case 182:
         fprintf(fPtr, "Ortho");
         break;
      case 183:
         fprintf(fPtr, "PopMatrix");
         break;
      case 184:
         fprintf(fPtr, "PushMatrix");
         break;
      case 185:
         fprintf(fPtr, "Rotated");
         break;
      case 186:
         fprintf(fPtr, "Rotatef");
         break;
      case 187:
         fprintf(fPtr, "Scaled");
         break;
      case 188:
         fprintf(fPtr, "Scalef");
         break;
      case 189:
         fprintf(fPtr, "Translated");
         break;
      case 190:
         fprintf(fPtr, "Translatef");
         break;
      case 191:
         fprintf(fPtr, "Viewport");
         break;
      case 193:
         fprintf(fPtr, "DrawArrays");
         break;
      case 4099:
         fprintf(fPtr, "TexSubImage1D");
         break;
      case 4100:
         fprintf(fPtr, "TexSubImage2D");
         break;
      case 4116:
         fprintf(fPtr, "DrawArrays");
         break;
      case 4117:
         fprintf(fPtr, "BindTexture");
         break;
      default:
         fprintf(fPtr, "op %d unknown or unsupported", op);
   }

   GLX_FINISH_LOG;
   return;
}

void
glx_log_glx_request(int op) {
   GLX_CHECK_LOGGING;
   GLX_CHECK_FILE;

   GLX_START_LOG;
   switch (op) {
      case 1:
         fprintf(fPtr, "Render");
         break;
      case 2:
         fprintf(fPtr, "RenderLarge");
         break;
      case 3:
         fprintf(fPtr, "CreateContext");
         break;
      case 4:
         fprintf(fPtr, "DestroyContext");
         break;
      case 5:
         fprintf(fPtr, "MakeCurrent");
         break;
      case 6:
         fprintf(fPtr, "IsDirect");
         break;
      case 7:
         fprintf(fPtr, "QueryVersion");
         break;
      case 8:
         fprintf(fPtr, "WaitGL");
         break;
      case 9:
         fprintf(fPtr, "WaitX");
         break;
      case 10:
         fprintf(fPtr, "CopyContext");
         break;
      case 11:
         fprintf(fPtr, "SwapBuffers");
         break;
      case 12:
         fprintf(fPtr, "UseXFont");
         break;
      case 13:
         fprintf(fPtr, "CreateGLXPixmap");
         break;
      case 14:
         fprintf(fPtr, "GetVisualConfigs");
         break;
      case 15:
         fprintf(fPtr, "DestroyGLXPixmap");
         break;
      case 16:
         fprintf(fPtr, "VendorPrivate");
         break;
      case 17:
         fprintf(fPtr, "VendorPrivateWithReply");
         break;
      case 19:
         fprintf(fPtr, "QueryServerString");
         break;
      case 20:
         fprintf(fPtr, "ClientInfo");
         break;
      case 101:
         fprintf(fPtr, "NewList");
         break;
      case 102:
         fprintf(fPtr, "EndList");
         break;
      case 103:
         fprintf(fPtr, "DeleteLists");
         break;
      case 104:
         fprintf(fPtr, "GenLists");
         break;
      case 105:
         fprintf(fPtr, "FeedbackBuffer");
         break;
      case 106:
         fprintf(fPtr, "SelectBuffer");
         break;
      case 107:
         fprintf(fPtr, "RenderMode");
         break;
      case 108:
         fprintf(fPtr, "Finish");
         break;
      case 109:
         fprintf(fPtr, "PixelStoref");
         break;
      case 110:
         fprintf(fPtr, "PixelStorei");
         break;
      case 111:
         fprintf(fPtr, "ReadPixels");
         break;
      case 112:
         fprintf(fPtr, "GetBooleanv");
         break;
      case 113:
         fprintf(fPtr, "GetClipPlane");
         break;
      case 114:
         fprintf(fPtr, "GetDoublev");
         break;
      case 115:
         fprintf(fPtr, "GetError");
         break;
      case 116:
         fprintf(fPtr, "GetFloatv");
         break;
      case 117:
         fprintf(fPtr, "GetIntegerv");
         break;
      case 118:
         fprintf(fPtr, "GetLightfv");
         break;
      case 119:
         fprintf(fPtr, "GetLightiv");
         break;
      case 120:
         fprintf(fPtr, "GetMapdv");
         break;
      case 121:
         fprintf(fPtr, "GetMapfv");
         break;
      case 122:
         fprintf(fPtr, "GetMapiv");
         break;
      case 123:
         fprintf(fPtr, "GetMaterialfv");
         break;
      case 124:
         fprintf(fPtr, "GetMaterialiv");
         break;
      case 125:
         fprintf(fPtr, "GetPixelMapfv");
         break;
      case 126:
         fprintf(fPtr, "GetPixelMapuiv");
         break;
      case 127:
         fprintf(fPtr, "GetPixelMapusv");
         break;
      case 128:
         fprintf(fPtr, "GetPolygonStipple");
         break;
      case 129:
         fprintf(fPtr, "GetString");
         break;
      case 130:
         fprintf(fPtr, "GetTexEnvfv");
         break;
      case 131:
         fprintf(fPtr, "GetTexEnviv");
         break;
      case 132:
         fprintf(fPtr, "GetTexGendv");
         break;
      case 133:
         fprintf(fPtr, "GetTexGenfv");
         break;
      case 134:
         fprintf(fPtr, "GetTexGeniv");
         break;
      case 135:
         fprintf(fPtr, "GetTexImage");
         break;
      case 136:
         fprintf(fPtr, "GetTexParameterfv");
         break;
      case 137:
         fprintf(fPtr, "GetTexParameteriv");
         break;
      case 138:
         fprintf(fPtr, "GetTexLevelParameterfv");
         break;
      case 139:
         fprintf(fPtr, "GetTexLevelParameteriv");
         break;
      case 140:
         fprintf(fPtr, "IsEnabled");
         break;
      case 141:
         fprintf(fPtr, "IsList");
         break;
      case 142:
         fprintf(fPtr, "Flush");
         break;
      case 143:
         fprintf(fPtr, "AreTexturesResident");
         break;
      case 144:
         fprintf(fPtr, "DeleteTextures");
         break;
      case 145:
         fprintf(fPtr, "GenTextures");
         break;
      default:
         fprintf(fPtr, "Invalid (%d)", op);
   }

   GLX_FINISH_LOG;
   return;
}

void
glx_log_enum(GLenum the_enum) {
   GLX_CHECK_LOGGING;
   GLX_CHECK_FILE;

   GLX_START_LOG;
   switch (the_enum) {

#if 0
      case GL_FALSE:
         fprintf(fPtr, "GL_FALSE\n");
         break;

      case GL_TRUE:
         fprintf(fPtr, "GL_TRUE\n");
         break;
#endif

      case GL_BYTE:
         fprintf(fPtr, "GL_BYTE\n");
         break;

      case GL_UNSIGNED_BYTE:
         fprintf(fPtr, "GL_UNSIGNED_BYTE\n");
         break;

      case GL_SHORT:
         fprintf(fPtr, "GL_SHORT\n");
         break;

      case GL_UNSIGNED_SHORT:
         fprintf(fPtr, "GL_UNSIGNED_SHORT\n");
         break;

      case GL_INT:
         fprintf(fPtr, "GL_INT\n");
         break;

      case GL_UNSIGNED_INT:
         fprintf(fPtr, "GL_UNSIGNED_INT\n");
         break;

      case GL_FLOAT:
         fprintf(fPtr, "GL_FLOAT\n");
         break;

      case GL_DOUBLE:
         fprintf(fPtr, "GL_DOUBLE\n");
         break;

      case GL_2_BYTES:
         fprintf(fPtr, "GL_2_BYTES\n");
         break;

      case GL_3_BYTES:
         fprintf(fPtr, "GL_3_BYTES\n");
         break;

      case GL_4_BYTES:
         fprintf(fPtr, "GL_4_BYTES\n");
         break;

      case GL_LINES:
         fprintf(fPtr, "GL_LINES\n");
         break;

      case GL_POINTS:
         fprintf(fPtr, "GL_POINTS\n");
         break;

      case GL_LINE_STRIP:
         fprintf(fPtr, "GL_LINE_STRIP\n");
         break;

      case GL_LINE_LOOP:
         fprintf(fPtr, "GL_LINE_LOOP\n");
         break;

      case GL_TRIANGLES:
         fprintf(fPtr, "GL_TRIANGLES\n");
         break;

      case GL_TRIANGLE_STRIP:
         fprintf(fPtr, "GL_TRIANGLE_STRIP\n");
         break;

      case GL_TRIANGLE_FAN:
         fprintf(fPtr, "GL_TRIANGLE_FAN\n");
         break;

      case GL_QUADS:
         fprintf(fPtr, "GL_QUADS\n");
         break;

      case GL_QUAD_STRIP:
         fprintf(fPtr, "GL_QUAD_STRIP\n");
         break;

      case GL_POLYGON:
         fprintf(fPtr, "GL_POLYGON\n");
         break;

      case GL_EDGE_FLAG:
         fprintf(fPtr, "GL_EDGE_FLAG\n");
         break;


      /****  skipping vertex arrays for now ****/

      case GL_MATRIX_MODE:
         fprintf(fPtr, "GL_MATRIX_MODE\n");
         break;

      case GL_PROJECTION:
         fprintf(fPtr, "GL_PROJECTION\n");
         break;

      case GL_MODELVIEW:
         fprintf(fPtr, "GL_MODELVIEW\n");
         break;

      case GL_TEXTURE:
         fprintf(fPtr, "GL_TEXTURE\n");
         break;

      /**** skipping points for now ****/


      /**** skipping lines for now ****/

      /**** polygon ****/
      case GL_POINT:
         fprintf(fPtr, "GL_POINT\n");
         break;

      case GL_LINE:
         fprintf(fPtr, "GL_LINE\n");
         break;

      case GL_FILL:
         fprintf(fPtr, "GL_FILL\n");
         break;

      case GL_CCW:
         fprintf(fPtr, "GL_CCW\n");
         break;

      case GL_CW:
         fprintf(fPtr, "GL_CW\n");
         break;

      case GL_FRONT:
         fprintf(fPtr, "GL_FRONT\n");
         break;

      case GL_BACK:
         fprintf(fPtr, "GL_BACK\n");
         break;

      case GL_CULL_FACE:
         fprintf(fPtr, "GL_CULL_FACE\n");
         break;

      case GL_CULL_FACE_MODE:
         fprintf(fPtr, "GL_CULL_FACE_MODE\n");
         break;

      case GL_POLYGON_SMOOTH:
         fprintf(fPtr, "GL_POLYGON_SMOOTH\n");
         break;

      case GL_POLYGON_STIPPLE:
         fprintf(fPtr, "GL_POLYGON_STIPPLE\n");
         break;

      case GL_FRONT_FACE:
         fprintf(fPtr, "GL_FRONT_FACE\n");
         break;

      case GL_POLYGON_MODE:
         fprintf(fPtr, "GL_POLYGON_MODE\n");
         break;

      case GL_POLYGON_OFFSET_FACTOR:
         fprintf(fPtr, "GL_POLYGON_OFFSET_FACTOR\n");
         break;

      case GL_POLYGON_OFFSET_UNITS:
         fprintf(fPtr, "GL_POLYGON_OFFSET_UNITS\n");
         break;

      case GL_POLYGON_OFFSET_POINT:
         fprintf(fPtr, "GL_POLYGON_OFFSET_POINT\n");
         break;

      case GL_POLYGON_OFFSET_LINE:
         fprintf(fPtr, "GL_POLYGON_OFFSET_LINE\n");
         break;

      case GL_POLYGON_OFFSET_FILL:
         fprintf(fPtr, "GL_POLYGON_OFFSET_FILL\n");
         break;


      /**** skipping display lists for now ****/


      case GL_NEVER:
         fprintf(fPtr, "GL_NEVER\n");
         break;

      case GL_LESS:
         fprintf(fPtr, "GL_LESS\n");
         break;

      case GL_GEQUAL:
         fprintf(fPtr, "GL_GEQUAL\n");
         break;

      case GL_LEQUAL:
         fprintf(fPtr, "GL_LEQUAL\n");
         break;

      case GL_GREATER:
         fprintf(fPtr, "GL_GREATER\n");
         break;

      case GL_NOTEQUAL:
         fprintf(fPtr, "GL_NOTEQUAL\n");
         break;

      case GL_EQUAL:
         fprintf(fPtr, "GL_EQUAL\n");
         break;

      case GL_ALWAYS:
         fprintf(fPtr, "GL_ALWAYS\n");
         break;

      case GL_DEPTH_TEST:
         fprintf(fPtr, "GL_DEPTH_TEST\n");
         break;

      case GL_DEPTH_BITS:
         fprintf(fPtr, "GL_DEPTH_BITS\n");
         break;

      case GL_DEPTH_CLEAR_VALUE:
         fprintf(fPtr, "GL_DEPTH_CLEAR_VALUE\n");
         break;

      case GL_DEPTH_FUNC:
         fprintf(fPtr, "GL_DEPTH_FUNC\n");
         break;

      case GL_DEPTH_RANGE:
         fprintf(fPtr, "GL_DEPTH_RANGE\n");
         break;

      case GL_DEPTH_WRITEMASK:
         fprintf(fPtr, "GL_DEPTH_WRITEMASK\n");
         break;

      case GL_DEPTH_COMPONENT:
         fprintf(fPtr, "GL_DEPTH_COMPONENT\n");
         break;


      /**** lighting ****/

      case GL_LIGHTING:
         fprintf(fPtr, "GL_LIGHTING\n");
         break;

      case GL_LIGHT0:
         fprintf(fPtr, "GL_LIGHT0\n");
         break;

      case GL_LIGHT1:
         fprintf(fPtr, "GL_LIGHT1\n");
         break;

      case GL_LIGHT2:
         fprintf(fPtr, "GL_LIGHT2\n");
         break;

      case GL_LIGHT3:
         fprintf(fPtr, "GL_LIGHT3\n");
         break;

      case GL_LIGHT4:
         fprintf(fPtr, "GL_LIGHT4\n");
         break;

      case GL_LIGHT5:
         fprintf(fPtr, "GL_LIGHT5\n");
         break;

      case GL_LIGHT6:
         fprintf(fPtr, "GL_LIGHT6\n");
         break;

      case GL_LIGHT7:
         fprintf(fPtr, "GL_LIGHT7\n");
         break;

      case GL_SPOT_EXPONENT:
         fprintf(fPtr, "GL_SPOT_EXPONENT\n");
         break;

      case GL_SPOT_CUTOFF:
         fprintf(fPtr, "GL_SPOT_CUTOFF\n");
         break;

      case GL_CONSTANT_ATTENUATION:
         fprintf(fPtr, "GL_CONSTANT_ATTENUATION\n");
         break;

      case GL_LINEAR_ATTENUATION:
         fprintf(fPtr, "GL_LINEAR_ATTENUATION\n");
         break;

      case GL_QUADRATIC_ATTENUATION:
         fprintf(fPtr, "GL_QUADRATIC_ATTENUATION\n");
         break;

      case GL_AMBIENT:
         fprintf(fPtr, "GL_AMBIENT\n");
         break;

      case GL_DIFFUSE:
         fprintf(fPtr, "GL_DIFFUSE\n");
         break;

      case GL_SPECULAR:
         fprintf(fPtr, "GL_SPECULAR\n");
         break;

      case GL_SHININESS:
         fprintf(fPtr, "GL_SHININESS\n");
         break;

      case GL_EMISSION:
         fprintf(fPtr, "GL_EMISSION\n");
         break;

      case GL_POSITION:
         fprintf(fPtr, "GL_POSITION\n");
         break;

      case GL_SPOT_DIRECTION:
         fprintf(fPtr, "GL_SPOT_DIRECTION\n");
         break;

      case GL_AMBIENT_AND_DIFFUSE:
         fprintf(fPtr, "GL_AMBIENT_AND_DIFFUSE\n");
         break;

      case GL_COLOR_INDEXES:
         fprintf(fPtr, "GL_COLOR_INDEXES\n");
         break;

      case GL_LIGHT_MODEL_TWO_SIDE:
         fprintf(fPtr, "GL_LIGHT_MODEL_TWO_SIDE\n");
         break;

      case GL_LIGHT_MODEL_LOCAL_VIEWER:
         fprintf(fPtr, "GL_LIGHT_MODEL_LOCAL_VIEWER\n");
         break;

      case GL_LIGHT_MODEL_AMBIENT:
         fprintf(fPtr, "GL_LIGHT_MODEL_AMBIENT\n");
         break;

      case GL_FRONT_AND_BACK:
         fprintf(fPtr, "GL_FRONT_AND_BACK\n");
         break;

      case GL_SHADE_MODEL:
         fprintf(fPtr, "GL_SHADE_MODEL\n");
         break;

      case GL_FLAT:
         fprintf(fPtr, "GL_FLAT\n");
         break;

      case GL_SMOOTH:
         fprintf(fPtr, "GL_SMOOTH\n");
         break;

      case GL_COLOR_MATERIAL:
         fprintf(fPtr, "GL_COLOR_MATERIAL\n");
         break;

      case GL_COLOR_MATERIAL_FACE:
         fprintf(fPtr, "GL_COLOR_MATERIAL_FACE\n");
         break;

      case GL_COLOR_MATERIAL_PARAMETER:
         fprintf(fPtr, "GL_COLOR_MATERIAL_PARAMETER\n");
         break;

      case GL_NORMALIZE:
         fprintf(fPtr, "GL_NORMALIZE\n");
         break;

      /** buffers **/

      case GL_RGB:
         fprintf(fPtr, "GL_RGB\n");
         break;

      case GL_RGBA:
         fprintf(fPtr, "GL_RGBA\n");
         break;

      /** utility enumerations **/

      case GL_VENDOR:
         fprintf(fPtr, "GL_VENDOR\n");
         break;

      case GL_RENDERER:
         fprintf(fPtr, "GL_RENDERER\n");
         break;

      case GL_VERSION:
         fprintf(fPtr, "GL_VERSION\n");
         break;

      case GL_EXTENSIONS:
         fprintf(fPtr, "GL_EXTENSIONS\n");
         break;

      default:
         fprintf(fPtr, "unhandled\n");

   }

   GLX_FINISH_LOG;
   return;
}


void
glx_log_mask(GLenum mask) {
   int temp = 0;
   GLX_CHECK_LOGGING;
   GLX_CHECK_FILE;

   GLX_START_LOG;
   if (mask & GL_ALL_ATTRIB_BITS) {
       fprintf(fPtr, " all bits");
       mask = 0;
   }

   temp = 0x001;
   while (mask) {
      switch (mask & temp) {
         case GL_CURRENT_BIT:
            fprintf(fPtr, " gl_current_bit");
            mask = mask & ~GL_CURRENT_BIT;
            break;
         case GL_POINT_BIT:
            fprintf(fPtr, " gl_point_bit");
            mask = mask & ~GL_POINT_BIT;
            break;
         case GL_LINE_BIT:
            fprintf(fPtr, " gl_line_bit");
            mask = mask & ~GL_LINE_BIT;
            break;
         case GL_POLYGON_BIT:
            fprintf(fPtr, " gl_polygon_bit");
            mask = mask & ~GL_POLYGON_BIT;
            break;
         case GL_POLYGON_STIPPLE_BIT:
            fprintf(fPtr, " gl_polygon_stipple_bit");
            mask = mask & ~GL_POLYGON_STIPPLE_BIT;
            break;
         case GL_PIXEL_MODE_BIT:
            fprintf(fPtr, " gl_pixel_mode_bit");
            mask = mask & ~GL_PIXEL_MODE_BIT;
            break;
         case GL_LIGHTING_BIT:
            fprintf(fPtr, " gl_lighting_bit");
            mask = mask & ~GL_LIGHTING_BIT;
            break;
         case GL_FOG_BIT:
            fprintf(fPtr, " gl_fog_bit");
            mask = mask & ~GL_FOG_BIT;
            break;
         case GL_DEPTH_BUFFER_BIT:
            fprintf(fPtr, " gl_depth_buffer_bit");
            mask = mask & ~GL_DEPTH_BUFFER_BIT;
            break;
         case GL_ACCUM_BUFFER_BIT:
            fprintf(fPtr, " gl_accum_buffer_bit");
            mask = mask & ~GL_ACCUM_BUFFER_BIT;
            break;
         case GL_STENCIL_BUFFER_BIT:
            fprintf(fPtr, " gl_stencil_buffer_bit");
            mask = mask & ~GL_STENCIL_BUFFER_BIT;
            break;
         case GL_VIEWPORT_BIT:
            fprintf(fPtr, " gl_viewport_bit");
            mask = mask & ~GL_VIEWPORT_BIT;
            break;
         case GL_TRANSFORM_BIT:
            fprintf(fPtr, " gl_transform_bit");
            mask = mask & ~GL_TRANSFORM_BIT;
            break;
         case GL_ENABLE_BIT:
            fprintf(fPtr, " gl_enable_bit");
            mask = mask & ~GL_ENABLE_BIT;
            break;
         case GL_COLOR_BUFFER_BIT:
            fprintf(fPtr, " gl_color_buffer_bit");
            mask = mask & ~GL_COLOR_BUFFER_BIT;
            break;
         case GL_HINT_BIT:
            fprintf(fPtr, " gl_hint_bit");
            mask = mask & ~GL_HINT_BIT;
            break;
         case GL_EVAL_BIT:
            fprintf(fPtr, " gl_eval_bit");
            mask = mask & ~GL_EVAL_BIT;
            break;
         case GL_LIST_BIT:
            fprintf(fPtr, " gl_list_bit");
            mask = mask & ~GL_LIST_BIT;
            break;
         case GL_TEXTURE_BIT:
            fprintf(fPtr, " gl_texture_bit");
            mask = mask & ~GL_TEXTURE_BIT;
            break;
         case GL_SCISSOR_BIT:
            fprintf(fPtr, " gl_scissor_bit");
            mask = mask & ~GL_SCISSOR_BIT;
            break;
         default:
            fprintf(fPtr, " unknown");
            mask = 0;
            break;
      }
      temp = temp << 1;
   }

   fprintf(fPtr, "\n");

   GLX_FINISH_LOG;
   return;
}

