/*
 * Functions for parsing a simple text file.
 * Copyright (C) 1999 Andreas Ehliar <ehliar@lysator.liu.se>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * ANDREAS EHLIAR, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR 
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE 
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 */


#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <ctype.h>

#include "glx_config.h"

static int parseconfig(char *string);
static int addvar(char *varname,char *value,int lineno);
static char *nextline(char *string);
static char *endtoken(char *string);
static char *skip_ws(char *string);


struct config{
  struct config *next;
  char *varname;
  char *value;
};

static int conflength;
static struct config *confvars=NULL;
static int override; /* The user may override variables considered safe */
static int insecure_override; /* The user may override DMA address configuration. */

/* Open and read the configuration file
 *
 * If the configuration file couldn't be read at all 0 will be returned
 * An error occuring while parsing the file is fatal though.
 */

int glx_readconf(char *filename)
{
  char *confstring=NULL;
  struct stat st;
  int conffd;
  int result;
  int error;


  conffd=open(filename,O_RDONLY);
  if(conffd == -1){
    fprintf(stderr,"Cannot open %s: %s\n",filename,strerror(errno));
    return CONFIG_NOTREAD;
  }

  fstat(conffd,&st);
  conflength=st.st_size;

  confstring = (char *) malloc(conflength+1);
  if(!confstring){
    fprintf(stderr,"Out of memory while reading config file\n");
    return CONFIG_FATAL;
  }

  confstring[conflength] = 0; /* You will lose if there is already a NUL
			       * in the config file. */

  result=read(conffd,confstring,conflength);
  if(result == -1){
    fprintf(stderr,"An error occured while reading the config file: %s\n",strerror(errno));
    free(confstring);
    close(conffd);
    confstring=NULL;
    return CONFIG_FATAL;
  }


  error=parseconfig(confstring);

  free(confstring);
  close(conffd);

  return error;
}


/* Add a variable to confvars */
static int addvar(char *varname,char *value,int lineno)
{
  struct config *tmp;

  tmp=confvars;

  while(tmp){
    if(!strcasecmp(varname,tmp->varname)){
      fprintf(stderr,"Error at line %d in config file: %s already defined\n",lineno,varname);
      return CONFIG_FATAL;
    }
    tmp=tmp->next;
  }
  
  tmp=malloc(sizeof(struct config));
  if(!tmp){
    /* If you don't have enough memory for a struct config
     * you have no business running the glx module anyway :) */
    fprintf(stderr,"Out of memory while reading config file\n");
    return CONFIG_FATAL;
  }

  tmp->varname=strdup(varname);
  tmp->value=strdup(value);

  if(tmp->varname && tmp->value){
    if(!confvars){
      confvars=tmp;
      tmp->next=NULL;
    }else{
      tmp->next=confvars;
      confvars=tmp;
    }
  }else{
    fprintf(stderr,"Out of memory while reading config file\n");
    return CONFIG_FATAL;
  }

  /* Check for options used by glx_config.c: */
  if(!strcasecmp(varname,"allow_override")){
    override=atoi(value);
  }else if(!strcasecmp(varname,"allow_insecure_override")){
    insecure_override=atoi(value);
  }
  return CONFIG_NOERR;

}

/* Helper functions for the parser */
static char *nextline(char *string)
{
  while(*string != '\n' && *string != 0){


    string++;
  }

  if(*string == '\n'){
    string++;
  }

  return string;
}


static char *skip_ws(char *string)
{
  while(*string == ' ' || *string == '\t'){
    string++;
  }

  return string;
}


static char *endtoken(char *string)
{
  while(*string != 0
	&& *string != ' '
	&& *string != '\t'
	&& *string != '='
	&& *string != '#'
	&& *string != '\n'){
    string++;
  }
  return string;
}

/* This will add every variable assignment to confvars
 *
 * A valid line looks like this:
 * varname = value # optional comment
 * # comment
 *
 * Blank lines are also accepted
 *
 * C is not the optimal language for text file parsing...
 * (This function is quite messy)
 *
 * Note: This function will change the contents of string
 */

static int parseconfig(char *string)
{
  char *currpos;
  char *varend;
  char *varbegin;
  char *valueend;
  char *valuebegin;
  int lineno = 1;
  int error;


  if(!string){
    return CONFIG_NOERR;
  }

  currpos=string;

  while(*currpos != 0){


    /* First well skip lines that are blank or consists of only
     * a comment */
    while(*currpos == ' '
	  || *currpos == '\t'
	  || *currpos == '\n'
	  || *currpos == '#')
      {
	switch(*currpos){
	case '#':
	  currpos=nextline(currpos);
	  lineno++;
	  break;
	case '\n':
	  lineno++;
	default:
	  currpos++;
	}
      }
    

    if(*currpos == 0){
      return CONFIG_NOERR;
    }

    varbegin=currpos;

    /* get the variable name */

    if(*varbegin == '='){
      fprintf(stderr,"Missing variable name at line %d in config file\n",lineno);
      return CONFIG_FATAL;
    }

    varend=endtoken(varbegin);

    currpos=varend;

    currpos=skip_ws(currpos);

    if(*currpos == 0 || *currpos == '\n' || *currpos == '#'){
      fprintf(stderr,"Unterminated statement on line %d in config file\n",lineno);
      return CONFIG_FATAL;
    }
    
    if(*currpos != '='){
      fprintf(stderr,"Syntax error at line %d in config file\n",lineno);
      return CONFIG_FATAL;
    }


    /* Get the value */

    valuebegin=currpos+1;

    valuebegin=skip_ws(valuebegin);

    if(*valuebegin == '#' || *valuebegin == 0 || *valuebegin == '\n'){
      fprintf(stderr,"Value expected at line %d in config file\n",lineno);
      return CONFIG_FATAL;
    }


    valueend=endtoken(valuebegin);

    if(*valueend == '='){
      fprintf(stderr,"Unexpected = at line %d in config file\n",lineno);
      return CONFIG_FATAL;
    }

    currpos=valueend;
    /* Note: We can't have ' ' or '\t' in the variable value right now,
     * do we need that? */

    while(*currpos  == ' ' || *currpos == '\t'){
      /* Scan to end of line */
      currpos++;
    }

    if(*currpos != '#' && *currpos != '\n' && *currpos != 0){
      fprintf(stderr,"Unexpected character(s) after value at line %d in config file\n",lineno);
      return CONFIG_FATAL;
    }

    if(*currpos == '#'){
      currpos=nextline(currpos);
    }else if(*currpos != 0){
      /* Make sure we don't terminate as a result of *valueend = 0; */
      currpos++;
    }


    *valueend=0;
    *varend=0;

    error=addvar(varbegin,valuebegin,lineno);
    if(error != CONFIG_NOERR){
      return CONFIG_FATAL;
    }
    lineno++;


  }

  return CONFIG_NOERR;
}

#if 0
/* For debugging usage */
static void printvars(void)
{
  struct config *tmp;
  tmp=confvars;
  while(tmp){
    fprintf(stderr,"%s = %s\n",tmp->varname,tmp->value);
    tmp=tmp->next;
  }
}
#endif

void glx_freeconfig(void)
{
  struct config *tmp;
  while(confvars){
    free(confvars->varname);
    free(confvars->value);

    tmp=confvars->next;
    free(confvars);
    confvars=tmp;
  }
  confvars=NULL; /* just to be safe... */
}

char *glx_getvar(const char *varname)
{
  struct config *tmp=confvars;

  if(override){
    /* I think I got all of the extremely dangerous variables here */
    if((strcasecmp("mga_logfile",varname) &&
	strcasecmp("mga_dma",varname) &&
	strcasecmp("mga_dmaadr",varname) &&
	strcasecmp("mga_dmasize",varname)) ||
       insecure_override){
      
      const char *orig=varname;
      char *tmp;
      char *result;
      char *envname;

      tmp=malloc(strlen(varname)+5);
      if(!tmp){
	fprintf(stderr,"Couldn't allocate memory in glx_getvar()\n");
	return NULL;
      }
      envname=tmp;

      tmp[0]='G';
      tmp[1]='L';
      tmp[2]='X';
      tmp[3]='_';
      tmp+=4;

      /* Convert varname to all uppercase for comparison with
       * environment */
      while(*orig){
	*tmp=toupper(*orig);
	tmp++;
	orig++;
      }
      *tmp=0;

      result=getenv(envname);
      free(envname);
      if(result){
	return result;
      }
    }
  }


  while(tmp){
    if(!strcasecmp(varname,tmp->varname)){
      return tmp->value;
    }
    tmp=tmp->next;
  }
  return NULL;
}


int glx_getint(const char *varname){
  char *tmp;

  tmp=glx_getvar(varname);

  if(!tmp){
    return 0;
  }

  return atoi(tmp);
}



