/* $Id: mach64glx.h,v 1.7 1999/12/20 20:16:54 johnc Exp $ */

/*
 * GLX Hardware Device Driver for ATI Rage Pro
 * Copyright (C) 1999 Gareth Hughes
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Based on MGA driver: mgalib.h 1.7
 *
 *    Gareth Hughes <garethh@bell-labs.com>
 */

#ifndef __MACH64_LIB_H__
#define __MACH64_LIB_H__

#include <stdio.h>

#include "types.h"		/* MESA types */
#ifndef MESA31
#error "The Mach64 driver now requires Mesa 3.1 or higher"
#endif

#include "xsmesaP.h"	/* GLX header */
extern int __glx_is_server;
extern void xsmesa_setup_DD_pointers( GLcontext *ctx );

#include "rage_reg.h"

#include "mach64log.h"
#include "mm.h"

#define MACH64_MAXFIFO		64

typedef unsigned int mach64UI32;
typedef unsigned short mach64UI16;
typedef unsigned char mach64UI8;
typedef int mach64I32;
typedef short mach64I16;
typedef char mach64I8;

typedef mach64UI8 mach64Color[4];


/*
 * main buffer structure for drawing destination
 */
#define mach64BufferMagic		0x65e813a1
#define VALID_MACH64_BUFFER( b )	( (b) && ( (b)->magic == mach64BufferMagic ) )

struct mach64_buffer_t
{
    mach64UI32	magic;
    struct mach64_buffer_t *next;

    void	*backBuffer;	/* pointer to buffer in VM */
    PMemBlock	backBufferBlock; /* location in card's space, NULL if software only */

    void	*depthBuffer;
    PMemBlock	depthBufferBlock;

    int		width, height;	/* in pixels */
    int		pitch;			/* in pixels */
};
typedef struct mach64_buffer_t mach64Buffer;
typedef struct mach64_buffer_t *mach64BufferPtr;

/*
 * main hardware context structure
 */
#define mach64ContextMagic		0x47323030
#define VALID_MACH64_CONTEXT( ctx )	( (ctx) && ( (ctx)->magic == mach64ContextMagic ) )

/* for SoftwareFallback */
#define FALLBACK_ALPHATEST	1
#define FALLBACK_BLENDING	2
#define FALLBACK_TEXTURE	4



/* For mach64Ctx->new_state.
 */
#define MACH64_NEW_DEPTH	0x1
#define MACH64_NEW_ALPHA	0x2
#define MACH64_NEW_FOG		0x4
#define MACH64_NEW_CLIP		0x8
#define MACH64_NEW_MASK		0x10
#define MACH64_NEW_TEXTURE	0x20
#define MACH64_NEW_CULL		0x40
#define MACH64_NEW_CONTEXT	0x100

struct mach64_context_t
{
    mach64UI32		magic;
    int			refcount;

    struct mach64_buffer_t *DB;
    GLcontext		*gl_ctx;

    int			SoftwareFallback;  /* or'ed values of FALLBACK_* */

    /* Support for CVA and the fast paths */
    unsigned int	setupdone;
    unsigned int	setupindex;
    unsigned int	renderindex;
    unsigned int	using_fast_path;
    unsigned int	using_immediate_fast_path;

    /* Shortcircuit some state changes */
    points_func		PointsFunc;
    line_func		LineFunc;
    triangle_func	TriangleFunc;
    quad_func		QuadFunc;

    /* Manage our own state */
    GLuint		new_state;
    GLuint		reg_dirty;

    GLfloat		map[16];
};

#define MACH64_CONTEXT(ctx)	((mach64ContextPtr)(((XSMesaContext)(ctx)->DriverCtx)->hw_ctx))


struct mach64_context_t;
typedef struct mach64_context_t mach64Context;
typedef struct mach64_context_t *mach64ContextPtr;


/*
 * texture object
 */
#define MACH64_TEXTURE_OBJECT_MAGIC 0x9a0b3fd2

typedef struct mach64_texture_object_s {
	mach64UI32	magic;
	struct mach64_texture_object_s *next;	/* driver global chain of textures */
	struct gl_texture_object *tObj;		/* mesa texture */
	mach64ContextPtr ctx;

	PMemBlock	memBlock;				/* buffer for all miplevels */
	mach64UI32	totalSize;				/* for adding to c_textureUtilization */

	int		texelBytes;

	int		widthLog2;
	int		heightLog2;
	int		maxLog2;

	int		hasAlpha;

	mach64UI32 	age;   				/* set to mgaglx.swapBufferCount when bound */
} mach64TextureObject_t;

typedef mach64TextureObject_t *mach64TextureObjectPtr;



/*
 * main driver variables structure
 */
typedef struct
{
	mach64Buffer			*bufferList;
	mach64TextureObject_t	*textureList;

	mach64UI32	dmaActive;

	struct _dma_buffer	*dma_buffer;	/* dmaBuffers[ activeDmaBuffer ] */

	mach64UI32	activeDmaBuffer;


    /* logging stuff */
    mach64UI32		logLevel;
    FILE		*logFile;

    /* dma stuff */
    mach64UI32		dmaDriver;
    mach64UI32		dmaSize;
    mach64UI32		dmaAdr;
    mach64UI32		cmdSize;
    mach64UI32		cardCmds;
    mach64UI32		systemTexture;
    mach64UI32		tableSize;

	/* driver register state */
#if 0 /* FIXME: GH */
    /* bookkeeping for textureing */
    struct mach64_texture_object_s *TexObjList;
    struct mach64_texture_object_s *currentTextureObject[2];
    mach64UI16 		*currentTexturePalette;
    mach64UI32		deferTextureLoads;
    mach64UI32		default32BitTextures;
    mach64UI32		swapBuffersCount;
    struct gl_texture_object *currentMesaTexture;
#endif

    /* options */
    mach64UI32		nullprims;	/* skip all primitive generation */
    mach64UI32		boxes;		/* draw performance boxes */
    mach64UI32		noFallback;	/* don't fall back to software, do best-effort rendering */
    mach64UI32		nosgram;	/* don't block clears */
    mach64UI32		skipDma;	/* don't send anything to the hardware */

    /* performance counters */
    mach64UI32		swapBuffersCount;	/* also used for texture thrash checking */

    mach64UI32		c_textureUtilization;
    mach64UI32		c_textureSwaps;
    mach64UI32		c_setupPointers;
    mach64UI32		c_triangles;
    mach64UI32		c_drawWaits;
    mach64UI32		c_signals;
    mach64UI32		c_dmaFlush;
    mach64UI32		c_overflows;

    mach64UI32		hardwareWentIdle; /* cleared each swapbuffers, set if
					     a waitfordmacompletion ever exited
					     without having to wait */

    mach64UI32		timeTemp;	/* for time counting in logfiles */

    /* stuff we get from the apropriate fbdev/mach64info struct */
    mach64UI32		depth;		/* should be 15 or 16 */
    mach64UI32		virtualX;
    mach64UI32		virtualY;
    mach64UI32		displayWidth;
    mach64UI32		videoRam;
    mach64UI32		linearBase;
    mach64UI32		MMIOBase;
    mach64UI32		bytesPerPixel;		/* should be 2 or 4 */

    mach64UI32		registers[0x200];	/* read at startup */
} mach64Glx_t;


extern int			xf86VTSema;
extern XSMesaContext	XSMesa;
extern mach64Glx_t		mach64glx;
extern mach64BufferPtr	mach64DB;
extern mach64ContextPtr	mach64Ctx;


#define MACH64PACKCOLOR555( r, g, b, a )				\
    ((((r) & 0xf8) << 7) | (((g) & 0xf8) << 2) | (((b) & 0xf8) >> 3) |	\
     ((a) ? 0x8000 : 0))

#define MACH64PACKCOLOR565( r, g, b )					\
    ((((r) & 0xf8) << 8) | (((g) & 0xfc) << 3) | (((b) & 0xf8) >> 3))

#define MACH64PACKCOLOR888( r, g, b )					\
    (((r) << 16) | ((g) << 8) | (b))

#define MACH64PACKCOLOR8888( r, g, b, a )				\
    (((a) << 24) | ((r) << 16) | ((g) << 8) | (b))

#define MACH64PACKCOLOR4444( r, g, b, a )				\
    ((((a) & 0xf0) << 8) | (((r) & 0xf0) << 4) | ((g) & 0xf0) | ((b) >> 4))

static __inline__ mach64UI32 mach64PackColor( mach64UI8 r, mach64UI8 g,
					      mach64UI8 b, mach64UI8 a )
{
#if 0	// JDC: FIXME
    switch ( mach64DB->Attrib & MACH64_PF_MASK )
    {
    case MACH64_PF_555:
	return MACH64PACKCOLOR555( r, g, b, a );
    case MACH64_PF_565:
	return MACH64PACKCOLOR565( r, g, b );
    case MACH64_PF_888:
	return MACH64PACKCOLOR888( r, g, b );
    case MACH64_PF_8888:
	return MACH64PACKCOLOR8888( r, g, b, a );
    default:
	return 0;
    }
#else
	return MACH64PACKCOLOR555( r, g, b, a );
#endif

}

/* ===== mach64glx.c ===== */

void (*mach64GLXSwapBuffers)( XSMesaBuffer b );


mach64ContextPtr mach64CreateContext( GLcontext *ctx );
int mach64DestroyContext( mach64ContextPtr ctx );

int usec( void );
void mach64LibInit();
void mach64SoftReset();

int mach64MakeCurrent( mach64ContextPtr ctx, mach64BufferPtr buf );


void mach64GLXCreateDepthBuffer( GLcontext* ctx );
void mach64GLXDestroyImage( GLXImage* image );
GLXImage* mach64GLXCreateImage( WindowPtr pwindow, int depth,
				int width, int height, GLXImage *old_image );
GLboolean mach64GLXMakeCurrent( XSMesaContext c );
GLboolean mach64GLXBindBuffer( XSMesaContext c, XSMesaBuffer b );
XSMesaContext mach64GLXCreateContext( XSMesaVisual v,
				      XSMesaContext share_list );
void mach64GLXDestroyContext( XSMesaContext c );


/* ===== mach64clear.c ===== */

GLbitfield mach64Clear( GLcontext *ctx, GLbitfield mask, GLboolean all,
			    GLint x, GLint y, GLint width, GLint height );


/* ===== mach64span.c ===== */

void mach64DDInitSpans( GLcontext *ctx );


/* ===== mach64swap.c ===== */

void mach64ServerSwapBuffers( XSMesaBuffer b );


/* ===== mach64dd.c ===== */

void mach64_setup_DD_pointers( GLcontext *ctx );
void mach64DDExtensionsInit( GLcontext *ctx );


/* ===== mach64state.c ===== */

void mach64DDInitStatePointers( GLcontext *ctx );
void mach64DDUpdateState( GLcontext *ctx );


/* ===== mach64tri.c ===== */

void mach64Triangle( GLcontext *ctx, GLuint e0, GLuint e1, GLuint e2, GLuint pv );


/* ===== mach64tex.c ===== */

void mach64CreateTexObj( mach64ContextPtr ctx, struct gl_texture_object *tObj );
GLboolean mach64IsTextureResident( GLcontext *ctx, struct gl_texture_object *tObj );
void mach64DeleteTexture( GLcontext *ctx, struct gl_texture_object *tObj );
void mach64TexSubImage( GLcontext *ctx, GLenum target,
		  struct gl_texture_object *tObj, GLint level,
		  GLint xoffset, GLint yoffset,
		  GLsizei width, GLsizei height,
		  GLint internalFormat,
		  const struct gl_texture_image *image );
void mach64TexImage( GLcontext *ctx, GLenum target,
		  struct gl_texture_object *tObj, GLint level,
		  GLint internalFormat,
		  const struct gl_texture_image *image );


/* ===== mach64dma.c / mach64dmainit.c ===== */

#include "mach64dma.h"


/* ===== mach64direct.c ===== */

// declared in serverglx/glxvisual, will be 0 if we are
// a direct rendering client
extern int __glx_is_server;

int mach64GLXVendorPrivate( ClientPtr client,
				XSMesaContext ctx, 
				xGLXVendorPrivateReq *stuff );


int mach64GLXAllowDirect( ClientPtr client );

void mach64DirectClientSwapBuffers( XSMesaBuffer b ); 

void mach64DirectClientDMAFlush( int wait ); 

#endif


/*
 * Local Variables:
 * mode: c
 * c-basic-offset: 4
 * End:
 */
