/* xsmesa1.c - originally was xmesa1.c, but has been changed for use
 * within the Xserver
 */


/*
 * Mesa 3-D graphics library
 * Version:  2.5
 * Copyright (C) 1995-1996  Brian Paul
 *
 * GLX Server Extension
 * Copyright (C) 1996  Steven G. Parker  (sparker@cs.utah.edu)
 * Copyright (C) 1998, 1999 Terence Ripperda (ripperda@sgi.com)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * STEVEN PARKER, TERENCE RIPPERDA, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR 
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, 
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE 
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */



/*
 * Mesa/X11 interface, part 1.
 *
 * This file contains the implementations of all the XMesa* functions.
 *
 *
 * NOTES:
 *
 * The window coordinate system origin (0,0) is in the lower-left corner
 * of the window.  X11's window coordinate origin is in the upper-left
 * corner of the window.  Therefore, most drawing functions in this
 * file have to flip Y coordinates.
 *
 * Byte swapping:  If the Mesa host and the X display use a different
 * byte order then there's some trickiness to be aware of when using
 * XImages.  The byte ordering used for the XImage is that of the X
 * display, not the Mesa host.
 *
 * The color-to-pixel encoding for True/DirectColor must be done
 * according to the display's visual red_mask, green_mask, and blue_mask.
 * If XPutPixel is used to put a pixel into an XImage then XPutPixel will
 * do byte swapping if needed.  If one wants to directly "poke" the pixel
 * into the XImage's buffer then the pixel must be byte swapped first.  In
 * Mesa, when byte swapping is needed we use the PF_TRUECOLOR pixel format
 * and use XPutPixel everywhere except in the implementation of
 * glClear(GL_COLOR_BUFFER_BIT).  We want this function to be fast so
 * instead of using XPutPixel we "poke" our values after byte-swapping
 * the clear pixel value if needed.
 *
 */

#include <assert.h>
#include <ctype.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "xsmesaP.h"
#include "xsmdither.h"
#include "mesaglx/context.h"
#include "mesaglx/macros.h"
#include "mesaglx/matrix.h"
#include "mesaglx/types.h"

#include "glx_log.h"

#define GC XXGC
#include "gcstruct.h"
#include "pixmapstr.h"
#include "windowstr.h"
#include "scrnintstr.h"
#undef GC

#ifdef XFree86LOADER
    #include "xf86_libc.h"
#endif

/*
 * Current X/Mesa context pointer:
 */
XSMesaContext XSMesa = NULL;

/**********************************************************************/
/*****                     X Utility Functions                    *****/
/**********************************************************************/

/*
 * X/Mesa error reporting function:
 */
static void error( const char *msg )
{
    if (getenv("MESA_DEBUG"))
        fprintf( stderr, "X/Mesa error: %s\n", msg );
}

/*
 * Error handling.
 */
static int mesaXErrorFlag = 0;

static int mesaHandleXError( Display *dpy, XErrorEvent *event )
{
    mesaXErrorFlag = 1;
    return(0);
}

/*
 * Apply gamma correction to an intensity value in [0..max].  Return the
 * new intensity value.
 */
static GLint gamma_adjust( GLfloat gamma, GLint value, GLint max )
{
    double x = (double) value / (double) max;
    return(GLint) ((GLfloat) max * pow( x, 1.0F/gamma ) );
}

/*
 * Determine if a given X window ID is valid (window exists).
 * Do this by calling XGetWindowAttributes() for the window and
 * checking if we catch an X error.
 * Input:  dpy - the display
 *         win - the window to check for existance
 * Return:  GL_TRUE - window exists
 *          GL_FALSE - window doesn't exist
 */
static GLboolean WindowExistsFlag;

static int window_exists_err_handler( Display* dpy, XErrorEvent* xerr )
{
    if (xerr->error_code == BadWindow)
    {
        WindowExistsFlag = GL_FALSE;
    }
    return(0);
}

static GLboolean window_exists( Display *dpy, Window win )
{
#if 0
    /* I don't know what to do with this */
    XWindowAttributes wa;
    int (*old_handler)( Display*, XErrorEvent* );
    WindowExistsFlag = GL_TRUE;
    old_handler = XSetErrorHandler(window_exists_err_handler);
    XGetWindowAttributes( dpy, win, &wa ); /* dummy request */
    XSetErrorHandler(old_handler);
    return(WindowExistsFlag);
#endif
    return(GL_FALSE);
}

/**********************************************************************/
/*****                Linked list of XSMesaBuffers                 *****/
/**********************************************************************/

static XSMesaBuffer XSMesaBufferList = NULL;

/* Allocate a new XMesaBuffer, add to linked list */
static XSMesaBuffer alloc_xsmesa_buffer(void)
{
    XSMesaBuffer b = (XSMesaBuffer) calloc( 1, sizeof(struct xsmesa_buffer) );
    if (b)
    {
        b->Next = XSMesaBufferList;
        XSMesaBufferList = b;
    }
    return(b);
}

/*
 * Find an XSMesaBuffer by matching X display and colormap but NOT matching
 * the notThis buffer.
 */
static XSMesaBuffer find_xsmesa_buffer(ScreenPtr screen, ColormapPtr cmap,
                                       XSMesaBuffer notThis)
{
    XSMesaBuffer b;
    for (b=XSMesaBufferList; b; b=b->Next)
    {
        if (b->xsm_visual->pScreen==screen && b->cmap==cmap && b!=notThis)
        {
            return(b);
        }
    }
    return(NULL);
}

/*
 * Free an XSMesaBuffer, remove from linked list, perhaps free X colormap
 * entries.
 */
static void free_xsmesa_buffer(XSMesaBuffer buffer)
{
    XSMesaBuffer prev = NULL, b;
    for (b=XSMesaBufferList; b; b=b->Next)
    {
        if (b==buffer)
        {
            /* unlink bufer from list */
            if (prev)
                prev->Next = buffer->Next;
            else
                XSMesaBufferList = buffer->Next;
            /* Check to free X colors */
            if (buffer->num_alloced>0)
            {
                /* If no other buffer uses this X colormap then free the colors. */
                if (!find_xsmesa_buffer(buffer->xsm_visual->pScreen, buffer->cmap, buffer))
                {
                    /*
                    XFreeColors(buffer->display, buffer->cmap,
                    buffer->alloced_colors, buffer->num_alloced, 0);
                    */
                }
            }
            free(buffer);
            return;
        }
        /* continue search */
        prev = b;
    }
    /* buffer not found in XSMesaBufferList */
    gl_problem(NULL,"free_xsmesa_buffer() - buffer not found\n");
}


/* Copy X color table stuff from on XSMesaBuffer to another. */
static void copy_colortable_info(XSMesaBuffer dst, const XSMesaBuffer src)
{
    MEMCPY(dst->color_table, src->color_table, sizeof(src->color_table));
    MEMCPY(dst->pixel_to_r, src->pixel_to_r, sizeof(src->pixel_to_r));
    MEMCPY(dst->pixel_to_g, src->pixel_to_g, sizeof(src->pixel_to_g));
    MEMCPY(dst->pixel_to_b, src->pixel_to_b, sizeof(src->pixel_to_b));
    dst->num_alloced = src->num_alloced;
    MEMCPY(dst->alloced_colors, src->alloced_colors,
           sizeof(src->alloced_colors));
}

/**********************************************************************/
/*****                   Misc Private Functions                   *****/
/**********************************************************************/

/*
 * Return number of bits set in n.
 */
static int bitcount( unsigned long n )
{
    int bits;
    for (bits=0; n>0; n=n>>1)
    {
        if (n&1)
        {
            bits++;
        }
    }
    return(bits);
}

/*
 * Setup an off-screen pixmap or Ximage to use as the back buffer.
 * Input:  b - the X/Mesa buffer
 */
void xsmesa_alloc_back_buffer( XSMesaBuffer b )
{
    ScreenPtr screen = b->xsm_visual->pScreen;
    if (b->db_state==BACK_XIMAGE)
    {
        /* Allocate new back buffer :
	 * (also deallocate the old backimage, if any) */
        b->backimage = GLXProcs.CreateImage(b->pixmap_flag ? (WindowPtr)NULL : (WindowPtr)b->frontbuffer,
                                            b->xsm_visual->pVisual->nplanes, 
                                            b->width, 
                                            b->height,
					    b->backimage);
        if (!b->backimage)
            error("alloc_back_buffer: XCreateImage failed.");
      
        b->backpixmap = None;
    }
    else if (b->db_state==BACK_PIXMAP)
    {
        PixmapPtr old_pixmap = b->backpixmap;
        /* Free the old back pixmap */
        if (b->backpixmap)
            (*screen->DestroyPixmap)( b->backpixmap );
        /* Allocate new back pixmap */
        b->backpixmap = (*screen->CreatePixmap)(screen,
                                                b->width, b->height,
                                                b->xsm_visual->pVisual->nplanes);
        /* update other references to backpixmap */
        if (b->buffer==(DrawablePtr)old_pixmap)
            b->buffer = (DrawablePtr)b->backpixmap;
        
        b->backimage = NULL;
    }
}

/*
 * A replacement for XAllocColor.  This function should never
 * fail to allocate a color.  When XAllocColor fails, we return
 * the nearest matching color.  If we have to allocate many colors
 * this function isn't too efficient; the XQueryColors() could be
 * done just once.
 * Written by Michael Pichler, Brian Paul, Mark Kilgard
 * Input:  dpy - X display
 *         cmap - X colormap
 *         cmapSize - size of colormap
 * In/Out: color - the XColor struct
 * Output:  exact - 1=exact color match, 0=closest match
 *          alloced - 1=XAlloc worked, 0=XAlloc failed
 */
static void
noFaultXAllocColor( ClientPtr client, ColormapPtr cmap, int cmapSize,
                    XColor *color, int *exact, int *alloced )
{
    Pixel  *pixtable;
    XColor subColor;
    xrgb   *rgbtable;
    int    i, bestmatch;
    double mindist;       /* 3*2^16^2 exceeds long int precision. */

    /* First try just using XAllocColor. */
    if (AllocColor(cmap, &color->red, &color->green, &color->blue,
                   &color->pixel, client->index)==Success)
    {
        *exact = 1;
        *alloced = 1;
        return;
    }

    /* Alloc failed, search for closest match */

    /* Retrieve color table entries. */
    pixtable = (Pixel *) malloc(cmapSize * sizeof(Pixel));
    rgbtable = (xrgb*)malloc(cmapSize*sizeof(xrgb));
    for (i = 0; i < cmapSize; i++)
        pixtable[i] = i;
    QueryColors(cmap, cmapSize, pixtable, rgbtable);

    /* Find best match. */
    bestmatch = -1;
    mindist = 0.0;
    for (i = 0; i < cmapSize; i++)
    {
        double dr = 0.30 * ((double) color->red - (double) rgbtable[i].red);
        double dg = 0.59 * ((double) color->green - (double) rgbtable[i].green);
        double db = 0.11 * ((double) color->blue - (double) rgbtable[i].blue);
        double dist = dr * dr + dg * dg + db * db;
        if (bestmatch < 0 || dist < mindist)
        {
            bestmatch = i;
            mindist = dist;
        }
    }

    /* Return result. */
    subColor.red   = rgbtable[bestmatch].red;
    subColor.green = rgbtable[bestmatch].green;
    subColor.blue  = rgbtable[bestmatch].blue;
    free(pixtable);
    free(rgbtable);
    /* Try to allocate the closest match color.  This should only
    * fail if the cell is read/write.  Otherwise, we're incrementing
    * the cell's reference count.
    */
    if (AllocColor(cmap, &subColor.red, &subColor.green, &subColor.blue,
                   &subColor.pixel, client->index) == Success)
    {
        *alloced = 1;
    }
    else
    {
        /* do this to work around a problem reported by Frank Ortega */
        subColor.pixel = (unsigned long) bestmatch;
        subColor.red   = rgbtable[bestmatch].red;
        subColor.green = rgbtable[bestmatch].green;
        subColor.blue  = rgbtable[bestmatch].blue;   
        subColor.flags = DoRed | DoGreen | DoBlue;
        *alloced = 0;
    }
    *color = subColor;
    *exact = 0;
}

/*
 * Do setup for PF_GRAYSCALE pixel format.
 * Note that buffer may be NULL.
 */
static GLboolean setup_grayscale( XSMesaVisual v, ClientPtr client,
                                  XSMesaBuffer buffer, ColormapPtr cmap )
{
    if (v->pVisual->nplanes<4 || v->pVisual->nplanes>16)
    {
        return(GL_FALSE);
    }

    if (buffer)
    {
        XSMesaBuffer prevBuffer;

        if (!cmap)
        {
            return(GL_FALSE);
        }

        prevBuffer = find_xsmesa_buffer(buffer->xsm_visual->pScreen, cmap, buffer);
        if (prevBuffer)
        {
            /* Copy colormap stuff from previous XSMesaBuffer which uses same
            * X colormap.  Do this to avoid time spent in noFaultXAllocColor.
            */
            copy_colortable_info(buffer, prevBuffer);
        }
        else
        {
            /* Allocate 256 shades of gray */
            int gray;
            int colorsfailed = 0;
            for (gray=0;gray<256;gray++)
            {
                GLint r = gamma_adjust( v->RedGamma,   gray, 255 );
                GLint g = gamma_adjust( v->GreenGamma, gray, 255 );
                GLint b = gamma_adjust( v->BlueGamma,  gray, 255 );
                int exact, alloced;
                XColor xcol;
                xcol.red   = (r << 8) | r;
                xcol.green = (g << 8) | g;
                xcol.blue  = (b << 8) | b;
                noFaultXAllocColor( client, cmap, v->pVisual->ColormapEntries,
                                    &xcol, &exact, &alloced );
                if (!exact)
                {
                    colorsfailed++;
                }
                if (alloced)
                {
                    assert(buffer->num_alloced<256);
                    buffer->alloced_colors[buffer->num_alloced] = xcol.pixel;
                    buffer->num_alloced++;
                }

                assert(gray < 576);
                buffer->color_table[gray] = xcol.pixel;
                assert(xcol.pixel < 65536);
                buffer->pixel_to_r[xcol.pixel] = gray * 30 / 100;
                buffer->pixel_to_g[xcol.pixel] = gray * 59 / 100;
                buffer->pixel_to_b[xcol.pixel] = gray * 11 / 100;
            }

            if (colorsfailed && getenv("MESA_DEBUG"))
            {
                fprintf( stderr,
                         "Note: %d out of 256 needed colors do not match exactly.\n",
                         colorsfailed );
            }
        }
    }

#define WEIGHT
#ifdef WEIGHT
    v->rmult = 30 * 255 / 100;
    v->gmult = 59 * 255 / 100;
    v->bmult = 11 * 255 / 100;
#else
    v->rmult = 255/3;
    v->gmult = 255/3;
    v->bmult = 255/3;
#endif
    v->dithered_pf = PF_GRAYSCALE;
    v->undithered_pf = PF_GRAYSCALE;
    return(GL_TRUE);
}

/*
 * Setup RGB rendering for a window with a PseudoColor, StaticColor,
 * or 8-bit TrueColor visual visual.  We try to allocate a palette of 225
 * colors (5 red, 9 green, 5 blue) and dither to approximate a 24-bit RGB
 * color.  While this function was originally designed just for 8-bit
 * visuals, it has also proven to work from 4-bit up to 16-bit visuals.
 * Dithering code contributed by Bob Mercier.
 */
static GLboolean setup_dithered_color( XSMesaVisual v, ClientPtr client,
                                       XSMesaBuffer buffer, ColormapPtr cmap )
{
    if (v->pVisual->nplanes<4 || v->pVisual->nplanes>16)
    {
        return(GL_FALSE);
    }

    if (buffer)
    {
        XSMesaBuffer prevBuffer;

        if (!cmap)
        {
            return(GL_FALSE);
        }

        prevBuffer = find_xsmesa_buffer(buffer->xsm_visual->pScreen, cmap, buffer);
        if (prevBuffer)
        {
            /* Copy colormap stuff from previous, matching XSMesaBuffer.
            * Do this to avoid time spent in noFaultXAllocColor.
            */
            copy_colortable_info(buffer, prevBuffer);
        }
        else
        {
            /* Allocate X colors and initialize color_table[], red_table[], etc */
            int r, g, b, i;
            int colorsfailed = 0;
            for (r = 0; r < _R; r++)
            {
                for (g = 0; g < _G; g++)
                {
                    for (b = 0; b < _B; b++)
                    {
                        XColor xcol;
                        int exact, alloced;
                        xcol.red  =gamma_adjust(v->RedGamma,   r*65535/(_R-1),65535);
                        xcol.green=gamma_adjust(v->GreenGamma, g*65535/(_G-1),65535);
                        xcol.blue =gamma_adjust(v->BlueGamma,  b*65535/(_B-1),65535);
                        noFaultXAllocColor(client, cmap,
                                           v->pVisual->ColormapEntries, &xcol,
                                           &exact, &alloced);
                        if (!exact)
                        {
                            colorsfailed++;
                        }
                        if (alloced)
                        {
                            assert(buffer->num_alloced<256);
                            buffer->alloced_colors[buffer->num_alloced] = xcol.pixel;
                            buffer->num_alloced++;
                        }
                        i = _MIX( r, g, b );
                        assert(i < 576);
                        buffer->color_table[i] = xcol.pixel;
                        assert(xcol.pixel < 65536);
                        buffer->pixel_to_r[xcol.pixel] = r * 255 / (_R-1);
                        buffer->pixel_to_g[xcol.pixel] = g * 255 / (_G-1);
                        buffer->pixel_to_b[xcol.pixel] = b * 255 / (_B-1);
                    }
                }
            }

            if (colorsfailed && getenv("MESA_DEBUG"))
            {
                fprintf( stderr,
                         "Note: %d out of %d needed colors do not match exactly.\n",
                         colorsfailed, _R*_G*_B );
            }
        }
    }

    v->rmult = 255;
    v->gmult = 255;
    v->bmult = 255;
    v->dithered_pf = PF_DITHER;
    v->undithered_pf = PF_LOOKUP;
    return(GL_TRUE);
}

/*
 * Setup RGB rendering for a window with a True/DirectColor visual.
 */
static GLboolean setup_truecolor( XSMesaVisual v, ClientPtr client,
                                  XSMesaBuffer buffer, ColormapPtr cmap )
{
    unsigned long rmask, gmask, bmask;

    /* Compute red multiplier (mask) and bit shift */
    v->rshift = 0;
    rmask = v->pVisual->redMask;
    while ((rmask & 1)==0)
    {
        v->rshift++;
        rmask = rmask >> 1;
    }

    /* Compute green multiplier (mask) and bit shift */
    v->gshift = 0;
    gmask = v->pVisual->greenMask;
    while ((gmask & 1)==0)
    {
        v->gshift++;
        gmask = gmask >> 1;
    }

    /* Compute blue multiplier (mask) and bit shift */
    v->bshift = 0;
    bmask = v->pVisual->blueMask;
    while ((bmask & 1)==0)
    {
        v->bshift++;
        bmask = bmask >> 1;
    }

    /*
    * Compute component-to-pixel lookup tables and dithering kernel
    */
    {
        static GLubyte kernel[16] =
        {
            0*16,  8*16,  2*16, 10*16,
            12*16,  4*16, 14*16,  6*16,
            3*16, 11*16,  1*16,  9*16,
            15*16,  7*16, 13*16,  5*16,
        };
        GLint rBits = bitcount(rmask);
        GLint gBits = bitcount(gmask);
        GLint bBits = bitcount(bmask);
        GLint minBits;
        GLint i;

        /* convert pixel components in [0,_mask] to RGB values in [0,255] */
        for (i=0; i<=rmask; i++)
            v->PixelToR[i] = (i * 255) / rmask;
        for (i=0; i<=gmask; i++)
            v->PixelToG[i] = (i * 255) / gmask;
        for (i=0; i<=bmask; i++)
            v->PixelToB[i] = (i * 255) / bmask;

        /* convert RGB values from [0,255] to pixel components */
        for (i=0;i<256;i++)
        {
            GLint r = gamma_adjust(v->RedGamma,   i, 255);
            GLint g = gamma_adjust(v->GreenGamma, i, 255);
            GLint b = gamma_adjust(v->BlueGamma,  i, 255);
            v->RtoPixel[i] = (r >> (8-rBits)) << v->rshift;
            v->GtoPixel[i] = (g >> (8-gBits)) << v->gshift;
            v->BtoPixel[i] = (b >> (8-bBits)) << v->bshift;
        }
        /* overflow protection */
        for (i=256;i<512;i++)
        {
            v->RtoPixel[i] = v->RtoPixel[255];
            v->GtoPixel[i] = v->GtoPixel[255];
            v->BtoPixel[i] = v->BtoPixel[255];
        }

        /* setup dithering kernel */
        minBits = rBits;
        if (gBits < minBits)  minBits = gBits;
        if (bBits < minBits)  minBits = bBits;
        for (i=0;i<16;i++)
        {
            v->Kernel[i] = kernel[i] >> minBits;
        }

        v->rmult = v->gmult = v->bmult = 255;
        v->undithered_pf = PF_TRUECOLOR;
        v->dithered_pf = (v->pVisual->nplanes<24) ? PF_TRUEDITHER : PF_TRUECOLOR;
    }

    /*
     * Now check for TrueColor visuals which we can optimize.
     */

    if (   v->pVisual->redMask  ==0x0000ff
           && v->pVisual->greenMask==0x00ff00
           && v->pVisual->blueMask ==0xff0000
           && sizeof(GLuint)==4
           && v->RedGamma==1.0 && v->GreenGamma==1.0 && v->BlueGamma==1.0)
    {
        /* common 24-bit config used on SGI, Sun */
        v->undithered_pf = v->dithered_pf = PF_8A8B8G8R;
    }
    else if (v->pVisual->redMask  ==0xff0000
             &&   v->pVisual->greenMask==0x00ff00
             &&   v->pVisual->blueMask ==0x0000ff
             && sizeof(GLuint)==4
             && v->RedGamma==1.0 && v->GreenGamma==1.0 && v->BlueGamma==1.0)
    {
        /* common 24-bit config used on Linux, HP, IBM */
        v->undithered_pf = v->dithered_pf = PF_8R8G8B;
    }
    else if (v->pVisual->redMask  ==0xf800
             &&   v->pVisual->greenMask==0x07e0
             &&   v->pVisual->blueMask ==0x001f
             && sizeof(GLushort)==2
             && v->RedGamma==1.0 && v->GreenGamma==1.0 && v->BlueGamma==1.0)
    {
        /* 5-6-5 color weight on common PC VGA boards */
        v->undithered_pf = PF_5R6G5B;
        v->dithered_pf = PF_TRUEDITHER;
    }

    return(GL_TRUE);
}

/*
 * Setup RGB rendering for a window with a monochrome visual.
 */
static GLboolean setup_monochrome( XSMesaVisual v )
{
    v->rmult = 255;
    v->gmult = 255;
    v->bmult = 255;
    v->dithered_pf = v->undithered_pf = PF_1BIT;
    /* if black=1 then we must flip pixel values
    v->bitFlip = (v->pScreen->black_pixel != 0); */
    return(GL_TRUE);
}

/*
 * When a context is "made current" for the first time, we can finally
 * finish initializing the context's visual and buffer information.
 * Input:  v - the XSMesaVisual to initialize
 *         b - the XSMesaBuffer to initialize (may be NULL)
 *         rgb_flag - TRUE = RGBA mode, FALSE = color index mode
 *         window - the window/pixmap we're rendering into
 *         cmap - the colormap associated with the window/pixmap
 * Return:  GL_TRUE=success, GL_FALSE=failure
 */
static GLboolean initialize_visual_and_buffer( XSMesaVisual v,
                                               ClientPtr client,
                                               XSMesaBuffer b,
                                               GLboolean rgb_flag,
                                               DrawablePtr window,
                                               ColormapPtr cmap )
{
/*      XGCValues gcvalues; */
    XID aluval;

    if (b)
    {
        assert(b->xsm_visual == v);
    }

    if (rgb_flag==GL_FALSE)
    {
        /* COLOR-INDEXED WINDOW:
         * Even if the visual is TrueColor or DirectColor we treat it as
         * being color indexed.  This is weird but might be useful to someone.
         */
        v->dithered_pf = v->undithered_pf = PF_INDEX;
        v->rmult = v->gmult = v->bmult = 0;
        v->index_bits = v->pVisual->nplanes;
    }
    else
    {
        /* RGB WINDOW:
         * We support RGB rendering into almost any kind of visual.
         */
        int xclass;
#if defined(__cplusplus) || defined(c_plusplus)
        xclass = v->pVisual->c_class;
#else
        xclass = v->pVisual->class;
#endif
        if (xclass==TrueColor || xclass==DirectColor)
        {
            setup_truecolor( v, client, b, cmap );
        }
        else if (xclass==StaticGray && v->pVisual->nplanes==1)
        {
            setup_monochrome( v );
        }
        else if (xclass==GrayScale || xclass==StaticGray)
        {
            if (!setup_grayscale( v, client, b, cmap ))
            {
                return(GL_FALSE);
            }
        }
        else if ((xclass==PseudoColor || xclass==StaticColor)
            && v->pVisual->nplanes>=4 && v->pVisual->nplanes<=16)
        {
            if (!setup_dithered_color( v, client, b, cmap ))
            {
                return(GL_FALSE);
            }
        }
        else
        {
            error("XSMesa: RGB mode rendering not supported in given visual.");
            return(GL_FALSE);
        }
        v->index_bits = 0;
    }

    /*
     * If MESA_INFO env var is set print out some debugging info
     * which can help Brian figure out what's going on when a user
     * reports bugs.
     */
    if (getenv("MESA_INFO"))
    {
        printf("v = %p\n", v);
        printf("dithered pf = %d\n", v->dithered_pf);
        printf("undithered pf = %d\n", v->undithered_pf);
        printf("level = %d\n", v->level);
        printf("depth = %d\n", v->pVisual->nplanes);
        /*
        printf("bits per pixel = %d\n", bits_per_pixel(v->display, v->visinfo));
        */
    }

    if (b && window)
    {
        /* Do window-specific initializations */

        /* Window dimensions */
        b->width = window->width;
        b->height = window->height;

        b->frontbuffer = (DrawablePtr)window;

        assert( v->gl_visual );

        /* Setup for single/double buffering */
        if (v->gl_visual->DBflag)
        {
            /* Double buffered */
            xsmesa_alloc_back_buffer( b );
            if (b->db_state==BACK_PIXMAP)
            {
                b->buffer = (DrawablePtr)b->backpixmap;
            }
            else
            {
                b->buffer = XIMAGE;
            }
        }
        else
        {
            /* Single Buffered */
            b->buffer = b->frontbuffer;
        }

        /* X11 graphics contexts */
	aluval = GXcopy;

        b->gc1 = GetScratchGC(window->depth, v->pScreen);
	DoChangeGC(b->gc1, GCFunction, &aluval, 0);

        b->gc2 = GetScratchGC(window->depth, v->pScreen);
	DoChangeGC(b->gc2, GCFunction, &aluval, 0);

        /*
        * Don't generate Graphics Expose/NoExpose events in swapbuffers().
        * Patch contributed by Michael Pichler May 15, 1995.
        b->cleargc = GetScratchGC(window->depth, v->pScreen);
        b->cleargc->alu = GXcopy;
        b->cleargc->graphicsExposures=0;
        */
        b->cleargc = GetScratchGC(window->depth, v->pScreen);
    }

    return(GL_TRUE);
}

/*
 * Convert an RGBA color to a pixel value.
 */
unsigned long xsmesa_color_to_pixel( XSMesaContext xsmesa,
                                     GLubyte r, GLubyte g, GLubyte b, GLubyte a)
{
    switch (xsmesa->pixelformat)
    {
        case PF_INDEX:
            return(0);
        case PF_TRUECOLOR:
            {
                unsigned long p;
                PACK_TRUECOLOR( p, r, g, b );
                return(p);
            }
        case PF_8A8B8G8R:
            return(PACK_8A8B8G8R( r, g, b, a ));
        case PF_8R8G8B:
            return(PACK_8R8G8B( r, g, b ));
        case PF_5R6G5B:
            return(PACK_5R6G5B( r, g, b ));
        case PF_DITHER:
            {
                DITHER_SETUP;
                return(DITHER( 1, 0, r, g, b ));
            }
        case PF_1BIT:
            /* 382 = (3*255)/2 */
            return((r+g+b) > 382U) ^ xsmesa->xsm_visual->bitFlip;
        case PF_LOOKUP:
            {
                LOOKUP_SETUP;
                return(LOOKUP( r, g, b ));
            }
        case PF_GRAYSCALE:
            return(GRAY_RGB( r, g, b ));
        case PF_TRUEDITHER:
            {
                unsigned long p;
                PACK_TRUEDITHER(p, 1, 0, r, g, b);
                return(p);
            }
        case PF_HPCR:
        default:
            gl_problem(NULL, "Bad pixel format in xsmesa_color_to_pixel");
            return(0);
    }
    return(0);  /*never get here*/
}

/**********************************************************************/
/*****                       Public Functions                     *****/
/**********************************************************************/

/*
 * Create a new X/Mesa visual.
 * Input:  display - the X display
 *         visinfo - the XVisualInfo
 *         rgb_flag - TRUE=RGB(A) mode, FALSE=CI mode
 *         alpha_flag - need alpha planes?
 *         db_flag - TRUE=double bufferd, FALSE=single
 *         ximage_flag - TRUE=use XImage for back buffer, FALSE=use Pixmap
 *         depth_size - requested min bits per depth buffer value
 *         stencil_size - requested min bits per stencil buffer value
 *         accum_size - requested min bits per accum buffer value (per channel)
 *         level - 0=normal, 1=overaly, -1=underlay, etc.
 * Return:  New XSMesaVisual or NULL if something goes wrong
 */
XSMesaVisual XSMesaCreateVisual( ScreenPtr pScreen,
                                 VisualPtr pVisual,
                                 GLboolean rgb_flag,
                                 GLboolean alpha_flag,
                                 GLboolean db_flag,
                                 GLboolean ximage_flag,
                                 GLint depth_size,
                                 GLint stencil_size,
                                 GLint accum_size,
                                 GLint level )
{
    char *gamma;
    XSMesaVisual v;
    GLfloat red_scale, green_scale, blue_scale, alpha_scale;
    GLint red_bits, green_bits, blue_bits, alpha_bits;
    GLboolean stereo_flag = GL_FALSE;

    v = (XSMesaVisual) calloc( 1, sizeof(struct xsmesa_visual) );
    if (!v)
    {
        return(NULL);
    }

    v->pScreen = pScreen;
    v->pVisual = pVisual;

    /* check for MESA_GAMMA environment variable */
    gamma = getenv("MESA_GAMMA");
    if (gamma)
    {
        v->RedGamma = v->GreenGamma = v->BlueGamma = 0.0;
        sscanf( gamma, "%f %f %f", &v->RedGamma, &v->GreenGamma, &v->BlueGamma );
        if (v->RedGamma<=0.0)    v->RedGamma = 1.0;
        if (v->GreenGamma<=0.0)  v->GreenGamma = v->RedGamma;
        if (v->BlueGamma<=0.0)   v->BlueGamma = v->RedGamma;
    }
    else
    {
        v->RedGamma = v->GreenGamma = v->BlueGamma = 1.0;
    }

    v->ximage_flag = ximage_flag;
    v->level = level;

    (void) initialize_visual_and_buffer( v, NULL, NULL, rgb_flag, 0, 0 );

    red_scale   = (GLfloat) v->rmult;
    green_scale = (GLfloat) v->gmult;
    blue_scale  = (GLfloat) v->bmult;
    alpha_scale = 255.0;
    {
        int xclass;
#if defined(__cplusplus) || defined(c_plusplus)
        xclass = v->pVisual->c_class;
#else
        xclass = v->pVisual->class;
#endif
        if (xclass==TrueColor || xclass==DirectColor)
        {
            red_bits   = bitcount(v->pVisual->redMask);
            green_bits = bitcount(v->pVisual->greenMask);
            blue_bits  = bitcount(v->pVisual->blueMask);
            alpha_bits = 0;
        }
        else
        {
            /* this is an approximation */
            int depth = v->pVisual->nplanes;
            red_bits = depth / 3;
            depth -= red_bits;
            green_bits = depth / 2;
            depth -= green_bits;
            blue_bits = depth;
            alpha_bits = 0;
            assert( red_bits + green_bits + blue_bits == v->pVisual->nplanes );
        }
    }
    v->gl_visual = gl_create_visual( rgb_flag, alpha_flag, db_flag, 
                                     stereo_flag, depth_size, stencil_size, accum_size,
                                     v->index_bits,
                                     red_bits, green_bits,
                                     blue_bits, alpha_bits );
    if (!v->gl_visual)
    {
        free(v);
        return(NULL);
    }

    return(v);
}

void XSMesaDestroyVisual( XSMesaVisual v )
{
    gl_destroy_visual( v->gl_visual );
    free(v);
}

/*
 * Create a new XSMesaContext.
 * Input:  v - XSMesaVisual
 *         share_list - another XSMesaContext with which to share display
 *                      lists or NULL if no sharing is wanted.
 * Return:  an XSMesaContext or NULL if error.
 */
XSMesaContext XSMesaCreateContext( XSMesaVisual v,
                                   XSMesaContext share_list )
{
    XSMesaContext c;

    ErrorF ( "### Creating new xsmesa context for sw...\n" );

    c = (XSMesaContext) calloc( 1, sizeof(struct xsmesa_context) );
    if (!c)
    {
        return(NULL);
    }

    c->gl_ctx = gl_create_context( v->gl_visual,
                                   share_list ? share_list->gl_ctx : NULL,
                                   (void *) c , GL_TRUE /* direct rendering */);
    if (!c->gl_ctx)
    {
        free(c);
        return(NULL);
    }

    c->xsm_visual = v;
    c->xsm_buffer = NULL;       /* set later by XSMesaMakeCurrent */
    c->pixelformat = v->dithered_pf;    /* Dithering is enabled by default */
    c->gl_ctx->Driver.UpdateState = xsmesa_setup_DD_pointers;

    return(c);
}

void XSMesaDestroyContext( XSMesaContext c )
{
    if (c->gl_ctx)
        gl_destroy_context( c->gl_ctx );

    free( c );

    if (XSMesa == c) {
       XSMesa = 0;       
       gl_make_current(NULL, NULL);                                            
    }
}

/*
 * Create a new XSMesaBuffer from an X window.
 * Input:  v - the XSMesaVisual
 *         W - the window
 * Return:  new XSMesaBuffer or NULL if error
 */
XSMesaBuffer XSMesaCreateWindowBuffer( XSMesaVisual v, ClientPtr client,
                                       WindowPtr w )
{
    XSMesaBuffer b = alloc_xsmesa_buffer();
    if (!b)
    {
        return(NULL);
    }

    assert(v);

    /*
    if (v->visinfo->depth != attr.depth)
    {
    if (getenv("MESA_DEBUG"))
    {
        fprintf(stderr, "XSMesaCreateWindowBuffer: depth mismatch between visual and window!\n");
    }
    return NULL;
    }
    */

    b->xsm_visual = v;
    b->pixmap_flag = GL_FALSE;

    /* removed from xsmesa:
    
    b->display = v->display;
    
    */

 #if 0
    in 2.5 originally:

    if (attr.colormap)
    {
        b->cmap = attr.colormap;
    }
    else
    {
        if (getenv("MESA_DEBUG"))
        {
            fprintf(stderr, "Window %d has no colormap!\n", w);
        }
        /* this is weird, a window w/out a colormap!? */
        /* OK, let's just allocate a new one and hope for the best */
        b->cmap = XCreateColormap(v->display, w, attr.visual, AllocNone);
    }
#endif

    /* only in xsmesa */
    b->cmap = (ColormapPtr) LookupIDByType (wColormap(w), RT_COLORMAP);

    /* determine back buffer implementation */
    if (v->gl_visual->DBflag)
    {
        if (v->ximage_flag)
        {
            b->db_state = BACK_XIMAGE; /* always XIMAGE */
        }
        else
        {
            b->db_state = BACK_PIXMAP;
        }
    }
    else
    {
        b->db_state = 0;
    }

    b->gl_buffer = gl_create_framebuffer( v->gl_visual );
    if (!b->gl_buffer)
    {
        free_xsmesa_buffer(b);
        return(NULL);
    }

    if (!initialize_visual_and_buffer( v, client, b, v->gl_visual->RGBAflag,
                                       (DrawablePtr)w, b->cmap ))
    {
        gl_destroy_framebuffer( b->gl_buffer );
        free_xsmesa_buffer(b);
        return(NULL);
    }
    return(b);
}

/*
 * Create a new XSMesaBuffer from an X pixmap.
 * Input:  v - the XSMesaVisual
 *         p - the pixmap
 *         cmap - the colormap, may be 0 if using a TrueColor or DirectColor
 *                visual for the pixmap
 * Return:  new XSMesaBuffer or NULL if error
 */
XSMesaBuffer XSMesaCreatePixmapBuffer( XSMesaVisual v, ClientPtr client,
                                       PixmapPtr p, ColormapPtr cmap )
{
    XSMesaBuffer b = alloc_xsmesa_buffer();
    if (!b)
    {
        return(NULL);
    }

    assert(v);

    b->xsm_visual = v;
    b->pixmap_flag = GL_TRUE;
    b->cmap = cmap;

    /* determine back buffer implementation */
    if (v->gl_visual->DBflag)
    {
        if (v->ximage_flag)
        {
            b->db_state = BACK_XIMAGE; /* always XIMAGE */
        }
        else
        {
            b->db_state = BACK_PIXMAP;
        }
    }
    else
    {
        b->db_state = 0;
    }

    b->gl_buffer = gl_create_framebuffer( v->gl_visual );
    if (!b->gl_buffer)
    {
        free_xsmesa_buffer(b);
        return(NULL);
    }

    if (!initialize_visual_and_buffer(v, client, b, v->gl_visual->RGBAflag,
                                      (DrawablePtr)p, cmap))
    {
        gl_destroy_framebuffer( b->gl_buffer );
        free_xsmesa_buffer(b);
        return(NULL);
    }

    return(b);
}

/*
 * Deallocate an XSMesaBuffer structure and all related info.
 */
void XSMesaDestroyBuffer( XSMesaBuffer b )
{
    if (b->gc1)     FreeScratchGC(b->gc1);
    if (b->gc2)     FreeScratchGC(b->gc2);
    if (b->cleargc) FreeScratchGC(b->cleargc);

    if (b->backimage)
        GLXProcs.DestroyImage(b->backimage);
    if (b->backpixmap)
        ( *b->xsm_visual->pScreen->DestroyPixmap)( b->backpixmap );

    gl_destroy_framebuffer( b->gl_buffer );
    free_xsmesa_buffer(b);
}

GLboolean XSMesaMakeCurrent( XSMesaContext c )
{
    if (c == XSMesa) return (GL_TRUE);                                          
    XSMesa = c;                                                                 
    if (c)                                                                      
        gl_make_current(c->gl_ctx,  
           c->xsm_buffer ? c->xsm_buffer->gl_buffer : NULL);
    else                                                                        
        gl_make_current(NULL, NULL);                                            
    return(GL_TRUE); 
}

/*
 * Bind buffer b to context c and make c the current rendering context.
 */
GLboolean XSMesaBindBuffer( XSMesaContext c, XSMesaBuffer b )
{
    if ((c && !b) || (!c && b))
    {
        return(GL_FALSE);
    }

    if (c)
    {
        if (c->gl_ctx == gl_get_current_context() && c->xsm_buffer == b
            && c->xsm_buffer->wasCurrent)
        {
            /* same context and buffer, do nothing */
            return(GL_TRUE);
        }
        if (!c->gl_ctx)
	{
	    ErrorF("XSMesaBindBuffer: no gl_ctx\n");
	    return(GL_FALSE);
	}
 
        c->xsm_buffer = b;
        XSMesa = c;
        gl_make_current( c->gl_ctx, b->gl_buffer );
        c->gl_ctx->Driver.UpdateState(c->gl_ctx);
        if (c->gl_ctx->Viewport.Width==0)
        {
            /* initialize viewport to window size */
            gl_Viewport( c->gl_ctx, 0, 0, b->width, b->height );
            c->gl_ctx->Scissor.Width = b->width;
            c->gl_ctx->Scissor.Height = b->height;
        }

        if (c->xsm_visual->gl_visual->RGBAflag)
        {
            /*
             * Must recompute and set these pixel values because colormap
             * can be different for different windows.
             */
            XID value;

            c->pixel = xsmesa_color_to_pixel( c, c->red, c->green,
                                              c->blue, c->alpha );
            value = (XID)c->pixel;
            DoChangeGC(c->xsm_buffer->gc1, GCForeground, &value, 0);
            c->clearpixel = xsmesa_color_to_pixel( c,
                                                   c->clearcolor[0],
                                                   c->clearcolor[1],
                                                   c->clearcolor[2],
                                                   c->clearcolor[3] );
            value = (XID)c->clearpixel;
            DoChangeGC(c->xsm_buffer->cleargc, GCForeground, &value, 0);
        }
        /* Solution to Stephane Rehel's problem with glXReleaseBuffersMESA(): */
        c->xsm_buffer->wasCurrent = GL_TRUE;
    }
    else
    {
        /* Detach */
        gl_make_current( NULL, NULL );
        XSMesa = NULL;
    }
    return(GL_TRUE);
}

XSMesaContext XSMesaGetCurrentContext( void )
{
    return(XSMesa);
}

XSMesaBuffer XSMesaGetCurrentBuffer( void )
{
    if (XSMesa)
    {
        return(XSMesa->xsm_buffer);
    }
    else
    {
        return(0);
    }
}

/*
 * Copy the back buffer to the front buffer.  If there's no back buffer
 * this is a no-op.
 */
void XSMesaSwapBuffers( XSMesaBuffer b )
{
#ifdef PROFILE
    GLdouble t0 = gl_time();
#endif

    if (b->db_state)
    {

        ValidateGC(b->frontbuffer, b->cleargc);
        if (b->backimage)
        {
            (*b->cleargc->ops->PutImage)((DrawablePtr)b->frontbuffer,
                                         b->cleargc,
                                         b->frontbuffer->depth,
                                         0, 0, b->width, b->height,
                                         0, ZPixmap, b->backimage->data);
        }
        else
        {
            /* Copy pixmap to window on server */
            (*b->cleargc->ops->CopyArea)((DrawablePtr)b->backpixmap,
                                         b->frontbuffer,
                                         b->cleargc,
                                         0, 0, b->width, b->height,
                                         0, 0);
        }
    }

#ifdef PROFILE
    XSMesa->gl_ctx->SwapCount++;
    XSMesa->gl_ctx->SwapTime += gl_time() - t0;
#endif
}

/*
 * Return a pointer to the XSMesa backbuffer Pixmap or XImage.  This function
 * is a way to get "under the hood" of X/Mesa so one can manipulate the
 * back buffer directly.
 * Output:  pixmap - pointer to back buffer's Pixmap, or 0
 *          ximage - pointer to back buffer's XImage, or NULL
 * Return:  GL_TRUE = context is double buffered
 *          GL_FALSE = context is single buffered
 */
GLboolean XSMesaGetBackBuffer( XSMesaBuffer b, PixmapPtr *pixmap,
                               XImage **ximage )
{
    if (b->db_state)
    {
        if (pixmap)  *pixmap = b->backpixmap;
        if (ximage)  *ximage = (XImage *)b->backimage;
        return(GL_TRUE);
    }
    else
    {
        *pixmap = 0;
        *ximage = NULL;
        return(GL_FALSE);
    }
}

/*
 * Return the depth buffer associated with an XSMesaBuffer.
 * Input:  b - the XSMesa buffer handle
 * Output:  width, height - size of buffer in pixels
 *          bytesPerValue - bytes per depth value (2 or 4)
 *          buffer - pointer to depth buffer values
 * Return:  GL_TRUE or GL_FALSE to indicate success or failure.
 */
GLboolean XSMesaGetDepthBuffer( XSMesaBuffer b, GLint *width, GLint *height,
                                GLint *bytesPerValue, void **buffer )
{
    if ((!b->gl_buffer) || (!b->gl_buffer->Depth))
    {
        *width = 0;
        *height = 0;
        *bytesPerValue = 0;
        *buffer = 0;
        return(GL_FALSE);
    }
    else
    {
        *width = b->gl_buffer->Width;
        *height = b->gl_buffer->Height;
        *bytesPerValue = sizeof(GLdepth);
        *buffer = b->gl_buffer->Depth;
        return(GL_TRUE);
    }
}

XSMesaBuffer XSMesaFindBuffer( ScreenPtr pScreen, DrawablePtr d )
{
    XSMesaBuffer b;
    for (b=XSMesaBufferList; b; b=b->Next)
    {
        if (b->xsm_visual->pScreen==pScreen && b->frontbuffer==d)
        {
            return(b);
        }
    }
    return(NULL);
}

#if 0
/*
 * Look for XSMesaBuffers whose X window has been destroyed.
 * Deallocate any such XMesaBuffers.
 */
void XSMesaGarbageCollect( void )
{
    XSMesaBuffer b, next;
    for (b=XSMesaBufferList; b; b=next)
    {
        next = b->Next;
        if (!b->pixmap_flag)
        {
            if (!window_exists( b->display, b->frontbuffer ))
            {
                /* found a dead window, free the ancillary info */
                XSMesaDestroyBuffer( b );
            }
        }
    }
}
#endif

unsigned long XSMesaDitherColor( XSMesaContext xsmesa, GLint x, GLint y,
                                 GLfloat red, GLfloat green,
                                 GLfloat blue, GLfloat alpha )
{
    GLint r = (GLint) (red   * (GLfloat) xsmesa->xsm_visual->rmult);
    GLint g = (GLint) (green * (GLfloat) xsmesa->xsm_visual->gmult);
    GLint b = (GLint) (blue  * (GLfloat) xsmesa->xsm_visual->bmult);
    GLint a = (GLint) (alpha * 255.0F);

    switch (xsmesa->pixelformat)
    {
        case PF_INDEX:
            return(0);
        case PF_TRUECOLOR:
            {
                unsigned long p;
                PACK_TRUECOLOR( p, r, g, b );
                return(p);
            }
        case PF_8A8B8G8R:
            return(PACK_8A8B8G8R( r, g, b, a ));
        case PF_8R8G8B:
            return(PACK_8R8G8B( r, g, b ));
        case PF_5R6G5B:
            return(PACK_5R6G5B( r, g, b ));
        case PF_DITHER:
            {
                DITHER_SETUP;
                return(DITHER( x, y, r, g, b ));
            }
        case PF_1BIT:
            /* 382 = (3*255)/2 */
            return((r+g+b) > 382U) ^ xsmesa->xsm_visual->bitFlip;
        case PF_LOOKUP:
            {
                LOOKUP_SETUP;
                return(LOOKUP( r, g, b ));
            }
        case PF_GRAYSCALE:
            return(GRAY_RGB( r, g, b ));
        case PF_TRUEDITHER:
            {
                unsigned long p;
                PACK_TRUEDITHER(p, x, y, r, g, b);
                return(p);
            }
        case PF_HPCR:
        default:
            gl_problem(NULL, "Bad pixel format in XSMesaDitherColor");
            return(0);
    }
    return(0);  /*never get here*/
}

/*
 * Copy a context...
 */
void XSMesaCopyContext(XSMesaContext src,
                       XSMesaContext dst,
                       unsigned int mask)
{
    XSMesaContext xsm_src, xsm_dst;
    xsm_src = (XSMesaContext) src;
    xsm_dst = (XSMesaContext) dst;
    gl_copy_context( xsm_src->gl_ctx, xsm_dst->gl_ctx, mask );        
}

char* XSMesaFeedbackBuffer( XSMesaContext ctx )
{
    return(char*)ctx->gl_ctx->Feedback.Buffer;
}

char* XSMesaSelectionBuffer( XSMesaContext ctx )
{
    return(char*)ctx->gl_ctx->Select.Buffer;
}

