 /***************************************************************************\
|*                                                                           *|
|*       Copyright 1993-1999 NVIDIA, Corporation.  All rights reserved.      *|
|*                                                                           *|
|*     NOTICE TO USER:   The source code  is copyrighted under  U.S. and     *|
|*     international laws.  Users and possessors of this source code are     *|
|*     hereby granted a nonexclusive,  royalty-free copyright license to     *|
|*     use this code in individual and commercial software.                  *|
|*                                                                           *|
|*     Any use of this source code must include,  in the user documenta-     *|
|*     tion and  internal comments to the code,  notices to the end user     *|
|*     as follows:                                                           *|
|*                                                                           *|
|*       Copyright 1993-1999 NVIDIA, Corporation.  All rights reserved.      *|
|*                                                                           *|
|*     NVIDIA, CORPORATION MAKES NO REPRESENTATION ABOUT THE SUITABILITY     *|
|*     OF  THIS SOURCE  CODE  FOR ANY PURPOSE.  IT IS  PROVIDED  "AS IS"     *|
|*     WITHOUT EXPRESS OR IMPLIED WARRANTY OF ANY KIND.  NVIDIA, CORPOR-     *|
|*     ATION DISCLAIMS ALL WARRANTIES  WITH REGARD  TO THIS SOURCE CODE,     *|
|*     INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY, NONINFRINGE-     *|
|*     MENT,  AND FITNESS  FOR A PARTICULAR PURPOSE.   IN NO EVENT SHALL     *|
|*     NVIDIA, CORPORATION  BE LIABLE FOR ANY SPECIAL,  INDIRECT,  INCI-     *|
|*     DENTAL, OR CONSEQUENTIAL DAMAGES,  OR ANY DAMAGES  WHATSOEVER RE-     *|
|*     SULTING FROM LOSS OF USE,  DATA OR PROFITS,  WHETHER IN AN ACTION     *|
|*     OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,  ARISING OUT OF     *|
|*     OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOURCE CODE.     *|
|*                                                                           *|
|*     U.S. Government  End  Users.   This source code  is a "commercial     *|
|*     item,"  as that  term is  defined at  48 C.F.R. 2.101 (OCT 1995),     *|
|*     consisting  of "commercial  computer  software"  and  "commercial     *|
|*     computer  software  documentation,"  as such  terms  are  used in     *|
|*     48 C.F.R. 12.212 (SEPT 1995)  and is provided to the U.S. Govern-     *|
|*     ment only as  a commercial end item.   Consistent with  48 C.F.R.     *|
|*     12.212 and  48 C.F.R. 227.7202-1 through  227.7202-4 (JUNE 1995),     *|
|*     all U.S. Government End Users  acquire the source code  with only     *|
|*     those rights set forth herein.                                        *|
|*                                                                           *|
 \***************************************************************************/

#include <stdlib.h>

/*
 * X includes.
 */
#include "X.h"
#include "Xproto.h"
#include "windowstr.h"
/*
 * Mesa includes.
 */
#include "context.h"
#include "depth.h"
#include "macros.h"
#include "texstate.h"
#include "triangle.h"
#include "vb.h"
#include "types.h"
/*
 * GLX includes.
 */
#include "xsmesaP.h"
#include "glx_log.h"
/*
 * Riva includes.
 */
#include "riva_glx.h"
/*
 * Texture heap.
 */
RIVA_TEX_HEAP rivaTexHeap = {0};
/*
 * Inner loop macros for converting OpenGL texel formats to RIVA formats.
 */
#define UV_ARGB_LOOP                                                        \
for (j = ttop, v = vtop; j < theight; j += tinc, v = RIVA_V_INC(v))         \
{                                                                           \
    for (i = sleft, u = uleft; i < swidth; i += sinc2, u = RIVA_U_INC2(u))  \
    {                                                                       \
        ii = (i >> TEX_FIX_SCALE) + img->Border;                            \
        jj = (j >> TEX_FIX_SCALE) + img->Border;                            \
        FETCH_TEXEL;                                                        \
        tex0 = ((alpha & 0xF0) << 8)                                        \
             | ((red   & 0xF0) << 4)                                        \
             | ((green & 0xF0))                                             \
             | ((blue  & 0xF0) >> 4);                                       \
        ii = ((i + sinc) >> TEX_FIX_SCALE) + img->Border;                   \
        FETCH_TEXEL;                                                        \
        tex1 = ((alpha & 0xF0) << 8)                                        \
             | ((red   & 0xF0) << 4)                                        \
             | ((green & 0xF0))                                             \
             | ((blue  & 0xF0) >> 4);                                       \
        ptex = (unsigned long *)(pdst + RIVA_UV_OFFSET(u, v));              \
       *ptex = tex0 | (tex1 << 16);                                         \
    }                                                                       \
}                                                                           \
texFormat = 0x00200000
#define UV_RGB_LOOP                                                         \
for (j = ttop, v = vtop; j < theight; j += tinc, v = RIVA_V_INC(v))         \
{                                                                           \
    for (i = sleft, u = uleft; i < swidth; i += sinc2, u = RIVA_U_INC2(u))  \
    {                                                                       \
        ii = (i >> TEX_FIX_SCALE) + img->Border;                            \
        jj = (j >> TEX_FIX_SCALE) + img->Border;                            \
        FETCH_TEXEL;                                                        \
        tex0 = ((red   & 0xF8) << 8)                                        \
             | ((green & 0xFC) << 3)                                        \
             | ((blue  & 0xF8) >> 3);                                       \
        ii = ((i + sinc) >> TEX_FIX_SCALE) + img->Border;                   \
        FETCH_TEXEL;                                                        \
        tex1 = ((red   & 0xF8) << 8)                                        \
             | ((green & 0xFC) << 3)                                        \
             | ((blue  & 0xF8) >> 3);                                       \
        ptex = (unsigned long *)(pdst + RIVA_UV_OFFSET(u, v));              \
       *ptex = tex0 | (tex1 << 16);                                         \
    }                                                                       \
}                                                                           \
texFormat = 0x00300000

/*
 * Debugging help.
 */
#if 0
static void CHECK_TEX_HEAP_LISTS(void)
{
    int i, j;

    /*
     * Check free list.
     */
    j = 0;
    for (i = rivaTexHeap.Free; i != NULL_INDEX; i = rivaTexHeap.Blocks[i].Next)
    {
        j += rivaTexHeap.Blocks[i].Size;
        if (rivaTexHeap.Blocks[i].Owner)
        {
            ErrorF("  Free block #%d: prev:%d next:%d size:%d\n", i, rivaTexHeap.Blocks[i].Prev,rivaTexHeap.Blocks[i].Next,rivaTexHeap.Blocks[i].Size);
            ErrorF("!!Free block has owner!!\n");
            exit (1);
        }
        if (rivaTexHeap.Blocks[i].Size == rivaTexHeap.Blocks[i].Next - i)
        {
            ErrorF("  Free block #%d: prev:%d next:%d size:%d\n", i, rivaTexHeap.Blocks[i].Prev,rivaTexHeap.Blocks[i].Next,rivaTexHeap.Blocks[i].Size);
            ErrorF("!!Free block adjacent to another free block!!\n");
            exit (1);
        }
        if ((rivaTexHeap.Blocks[i].Prev != NULL_INDEX) && (rivaTexHeap.Blocks[i].Prev >= i))
        {
            ErrorF("  Free block #%d: prev:%d next:%d size:%d\n", i, rivaTexHeap.Blocks[i].Prev,rivaTexHeap.Blocks[i].Next,rivaTexHeap.Blocks[i].Size);
            ErrorF("!!Free block bad prev link!!\n");
            exit (1);
        }
        if ((rivaTexHeap.Blocks[i].Next != NULL_INDEX) && (rivaTexHeap.Blocks[i].Next <= i))
        {
            ErrorF("  Free block #%d: prev:%d next:%d size:%d\n", i, rivaTexHeap.Blocks[i].Prev,rivaTexHeap.Blocks[i].Next,rivaTexHeap.Blocks[i].Size);
            ErrorF("!!Free block bad next link!!\n");
            exit (1);
        }
    }
    if (j != rivaTexHeap.FreeBlocks)
    {
        ErrorF("!!Inconsistent Free count  calc:%d tex heap free:%d!!\n", j, rivaTexHeap.FreeBlocks);
        exit (1);
    }
    for (i = rivaTexHeap.Head; i != NULL_INDEX; i = rivaTexHeap.Blocks[i].Next)
    {
        if (!rivaTexHeap.Blocks[i].Owner)
        {
            ErrorF("  Alloc block #%d: owner:%x prev:%d next:%d size:%d\n", i, rivaTexHeap.Blocks[i].Owner,
               rivaTexHeap.Blocks[i].Prev,rivaTexHeap.Blocks[i].Next,rivaTexHeap.Blocks[i].Size);
            ErrorF("!!Alloc block has no owner!!\n");
            exit (1);
        }
        if (((RIVA_TEX_BLOCK *)rivaTexHeap.Blocks[i].Owner->DriverData)->Index != i)
        {
            ErrorF("  Alloc block #%d: owner:%x prev:%d next:%d size:%d\n", i, rivaTexHeap.Blocks[i].Owner,
               rivaTexHeap.Blocks[i].Prev,rivaTexHeap.Blocks[i].Next,rivaTexHeap.Blocks[i].Size);
            ErrorF("!!Alloc block has incorrect owner!!\n");
            exit (1);
        }
        if ((rivaTexHeap.Blocks[i].Prev != NULL_INDEX) && (rivaTexHeap.Blocks[i].Prev >= i))
        {
            ErrorF("  Alloc block #%d: owner:%x prev:%d next:%d size:%d\n", i, rivaTexHeap.Blocks[i].Owner,
               rivaTexHeap.Blocks[i].Prev,rivaTexHeap.Blocks[i].Next,rivaTexHeap.Blocks[i].Size);
            ErrorF("!!Alloc block bad prev link!!\n");
            exit (1);
        }
        if ((rivaTexHeap.Blocks[i].Next != NULL_INDEX) && (rivaTexHeap.Blocks[i].Next <= i))
        {
            ErrorF("  Alloc block #%d: owner:%x prev:%d next:%d size:%d\n", i, rivaTexHeap.Blocks[i].Owner,
               rivaTexHeap.Blocks[i].Prev,rivaTexHeap.Blocks[i].Next,rivaTexHeap.Blocks[i].Size);
            ErrorF("!!Alloc block bad next link!!\n");
            exit (1);
        }
        for (j = rivaTexHeap.HeadLRU; j != rivaTexHeap.TailLRU; j = rivaTexHeap.Blocks[j].NextLRU)
        {
            if (j == i)
                break;
        }
        if (j != i)
        {
            ErrorF("  Alloc block #%d: owner:%x prev:%d next:%d size:%d\n", i, rivaTexHeap.Blocks[i].Owner,
                rivaTexHeap.Blocks[i].Prev,rivaTexHeap.Blocks[i].Next,rivaTexHeap.Blocks[i].Size);
            ErrorF("!!Alloc block not in LRU list!!\n");
            exit (1);
        }
    }
    for (i = rivaTexHeap.Tail; i != NULL_INDEX; i = rivaTexHeap.Blocks[i].Prev)
    {
        if (!rivaTexHeap.Blocks[i].Owner)
        {
            ErrorF("  Alloc block #%d: owner:%x prev:%d next:%d size:%d\n", i, rivaTexHeap.Blocks[i].Owner,
                   rivaTexHeap.Blocks[i].Prev,rivaTexHeap.Blocks[i].Next,rivaTexHeap.Blocks[i].Size);
            ErrorF("!!Alloc block has no owner!!\n");
            exit (1);
        }
        if (((RIVA_TEX_BLOCK *)rivaTexHeap.Blocks[i].Owner->DriverData)->Index != i)
        {
            ErrorF("  Alloc block #%d: owner:%x prev:%d next:%d size:%d\n", i, rivaTexHeap.Blocks[i].Owner,
                   rivaTexHeap.Blocks[i].Prev,rivaTexHeap.Blocks[i].Next,rivaTexHeap.Blocks[i].Size);
            ErrorF("!!Alloc block has incorrect owner!!\n");
            exit (1);
        }
        if ((rivaTexHeap.Blocks[i].Prev != NULL_INDEX) && (rivaTexHeap.Blocks[i].Prev >= i))
        {
            ErrorF("  Alloc block #%d: owner:%x prev:%d next:%d size:%d\n", i, rivaTexHeap.Blocks[i].Owner,
                   rivaTexHeap.Blocks[i].Prev,rivaTexHeap.Blocks[i].Next,rivaTexHeap.Blocks[i].Size);
            ErrorF("!!Alloc block bad prev link!!\n");
            exit (1);
        }
        if ((rivaTexHeap.Blocks[i].Next != NULL_INDEX) && (rivaTexHeap.Blocks[i].Next <= i))
        {
            ErrorF("  Alloc block #%d: owner:%x prev:%d next:%d size:%d\n", i, rivaTexHeap.Blocks[i].Owner,
                   rivaTexHeap.Blocks[i].Prev,rivaTexHeap.Blocks[i].Next,rivaTexHeap.Blocks[i].Size);
            ErrorF("!!Alloc block bad next link!!\n");
            exit (1);
        }
    }
    for (i = rivaTexHeap.HeadLRU; i != NULL_INDEX; i = rivaTexHeap.Blocks[i].NextLRU)
    {
        if (!rivaTexHeap.Blocks[i].Owner)
        {
            ErrorF("  LRU block #%d: owner:%x prev:%d next:%d size:%d\n", i, rivaTexHeap.Blocks[i].Owner,
                   rivaTexHeap.Blocks[i].PrevLRU,rivaTexHeap.Blocks[i].NextLRU,rivaTexHeap.Blocks[i].Size);
            ErrorF("!!LRU block has no owner!!\n");
            exit (1);
        }
        if (((RIVA_TEX_BLOCK *)rivaTexHeap.Blocks[i].Owner->DriverData)->Index != i)
        {
            ErrorF("  LRU block #%d: owner:%x prev:%d next:%d size:%d\n", i, rivaTexHeap.Blocks[i].Owner,
                   rivaTexHeap.Blocks[i].PrevLRU,rivaTexHeap.Blocks[i].NextLRU,rivaTexHeap.Blocks[i].Size);
            ErrorF("!!Alloc block has incorrect owner!!\n");
            exit (1);
        }
    }
    for (i = rivaTexHeap.TailLRU; i != NULL_INDEX; i = rivaTexHeap.Blocks[i].PrevLRU)
    {
        if (!rivaTexHeap.Blocks[i].Owner)
        {
            ErrorF("  LRU block #%d: owner:%x prev:%d next:%d size:%d\n", i, rivaTexHeap.Blocks[i].Owner,
                   rivaTexHeap.Blocks[i].PrevLRU,rivaTexHeap.Blocks[i].NextLRU,rivaTexHeap.Blocks[i].Size);
            ErrorF("!!LRU block has no owner!!\n");
            exit (1);
        }
        if (((RIVA_TEX_BLOCK *)rivaTexHeap.Blocks[i].Owner->DriverData)->Index != i)
        {
            ErrorF("  LRU block #%d: owner:%x prev:%d next:%d size:%d\n", i, rivaTexHeap.Blocks[i].Owner,
                   rivaTexHeap.Blocks[i].PrevLRU,rivaTexHeap.Blocks[i].NextLRU,rivaTexHeap.Blocks[i].Size);
            ErrorF("!!Alloc block has incorrect owner!!\n");
            exit (1);
        }
    }
}
#else
#define CHECK_TEX_HEAP_LISTS()
#endif    
/*
 * Move texels into texture buffer.
 */
static unsigned long RivaLoadTexels
(
    GLcontext                      *ctx,
    const struct gl_texture_image  *img,
    struct       gl_texture_object *tObj,
    unsigned short                 *pdst,
    unsigned                        left,
    unsigned                        top,
    unsigned                        width,
    unsigned                        height
)
{
    unsigned ttop, theight, tinc, vtop, sleft, swidth, sinc, sinc2, uleft;
    unsigned texFormat, i, ii, j, jj, u, v, scale, right;
    unsigned long *ptex, tex0, tex1;
    GLubyte  red, green, blue, alpha;
    GLint    iwidth   = img->Width;   /* Includes border width */
    GLint    iwidth2  = img->Width2;  /* Power of 2 width      */
    GLint    iheight2 = img->Height2; /* Power of 2 height     */
    GLubyte *texture  = img->Data;    /* Texel data            */

    /*
     * Force horizontal alignment to even texels so the inner loop can go at 2 texels per transfer.
     */
    right   = left + width;
    if (left  & 0x01) left--;
    if (right & 0x01) right++;
    width   = right - left;
    /*
     * RIVA 128 only accesses square textures.  Must resample the source texture if
     * it isn't square. This algorithm could be improved to do some type of
     * filtering when scaling.
     */
    vtop    = top;
    uleft   = left;
    ttop    = top    << TEX_FIX_SCALE;
    theight = height << TEX_FIX_SCALE;
    sleft   = left   << TEX_FIX_SCALE;
    swidth  = width  << TEX_FIX_SCALE;
    tinc    = sinc = 1  << TEX_FIX_SCALE;
    sinc2   =        2  << TEX_FIX_SCALE;
    if (iheight2 > iwidth2)
    {
        /*
         * Scale width.
         */
        scale   = img->HeightLog2 - img->WidthLog2;
        sinc  >>= scale;
        sinc2 >>= scale;
        uleft <<= scale;
    }
    else if (iheight2 < iwidth2)
    {
        /*
         * Scale height.
         */
        scale  = img->WidthLog2 - img->HeightLog2;
        tinc >>= scale;
        vtop <<= scale;
    }
    /*
     * Make sure there is texture data to convert.
     */
    if (texture)
    {
        /*
         * Expand U and V bits for swizzling.
         */
        if (vtop)
            vtop = ((vtop & 0x0001) << 1)
                 | ((vtop & 0x0002) << 2)
                 | ((vtop & 0x0004) << 3)
                 | ((vtop & 0x0008) << 4)
                 | ((vtop & 0x0010) << 5)
                 | ((vtop & 0x0020) << 6)
                 | ((vtop & 0x0040) << 7)
                 | ((vtop & 0x0080) << 8)
                 | ((vtop & 0x0100) << 9)
                 | ((vtop & 0x0200) << 10)
                 | ((vtop & 0x0400) << 11);
        if (uleft)
            uleft = ((uleft & 0x0001) << 0)
                  | ((uleft & 0x0002) << 1)
                  | ((uleft & 0x0004) << 2)
                  | ((uleft & 0x0008) << 3)
                  | ((uleft & 0x0010) << 4)
                  | ((uleft & 0x0020) << 5)
                  | ((uleft & 0x0040) << 6)
                  | ((uleft & 0x0080) << 7)
                  | ((uleft & 0x0100) << 8)
                  | ((uleft & 0x0200) << 9)
                  | ((uleft & 0x0400) << 10);
        /*
         * Convert source texels into RIVA ARGB_4444 or RGB_565.
         */
        switch (img->Format)
        {
            case GL_COLOR_INDEX:
                {
                    int      index;
                    GLubyte *palette;
    
                    if (ctx && ctx->Texture.SharedPalette)
                        palette = ctx->Texture.Palette;
                    else
                        palette = tObj->Palette;
                    switch (tObj->PaletteFormat)
                    {
                        case GL_ALPHA:
                            red   = 
                            green = 
                            blue  = 0xFF;
                            #define FETCH_TEXEL                         \
                            index = texture[jj * iwidth + ii];          \
                            alpha = tObj->Palette[index];
                            UV_ARGB_LOOP;
                            #undef FETCH_TEXEL
                            break;
                        case GL_LUMINANCE:
                        case GL_INTENSITY:
                            #define FETCH_TEXEL                         \
                            index = texture[jj * iwidth + ii];          \
                            red   =                                     \
                            green =                                     \
                            blue  = palette[index];
                            UV_RGB_LOOP;
                            #undef FETCH_TEXEL
                            break;
                        case GL_LUMINANCE_ALPHA:
                            #define FETCH_TEXEL                         \
                            index = texture[jj * iwidth + ii] << 1;     \
                            red   =                                     \
                            green =                                     \
                            blue  = palette[index + 0];                 \
                            alpha = palette[index + 1];
                            UV_ARGB_LOOP;
                            #undef FETCH_TEXEL
                            break;
                        case GL_RGB:
                            #define FETCH_TEXEL                         \
                            index = texture[jj * iwidth + ii] * 3;      \
                            red   = palette[index + 0];                 \
                            green = palette[index + 1];                 \
                            blue  = palette[index + 2];
                            UV_RGB_LOOP;
                            #undef FETCH_TEXEL
                            break;
                        case GL_RGBA:
                            #define FETCH_TEXEL                         \
                            index = texture[jj * iwidth + ii] << 2;     \
                            red   = palette[index + 0];                 \
                            green = palette[index + 1];                 \
                            blue  = palette[index + 2];                 \
                            alpha = palette[index + 3];
                            UV_ARGB_LOOP;
                            #undef FETCH_TEXEL
                            break;
                        default:
                            red   = 
                            green = 
                            blue  = 0xFF;
                            #define FETCH_TEXEL
                            UV_RGB_LOOP;
                            #undef FETCH_TEXEL
                    }
                }
                break;
            case GL_ALPHA:
                red   = 
                green = 
                blue  = 0xFF;
                #define FETCH_TEXEL                                     \
                alpha = texture[jj * iwidth + ii];
                UV_ARGB_LOOP;
                #undef FETCH_TEXEL
                break;
            case GL_LUMINANCE:
            case GL_INTENSITY:
                #define FETCH_TEXEL                                     \
                red   =                                                 \
                green =                                                 \
                blue  = texture[jj * iwidth + ii];
                UV_RGB_LOOP;
                #undef FETCH_TEXEL
                break;
            case GL_LUMINANCE_ALPHA:
                #define FETCH_TEXEL                                     \
                red   =                                                 \
                green =                                                 \
                blue  = texture[((jj * iwidth + ii) << 1) + 0];         \
                alpha = texture[((jj * iwidth + ii) << 1) + 1];
                UV_ARGB_LOOP;
                #undef FETCH_TEXEL
                break;
            case GL_RGB:
                #define FETCH_TEXEL                                     \
                red   = texture[(jj * iwidth + ii) * 3 + 0];            \
                green = texture[(jj * iwidth + ii) * 3 + 1];            \
                blue  = texture[(jj * iwidth + ii) * 3 + 2];
                UV_RGB_LOOP;
                #undef FETCH_TEXEL
                break;
            case GL_RGBA:
                #define FETCH_TEXEL                                     \
                red   = texture[((jj * iwidth + ii) << 2) + 0];         \
                green = texture[((jj * iwidth + ii) << 2) + 1];         \
                blue  = texture[((jj * iwidth + ii) << 2) + 2];         \
                alpha = texture[((jj * iwidth + ii) << 2) + 3];
                UV_ARGB_LOOP;
                #undef FETCH_TEXEL
                break;
            default:
                red   = 
                green = 
                blue  = 0xFF;
                #define FETCH_TEXEL
                UV_RGB_LOOP;
                #undef FETCH_TEXEL
        }
    }
    else
    {
        /*
         * This must be a texture place holder.  Just give it a dummy format.
         */
        texFormat = 0x00000000;
    }
    /*
     * Return the texture format and log size. Must be at least 4x4 for RIVA 128.
     */
    if (img->MaxLog2 < 2)
        return (texFormat | 0x22000000);
    else
        return (texFormat | (img->MaxLog2 << 28) | (img->MaxLog2 << 24));
}
/*
 * Allocate a texture in the heap. Kick out older textures if needed.
 */
static int RivaAllocateTexture
(
    GLcontext                 *ctx,
    struct  gl_texture_object *tObj
)
{
    unsigned level, mipSize, totalSize, BlockSize, i, j;
    struct   gl_texture_image *img = tObj->Image[tObj->BaseLevel];

    /*
     * Major sanity checks first.
     */
    if (tObj->DriverData)
        return (NULL_INDEX);
    /*
     * Calculate the memory requirements for a texture loaded into the texture buffer.
     * Determine the # of 8K blocks used be the texture, including mipmaps.
     */
    if (!img)
        return (NULL_INDEX);
    /*
     * Find offset of level from base.
     */
    mipSize    = 2 << (max(img->HeightLog2, img->WidthLog2) * 2);
    totalSize  = 0;
    for (level = tObj->BaseLevel; (level <= tObj->P) && (tObj->Image[level]); level++)
    {
        totalSize += mipSize;
        mipSize  >>= 2;
    }
    BlockSize = (totalSize + RIVA_TEX_BLOCK_SIZE - 1) / RIVA_TEX_BLOCK_SIZE;
    if ((BlockSize > (rivaTexHeap.Total - 2)) || (BlockSize == 0))
        return (NULL_INDEX);
    /*
     * Check for free blocks before doing a search.
     */
    while (BlockSize > rivaTexHeap.FreeBlocks)
        RivaDeleteTexture(ctx, rivaTexHeap.Blocks[rivaTexHeap.TailLRU].Owner);
    /*
     * Now start looking for a free section big enough.
     * Perhaps there should be a sorted-by-size free list.
     */
    do
    {
        for (i = rivaTexHeap.Free;
            (i != NULL_INDEX) && (BlockSize > rivaTexHeap.Blocks[i].Size);
             i = rivaTexHeap.Blocks[i].Next);
        if (i == NULL_INDEX)
            RivaDeleteTexture(ctx, rivaTexHeap.Blocks[rivaTexHeap.TailLRU].Owner);
    } while (i == NULL_INDEX);
    /*
     * Whew. Found something.
     */
    if (BlockSize == rivaTexHeap.Blocks[i].Size)
    {
        /*
         * Exact match. Remove entire block from free list.
         */
        if (rivaTexHeap.Blocks[i].Next != NULL_INDEX)
            rivaTexHeap.Blocks[rivaTexHeap.Blocks[i].Next].Prev = rivaTexHeap.Blocks[i].Prev;
        if (rivaTexHeap.Blocks[i].Prev != NULL_INDEX)
            rivaTexHeap.Blocks[rivaTexHeap.Blocks[i].Prev].Next = rivaTexHeap.Blocks[i].Next;
        else
            rivaTexHeap.Free                                    = rivaTexHeap.Blocks[i].Next;
    }
    else
    {
        /*
         * Break out a chunk.
         */
        j = i + BlockSize;
        rivaTexHeap.Blocks[j].Owner = NULL;
        rivaTexHeap.Blocks[j].Size  = rivaTexHeap.Blocks[i].Size - BlockSize;
        rivaTexHeap.Blocks[j].Next  = rivaTexHeap.Blocks[i].Next;
        rivaTexHeap.Blocks[j].Prev  = rivaTexHeap.Blocks[i].Prev;
        /*
         * Link new, smaller free block back into free list.
         */
        if (rivaTexHeap.Blocks[i].Next != NULL_INDEX)
            rivaTexHeap.Blocks[rivaTexHeap.Blocks[i].Next].Prev = j;
        if (rivaTexHeap.Blocks[i].Prev != NULL_INDEX)
            rivaTexHeap.Blocks[rivaTexHeap.Blocks[i].Prev].Next = j;
        else
            rivaTexHeap.Free                                    = j;
    }
    /*
     * Insert into allocated list.
     */
    if (rivaTexHeap.Head == NULL_INDEX)
    {
        /*
         * First allocated block.
         */
        rivaTexHeap.Head           = rivaTexHeap.Tail           = i;
        rivaTexHeap.Blocks[i].Next = rivaTexHeap.Blocks[i].Prev = NULL_INDEX;
    }
    else if (i > rivaTexHeap.Tail)
    {
        /*
         * After the end.
         */
        rivaTexHeap.Blocks[rivaTexHeap.Tail].Next = i;
        rivaTexHeap.Blocks[i].Prev                = rivaTexHeap.Tail;
        rivaTexHeap.Blocks[i].Next                = NULL_INDEX;
        rivaTexHeap.Tail                          = i;
    }
    else if (i < rivaTexHeap.Head)
    {
        /*
         * Before the beginning.
         */
        rivaTexHeap.Blocks[rivaTexHeap.Head].Prev = i;
        rivaTexHeap.Blocks[i].Prev                = NULL_INDEX;
        rivaTexHeap.Blocks[i].Next                = rivaTexHeap.Head;
        rivaTexHeap.Head                          = i;
    }
    else
    {
        /*
         * Just look past the old free block to find an allocated block.
         */
        j                                                   = i + rivaTexHeap.Blocks[i].Size;
        rivaTexHeap.Blocks[i].Prev                          = rivaTexHeap.Blocks[j].Prev;
        rivaTexHeap.Blocks[i].Next                          = j;
        rivaTexHeap.Blocks[rivaTexHeap.Blocks[j].Prev].Next = i;
        rivaTexHeap.Blocks[j].Prev                          = i;  
    }
    /*
     * Update free amount.
     */
    rivaTexHeap.FreeBlocks -= BlockSize;
    /*
     * Insert into head of LRU.
     */
    if (rivaTexHeap.HeadLRU != NULL_INDEX)
        rivaTexHeap.Blocks[rivaTexHeap.HeadLRU].PrevLRU = i;
    else
        rivaTexHeap.TailLRU                             = i;
    rivaTexHeap.Blocks[i].PrevLRU = NULL_INDEX;
    rivaTexHeap.Blocks[i].NextLRU = rivaTexHeap.HeadLRU;
    rivaTexHeap.HeadLRU           = i;
    /*
     * Complete allocation of block.
     */
    rivaTexHeap.Blocks[i].maxLevel       = level - 1;
    rivaTexHeap.Blocks[i].baseWidthLog2  = img->WidthLog2;
    rivaTexHeap.Blocks[i].baseHeightLog2 = img->HeightLog2;
    rivaTexHeap.Blocks[i].Size           = BlockSize;
    rivaTexHeap.Blocks[i].Owner          = tObj;
    rivaTexHeap.Blocks[i].Format         = 0x00000000;
    rivaTexHeap.Blocks[i].sAdjust        = 0.5F / (float)img->Width2;
    rivaTexHeap.Blocks[i].tAdjust        = 0.5F / (float)img->Height2;
    tObj->DriverData                     = (void *)&rivaTexHeap.Blocks[i];
    CHECK_TEX_HEAP_LISTS();
    return (i);
}
/*
 * Free a texture in the texture buffer heap.
 */
void RivaDeleteTexture
(
    GLcontext                *ctx,
    struct gl_texture_object *tObj
)
{
    int             i, prev, next;
    RIVA_TEX_BLOCK *pblk;

    /*
     * Make sure the texture is still resident in the texture buffer.
     */
    if ((pblk = (RIVA_TEX_BLOCK *)tObj->DriverData))
    {
        if (pblk->Owner != tObj)
        {
            /*
             * This isn't good.  Somehow things have gotten out of sync.  Bail.
             */
            ErrorF("Inconsistent texture owner in RivaDeleteTexture!\n");
        }
        tObj->DriverData = NULL;
        pblk->Owner      = NULL;
        /*
         * Remove this block from the allocated list.
         */
        if (pblk->Next != NULL_INDEX)
            rivaTexHeap.Blocks[pblk->Next].Prev = pblk->Prev;
        else
            rivaTexHeap.Tail                    = pblk->Prev;
        if (pblk->Prev != NULL_INDEX)
            rivaTexHeap.Blocks[pblk->Prev].Next = pblk->Next;
        else
            rivaTexHeap.Head                    = pblk->Next;
        /*
         * Remove from the LRU list.
         */
        if (pblk->NextLRU != NULL_INDEX)
            rivaTexHeap.Blocks[pblk->NextLRU].PrevLRU = pblk->PrevLRU;
        else
            rivaTexHeap.TailLRU                       = pblk->PrevLRU;
        if (pblk->PrevLRU != NULL_INDEX)
            rivaTexHeap.Blocks[pblk->PrevLRU].NextLRU = pblk->NextLRU;
        else
            rivaTexHeap.HeadLRU                       = pblk->NextLRU;
        /*
         * Update free amount.
         */
        rivaTexHeap.FreeBlocks += pblk->Size; 
        /*
         * Insert into the free list.
         */
        if (rivaTexHeap.Free != NULL_INDEX)
        {
            /*
             * Look for adjoining free areas.
             */
            next = pblk->Index + pblk->Size;
            prev = pblk->Prev;
            if ((pblk->Next == NULL_INDEX && next < rivaTexHeap.Total)
             || (pblk->Next != NULL_INDEX && (pblk->Next > next)))
            {
                /*
                 * Free blocks directly following. Prepend to it.
                 */
                prev        = rivaTexHeap.Blocks[next].Prev; 
                pblk->Size += rivaTexHeap.Blocks[next].Size;
                pblk->Next  = rivaTexHeap.Blocks[next].Next;
                pblk->Prev  = prev;
                if ((prev + rivaTexHeap.Blocks[prev].Size) == pblk->Index)
                {
                    /*
                     * Preceding free block adjoins this.  Combine the two.
                     */
                    if (rivaTexHeap.Blocks[next].Next != NULL_INDEX)
                        rivaTexHeap.Blocks[rivaTexHeap.Blocks[next].Next].Prev = prev;
                    rivaTexHeap.Blocks[prev].Next  = pblk->Next;
                    rivaTexHeap.Blocks[prev].Size += pblk->Size;
                }
                else
                {
                    /*
                     * Link into the chain after the preceding free block.
                     */
                    if (rivaTexHeap.Blocks[next].Next != NULL_INDEX)
                        rivaTexHeap.Blocks[rivaTexHeap.Blocks[next].Next].Prev = pblk->Index;
                    rivaTexHeap.Blocks[prev].Next = pblk->Index;
                }
            }
            else if ((prev == NULL_INDEX && pblk->Index >  rivaTexHeap.Free)
                 || ((prev != NULL_INDEX && pblk->Index != rivaTexHeap.Blocks[prev].Index + rivaTexHeap.Blocks[prev].Size)))
            {
                /*
                 * Free block before.  Append to it.
                 */
                prev = (prev == NULL_INDEX) ? rivaTexHeap.Free : rivaTexHeap.Blocks[prev].Index + rivaTexHeap.Blocks[prev].Size;
                rivaTexHeap.Blocks[prev].Size += pblk->Size;
            }
            /*
             * If the block wasn't absorbed into adjoining blocks, insert it now.
             */
            else 
            {
                /*
                 * Do a (groan) linear search to insert it.
                 */
                for (i = rivaTexHeap.Free;
                    (i < pblk->Index) && (rivaTexHeap.Blocks[i].Next != NULL_INDEX);
                     i = rivaTexHeap.Blocks[i].Next);
                if (i > pblk->Index)
                {
                    /*
                     * Insert before block i.
                     */
                    pblk->Prev                 = rivaTexHeap.Blocks[i].Prev;
                    pblk->Next                 = i;
                    rivaTexHeap.Blocks[i].Prev = pblk->Index;
                    if (pblk->Prev != NULL_INDEX)
                        rivaTexHeap.Blocks[pblk->Prev].Next = pblk->Index;
                }
                else
                {
                    /*
                     * Insert after block i.
                     */
                    pblk->Prev                 = i;
                    pblk->Next                 = NULL_INDEX;
                    rivaTexHeap.Blocks[i].Next = pblk->Index;
                }
            }
            /*
             * Check for head of list.
             */
            if (rivaTexHeap.Free > pblk->Index)
                rivaTexHeap.Free = pblk->Index;
        }
        else
        {
            /*
             * Free list was empty.  Make this the first entry.
             */
            pblk->Next       = NULL_INDEX;
            pblk->Prev       = NULL_INDEX;
            rivaTexHeap.Free = pblk->Index;
        }
        /*
         * Make sure we stop using this texture before loading new texels.
         */
        if (rivaContext.texObj == tObj)
        {
            rivaContext.texObj    = NULL;
            rivaContext.texOffset = rivaTexHeap.DefaultOffset;
            rivaContext.texFormat = 0x22300000;
            rivaReload3D          = 1;
            while (riva.Busy(&riva));
        }
    }
    CHECK_TEX_HEAP_LISTS();
}
/*
 * Load a complete texture and mipmaps into buffer.
 */
static GLboolean RivaLoadTexture
(
    GLcontext                *ctx,
    struct gl_texture_object *tObj
)
{
    RIVA_TEX_BLOCK *pblk;
    int             level, mipWidthLog2, mipHeightLog2;

    if ((!tObj->DriverData) && (RivaAllocateTexture(ctx, tObj) == NULL_INDEX))
        return (GL_FALSE);
    /*
     * Load all the mipmaps.
     */
    pblk = (RIVA_TEX_BLOCK *)tObj->DriverData;
    /*
     * Reset max and min mipmaps.
     */
    pblk->Format  = 0x00000000;
    mipWidthLog2  = pblk->baseWidthLog2;
    mipHeightLog2 = pblk->baseHeightLog2;
    for (level = tObj->BaseLevel; (level <= tObj->P) && (tObj->Image[level]); level++)
    {
        if (mipWidthLog2 == tObj->Image[level]->WidthLog2 && mipHeightLog2 == tObj->Image[level]->HeightLog2)
        {
            RivaTexSubImage(ctx, 
                            0, 
                            tObj, 
                            level,
                            0,
                            0,
                            tObj->Image[level]->Width,
                            tObj->Image[level]->Height,
                            tObj->Image[level]->IntFormat,
                            tObj->Image[level]);
            mipWidthLog2  -= 1;
            mipHeightLog2 -= 1;
        }
        else
            break;
    }
    return (pblk->Format ? GL_TRUE : GL_FALSE);
}
/*
 * Load a texture/subtexture into buffer.
 */
void RivaTexSubImage
(
    GLcontext   *ctx,
    GLenum       target,
    struct       gl_texture_object *tObj,
    GLint        level,
    GLint        xoffset,
    GLint        yoffset,
    GLint        width,
    GLint        height,
    GLint        internalFormat,
    const struct gl_texture_image *image
)
{
    RIVA_TEX_BLOCK *pblk;
    int             mipSize, levelOffset;
    unsigned        format;

    if ((pblk = (RIVA_TEX_BLOCK *)tObj->DriverData))
    {
        /*
         * Make sure space has been allocated for this level.
         */
        if (pblk->maxLevel < level)
        {
            /*
             * Re-allocate space for this texture. Yep, its recursive.
             */
            RivaDeleteTexture(ctx, tObj);
            RivaLoadTexture(ctx, tObj);
        }
        else
        {
            /*
             * Find offset of level from base.
             */
            for (levelOffset = 0, mipSize = 2 << (max(image->HeightLog2, image->WidthLog2) * 2);
                 level > tObj->BaseLevel;
                 level--)
            {
                mipSize    <<= 2;
                levelOffset |= mipSize;
            }
            /*
             * Make sure we stop using this texture before loading new texels.
             */
            if (rivaContext.texObj == tObj)
            {
                rivaReload3D = 1;
                while (riva.Busy(&riva));
            }
            /*
             * Move the texels into video memory.
             */
            format = RivaLoadTexels(ctx,
                                    image,
                                    tObj,
                                    (unsigned short *)(RIVA_TEX_BLOCK_ADDRESS(pblk->Index) + levelOffset),
                                    xoffset,
                                    yoffset,
                                    width,
                                    height);
            if (!pblk->Format)
                pblk->Format = format;
            /*
             * Set min and max mipmap sizes when they get loaded.
             */
            if ((pblk->Format & 0xF0000000) < (format & 0xF0000000))
                pblk->Format = (pblk->Format & 0x0FFFFFFF) | (format & 0xF0000000);
            if ((pblk->Format & 0x0F000000) > (format & 0x0F000000))
                pblk->Format = (pblk->Format & 0xF0FFFFFF) | (format & 0x0F000000);
            if (rivaTexHeap.HeadLRU != pblk->Index)
            {
                /*
                 * Remove from the LRU list.
                 */
                if (pblk->NextLRU != NULL_INDEX)
                    rivaTexHeap.Blocks[pblk->NextLRU].PrevLRU = pblk->PrevLRU;
                else
                    rivaTexHeap.TailLRU                       = pblk->PrevLRU;
                if (pblk->PrevLRU != NULL_INDEX)
                    rivaTexHeap.Blocks[pblk->PrevLRU].NextLRU = pblk->NextLRU;
                else
                    rivaTexHeap.HeadLRU                       = pblk->NextLRU;
                /*
                 * Insert into front of LRU.
                 */
                if (rivaTexHeap.HeadLRU != NULL_INDEX)
                    rivaTexHeap.Blocks[rivaTexHeap.HeadLRU].PrevLRU = pblk->Index;
                else
                    rivaTexHeap.TailLRU                             = pblk->Index;
                rivaTexHeap.Blocks[pblk->Index].PrevLRU = NULL_INDEX;
                rivaTexHeap.Blocks[pblk->Index].NextLRU = rivaTexHeap.HeadLRU;
                rivaTexHeap.HeadLRU                     = pblk->Index;
            }
        }
    }
}
void RivaTexImage
(
    GLcontext   *ctx,
    GLenum       target,
    struct       gl_texture_object *tObj,
    GLint        level,
    GLint        internalFormat,
    const struct gl_texture_image *image
)
{
    RIVA_TEX_BLOCK *pblk;
    int             mipWidthLog2, mipHeightLog2;

    if ((pblk = (RIVA_TEX_BLOCK *)tObj->DriverData))
    {
        /*
         * Check if this image matches the size previously allocated.
         */
        mipWidthLog2  = pblk->baseWidthLog2  >> (level - tObj->BaseLevel);
        mipHeightLog2 = pblk->baseHeightLog2 >> (level - tObj->BaseLevel);
        if (mipWidthLog2   == tObj->Image[level]->WidthLog2
         && mipHeightLog2  == tObj->Image[level]->HeightLog2
         && pblk->maxLevel == tObj->P)
        {
            RivaTexSubImage(ctx,
                            target,
                            tObj,
                            level,
                            0,
                            0,
                            image->Width,
                            image->Height,
                            internalFormat,
                            image);
        }
        else
        {
            /*
             * Delete texture in hardware and lazy load it later unless its the current texture.
             */
            if (rivaContext.texObj != tObj)
            {
                RivaDeleteTexture(ctx, tObj);
            }
            else
            {
                RivaDeleteTexture(ctx, tObj);
                RivaBindTexture(ctx, 0, tObj);
            }
        }
    }
}
/*
 * Bind texture to current context.
 */
void RivaBindTexture
(
    GLcontext                *ctx,
    GLenum                    target,
    struct gl_texture_object *tObj
)
{
    RIVA_TEX_BLOCK *pblk;

    /*
     * Load the texture if not already present.
     */
    if (!tObj || ((!tObj->DriverData) && (!RivaLoadTexture(ctx, tObj))))
    {
        /*
         * Set default texture if no texture images available yet.
         */
        rivaContext.texObj    = NULL;
        rivaContext.texOffset = rivaTexHeap.DefaultOffset;
        rivaContext.texFormat = 0x22300000;
        rivaContext.sAdjust   = 0.0F;
        rivaContext.tAdjust   = 0.0F;
    }
    else
    {
        /*
         * Set this as the active texture.
         */
        pblk                    = (RIVA_TEX_BLOCK *)tObj->DriverData;
        rivaContext.texObj      = tObj;    
        rivaContext.texOffset   = RIVA_TEX_BLOCK_OFFSET(pblk->Index);
        rivaContext.triControl &= 0xFFFFFF00;
        /*
         * Wrap/mirror/clamp.
         */
        rivaContext.triControl |= (tObj->WrapS == GL_CLAMP) ? 0x30 : 0x10;
        rivaContext.triControl |= (tObj->WrapT == GL_CLAMP) ? 0xC0 : 0x40;
        /*
         * Filtering. Make sure we adhere to OGL texture sampling with each filter mode.
         */
        if (tObj->MagFilter == GL_LINEAR)
        {
            rivaContext.triControl |= 0x0002;
            rivaContext.sAdjust     = pblk->sAdjust;
            rivaContext.tAdjust     = pblk->tAdjust;
        }
        else
        {
            rivaContext.sAdjust     = 0.0F;
            rivaContext.tAdjust     = 0.0F;
        }
        /*
         * Mipmapping.
         */
        if ((tObj->MinFilter == GL_NEAREST) || (tObj->MinFilter == GL_LINEAR))
            rivaContext.texFormat =  (pblk->Format & 0xF0FFFFFF)
                                  | ((pblk->Format & 0xF0000000) >> 4);
        else
            rivaContext.texFormat =   pblk->Format;
        /*
         * Update LRU list.
         */
        if (rivaTexHeap.HeadLRU != pblk->Index)
        {
            /*
             * Remove from the LRU list.
             */
            if (pblk->NextLRU != NULL_INDEX)
                rivaTexHeap.Blocks[pblk->NextLRU].PrevLRU = pblk->PrevLRU;
            else
                rivaTexHeap.TailLRU                       = pblk->PrevLRU;
            if (pblk->PrevLRU != NULL_INDEX)
                rivaTexHeap.Blocks[pblk->PrevLRU].NextLRU = pblk->NextLRU;
            else
                rivaTexHeap.HeadLRU                       = pblk->NextLRU;
            /*
             * Insert into front of LRU.
             */
            if (rivaTexHeap.HeadLRU != NULL_INDEX)
                rivaTexHeap.Blocks[rivaTexHeap.HeadLRU].PrevLRU = pblk->Index;
            else
                rivaTexHeap.TailLRU                             = pblk->Index;
            rivaTexHeap.Blocks[pblk->Index].PrevLRU = NULL_INDEX;
            rivaTexHeap.Blocks[pblk->Index].NextLRU = rivaTexHeap.HeadLRU;
            rivaTexHeap.HeadLRU                     = pblk->Index;
        }
    }
    /*
     * Make sure state gets reloaded.
     */
    rivaReload3D = 1;
}
/*
 * Release all textures (like for a mode switch).
 */
void RivaReleaseTextures
(
    void
)
{
    int i, next;

    /*
     * Run through all the loaded textures and unlink them from their texture objects.
     */
    if ((i = rivaTexHeap.Head) != NULL_INDEX)
    {
        do
        {
            next = rivaTexHeap.Blocks[i].Next;
            rivaTexHeap.Blocks[i].Owner->DriverData = NULL;
            i = next;
        } while (i != NULL_INDEX);
    }
    /*
     * Reset texture heap lists.
     */
    rivaTexHeap.FreeBlocks         = rivaTexHeap.Total - 2;
    rivaTexHeap.Free               = 2;
    rivaTexHeap.Head               = NULL_INDEX;
    rivaTexHeap.Tail               = NULL_INDEX;
    rivaTexHeap.HeadLRU            = NULL_INDEX;
    rivaTexHeap.TailLRU            = NULL_INDEX;
    rivaTexHeap.Blocks[2].Size     = rivaTexHeap.Total - 2;
    rivaTexHeap.Blocks[2].Next     = NULL_INDEX;
    rivaTexHeap.Blocks[2].Prev     = NULL_INDEX;
    rivaTexHeap.Blocks[2].NextLRU  = NULL_INDEX;
    rivaTexHeap.Blocks[2].PrevLRU  = NULL_INDEX;
    /*
     * Reset context.
     */
    rivaContext.texObj    = NULL;
    rivaContext.texOffset = rivaTexHeap.DefaultOffset;
    rivaContext.texFormat = 0x22300000;
}
/*
 * Initialize/Reinitialize texture buffer heap.
 *
 *  The texture heap is broken up into 8K sized blocks.  This is the minimum texture size
 *  allowed by OpenGL (64X64).  16 bit texels uses 8K minimum.  By pre-allocating the blocks
 *  now, alloc/free overhead is saved as well as reducing memory fragmentation.  Texture
 *  heaps are very dynamic.  I prefer avoiding generic memory managers for specialized cases.
 */
int RivaInitTextureHeap
(
    void
)
{
    int             texMemStart, texMemEnd, texBlkTotal, i, j, u, v;
    unsigned short *ptex;

    /*
     * Free any previously allocated heap blocks.
     */
    if (rivaTexHeap.Blocks)
        xfree (rivaTexHeap.Blocks);
    rivaTexHeap.Blocks = NULL;
    /*
     * How much memory do we have?
     */
    texMemStart = rivaBufferOffset[RIVA_TEXTURE_BUFFER] + 2 * RIVA_TEX_ALIGN;
    texMemEnd   = rivaBufferOffset[RIVA_DEPTH_BUFFER] > rivaBufferOffset[RIVA_TEXTURE_BUFFER]
                ? rivaBufferOffset[RIVA_DEPTH_BUFFER] : riva.RamAmountKBytes * 1024;
    texBlkTotal = (texMemEnd - texMemStart) / RIVA_TEX_BLOCK_SIZE;
    /*
     * No memory? No bueno.
     */
    if (texBlkTotal <= 0)
        return (GL_FALSE);
    /*
     * Set all the heap parameters. First two blocks are reserved for default and AA textures.
     */
    rivaTexHeap.Total           = texBlkTotal;
    rivaTexHeap.FreeBlocks      = texBlkTotal - 2;
    rivaTexHeap.Free            = 2;
    rivaTexHeap.Head            = NULL_INDEX;
    rivaTexHeap.Tail            = NULL_INDEX;
    rivaTexHeap.HeadLRU         = NULL_INDEX;
    rivaTexHeap.TailLRU         = NULL_INDEX;
    rivaTexHeap.Blocks          = (RIVA_TEX_BLOCK *)xalloc(sizeof(RIVA_TEX_BLOCK) * texBlkTotal);
    /*
     * Init and load default texture.
     */
    ptex = (unsigned short *)RIVA_TEX_BLOCK_ADDRESS(0);
    for (i = 0; i < 16; i++)
        ptex[i] = 0xFFFF;
    rivaTexHeap.DefaultOffset    = RIVA_TEX_BLOCK_OFFSET(0);
    rivaTexHeap.Blocks[0].Format = 0x22300000;
    /*
     * Init and load default AA texture.
     */
    ptex = (unsigned short *)RIVA_TEX_BLOCK_ADDRESS(1);
    for (j = v = 0; j < 16; j++, v = RIVA_V_INC(v))
        for (i = u = 0; i < 16; i++, u = RIVA_U_INC(u))
        {
            float          dist, alpha;
            unsigned short texel;

            dist  = (((float)j - 7.5)*((float)j - 7.5) + ((float)i - 7.5)*((float)i - 7.5)) / 64.0;
            if (dist < 0.0F) dist = 0.0F;
            if (dist > 1.0F) dist = 1.0F;
            alpha = (cos(sqrt(dist) * 3.141592654) * 0.5 + 0.55) * 15.0;
            RivaFloatToUInt(texel, alpha);
            ptex[RIVA_UV_OFFSET(u, v)] = (texel << 12) | 0x0FFF;
        }
    rivaTexHeap.DefaultAAOffset  = RIVA_TEX_BLOCK_OFFSET(1);
    rivaTexHeap.Blocks[1].Format = 0x44200000;
    /*
     * Init the block linked-list/array.
     */
    for (i = 2; i < texBlkTotal; i++)
    {
        rivaTexHeap.Blocks[i].Owner    = NULL;
        rivaTexHeap.Blocks[i].Priority = TEX_MIN_PRIORITY;
        rivaTexHeap.Blocks[i].Format   = 0x00000000;
        rivaTexHeap.Blocks[i].Index    = i;
    }
    rivaTexHeap.Blocks[2].Size     = texBlkTotal - 2;
    rivaTexHeap.Blocks[2].Next     = NULL_INDEX;
    rivaTexHeap.Blocks[2].Prev     = NULL_INDEX;
    rivaTexHeap.Blocks[2].NextLRU  = NULL_INDEX;
    rivaTexHeap.Blocks[2].PrevLRU  = NULL_INDEX;
    return (GL_TRUE);
}



