/*
 * GLX Hardware Device Driver for Matrox G200/G400
 * warp engine interface
 * Copyright (c) 1999 Ralph Giles
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * MATROX GRAPHICS INC., OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR 
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE 
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 * original by Ralph Giles <giles@ashlu.bc.ca> 1999 July 27
 * 
 */

#include "mgacommon.h"
#include "warp.h"	/* matrox microcode */
#include "mgawarp.h"
#include "os.h" /* ErrorF */
#include "mm.h"
#include "mgadma.h"
#include "mgadirect.h"
#include "mgalog.h"
#include "glx_config.h"
#include <assert.h>
#include <stdlib.h>
#include "mgalib.h"

#include "g200_mac.h"

#include "xsmesaP.h"

/* The warp microcode routines are defined separately in warp.h
 * these get loaded into the setup engine(s) to do the triangle
 * transforms. the extension specifies the enabled path:
 *
 *  t is for single-stage textured
 *  t2 is for two-stage textured (G400 only)
 *  g is for Gouraud color interpolation (always on)
 *  z for Z buffer interpolated
 *  s is for specular hilight
 *  a is for alpha blended (?)
 *  f is for fogged
 *
 *  fill in null values in the vertex structure for features you're
 *  not using.
 */


/*
  Bus mastering (3) or direct access warp (0)
*/
int mgaWarpWMISC;

/*
  Warp microcode stuff - The array is large because the ordering of
  the flags used to index it isn't appropriate - fix this.
*/
static struct {
  void *Physaddr;  /* Phys addr for bus mastering, zero for direct access */
  void *Virtaddr;  /* Virt addr for direct access, unused for bus mastering */
  int size;        /* only for direct access */
} mgaWarpIndex[0x80];

/* WAITFIFO() needs this, it's some internal XFree86 thing. */
extern Bool MGAUsePCIRetry;


/*
  a debug function
*/
void printWarp(mga_warp_vertex2 *v)
{
  mgaMsg(1, "Warp vector dump:\n");
  mgaMsg(1, "(%f, %f, %f, %f) - (%f, %f)\n", v->x, v->y, v->z, v->rhw, v->tu0,
	 v->tv0);
  mgaMsg(1, "(%d, %d, %d, %d) - (%d, %d, %d, %d)\n",
	 v->color.blue, v->color.green, v->color.red, v->color.alpha,
	 v->specular.blue, v->specular.green, v->specular.red,
	 v->specular.alpha);
}




/*
  This function puts commands in the DMA command buffer to render
  triangle vertexes that have been stored in secondary dma memory.
  It's called when buffers are flushed, and before any rasterization
  state changes.
*/
void mgaWarpFinishSerie( void ) {
	mgaUI32		*pos;
	mgaUI32		*start;
	
	start = mgaglx.warp_serieStart;
	
	/* get current secondary position */
	pos = dma_buffer->virtualAddress + dma_buffer->maxPrimaryDwords 
		+ dma_buffer->secondaryDwords;

	if ( !start || ( pos <= start ) ) {
  		return; /* Nothing to do */
	}
	mgaMsg(10, "Warp serie is being finished now (%d, %d)\n", start, pos);

	mgaglx.warp_serieStart = pos;

	/* this might cause an overflow flush, which will reset warp_serieStart again */
	mgaSecondaryDma( TT_VERTEX, start, pos - start );
}




/*
  Initialisation functions come below this line
*/


/*
  For debuging purposes
*/
void dumpWarpCodeOffsets(void)
{
  mgaMsg(1, "tgz     %p, %p, %d\n", mgaWarpIndex[MGA_WARP_TGZ].Physaddr,
	 mgaWarpIndex[MGA_WARP_TGZ].Virtaddr,
	 mgaWarpIndex[MGA_WARP_TGZ].size);
  mgaMsg(1, "tgza    %p, %p, %d\n", mgaWarpIndex[MGA_WARP_TGZA].Physaddr,
	 mgaWarpIndex[MGA_WARP_TGZA].Virtaddr,
	 mgaWarpIndex[MGA_WARP_TGZA].size);
  mgaMsg(1, "tgzaf   %p, %p, %d\n", mgaWarpIndex[MGA_WARP_TGZAF].Physaddr,
	 mgaWarpIndex[MGA_WARP_TGZAF].Virtaddr,
	 mgaWarpIndex[MGA_WARP_TGZAF].size);
  mgaMsg(1, "tgzf    %p, %p, %d\n", mgaWarpIndex[MGA_WARP_TGZF].Physaddr,
	 mgaWarpIndex[MGA_WARP_TGZF].Virtaddr,
	 mgaWarpIndex[MGA_WARP_TGZF].size);
  mgaMsg(1, "tgzs    %p, %p, %d\n", mgaWarpIndex[MGA_WARP_TGZS].Physaddr,
	 mgaWarpIndex[MGA_WARP_TGZS].Virtaddr,
	 mgaWarpIndex[MGA_WARP_TGZS].size);
  mgaMsg(1, "tgzsa   %p, %p, %d\n", mgaWarpIndex[MGA_WARP_TGZSA].Physaddr,
	 mgaWarpIndex[MGA_WARP_TGZSA].Virtaddr,
	 mgaWarpIndex[MGA_WARP_TGZSA].size);
  mgaMsg(1, "tgzsaf  %p, %p, %d\n", mgaWarpIndex[MGA_WARP_TGZSAF].Physaddr,
	 mgaWarpIndex[MGA_WARP_TGZSAF].Virtaddr,
	 mgaWarpIndex[MGA_WARP_TGZSAF].size);
  mgaMsg(1, "tgzsf   %p, %p, %d\n", mgaWarpIndex[MGA_WARP_TGZSF].Physaddr,
	 mgaWarpIndex[MGA_WARP_TGZSF].Virtaddr,
	 mgaWarpIndex[MGA_WARP_TGZSF].size);
if(MGA_IS_G400(MGAchipset)) {
  mgaMsg(1, "t2gz     %p, %p, %d\n", mgaWarpIndex[MGA_WARP_T2GZ].Physaddr,
	 mgaWarpIndex[MGA_WARP_T2GZ].Virtaddr,
	 mgaWarpIndex[MGA_WARP_T2GZ].size);
  mgaMsg(1, "t2gza    %p, %p, %d\n", mgaWarpIndex[MGA_WARP_T2GZA].Physaddr,
	 mgaWarpIndex[MGA_WARP_T2GZA].Virtaddr,
	 mgaWarpIndex[MGA_WARP_T2GZA].size);
  mgaMsg(1, "t2gzaf   %p, %p, %d\n", mgaWarpIndex[MGA_WARP_T2GZAF].Physaddr,
	 mgaWarpIndex[MGA_WARP_T2GZAF].Virtaddr,
	 mgaWarpIndex[MGA_WARP_T2GZAF].size);
  mgaMsg(1, "t2gzf    %p, %p, %d\n", mgaWarpIndex[MGA_WARP_T2GZF].Physaddr,
	 mgaWarpIndex[MGA_WARP_T2GZF].Virtaddr,
	 mgaWarpIndex[MGA_WARP_T2GZF].size);
  mgaMsg(1, "t2gzs    %p, %p, %d\n", mgaWarpIndex[MGA_WARP_T2GZS].Physaddr,
	 mgaWarpIndex[MGA_WARP_T2GZS].Virtaddr,
	 mgaWarpIndex[MGA_WARP_T2GZS].size);
  mgaMsg(1, "t2gzsa   %p, %p, %d\n", mgaWarpIndex[MGA_WARP_T2GZSA].Physaddr,
	 mgaWarpIndex[MGA_WARP_T2GZSA].Virtaddr,
	 mgaWarpIndex[MGA_WARP_T2GZSA].size);
  mgaMsg(1, "t2gzsaf  %p, %p, %d\n", mgaWarpIndex[MGA_WARP_T2GZSAF].Physaddr,
	 mgaWarpIndex[MGA_WARP_T2GZSAF].Virtaddr,
	 mgaWarpIndex[MGA_WARP_T2GZSAF].size);
  mgaMsg(1, "t2gzsf   %p, %p, %d\n", mgaWarpIndex[MGA_WARP_T2GZSF].Physaddr,
	 mgaWarpIndex[MGA_WARP_T2GZSF].Virtaddr,
	 mgaWarpIndex[MGA_WARP_T2GZSF].size);
}
}

/*
  This function loads the specified pipe in WMISC=0 mode (manual
  ucode download, no dma required)
  WARNING! the size of the pipes must be a multiple of 8.  This has been
  checked for all G200 pipes, but not yet for the G400 pipes.
  another WARNING.  Make sure warp is idle before calling this function
  (or change this function).  You can check the status of the pipes with
  the WBRKSTS registers.  Sadly enough, these are undocumented :-( (help,
  matrox!)
*/

static void warpLoadPipeManual(int pipe)
{
  mgaUI32 *from = mgaWarpIndex[pipe].Virtaddr;
  int size = mgaWarpIndex[pipe].size;

  if((size % 8) || (size == 0) || (from == 0))
    mgaMsg(1, "warpLoadPipeManual: Bad size (%d, %d) or source (%p)."
	   " Prepare to crash.\n", pipe, size, from);

  /* in the ddk, this is much more complicated.  The original author of
     this piece of code likes it more to be lazy then to be tired.  We
     might want to fix this when pipes get switched a lot. */
  OUTREG(MGAREG_WIMEMADDR, 0);
  if (MGA_IS_G400(MGAchipset)) 
    OUTREG(0x1e7c, 0);
  while(size>0) {
    OUTREG(MGAREG_WIMEMDATA, *from);
    if (MGA_IS_G400(MGAchipset)) 
      OUTREG(MGAREG_WIMEMDATA1, *from);
    from++;
    size-=4;
    mgaMsg(2, "MGAREG_WIMEMADDR %x\n", INREG( MGAREG_WIMEMADDR ));
  }
}

/*
  Initialise mgaWarpPipes for direct access mode
*/
static void warpInitialisePipesForDirectAccess()
{
  if (MGA_IS_G200(MGAchipset)) {
#define mgaWarpInstallCode(which, where) {\
      mgaWarpIndex[MGA_WARP_ ## where ].Physaddr = 0; \
      mgaWarpIndex[MGA_WARP_ ## where ].Virtaddr = WARP_G200_ ## which; \
      mgaWarpIndex[MGA_WARP_ ## where ].size = sizeof(WARP_G200_ ## which); \
    }
    mgaWarpInstallCode(tgz,    TGZ);
    mgaWarpInstallCode(tgza,   TGZA);
    mgaWarpInstallCode(tgzaf,  TGZAF);
    mgaWarpInstallCode(tgzf,   TGZF);
    mgaWarpInstallCode(tgzs,   TGZS);
    mgaWarpInstallCode(tgzsa,  TGZSA);
    mgaWarpInstallCode(tgzsaf, TGZSAF);
    mgaWarpInstallCode(tgzsf,  TGZSF);
#undef mgaWarpInstallCode
   } else { /*G400*/
#define mgaWarpInstallCode(which, where) {\
      mgaWarpIndex[MGA_WARP_ ## where ].Physaddr = 0; \
      mgaWarpIndex[MGA_WARP_ ## where ].Virtaddr = WARP_G400_ ## which; \
      mgaWarpIndex[MGA_WARP_ ## where ].size = sizeof(WARP_G400_ ## which); \
    }
    mgaWarpInstallCode(t2gz,    T2GZ);
    mgaWarpInstallCode(t2gza,   T2GZA);
    mgaWarpInstallCode(t2gzaf,  T2GZAF);
    mgaWarpInstallCode(t2gzf,   T2GZF);
    mgaWarpInstallCode(t2gzs,   T2GZS);
    mgaWarpInstallCode(t2gzsa,  T2GZSA);
    mgaWarpInstallCode(t2gzsaf, T2GZSAF);
    mgaWarpInstallCode(t2gzsf,  T2GZSF);
    mgaWarpInstallCode(tgz,     TGZ);
    mgaWarpInstallCode(tgza,    TGZA);
    mgaWarpInstallCode(tgzaf,   TGZAF);
    mgaWarpInstallCode(tgzf,    TGZF);
    mgaWarpInstallCode(tgzs,    TGZS);
    mgaWarpInstallCode(tgzsa,   TGZSA);
    mgaWarpInstallCode(tgzsaf,  TGZSAF);
    mgaWarpInstallCode(tgzsf,   TGZSF);
#undef mgaWarpInstallCode
  }
  dumpWarpCodeOffsets();
}


/*
  This function copies the WARP microcode to somewhere into the
  AGP aperture (or somewhere in system mem) for WMISC=3 mode (bus
  mastering, warp code mem acts like cache)
*/
static int warpInstallMicrocode()
{
  unsigned int microcode_size;
  if (MGA_IS_G200(MGAchipset)) {
    void *pcbase, *vcbase;
    /* FIXME: if we ever want clean deallocation of memory, we'll have
       to keep track of this block in some persistent scope */
    PMemBlock block;
#define CODESIZE(which) ((sizeof(which)/256 + 1)*256)
    microcode_size = CODESIZE(WARP_G200_tgz) +
      CODESIZE(WARP_G200_tgza) + CODESIZE(WARP_G200_tgzaf) +
      CODESIZE(WARP_G200_tgzf) + CODESIZE(WARP_G200_tgzs) +
      CODESIZE(WARP_G200_tgzsa) + CODESIZE(WARP_G200_tgzsaf) +
      CODESIZE(WARP_G200_tgzsf);
#undef CODESIZE
    block=mmAllocMem(sysmemHeap, microcode_size, 8, 0);
    if(!block) {
      mgaMsg(1, "Failed to allocate %d bytes for warp microcode\n",
	     microcode_size);
      mmDumpMemInfo(sysmemHeap);
      return 0;
    }
    mgaMsg(1, "Allocated %d bytes for G200 warp microcode\n",
	   microcode_size);
    pcbase = (void *)(sysmemPhysical + mmOffset( block ));
    vcbase = (void *)(sysmemVirtual + mmOffset( block ));
    memset(vcbase, 0, microcode_size);
#define mgaWarpInstallCode(which,where) {\
      mgaWarpIndex[where].Physaddr  = pcbase; \
      mgaWarpIndex[where].size  = sizeof(WARP_G200_ ## which); \
      memcpy(vcbase, WARP_G200_ ## which, sizeof(WARP_G200_ ## which)); \
      pcbase += (sizeof(WARP_G200_ ## which) / 256 + 1) * 256; \
      vcbase += (sizeof(WARP_G200_ ## which) / 256 + 1) * 256; \
    }
      
    mgaWarpInstallCode(tgz,    MGA_WARP_TGZ);
    mgaWarpInstallCode(tgza,   MGA_WARP_TGZA);
    mgaWarpInstallCode(tgzaf,  MGA_WARP_TGZAF);
    mgaWarpInstallCode(tgzf,   MGA_WARP_TGZF);
    mgaWarpInstallCode(tgzs,   MGA_WARP_TGZS);
    mgaWarpInstallCode(tgzsa,  MGA_WARP_TGZSA);
    mgaWarpInstallCode(tgzsaf, MGA_WARP_TGZSAF);
    mgaWarpInstallCode(tgzsf,  MGA_WARP_TGZSF);

#undef mgaWarpInstallCode
  } else { /*G400*/
    void *pcbase, *vcbase;
    /* FIXME: if we ever want clean deallocation of memory, we'll have
       to keep track of this block in some persistent scope */
    PMemBlock block;
#define CODESIZE(which) ((sizeof(which)/256 + 1)*256)
    microcode_size = CODESIZE(WARP_G400_t2gz) + CODESIZE(WARP_G400_t2gza) +
      CODESIZE(WARP_G400_t2gzaf) + CODESIZE(WARP_G400_t2gzf) +
      CODESIZE(WARP_G400_t2gzs) + CODESIZE(WARP_G400_t2gzsa) +
      CODESIZE(WARP_G400_t2gzsaf) + CODESIZE(WARP_G400_t2gzsf) +
      CODESIZE(WARP_G400_tgz) + CODESIZE(WARP_G400_tgza) +
      CODESIZE(WARP_G400_tgzaf) + CODESIZE(WARP_G400_tgzf) +
      CODESIZE(WARP_G400_tgzs) + CODESIZE(WARP_G400_tgzsa) +
      CODESIZE(WARP_G400_tgzsaf) + CODESIZE(WARP_G400_tgzsf);
#undef CODESIZE
    block=mmAllocMem(sysmemHeap, microcode_size, 8, 0);
    if(!block) {
      mgaMsg(1, "Failed to allocate %d bytes for warp microcode\n",
	     microcode_size);
      return 0;
    }
    mgaMsg(1, "Allocated %d bytes for G400 warp microcode\n",
	   microcode_size);
    pcbase = (void *)(sysmemPhysical + mmOffset( block ));
    vcbase = (void *)(sysmemVirtual + mmOffset( block ));
#define mgaWarpInstallCode(which, where) {\
      mgaWarpIndex[where].Physaddr = pcbase; \
      mgaWarpIndex[where].size = sizeof(WARP_G400_ ## which); \
      memcpy(vcbase, WARP_G400_ ## which, sizeof(WARP_G400_ ## which)); \
      pcbase += (sizeof(WARP_G400_ ## which) / 256 + 1) * 256; \
      vcbase += (sizeof(WARP_G400_ ## which) / 256 + 1) * 256; \
    }
    mgaWarpInstallCode(t2gz,    MGA_WARP_T2GZ);
    mgaWarpInstallCode(t2gza,   MGA_WARP_T2GZA);
    mgaWarpInstallCode(t2gzaf,  MGA_WARP_T2GZAF);
    mgaWarpInstallCode(t2gzf,   MGA_WARP_T2GZF);
    mgaWarpInstallCode(t2gzs,   MGA_WARP_T2GZS);
    mgaWarpInstallCode(t2gzsa,  MGA_WARP_T2GZSA);
    mgaWarpInstallCode(t2gzsaf, MGA_WARP_T2GZSAF);
    mgaWarpInstallCode(t2gzsf,  MGA_WARP_T2GZSF);
    mgaWarpInstallCode(tgz,     MGA_WARP_TGZ);
    mgaWarpInstallCode(tgza,    MGA_WARP_TGZA);
    mgaWarpInstallCode(tgzaf,   MGA_WARP_TGZAF);
    mgaWarpInstallCode(tgzf,    MGA_WARP_TGZF);
    mgaWarpInstallCode(tgzs,    MGA_WARP_TGZS);
    mgaWarpInstallCode(tgzsa,   MGA_WARP_TGZSA);
    mgaWarpInstallCode(tgzsaf,  MGA_WARP_TGZSAF);
    mgaWarpInstallCode(tgzsf,   MGA_WARP_TGZSF);
#undef mgaWarpInstallCode
  }
  dumpWarpCodeOffsets();
  
  return 1;
}


void mgaWarpUpdateState(GLcontext *ctx)
{
   int index = mgaCtx->setupindex;
   void *code;

   /* These are true for all the triangle functions, so don't
    * participate in the indexing.
    */
   index &= ~(MGA_WIN_BIT|MGA_TEX0_BIT|MGA_RGBA_BIT);

   /* Minimize thrashing by always interpolating alpha - doesn't seem
    * to make too much difference to q3test.
    */
   index |= (mgaglx.warp_residual_pipe & MGA_ALPHA_BIT);
   mgaglx.warp_residual_pipe |= index;

   if (mgaglx.warp_no_pipe_swaps)
      index = MGA_WARP_TGZA;

   /* Select new WARP pipe and set warp pipe options */
   if (index != mgaglx.warp_index)
   {
      DMALOCALS;

      mgaWarpFinishSerie();

      mgaMsg( 15, "Switching to warp pipe %i\n", index );
      
      if (0) mgaPrintSetupFlags("setting warp pipe", index); 

      mgaglx.warp_index = index;

      code = mgaWarpIndex[index].Physaddr;

      if ((!code) && (mgaWarpWMISC == 3)) {
	 mgaPrintSetupFlags("Invalid warp code selected", index);
	 code = mgaWarpIndex[0].Physaddr;
	 if (!code) FatalError("MGA code did a boo-boo thing\n");
      } else if(mgaWarpWMISC == 0) {
	 /* code == 0 and WMISC==0, manual pipe load 
	  */
	 warpLoadPipeManual(index);
      }

      MGADMAGETPTR(100);
     
      if (MGA_IS_G200(MGAchipset)) { 
	 DMAOUTREG(MGAREG_WIADDR, WIA_wmode_suspend);
	 DMAOUTREG(MGAREG_WVRTXSZ, 7);
	 DMAOUTREG(MGAREG_WFLAG, 0);  // FIXME! Read before write ?
	 DMAOUTREG(0x2d00 + 24*4, 0); // tex w/h
    
	 DMAOUTREG(0x2d00 + 25*4, 0x100);
	 DMAOUTREG(0x2d00 + 34*4, 0); // tex w/h
	 DMAOUTREG(0x2d00 + 42*4, 0xFFFF);
	 DMAOUTREG(0x2d00 + 60*4, 0xFFFF);

	 /* Attention! dma pading required due to hw bug (see specs) */
	 DMAOUTREG(MGAREG_DMAPAD, 0xffffffff);
	 DMAOUTREG(MGAREG_DMAPAD, 0xffffffff);
	 DMAOUTREG(MGAREG_DMAPAD, 0xffffffff);
	 DMAOUTREG(MGAREG_WIADDR,(unsigned long)(code) | WIA_wmode_start |
		   (mgaglx.use_agp ? WIA_wagp_agp : WIA_wagp_pci));
      }
      else {
	 float fParam = 12800.0f;

	 DMAOUTREG(MGAREG_WIADDR2, WIA_wmode_suspend);

	 if (index & MGA_TEX1_BIT) {
	    DMAOUTREG(MGAREG_WVRTXSZ, 0x00001e09);
	    DMAOUTREG(MGAREG_WACCEPTSEQ, 0x1e000000);
	 } else {
	    DMAOUTREG(MGAREG_WVRTXSZ, 0x00001807);
	    DMAOUTREG(MGAREG_WACCEPTSEQ, 0x18000000);
	 }

	 DMAOUTREG(MGAREG_WFLAG, 0);
	 DMAOUTREG(MGAREG_WFLAG1, 0);
       
	 DMAOUTREG(0x2d00 + 56*4, *((mgaUI32 *)(&fParam)));
	 DMAOUTREG(MGAREG_DMAPAD, 0);
	 DMAOUTREG(MGAREG_DMAPAD, 0);

	 /* Perhaps only the registers for the Tex Stage 1 should
	  * be set on the microcode that does multitex, however it
	  * makes no difference :(
	  */       
	 DMAOUTREG(0x2d00 + 49*4, 0);  /* Tex stage 0 */
	 DMAOUTREG(0x2d00 + 57*4, 0);  /* Tex stage 0 */
	 DMAOUTREG(0x2d00 + 53*4, 0);  /* Tex stage 1 */
	 DMAOUTREG(0x2d00 + 61*4, 0);  /* Tex stage 1 */

	 DMAOUTREG(0x2d00 + 54*4, 0x40); /* Tex stage 0 : w */
	 DMAOUTREG(0x2d00 + 62*4, 0x40); /* Tex stage 0 : h */
	 DMAOUTREG(0x2d00 + 52*4, 0x40); /* Tex stage 1 : w */
	 DMAOUTREG(0x2d00 + 60*4, 0x40); /* Tex stage 1 : h */
       
	 /* Attention! dma pading required due to hw bug (see specs) */
	 DMAOUTREG(MGAREG_DMAPAD, 0xffffffff);
	 DMAOUTREG(MGAREG_DMAPAD, 0xffffffff);
	 DMAOUTREG(MGAREG_DMAPAD, 0xffffffff);
	 DMAOUTREG(MGAREG_WIADDR2,(unsigned long)(code) | WIA_wmode_start |
		   (mgaglx.use_agp ? WIA_wagp_agp : WIA_wagp_pci));
      }
      DMAADVANCE();

      mgaCtx->new_state |= MGA_NEW_WARP;
   }
}


int mgaWarpInit(void)
{
  int wmisc;

  mgaMsg(1, "Configuring WARP\n");

  mgaglx.warp_serieStart = 0;
  mgaglx.warp_index = -1;	/* impossible value */

  if (glx_getint("mga_warp_tgza") || mgaglx.dmaDriver == 0 ) {
  	mgaMsg( 1, "Only using a single warp pipe.\n" );
 	mgaglx.warp_no_pipe_swaps = 1;
  }

  /* Find out wether we'll use busmastering or manual warp instruction
     loading */
  if (mgaglx.dmaDriver >= 2)
    mgaWarpWMISC = WM_wucodecache_enable | WM_wmaster_enable;
  else
    mgaWarpWMISC = 0;

  if (__glx_is_server) {
     WAITFIFO(3);
     if (MGA_IS_G200(MGAchipset)) {
	OUTREG(MGAREG_WIADDR, WIA_wmode_suspend);
	OUTREG(MGAREG_WGETMSB, 0x1606);
	OUTREG(MGAREG_WVRTXSZ, 7); /* may be set on something else later on*/
     } else {
	OUTREG(MGAREG_WIADDR2, WIA_wmode_suspend);
	OUTREG(MGAREG_WGETMSB, 0x00000E00);
	OUTREG(MGAREG_WVRTXSZ, 0x00001807);
	OUTREG(MGAREG_WACCEPTSEQ, 0x18000000);
     }
    
     /*
       wmisc can only be written once.  Be sure you write the correct thing
       to it !  You'll need a reboot to be able to reset it according to the
       docs.  I don't have the impression however that a reboot is required.
     */
     WAITFIFO(1);
     OUTREG(MGAREG_WMISC, mgaWarpWMISC | WM_wcacheflush_enable);
     wmisc = INREG(MGAREG_WMISC);
     if(wmisc != mgaWarpWMISC) {
	FatalError("[mga] WARP engine wrongly configured (%d != %d)."
	       "  Switch off your PC and try again.\n", wmisc, mgaWarpWMISC);
     }
  }

  if(mgaWarpWMISC == 3)
    /* Copy the warp microcode to somewhere in the agp aperture */
    if(!warpInstallMicrocode())
      return 0;

  if(mgaWarpWMISC == 0)
    warpInitialisePipesForDirectAccess();

  mgaMsg(1, "Warp initialisation successfully completed\n");

  return 1;
}
