/*
 * GLX Hardware Device Driver for Matrox Millenium G200
 * Copyright (C) 1999 Wittawat Yamwong
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR 
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE 
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 *
 *    Wittawat Yamwong <Wittawat.Yamwong@stud.uni-hannover.de>
 */

/* $Id: mgalib.h,v 1.14 1999/12/19 14:18:46 keithw Exp $ */

#ifndef MGALIB_INC
#define MGALIB_INC

#include <stdio.h>

#include "mesaglx/types.h"

#include "mgacommon.h"
#include "mgabuf.h"
#include "mm.h"
#include "g200_mac.h"
#include "mgalog.h"
#include "mgadma.h"
#include "mgatex.h"
#include "mgavb.h"


#ifndef MESA31
#error "The MGA driver now requires Mesa 3.1 or higher"
#endif


#define mgaContextMagic 0x47323030
#define VALID_MGA_CONTEXT(ctx) ((ctx) && (ctx)->magic == mgaContextMagic)

/* for SoftwareFallback */
#define FALLBACK_ALPHATEST 1
#define FALLBACK_BLENDING  2
#define FALLBACK_TEXTURE   4



/* For mgaCtx->new_state.
 */
#define MGA_NEW_DEPTH   0x1
#define MGA_NEW_ALPHA   0x2
#define MGA_NEW_FOG     0x4
#define MGA_NEW_CLIP    0x8
#define MGA_NEW_MASK    0x10
#define MGA_NEW_TEXTURE 0x20
#define MGA_NEW_CULL    0x40
#define MGA_NEW_WARP    0x80
#define MGA_NEW_CONTEXT 0x100


typedef void (*mga_interp_func)( GLfloat t, 
				 GLfloat *result,
				 const GLfloat *in,
				 const GLfloat *out );
				 

struct mga_context_t {
   mgaUI32 magic;

   mgaBufferPtr DB;
   GLcontext *gl_ctx;

   mgaUI32 regDWGCTL;
   mgaUI32 regALPHACTRL;
   mgaUI32 regSTENCIL;
   mgaUI32 regSTENCILCTL;   
   mgaUI32 regFOGCOLOR;
   mgaUI32 regWFLAG;
   mgaUI32 regTDUAL[2];		/* G400 dual texturing */

   /* How big is yours?
    */
   mgaUI32 vertsize;

   /* Map GL texture units onto hardware.
    */
   mgaUI32 multitex;
   mgaUI32 tmu_source[2];
   mgaUI32 tex_dest[2];

   /* shared texture palette */
   mgaUI16	GlobalPalette[256];
  
   int SoftwareFallback;  /* or'ed values of FALLBACK_* */

   /* Support for CVA and the fast paths */
   unsigned int setupdone;
   unsigned int setupindex;
   unsigned int renderindex;
   unsigned int using_fast_path;
   unsigned int using_immediate_fast_path;
   mga_interp_func interp;

   /* Shortcircuit some state changes */
   points_func   PointsFunc;
   line_func     LineFunc;
   triangle_func TriangleFunc;
   quad_func     QuadFunc;

   /* Manage our own state */
   GLuint        new_state; 
   GLuint        reg_dirty;

   GLfloat       map[16];
};


extern GLuint mgaDDRegisterPipelineStages( struct gl_pipeline_stage *out,
					   const struct gl_pipeline_stage *in,
					   GLuint nr );

extern GLboolean mgaDDBuildPrecalcPipeline( GLcontext *ctx );


/*
 * Driver functions that uses mgaCtx or mgaDB and can be called
 * directly from API such as glClear() should use CHECK_CONTEXT
 * before doing anything with mgaCtx and mgaDB.
 *
 * Drawing functions such as Triangle are protected by CHECK_CONTEXT
 * in mgaRenderVB, so that they don't need it.
 *
 * mgaDB will be invalid when the frame buffer have to be reallocate
 * but there was not enough video memory avialable. 
 * TODO: fix this.
 */
#define CHECK_CONTEXT(x) do {					\
  if (!VALID_MGA_CONTEXT(mgaCtx) || !VALID_MGA_BUFFER(mgaDB)) {	\
    x								\
  }								\
} while(0)

extern mgaBufferPtr mgaDB;
extern mgaBufferPtr mgaFrontBuffer;
extern mgaContextPtr mgaCtx;
extern int mgaIn3DContext;

typedef struct {
	mgaBufferPtr	bufferList;

	/* logging stuff */
	mgaUI32		logLevel;
	FILE		*logFile;

	/* dma stuff */
	mgaUI32		dmaDriver;	
        mgaUI32         dmaSize;
        mgaUI32         dmaAdr;
        mgaUI32         cmdSize;
        mgaUI32         cardCmds;
        mgaUI32         systemTexture;
        mgaUI32         noSetupDma;

        int             use_agp;

        /* bookkeeping for textureing */
        struct mga_texture_object_s *TexObjList;
        struct mga_texture_object_s *currentTextureObject[2];
        mgaUI16 	*currentTexturePalette;
        mgaUI32         default32BitTextures;
        mgaUI32         swapBuffersCount;
        
	/* warp stuff */
	mgaUI32	       *warp_serieStart;	
        mgaUI32         warp_index;             /* current active pipe */
        mgaUI32         warp_no_pipe_swaps;
        mgaUI32         warp_residual_pipe;

	/* options */
	mgaUI32		nullprims;  /* skip all primitive generation */
	mgaUI32		boxes;	    /* draw performance boxes */
	mgaUI32		noFallback; /* don't fall back to software, do
                                       best-effort rendering */
	mgaUI32		nosgram;    /* don't use block clears */
 	mgaUI32		skipDma;    /* don't send anything to the hardware */
 	
	/* performance counters */	
	mgaUI32		c_textureUtilization;
	mgaUI32		c_textureSwaps;
	mgaUI32		c_setupPointers;
	mgaUI32		c_triangles;
	mgaUI32		c_points;
	mgaUI32		c_lines;
	mgaUI32		c_drawWaits;
	mgaUI32		c_signals;
	mgaUI32		c_dmaFlush;
	mgaUI32		c_overflows;
	
	mgaUI32		hardwareWentIdle; /* cleared each swapbuffers,
					     set if a
					     waitfordmacompletion ever
					     exited without having to
					     wait */

	mgaUI32		timeTemp; /* for time counting in logfiles */
} mgaGlx_t;

extern mgaGlx_t	mgaglx;

int	usec( void );
void mgaLibInit();
void mgaSoftReset();

int mgaMakeCurrent(mgaContextPtr ctx, mgaBufferPtr buf);

#define MGAPACKCOLOR555(r,g,b,a) \
  ((((r) & 0xf8) << 7) | (((g) & 0xf8) << 2) | (((b) & 0xf8) >> 3) | \
    ((a) ? 0x8000 : 0))

#define MGAPACKCOLOR565(r,g,b) \
  ((((r) & 0xf8) << 8) | (((g) & 0xfc) << 3) | (((b) & 0xf8) >> 3))

#define MGAPACKCOLOR888(r,g,b) \
  (((r) << 16) | ((g) << 8) | (b))

#define MGAPACKCOLOR8888(r,g,b,a) \
  (((a) << 24) | ((r) << 16) | ((g) << 8) | (b))

#define MGAPACKCOLOR4444(r,g,b,a) \
  ((((a) & 0xf0) << 8) | (((r) & 0xf0) << 4) | ((g) & 0xf0) | ((b) >> 4))

static __inline__ mgaUI32 mgaPackColor(mgaUI8 r, mgaUI8 g, 
				   mgaUI8 b, mgaUI8 a)
{
  switch (mgaDB->Attrib & MGA_PF_MASK) {
  case MGA_PF_555:
    return MGAPACKCOLOR555(r,g,b,a);
  case MGA_PF_565:
    return MGAPACKCOLOR565(r,g,b);
  case MGA_PF_888:
    return MGAPACKCOLOR888(r,g,b);
  case MGA_PF_8888:
    return MGAPACKCOLOR8888(r,g,b,a);
  default:
    return 0;
  }
}

#endif
