/*
 * Copyright (C) 1999 John Carmack <johnc@idsoftware.com>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR 
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE 
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * 
 */


#if 0
#include "xsmesaP.h"
#include <stdio.h>
#include <GL/gl.h>
#endif

#include <stdlib.h>
#include "mach64glx.h"

/*

NOTES:

rage pro has a single byte greyscale texture format

*/


/*
 * mach64DestroyTexObj
 * Free all memory associated with a texture and NULL any pointers
 * to it.
 */
static void mach64DestroyTexObj( mach64ContextPtr ctx, mach64TextureObjectPtr t ) {
	mach64TextureObjectPtr	p,prev;

	mach64Msg( 10,"mach64DestroyTexObj( %p )\n", t->tObj );
	
 	if ( !t ) {
  		return;
  	}
  	
 	if ( t->magic != MACH64_TEXTURE_OBJECT_MAGIC ) {
 		mach64Error( "mach64DestroyTexObj: t->magic != MACH64_TEXTURE_OBJECT_MAGIC\n" );
		return;
	}

	/* free the texture memory */
	mmFreeMem( t->memBlock );
 
 	/* free mesa's link */   
	t->tObj->DriverData = NULL;

	/* remove from the driver texobj list */
	p = mach64glx.textureList;
	prev = NULL;
	while( p ) {
		if (p == t) {
			if ( prev ) {
				prev->next = t->next;
			} else {
				mach64glx.textureList = t->next;
			}
    			break;
    		}
		prev = p;
		p = p->next;
	}

	/* clear magic to catch any bad future references */
	t->magic = 0;
	
	/* free the structure */
	free( t );
}


/*
 * mach64DestroyOldestTexObj
 * Throw out a texture to try to make room for a new texture
 */
static int mach64DestroyOldestTexObj( void ) {
	mach64TextureObjectPtr t, oldest;
	mach64UI32 old;
 
	mach64Msg(10,"  Swapping out texture.\n");
	
 	/* find the best texture to toss */
	old = 0x7fffffff;
	oldest = NULL;
	for ( t = mach64glx.textureList; t ; t = t->next ) {
		if ( t->age < old ) {
			old = t->age;
			oldest = t;
		}
	}

#if 0	/* can't enable this until I get textures loading in the command stream */

	/* if the oldest texture was in use on the previous frame, then
	we are in a texture thrashing state.  Note that we can't just
	test for "in THIS frame", because textures from the same working
	set may be used in different order, and it could register as not
	thrashing.  The solution is to pick the MOST recently used texture
	that isn't currently needed for multitexture.  This will allow the
	other textures to stay resident for the next frame, rather than
	swapping everything out in order. */
	
	if ( old >= mach64glx.swapBuffersCount - 1 ) {
		/* newly created texture objects are always added to the
		front of the list, so just find the first one that isn't
		used for multitexture */
	        mach64Msg( 10, "mach64DestroyOldestTexObj: thrashing\n" );
		oldest = mach64glx.textureList;

	} else {
		mach64Msg( 10, "mach64DestroyOldestTexObj\n" );
	}
#endif
	
	if ( !oldest ) {
		/* This shouldn't happen unless the 2D resolution is high enough that
		a single texture can't be allocated in the remaining memory */
		mach64Error("  No Texture to swap out -> Out of Memory!\n"); 
		mmDumpMemInfo( textureHeap ); 
		return -1;
	}

	/* just destroy the texture, because it can always
	be recreated directly from the mesa texture */
	
	mach64DestroyTexObj( mach64Ctx, oldest );

	return 0;
}

/*
 * mach64ConvertTexture
 * Converts a block of mesa format texture to the apropriate hardware format
 */
static void mach64ConvertTexture( mach64UI32 *destPtr, int texelBytes, 
	struct gl_texture_image *image, int x, int y, int width, int height ) {
	int		i, j;
	mach64UI8		*src;
	
	if ( texelBytes == 1 && ( image->Format == GL_COLOR_INDEX || image->Format == GL_INTENSITY || image->Format == GL_LUMINANCE || image->Format == GL_ALPHA ) ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (mach64UI8 *)image->Data + ( ( y + i ) * image->Width + x );
			for ( j = width >> 2  ; j ; j-- ) {
				int	pix;
			
				pix = src[0] | ( src[1] << 8 ) | ( src[2] << 16 ) | ( src[3] << 24 );
				*destPtr++ = pix;
				src += 4;
			}
		}
	} else if ( texelBytes == 2 && image->Format == GL_RGB ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (mach64UI8 *)image->Data + ( ( y + i ) * image->Width + x ) * 3;
			for ( j = width >> 1  ; j ; j-- ) {
				int	pix;
			
				pix = MACH64PACKCOLOR565(src[0],src[1],src[2]) |  
					( MACH64PACKCOLOR565(src[3],src[4],src[5]) << 16 );
				*destPtr++ = pix;
				src += 6;
			}
		}
	} else if ( texelBytes == 2 && image->Format == GL_RGBA ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (mach64UI8 *)image->Data + ( ( y + i ) * image->Width + x ) * 4;
			for ( j = width >> 1  ; j ; j-- ) {
				int	pix;
			
				pix = MACH64PACKCOLOR4444(src[0],src[1],src[2],src[3]) |  
					( MACH64PACKCOLOR4444(src[4],src[5],src[6],src[7]) << 16 );
				*destPtr++ = pix;
				src += 8;
			}
		}
	} else if ( texelBytes == 2 && image->Format == GL_LUMINANCE ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (mach64UI8 *)image->Data + ( ( y + i ) * image->Width + x );
			for ( j = width >> 1  ; j ; j-- ) {
				int	pix;
				/* FIXME: should probably use 555 texture to get true grey */
				pix = MACH64PACKCOLOR565(src[0],src[0],src[0]) |  
					( MACH64PACKCOLOR565(src[1],src[1],src[1]) << 16 );
				*destPtr++ = pix;
				src += 2;
			}
		}
	} else if ( texelBytes == 2 && image->Format == GL_INTENSITY ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (mach64UI8 *)image->Data + ( ( y + i ) * image->Width + x );
			for ( j = width >> 1  ; j ; j-- ) {
				int	pix;
			
				pix = MACH64PACKCOLOR4444(src[0],src[0],src[0],src[0]) |  
					( MACH64PACKCOLOR4444(src[1],src[1],src[1],src[1]) << 16 );
				*destPtr++ = pix;
				src += 2;
			}
		}
	} else if ( texelBytes == 2 && image->Format == GL_ALPHA ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (mach64UI8 *)image->Data + ( ( y + i ) * image->Width + x );
			for ( j = width >> 1  ; j ; j-- ) {
				int	pix;
			
				pix = MACH64PACKCOLOR4444(255,255,255,src[0]) |  
					( MACH64PACKCOLOR4444(255,255,255,src[1]) << 16 );
				*destPtr++ = pix;
				src += 2;
			}
		}
	} else if ( texelBytes == 2 && image->Format == GL_LUMINANCE_ALPHA ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (mach64UI8 *)image->Data + ( ( y + i ) * image->Width + x ) * 2;
			for ( j = width >> 1  ; j ; j-- ) {
				int	pix;
			
				pix = MACH64PACKCOLOR4444(src[0],src[0],src[0],src[1]) |  
					( MACH64PACKCOLOR4444(src[2],src[2],src[2],src[3]) << 16 );
				*destPtr++ = pix;
				src += 4;
			}
		}
	} else if ( texelBytes == 4 && image->Format == GL_RGB ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (mach64UI8 *)image->Data + ( ( y + i ) * image->Width + x ) * 3;
			for ( j = width ; j ; j-- ) {
				int	pix;
			
				pix = (255<<24) | MACH64PACKCOLOR888(src[0],src[1],src[2]);
				*destPtr++ = pix;
				src += 3;
			}
		}
	} else if ( texelBytes == 4 && image->Format == GL_RGBA ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (mach64UI8 *)image->Data + ( ( y + i ) * image->Width + x ) * 4;
			for ( j = width  ; j ; j-- ) {
				int	pix;
			
				pix = MACH64PACKCOLOR8888(src[0],src[1],src[2],src[3]);
				*destPtr++ = pix;
				src += 4;
			}
		}
	} else if ( texelBytes == 4 && image->Format == GL_LUMINANCE ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (mach64UI8 *)image->Data + ( ( y + i ) * image->Width + x );
			for ( j = width ; j ; j-- ) {
				int	pix;
			
				pix = (255<<24) | MACH64PACKCOLOR888(src[0],src[0],src[0]);
				*destPtr++ = pix;
				src += 1;
			}
		}
	} else if ( texelBytes == 4 && image->Format == GL_INTENSITY ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (mach64UI8 *)image->Data + ( ( y + i ) * image->Width + x );
			for ( j = width ; j ; j-- ) {
				int	pix;
			
				pix = MACH64PACKCOLOR8888(src[0],src[0],src[0],src[0]);
				*destPtr++ = pix;
				src += 1;
			}
		}
	} else if ( texelBytes == 4 && image->Format == GL_ALPHA ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (mach64UI8 *)image->Data + ( ( y + i ) * image->Width + x );
			for ( j = width ; j ; j-- ) {
				int	pix;
			
				pix = MACH64PACKCOLOR8888(255,255,255,src[0]);
				*destPtr++ = pix;
				src += 1;
			}
		}
	} else if ( texelBytes == 4 && image->Format == GL_LUMINANCE_ALPHA ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (mach64UI8 *)image->Data + ( ( y + i ) * image->Width + x ) * 2;
			for ( j = width ; j ; j-- ) {
				int	pix;
			
				pix = MACH64PACKCOLOR8888(src[0],src[0],src[0],src[1]);
				*destPtr++ = pix;
				src += 2;
			}
		}
	} else {
		mach64Error( "Unsupported texelBytes %i, image->Format %i\n", texelBytes, image->Format );
	}
}


/*
 * mach64UploadSubImage
 * Perform an iload based update of a resident buffer.  This is used for
 * both initial loading of the entire image, and texSubImage updates.
 */
static void mach64UploadSubImage( mach64TextureObjectPtr t,int level,	     
		     int x, int y, int width, int height ) {
	int		x2;
	int		dwords;
        mach64UI32		*dest;
	struct gl_texture_image *image;
	int		texelBytes, texelsPerDword;

	/* the ragePro only handles a single level properly */
	if ( level != 0 ) {
		mach64Msg( 1, "mach64UploadSubImage: bad level: %i\n", level );
 	 	return;
	}

	image = t->tObj->Image[0];
	if ( !image ) {
		mach64Error( "mach64UploadSubImage: NULL image\n" );
		return;
	}
	
	if ( image->Format == GL_RGBA || image->Format == GL_ALPHA 
	|| image->Format == GL_LUMINANCE_ALPHA ) {
		t->hasAlpha = 1;
	} else {
		t->hasAlpha = 0;
	}		
	
	texelBytes = t->texelBytes;	
	texelsPerDword = 2;
		
	/* we can't do a subimage update if pitch is < 32 texels due to hardware XY addressing
	limits,	so we will need to linearly upload all modified rows */
	if ( image->Width < 32 ) {
		x = 0;
		width = image->Width * height;
		height = 1;
		/* we are going to assume that 1x1 textures aren't going to cause
		a bus error if we read up to four texels from that location */
		if ( width < texelsPerDword ) {
			width = texelsPerDword;
		}
	} else {
		/* pad the size out to dwords.  The image is a pointer to the entire image,
		so we can safely reference outside the x,y,width,height bounds if we need to */
		x2 = x + width;
		x2 = ( x2 + ( texelsPerDword - 1 ) ) & ~(texelsPerDword-1);
		
		x = ( x + ( texelsPerDword - 1 ) ) & ~(texelsPerDword-1);
		width = x2 - x;
	}
  
 #if 0	// FIXME: get dma based texture loading working   
	/* we may not be able to upload the entire texture in one batch due
	to register limits or dma buffer limits.  Recursively split it up. */
	while ( 1 ) {
		dwords = height * width / texelsPerDword;
		if ( dwords <= dma_buffer->maxSecondaryDwords ) {
			break;
		}
		mach64Msg(10, "mach64UploadSubImage: recursively subdividing\n" );

		mach64UploadSubImage( t, level, x, y, width, height >> 1 );
		y += ( height >> 1 );
		height -= ( height >> 1 );
	}
#endif	
	mach64Msg(10, "mach64UploadSubImage: %i,%i of %i,%i at %i,%i\n", width, height,
		image->Width, image->Height, x, y );

	/* bump the performance counter */
	mach64glx.c_textureSwaps += ( dwords << 2 );

	/* FIXME: do this with DMA instead of writing into the framebuffer! */	
	dest = (mach64UI32 *)( mach64glx.linearBase + t->memBlock->ofs );
	
	/* fill in the secondary buffer with properly converted texels from the
	mesa buffer */
	mach64ConvertTexture( dest, texelBytes, image, x, y, width, height );
}

static int Log2( unsigned a ) {
	unsigned	i;
	
	for ( i = 0 ; i < 32 ; i++ ) {
		if ( ( 1<<i ) >= a ) {
			return i;
		}	
	}
	return 31;
}

/*
 * mach64CreateTexObj
 * Allocate space for and load the mesa images into the texture memory block.
 * This will happen before drawing with a new texture, or drawing with a
 * texture after it was swapped out or teximaged again.
 */
void mach64CreateTexObj( mach64ContextPtr ctx, struct gl_texture_object *tObj ) {
	mach64TextureObjectPtr		t;
	int					ofs, size;
	PMemBlock				mem;
	struct gl_texture_image *image;
	
	mach64Msg( 10,"mach64CreateTexObj( %p )\n", tObj );

	image = tObj->Image[ 0 ];
	if ( !image ) {
		return;
	}

	t = malloc( sizeof( *t ) );
	if ( !t ) {
		FatalError( "mach64CreateTexObj: Failed to malloc textureObject\n" );
	}
	memset( t, 0, sizeof( *t ) );
	
	/* we only need one mip level, because the RagePro has busted mip
	map hardware */


	/* texture format options */
	t->texelBytes = 2;
	
	size = image->Width * image->Height * t->texelBytes;
	size = ( size + 31 ) & ~31;	/* 32 byte aligned */
	ofs = size;

	t->widthLog2 = Log2( image->Width );
	t->heightLog2 = Log2( image->Height );
	t->maxLog2 = t->widthLog2 > t->heightLog2 ? t->widthLog2 : t->heightLog2;
	
	t->totalSize = ofs;
	
	/* allocate a buffer for all levels, swapping out stuff if needed */
        while ( ( mem = mmAllocMem( textureHeap, ofs, 6, 0 ) ) == 0 ) {
        	if ( mach64DestroyOldestTexObj() ) {
        		/* can't hold this texture at all */
			mach64Msg( 10,"mach64CreateTexObj: Couldn't allocate buffer\n" );
        		free( t );
        		return;
        	}
        }

	/* dump the heap contents if loglevel is high enough */
	if ( mach64glx.logLevel >= 15 ) {
		mmDumpMemInfo( textureHeap );
	}

	/* fill in our texture object */
	t->magic = MACH64_TEXTURE_OBJECT_MAGIC;
	t->tObj = tObj;
	t->ctx = ctx;
	t->next = mach64glx.textureList;
	mach64glx.textureList = t;
	
	t->memBlock = mem;

	/* base image */	  
	image = tObj->Image[ 0 ];

  	tObj->DriverData = t;
  
	/* load the texels */
	mach64UploadSubImage( t, 0, 0, 0, image->Width, image->Height );
}


#if 0	// FIXME: also fix in mga
/*
 * mgaDestroyContextTextures
 */
void mgaDestroyContextTextures( mgaContextPtr ctx ) {
	mach64Msg( 1, "mgaDestroyContextTextures( %p )\n", ctx );
	/* FIXME: do we actually need to do anything here?  Shouldn't MESA
	call the normal delete function before freeing textures? */
#if 0
  while (ctx->textureList)
    mgaDestroyTexObj(ctx,ctx->textureList);
#endif
}
#endif


/*
============================================================================

Driver functions called directly from mesa

============================================================================
*/


/*
 * mach64TexImage
 */ 
void mach64TexImage( GLcontext *ctx, GLenum target,
		  struct gl_texture_object *tObj, GLint level,
		  GLint internalFormat,
		  const struct gl_texture_image *image ) {
	mach64TextureObject_t *t;

	mach64Msg( 10,"mach64TexImage( %p, level %i )\n", tObj, level );
  
  	/* only level 0 will matter on rage pro */
  	if ( level != 0 ) {
  		return;
  	}
  		
  	/* free the driver texture if it exists */
	t = (mach64TextureObjectPtr) tObj->DriverData;
	if ( t ) {
 	 	mach64DestroyTexObj( mach64Ctx, t );
	}
	
	/* create it */
	mach64CreateTexObj( mach64Ctx, tObj );
}

/*
 * mach64TexSubImage
 */      
void mach64TexSubImage( GLcontext *ctx, GLenum target,
		  struct gl_texture_object *tObj, GLint level,
		  GLint xoffset, GLint yoffset,
		  GLsizei width, GLsizei height,
		  GLint internalFormat,
		  const struct gl_texture_image *image ) {
	mach64TextureObject_t *t;

	mach64Msg(10,"mach64TexSubImage() Size: %d,%d of %d,%d; Level %d\n",
		width, height, image->Width,image->Height, level);

  	/* only level 0 will matter on rage pro */
  	if ( level != 0 ) {
  		return;
  	}

	/* immediately upload it if it is resident */
	t = (mach64TextureObject_t *) tObj->DriverData;
	if ( t ) {
		mach64UploadSubImage( t, 0, xoffset, yoffset, width, height );
	}
}

/*
 * mach64DeleteTexture
 */      
void mach64DeleteTexture( GLcontext *ctx, struct gl_texture_object *tObj ) {
	mach64Msg( 10, "mach64DeleteTexture( %p )\n", tObj );
	
	/* delete our driver data */
	if ( tObj->DriverData ) {
		mach64DestroyTexObj( mach64Ctx, (mach64TextureObject_t *)(tObj->DriverData) );
	}
}

/*
 * mach64IsTextureResident
 */      
GLboolean mach64IsTextureResident( GLcontext *ctx, struct gl_texture_object *tObj ) {
	GLboolean			is;
	
   	is = (tObj->DriverData != NULL);

	mach64Msg( 10, "mach64IsTextureResident( %p ) == %i\n", tObj, is );
   	
	return is;
}

