/* $Id: mach64state.c,v 1.6 1999/12/14 04:32:37 johnc Exp $ */

/*
 * GLX Hardware Device Driver for ATI Rage Pro
 * Copyright (C) 1999 Gareth Hughes
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Based on MGA driver: mgastate.c ???
 *
 *    Gareth Hughes <garethh@bell-labs.com>
 */

#include <stdio.h>

#include "mach64glx.h"
#include "pb.h"
#include "dd.h"

/*

NOTES:


*/

/*
 * mach64IsTexturingEnabled
 * FIXME: i'm sure there is an easier way of checking this
 */
struct gl_texture_object *mach64IsTexturingEnabled( const GLcontext *ctx ) {
	struct gl_texture_object	*tObj;

	if ( ctx->Texture.Enabled != TEXTURE0_2D ) {
		return 0;
	}

	tObj = ctx->Texture.Unit[0].Current;
	if ( !tObj ) {
		return 0;
	}
	
	/* check for weird mesa state where it looks like we
	are enabled but the texture object's image is an invalid pointer.
	This should probably be fixed in mesa someplace */
	if ( tObj != ctx->Texture.Unit[0].CurrentD[2] ) {
		return 0;
	}
	
	return tObj;
}

static int mach64CalcTEX_CNTL( const GLcontext *ctx ) {
	struct gl_texture_object	*tObj;
	mach64TextureObject_t		*t;
	int		clampS, clampT;
	int		reg;
	
	if ( !mach64IsTexturingEnabled( ctx ) ) {
		return 0;
	}
		
	tObj = ctx->Texture.Unit[0].Current;
	t = (mach64TextureObject_t *)tObj->DriverData;
	
	clampS = ( tObj->WrapS == GL_CLAMP ) ? 1 : 0;
	clampT = ( tObj->WrapT == GL_CLAMP ) ? 1 : 0;
	
	reg = 		
		0					/* LOD bias */
		| ( 0 << 4 )		/* composite factor */
		| ( 0 << 8 )		/* 1 = enable texture composite */
		| ( 1 << 9 )		/* composite combine factor = multiply */
		| ( 1 << 11 )		/* 1 = bilinear on composite minify */			
		| ( 1 << 12 )		/* 1 = bilinear on composite magnify */
		| ( 1 << 13 ) 		/* 1 = composite factor from texture alpha */
		| ( 0 << 14 )		/* 1 = textures in block tiled memory format */
		| ( 0 << 15 )		/* 1 = secondary texture in block tiled format */
		| ( 0 << 16 )		/* 1 = disable texel cache stalls */
		| ( clampS << 17 )	/* texture clamping */
		| ( clampT << 18 )	/* texture clamping */
		| ( 0 << 19 )		/* 1 = disable setup engine ST multiply by W */
		| ( 0 << 20 )		/* 1 = textures are in AGP memory */
		| ( 0 << 21 )		/* 1 = texture is in VQ compressed format */
		| ( 0 << 22 )		/* 1 = composite texture is in VQ format */
		| ( 0 << 23 )		/* 1 = flush texel cache */
		| ( 0 << 24 )		/* secondary texture S clamp */
		| ( 0 << 25 )		/* secondary texture T clamp */
		| ( 0 << 28 )		/* 1 = "wrap" texture S */
		| ( 0 << 29 )		/* 1 = "wrap" texture T */
		| ( 0 << 30 )		/* 1 = cut texture cache in half */
		| ( 0 << 31 )		/* 1 = use secondary ST for secondary texture (TEXTURING STOPS!) */
		;

	return reg;
}

static int mach64CalcSCALE_3D_CNTL( const GLcontext *ctx ) {
	int	reg;
	int	s3df;
	int	afn;
	int	asat;
	int	asrc, adst;
	int	tlf;
	int	tfilt;
	int	tblend;
	int	texalpha;
	struct gl_texture_object	*tObj;
	
	if ( mach64IsTexturingEnabled( ctx ) ) {
		s3df = 2;	// texturing
		tObj = ctx->Texture.Unit[0].Current;
		texalpha = ((mach64TextureObject_t *)(tObj->DriverData))->hasAlpha;
	
		if ( tObj->MinFilter == GL_LINEAR || 
			tObj->MinFilter == GL_LINEAR_MIPMAP_NEAREST ||
			tObj->MinFilter == GL_LINEAR_MIPMAP_LINEAR ) {
			tblend = 2;
		} else {
			tblend = 0;
		}
		
		if ( tObj->MagFilter == GL_LINEAR ) {
			tfilt = 1;
		} else {
			tfilt = 0;
		}
		
		switch ( ctx->Texture.Unit[0].EnvMode ) {
		case GL_REPLACE:
			tlf = 0;
			break;
		case GL_MODULATE:
			tlf = 1;
			break;
		case GL_DECAL:
			tlf = 2;
			break;
		case GL_BLEND:
		default:	
			tlf = 1;	/* not supported by RagePro! */
			break;
		}
	} else {
		s3df = 3;	// shading
		texalpha = 0;
		tlf = 1;	// 0 = replace, 1 = modulate, 2 = alpha decal (or is it blend?)
		tfilt = 1;
		tblend = 2;
	}			
	

	/* blending control */
	if ( ctx->Color.BlendEnabled ) {
 	   	afn = 1;		// blending on, fog off
    	
 	   	// FIXME: check for fog enabled here, which can't be done by hardware
    	
		switch ( ctx->Color.BlendSrcRGB ) {
		case GL_ZERO:
			asrc = 0; asat = 0; break;
		case GL_ONE:
			asrc = 1; asat = 0; break;
		case GL_DST_COLOR:
			asrc = 2; asat = 0; break;
		case GL_ONE_MINUS_DST_COLOR:
			asrc = 3; asat = 0; break;
		case GL_SRC_ALPHA:
			asrc = 4; asat = 0; break;
		case GL_ONE_MINUS_SRC_ALPHA:
			asrc = 5; asat = 0; break;
		case GL_DST_ALPHA:
			// asrc = 6; asat = 0; break;
			asrc = 0; asat = 0; break;		/* we don't have an alpha buffer (FIXME: 32 bit) */
		case GL_ONE_MINUS_DST_ALPHA:
			// asrc = 7; asat = 0; break;
			asrc = 0; asat = 0; break;		/* we don't have an alpha buffer (FIXME: 32 bit) */
		case GL_SRC_ALPHA_SATURATE:
			asrc = 0; asat = 1; break;
		default:		/* never happens */
		}

		switch ( ctx->Color.BlendDstRGB ) {
		case GL_ZERO:
			adst = 0; break;
		case GL_ONE:
			adst = 1; break;
		case GL_SRC_COLOR:
			adst = 2; break;
		case GL_ONE_MINUS_SRC_COLOR:
			adst = 3; break;
		case GL_SRC_ALPHA:
			adst = 4; break;
		case GL_ONE_MINUS_SRC_ALPHA:
			adst = 5; break;
		case GL_DST_ALPHA:
			// adst = 6; break;
			adst = 0; break;	/* we don't have an alpha buffer (FIXME: 32 bit) */
		case GL_ONE_MINUS_DST_ALPHA:
			// adst = 7; break;
			adst = 0; break;	/* we don't have an alpha buffer (FIXME: 32 bit) */
		default:		/* never happens */
		}
	} else {
 	   	/* blending disabled */
    	
		asat = 0;	// set if alpha saturate blending
		if ( !( ctx->RasterMask & FOG_BIT ) ) {
			afn = 0;	// blending off, fog off
			asrc = 1;
			adst = 0;
		} else {
			afn = 2;	// blending off, fog on 
			asrc = 5;	// fog color is now dest and fog factor is alpha, so
			adst = 4;	// use GL_ONE_MINUS_SRC_ALPHA, GL_SRC_ALPHA
		}
	}
    
	reg = 
	(1<<0)			// 0 = zero extend, 1 = dynamic range extend pixels to 32 bit
	| (1<<1)		// 1 = 2D dither, 0 = error diffusion dither
	| (1<<2)		// 1 = enable dither
	| (0<<3)		// 1 = reset error diffusion each line
	| (0<<4)		// 1 = round instead of dither
	| (0<<5)		// 1 = disable texture cache
	| (s3df<<6)	 
	| (0<<8)		// 1 = edge anti-alias
	| (0<<9)		// 1 = split texture cache
	| (0<<10)		// 1 = apple YUV mode
	| (afn<<11)		// alpha / fog control 
	| (asat<<13)	// alpha saturate blending
	| (0<<14)		// 1 = limit red dither range (what for???)
	| (0<<15)		// 1 = signed dst blend clamp for mpeg
	| (asrc<<16)	// blend src
	| (adst<<19)	// blend dst
	| (tlf<<22)		// texture environment
	| (1<<24)		// 1 = disable mip-mapping (its broke on all ragepros!)
	| (tfilt<<25)	// 1 = bilinear filter texture on mag
	| (tblend<<26)	// minification filtering
	| (0<<28)		// 1 = LSB of alpha for texture masking
	| (0<<29)		// alpha masking mode
	| (texalpha<<30)	// 1 = texture has alpha
	| (0<<31)		// 1 = source pixel from host register
	;
	
	return reg;
}



static int mach64CalcZ_CNTL( const GLcontext *ctx ) {
	int		zcntl;

	if ( !ctx->Depth.Test ) {
		return 0;
	}
    
	zcntl = 1;
	
	switch( ctx->Depth.Func )
	{
	case GL_NEVER:
	    zcntl |= 0 << 4; break;
	case GL_LESS:
	    zcntl |= 1 << 4; break;
	case GL_LEQUAL:
	    zcntl |= 2 << 4; break;
	case GL_EQUAL:
	    zcntl |= 3 << 4; break;
	case GL_GEQUAL:
	    zcntl |= 4 << 4; break;
	case GL_GREATER:
	    zcntl |= 5 << 4; break;
	case GL_NOTEQUAL:
	    zcntl |= 6 << 4; break;
	case GL_ALWAYS:
	    zcntl |= 7 << 4; break;
	default:
	    break;
	}
	if ( ctx->Depth.Mask ) {
		zcntl |= 256;	/* enable writes to depth buffer */
	}
	
	return zcntl;
}


static int mach64CalcALPHA_TST_CNTL( GLcontext *ctx ) {
	int		r;
	int		enable;
	int		mode;
	int		adest;
	int		asrc;
	int		ref;
	int		specen;

	if ( ctx->Color.AlphaEnabled ) {
		enable = 1;
		ref =  ctx->Color.AlphaRef;

		switch ( ctx->Color.AlphaFunc ) {
		default:
		case GL_NEVER:
		    mode = 0; break;
		case GL_LESS:
		    mode = 1; break;
		case GL_LEQUAL:
		    mode = 2; break;
		case GL_EQUAL:
		    mode = 3; break;
		case GL_GEQUAL:
		    mode = 4; break;
		case GL_GREATER:
		    mode = 5; break;
		case GL_NOTEQUAL:
		    mode = 6; break;
		case GL_ALWAYS:
		    mode = 7; break;
		}
		
		/* we are going to always take the texel alpha, because
		we can't get modulated values (rage pro limitation),
		and alpha testing from vertex shading doesn't make much sense */
		asrc = 0;			
	} else {
		enable = 0;
		mode = 0;
		asrc = 0;
		ref = 0;
	}	
	
	adest = 4;		/* destination alpha is source alpha */
	specen = 0;		/* FIXME; how do we properly set specular enable? */ 
		
	r = 
	(enable<<0)
	| (mode<<4)
	| (0<<7)		/* ALPHA_MOD_MSB weird stuff */
	| (adest<<8)	/* select value to write for alpha */
	| (asrc<<12)	/* source alpha for comparison */
	| (ref<<16)		/* alpha reference value */
	| (0<<24)		/* 4 bit color index weird stuff */
	| (0<<28)		/* 4 bit color index weird stuff */
	| (0<<29)		/* 4 bit color index weird stuff */
	| (0<<30)		/* composite shadow */
	| (specen<<31);	/* specular lighting enable */	

	return r;	
}



/* =============================================================
 */


/*
 * mach64DDUpdateState
 * This will be called before any beginrender if state has changed
 */
void mach64DDUpdateState( GLcontext *ctx ) {
	DMALOCALS;
	int		textureFormat;
	
	mach64glx.c_setupPointers++;

	/* for now, we are going to set every register we care about */

	if ( !mach64DB ) {
		return;
	}

	/* no acceleration if our back buffer isn't on card */
	if ( !mach64DB->backBufferBlock ) {
		return;	
	}
	
	/* no acceleration if our back buffer isn't on card */
	if ( ctx->Depth.Test && !mach64DB->depthBufferBlock ) {
		return;	
	}
	
	/* no acceleration if any stencil operations are enabled */
	if ( ctx->Stencil.Enabled ) {
		return;	
	}

	/* no acceleration if stipple is on */
	if ( ctx->Polygon.StippleFlag ) {
		return;	
	}

	/* check for uploading a texture */
	if ( mach64IsTexturingEnabled( ctx ) ) {
		struct gl_texture_object	*tObj;
		mach64TextureObject_t		*t;
		
		tObj = ctx->Texture.Unit[0].Current;
		
		/* move a texture to the card if needed */
		if ( !tObj->DriverData ) {
			/* this will program registers, so it can't be inside a DMAGETPTR() block */
			mach64CreateTexObj( mach64Ctx, ctx->Texture.Unit[0].Current );
		}
		t = (mach64TextureObject_t *)tObj->DriverData;
		
		if ( t->hasAlpha ) {
			textureFormat = 15;
		} else {
			textureFormat = 4;
		}
		
		DMAGETPTR( 40 );
		
		DMAOUTREG( MACH64REG_TEX_0_OFF + (t->maxLog2<<2), t->memBlock->ofs );

		DMAOUTREG( MACH64REG_TEX_SIZE_PITCH, 
			t->widthLog2
			| ( t->maxLog2 << 4 )
			| ( t->heightLog2 << 8 ) 
			| ( t->maxLog2 << 12 )
			| ( t->widthLog2 << 16 )
			| ( t->maxLog2 << 20 ) 
			| ( t->heightLog2 << 24 )
			);	/* FIXME: multitexture maps */

		DMAOUTREG( MACH64REG_TEX_CNTL, mach64CalcTEX_CNTL( ctx ) );
		
		DMAADVANCE();
		
		/* note it as recently used to keep from swapping out */
		t->age = mach64glx.swapBuffersCount;
	} else {
		textureFormat = 4;		/* RGB 565 (FIXME: others) */
	}

	DMAGETPTR( 40 );

	/* draw to the back buffer */	
	DMAOUTREG( MACH64REG_DST_OFF_PITCH, 
		( (mach64DB->pitch/8) << 22 ) | (mach64DB->backBufferBlock->ofs>>3) ) ;
		
	/* use our depth buffer */
	if ( !mach64DB->depthBufferBlock ) {
		DMAOUTREG( MACH64REG_Z_CNTL, 0 );
	} else {
		DMAOUTREG( MACH64REG_Z_OFF_PITCH,
			( (mach64DB->pitch/8) << 22 ) | (mach64DB->depthBufferBlock->ofs>>3) ) ;
			
		DMAOUTREG( MACH64REG_Z_CNTL, mach64CalcZ_CNTL( ctx ) );
	}	

	DMAOUTREG( MACH64REG_DP_MIX, 0x00070007 );	/* only use src */

	DMAOUTREG( MACH64REG_DP_SRC, 
		DP_bkgd_src_3d_data | DP_frgd_src_3d_data | DP_mono_src_1 );
	
	DMAOUTREG( MACH64REG_GUI_TRAJ_CNTL, 3 );		/* left to right, top to bottom */ 

	DMAOUTREG( MACH64REG_SETUP_CNTL, (1<<6) );		/* turn on the LOG_MAX_INC bit */
	
	DMAOUTREG( MACH64REG_CLR_CMP_CNTL, 0 );			/* disable color compare */

	/* lots of blending / texturing bits */
	DMAOUTREG( MACH64REG_SCALE_3D_CNTL, mach64CalcSCALE_3D_CNTL( ctx ) );
	
	/* alpha test control */ 
	DMAOUTREG( MACH64REG_ALPHA_TST_CNTL, mach64CalcALPHA_TST_CNTL( ctx ) );
	
	/* scissor rect */
	if ( ctx->Scissor.Enabled ) {
		int		x1,x2,y1,y2;

		/* mesa does not rage check scissor for us */	
		x1 = ctx->Scissor.X;
		x2 = ctx->Scissor.X + ctx->Scissor.Width - 1;
		y1 = mach64DB->height - ctx->Scissor.Y - ctx->Scissor.Height;
		y2 = mach64DB->height - ctx->Scissor.Y - 1;
		if ( x1 < 0 ) x1 = 0;
		if ( y1 < 0 ) y1 = 0;
		if ( x2 >= mach64DB->width ) x2 = mach64DB->width - 1;
		if ( y2 >= mach64DB->height ) y2 = mach64DB->height - 1;

		if ( ( x1 > x2 ) || ( y1 > y2 ) ) {
			x1 = 0; x2 = 0;
			y2 = 0; y1 = 1;
		}

		DMAOUTREG( MACH64REG_SC_LEFT_RIGHT, x1 | ( x2 << 16 ) );
		DMAOUTREG( MACH64REG_SC_TOP_BOTTOM, y1 | ( y2 << 16 ) );
	} else {
		DMAOUTREG( MACH64REG_SC_LEFT_RIGHT, 0 | ( ( mach64DB->width - 1 ) << 16 ) );
		DMAOUTREG( MACH64REG_SC_TOP_BOTTOM, 0 | ( ( mach64DB->height - 1 ) << 16 )  );
	}	
		
	/* fog color */
	if ( ctx->Fog.Enabled ) {
		int		r, g, b;
		int		color;
		
		r = 255 * ctx->Fog.Color[0];
		g = 255 * ctx->Fog.Color[1];
		b = 255 * ctx->Fog.Color[2];
		color = MACH64PACKCOLOR565(r,g,b);
		/* aliased on top of DP_FRGRD_COLOR */
		DMAOUTREG( MACH64REG_DP_FOG_CLR, color );	
	}

	/* color plane write mask */
	if ( !ctx->Color.BlendEnabled ) {
		int		color;
		
		color = MACH64PACKCOLOR565( ctx->Color.ColorMask[0], 
			ctx->Color.ColorMask[1], ctx->Color.ColorMask[2] );
			
		DMAOUTREG( MACH64REG_DP_WRITE_MASK, color );
	} else {
		/* Rage Pro can't color mask and alpha blend at the same time */ 	
		DMAOUTREG( MACH64REG_DP_WRITE_MASK, 0xffffffff );
	}	


	/* pixel widths */
	/* is this correct for fog color with alpha texture formats? */
	DMAOUTREG( MACH64REG_DP_PIX_WIDTH, 
		4	/* dest pix width FIXME: others */
		| (textureFormat<<4) 
		| (textureFormat<<8) 
		| (textureFormat<<12) 
		| (textureFormat<<28) 
		);

	DMAADVANCE();


	/* if we are going to render with hardware, go to our tri function */
	ctx->Driver.TriangleFunc = mach64Triangle;
}




void mach64DDInitStatePointers( GLcontext *ctx )
{
    ctx->Driver.Enable = 0;
    ctx->Driver.LightModelfv = 0;
    ctx->Driver.AlphaFunc = 0;
    ctx->Driver.BlendEquation = 0;
    ctx->Driver.BlendFunc = 0;
    ctx->Driver.BlendFuncSeparate = 0;
    ctx->Driver.DepthFunc = 0;
    ctx->Driver.DepthMask = 0;
    ctx->Driver.Fogfv = 0;
    ctx->Driver.Scissor = 0;
    ctx->Driver.CullFace = 0;
    ctx->Driver.FrontFace = 0;
    ctx->Driver.ShadeModel = 0;
    ctx->Driver.ColorMask = 0;
    ctx->Driver.ReducedPrimitiveChange = 0;
    ctx->Driver.RenderStart = mach64DDUpdateState;
    ctx->Driver.RenderFinish = 0;
	ctx->Driver.UpdateState = 0;	/* we update state in render start */
		
#if 0 /* FIXME: GH */
    ctx->Driver.Viewport = mach64DDViewport;
    ctx->Driver.DepthRange = mach64DDDepthRange;
    ctx->Driver.RegisterVB = mach64DDRegisterVB;
    ctx->Driver.UnregisterVB = mach64DDUnregisterVB;
    ctx->Driver.TexEnv = mach64TexEnv;
    ctx->Driver.TexImage = mach64TexImage;
    ctx->Driver.TexParameter = mach64TexParameter;
    ctx->Driver.UpdateTexturePalette = mach64UpdateTexturePalette;
#endif

    ctx->Driver.TexImage = mach64TexImage;
    ctx->Driver.TexSubImage = mach64TexSubImage;
    ctx->Driver.DeleteTexture = mach64DeleteTexture;
    ctx->Driver.IsTextureResident = mach64IsTextureResident;
}



/*
 * Local Variables:
 * mode: c
 * c-basic-offset: 4
 * End:
 */
