#!/usr/bin/perl -w
#
##############################################################################
#          Generic parser for easy hardware register specification
# V1.0
##############################################################################
# Author: simon pogarcic, sim@suse.de
#
# Permission is hereby granted, free of charge, to any person obtaining a
# copy of this software and associated documentation files (the "Software"),
# to deal in the Software without restriction, including without limitation
# the rights to use, copy, modify, merge, publish, distribute, sublicense,
# and/or sell copies of the Software, and to permit persons to whom the
# Software is furnished to do so, subject to the following conditions:
# 
# The above copyright notice and this permission notice shall be included
# in all copies or substantial portions of the Software.
# 
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
# OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
# SIMON POGARCIC, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM, 
# DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR 
# THERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE 
# OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
#
##############################################################################
#
#          PUT YOUR DEFINITIONS WHICH DOESN'T FIT FOR PARSER HERE
#   They will appear at the start of automaticly generated register file
#
##############################################################################
$code_start = <<'EOF';
/*
 * GLX Hardware Device Driver for ATI Rage Pro
 * Copyright (C) 1999 Gareth Hughes
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM,
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 *
 *    Gareth Hughes <garethh@bell-labs.com>
 */

#ifndef __MACH64_REGS_H__
#define __MACH64_REGS_H__

EOF

$code_end = <<'EOF';

#endif /* __MACH64_REGS_H__ */

EOF

$text = <<'EOF';
###############################################################################

# ==========================================================================
# === REGISTER =============== OFFSET ======= ACCESS = CARD ================
# ==========================================================================

@REGION0



@BCOMM Setup and Control Memory Mapped Registers


BUS_CNTL		0x04A0	rw
@VALUES BUS_
	dbl_resync_		[0]
	mstr_reset_		[1]
	flush_buf_		[2]
	stop_req_		[3]{enable/disable}
	aper_reg_		[4]{enable/disable}
	extra_pipe_		[5]{enable/disable}
	master_			[6]{enable/disable}
	rom_wrt_		[7]
	pci_read_retry_		[13]
	pci_wrt_retry_		[15]
	retry_ws_		[16-19]
	mstr_rd_mult_		[20]
	mstr_rd_line_		[21]
	suspend_		[22]
	lat16x_			[23]
	rd_discard_		[24]
	rd_abort_		[25]
	mstr_ws_		[26]{8/32}
	ext_reg_		[27]
	mstr_disconnect_	[28]{dis/en}
	wrt_burst_		[29]
	read_burst_		[30]
	rdy_read_dly_		[31]
@ENDVALUES

CONFIG_CHIP_ID		0x04E0	ro
@VALUES CFG_CHIP_
	type_			[0-15]
	class_			[16-23]
	major_			[24-26]
	fnd_id_			[27-29]
	minor_			[30-31]
@ENDVALUES

CONFIG_CNTL		0x04DC	rw
@VALUES CFG_
	mem_ap_size_		[0-1]
		+2x8mb			2
	mem_vga_ap_		[2]
	mem_ap_loc_		[4-13]
	vga_dis_		[19]{enable/disable}
@ENDVALUES

CONFIG_STAT0		0x04E4	rw
@VALUES CFG_
	mem_type_		[0-2]
		+disable		0
		+basic_dram		1
		+edo			2
		+hp_dram		3
		+sdram			4
		+sgram			5
	rom_remap_		[3]
	vga_en_			[4]
	clock_			[5]
	vfc_sense_		[7]{none/detected}
	board_id_		[8-15]
	pci33_			[22]
	fullagp_		[23]{256/208}
@ENDVALUES

CONFIG_STAT1		0x0494	ro
@VALUES CFG_
	subsys_dev_id_		[0-7]
	subsys_ven_id_		[8-23]
	dimm_type_		[29-31]
@ENDVALUES

CONFIG_STAT2		0x0498	ro
@VALUES CFG_
	rom_wrt_		[5]
	agp_vco_gain_		[6-7]
	bus_type_		[8]
	agp_skew_		[9-11]
	x1_clk_skew_		[12-14]
		+0_taps			0
		+1_taps			1
		+2_taps			2
		+3_taps			3
		+4_taps			4
		+5_taps			5
		+6_taps			6
		+7_taps			7
	cs_			[15]
	mem_typeb_		[16-18]
		+disable		7
		+dram			6
		+edo			5
		+hp_dram		4
		+sdram			3
		+sgram			2
		+wram			1
	id_disable_		[19]{normal/not_connected}
	chg_id0_		[20]
	prefetch_		[21]
	pre_test_		[23]
	chg_id1_		[24]
	vfc_senseb_		[25]{attached/not_attached}
	vga_			[26]{enable/disable}
	int_			[27]{enable/disable}
	rom_remapb_		[29]
	id_sel_			[30]{ad16/ad17}
@ENDVALUES

CRC_SIG			0x04E8	ro

CUSTOM_MACRO_CNTL	0x04D4	rw
@VALUES CMC_
	cmd_fifo_ext_		[0]{normal/enable}
	dsp_fifo_ext_		[1]{normal/enable}
	rdbuf_fifo_ext_		[2]{normal/enable}
	wrbuf_fifo_ext_		[3]{normal/enable}
	gwbuf_fifo_ext_		[4]{normal/enable}
	cache_a_ext_		[5]{normal/enable}
	cache_b_ext_		[6]{normal/enable}
	tag_ram_ext_		[7]{normal/enable}
	rdret_fifo_ext_		[8]{normal/enable}
	mem_cas_skew_		[16-17]
	mem_cntl_		[18]{no_delay/delay}
	mem_cs01_		[19]{no_delay/delay}
	mem_cs23_		[20]{no_delay/delay}
	mem_dqm_		[21]{no_delay/delay}
	mem_md_			[22]{no_speedup/speedup}
@ENDVALUES

DSP_CONFIG		0x0420	rw

DSP_ON_OFF		0x0424	rw

EXT_MEM_CNTL		0x04AC	rw
@VALUES MEM_
	cs_			[0]{every_other_clk/every_clk}
	sdram_			[1]{normal/reset}
	cyc_test_		[2-3]
		+normal			0
		+init			2
		+run			3
	tile_select_		[4-7]
		+none			0
		+256_1k			1
		+512_1k			2
		+128_1k			4
		+128_2k			5
		+128_4k			6
		+128_8k			7
		+128_640		8
		+128_1280		9
		+128_2560		10
	clk_select_		[8-9]
		+sdram			0
		+xclk			2
		+inv_xclk		3
	cas_latency_		[10-11]
		+1_clk			0
		+2_clk			1
		+3_clk			2
		+4_clk			3
	tile_bndry_		[12-15]
	mda_drive_		[16]{no_boost/boost}
	mdb_drive_		[17]{no_boost/boost}
	mde_			[18]{no_delay/delay}
	mdo_			[19]{no_delay/delay}
	ma_drive_		[20]{no_boost/boost}
	ma_			[21]{no_delay/delay}
	gcmrs_			[24-27]
		+burst_0		0
		+burst_2		1
		+burst_4		2
		+burst_8		3
		+sequential		0
		+interleave		4
		+burst_rw		0
		+burst_ro		8
	cs_strap_		[28]{enable/disable}
	sdram_cfg_		[29]{222/114}
	all_page_		[30]{enable/disable}
	group_fault_		[31]
@ENDVALUES

GEN_TEST_CNTL		0x04D0	rw
@VALUES	GEN_
	cur_			[7]
	gui_			[8]{reset/set}
	mem_ctrl_		[9]{normal/soft_reset}
	test_vect_mode_		[12-15]
		+normal			0
		+iddq			3
	test_mode_		[16-20]
		+disable		0
		+video_port		5
		+fifo			6
		+debug			7
		+ring_osc		8
		+delay_path		9
		+reg_block		10
		+pll			11
		+palette		12
		+dac			13
		+ramdac			14
	crc_			[21]
	debug_mode_		[24-31]
@ENDVALUES

MEM_ADDR_CONFIG		0x0434	rw
@VALUES	MEM_
	row_mapping_		[0-2]
	col_mapping_		[4-6]
	group_size_		[8-9]
		+2mb			0
		+4mb			1
		+8mb			2
	wbank_cntl_		[16-25]
@ENDVALUES

MEM_BUF_CNTL		0x042C	rw
@VALUES	MEM_
	z_wb_flush_		[0-3]
	vid_wb_flush_		[4-7]
	gui_wb_flush_		[8-12]
	hst_wb_flush_		[13-15]
	scl_thresh_		[16-21]
	rb_cache_		[23]{validate/invalidate}
@ENDVALUES

MEM_CNTL		0x04B0	rw
@VALUES	MEM_
	size_			[0-3]
		+512k			0
		+1mb			1
		+2mb			3
		+4mb			7
		+6mb			9
		+8mb			11
		+16mb			15
	latency_		[4-5]
		+1_clk			0
		+2_clk			1
		+3_clk			2
		+4_clk			3
	latch_			[6-7]
		+cas			0
		+hclk			1
		+pos_xclk		2
		+neg_xclk		3
	trip_			[8-9]
		+1_clk			0
		+2_clk			1
		+3_clk			2
		+4_clk			3
	trcd_ras_cas_		[10-11]
		+1_clk			0
		+2_clk			1
		+3_clk			2
		+4_clk			3
	trcd_cas_ras_		[12]{no_delay/delay}
	tr2w_			[13]{1_clk/2_clk}
	cas_			[14]{low_clk_low/low_clk_high}
	oe_pullback		[15]{1_clk/0_clk}
	tras_			[16-18]
		+1_clk			0
		+2_clk			1
		+3_clk			2
		+4_clk			3
		+5_clk			4
		+6_clk			5
		+7_clk			6
		+8_clk			7
	refresh_		[19]{enable/disable}
	refresh_rate_		[20-22]
		+10mhz			0
		+44mhz			1
		+50mhz			2
		+55mhz			3
		+66mhz			4
		+75mhz			5
		+80mhz			6
		+100mhz			7
	lower_aper_		[24-25]
		+little_endian		0
		+big_endian_16bpp	1
		+big_endian_32bpp	2
	upper_aper_		[26-27]
		+little_endian		0
		+big_endian_16bpp	1
		+big_endian_32bpp	2
	page_size_		[28-29]
		+2k			0
		+4k			1
		+8k			2
		+16k			3
@ENDVALUES

SCRATCH_REG0		0x0480	rw

SCRATCH_REG1		0x0484	rw

TIMER_CONFIG		0x0428	rw

VGA_DSP_CONFIG		0x044C	rw

VGA_DSP_ON_OFF		0x0450	rw



@BCOMM Accelerator CRTC and DAC Memory Mapped Registers


CUR_CLR0		0x0460	rw
@VALUES CUR_
	clr0_8_			[0-7]
	clr0_b_			[8-15]
	clr0_g_			[16-23]
	clr0_r_			[24-31]
@ENDVALUES

CUR_CLR1		0x0464	rw
@VALUES CUR_
	clr1_8_			[0-7]
	clr1_b_			[8-15]
	clr1_g_			[16-23]
	clr1_r_			[24-31]
@ENDVALUES

CUR_OFFSET		0x0468	rw
@VALUES CUR_
	offset_		[0-19]
@ENDVALUES

CUR_HORZ_VERT_POSN	0x046C	rw
@VALUES CUR_
	horz_posn_		[0-10]
	vert_posn_		[16-26]
@ENDVALUES

CUR_HORZ_VERT_OFF	0x0470	rw
@VALUES CUR_
	horz_off_		[0-5]
	vert_off_		[16-21]
@ENDVALUES



@BCOMM Draw Engine Trajectory Memory Mapped Registers


ALPHA_TST_CNTL		0x0550	rw
@VALUES ALPHA_
	tst_			[0]
	test_			[4-6]
		+anever			0
		+alt			1
		+alte			2
		+ae			3
		+agte			4
		+agt			5
		+ane			6
		+aalways		7
	mod_msb_		[7]
	dst_			[8-10]
		+zero			0
		+one			1
		+src_alpha		4
		+om_src_alpha		5
		+dst_alpha		6
		+om_dst_alpha		7
	tst_src_sel_		[12]{texel/src}
	ref_			[16-23]
	comp_ci4_rgb_index_	[24-27]
	comp_ci4_low_nibble_	[28]{dis/en}
	comp_ci4_high_nibble_	[29]{dis/en}
	comp_shadow_		[30]
	spec_light_		[31]{off/on}
@ENDVALUES

COMPOSITE_SHADOW_ID	0x0798	rw
@VALUES COMP_
	shadow_id_		[0-23]
@ENDVALUES

CONTEXT_LOAD_CNTL	0x072c	rw

CONTEXT_MASK		0x0720	rw

DST_BRES_DEC		0x052C	rw
@VALUES DST_
	bres_dec_		[0-17]
@ENDVALUES

DST_BRES_ERR		0x0524	rw
@VALUES DST_
	bres_err_		[0-17]
@ENDVALUES

DST_BRES_INC		0x0528	rw
@VALUES DST_
	bres_inc_		[0-17]
@ENDVALUES

DST_BRES_LNTH		0x0520	rw
@VALUES DST_
	bres_lnth_		[0-14]
	draw_trap_		[15]
		+init			1
	trail_x_		[16-30]
	bres_lnth_line_		[31]{en/dis}
@ENDVALUES

DST_CNTL		0x0530	rw
@VALUES DST_
	x_dir_			[0]{right_left/left_right}
	y_dir_			[1]{bottom_top/top_bottom}
	y_major_		[2]{x_major_line/y_major_line}
	x_tile_			[3]
	y_tile_			[4]
	last_pel_		[5]
	poly_			[6]
	24_rot_en_		[7]
	24_rot_			[8-10]
	dst_bres_sign_		[11]{pos/neg}
	poly_rtedge_		[12]{en/dis}
	trail_x_dir_		[13]{right_left/left_right}
	trap_fill_dir_		[14]{right_left/left_right}
	trail_bres_sign_	[15]{pos/neg}
	bres_sign_		[17]{auto/override}
	alpha_overlap_		[19]{disallow/allow}
	sub_pix_		[20]{off/on}
@ENDVALUES

DST_HEIGHT		0x0514	rw
@VALUES DSTH_
	height_			[0-14]
@ENDVALUES

DST_HEIGHT_WIDTH	0x0518	wo
@VALUES DSTHW_
	height_			[0-15]
	width_			[16-31]
@ENDVALUES

DST_OFF_PITCH		0x0500	rw
@VALUES DSTOP_
	offset_			[0-19]
	pitch_			[22-31]
@ENDVALUES

DST_WIDTH		0x0510	rw
@VALUES DSTW_
	width_			[0-15]
	width_fill_		[31]{en/dis}
@ENDVALUES

DST_WIDTH_HEIGHT	0x06ec	wo
@VALUES DSTWH_
	width_			[0-15]
	height_			[16-31]
@ENDVALUES

DST_X			0x0504	rw
@VALUES DSTX_
	x_			[0-14]
	scale_y_sec_		[15]
		+lsb			1
@ENDVALUES

DST_X_WIDTH		0x051c	wo
@VALUES DSTXW_
	x_			[0-14]
	scale_y_sec_		[15]
		+lsb			1
	dst_width_		[16-31]
@ENDVALUES

DST_X_Y			0x06e8	wo
@VALUES DSTXY_
	x_			[0-14]
	scale_y_sec_		[15]
		+lsb			1
	y_			[16-30]
	scale_y_		[31]
		+lsb			1
@ENDVALUES

DST_Y			0x0508	rw
@VALUES DSTX_
	y_			[0-14]
	scale_y_		[15]
		+lsb			1
@ENDVALUES

DST_Y_X			0x050c	wo
@VALUES DSTYX_
	y_			[0-14]
	scale_y_		[15]
		+lsb			1
	x_			[16-30]
	scale_y_sec_		[31]
		+lsb			1
@ENDVALUES

LEAD_BRES_DEC		0x052c	rw

LEAD_BRES_INC		0x0528	rw

LEAD_BRES_LNTH		0x0544	rw
@VALUES LEAD_
	bres_lnth_		[0-14]
	draw_trap_		[15]
		+init			1
	trail_x_		[16-30]
	bres_lnth_line_		[31]{en/dis}
@ENDVALUES

SRC_CNTL		0x05B4	rw
@VALUES SRC_
	patt_			[0]
	patt_rot_		[1]
	linear_			[2]
	byte_align_		[3]
	line_x_dir_		[4]
	8x8x8_brush_		[5]
	fast_fill_		[6]
	track_dst_		[7]
	bm_			[8]
	bm_gui_			[9]{async/sync}
	bm_op_			[10-11]
		+buf_to_sysmem		0
		+sysmem_to_buf		1
		+reg_to_sysmem		2
		+sysmem_to_reg		3
	8x8x8_brush_next_	[12]{new/hold}
	color_reg_write_	[13]{enable/disable}
	block_write_		[14]{enable/disable}
@ENDVALUES

SRC_HEIGHT1		0x0594	rw
@VALUES SRC_
	height1_		[0-14]
@ENDVALUES

SRC_HEIGHT2		0x05AC	rw
@VALUES SRC_
	height2_		[0-14]
@ENDVALUES

SRC_HEIGHT1_WIDTH1	0x0598	wo
@VALUES SRCHW_
	height1_		[0-14]
	width1_			[16-29]
@ENDVALUES

SRC_HEIGHT2_WIDTH2	0x05b0	wo
@VALUES SRCHW_
	height2_		[0-14]
	width2_			[16-29]
@ENDVALUES

SRC_OFF_PITCH		0x0580	rw
@VALUES SRC_
	offset_			[0-19]
	pitch_			[22-31]
@ENDVALUES

SRC_WIDTH1		0x0590	rw
@VALUES SRC_
	width1_			[0-13]
@ENDVALUES

SRC_WIDTH2		0x05a8	rw
@VALUES SRC_
	width2_			[0-13]
@ENDVALUES

SRC_X			0x0584	rw
@VALUES SRC_
	x_			[0-13]
@ENDVALUES

SRC_X_START		0x059c	rw
@VALUES SRC_
	x_start_		[0-13]
@ENDVALUES

SRC_Y			0x0588	rw
@VALUES SRC_
	y_			[0-14]
@ENDVALUES

SRC_Y_START		0x05a0	rw
@VALUES SRC_
	y_start_		[0-14]
@ENDVALUES

SRC_Y_X			0x058c	wo
@VALUES SRCYX_
	y_			[0-14]
	x_			[16-29]
@ENDVALUES

SRC_Y_X_START		0x05a4	wo
@VALUES SRCYX_
	y_start_		[0-14]
	x_start_		[16-29]
@ENDVALUES

TRAIL_BRES_DEC	       	0x0540	rw
@VALUES TRAIL_
	bres_dec_		[0-17]
@ENDVALUES

TRAIL_BRES_ERR		0x0538	rw
@VALUES TRAIL_
	bres_err_		[0-17]
@ENDVALUES

TRAIL_BRES_INC		0x053c	rw
@VALUES TRAIL_
	bres_inc_		[0-17]
@ENDVALUES

USR_DST_PITCH		0x06f0	wo
@VALUES USR_
	dst_pitch_		[0-9]
@ENDVALUES

Z_CNTL			0x054c	rw
@VALUES Z_
	tst_			[0]
	src_			[1]{normal/with_2d}
	test_			[4-6]
		+znever			0
		+zlt			1
		+zlte			2
		+ze			3
		+zgte			4
		+zgt			5
		+zne			6
		+zalways		7
	mask_			[8]
@ENDVALUES

Z_OFF_PITCH		0x0548	rw
@VALUES Z_
	offset_			[0-19]
	pitch_			[22-31]
@ENDVALUES



@BCOMM Draw Engine Control Memory Mapped Registers


CLR_CMP_CLR		0x0700	rw

CLR_CMP_CNTL		0x0708	rw
@VALUES CLR_
	cmp_fcn_		[0-2]
		+false			0
		+true			1
		+dst_neq_cmp		4
		+dst_eq_cmp		5
	cmp_src_		[23-24]
		+dst			0
		+2d_src			1
		+texel			2
		+scaler			2
@ENDVALUES

CLR_CMP_MASK		0x0704	rw

DP_BKGD_CLR		0x06c0	rw

DP_CHAIN_MASK		0x06cc	rw

DP_FOG_CLR		0x06c4	rw

DP_FGRD_BKGD_CLR	0x06e0	wo
@VALUES DP_
	frgd_clr_		[0-15]
	bkgn_clr_		[16-31]
@ENDVALUES

DP_FRGD_CLR		0x06C4	rw

DP_FGRD_CLR_MIX		0x06DC	wo
@VALUES DPM_
	frgd_clr_		[0-15]
	frgd_mix_		[16-20]
	bkgd_mix_		[24-28]
@ENDVALUES

DP_MIX			0x06D4	rw
@VALUES DP_
	bkgd_mix_		[0-4]
	frgd_mix_		[16-20]
@ENDVALUES

DP_PIX_WIDTH		0x06D0	rw
@VALUES DP_
	dst_pix_width_		[0-3]
		+mono			0
		+8bpp_pseudo		2
		+16bpp_1555		3
		+16bpp_565		4
		+32bpp_8888		6
		+8bpp_332		7
		+y8_grey		8
		+rgb8_grey		9
		+vyuy422		11
		+yvyu422		12
		+ayuv444		14
		+argb4444		15
	comp_pix_width_		[4-7]
		+8bpp_pseudo		2
		+16bpp_1555		3
		+16bpp_565		4
		+32bpp_8888		6
		+8bpp_332		7
		+y8_grey		8
		+rgb8_grey		9
		+vyuy422		11
		+yvyu422		12
		+ayuv444		14
		+argb4444		15
	src_pix_width_		[8-11]
		+mono			0
		+8bpp_pseudo		2
		+16bpp_1555		3
		+16bpp_565		4
		+32bpp_8888		6
		+8bpp_332		7
		+y8_grey		8
		+vyuy422		11
		+yvyu422		12
		+ayuv444		14
		+argb4444		15
	host_triple_		[13]
	host_pix_width_		[16-19]
		+mono			0
		+8bpp_pseudo		2
		+16bpp_1555		3
		+16bpp_565		4
		+32bpp_8888		6
		+8bpp_332		7
		+y8_grey		8
		+rgb8_grey		9
		+vyuy422		11
		+yvyu422		12
		+ayuv444		14
		+argb4444		15
	ci4_rgb_index_		[20-23]
	byte_pix_order_		[24]{msb_to_lsb/lsb_to_msb}
	conversion_temp_	[25]{r6500k/rgb9300k}
	ci4_rgb_low_nibble_	[26]
	ci4_rgb_high_nibble_	[27]
	scale_pix_width_		[4-7]
		+8bpp_pseudo		2
		+16bpp_1555		3
		+16bpp_565		4
		+32bpp_8888		6
		+8bpp_332		7
		+y8_grey		8
		+rgb8_grey		9
		+vyuy422		11
		+yvyu422		12
		+ayuv444		14
		+argb4444		15
@ENDVALUES

DP_SET_GUI_ENGINE	0x06FC	wo
@VALUES SET_
	dst_pix_width_		[3-5]
		+mono			0
		+8bpp			2
		+15bpp			3
		+16bpp			4
		+32bpp			6
	src_pix_width_		[6]{mono/same}
	offset_			[7-9]
		+0k			0
		+256k			1
		+512k			2
		+768k			3
		+1mb			4
	pitch_			[10-13]
		+usr			0
		+320			1
		+352			2
		+384			3
		+640			4
		+800			5
		+896			6
		+512			7
		+1024			8
		+1152			9
		+1280			10
		+400			11
		+832			12
		+1600			13
		+448			14
		+2048			15
	dst_pitch_mult_		[14]{normal/by_2}
	off_pitch_		[15]{0/copy_dst}
	src_hw1_hw2_		[16-17]
		+8_8			0
		+1_32			1
		+8_24			2
	drawing_combo_		[20-23]
	bm_op_			[24-25]
		+buf_to_sysmem		0
		+sysmem_to_buf		1
		+reg_to_sysmem		2
		+sysmem_to_reg		3
	bm_			[26]
	bm_gui_			[27]{async/sync}
	host_triple_		[28]
	fast_fill_		[29]{enable/disable}
	block_write_		[29]{enable/disable}
@ENDVALUES

DP_SET_GUI_ENGINE2	0x06F8	wo

DP_SRC			0x06D8	rw
@VALUES DP_
	bkgd_src_		[0-2]
		+background		0
		+foreground		1
		+host_data		2
		+blit_src		3
		+pattern		4
		+scaler_data		5
		+3d_data		5
	frgd_src_		[8-10]
		+background		0
		+foreground		1
		+host_data		2
		+blit_src		3
		+pattern		4
		+scaler_data		5
		+3d_data		5
	mono_src_		[16-17]
		+1			0
		+pattern		1
		+host_data		2
		+blit_src		3
@ENDVALUES

DP_WRITE_MASK		0x06C8	rw

FIFO_STAT		0x0710	ro
@VALUES FIFO_
	stat_			[0-15]
@ENDVALUES

GUI_CMDFIFO_DEBUG	0x0170	ro
@VALUES GUI_
	reg_index_		[0-7]
	radr_			[8-15]
	wadr_			[16-23]
	ren_			[24]
	wen_			[25]
	snoop_			[31]
@ENDVALUES

GUI_CMDFIFO_DATA	0x0174	ro

GUI_CNTL		0x0178	ro
@VALUES GUI_
	cmdfifo_size_		[0-1]
		+192			0
		+128			1
		+64			2
@ENDVALUES

GUI_STAT		0x0738	ro

GUI_TRAJ_CNTL		0x0730	rw		######## I'm up to here #######
@VALUES GUI_
	dst_x_dir_		[0]
	dst_y_dir_		[1]
	dst_y_major_		[2]
	dst_x_tile_		[3]
	dst_y_tile_		[4]
	dst_last_pel_		[5]
	dst_polygon_		[6]
	dst_24_rot_en_		[7]
	dst_24_rot_		[8-10]
	dst_bres_zero_		[11]
	dst_poly_rtedge_	[12]{enable/disable}
	trail_x_dir_		[13]
	trap_fill_dir_		[14]
	trail_bres_sign_	[15]
@ENDVALUES

HOST_CNTL		0x0640
@VALUES HOST_
	byte_align_		[0]
	big_endian_		[1]
@ENDVALUES

HOST_DATA0		0x0600	wo
HOST_DATA1		0x0604	wo
HOST_DATA2		0x0608	wo
HOST_DATA3		0x060c	wo
HOST_DATA4		0x0610	wo
HOST_DATA5		0x0614	wo
HOST_DATA6		0x0618	wo
HOST_DATA7		0x061c	wo
HOST_DATA8		0x0620	wo
HOST_DATA9		0x0624	wo
HOST_DATAA		0x0628	wo
HOST_DATAB		0x062c	wo
HOST_DATAC		0x0630	wo
HOST_DATAD		0x0634	wo
HOST_DATAE		0x0638	wo
HOST_DATAF		0x063c	wo

PAT_CNTL		0x0688	rw
@VALUES PAT_
	mono_			[0]
	clr_4x2_		[1]
	clr_8x1_		[2]
@ENDVALUES

PAT_REG0		0x0680	rw
PAT_REG1		0x0684	rw

SC_BOTTOM		0x06b0	rw
@VALUES SC_
	top_			[0-14]
@ENDVALUES

SC_LEFT			0x06a0	rw
@VALUES SC_
	left_			[0-13]
@ENDVALUES

SC_LEFT_RIGHT		0x06a8	wo
@VALUES SCLR_
	left_			[0-13]
	right_			[16-29]
@ENDVALUES

SC_RIGHT		0x06a4	rw
@VALUES SC_
	right_			[0-13]
@ENDVALUES

SC_TOP			0x06ac	rw
@VALUES SC_
	top_			[0-14]
@ENDVALUES

SC_TOP_BOTTOM		0x06b4	wo
@VALUES SCTB_
	top_			[0-14]
	bottom_			[16-30]
@ENDVALUES



@BCOMM Scaler and 3D Accelerator Memory Mapped Registers


@BCOMM Front-End Scaler Pipe Memory Mapped Registers


SCALE_3D_CNTL		0x05fc	rw

SCALE_DST_Y_X		0x078c	rw

SCALE_HACC		0x07c8	rw

SCALE_HEIGHT		0x05e0	rw

SCALE_OFF		0x05c0	rw

SCALE_OFF_ACC		0x0788	wo

SCALE_PITCH		0x0750	rw

SCALE_PITCH_BOTH	0x0780	wo

SCALE_VACC		0x05f8	rw

SCALE_WIDTH		0x05dc	rw

SCALE_X_INC		0x07c0	rw

SCALE_Y_INC		0x07cc	rw

SECONDARY_SCALE_HACC	0x07a4	rw

SECONDARY_SCALE_OFF	0x0778	rw

SECONDARY_SCALE_OFF_ACC	0x0784	wo

SECONDARY_SCALE_PITCH	0x0768	rw

SECONDARY_SCALE_VACC	0x07d4	rw

SECONDARY_SCALE_X_INC	0x079c	rw

SECONDARY_SCALE_Y_INC	0x07d0	rw



@BCOMM Texture Mapping Memory Mapped Registers


LOG_MAX_INC		0x0748	rw

S_START			0x0754	rw

S_X_INC			0x074c	rw

S_Y_INC			0x0750	rw

SECONDARY_S_START	0x0564	rw

SECONDARY_S_X_INC	0x055c	rw

SECONDARY_S_Y_INC	0x0560	rw

SECONDARY_STW_EXP	0x0558	rw

SECONDARY_TEX_OFFSET	0x0778	rw

SECONDARY_T_START	0x057c	rw

SECONDARY_T_X_INC	0x0574	rw

SECONDARY_T_Y_INC	0x0578	rw

SECONDARY_W_START	0x0570	rw

SECONDARY_W_X_INC	0x0568	rw

SECONDARY_W_Y_INC	0x056c	rw

STW_EXP			0x0744	rw

T_START			0x076c	rw

T_X_INC			0x0764	rw

T_Y_INC			0x0768	rw

TEX_0_OFF		0x05c0	rw

TEX_1_OFF		0x05c4	rw

TEX_2_OFF		0x05c8	rw

TEX_3_OFF		0x05cc	rw

TEX_4_OFF		0x05d0	rw

TEX_5_OFF		0x05d4	rw

TEX_6_OFF		0x05d8	rw

TEX_7_OFF		0x05dc	rw

TEX_8_OFF		0x05e0	rw

TEX_9_OFF		0x05e4	rw

TEX_10_OFF		0x05e8	rw

TEX_CNTL		0x0774	rw

TEX_PALETTE		0x077c	rw

TEX_PALETTE_INDEX	0x0740	rw

TEX_SIZE_PITCH		0x0770	rw

W_START			0x0760	rw

W_X_INC			0x0758	rw

W_Y_INC			0x075c	rw



@BCOMM Specular, Color, Z and Alpha Interpolator Memory Mapped Registers


ALPHA_START		0x07f8	rw

ALPHA_X_INC		0x07f0	rw

ALPHA_Y_INC		0x07f4	rw

BLUE_START		0x07e0	rw

BLUE_X_INC		0x07d8	rw

BLUE_Y_INC		0x07dc	rw

GREEN_START		0x07d4	rw

GREEN_X_INC		0x07cc	rw

GREEN_Y_INC		0x07d0	rw

RED_START		0x07c8	rw

RED_X_INC		0x07c0	rw

RED_Y_INC		0x07c4	rw

SPECULAR_BLUE_START	0x07bc	rw

SPECULAR_BLUE_X_INC	0x07b4	rw

SPECULAR_BLUE_Y_INC	0x07b8	rw

SPECULAR_GREEN_START	0x07b0	rw

SPECULAR_GREEN_X_INC	0x07a8	rw

SPECULAR_GREEN_Y_INC	0x07ac	rw

SPECULAR_RED_START	0x07a4	rw

SPECULAR_RED_X_INC	0x079c	rw

SPECULAR_RED_Y_INC	0x07a0	rw

Z_START			0x07ec	rw

Z_X_INC			0x07e4	rw

Z_Y_INC			0x07e8	rw



@BCOMM Setup Engine Memory Mapped Registers


ONE_OVER_AREA		0x029c	rw

ONE_OVER_AREA_UC	0x0300	rw

SETUP_CNTL		0x0304	rw

VERTEX_1_ARGB		0x0254	rw

VERTEX_1_S		0x0240	rw

VERTEX_1_SECONDARY_S	0x0328	rw

VERTEX_1_SECONDARY_T	0x032c	rw

VERTEX_1_SECONDARY_W	0x0330	rw

VERTEX_1_SPEC_ARGB	0x024c	rw

VERTEX_1_T		0x0244	rw

VERTEX_1_W		0x0248	rw

VERTEX_1_X_Y		0x0258	rw

VERTEX_1_Z		0x0250	rw

VERTEX_2_ARGB		0x0274	rw

VERTEX_2_S		0x0260	rw

VERTEX_2_SECONDARY_S	0x0334	rw

VERTEX_2_SECONDARY_T	0x0338	rw

VERTEX_2_SECONDARY_W	0x033c	rw

VERTEX_2_SPEC_ARGB	0x026c	rw

VERTEX_2_T		0x0264	rw

VERTEX_2_W		0x0268	rw

VERTEX_2_X_Y		0x0278	rw

VERTEX_2_Z		0x0270	rw

VERTEX_3_ARGB		0x0294	rw

VERTEX_3_S		0x0280	rw

VERTEX_3_SECONDARY_S	0x02a0	rw

VERTEX_3_SECONDARY_T	0x02a4	rw

VERTEX_3_SECONDARY_W	0x02a8	rw

VERTEX_3_SPEC_ARGB	0x028c	rw

VERTEX_3_T		0x0284	rw

VERTEX_3_W		0x0288	rw

VERTEX_3_X_Y		0x0298	rw

VERTEX_3_Z		0x0290	rw



@BCOMM Bus Mastering DMA Memory Mapped Registers


BM_ADDR			0x0648	wo

BM_COMMAND		0x0188	ro

BM_DATA			0x0648	wo

BM_FRAME_BUF_OFFSET	0x0180	ro

BM_GUI_TABLE		0x01b8	rw

BM_GUI_TABLE_CMD	0x064c	rw

BM_HOSTDATA		0x0644	wo

BM_STATUS		0x018c	ro

BM_SYSTEM_MEM_ADDR	0x0184	ro

BM_SYSTEM_TABLE		0x01bc	rw



@BCOMM AGP Memory Mapped Registers


AGP_BASE		0x0148	rw

AGP_CNTL		0x014c	rw



@END

###############################################################################
EOF

sub myprint;

$curr = `pwd`;
chomp $curr;

open OUT, ">$curr/mach64regs.h";

myprint $code_start;
myprint "/*************** (START) AUTOMATICLY GENERATED REGISTER FILE *****************/";
myprint "\n/*\n * Generated on ", `date`, " */\n\n";

for $parse (0,1) {

    $mode=0;

    my @lines=split(/\n/, $text);

    MAIN_LOOP:
    for (my $line_index=0; $line_index < $#lines; $line_index++) {
	$line = $lines[$line_index];

	$line =~ s/^\s*//;
	$line =~ s/\s*$//;
	$line =~ s/#.*//;
	$line || next MAIN_LOOP;

	if ($line =~ /\@BCOMM\s+(.*)/) {
		$comm = $1;
		$parse || myprint "\n/*\n * ", $comm, "\n */\n\n";
		next;
	}

	if ($line =~ /\@LCOMM\s+(.*)/) {
		$comm = $1;
		$parse || myprint "\n/* ", $comm, " */\n\n";
		next;
	}

	if ($line =~ /\@PCI/) {
		$name = "PCI Config Region";
		$parse || do {  myprint "\n"; $mode = 2; };
		next;
	}

	if ($line =~ /\@REGION0/) {
		$name = "Region 0 Registers";
		$parse || do {  myprint "\n"; $mode = 3; };
		next;
	}

	if ($line =~ /\@VALUES\s+(\w+)/) {
		$vpref = $1;
		$lastmode = $mode;
		$mode = 0;
		$parse || do { myprint "\n"; $mode = 4; };
		($strunset, $strset) = ("disable", "enable");
		$single_bit = 0;
		next;
	}

	if ($line =~ /\@ENDVALUES/) {
		$name = "";
		if( $single_bit ) {
		    val_formats("$vpref$lastval$l_strunset", "0x0", "");
		    val_formats("$vpref$lastval$l_strset", $bvset, "");
		}
		$parse || myprint "\n";
		$mode = $lastmode;
		next;
	}

	if ($line =~ /\@END/) {
		$name = "";
		$parse || myprint "\n\n\n";
		$mode = 0;
		next;
	}

	PCI_MODE: { $mode == 2 || last PCI_MODE;
	    ($reg, $offset, $access) = split ' ', $line;
	    reg_simplef($reg, $offset);
	    next MAIN_LOOP;
	}

	REGION0_MODE: { $mode == 3 || last REGION0_MODE;
	    ($reg, $offset, $access) = split ' ', $line;
	    reg_simplef($reg, $offset);
	    next MAIN_LOOP;
	}

	VALUES_MODE: { $mode == 4 || last VALUES_MODE;
	    if( $line =~ /(\w+)\s+\[\s*([0-9- ]+)\s*\]/ ) {
	        ($val, $bit) = ($1, $2);
		if( $line !~ /\]\s*\*/ ) {
		    ($strunset, $strset) = ("disable", "enable");
		}
		if( $line =~ /\]\s*\{\s*(\w+)\s*\/(\w+)\s*\}/ ) {
		    ($strunset, $strset) = ($1, $2);
		}
		if( $single_bit ) {
		    val_formats("$vpref$lastval$l_strunset", "0x0", "");
		    val_formats("$vpref$lastval$l_strset", $bvset, "");
		}

		if( $bit =~ /(\d+)\s*-\s*(\d+)/ ) {
		    ($bfrom, $bto) = ($1, $2);
		    $single_bit = 0;
		}
		elsif ( $bit =~ /(\d+)/ ) {
		    $bfrom = $bto = $1;
		    ($lastval,$l_strunset,$l_strset)=($val,$strunset,$strset);
		    $single_bit = 1;
		}
		else {
		    print("/* BIT RANGE SYNTAX ERROR ENCOUNTERD! */\n");
		    myprint("/* BIT RANGE SYNTAX ERROR ENCOUNTERD! */\n");
		}

		($bvmask, $bvset) = get_bits_value($bfrom,$bto);
		$comment = "/* bits $bfrom-$bto */";
		if($bfrom == $bto) { $comment = "/* bit $bfrom */"; }
		val_formats("$vpref${val}MASK", $bvmask, $comment);
		if (($bto - $bfrom > 0 || $bto eq '') && $lines[$line_index+1] !~ /\s*\+(\w+)\s+(\d+)/) {
			val_formats("$vpref${val}SHIFT", $bfrom, "");
		}
	    }
	    elsif($line =~ /\s*\+(\w+)\s+(\d+)/) {
		($vsuf, $rv) = ($1, $2);
		$single_bit = 0;
		$srv = shift_value_nrbits($rv, $bfrom);
		$comment = "/* val $rv, shift $bfrom */";
		val_formats("$vpref$val$vsuf", $srv, $comment);
	    }
	    else {
		print("/* VALUE LINE SYNTAX ERROR ENCOUNTERD! */\n");
		myprint("/* VALUE LINE SYNTAX ERROR ENCOUNTERD! */\n");
	    }
	    next MAIN_LOOP;
	}
    }
}

myprint "/**************** (END) AUTOMATICLY GENERATED REGISTER FILE ******************/\n";
myprint $code_end;

close OUT;



sub shift_value_nrbits
{
	my ($val, $nrbits) = @_;
	my ($ret);

	$ret = sprintf "0x%x", ($val << $nrbits);

	return $ret;
}



sub get_bits_value
{
	my ($start, $end) = @_;
	my ($cnt, $bmask, $bset);

	$bset = 0;
	for $cnt ($start..$end) {
	    $bset |= (1 << $cnt);
	}
	$bmask = sprintf "0x%x", ~$bset;
	$bset = sprintf "0x%x", $bset;

	return ($bmask,$bset);
}



sub val_formats
{
	my ($reg, $rval, $comm) = @_;
	my ($cs, $len, $rtab, $vtab);
	
	$len = length($reg);
	$rtab =" "."\t"x(5 - int(($len + 13)/8) );

	$len = length($rval);
	$vtab =" "."\t"x(2 - int(($len + 1)/8) );

	myprint "    #define ", $reg, $rtab, $rval, $vtab, "$comm\n";
}



sub reg_simplef
{
	my ($reg, $offset) = @_;
	my ($len, $tab);
	
	$len = length($reg);
	$tab =" "."\t"x(5 - int(($len + 9)/8) );

	myprint "#define MACH64REG_", $reg, $tab, $offset;
	myprint "\n";
}




sub myprint { foreach $txt (@_) { print OUT $txt; } }



