/*
 * GLX Hardware Device Driver for Intel i810
 * Copyright (C) 1999 Keith Whitwell
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * KEITH WHITWELL, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR 
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE 
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 */

/* $Id: i810tex.c,v 1.1 1999/12/13 12:47:45 keithw Exp $ */

#include "xsmesaP.h"
#include <stdlib.h>
#include <stdio.h>

#include <GL/gl.h>

#include "mm.h"
#include "i810lib.h"
#include "i810tex.h"
#include "i810log.h"
#include "simple_list.h"
#include "enums.h"

static void i810SetTexWrapping(i810TextureObjectPtr tex, GLenum s, GLenum t)
{
   unsigned int val = tex->Setup[I810_TEXREG_MCS];

   val &= ~(MCS_U_STATE_MASK|MCS_V_STATE_MASK);
   val |= (MCS_U_WRAP|MCS_V_WRAP);
   
   if (s != GL_REPEAT) 
      val ^= (MCS_U_WRAP^MCS_U_CLAMP);

   if (t != GL_REPEAT) 
      val ^= (MCS_V_WRAP^MCS_V_CLAMP);

   tex->Setup[I810_TEXREG_MCS] = val;
   tex->reg_dirty |= (1 << I810_TEXREG_MCS);
}

static void i810SetTexFilter(i810TextureObjectPtr t, GLenum minf, GLenum magf)
{
   GLuint LastLevel;

   switch (minf) {
   case GL_NEAREST:
      I810_SET_FIELD(t->Setup[I810_TEXREG_MF],
		     MF_MIN_MASK | MF_MIP_MASK,
		     MF_MIN_NEAREST | MF_MIP_NONE);
      break;
   case GL_LINEAR:
      I810_SET_FIELD(t->Setup[I810_TEXREG_MF],
		     MF_MIN_MASK | MF_MIP_MASK,
		     MF_MIN_LINEAR | MF_MIP_NONE);
      break;
   case GL_NEAREST_MIPMAP_NEAREST:
      I810_SET_FIELD(t->Setup[I810_TEXREG_MF],
		     MF_MIN_MASK | MF_MIP_MASK,
		     MF_MIN_NEAREST | MF_MIP_NEAREST);
      break;
   case GL_LINEAR_MIPMAP_NEAREST:
      I810_SET_FIELD(t->Setup[I810_TEXREG_MF],
		     MF_MIN_MASK | MF_MIP_MASK,
		     MF_MIN_LINEAR | MF_MIP_NEAREST);
      break;
   case GL_NEAREST_MIPMAP_LINEAR:
      I810_SET_FIELD(t->Setup[I810_TEXREG_MF],
		     MF_MIN_MASK | MF_MIP_MASK,
		     MF_MIN_NEAREST | MF_MIP_DITHER );
      break;
   case GL_LINEAR_MIPMAP_LINEAR:
      I810_SET_FIELD(t->Setup[I810_TEXREG_MF],
		     MF_MIN_MASK | MF_MIP_MASK,
		     MF_MIN_LINEAR  | MF_MIP_DITHER );
      break;
   default:
      i810Error("i810SetTexFilter(): not supported min. filter %d\n",(int)minf);
      break;
   }

   switch (magf) {
   case GL_NEAREST:
      I810_SET_FIELD(t->Setup[I810_TEXREG_MF],
		     MF_MAG_MASK,MF_MAG_NEAREST);
      break;
   case GL_LINEAR:
      I810_SET_FIELD(t->Setup[I810_TEXREG_MF],
		     MF_MAG_MASK,MF_MAG_LINEAR);
      break;
   default:
      i810Error("i810SetTexFilter(): not supported mag. filter %d\n",(int)magf);
      break;
   }
  

   if (t->globj->MinFilter != GL_NEAREST && 
       t->globj->MinFilter != GL_LINEAR) {
      LastLevel = t->max_level;
   } else {
      LastLevel = t->min_level;
   }

   I810_SET_FIELD(t->Setup[I810_TEXREG_MLL],
		  MLL_MAX_MIP_MASK,
		  (t->min_level << (MLL_MAX_MIP_SHIFT+4)));

   I810_SET_FIELD(t->Setup[I810_TEXREG_MLL],
		  MLL_MIN_MIP_MASK,
		  (LastLevel << MLL_MIN_MIP_SHIFT));

   /* See OpenGL 1.2 specification */
   if (magf == GL_LINEAR && (minf == GL_NEAREST_MIPMAP_NEAREST || 
			     minf == GL_NEAREST_MIPMAP_LINEAR))
   {
      /* c = 0.5 */
      I810_SET_FIELD(t->Setup[I810_TEXREG_MLC],
		     MLC_LOD_BIAS_MASK,
		     0x10);
   } else {
      /* c = 0 */
      I810_SET_FIELD(t->Setup[I810_TEXREG_MLC],
		     MLC_LOD_BIAS_MASK,
		     0x0);
   }

   t->reg_dirty |= ((1<<I810_TEXREG_MLC) |
		    (1<<I810_TEXREG_MLL) |
		    (1<<I810_TEXREG_MF));
}


/* Need a fallback ?
 */
static void i810SetTexBorderColor(i810TextureObjectPtr t, GLubyte color[4])
{
/*    t->Setup[I810_TEXREG_TEXBORDERCOL] =  */
/*      I810PACKCOLOR8888(color[0],color[1],color[2],color[3]); */
}



static void ReplicateMesaTexState(i810TextureObjectPtr t,
                                  struct gl_texture_object *mesatex)
{
   i810SetTexWrapping(t,mesatex->WrapS,mesatex->WrapT);
   i810SetTexFilter(t,mesatex->MinFilter,mesatex->MagFilter);
   i810SetTexBorderColor(t,mesatex->BorderColor);
}

i810TextureObjectPtr i810CreateTexObj(i810ContextPtr ctx,
				      struct gl_texture_object *tObj)
{
   i810TextureObjectPtr t;
   GLuint height, width, pitch, i, textureFormat, log_pitch;
   struct gl_texture_image *image;

   image = tObj->Image[ 0 ];
   if ( !image ) {
      fprintf(stderr, "no image at level zero - not creating texobj\n");
      return 0;
   }
   
   t = (i810TextureObjectPtr) calloc(1,sizeof(*t));
   if (!t)
      return 0;

   switch( image->Format ) {
   case GL_RGB:
   case GL_LUMINANCE:
   case GL_ALPHA:
      t->texelBytes = 2;
      textureFormat = MI1_FMT_16BPP | MI1_PF_16BPP_RGB565;
      break;
   case GL_LUMINANCE_ALPHA:
   case GL_INTENSITY:
   case GL_RGBA:
      t->texelBytes = 2;
      textureFormat = MI1_FMT_16BPP | MI1_PF_16BPP_ARGB4444;
      break;
   case GL_COLOR_INDEX:
      textureFormat = MI1_FMT_8CI | MI1_PF_8CI_ARGB4444;
      t->texelBytes = 1;
      t->UsePalette = 1;
      break;
   default:
      i810Error( "i810CreateTexObj: bad image->Format\n" );
      free( t );      
      return 0;	
   }


   /* Figure out the size now (and count the levels).  Upload won't be done
    * until later.
    */ 
   width = image->Width * t->texelBytes;
   for (pitch = 32, log_pitch=2 ; pitch < width ; pitch *= 2 )
      log_pitch++;
   
   t->dirty_images = 0;
   
   for ( height = i = 0 ; i < I810_TEX_MAXLEVELS && tObj->Image[i] ; i++ ) {
      t->image[i].image = tObj->Image[i];
      t->image[i].offset = height * pitch;
      t->image[i].internalFormat = image->Format;
      t->dirty_images |= (1<<i);
      height += t->image[i].image->Height;
   }

   t->Pitch = pitch;
   t->totalSize = height*pitch;
   t->max_level = i-1;
   t->min_level = 0;
   t->globj = tObj;
   t->age = 0;
   t->reg_dirty = ~0;

   t->Setup[I810_TEXREG_MI0] = GFX_OP_MAP_INFO;

   t->Setup[I810_TEXREG_MI1] = (MI1_MAP_0 |
				textureFormat |
				log_pitch);			 

   t->Setup[I810_TEXREG_MI2] = (MI2_DIMENSIONS_ARE_LOG2 |
				(image->HeightLog2 << 16) |
				(image->WidthLog2));

   t->Setup[I810_TEXREG_MI3] = 0;
  
   t->Setup[I810_TEXREG_MLC] = (GFX_OP_MAP_LOD_CTL | 
				MLC_MAP_0 |
				MLC_DITHER_WEIGHT_FULL |
				MLC_UPDATE_LOD_BIAS |
				0x0);

   t->Setup[I810_TEXREG_MLL] = (GFX_OP_MAP_LOD_LIMITS |
				MLL_MAP_0  |
				MLL_UPDATE_MAX_MIP | 
				(t->min_level << MLL_MAX_MIP_SHIFT) |
				MLL_UPDATE_MIN_MIP |
				t->max_level);

   /* I think this is context state, really.
    */
   t->Setup[I810_TEXREG_MCS] = (GFX_OP_MAP_COORD_SETS |
				MCS_COORD_0 |
				MCS_UPDATE_NORMALIZED |
				MCS_NORMALIZED_COORDS |
				MCS_UPDATE_V_STATE |
				MCS_V_WRAP |
				MCS_UPDATE_U_STATE |
				MCS_U_WRAP);

   t->Setup[I810_TEXREG_MF] = (GFX_OP_MAP_FILTER |
			       MF_MAP_0 |
			       MF_UPDATE_ANISOTROPIC |
			       0 |
			       MF_UPDATE_MIP_FILTER |
			       MF_MIP_NEAREST |
			       MF_UPDATE_MAG_FILTER |
			       MF_MAG_NEAREST |
			       MF_UPDATE_MIN_FILTER |
			       MF_MIN_NEAREST);

   t->current_unit = 0;

   ReplicateMesaTexState(t,tObj);
   tObj->DriverData = t;
   insert_at_head(&ctx->SwappedOut, t);
   return t;
}

int i810DestroyTexObj(i810ContextPtr ctx, i810TextureObjectPtr t)
{
   if (t) {
      if (t->age > i810glx.dma_buffer_age)
	 i810WaitDrawingEngine();

      mmFreeMem(t->MemBlock);
      t->MemBlock = 0;      
      remove_from_list(t);
      free(t);
   }
   return 0;
}


/* Card memory management
 */
static void i810SwapOutOldestTexObj( i810ContextPtr ctx )
{
   i810TextureObjectPtr t = ctx->TexObjList.prev;

   fprintf(stderr, "swap out oldest, age %d, dma age %d\n",
	   t->age, i810glx.dma_buffer_age);

   if (t->age > i810glx.dma_buffer_age)
      i810WaitDrawingEngine();	

   mmFreeMem( t->MemBlock );
   t->MemBlock = 0;      
   t->dirty_images = ~0;
   move_to_tail(&ctx->SwappedOut, t);
}

/* Just allocates the memory - doesn't do the upload.
 */
static int i810SwapInTexObj( i810ContextPtr ctx, i810TextureObjectPtr t)
{
   i810Msg(10,"  Swapping in texture.\n");
   i810glx.c_textureSwaps++;

   /* Try to allocate textureHeap memory, swapping out stuff until we
    * can.
    */
   while (1)
   {
      t->MemBlock = mmAllocMem( i810glx.sysmemHeap, t->totalSize, 12, 0 );

      if (t->MemBlock)
	 break;

      if (is_empty_list(&ctx->TexObjList))
	 return -1;

      i810SwapOutOldestTexObj( ctx );
   }
 
   t->Setup[I810_TEXREG_MI3] = t->MemBlock->ofs;
   t->BufAddr = i810glx.sysmemVirtual + t->MemBlock->ofs;
   t->reg_dirty |= ((1<<I810_TEXREG_MI0) |
		    (1<<I810_TEXREG_MI1) |
		    (1<<I810_TEXREG_MI2) |
		    (1<<I810_TEXREG_MI3));
   return 0;
}

/* Upload an image from mesa's internal copy.
 */
static int i810UploadTexLevel( i810TextureObjectPtr t, int level )
{
   const struct gl_texture_image *image = t->image[level].image;
   int i,j;

   i810Msg(10,"CopyImage():\n");

   /* Need triangle (rather than pixel) fallbacks to simulate this using
    * normal textured triangles.
    */
   if (image->Border != 0) 
      i810Error("Not supported texture border %d.\n",image->Border);

   switch (t->image[level].internalFormat) {
   case GL_RGB:
   {
      GLushort *dst = (GLushort *)(t->BufAddr + t->image[level].offset);
      GLubyte  *src = (GLubyte *)image->Data;
      i810Msg(10,"  CopyRGB: %p %p\n", dst, src);      

      for (j = 0 ; j < image->Height ; j++, dst += (t->Pitch/2)) {
	 for (i = 0 ; i < image->Width ; i++) {
	    dst[i] = I810PACKCOLOR565(src[0],src[1],src[2]);
	    src += 3;
	 }
      }
   }
   break;
      
   case GL_RGBA:
   {
      GLushort *dst = (GLushort *)(t->BufAddr + t->image[level].offset);
      GLubyte  *src = (GLubyte *)image->Data;
      i810Msg(10,"  CopyRGBA: %p %p\n", dst, src);      

      for (j = 0 ; j < image->Height ; j++, dst += (t->Pitch/2)) {
	 for (i = 0 ; i < image->Width ; i++) {
	    dst[i] = I810PACKCOLOR4444(src[0],src[1],src[2],src[3]);
	    src += 4;
	 }
      }
   }
   break;
      
   case GL_LUMINANCE:
   {
      GLushort *dst = (GLushort *)(t->BufAddr + t->image[level].offset);
      GLubyte  *src = (GLubyte *)image->Data;
      i810Msg(10,"  CopyL: %p %p\n", dst, src);      

      for (j = 0 ; j < image->Height ; j++, dst += (t->Pitch/2)) {
	 for (i = 0 ; i < image->Width ; i++) {
	    dst[i] = I810PACKCOLOR565(src[0],src[0],src[0]);
	    src ++;
	 }
      }
   }
   break;

   case GL_INTENSITY:
   {
      GLushort *dst = (GLushort *)(t->BufAddr + t->image[level].offset);
      GLubyte  *src = (GLubyte *)image->Data;
      int i;
      i810Msg(10,"  CopyI: %p %p\n", dst, src);      

      for (j = 0 ; j < image->Height ; j++, dst += (t->Pitch/2)) {
	 for (i = 0 ; i < image->Width ; i++) {
	    dst[i] = I810PACKCOLOR4444(src[0],src[0],src[0],src[0]);
	    src ++;
	 }
      }
   }
   break;
      
   case GL_LUMINANCE_ALPHA:
   {
      GLushort *dst = (GLushort *)(t->BufAddr + t->image[level].offset);
      GLubyte  *src = (GLubyte *)image->Data;
      i810Msg(10,"  CopyLA: %p %p\n", dst, src);      

      for (j = 0 ; j < image->Height ; j++, dst += (t->Pitch/2)) {
	 for (i = 0 ; i < image->Width ; i++) {
	    dst[i] = I810PACKCOLOR4444(src[0],src[0],src[0],src[1]);
	    src += 2;
	 }
      }
   }
   break;

   case GL_COLOR_INDEX:
      {
	 GLubyte *dst = (GLubyte *)(t->BufAddr + t->image[level].offset);
	 GLubyte *src = (GLubyte *)image->Data;
	 i810Msg(10,"  CopyIndex: %p %p\n", dst, src);      

	 for (j = 0 ; j < image->Height ; j++, dst += t->Pitch) {
	    for (i = 0 ; i < image->Width ; i++) {
	       dst[i] = src[0];
	       src += 1;
	    }
	 }

	 t->UsePalette = I810_USE_PALETTE;
      }
   break;
      
   default:
      i810Error("Not supported texture format %d\n",image->Format);
      FatalError("bummer");
      return -1;
   }

   return 0;
}



static int i810UploadTexImages( i810ContextPtr ctx, i810TextureObjectPtr t )
{
   int i;
   i810glx.c_textureSwaps++;

   if (!t->MemBlock) 
      i810SwapInTexObj( ctx, t ); 

   if (t->age > i810glx.dma_buffer_age)
      i810WaitDrawingEngine();

   for (i = t->min_level ; i <= t->max_level ; i++)
      if (t->dirty_images & (1<<i)) {
	 if (i810UploadTexLevel( t, i ) != 0)
	    return -1;
      }

   t->dirty_images = 0;
   return 0;
}

static void i810TexSetUnit( i810TextureObjectPtr t, GLuint unit )
{
   if (t->current_unit == unit) return;

   t->Setup[I810_TEXREG_MI1] ^= (MI1_MAP_0 ^ MI1_MAP_1);
   t->Setup[I810_TEXREG_MLC] ^= (MLC_MAP_0 ^ MLC_MAP_1);
   t->Setup[I810_TEXREG_MLL] ^= (MLL_MAP_0 ^ MLL_MAP_1);
   t->Setup[I810_TEXREG_MCS] ^= (MCS_COORD_0 ^ MCS_COORD_1);
   t->Setup[I810_TEXREG_MF]  ^= (MF_MAP_0 ^ MF_MAP_1);

   t->reg_dirty = ~0;
   t->current_unit = unit;
}




static void i810UpdateTex0State( GLcontext *ctx )
{
   i810ContextPtr imesa = I810_CONTEXT(ctx);
   struct gl_texture_object	*tObj;
   i810TextureObjectPtr t;
   int ma_modulate_op;


   tObj = ctx->Texture.Unit[0].Current;

   if ( tObj != ctx->Texture.Unit[0].CurrentD[2] ) 
      tObj = 0;

   if (!(ctx->Texture.Enabled & 0xf) || !tObj || !tObj->Complete) {

      imesa->Setup[I810_CTXREG_MC0] = ( GFX_OP_MAP_COLOR_STAGES |
					MC_STAGE_0 |
					MC_UPDATE_DEST |
					MC_DEST_CURRENT |
					MC_UPDATE_ARG1 |
					MC_ARG1_ITERATED_COLOR | 
					MC_UPDATE_ARG2 |
					MC_ARG2_ONE |
					MC_UPDATE_OP |
					MC_OP_ARG1 );

      imesa->Setup[I810_CTXREG_MA0] = ( GFX_OP_MAP_ALPHA_STAGES |
					MA_STAGE_0 |
					MA_UPDATE_ARG1 |
					MA_ARG1_ITERATED_ALPHA |
					MA_UPDATE_ARG2 |
					MA_ARG2_TEX0_ALPHA |
					MA_UPDATE_OP |
					MA_OP_ARG1 );
    
      imesa->reg_dirty |= ((1<<I810_CTXREG_MA0) |
			   (1<<I810_CTXREG_MC0));
      return;
   }

   t = tObj->DriverData;
  
   if (!t) {
      t = i810CreateTexObj( imesa, tObj );
      if (!t) return;
   }

   if (t->current_unit != 0)
      i810TexSetUnit( t, 0 );
    
   if (t->dirty_images) 
      i810UploadTexImages(imesa, t);

   t->age = ++i810glx.current_texture_age;
   move_to_head(&imesa->TexObjList, t);

   imesa->CurrentTex0Obj = t;

   if (0)
      if (t->UsePalette == I810_USE_PALETTE) {      
	 if (ctx->Texture.SharedPalette) {
	    if (imesa->GlobalPaletteUpdated)
	       i810LoadTexturePalette(imesa->GlobalPalette,0,256);
	    imesa->GlobalPaletteUpdated = 0;
	 }
	 else {
	    i810LoadTexturePalette(t->Palette,0,256);
	    imesa->GlobalPaletteUpdated = 1;
	 }
      }		      


   if (t->reg_dirty) 
      i810DmaExecute(t->Setup,I810_TEX_SETUP_SIZE);

  
   switch (ctx->Texture.Unit[0].EnvMode) {
      case GL_REPLACE:
      imesa->Setup[I810_CTXREG_MC0] = ( GFX_OP_MAP_COLOR_STAGES |
					MC_STAGE_0 |
					MC_UPDATE_DEST |
					MC_DEST_CURRENT |
					MC_UPDATE_ARG1 |
					MC_ARG1_TEX0_COLOR | 
					MC_UPDATE_ARG2 |
					MC_ARG2_ONE |
					MC_UPDATE_OP |
					MC_OP_ARG1 );

      if (t->image[0].internalFormat == GL_RGB) {
	 ma_modulate_op = MA_OP_ARG1;
      } else {
	 ma_modulate_op = MA_OP_ARG2;
      }

      imesa->Setup[I810_CTXREG_MA0] = ( GFX_OP_MAP_ALPHA_STAGES |
					MA_STAGE_0 |
					MA_UPDATE_ARG1 |
					MA_ARG1_ITERATED_ALPHA |
					MA_UPDATE_ARG2 |
					MA_ARG2_TEX0_ALPHA |
					MA_UPDATE_OP |
					ma_modulate_op );
      break;
   case GL_MODULATE:
      imesa->Setup[I810_CTXREG_MC0] = ( GFX_OP_MAP_COLOR_STAGES |
					MC_STAGE_0 |
					MC_UPDATE_DEST |
					MC_DEST_CURRENT |
					MC_UPDATE_ARG1 |
					MC_ARG1_TEX0_COLOR | 
					MC_UPDATE_ARG2 |
					MC_ARG2_ITERATED_COLOR |
					MC_UPDATE_OP |
					MC_OP_MODULATE );

      if (t->image[0].internalFormat == GL_RGB) {
	 ma_modulate_op = MA_OP_ARG1;
      } else {
	 ma_modulate_op = MA_OP_MODULATE;
      }
	
      imesa->Setup[I810_CTXREG_MA0] = ( GFX_OP_MAP_ALPHA_STAGES |
					MA_STAGE_0 |
					MA_UPDATE_ARG1 |
					MA_ARG1_ITERATED_ALPHA |
					MA_UPDATE_ARG2 |
					MA_ARG2_TEX0_ALPHA |
					MA_UPDATE_OP |
					MA_OP_MODULATE );
      break;
   case GL_DECAL:

      if (t->image[0].internalFormat == GL_RGB) {
	 imesa->Setup[I810_CTXREG_MC0] = ( GFX_OP_MAP_COLOR_STAGES |
					   MC_STAGE_0 |
					   MC_UPDATE_DEST |
					   MC_DEST_CURRENT |
					   MC_UPDATE_ARG1 |
					   MC_ARG1_COLOR_FACTOR | 
					   MC_UPDATE_ARG2 |
					   MC_ARG2_TEX0_COLOR |
					   MC_UPDATE_OP |
					   MC_OP_ARG2 );

      } else {
	 imesa->Setup[I810_CTXREG_MC0] = ( GFX_OP_MAP_COLOR_STAGES |
					   MC_STAGE_0 |
					   MC_UPDATE_DEST |
					   MC_DEST_CURRENT |
					   MC_UPDATE_ARG1 |
					   MC_ARG1_COLOR_FACTOR | 
					   MC_UPDATE_ARG2 |
					   MC_ARG2_TEX0_COLOR |
					   MC_UPDATE_OP |
					   MC_OP_LIN_BLEND_TEX0_ALPHA );
      }

      imesa->Setup[I810_CTXREG_MA0] = ( GFX_OP_MAP_ALPHA_STAGES |
					MA_STAGE_0 |
					MA_UPDATE_ARG1 |
					MA_ARG1_ALPHA_FACTOR |
					MA_UPDATE_ARG2 |
					MA_ARG2_ALPHA_FACTOR |
					MA_UPDATE_OP |
					MA_OP_ARG1 );
      break;
   case GL_BLEND:
      imesa->Setup[I810_CTXREG_MC0] = ( GFX_OP_MAP_COLOR_STAGES |
					MC_STAGE_0 |
					MC_UPDATE_DEST |
					MC_DEST_CURRENT |
					MC_UPDATE_ARG1 |
					MC_ARG1_COLOR_FACTOR | 
					MC_UPDATE_ARG2 |
					MC_ARG2_ITERATED_COLOR |
					MC_UPDATE_OP |
					MC_OP_LIN_BLEND_TEX0_COLOR );

      if (t->image[0].internalFormat == GL_RGB) {
	 imesa->Setup[I810_CTXREG_MA0] = ( GFX_OP_MAP_ALPHA_STAGES |
					   MA_STAGE_0 |
					   MA_UPDATE_ARG1 |
					   MA_ARG1_ALPHA_FACTOR |
					   MA_UPDATE_ARG2 |
					   MA_ARG2_ITERATED_ALPHA |
					   MA_UPDATE_OP |
					   MA_OP_ARG1 );
      } else {
	 imesa->Setup[I810_CTXREG_MA0] = ( GFX_OP_MAP_ALPHA_STAGES |
					   MA_STAGE_0 |
					   MA_UPDATE_ARG1 |
					   MA_ARG1_ALPHA_FACTOR |
					   MA_UPDATE_ARG2 |
					   MA_ARG2_ITERATED_ALPHA |
					   MA_UPDATE_OP |
					   MA_OP_LIN_BLEND_TEX0_ALPHA );
      }
      break;

   default:
      FatalError("unkown tex env mode");
      break;			
   }



   imesa->reg_dirty |= ((1<<I810_CTXREG_MA0) |
			(1<<I810_CTXREG_MC0));

}



static void i810UpdateTex1State( GLcontext *ctx )
{
   i810ContextPtr imesa = I810_CONTEXT(ctx);
   struct gl_texture_object	*tObj;
   i810TextureObjectPtr t;
   int ma_modulate_op;


   tObj = ctx->Texture.Unit[1].Current;

   if ( tObj != ctx->Texture.Unit[1].CurrentD[2] ) 
      tObj = 0;

   if (!(ctx->Texture.Enabled & 0xf0) || !tObj || !tObj->Complete) {

      imesa->Setup[I810_CTXREG_MC1] = ( GFX_OP_MAP_COLOR_STAGES |
					MC_STAGE_1 |
					MC_UPDATE_DEST |
					MC_DEST_CURRENT |
					MC_UPDATE_ARG1 |
					MC_ARG1_ONE | 
					MC_ARG1_DONT_REPLICATE_ALPHA |
					MC_ARG1_DONT_INVERT |
					MC_UPDATE_ARG2 |
					MC_ARG2_ONE |
					MC_ARG2_DONT_REPLICATE_ALPHA |
					MC_ARG2_DONT_INVERT |
					MC_UPDATE_OP |
					MC_OP_DISABLE );

      imesa->Setup[I810_CTXREG_MA1] = ( GFX_OP_MAP_ALPHA_STAGES |
					MA_STAGE_1 |
					MA_UPDATE_ARG1 |
					MA_ARG1_CURRENT_ALPHA |
					MA_ARG1_DONT_INVERT |
					MA_UPDATE_ARG2 |
					MA_ARG2_CURRENT_ALPHA |
					MA_ARG2_DONT_INVERT |
					MA_UPDATE_OP |
					MA_OP_ARG1 );

      imesa->reg_dirty |= ((1<<I810_CTXREG_MA1) |
			   (1<<I810_CTXREG_MC1));
      return;
   }

   t = tObj->DriverData;
  
   if (!t) {
      t = i810CreateTexObj( imesa, tObj );
      if (!t) return;
   }
    
   if (t->dirty_images) 
      i810UploadTexImages(imesa, t);

   if (t->current_unit != 1)
      i810TexSetUnit( t, 1 );

   t->age = ++i810glx.current_texture_age;
   move_to_head(&imesa->TexObjList, t);
   imesa->CurrentTex1Obj = t;

   if (0)
      if (t->UsePalette == I810_USE_PALETTE) {      
	 if (ctx->Texture.SharedPalette) {
	    if (imesa->GlobalPaletteUpdated)
	       i810LoadTexturePalette(imesa->GlobalPalette,0,256);
	    imesa->GlobalPaletteUpdated = 0;
	 }
	 else {
	    i810LoadTexturePalette(t->Palette,0,256);
	    imesa->GlobalPaletteUpdated = 1;
	 }
      }		      


   if (t->reg_dirty) 
      i810DmaExecute(t->Setup,I810_TEX_SETUP_SIZE);


   
   switch (ctx->Texture.Unit[1].EnvMode) {
   case GL_REPLACE:
      imesa->Setup[I810_CTXREG_MC1] = ( GFX_OP_MAP_COLOR_STAGES |
					MC_STAGE_1 |
					MC_UPDATE_DEST |
					MC_DEST_CURRENT |
					MC_UPDATE_ARG1 |
					MC_ARG1_TEX1_COLOR | 
					MC_UPDATE_ARG2 |
					MC_ARG2_ONE |
					MC_UPDATE_OP |
					MC_OP_ARG1 );

      if (t->image[0].internalFormat == GL_RGB) {
	 ma_modulate_op = MA_OP_ARG1;
      } else {
	 ma_modulate_op = MA_OP_ARG2;
      }

      imesa->Setup[I810_CTXREG_MA1] = ( GFX_OP_MAP_ALPHA_STAGES |
					MA_STAGE_1 |
					MA_UPDATE_ARG1 |
					MA_ARG1_CURRENT_ALPHA |
					MA_UPDATE_ARG2 |
					MA_ARG2_TEX1_ALPHA |
					MA_UPDATE_OP |
					ma_modulate_op );
      break;
   case GL_MODULATE:
      imesa->Setup[I810_CTXREG_MC1] = ( GFX_OP_MAP_COLOR_STAGES |
					MC_STAGE_1 |
					MC_UPDATE_DEST |
					MC_DEST_CURRENT |
					MC_UPDATE_ARG1 |
					MC_ARG1_TEX1_COLOR | 
					MC_UPDATE_ARG2 |
					MC_ARG2_CURRENT_COLOR |
					MC_UPDATE_OP |
					MC_OP_MODULATE );

      if (t->image[0].internalFormat == GL_RGB) {
	 ma_modulate_op = MA_OP_ARG1;
      } else {
	 ma_modulate_op = MA_OP_MODULATE;
      }
	
      imesa->Setup[I810_CTXREG_MA1] = ( GFX_OP_MAP_ALPHA_STAGES |
					MA_STAGE_1 |
					MA_UPDATE_ARG1 |
					MA_ARG1_CURRENT_ALPHA |
					MA_UPDATE_ARG2 |
					MA_ARG2_TEX1_ALPHA |
					MA_UPDATE_OP |
					ma_modulate_op );
      break;

   case GL_DECAL:
      if (t->image[0].internalFormat == GL_RGB) {
	 imesa->Setup[I810_CTXREG_MC1] = ( GFX_OP_MAP_COLOR_STAGES |
					   MC_STAGE_1 |
					   MC_UPDATE_DEST |
					   MC_DEST_CURRENT |
					   MC_UPDATE_ARG1 |
					   MC_ARG1_COLOR_FACTOR | 
					   MC_UPDATE_ARG2 |
					   MC_ARG2_TEX1_COLOR |
					   MC_UPDATE_OP |
					   MC_OP_ARG2 );

      } else {
	 imesa->Setup[I810_CTXREG_MC1] = ( GFX_OP_MAP_COLOR_STAGES |
					   MC_STAGE_1 |
					   MC_UPDATE_DEST |
					   MC_DEST_CURRENT |
					   MC_UPDATE_ARG1 |
					   MC_ARG1_COLOR_FACTOR | 
					   MC_UPDATE_ARG2 |
					   MC_ARG2_TEX1_COLOR |
					   MC_UPDATE_OP |
					   MC_OP_LIN_BLEND_TEX1_ALPHA );
      }

      imesa->Setup[I810_CTXREG_MA1] = ( GFX_OP_MAP_ALPHA_STAGES |
					MA_STAGE_1 |
					MA_UPDATE_ARG1 |
					MA_ARG1_ALPHA_FACTOR |
					MA_UPDATE_ARG2 |
					MA_ARG2_ALPHA_FACTOR |
					MA_UPDATE_OP |
					MA_OP_ARG1 );
      break;

   case GL_BLEND:
      imesa->Setup[I810_CTXREG_MC1] = ( GFX_OP_MAP_COLOR_STAGES |
					MC_STAGE_1 |
					MC_UPDATE_DEST |
					MC_DEST_CURRENT |
					MC_UPDATE_ARG1 |
					MC_ARG1_COLOR_FACTOR | 
					MC_UPDATE_ARG2 |
					MC_ARG2_CURRENT_COLOR |
					MC_UPDATE_OP |
					MC_OP_LIN_BLEND_TEX1_COLOR );

      if (t->image[0].internalFormat == GL_RGB) {
	 imesa->Setup[I810_CTXREG_MA1] = ( GFX_OP_MAP_ALPHA_STAGES |
					   MA_STAGE_1 |
					   MA_UPDATE_ARG1 |
					   MA_ARG1_ALPHA_FACTOR |
					   MA_UPDATE_ARG2 |
					   MA_ARG2_CURRENT_ALPHA |
					   MA_UPDATE_OP |
					   MA_OP_ARG1 );
      } else {
	 imesa->Setup[I810_CTXREG_MA1] = ( GFX_OP_MAP_ALPHA_STAGES |
					   MA_STAGE_1 |
					   MA_UPDATE_ARG1 |
					   MA_ARG1_ALPHA_FACTOR |
					   MA_UPDATE_ARG2 |
					   MA_ARG2_ITERATED_ALPHA |
					   MA_UPDATE_OP |
					   MA_OP_LIN_BLEND_TEX1_ALPHA );
      }
      break;

   default:
      FatalError("unkown tex 1 env mode");
      break;			
   }

   imesa->reg_dirty |= ((1<<I810_CTXREG_MA1) |
			(1<<I810_CTXREG_MC1));
}


void i810UpdateTextureState( GLcontext *ctx )
{
   i810UpdateTex0State( ctx );
   i810UpdateTex1State( ctx );
}



/*****************************************
 * Driver functions
 *****************************************/

void i810TexEnv( GLcontext *ctx, GLenum pname, const GLfloat *param )
{
   if (!i810Ctx) 
      return;
   

   if (pname == GL_TEXTURE_ENV_MODE) {

      i810Ctx->new_state |= I810_NEW_TEXTURE;

   } else if (pname == GL_TEXTURE_ENV_COLOR) {

      struct gl_texture_unit *texUnit = 
	 &ctx->Texture.Unit[ctx->Texture.CurrentUnit];
      GLubyte c[4];
      GLuint col;

      FLOAT_RGBA_TO_UBYTE_RGBA(texUnit->EnvColor, c);
      col = i810PackColor(c[0],c[1],c[2],c[3]);
    
      if (i810DB->Setup[I810_CTXREG_CF1] != col) {
	 i810DB->Setup[I810_CTXREG_CF1] = col;      
	 i810Ctx->reg_dirty |= (1<<I810_CTXREG_CF1);
      }
   } 
}

void i810TexImage( GLcontext *ctx, 
		   GLenum target,
		   struct gl_texture_object *tObj, 
		   GLint level,
		   GLint internalFormat,
		   const struct gl_texture_image *image )
{
   i810TextureObjectPtr t;

   CHECK_CONTEXT( return; );
  
   i810Msg(10,"i810TexImage(%d): level %d internalFormat %x\n", 
	   tObj->Name, level, internalFormat);

   if (target != GL_TEXTURE_2D)
      return;

   if (level >= I810_TEX_MAXLEVELS)
      return;

   t = (i810TextureObjectPtr) tObj->DriverData;
   if (t) {
      /* if this is the current object, it will force an update */
      i810DestroyTexObj( i810Ctx, t );
      tObj->DriverData = 0;
      i810Ctx->new_state |= I810_NEW_TEXTURE;
   }
}

void i810TexSubImage( GLcontext *ctx, GLenum target,
		      struct gl_texture_object *tObj, GLint level,
		      GLint xoffset, GLint yoffset,
		      GLsizei width, GLsizei height,
		      GLint internalFormat,
		      const struct gl_texture_image *image ) 
{
   i810TextureObjectPtr t;
   i810Msg(10,"i810TexSubImage():\n");
   i810Msg(10,"  Size: %d,%d of %d,%d; Level %d\n",
	   width, height, image->Width,image->Height,
	   level);

   CHECK_CONTEXT( return; );

   if ( target != GL_TEXTURE_2D ) 
      return;
   
   t = (i810TextureObjectPtr) tObj->DriverData;
   if (t) {
      i810DestroyTexObj( i810Ctx, t );
      tObj->DriverData = 0;
      i810Ctx->new_state |= I810_NEW_TEXTURE;
      i810glx.c_textureSwaps++;
   }
}

void i810TexParameter( GLcontext *ctx, GLenum target,
		      struct gl_texture_object *tObj,
		      GLenum pname, const GLfloat *params )
{
   i810TextureObjectPtr t = (i810TextureObjectPtr) tObj->DriverData;

   CHECK_CONTEXT( return; );

   if (!t || target != GL_TEXTURE_2D)
      return;

   switch (pname) {
   case GL_TEXTURE_MIN_FILTER:
   case GL_TEXTURE_MAG_FILTER:
      i810SetTexFilter(t,tObj->MinFilter,tObj->MagFilter);
      break;

   case GL_TEXTURE_WRAP_S:
   case GL_TEXTURE_WRAP_T:
      i810SetTexWrapping(t,tObj->WrapS,tObj->WrapT);
      break;
  
   case GL_TEXTURE_BORDER_COLOR:
      i810SetTexBorderColor(t,tObj->BorderColor);
      break;

   default:
      return;
   }
   i810Ctx->new_state |= I810_NEW_TEXTURE;
}

void i810BindTexture( GLcontext *ctx, GLenum target,
		     struct gl_texture_object *tObj )
{
   CHECK_CONTEXT( return; );

   if (target != GL_TEXTURE_2D)
      return;

   i810Ctx->new_state |= I810_NEW_TEXTURE;
}

void i810DeleteTexture( GLcontext *ctx, struct gl_texture_object *tObj )
{
   i810TextureObjectPtr t = (i810TextureObjectPtr)tObj->DriverData;
   CHECK_CONTEXT( return; );

   i810DestroyTexObj(i810Ctx,t);
   tObj->DriverData=0;
}

void i810UpdateTexturePalette( GLcontext *ctx, struct gl_texture_object *tObj )
{
   i810TextureObjectPtr t;
   GLushort *dst;
   GLubyte *src;
   int i, size, next;
   GLenum format;

   CHECK_CONTEXT( return; );

   return;


   if (tObj) 
   {
      i810Msg(8,"i810UpdateTexturePalette(): size %d\n", tObj->PaletteSize);
      t = (i810TextureObjectPtr) tObj->DriverData;;
      if (!t) return;
      size = tObj->PaletteSize;
      format = tObj->PaletteFormat;
      dst = t->Palette;
      src = (GLubyte *) tObj->Palette;
   }
   else
   {
      /* shared palette */
      i810Ctx->GlobalPaletteUpdated = 1;
      size = ctx->Texture.PaletteSize;
      format = ctx->Texture.PaletteFormat;
      dst = i810Ctx->GlobalPalette;
      src = (GLubyte *) ctx->Texture.Palette;
    
      i810Msg(8, "i810UpdateTexturePalette(): size %d intFormat %x format %x\n",
	      size, ctx->Texture.PaletteIntFormat, ctx->Texture.PaletteFormat);
   }

   if (size > 256) {
      i810Error("i810UpdateTexturePalette(): palette > 256 entries!\n");
      return;
   }

   switch (format)
   { 
   case GL_RGB: next = 3; break;
   case GL_RGBA:next = 4; break;
   default : 
      i810Error("i810UpdateTexturePalette(): unsupported palette format %x\n");
      next = 4;
   }

   for (i=0; i < size; i++) {
      *dst = I810PACKCOLOR565(src[RCOMP],src[GCOMP],src[BCOMP]);
      dst++;
      src += next;
   }  
}
