
/* -*- mode: C; c-basic-offset:8 -*- */
/*
 * GLX Hardware Device Driver for Intel i810
 * Copyright (C) 1999 Keith Whitwell
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * KEITH WHITWELL, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR 
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE 
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 *
 * based on the original by Jeff Hartmann <slicer@ionet.net>
 * as rewritten by John Carmack <johnc@idsoftware.com>
 */

/*

This file is only entered at startup.  After i810GlxInit completes,
nothing here will be executed again.

*/

#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <sys/mman.h>
#include <stdio.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <signal.h>

#include "context.h"
#include "depth.h"
#include "macros.h"
#include "texstate.h"
#include "triangle.h"
#include "vb.h"
#include "types.h"

#include "xsmesaP.h"
#include "glx_log.h"
#include "mesaglx/context.h"
#include "mesaglx/matrix.h"
#include "mesaglx/types.h"

#define GC XXGC
#include "gcstruct.h"
#include "pixmapstr.h"
#include "servermd.h" /* PixmapBytePad */
#include "scrnintstr.h"
#include "regionstr.h"
#include "windowstr.h"
#undef GC

#include "vga.h"
#include "vgaPCI.h"
#include "i810.h"
#include "xaa/xf86xaa.h"

#include "mm.h"
#include "i810buf.h"
#include "i810dd.h"
#include "i810lib.h"
#include "i810log.h"
#include "i810direct.h"
#include "i810glx.h"

#include <agpgart.h>

/* private vars */
i810BatchBuffer	*dmaBuffers[2];
static I810MemRange I810SysMem2;


static const char *getenvSafe(const char *name ) {
   const char	*r;
	
   r = getenv( name );
   if ( !r ) {
      return "";
   }
   return r;
}


static void AllocateCommandBuffers( void ) 
{
   PMemBlock block;
   GLuint bufferBytes = i810glx.cmdSize;
   GLuint start, i;

   if ( 1 || !bufferBytes ) {
      fprintf(stderr,"temporarily hardwiring GLX_I810_CMDSIZE = 1\n" );
      bufferBytes = 1;
   }

   bufferBytes *= 0x100000;
	
   block = mmAllocMem( i810glx.sysmemHeap, bufferBytes, 8, 0 );

   if ( !block ) {
      fprintf(stderr, 
	      "failed to allocate 0x%x bytes from "
	      "sysmemHeap for command buffers.\n", 
	      bufferBytes );

      FatalError("Couldn't get dma buffer\n");
   }

   start = block->ofs;
		
   /* setup the two buffers that will be ping-ponged 
    */
   for (i = 0 ; i < 2 ; i++) {
      dmaBuffers[i] = calloc(1,sizeof(i810BatchBuffer));
      dmaBuffers[i]->virtual_start = i810glx.sysmemVirtual + start;
      dmaBuffers[i]->mem.Start = start;
      dmaBuffers[i]->mem.Size = bufferBytes / 2;
      dmaBuffers[i]->mem.End = (start += bufferBytes / 2);
   }
	
   i810DmaResetBuffer();
}


/* Another bit of duplication caused by 'static' code in the X server.
 */
static int    gartfd;
static struct gart_info gartinf;
static char *gart_buf;

static int AllocateGARTMemory( int size ) 
{
   int i, pages = size / 4096; 
   int start = vga256InfoRec.videoRam / 4; /* a crock! */
   struct stat sb;

   if (stat("/dev/agpgart", &sb) != 0) {
      ErrorF("Stat failed on /dev/agpgart: %s\n", 
	     sys_errlist[errno]);
      return -1;
   }

   gartfd = open("/dev/agpgart", O_RDWR);
   if (gartfd == -1) {	
      ErrorF("unable to open /dev/agpgart: %s\n", 
	     sys_errlist[errno]);
      FatalError("Aborting");
   }

   if (ioctl(gartfd, GARTIOCINFO, &gartinf) != 0) {
      ErrorF("error doing ioctl(GARTIOCINFO): %s\n", 
	     sys_errlist[errno]);
      FatalError("Aborting");
   }

   /* Treat the gart like video memory - we assume we own all that is
    * there, so ignore EBUSY errors.  Don't try to remove it on
    * failure, either.
    */
   for (i = start; i < start+pages; i++) 
      if (ioctl(gartfd, GARTIOCINSERT, &i) != 0) {
	 if (errno != EBUSY) 
	 {	
	    perror("gart insert");
	    ErrorF("GART: allocation of %d pages failed at page %d\n", 
		   pages, i);
	    FatalError("Aborting");
	 }	
      } 

   ErrorF("GART: allocated %dK system ram\n", pages * 4);
   
   I810SysMem2.Start = start * 4096;
   I810SysMem2.End = (start + pages) * 4096;
   I810SysMem2.Size = pages * 4096;

   gart_buf = mmap( 0, 
		    gartinf.size * 0x100000,
		    PROT_READ | PROT_WRITE,
		    MAP_SHARED,		    
		    gartfd,
		    0 );
   
   if ((unsigned int)gart_buf == ~0) {
      perror("/dev/agpgart");
      FatalError("Couldn't mmap /dev/agpgart - aborting");
   }

   return 0;
}


/*
 * AllocateSystemMemory
 * Looks at environment variables to determine if a block
 * of physical memory has been left for graphics after the
 * memory available to the kernel.
 * System memory can be used for dma command buffers or
 * textures.
 */
static void AllocateSystemMemory( void ) 
{
   GLuint sysmemBytes = i810glx.dmaSize;

   if ( !sysmemBytes ) 
      sysmemBytes = 8;

   sysmemBytes *= 0x100000;

   if (AllocateGARTMemory(sysmemBytes) != 0) 
      FatalError( "AllocateGARTMemory failed.\n" );

   
   i810glx.sysmemVirtual = (unsigned char *)gart_buf;

   /* Suck in any leftover memory from the 2d server.

   i810glx.sysmemHeap = mmInit( I810SysMem.Start,
				I810SysMem.Size );
    */

   /* Add our new memory.
    */
   i810glx.sysmemHeap = 
/*        mmAddRange( i810glx.sysmemHeap,  */
      mmInit(
	 I810SysMem2.Start,
	 I810SysMem2.Size );

      			
   /* Manage dcache mem via. a seperate heap.
    */
   i810glx.cardHeap = mmInit( I810DcacheMem.Start,
			      I810DcacheMem.Size );

   i810Msg( 1, "sysmemSize: 0x%x\n", sysmemBytes );
   i810Msg( 1, "cardSize: 0x%lx\n", I810DcacheMem.Size );
}



/*
 * i810DmaInit
 *
*/
void i810DmaInit(void) {

   /* Server init - queries environment variables.  The client
    * gets these values from the sever and initializes them in
    * i810direct.c 
    */
   if (__glx_is_server) {
      i810glx.dmaDriver = atoi( getenvSafe("GLX_I810_DMA") );
      i810glx.dmaSize = atoi( getenvSafe("GLX_I810_DMASIZE") );
      i810glx.cmdSize = atoi( getenvSafe("GLX_I810_CMDSIZE") );

      if (i810glx.dmaDriver != 3)
	 FatalError("GLX_I810_DMA not set\n");
   }
  
   /* get some system memory and make it write combining if we can */
   AllocateSystemMemory();

   /* read the command environment variable */
   i810Msg(1,"i810DmaInit: GLX_I810_DMA = %i\n", i810glx.dmaDriver );

   /* setup the two command buffers in the apropriate memory space */
   AllocateCommandBuffers();
	
   /* prepare the first buffer for use */
   i810DmaResetBuffer();
}


/*
 * This function should only verify that the current hardware is supported.
 * It should do no setup.
 */
static GLboolean i810_detect_hw( void ) 
{  
   if (I810Chipset == -1)
      return GL_FALSE;

   if (vga256InfoRec.depth != 15 &&
       vga256InfoRec.depth != 16) 
   {
      i810Error("Unsupported depth: %d, only 15 and 16d bpp "
		"are supported right now\n",
		vga256InfoRec.depth);
      return GL_FALSE;
   }

   return GL_TRUE;
}

/*
 * i810InitLogging
 *
 */
void i810InitLogging( void )
{		
   i810glx.logLevel = 100;
}



/*
 * i810InitGLX
 * This is the initial entry point for the i810 hardware driver,
 * called at X server module load time, or libGL direct rendering
 * init time.
 */
GLboolean i810InitGLX( void ) {

   fprintf(stderr, "\n\n\n\ni810InitGLX\n");

   i810InitLogging();

   i810Msg(1,"virtual (x, y) (%d, %d)\n", vga256InfoRec.virtualX,
	   vga256InfoRec.virtualY);
   i810Msg(1,"width: %d\n", vga256InfoRec.displayWidth);
   i810Msg(1,"depth: %d\n", vga256InfoRec.depth);
   i810Msg(1,"memBase: %p\n", vgaLinearBase);
   i810Msg(1,"videoRam: 0x%08x\n", vga256InfoRec.videoRam);

   if (!i810_detect_hw()) {
      ErrorF("Couldn't find i810 hardware\n\n\n");
      return GL_FALSE;
   }

   /* init the dma system */  
   i810DmaInit();

   /* Register as a glx driver */
   GLXProcs.CreateContext = i810GLXCreateContext;
   GLXProcs.DestroyContext = i810GLXDestroyContext;
   GLXProcs.SwapBuffers = i810GLXSwapBuffers;
   GLXProcs.CreateImage = i810GLXCreateImage;
   GLXProcs.DestroyImage = i810GLXDestroyImage;
   GLXProcs.CreateDepthBuffer = i810GLXCreateDepthBuffer;
   GLXProcs.MakeCurrent = i810GLXMakeCurrent;
   GLXProcs.BindBuffer = i810GLXBindBuffer;
   GLXProcs.SwapBuffers = i810GLXSwapBuffers;
   GLXProcs.VendorPrivate = i810GLXVendorPrivate;
   GLXProcs.AllowDirect = i810GLXAllowDirect;

   if (!__glx_is_server) {
      GLXProcs.ValidateFrontBuffer = i810ClientGetGeometry;
   }

   /* these vars can be changed between invocations of direct clients */
   if (getenv("GLX_I810_NULLPRIMS") ) {
      i810Msg( 1, "enabling GLX_I810_NULLPRIMS\n" );
      i810glx.nullprims = 1;
   }
   if (getenv("GLX_I810_SKIPDMA") ) {
      i810Msg( 1, "enabling GLX_I810_SKIPDMA\n" );
      i810glx.skipDma = 1;
   }
   if (getenv("GLX_I810_BOXES") ) {
      i810Msg( 1, "enabling GLX_I810_BOXES\n" );
      i810glx.boxes = 1;
   }
   if (getenv("GLX_I810_NOFALLBACK") ) {
      i810Msg( 1, "enabling GLX_I810_NOFALLBACK\n" );
      i810glx.noFallback = 1;
   }
	
   i810Error("i810InitGLX completed\n");
   return GL_TRUE;
}


