
/*
 * GLX Server Extension
 * Copyright (C) 1996  Steven G. Parker  (sparker@cs.utah.edu)
 * Copyright (C) 1998, 1999  Terence Ripperda (ripperda@sgi.com)
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * STEPHEN PARKER, TERENCE RIPPERDA, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR 
 * ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, 
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE 
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

/*
 * glX Client Commands
 * modified July 14, 1998 by Terence Ripperda (ripperda@engr.sgi.com)
 *
 * to help clean up some of the code and files I am splitting them 
 * up along more distinguished lines. Other c files in this directory
 * are based and ordered on sections from the GLX Extension for OpenGL
 * Protocol Specification Version 1.2. Unfortunately, there appear to
 * be many mistakes in that guide. The other current sources I have
 * for specs are OpenGL Graphics for the X Window System (which appears
 * to be a spec on glX, the client, rather than glX, the protocol), 
 * and Mark Kilgard's OpenGL Programming for the X Window System.
 *
 * Many functions listed in the Protocol Spec appear to be different
 * from the other two sources, and many appear to be missing. It may
 * be that the protocol spec defines glX commands that communicate
 * between the client and server, while the other spec defines 
 * communication between the glX Client and your application.
 *
 * The functions and ordering in this file are based on the OpenGL
 * Graphics with the X Window System spec, but (for the most part)
 * correspond with the Protocol Spec Section 9.1
 */
 


#include "glxlib.h"
#include "pointers.h"
#include "glxcommon.h"
#include "extensions/Xext.h"
#include "extensions/extutil.h"
#include "GL/GLXproto.h"
#include <stdio.h>
#include <assert.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>

#include "direct_protocol.h"

GLContext* GLCurrent;

#define MAJOR_VERSION 1
#define MINOR_VERSION 2

/* 
   I'm overriding this string to include passing of some glx extensions 
   here, to allow the server to know we're connecting, and that we support
   these special extensions.
*/
static char gl_extension_string[] = 
#ifdef BUILD_COLOR_TABLE_EXT
    "GL_color_table " "GL_EXT_paletted_texture " 
    "GL_EXT_shared_texture_palette "
#endif
    "";

static char glx_extension_string[] = 
   GLX_TRY_DIRECT_STRING 
   GLX_NOHW_STRING;


#define MAX_RENDER_BUFFER_SIZE        (64*1024) 
#define MAX_LARGE_RENDER_BUFFER_SIZE  (128*1024) 

static XExtensionInfo _gl_info_data;
static XExtensionInfo* gl_info = &_gl_info_data;
static char* gl_extension_name = "GLX";

#define GLCheckExtension(dpy, i, val) \
   XextCheckExtension(dpy, i, gl_extension_name, val)

#define GLSimpleCheckExtension(dpy, i) \
   XextSimpleCheckExtension(dpy, i, gl_extension_name)




static /* const */ XExtensionHooks gl_extension_hooks = {
    NULL,         /* create_gc */
    NULL,         /* copy_gc */
    NULL,         /* flush_gc */
    NULL,         /* free_gc */
    NULL,         /* create_font */
    NULL,         /* free_font */
    NULL,         /* close_display */
    NULL,         /* wire_to_event */
    NULL,         /* event_to_wire */
    NULL,         /* error */
    NULL,         /* error_string */
};




XEXT_GENERATE_FIND_DISPLAY (__gl_find_display, gl_info, gl_extension_name, 
				   &gl_extension_hooks, (int) NULL, NULL)


void GLXError()
{
    fprintf(stderr, "Need to throw an error...\n");
}

/** internal forward declarations **/

static void initialize_display(Display* dpy);
static int score_attrib(Display*, XVisualInfo*, int, int, int*);
static int match_bool(Display*, XVisualInfo*, int, int); 


/** glX Client Command Suite **/

Bool glXQueryExtension(Display* dpy, int* errorBase, int* eventBase)
{
    XExtDisplayInfo* info = NULL;

    if (dpy == NULL) return False;
    info = __gl_find_display(dpy);

    if(XextHasExtension(info)){
	if (eventBase) *eventBase = info->codes->first_event;
	if (errorBase) *errorBase = info->codes->first_error;
	return True;
    } else {
	return False;
    }
}

Bool glXQueryVersion(Display* dpy, int* maj, int* min)
{
    XExtDisplayInfo* info = NULL;
    xGLXQueryVersionReq* req;
    xGLXQueryVersionReply reply;
    Proto *proto = __glx_get_proto(dpy); /* ind */

    if (dpy == NULL) return False;
    info =  __gl_find_display(dpy); 
    if (!XextHasExtension(info)) {
	return False;
    }

    proto->GLXRenderFlush();

    protoLockDisplay(dpy);
    protoGetReq(GLXQueryVersion, req);
    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXQueryVersion;
    req->length = 3;
    req->major_version = MAJOR_VERSION;
    req->minor_version = MINOR_VERSION;

    proto->XReply(dpy, (xReply*)&reply, 0, xFalse);

    protoUnlockDisplay(dpy);
    protoSyncHandle();

    if (maj) *maj = reply.major_version; 
    if (min) *min = reply.minor_version; 
	
    return True;
}


int glXGetConfig(Display* dpy, XVisualInfo* vis, int attrib, int* value)
{
    XExtDisplayInfo* info = NULL; 
    GLDisplayPrivate* priv;
    int i;
    GLVisualInfo* glvis;
    GLScreenInfo* si;

    if (dpy == NULL) return GLX_NO_EXTENSION;
    info =  __gl_find_display(dpy); 

    if(!info->data)
	initialize_display(dpy);
    priv=(GLDisplayPrivate*)info->data;

    if(!priv->hasExtension)
	return GLX_NO_EXTENSION;
    if(vis->screen < 0 || vis->screen >= priv->numScreens)
	return GLX_BAD_SCREEN;

    si=&priv->screenInfo[vis->screen];
    /* Find the requested visual... */
    for(i=0;i<si->numGLVisuals;i++){
	if(si->visualInfo[i].visualid == vis->visualid)
	    break;
    }
    if(i==si->numGLVisuals){
	/* GL not supported. */
	if(attrib == GLX_USE_GL){
	    if (value) *value=GL_FALSE;
	    return 0;
	} else {
	    return GLX_BAD_VISUAL;
	}
    }

    glvis=&si->visualInfo[i];
    switch(attrib){
    case GLX_USE_GL:
	if (value) *value=GL_TRUE;
	break;
    case GLX_BUFFER_SIZE:
	if (value) *value=glvis->buffer_size;
	break;
    case GLX_LEVEL:
	if (value) *value=glvis->level;
	break;
    case GLX_RGBA:
	if (value) *value=glvis->rgba;
	break;
    case GLX_DOUBLEBUFFER:
	if (value) *value=glvis->doublebuffer;
	break;
    case GLX_STEREO:
	if (value) *value=glvis->stereo;
	break;
    case GLX_AUX_BUFFERS:
	if (value) *value=glvis->aux_buffers;
	break;
    case GLX_RED_SIZE:
	if (value) *value=glvis->red_size;
	break;
    case GLX_GREEN_SIZE:
	if (value) *value=glvis->green_size;
	break;
    case GLX_BLUE_SIZE:
	if (value) *value=glvis->blue_size;
	break;
    case GLX_ALPHA_SIZE:
	if (value) *value=glvis->alpha_size;
	break;
    case GLX_DEPTH_SIZE:
	if (value) *value=glvis->depth_size;
	break;
    case GLX_STENCIL_SIZE:
	if (value) *value=glvis->stencil_size;
	break;
    case GLX_ACCUM_RED_SIZE:
	if (value) *value=glvis->accum_red_size;
	break;
    case GLX_ACCUM_GREEN_SIZE:
	if (value) *value=glvis->accum_green_size;
	break;
    case GLX_ACCUM_BLUE_SIZE:
	if (value) *value=glvis->accum_blue_size;
	break;
    case GLX_ACCUM_ALPHA_SIZE:
	if (value) *value=glvis->accum_alpha_size;
	break;
    default:
	return GLX_BAD_ATTRIBUTE;
    }
    return 0;
}


XVisualInfo* glXChooseVisual(Display* dpy, int screen, int* attribList)
{
    XVisualInfo* vinfo;
    int nvisuals;
    int maxscore;
    int maxvis;
    int vis;
    int score;
    XVisualInfo tmpl;

    int buffer_size=0;
    int level=0;
    int rgba=0;
    int doublebuffer=0;
    int stereo=0;
    int aux_buffers=0;
    int red_size=0;
    int green_size=0;
    int blue_size=0;
    int alpha_size=0;
    int depth_size=0;
    int stencil_size=0;
    int accum_red_size=0;
    int accum_green_size=0;
    int accum_blue_size=0;
    int accum_alpha_size=0;

    if (dpy == NULL) return NULL;
    if (attribList) {
      while(*attribList != None){
	switch(*attribList++){
	case GLX_USE_GL:
	    break;
	case GLX_BUFFER_SIZE:
	    buffer_size=*attribList++;
	    break;
	case GLX_LEVEL:
	    level=*attribList++;
	    break;
	case GLX_RGBA:
	    rgba=GL_TRUE;
	    break;
	case GLX_DOUBLEBUFFER:
	    doublebuffer=GL_TRUE;
	    break;
	case GLX_STEREO:
	    stereo=GL_TRUE;
	    break;
	case GLX_AUX_BUFFERS:
	    aux_buffers=*attribList++;
	    break;
	case GLX_RED_SIZE:
	    red_size=*attribList++;
	    break;
	case GLX_GREEN_SIZE:
	    green_size=*attribList++;
	    break;
	case GLX_BLUE_SIZE:
	    blue_size=*attribList++;
	    break;
	case GLX_ALPHA_SIZE:
	    alpha_size=*attribList++;
	    break;
	case GLX_DEPTH_SIZE:
	    depth_size=*attribList++;
	    break;
	case GLX_STENCIL_SIZE:
	    stencil_size=*attribList++;
	    break;
	case GLX_ACCUM_RED_SIZE:
	    accum_red_size=*attribList++;
	    break;
	case GLX_ACCUM_GREEN_SIZE:
	    accum_green_size=*attribList++;
	    break;
	case GLX_ACCUM_BLUE_SIZE:
	    accum_blue_size=*attribList++;
	    break;
	case GLX_ACCUM_ALPHA_SIZE:
	    accum_alpha_size=*attribList++;
	    break;
	default:
	    return NULL;
	}
      }
    }

    vinfo=XGetVisualInfo(dpy, VisualNoMask, NULL, &nvisuals);
    if(!vinfo)
	return NULL;
    maxscore=0;
    maxvis=-1;
    for(vis=0;vis<nvisuals;vis++){
	score=0;

	if(!match_bool(dpy, &vinfo[vis], GLX_USE_GL, GL_TRUE))
	    continue;
        else score += 100;

	if(!match_bool(dpy, &vinfo[vis], GLX_RGBA, rgba))
	    continue;
        else score += 100;

	if(!match_bool(dpy, &vinfo[vis], GLX_DOUBLEBUFFER, doublebuffer))
	    continue;
        else score += 100;

	if(!match_bool(dpy, &vinfo[vis], GLX_STEREO, stereo))
	    continue;
        else score += 100;

	if(rgba){
	    if(!score_attrib(dpy, &vinfo[vis], GLX_RED_SIZE, red_size, &score))
		continue;
	    if(!score_attrib(dpy, &vinfo[vis], GLX_GREEN_SIZE, green_size, &score))
		continue;
	    if(!score_attrib(dpy, &vinfo[vis], GLX_BLUE_SIZE, blue_size, &score))
		continue;
	    if(!score_attrib(dpy, &vinfo[vis], GLX_ALPHA_SIZE, alpha_size, &score))
		continue;
	    if(!score_attrib(dpy, &vinfo[vis], GLX_ACCUM_RED_SIZE, accum_red_size, &score))
		continue;
	    if(!score_attrib(dpy, &vinfo[vis], GLX_ACCUM_GREEN_SIZE, accum_green_size, &score))
		continue;
	    if(!score_attrib(dpy, &vinfo[vis], GLX_ACCUM_BLUE_SIZE, accum_blue_size, &score))
		continue;
	    if(!score_attrib(dpy, &vinfo[vis], GLX_ACCUM_ALPHA_SIZE, accum_alpha_size, &score))
		continue;
	} else {
	    if(!score_attrib(dpy, &vinfo[vis], GLX_BUFFER_SIZE, buffer_size, &score))
		continue;
	}
	if(!score_attrib(dpy, &vinfo[vis], GLX_LEVEL, level, &score))
	    continue;
	if(!score_attrib(dpy, &vinfo[vis], GLX_AUX_BUFFERS, aux_buffers, &score))
	    continue;
	if(!score_attrib(dpy, &vinfo[vis], GLX_DEPTH_SIZE, depth_size, &score))
	    continue;
	if(!score_attrib(dpy, &vinfo[vis], GLX_STENCIL_SIZE, stencil_size, &score))
	    continue;

	/*
	 * Now that we have the score, see if we should use this one...
	 */
	if(score > maxscore){
	    maxscore=score;
	    maxvis=vis;
	} else if(score == maxscore){
	    /*
	     * Prefer TrueColor over Direct and Pseudo over Static/Gray...
	     */
	    if((vinfo[vis].class == TrueColor && vinfo[maxvis].class != TrueColor)
	       || (vinfo[vis].class == PseudoColor && vinfo[maxvis].class != PseudoColor)){
		maxvis=vis;
	    }
	}
    }
    if(maxscore == 0){
	XFree(vinfo);
	return NULL;
    } else {
	tmpl.visualid=vinfo[maxvis].visualid;
	XFree(vinfo);
	return XGetVisualInfo(dpy, VisualIDMask, &tmpl, &nvisuals);
    }
}

GLXPixmap glXCreateGLXPixmap( Display *dpy, XVisualInfo *visinfo, 
                            Pixmap pixmap)
{
    xGLXCreateGLXPixmapReq *req;
    XExtDisplayInfo* info = NULL;
    GLXPixmap glx_pixmap = XAllocID(dpy);
    Proto *proto = __glx_get_proto(dpy);

    if (dpy == NULL) return (GLXPixmap) NULL;
    if (visinfo == NULL) return (GLXPixmap) NULL;

    info =  __gl_find_display(dpy);  
    protoLockDisplay(dpy);
    protoGetReq(GLXCreateGLXPixmap, req);
    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXCreateGLXPixmap;
    req->screen = visinfo->screen;
    req->visual = visinfo->visualid;
    req->pixmap = pixmap;
    req->glx_pixmap = glx_pixmap;

    protoUnlockDisplay(dpy);
    protoSyncHandle();
    proto->XFlush(dpy);
    return glx_pixmap;
}


void glXDestroyGLXPixmap(Display *dpy, GLXPixmap pixmap) 
{
    xGLXDestroyGLXPixmapReq * req;
    XExtDisplayInfo* info = NULL;
    Proto *proto = __glx_get_proto(dpy);

    if (dpy == NULL) return;
    info = __gl_find_display(dpy); 
    protoLockDisplay(dpy);
    protoGetReq(GLXDestroyGLXPixmap, req);
    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXDestroyGLXPixmap;
    req->glx_pixmap = pixmap;

    protoUnlockDisplay(dpy);
    protoSyncHandle();
    proto->XFlush(dpy);
    return;
}

GLboolean	WantDirectRendering( void ) {
	/* try direct if root unless explicitly asked not to */
	if ( getuid() != 0 ) {
		return GL_FALSE;
	}

    	if ( getenv("GLX_NO_DIRECT" ) ) {
    		return GL_FALSE;
    	}
	return GL_TRUE;
}

/*
  Unfortunately, this function is messy to add support for special features.
  Only the g200 supports attempts at Direct Rendering for now, but I'll keep
  the support generic here for now.
*/
void 
glx_send_client_info(Display *dpy)
{
    xGLXClientInfoReq* req;
    XExtDisplayInfo* info = NULL;
    unsigned int num_bytes = 0;
    char *tmp = NULL;
    GLboolean no_accel = GL_FALSE;
    GLboolean try_direct = GL_FALSE;

    if (dpy == NULL) return;
    info = __gl_find_display(dpy);

    if (getenv("GLX_NOHW")) no_accel = GL_TRUE;

    try_direct = WantDirectRendering();
    
    LockDisplay(dpy);
    GetReq(GLXClientInfo, req);
    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXClientInfo;
    req->major_version = GLX_MAJOR_VERSION;
    req->minor_version = GLX_MINOR_VERSION;
    num_bytes = strlen(gl_extension_string);
    if (no_accel) num_bytes += strlen(GLX_NOHW_STRING) + 1;
    if (try_direct) num_bytes += strlen(GLX_TRY_DIRECT_STRING) + 1;

    req->stringlength = num_bytes;
    tmp = (char *) malloc(num_bytes + 1);
    if (tmp == NULL) { 
       UnlockDisplay(dpy);
       SyncHandle();
       return;
    }
    strcpy(tmp, gl_extension_string);
    if (no_accel) strcat(tmp, GLX_NOHW_STRING);
    if (try_direct) strcat(tmp, GLX_TRY_DIRECT_STRING);
    req->length += GLX_pad(num_bytes) >> 2; 
    _XSend(dpy, tmp, num_bytes);
    free(tmp);
    UnlockDisplay(dpy);
    SyncHandle();
}

/* Need to resend the create context request after going direct...
 */
void __glx_recreate_context(GLContext *ctx, Display* dpy, 
			    XVisualInfo* vis, GLXContext shareList)
{
    xGLXCreateContextReq* req;
    XExtDisplayInfo* info = NULL;
    GLX_CONTEXT shareListID = shareList?((GLContext*)shareList)->contextID:0;
    Proto *proto = __glx_get_proto(dpy);
    info = __gl_find_display(dpy); 

    /** send request to (local) glX Server **/
    protoLockDisplay(dpy);
    protoGetReq(GLXCreateContext, req);
    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXCreateContext;
    req->length = 6;
    req->context = ctx->contextID;
    req->visual = vis->visualid;
    req->screen = vis->screen;
    req->sharelist = shareListID;
    req->direct = GL_TRUE;
    protoUnlockDisplay(dpy);
    protoSyncHandle();
    proto->GLXRenderFlush();
}


static int x_error;

static int tmp_handler( Display *dpy, XErrorEvent *ev )
{
   x_error = 1;
   return 0;
}


static int __glx_direct_count = 0;

void __glx_GoDirect( GLContext *ctx, Display *dpy, 
		     XVisualInfo* vis, GLXContext shareList)
{
   int (*handler)(Display *, XErrorEvent *);
   char *reply;

   ctx->ContextType = _GLX_INDIRECT;

   if (dpy->display_name[0] != ':' || !WantDirectRendering() )
      return;

   /* Verify that the context was correctly created and the direct
    * request was accepted.
    */
   fprintf(stderr, "sending isdirect request\n");
   if (!glXIsDirect(dpy, ctx)) 
      return;
   fprintf(stderr, "isdirect succeeded\n");

   if (dpy == directDisplay) { 
      ctx->ContextType = _GLX_DIRECT;
      __glx_direct_count++;
   }  else {
      
      /* Send the first non-standard GLX request.  It should now be safe
       * to do so.
       */
      x_error = 0;
      handler = XSetErrorHandler(tmp_handler);
      reply = __glx_request_direct( dpy );
      XSetErrorHandler(handler);
   
      if (!reply) 
	 return;

      if (!*reply || x_error) {
	 free( reply );
	 return;
      }

      fprintf(stderr, "going direct\n");

      if (__glx_init_direct( dpy, reply )) {
	 ctx->ContextType = _GLX_DIRECT;
	 __glx_direct_count++;
      }

      free( reply );
   }

   /* Need to recreate the context which was created on the server on
    * our local copy of the server.
    */
   __glx_recreate_context(ctx, dpy, vis, shareList);
}


GLXContext glXCreateContext(Display* dpy, XVisualInfo* vis,
			    GLXContext shareList, Bool direct)
{
    GLContext* ctx;
    int i;
    unsigned int max_size = 0;
    xGLXCreateContextReq* req;
    XExtDisplayInfo* info = NULL;
    GLX_CONTEXT shareListID = shareList?((GLContext*)shareList)->contextID:0;
    Proto *proto = &xProto;	/* always indirect */

    if (dpy == NULL) return NULL;
    if (vis == NULL) return NULL;
    info = __gl_find_display(dpy); 

    glx_send_client_info(dpy);

    ctx=(GLContext*)Xmalloc(sizeof(GLContext));
    if(!ctx)
	return 0;

    ctx->contextID=XAllocID(dpy);

    /** send request to glX Server **/
    protoLockDisplay(dpy);
    protoGetReq(GLXCreateContext, req);
    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXCreateContext;
    req->length = 6;
    req->context = ctx->contextID;
    req->visual = vis->visualid;
    req->screen = vis->screen;
    req->sharelist = shareListID;
    req->direct = direct;
    protoUnlockDisplay(dpy);
    protoSyncHandle();
    proto->GLXRenderFlush();

    /* Try to go direct, if requested & possible.
     */
    if (direct) 
       __glx_GoDirect( ctx, dpy, vis, shareList );

    ctx->dpy=dpy;
    ctx->bufferSize = 0;
    ctx->bufferLargeSize = 0;

    /** see if the user has defined a specific buffer size **/
    if (getenv("GLX_BUFFER")) {
       ctx->bufferSize = (atoi(getenv("GLX_BUFFER")) * 1024);
    }
    if (getenv("GLX_LARGE_BUFFER")) {
       ctx->bufferLargeSize = (atoi(getenv("GLX_LARGE_BUFFER")) * 1024);
    }

    /** If the user didn't define, use defaults **/
    if (!ctx->bufferSize) ctx->bufferSize = MAX_RENDER_BUFFER_SIZE;
    if (!ctx->bufferLargeSize) 
        ctx->bufferLargeSize = MAX_LARGE_RENDER_BUFFER_SIZE;

    /** make sure our buffers aren't too big for the X Server **/
    max_size = XMaxRequestSize(dpy)*4;
    if (ctx->bufferSize > max_size)      ctx->bufferSize      = max_size;
    if (ctx->bufferLargeSize > max_size) ctx->bufferLargeSize = max_size;

    /** allocate room for the larger of the two buffers **/
    if (ctx->bufferSize > ctx->bufferLargeSize) {
       max_size = ctx->bufferSize;
    } else {
       max_size = ctx->bufferLargeSize;
    }
    ctx->bufferStart=Xmalloc(max_size);
    if(!ctx->bufferStart){
	Xfree(ctx);
	return 0;
    }

    ctx->CompileFlag = GL_FALSE;
    ctx->drawable = 0;

    ctx->bufferLargeRenderEnd = ctx->bufferStart + ctx->bufferLargeSize;
    ctx->bufferRenderEnd      = ctx->bufferStart + ctx->bufferSize;
    ctx->bufferP=ctx->bufferStart;
    ctx->isRenderLargeReq=GL_FALSE;
    ctx->largeReqMaxDataLen=ctx->bufferLargeSize-8;

    ctx->pack_swap_bytes=GL_FALSE;
    ctx->pack_lsb_first=GL_FALSE;
    ctx->pack_row_length=0;
    ctx->pack_skip_rows=0;
    ctx->pack_skip_pixels=0;
    ctx->pack_alignment=4;
    ctx->unpack_swap_bytes = DEFAULT_UNPACK_SWAP_BYTES;
    ctx->unpack_lsb_first  = DEFAULT_UNPACK_LSB_FIRST;
    ctx->unpack_row_length = DEFAULT_UNPACK_ROW_LENGTH;
    ctx->unpack_skip_rows  = DEFAULT_UNPACK_SKIP_ROWS;
    ctx->unpack_skip_pixels= DEFAULT_UNPACK_SKIP_PIXELS;
    ctx->unpack_alignment  = DEFAULT_UNPACK_ALIGNMENT;
    ctx->need_unpacking    = GL_FALSE;
    ctx->feedback_buffer_size=0;
    ctx->selection_buffer_size=0;
    ctx->rendermode=GL_RENDER;

    ctx->Array.VertexPtr = NULL;
    ctx->Array.VertexFuncPtr = NULL;
    ctx->Array.VertexEnabled = GL_FALSE;
    ctx->Array.NormalEnabled = GL_FALSE;
    ctx->Array.NormalPtr = NULL;
    ctx->Array.NormalFuncPtr = NULL;
    ctx->Array.ColorEnabled = GL_FALSE;
    ctx->Array.ColorPtr = NULL;
    ctx->Array.ColorFuncPtr = NULL;
    ctx->Array.IndexEnabled = GL_FALSE;
    ctx->Array.IndexPtr = NULL;
    ctx->Array.IndexFuncPtr = NULL;
    for (i = 0; i < MAX_TEX_SETS; i++) {
       ctx->Array.TexCoordEnabled[i] = GL_FALSE;
       ctx->Array.TexCoordPtr[i] = NULL;
    }
    ctx->Array.TexCoordFuncPtr = NULL;
    ctx->Array.EdgeFlagEnabled = GL_FALSE;
    ctx->Array.EdgeFlagPtr = NULL;
    ctx->Array.EdgeFlagFuncPtr = NULL;
    ctx->ClientAttribStackDepth = 0;
    ctx->SGIhost = GL_FALSE;
    ctx->enabled_varrays = 0;

    glx_init_api_function_pointers(ctx);

    printf("@@Created GLX Context..\n");
    return (GLXContext)ctx;
}


Bool glXIsDirect(Display* dpy, GLXContext ctxp)
{
    xGLXIsDirectReply reply;
    xGLXIsDirectReq* req;
    XExtDisplayInfo* info = NULL;
    GLContext* ctx=(GLContext*)ctxp;
    Proto *proto = __glx_get_proto(dpy);

    if (dpy == NULL) return False;
    info = __gl_find_display(dpy);
  
    protoLockDisplay(dpy);
    protoGetReq(GLXIsDirect, req);
    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXIsDirect;
    req->length = 2;
    req->context_tag = ctx->contextID;

    proto->XReply(dpy, (xReply*)&reply, 0, xFalse);
    protoUnlockDisplay(dpy);
    protoSyncHandle();
    return reply.is_direct;
}


void glXDestroyContext(Display* dpy, GLXContext ctxp)
{
    XExtDisplayInfo* info = NULL;
    xGLXDestroyContextReq* req;
    GLContext* ctx=(GLContext*)ctxp;
    Proto *proto = __glx_get_proto(dpy);

    if (dpy == NULL) return;
    info = __gl_find_display(dpy); 
    GLSimpleCheckExtension(dpy, info);

    proto->GLXRenderFlush();
    protoLockDisplay(dpy);
    protoGetReq(GLXDestroyContext, req);
    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXDestroyContext;
    req->length = 2;
    req->context = ctx->contextID;

    protoUnlockDisplay(dpy);
    protoSyncHandle();

    if (ctx)
    {
       if (ctx->ContextType == _GLX_DIRECT) 
       {
	  fprintf(stderr, "deleted local context\n");

	  fprintf(stderr, "Nr direct contexts remaining: %d\n", 
		  __glx_direct_count - 1);

	  if (--__glx_direct_count == 0) {
	     fprintf(stderr, "Unloading glx.so\n");
	     __glx_unload_direct();
	  }

	  /* Now delete the context on the server.
	   */
	  proto = __glx_get_proto(dpy);
	  proto->GLXRenderFlush();
	  protoLockDisplay(dpy);
	  protoGetReq(GLXDestroyContext, req);
	  req->reqType = info->codes->major_opcode;
	  req->glx_opcode = X_GLXDestroyContext;
	  req->length = 2;
	  req->context = ctx->contextID;

	  protoUnlockDisplay(dpy);
	  protoSyncHandle();
	  proto->XFlush(dpy);

	  fprintf(stderr, "deleted context on server\n");
       }

       if (ctx->bufferStart) free(ctx->bufferStart);
       else 
          fprintf(stderr, "Why wasn't there a buffer? PLEASE report this!\n"); 
       free(ctx);
    } else 
       fprintf(stderr, "Why wasn't there a context?? PLEASE report this!\n");

    if (GLCurrent == ctx) 
       GLCurrent = 0;
}


void glXCopyContext(Display* dpy, GLXContext srcp, GLXContext dstp, GLuint mask)
{
    XExtDisplayInfo* info = NULL;
    xGLXCopyContextReq* req;
    GLContext* src=(GLContext*)srcp;
    GLContext* dst=(GLContext*)dstp;
    Proto *proto = __glx_get_proto(dpy);

    if (dpy == NULL) return;
    info = __gl_find_display(dpy); 
    GLSimpleCheckExtension(dpy, info);

    proto->GLXRenderFlush();
    protoLockDisplay(dpy);
    protoGetReq(GLXCopyContext, req);
    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXCopyContext;
    req->length = 5;
    req->source_context = src->contextID;
    req->dest_context = dst->contextID;
    req->mask = mask;
    if(GLCurrent)
	req->source_tag = 0;
    else
	req->source_tag = GLCurrent->contextTag;

    protoUnlockDisplay(dpy);
    protoSyncHandle();
}


Bool glXMakeCurrent(Display* dpy, GLXDrawable drawable, GLXContext ctxp)
{
    xGLXMakeCurrentReply reply;
    xGLXMakeCurrentReq* req;
    XExtDisplayInfo* info = NULL;
    GLContext* ctx = (GLContext*)ctxp;
    Proto *proto = __glx_get_proto(dpy);

    if (dpy == NULL) return False;
    info = __gl_find_display(dpy);

    proto->GLXRenderFlush();
    protoLockDisplay(dpy);
    protoGetReq(GLXMakeCurrent, req);
    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXMakeCurrent;
    req->drawable = drawable;
    if (GLCurrent) {
	req->old_context = GLCurrent->contextTag;
    } else {
	req->old_context = 0;
    }

    if (ctx) req->context = ctx->contextID; 
    else req->context = 0;

    if(!proto->XReply(dpy, (xReply*)&reply, 0, xFalse)){
        printf("make current failed\n");
	protoUnlockDisplay(dpy);
	protoSyncHandle();
	return False;
    }

    if (ctx) {
       ctx->contextTag = reply.context_tag; 
       ctx->drawable = drawable;
    }
    GLCurrent = ctx;
    protoUnlockDisplay(dpy);
    protoSyncHandle();

    /* Check if the server is an SGI */
    if (ctx) {
       const GLubyte *server_string = NULL;
       server_string = glGetString(GL_VENDOR);
       if (server_string) {
          if (!strcmp("SGI", server_string)) {
             printf("Talking to SGI server, implementing hack fixes..\n");
             ctx->SGIhost = GL_TRUE;
          }
       }
    }
    return True;
}


GLXContext glXGetCurrentContext()
{
    return (GLXContext)GLCurrent;
}


GLXDrawable glXGetCurrentDrawable()
{
    return GLCurrent?GLCurrent->drawable:0;
}

Display* glXGetCurrentDisplay()
{
    return GLCurrent?GLCurrent->dpy:0;
}


void glXWaitGL()
{
    XExtDisplayInfo* info = NULL;
    xGLXWaitGLReq* req;
    Display* dpy = NULL;
    Proto *proto;

    if (GLCurrent == NULL) return;

    dpy = GLCurrent->dpy;
    proto = __glx_get_proto(dpy);
    info = __gl_find_display(dpy);
    GLSimpleCheckExtension(dpy, info);

    proto->GLXRenderFlush();
    protoLockDisplay(dpy);
    protoGetReq(GLXWaitGL, req);

    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXWaitGL;
    req->length = 2;
    req->context_tag = GLCurrent->contextTag;

    protoUnlockDisplay(dpy);
    protoSyncHandle();
}

void glXWaitX()
{
    XExtDisplayInfo* info = NULL;
    xGLXWaitXReq* req;
    Display* dpy = NULL;
    Proto *proto;

    if (GLCurrent == NULL) return;

    dpy = GLCurrent->dpy;
    proto = __glx_get_proto(dpy);
    info = __gl_find_display(dpy);
    GLSimpleCheckExtension(dpy, info);

    proto->GLXRenderFlush();
    protoLockDisplay(dpy);
    protoGetReq(GLXWaitX, req);
    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXWaitX;
    req->length = 2;
    req->context_tag = GLCurrent->contextTag;

    protoUnlockDisplay(dpy);
    protoSyncHandle();

    /** should this be here? **/
    proto->XFlush(dpy);
}

/* Some MESA workarounds for binary apps to run */

/* With a true GLX, this is no longer needed, but for binary apps
   to resolve all symbols */

Bool glXReleaseBuffersMESA( Display *dpy, Window w )
{
   return GL_TRUE;
}

/* XXX hack, workaround for binary-only programs */
/* This is really, really painful, but should work properly */

void glXCopySubBufferMESA(Display* dpy, GLXDrawable drawable, int x, int y,
                          int width, int height)
{
   glXSwapBuffers(dpy, drawable);
}

void glXSwapBuffers(Display* dpy, GLXDrawable drawable)
{
    XExtDisplayInfo* info = NULL;
    xGLXSwapBuffersReq* req;
    Proto *proto = __glx_get_proto(dpy);

    if (dpy == NULL) return;
    info = __gl_find_display(dpy); 
    GLSimpleCheckExtension(dpy, info);

    proto->GLXRenderFlush();
    protoLockDisplay(dpy);
    protoGetReq(GLXSwapBuffers, req);
    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXSwapBuffers;
    req->length = 3;
    req->context_tag = GLCurrent?GLCurrent->contextTag:0;
    req->drawable = drawable;

    protoUnlockDisplay(dpy);
    protoSyncHandle();
    proto->XFlush(dpy);
}


void glXUseXFont(Font font, int first, int count, int listbase)
{
    XExtDisplayInfo* info = NULL;
    xGLXUseXFontReq* req;
    Display* dpy = NULL;
    Proto *proto;

    if (GLCurrent == NULL) return;

    dpy = GLCurrent->dpy;
    proto = __glx_get_proto(dpy);
    info = __gl_find_display(dpy);
    GLSimpleCheckExtension(GLCurrent->dpy, info);

    proto->GLXRenderFlush();
    protoLockDisplay(dpy);
    protoGetReq(GLXUseXFont, req);
    req->reqType = info->codes->major_opcode;

    req->glx_opcode = X_GLXUseXFont;
    req->length = 6;
    req->context_tag = GLCurrent->contextTag;
    req->font=font;
    req->first=first;
    req->count=count;
    req->listbase=listbase;
    protoUnlockDisplay(GLCurrent->dpy);
    protoSyncHandle();
}


const char *
glXGetClientString(Display *dpy, int name) {
   static char *extensions = glx_extension_string;
   static char *vendor = "Brian Paul";
   static char *version = "1.2 Mesa 3.0";

   if (dpy == NULL) return NULL;

   if (dpy == directDisplay) 
      return glGetString(name);	/* right? */

   switch (name) {
      case GLX_VENDOR:
         return vendor;
      case GLX_VERSION:
         return version;
      case GLX_EXTENSIONS:
         return extensions;
      default:
         return NULL;
   }
   return NULL;
}

const char *
glXQueryServerString(Display *dpy, int screen, int name) {
    XExtDisplayInfo* info = NULL;
    xGLXQueryServerStringReq* req;
    xGLXQueryServerStringReply reply;
    unsigned int num_bytes = 0;
    unsigned char *data = NULL;
    Proto *proto = __glx_get_proto(dpy);
    
    if (dpy == NULL) return NULL;
    info = __gl_find_display(dpy); 
    GLCheckExtension(dpy, info, NULL);
    proto->GLXRenderFlush();
    protoLockDisplay(dpy);
    protoGetReq(GLXQueryServerString, req);
    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXQueryServerString;
    req->length = 3;
    req->screen = screen;
    req->name = name;
    if (!proto->XReply(dpy, (xReply*)&reply, 0, xFalse)) {
        protoUnlockDisplay(dpy);
        protoSyncHandle();
        return NULL;
    }
    num_bytes = (reply.n + 3) & 0xFFFC;
    data = (unsigned char *) calloc(num_bytes, sizeof(char));
    proto->XRead(dpy, (char *) data, num_bytes);
    protoUnlockDisplay(dpy);
    protoSyncHandle();
    return (const char *) data;
}

const char * glXQueryExtensionsString( Display *dpy, int screen)
{
    XExtDisplayInfo* info = NULL;
    xGLXQueryServerStringReq* req;
    xGLXQueryServerStringReply reply;
    unsigned int num_bytes = 0;
    unsigned char *data = NULL;
    Proto *proto = __glx_get_proto(dpy);

    if (dpy == NULL) return NULL;
    info = __gl_find_display(dpy); 

    GLCheckExtension(dpy, info, NULL);

    /** send client info to server **/
    proto->GLXRenderFlush();
    protoLockDisplay(dpy);
    protoGetReq(GLXQueryServerString, req);
    req->reqType = info->codes->major_opcode;
    req->glx_opcode = X_GLXQueryServerString;
    req->length = 3;
    req->screen = screen;
    req->name = GLX_EXTENSIONS;

    /** add any client-side only extensions to list **/
    if (!proto->XReply(dpy, (xReply*)&reply, 0, xFalse)) {
        protoUnlockDisplay(dpy);
        protoSyncHandle();
        return NULL;
    }
    num_bytes = reply.n;
    if (num_bytes) {
       data = (unsigned char *) calloc(num_bytes, sizeof(char));               
       num_bytes = (num_bytes + 3) & 0xFFFC;
       proto->XRead(dpy, (char *) data, num_bytes);
    } else {
       data = NULL;
    }
    
    /** return merged list **/
    protoUnlockDisplay(dpy);
    protoSyncHandle();
    return (const char *) data;
}


/** internal functions **/

static void initialize_display(Display* dpy)
{
    xGLXGetVisualConfigsReq* req;
    xGLXGetVisualConfigsReply reply;
    XExtDisplayInfo* info = NULL;
    GLDisplayPrivate* priv;
    int screen;
    int* buf;
    int vis;
    GLVisualInfo* vi;
    Proto *proto = __glx_get_proto(dpy);

    if (dpy == NULL) return;
    info = __gl_find_display(dpy);
 
    info->data=(XPointer)Xmalloc(sizeof(GLDisplayPrivate));
    priv=(GLDisplayPrivate*)info->data;

    if(!XextHasExtension(info)){
	priv->hasExtension=GL_FALSE;
	return;
    }
    priv->hasExtension=GL_TRUE;
    priv->numScreens=ScreenCount(dpy);
    priv->screenInfo=(GLScreenInfo*)Xmalloc(sizeof(GLScreenInfo));

    /*
     * Get the visual configs...
     */
    protoLockDisplay(dpy);
    for(screen=0;screen<priv->numScreens;screen++){
	GLScreenInfo* si=&priv->screenInfo[screen];
	protoGetReq(GLXGetVisualConfigs, req);
	req->reqType = info->codes->major_opcode;
	req->glx_opcode = X_GLXGetVisualConfigs;
	req->screen=screen;

	/* Get the reply... */
	if(!proto->XReply(dpy, (xReply*)&reply, 0, False)){
	    /* Hmm...*/
	    si->numGLVisuals=0;
	    continue;
	}
	if(reply.num_props < 18){
	    si->numGLVisuals=0;
	    continue;
	}
	si->numGLVisuals=reply.num_visuals;
	buf=(int*)Xmalloc(reply.num_props*sizeof(int));
	if(!buf){
	    si->numGLVisuals=0;
	    continue;
	}
	si->visualInfo=(GLVisualInfo*)Xmalloc(si->numGLVisuals*sizeof(GLVisualInfo));
	vi=si->visualInfo;
	if(!vi){
	    si->numGLVisuals=0;
	    continue;
	}
	for(vis=0;vis<reply.num_visuals;vis++){
	    GLVisualInfo* v=&vi[vis];
	    proto->XRead(dpy, (char*)buf, 4L*reply.num_props);
	    v->visualid=buf[0];
	    v->class=buf[1];
	    v->rgba=buf[2];
	    v->red_size=buf[3];
	    v->green_size=buf[4];
	    v->blue_size=buf[5];
	    v->alpha_size=buf[6];
	    v->accum_red_size=buf[7];
	    v->accum_green_size=buf[8];
	    v->accum_blue_size=buf[9];
	    v->accum_alpha_size=buf[10];
	    v->doublebuffer=buf[11];
	    v->stereo=buf[12];
	    v->buffer_size=buf[13];
	    v->depth_size=buf[14];
	    v->stencil_size=buf[15];
	    v->aux_buffers=buf[16];
	    v->level=buf[17];
	}
	Xfree(buf);
    }
    protoUnlockDisplay(dpy);
    protoSyncHandle();
}

static int match_bool(Display* dpy, XVisualInfo* vis, int attrib,
		      int req_value)
{
    int visual_value;
    if(glXGetConfig(dpy, vis, attrib, &visual_value) != 0)
	return GL_FALSE;
    if((visual_value && req_value) || (!visual_value && !req_value)){
	return GL_TRUE;
    } else {
	return GL_FALSE;
    }
}

static int score_attrib(Display* dpy, XVisualInfo* vis, int attrib,
			int req_value, int* score)
{
    int visual_value;
    if(glXGetConfig(dpy, vis, attrib, &visual_value) != 0)
	return GL_FALSE;
    if(visual_value < req_value)
	/* Can't use this visual */
	return GL_FALSE;
    if(req_value == 0 && visual_value > 0){
	/* Slightly penalize if they didn't request it... */
	(*score)--;
    } else if(req_value > 0){
	/* Up the score... */
	(*score)+=100;
    }
    return GL_TRUE;
}




/* The following is all specific to the indirect path.
 */
void GLXLargeRenderFlush();

void GLXRenderFlush()
{
    XExtDisplayInfo* info;
    Display* dpy;
    int nbytes;
    xGLXRenderReq* req;
    Proto *proto;

    if(!GLCurrent) return; 
    dpy = GLCurrent->dpy;
    proto = __glx_get_proto(dpy);
    info = __gl_find_display(GLCurrent->dpy);
    nbytes = GLCurrent->bufferP - GLCurrent->bufferStart;
    if(nbytes == 0) return;

#if 1
    /** sanity check for now **/
    if (proto->IsDirect) {
       printf("REPORT ME: Direct protocal in GLXRenderFlush!\n");
       return;
    }

    if (GLCurrent->isRenderLargeReq) {
       printf("REPORT ME: WOAH!! Large render call to RenderFlush!\n");
       GLXLargeRenderFlush();
       return;
    }
#endif

    protoLockDisplay(dpy);
       protoGetReq( GLXRender, req ); 
       req->reqType = info->codes->major_opcode;
       req->glx_opcode = X_GLXRender;
       req->length += GLX_pad(nbytes)>>2;
       req->context_tag = GLCurrent->contextTag;
       proto->XSend(dpy, GLCurrent->bufferStart, nbytes);
    protoUnlockDisplay(dpy);
    protoSyncHandle();
    GLCurrent->bufferP=GLCurrent->bufferStart;
}


void GLXLargeRenderFlush()
{
    XExtDisplayInfo* info;
    Display* dpy;
    int nbytes;
    xGLXRenderLargeReq* req;
    Proto *proto;
   
    if(!GLCurrent)
	return;
    dpy=GLCurrent->dpy;
    proto = __glx_get_proto(dpy);
    info = __gl_find_display(GLCurrent->dpy);
    nbytes = GLCurrent->largeReqDataLen;

    if (nbytes == 0) return;

    protoLockDisplay(dpy);
       protoGetReq( GLXRenderLarge, req );
       req->reqType        = info->codes->major_opcode;
       req->glx_opcode     = X_GLXRenderLarge;
       req->length        += GLX_pad(nbytes)>>2;
       req->context_tag    = GLCurrent->contextTag;
       req->request_number = GLCurrent->largeReqNum;
       req->request_total  = GLCurrent->largeReqTotal;
       req->data_bytes     = GLCurrent->largeReqDataLen;
       proto->XSend(dpy, GLCurrent->bufferStart, nbytes);
    protoUnlockDisplay(dpy);
    protoSyncHandle();
    GLCurrent->largeReqNum++;
}



char *
GET_render_large_buffer(
   const  int   parmSize,           /* size of fixed small parms (unpadded) */ 
   const  int   dataSize,           /* size of variable large parm (unpadded) */ 
   const  int   opCode,             /* request command opcode */
   const  int   incr               /* min allocation incr for large data */ 
)
{
    /*
     * For render commands that may require handling as
     *   multi-part render larger requests 
     */
    char *p = NULL;
    int   dataPadded = GLX_pad(dataSize); /* padded */
    int   cmdTotal = parmSize + dataPadded;

    if ( cmdTotal <= GLCurrent->bufferSize ) {
       /*
        * Handle as normal render request
        */
       p = GLCurrent->bufferP;                                                  
       GLCurrent->bufferP += cmdTotal;                                        
                                                                                
       if (GLCurrent->bufferP > GLCurrent->bufferRenderEnd) {                   
          GLCurrent->bufferP = p;                                               
          __GLX_FLUSH;                                                            
          GLCurrent->bufferP = GLCurrent->bufferStart + cmdTotal;             
          p = GLCurrent->bufferStart;                                           
       }

       GLCurrent->isRenderLargeReq = GL_FALSE;

       __GLX_PUT_short( p, cmdTotal );
       __GLX_PUT_short( p, opCode );
    } else {
       /*
        * Handle as multi-part render larger request
        */
       int maxDataSize = (GLCurrent->largeReqMaxDataLen / incr) * incr;
       short reqTotal  = ((dataPadded + maxDataSize - 1) / maxDataSize) + 1;

       /* Flush it out... */                                                    
       __GLX_FLUSH;

       cmdTotal += 4; /* larger header for RenderLarge */
       GLCurrent->isRenderLargeReq = GL_TRUE;
       GLCurrent->largeReqNum = 1;  
       GLCurrent->largeReqTotal = reqTotal;  
       GLCurrent->largeReqDataLen = (parmSize+4);
       p = GLCurrent->bufferStart;

       __GLX_PUT_int( p, cmdTotal );
       __GLX_PUT_int( p, opCode );
    }
   return p;
}


/** Yes, this is sloppy and slow. Just trying to be right for now **/
void PUT_unpacked_bitmap( char *dest, const unsigned char *source, 
                 const int width, const int height) {

   int ngroups;
   int h, i, j, k;
   int curr_dest_bit = 0;
   int curr_dest_byte = 0;
   int alignment = GLCurrent->unpack_alignment;
   int skip_rows = GLCurrent->unpack_skip_rows;
   int skip_pixels = GLCurrent->unpack_skip_pixels;
   char transfer;
   char *row_start_dest = dest;
   int dest_row_width_in_bytes = 0;
   __GLX_DECLARE_LARGE_VARIABLES;

   if ( width <= 0 || height <= 0 ) return;

   if (GLCurrent->unpack_row_length) {
      ngroups = GLCurrent->unpack_row_length;
   } else {
      ngroups = width;
   }

   k = alignment * CEILING( ngroups, 8 * alignment );

   /* force dest width to be based on width, rather than possibly */
   /* unpack_row_length, since we will be unpacking here.         */
   dest_row_width_in_bytes = alignment * CEILING( width, 8 * alignment );

   __GLX_START_LARGE_RENDER(1);
   for (j = 0; j < height; j++) {
      dest = row_start_dest;
      *dest = 0;
      for (i = 0; i <width; i++) {
         h = 7 - ((i + skip_pixels) % 8);
         transfer = *(source + (((j + skip_rows) * k) + ((i + skip_pixels)/8)));
         transfer = 0x01 & (transfer >> h);
         /* printf("%d", transfer); */
         curr_dest_bit = 7 - (i % 8);
         *dest |= (transfer << curr_dest_bit);
         curr_dest_byte++;
         if (!(curr_dest_bit % 8)) {
           dest++;
           *dest = 0;
           __GLX_CHECK_LARGE_RENDER;
         }
      }
      /* printf("\n"); */
      row_start_dest += dest_row_width_in_bytes;
   }
   __GLX_END_LARGE_RENDER;
}

void PUT_unpacked_buffer( char *dest, const unsigned char *source, 
                 const int width, const int height, GLenum format, GLenum type) {
   int ngroups;
   int nbytes = GLX_data_size(type);
   int nelements = GLX_num_elements(format);
   int i, j, k;
   int offset = 0;
   int alignment = GLCurrent->unpack_alignment;
   int skip_rows = GLCurrent->unpack_skip_rows;
   int skip_pixels = GLCurrent->unpack_skip_pixels;
   int group_size_in_bytes;
   __GLX_DECLARE_LARGE_VARIABLES;

   if ( width < 0 || height < 0 ) return;

   if (GLCurrent->unpack_row_length) ngroups = GLCurrent->unpack_row_length;
   else ngroups = width;

   group_size_in_bytes = nbytes * nelements;
   k = group_size_in_bytes * ngroups;
   if (nbytes < alignment) {
      k = alignment * CEILING(k, alignment);
   }

   __GLX_START_LARGE_RENDER(group_size_in_bytes);
   for (j = 0; j < height; j++) {
      for (i = 0; i <width; i++) {
         offset = ((j + skip_rows) * k) + ((i + skip_pixels) * group_size_in_bytes);
         memcpy( dest, (source + offset), group_size_in_bytes);              
         dest += group_size_in_bytes;                            
         __GLX_CHECK_LARGE_RENDER;
      }
   }
   __GLX_END_LARGE_RENDER;
}

/** XXX needs swap_bytes **/
void GET_packed_buffer( char *dest, const unsigned char *source, 
                 const int width, const int height, GLenum format, GLenum type)
{
   int i, j, k;
   int nbytes = GLX_data_size(type);
   int nelements = GLX_num_elements(format);
   int group_size = nelements * nbytes;
   int offset = 0;

   if ( width < 0 || height < 0 ) return;

   k = group_size * width;
   if (nbytes < 4) k = 4 * CEILING( k, 4);

   for (j = 0; j < height; j++) {
      for (i = 0; i <width; i++) {
         offset = (j * k) + (i * group_size);
         memcpy(dest, (source + offset), group_size);
         dest += group_size;
      }
   }

}


void PUT_buffer( char*  dest, const unsigned char* source, const int num_bytes) 
{
   if ( num_bytes <= 0) return;
   if( ! GLCurrent->isRenderLargeReq ) {
      memcpy(dest, source, num_bytes);
   } else {
      /** large render split across multiple requests **/
      int maxData  = GLCurrent->largeReqMaxDataLen;
      int bytes_left = num_bytes;    /* running count of src still left */
      int bytes_to_send = 0;         /* number of bytes to send this request */
      char *src = (char *) source;

      GLXLargeRenderFlush();
      while (bytes_left > 0) {
         bytes_to_send = (bytes_left > maxData) ? maxData : bytes_left;
         memcpy(GLCurrent->bufferStart, src, bytes_to_send);
         src += bytes_to_send;
         bytes_left -= bytes_to_send;
         GLCurrent->largeReqDataLen = bytes_to_send;
         GLXLargeRenderFlush();
      }
      GLCurrent->isRenderLargeReq = GL_FALSE;
   }
}


void PUT_large_map2f(
         char*     dst, 
   const GLfloat*  src, 
   const GLint     uorder,
   const GLint     vorder,
   const GLint     ustride,
   const GLint     vstride,
   const GLint     nvalues
)
{
#ifdef DEBUG_RENDER_LARGE
    fprintf( stderr, 
             "PUT_large_map2f: uorder %d, vorder %d,\n\
                 ustride %d - vstride %d, nvalues %d,\n",
             uorder, vorder,
             ustride, vstride,
             nvalues
          );
#endif

    if( ! GLCurrent->isRenderLargeReq ) {
       int i,j,k;
       for(i=0;i<uorder;i++){
           const GLfloat* row=src;
           for(j=0;j<vorder;j++){
               for(k=0;k<nvalues;k++){
                   __GLX_PUT_float(dst, row[k]);
               }
               row+=vstride;
           }
           src+=ustride;
       }
    }
    else {
       /*
        * send as multiple requests
        */
       int leftOver = 0;       /* packet space left after last full row */ 
       /* int numRow; */       /* number of full rows in packet         */
       int currRow = 0;        /* current row being loaded              */
       int remainingDst = 0;   /* remaining dest cntrl pnts in curr row */
       int remainingSrc = 0;   /* remaining src cntrl pnts in curr row  */
       char* dstStart = GLCurrent->bufferStart;
       /* int reqTotal = GLCurrent->largeReqTotal; */
       int incr     = vstride * sizeof( GLfloat ); 
       int maxPoints = GLCurrent->largeReqMaxDataLen / incr;  
       /* int maxData  = maxPoints * incr; */

#ifdef DEBUG_RENDER_LARGE
       fprintf( stderr, 
                "PUT_largeMap2f: uorder %d, vorder %d, ustride %d, vstride %d\n\
                 maxdata 0x%x, maxpoints 0x%x, reqTotal %d\n",
                 uorder, vorder, ustride, vstride,
                 maxData, maxPoints, reqTotal 
              );
#endif


       while( currRow < uorder ) {
           int numRow, space, j, k;
           /*
            * The first time thru this will flush the previously buffered
            * first request with the fixed parms.
            */
           GLXLargeRenderFlush();
           /* 
            * Copy the remaining data from the incomplete last
            * row of the last req.
            */
           dst = dstStart;
           if( remainingDst ) {
              const GLfloat* row=src;
              for(j=0;j<remainingDst;j++){
                  for(k=0;k<nvalues;k++){
                      __GLX_PUT_float(dst, row[k]);
                  }
                  row+=vstride;
              }
              src += remainingSrc;
              dst += remainingDst;
              currRow++;            /* we've finished another row */
           }
           /*
            * Compute how may full rows will fit in the remaining
            * space of the current request and how much space will
            * be left over to start on another, partial, row.
            */
           space    = maxPoints - remainingDst;
           numRow   = space / vstride;
           leftOver = space % vstride; 
           /*
            * Copy the full rows.
            */
           while( numRow-- && currRow++ < uorder ) {
              const GLfloat* row=src;
              for(j=0;j<vorder;j++){
                  for(k=0;k<nvalues;k++){
                      __GLX_PUT_float(dst, row[k]);
                  }
                  row+=vstride;
              }
              src+=ustride;
           }
           /*
            * Use remaining space in the request to start another row.
            */
           if( leftOver && currRow < uorder ) {
              int srcIncr = min( leftOver, vorder); 
              const GLfloat* row=src;
              for(j=0;j<leftOver;j++){
                  for(k=0;k<nvalues;k++){
                      __GLX_PUT_float(dst, row[k]);
                  }
                  row+=vstride;
              }
              remainingDst = vorder - leftOver;
              remainingSrc = vorder - srcIncr;
              src += srcIncr; 
              dst += leftOver;
              /* we haven't completed the row, so don't incr currRow yet */
           }
           else {
              remainingDst = 0;
              remainingSrc = 0;
           }
           GLCurrent->bufferP = dst;
           GLCurrent->largeReqDataLen = dst - dstStart;
       }
       /*
        * Flush The last request in the series. 
        */
       GLXLargeRenderFlush();
       GLCurrent->isRenderLargeReq = GL_FALSE;
    }
}

void PUT_large_map2d(
         char*     dst, 
   const GLdouble* src, 
   const GLint     uorder,
   const GLint     vorder,
   const GLint     ustride,
   const GLint     vstride,
   const GLint     nvalues
)
{
#ifdef DEBUG_RENDER_LARGE
    fprintf( stderr, 
             "PUT_large_map2d: uorder %d, vorder %d,\n\
                 ustride %d - vstride %d, nvalues %d,\n",
             uorder, vorder,
             ustride, vstride,
             nvalues
          );
#endif

    if( ! GLCurrent->isRenderLargeReq ) {
       int i,j,k;
       for(i=0;i<uorder;i++){
           const GLdouble* row=src;
           for(j=0;j<vorder;j++){
               for(k=0;k<nvalues;k++){
                   __GLX_PUT_double(dst, row[k]);
               }
               row+=vstride;
           }
           src+=ustride;
       }
    }
    else {
       /*
        * send as multiple requests
        */
       int leftOver = 0;       /* packet space left after last full row */ 
       /* int numRow; */       /* number of full rows in packet         */
       int currRow = 0;        /* current row being loaded              */
       int remainingDst = 0;   /* remaining dest cntrl pnts in curr row */
       int remainingSrc = 0;   /* remaining src cntrl pnts in curr row  */
       char* dstStart = GLCurrent->bufferStart;
       /* int reqTotal = GLCurrent->largeReqTotal; */
       int incr     = vstride * sizeof( GLdouble ); 
       int maxPoints = GLCurrent->largeReqMaxDataLen / incr;  
       /* int maxData  = maxPoints * incr; */

#ifdef DEBUG_RENDER_LARGE
       fprintf( stderr, 
                "PUT_largeMap2d: uorder %d, vorder %d, ustride %d, vstride %d\n\
                 maxdata 0x%x, maxpoints 0x%x, reqTotal %d\n",
                 uorder, vorder, ustride, vstride,
                 maxData, maxPoints, reqTotal 
              );
#endif


       while( currRow < uorder ) {
           int numRow, space, j, k;
           /*
            * The first time thru this will flush the previously buffered
            * first request with the fixed parms.
            */
           GLXLargeRenderFlush();
           /* 
            * Copy the remaining data from the incomplete last
            * row of the last req.
            */
           dst = dstStart;
           if( remainingDst ) {
              const GLdouble* row=src;
              for(j=0;j<remainingDst;j++){
                  for(k=0;k<nvalues;k++){
                      __GLX_PUT_double(dst, row[k]);
                  }
                  row+=vstride;
              }
              src += remainingSrc;
              dst += remainingDst;
              currRow++;            /* we've finished another row */
           }
           /*
            * Compute how may full rows will fit in the remaining
            * space of the current request and how much space will
            * be left over to start on another, partial, row.
            */
           space    = maxPoints - remainingDst;
           numRow   = space / vstride;
           leftOver = space % vstride; 
           /*
            * Copy the full rows.
            */
           while( numRow-- && currRow++ < uorder ) {
              const GLdouble* row=src;
              for(j=0;j<vorder;j++){
                  for(k=0;k<nvalues;k++){
                      __GLX_PUT_double(dst, row[k]);
                  }
                  row+=vstride;
              }
              src+=ustride;
           }
           /*
            * Use remaining space in the request to start another row.
            */
           if( leftOver && currRow < uorder ) {
              int srcIncr = min( leftOver, vorder); 
              const GLdouble* row=src;
              for(j=0;j<leftOver;j++){
                  for(k=0;k<nvalues;k++){
                      __GLX_PUT_double(dst, row[k]);
                  }
                  row+=vstride;
              }

              remainingDst = vorder - leftOver;
              remainingSrc = vorder - srcIncr;
              src += srcIncr; 
              dst += leftOver;
              /* we haven't completed the row, so don't incr currRow yet */
           }
           else {
              remainingDst = 0;
              remainingSrc = 0;
           }
           GLCurrent->bufferP = dst;
           GLCurrent->largeReqDataLen = dst - dstStart;
       }
       /*
        * Flush The last request in the series. 
        */
       GLXLargeRenderFlush();
       GLCurrent->isRenderLargeReq = GL_FALSE;
    }
}


