/* -*- mode: C; c-basic-offset:8 -*- */
/*
 * GLX Hardware Device Driver for Matrox Millenium G200
 * Copyright (C) 1999 Wittawat Yamwong
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR 
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE 
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 *
 *    original by Wittawat Yamwong <Wittawat.Yamwong@stud.uni-hannover.de>
 *	9/20/99 rewrite by John Carmack <johnc@idsoftware.com>
 */

/* $Id: mgatex.c,v 1.15 1999/12/12 20:37:13 keithw Exp $ */

#include "xsmesaP.h"
#include <stdlib.h>
#include <stdio.h>

#include <GL/gl.h>

#include "g200_mac.h"
#include "mm.h"
#include "mgalib.h"
#include "mgadma.h"
#include "mgatex.h"
#include "mgalog.h"

#include "glx_config.h"

/*

NOTES:

Should we use 555 internal format instead of 565?  OpenGL doesn't
specify a 565 internal format, but it does give a bit more precision
at the expense of losing true greys.

We don't update the mesa texture object internal formats on images yet.

If we ever give up on PIO completely, texture palettes could be cached in
dma memory so they could be loaded with an out-of-line secondary dma.

Does the G400 have a texture palette for each texture unit, or a shared one?

Single byte GL_LUMINANCE textures could be supported on G200 by having a driver
level greyscale texture palette if we ever care.

4 bit paletted textures could be supported if we ever care.

If the G400 or future cards have significantly faster texturing from local memory,
it would be possible to use dma memory as a secondary cache to allow texture
swapping to be done with out-of-line secondary dma. (wouldn't work with PIO)

An extension could be implemented for the mga textrans functionality.

An extension could be implemented for the mga filter threshold functionality.

Are all the texture environments and texture types handled correctly?

G400 can support 2048 size textures if we code for it.

G400 anisotropic filtering should be supported with an extension.

G400 can support full mip levels instead of just 5 if we code for it

*/


/*
 * mgaDestroyTexObj
 * Free all memory associated with a texture and NULL any pointers
 * to it.
 */
static void mgaDestroyTexObj( mgaContextPtr ctx, mgaTextureObjectPtr t ) {
	int	i;
	mgaTextureObjectPtr	p,prev;

	mgaMsg( 10,"mgaDestroyTexObj( %p )\n", t->tObj );
	
 	if ( !t ) {
  		return;
  	}
  	
 	if ( t->magic != MGA_TEXTURE_OBJECT_MAGIC ) {
 		mgaError( "mgaDestroyTexObj: "
			  "t->magic != MGA_TEXTURE_OBJECT_MAGIC\n" );
		return;
	}

	/* free the texture memory */
	mmFreeMem( t->memBlock );
 
 	/* free mesa's link */   
	t->tObj->DriverData = NULL;

	/* see if it was the driver's current object */
	for ( i = 0 ; i < 2 ; i++ ) {
		if ( mgaglx.currentTextureObject[i] == t ) {
			mgaMsg( 10, "mgaDestroyTexObj: destroyed current\n" );
			mgaglx.currentTextureObject[i] = NULL;
		}  
	}
	if ( mgaglx.currentTexturePalette == t->Palette ) {
		mgaglx.currentTexturePalette = NULL;
	}
	
	/* remove from the driver texobj list */
	p = mgaglx.TexObjList;
	prev = NULL;
	while( p ) {
		if (p == t) {
			if ( prev ) {
				prev->next = t->next;
			} else {
				mgaglx.TexObjList = t->next;
			}
    			break;
    		}
		prev = p;
		p = p->next;
	}

	/* clear magic to catch any bad future references */
	t->magic = 0;
	
	/* free the structure */
	free( t );
}


/*
 * mgaSetTexWrappings
 */
static void mgaSetTexWrapping( mgaTextureObjectPtr t,
			       GLenum sWrap, GLenum tWrap ) {
	if (sWrap == GL_REPEAT) {
		t->texCtl &= ~TMC_clampu_enable;
	} else {
		t->texCtl |= TMC_clampu_enable;
	}
	if (tWrap == GL_REPEAT) {
		t->texCtl &= ~TMC_clampv_enable;
	} else {
		t->texCtl |= TMC_clampv_enable;
	}
}

/*
 * mgaSetTexFilter
 */
static void mgaSetTexFilter(mgaTextureObjectPtr t, GLenum minf, GLenum magf) {
	switch (minf) {
	case GL_NEAREST:
		MGA_SET_FIELD(t->texFilter,
		  TF_minfilter_MASK, TF_minfilter_nrst);
		break;
	case GL_LINEAR:
		MGA_SET_FIELD(t->texFilter,
		  TF_minfilter_MASK, TF_minfilter_bilin);
		break;
	case GL_NEAREST_MIPMAP_NEAREST:
		MGA_SET_FIELD(t->texFilter,
		  TF_minfilter_MASK, TF_minfilter_mm1s);
		break;
	case GL_LINEAR_MIPMAP_NEAREST:
		MGA_SET_FIELD(t->texFilter,
		  TF_minfilter_MASK, TF_minfilter_mm4s);
		break;
	case GL_NEAREST_MIPMAP_LINEAR:
		MGA_SET_FIELD(t->texFilter,
		  TF_minfilter_MASK, TF_minfilter_mm2s);
		break;
	case GL_LINEAR_MIPMAP_LINEAR:
		MGA_SET_FIELD(t->texFilter,
		  TF_minfilter_MASK, TF_minfilter_mm8s);
		break;
	default:
		mgaError("mgaSetTexFilter(): not supported min. filter %d\n",
			 (int)minf);
		break;
	}

	switch (magf) {
	case GL_NEAREST:
		MGA_SET_FIELD(t->texFilter,
		  TF_magfilter_MASK,TF_magfilter_nrst);
		break;
	case GL_LINEAR:
		MGA_SET_FIELD(t->texFilter,
		  TF_magfilter_MASK,TF_magfilter_bilin);
		break;
	default:
		mgaError("mgaSetTexFilter(): not supported mag. filter %d\n",
			 (int)magf);
		break;
	}
  
	/* See OpenGL 1.2 specification */
	if (magf == GL_LINEAR && (minf == GL_NEAREST_MIPMAP_NEAREST || 
                            minf == GL_NEAREST_MIPMAP_LINEAR)) {
		/* c = 0.5 */
		MGA_SET_FIELD(t->texFilter,TF_fthres_MASK,
		  0x20 << TF_fthres_SHIFT);
	} else {
		/* c = 0 */
		MGA_SET_FIELD(t->texFilter,TF_fthres_MASK,
		  0x10 << TF_fthres_SHIFT);
	}
}

/*
 * mgaSetTexBorderColor
 */
static void mgaSetTexBorderColor(mgaTextureObjectPtr t, GLubyte color[4]) {
	t->texBorderCol = 
		MGAPACKCOLOR8888(color[0],color[1],color[2],color[3]);
}

/*
 * mgaDestroyOldestTexObj
 * Throw out a texture to try to make room for a new texture
 */
static int mgaDestroyOldestTexObj( void ) {
	mgaTextureObjectPtr t, oldest;
	mgaUI32 old;
 
	mgaMsg(10,"  Swapping out texture.\n");
	
 	/* find the best texture to toss */
	old = 0x7fffffff;
	oldest = NULL;
	for ( t = mgaglx.TexObjList; t ; t = t->next ) {
		/* never swap out textures used by other multitexture units */
		if ( t == mgaglx.currentTextureObject[0]
		|| t == mgaglx.currentTextureObject[1] ) {
			continue;
		}

		if ( t->age < old ) {
			old = t->age;
			oldest = t;
		}
	}

	/* if the oldest texture was in use on the previous frame, then
	we are in a texture thrashing state.  Note that we can't just
	test for "in THIS frame", because textures from the same working
	set may be used in different order, and it could register as not
	thrashing.  The solution is to pick the MOST recently used texture
	that isn't currently needed for multitexture.  This will allow the
	other textures to stay resident for the next frame, rather than
	swapping everything out in order. */
	
	if ( old >= mgaglx.swapBuffersCount - 1 ) {
		/* newly created texture objects are always added to the
		front of the list, so just find the first one that isn't
		used for multitexture */
	        mgaMsg( 10, "mgaDestroyOldestTexObj: thrashing\n" );
		for ( t = mgaglx.TexObjList; t ; t = t->next ) {
			/* never swap out textures used by other
                           multitexture units */
			if ( t == mgaglx.currentTextureObject[0] || 
			     t == mgaglx.currentTextureObject[1] ) {
				continue;
			}
			
			break;
		}
		oldest = t;

	} else {
		mgaMsg( 10, "mgaDestroyOldestTexObj\n" );
	}
	
	if ( !oldest ) {
		/* This shouldn't happen unless the 2D resolution is
		high enough that a single texture can't be allocated
		in the remaining memory */
		mgaError("  No Texture to swap out -> Out of Memory!\n"); 
		mmDumpMemInfo( textureHeap ); 
		return -1;
	}

	/* just destroy the texture, because it can always
	be recreated directly from the mesa texture */
	
	mgaDestroyTexObj( mgaCtx, oldest );

	return 0;
}

/*
 * mgaLoadTexturePalette
 * FIXME: this is completely untested, and probably doesn't work...
 */
static void mgaLoadTexturePalette( mgaUI16 *pal ) {
	DMALOCALS;
	int	n;

	/* save the current palette for change checking */
	mgaglx.currentTexturePalette = pal;
  
	MGADMAGETPTR( 16 + 128 );
  
	DMAOUTREG(MGAREG_AR0,255);
	DMAOUTREG(MGAREG_AR3,0);
	MGADMA_PITCH(1024);
	MGADMA_YDSTLEN(0, 256);
	DMAOUTREG(MGAREG_YDSTORG,0);
	DMAOUTREG(MGAREG_FXBNDRY,0);
	DMAOUTREG(MGAREG_MACCESS,MA_pwidth_16 | MA_tlutload_enable);
	MGADMA_DWGCTL_EXEC(DC_opcod_iload | DC_atype_rstr | DC_linear_linear |
		  DC_shftzero_enable | DC_sgnzero_enable | 
		     (0xc << DC_bop_SHIFT) | DC_bltmod_bfcol);
	DMAADVANCE();
  
	/* FIXME: convert to using secondary DMA */
	/* Hardware palette format is TW16 (RGB565) */
	for ( n = 0; n < 128 ; n++ ) {
		DMAOUTREG(MGAREG_DMAPAD, ((mgaUI32 *)pal)[n]);
	}
	DMAADVANCE();

	// FIXME: this shouldn't need to regerence MGAydstorg
	MGADMAGETPTR( 10 );
	mgaMsg(7, "mgaLoadTexturePalette: mgaFrontBuffer = %d\n",
	       mgaFrontBuffer);
	DMAOUTREG(MGAREG_YDSTORG,MGAydstorg); /* for X server */
	if ( mgaFrontBuffer ) {
	  	DMAOUTREG(MGAREG_MACCESS,mgaFrontBuffer->Setup[MGA_SETUP_MACCESS]);
	  	DMAOUTREG(MGAREG_PITCH,mgaFrontBuffer->Setup[MGA_SETUP_PITCH]);
	} else {
		DMAOUTREG(MGAREG_PITCH,mgaDB->Setup[MGA_SETUP_PITCH]);
	    	DMAOUTREG(MGAREG_MACCESS,mgaDB->Setup[MGA_SETUP_MACCESS]);
	}
	DMAADVANCE();
	mgaWaitDrawingEngine();
}

/*
 * mgaConvertTexture
 * Converts a mesa format texture to the apropriate hardware format
 * Note that sometimes width may be larger than the texture, like 64x1
 * for an 8x8 texture.  This happens when we have to crutch the pitch
 * limits of the mga by uploading a block of texels as a single line.
 */
void mgaConvertTexture( mgaUI32 *destPtr, int texelBytes, 
			struct gl_texture_image *image,
			int x, int y, int width, int height ) {
	int		i, j;
	mgaUI8		*src;
	
	/* FIXME: g400 luminance_alpha internal format */
	if ( texelBytes == 1 && 
	     ( image->Format == GL_COLOR_INDEX || 
	       image->Format == GL_INTENSITY || 
	       image->Format == GL_LUMINANCE || 
	       image->Format == GL_ALPHA ) ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (mgaUI8 *)image->Data + ( ( y + i ) * image->Width + x );
			for ( j = width >> 2  ; j ; j-- ) {
				int	pix;
			
				pix = src[0] | ( src[1] << 8 ) | ( src[2] << 16 ) | ( src[3] << 24 );
				*destPtr++ = pix;
				src += 4;
			}
		}
	} else if ( texelBytes == 2 && image->Format == GL_RGB ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (mgaUI8 *)image->Data + ( ( y + i ) * image->Width + x ) * 3;
			for ( j = width >> 1  ; j ; j-- ) {
				int	pix;
			
				pix = MGAPACKCOLOR565(src[0],src[1],src[2]) |  
					( MGAPACKCOLOR565(src[3],src[4],src[5]) << 16 );
				*destPtr++ = pix;
				src += 6;
			}
		}
	} else if ( texelBytes == 2 && image->Format == GL_RGBA ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (mgaUI8 *)image->Data + ( ( y + i ) * image->Width + x ) * 4;
			for ( j = width >> 1  ; j ; j-- ) {
				int	pix;
			
				pix = MGAPACKCOLOR4444(src[0],src[1],src[2],src[3]) |  
					( MGAPACKCOLOR4444(src[4],src[5],src[6],src[7]) << 16 );
				*destPtr++ = pix;
				src += 8;
			}
		}
	} else if ( texelBytes == 2 && image->Format == GL_LUMINANCE ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (mgaUI8 *)image->Data + ( ( y + i ) * image->Width + x );
			for ( j = width >> 1  ; j ; j-- ) {
				int	pix;
				/* FIXME: should probably use 555 texture to get true grey */
				pix = MGAPACKCOLOR565(src[0],src[0],src[0]) |  
					( MGAPACKCOLOR565(src[1],src[1],src[1]) << 16 );
				*destPtr++ = pix;
				src += 2;
			}
		}
	} else if ( texelBytes == 2 && image->Format == GL_INTENSITY ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (mgaUI8 *)image->Data + ( ( y + i ) * image->Width + x );
			for ( j = width >> 1  ; j ; j-- ) {
				int	pix;
			
				pix = MGAPACKCOLOR4444(src[0],src[0],src[0],src[0]) |  
					( MGAPACKCOLOR4444(src[1],src[1],src[1],src[1]) << 16 );
				*destPtr++ = pix;
				src += 2;
			}
		}
	} else if ( texelBytes == 2 && image->Format == GL_ALPHA ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (mgaUI8 *)image->Data + ( ( y + i ) * image->Width + x );
			for ( j = width >> 1  ; j ; j-- ) {
				int	pix;
			
				pix = MGAPACKCOLOR4444(255,255,255,src[0]) |  
					( MGAPACKCOLOR4444(255,255,255,src[1]) << 16 );
				*destPtr++ = pix;
				src += 2;
			}
		}
	} else if ( texelBytes == 2 && image->Format == GL_LUMINANCE_ALPHA ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (mgaUI8 *)image->Data + ( ( y + i ) * image->Width + x ) * 2;
			for ( j = width >> 1  ; j ; j-- ) {
				int	pix;
			
				pix = MGAPACKCOLOR4444(src[0],src[0],src[0],src[1]) |  
					( MGAPACKCOLOR4444(src[2],src[2],src[2],src[3]) << 16 );
				*destPtr++ = pix;
				src += 4;
			}
		}
	} else if ( texelBytes == 4 && image->Format == GL_RGB ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (mgaUI8 *)image->Data + ( ( y + i ) * image->Width + x ) * 3;
			for ( j = width ; j ; j-- ) {
				int	pix;
			
				pix = (255<<24) | MGAPACKCOLOR888(src[0],src[1],src[2]);
				*destPtr++ = pix;
				src += 3;
			}
		}
	} else if ( texelBytes == 4 && image->Format == GL_RGBA ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (mgaUI8 *)image->Data + ( ( y + i ) * image->Width + x ) * 4;
			for ( j = width  ; j ; j-- ) {
				int	pix;
			
				pix = MGAPACKCOLOR8888(src[0],src[1],src[2],src[3]);
				*destPtr++ = pix;
				src += 4;
			}
		}
	} else if ( texelBytes == 4 && image->Format == GL_LUMINANCE ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (mgaUI8 *)image->Data + ( ( y + i ) * image->Width + x );
			for ( j = width ; j ; j-- ) {
				int	pix;
			
				pix = (255<<24) | MGAPACKCOLOR888(src[0],src[0],src[0]);
				*destPtr++ = pix;
				src += 1;
			}
		}
	} else if ( texelBytes == 4 && image->Format == GL_INTENSITY ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (mgaUI8 *)image->Data + ( ( y + i ) * image->Width + x );
			for ( j = width ; j ; j-- ) {
				int	pix;
			
				pix = MGAPACKCOLOR8888(src[0],src[0],src[0],src[0]);
				*destPtr++ = pix;
				src += 1;
			}
		}
	} else if ( texelBytes == 4 && image->Format == GL_ALPHA ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (mgaUI8 *)image->Data + ( ( y + i ) * image->Width + x );
			for ( j = width ; j ; j-- ) {
				int	pix;
			
				pix = MGAPACKCOLOR8888(255,255,255,src[0]);
				*destPtr++ = pix;
				src += 1;
			}
		}
	} else if ( texelBytes == 4 && image->Format == GL_LUMINANCE_ALPHA ) {
		for ( i = 0 ; i < height ; i++ ) {
 	       		src = (mgaUI8 *)image->Data + ( ( y + i ) * image->Width + x ) * 2;
			for ( j = width ; j ; j-- ) {
				int	pix;
			
				pix = MGAPACKCOLOR8888(src[0],src[0],
						       src[0],src[1]);
				*destPtr++ = pix;
				src += 2;
			}
		}
	} else {
		mgaError( "Unsupported texelBytes %i, image->Format %i\n", 
			  texelBytes, image->Format );
	}
}


/*
 * mgaUploadSubImage
 * Perform an iload based update of a resident buffer.  This is used for
 * both initial loading of the entire image, and texSubImage updates.
 */
static void mgaUploadSubImage( mgaTextureObjectPtr t,int level,	     
		     int x, int y, int width, int height ) {
	int		x2;
	int		dwords;
        mgaUI32		*dest;
        int		cmd;
        int		dstorg;
	struct gl_texture_image *image;
	int		texelBytes, texelsPerDword, texelMaccess;
 	DMALOCALS;

	/* the G200 only handles five mip levels, so ignore the
           smallest ones */

	/* FIXME: G400 mipmap addressing */
	if ( level < 0 || level >= MGA_TEX_MAXLEVELS ) {
		mgaMsg( 1, "mgaUploadSubImage: bad level: %i\n", level );
 	 	return;
	}

	image = t->tObj->Image[level];
	if ( !image ) {
		mgaError( "mgaUploadSubImage: NULL image\n" );
		return;
	}
	
	/* find the proper destination offset for this level */
   	dstorg = textureHeapPhysical + t->memBlock->ofs + t->offsets[level];

    	/* turn on PCI/AGP if needed */
    	if ( textureHeapPhysical ) {
    		dstorg |= 1 | use_agp;
    	}

	texelBytes = t->texelBytes;	
	switch( texelBytes ) {
	case 1:
		texelsPerDword = 4;
		texelMaccess = 0;
		break;
	case 2:
		texelsPerDword = 2;
		texelMaccess = 1;
		break;
	case 4:
		texelsPerDword = 1;
		texelMaccess = 2;
		break;
	}
	
		
		
	/* we can't do a subimage update if pitch is < 32 texels due
	to hardware XY addressing limits, so we will need to linearly
	upload all modified rows */
	if ( image->Width < 32 ) {
		x = 0;
		width = image->Width * height;
		height = 1;
		/* we are going to assume that 1x1 textures aren't
		going to cause a bus error if we read up to four
		texels from that location */
		if ( width < texelsPerDword ) {
			width = texelsPerDword;
		}
	} else {
		/* pad the size out to dwords.  The image is a pointer
		   to the entire image, so we can safely reference
		   outside the x,y,width,height bounds if we need to */
		x2 = x + width;
		x2 = ( x2 + ( texelsPerDword - 1 ) ) & ~(texelsPerDword-1);
		
		x = ( x + ( texelsPerDword - 1 ) ) & ~(texelsPerDword-1);
		width = x2 - x;
	}
	
	/* flush any warp verts before we allocate secondary memory */
	mgaWarpFinishSerie();
    
	/* we may not be able to upload the entire texture in one
	   batch due to register limits or dma buffer limits.
	   Recursively split it up. */
	while ( 1 ) {
		dwords = height * width / texelsPerDword;
		if ( dwords <= dma_buffer->maxSecondaryDwords ) {
			break;
		}
		mgaMsg(10, "mgaUploadSubImage: recursively subdividing\n" );

		mgaUploadSubImage( t, level, x, y, width, height >> 1 );
		y += ( height >> 1 );
		height -= ( height >> 1 );
	}
	
	mgaMsg(10, "mgaUploadSubImage: %i,%i of %i,%i at %i,%i\n", 
	       width, height, image->Width, image->Height, x, y );

	/* bump the performance counter */
	mgaglx.c_textureSwaps += ( dwords << 2 );
	
	/* allocate a secondary dma buffer for the texels and make sure that
	 * the MGADMAGETPTR() won't cause an overflow, which would leave the
	 * secondary allocation in the wrong buffer.
	 * this may cause a flush if it doesn't fit in the current buffer 
	 */
	dest = mgaAllocPrimaryAndSecondaryBuffer( 20, dwords );
	
	/* start the primary commands */
	MGADMAGETPTR_NO_OVERFLOW( 20 );

	if ( image->Width < 32 ) {
		/* linear destination, linear iload */
		cmd = DC_opcod_iload | 		/* image load */
	 	DC_atype_rpl |			/* raster replace mode */
		DC_linear_linear | 		/* linear source */
		DC_bltmod_bfcol |		/* source data is
                                                   pre-formatted color */
		(0xC << DC_bop_SHIFT)  | 	/* use source bit op */
	 	DC_sgnzero_enable |		/* normal scanning direction */
	 	DC_shftzero_enable |            /* required for iload */
	 	DC_clipdis_enable;		/* don't use the clip rect */
	
		MGADMA_DSTORG(dstorg);	        /* location of buffer
                                                   we are uploading to */
		MGADMA_MACCESS(texelMaccess);	/* 8 / 16 / 32 bit */
		MGADMA_PITCH( 1 << 15 );	/* linear destination */

		/* spec says we must use YDST / LEN instead of YDSTLEN
                   with linear dest */
		DMAOUTREG( MGAREG_YDST, y*width );	/* starting row */
		MGADMA_LEN( 1 );		/* we are going to
                                                   send all texels as
                                                   one row */
		
		MGADMA_FXBNDRY( 0, width*height-1 );	/* all the
                                                           texels we
                                                           are
                                                           uploading */

		DMAOUTREG( MGAREG_AR0, width * height - 1);  /* source
                                                                texel
                                                                count */
		DMAOUTREG( MGAREG_AR3, 0 );		     /* required */
	
		DMAOUTREG( MGAREG_DMAPAD, 0 );
		DMAOUTREG( MGAREG_DMAPAD, 0 );
		DMAOUTREG( MGAREG_DMAPAD, 0 );
		
		/* pad if needed so the exec is at the end of a
                   register quad */
		MGADMA_DWGCTL_EXEC(cmd);	
	} else {
		/* XY destination, linear iload */
		cmd = DC_opcod_iload | 		/* image load */
	 	DC_atype_rpl |			/* raster replace mode */
		DC_linear_linear | 		/* linear source */
		DC_bltmod_bfcol |		/* source data is
                                                   pre-formatted color */
		(0xC << DC_bop_SHIFT)  | 	/* use source bit op */
	 	DC_sgnzero_enable |		/* normal scanning direction */
	 	DC_shftzero_enable |	        /* required for iload */
	 	DC_clipdis_enable;		/* don't use the clip rect */
	
		MGADMA_DSTORG(dstorg);	        /* location of buffer
                                                   we are uploading to */
		MGADMA_MACCESS(texelMaccess);	/* 8 / 16 / 32 bit */
		MGADMA_PITCH( image->Width );	/* texture width */

		MGADMA_YDSTLEN( y, height );	/* top to bottom */
		MGADMA_FXBNDRY( x, x+width-1 );	/* full width */

		DMAOUTREG( MGAREG_AR0, width * height - 1);	/* source
                                                                   texel
                                                                   count */
		DMAOUTREG( MGAREG_AR3, 0 );			/* required */
	
		/* pad if needed so the exec is at the end of a register quad */
		MGADMA_DWGCTL_EXEC(cmd);
	}
		
	DMAADVANCE();

	/* fill in the secondary buffer with properly converted texels
	from the mesa buffer */
	mgaConvertTexture( dest, texelBytes, image, x, y, width, height );
		
	/* send the secondary data */	
	mgaSecondaryDma( TT_BLIT, dest, dwords );

	mgaglx.warp_serieStart = mgaAllocSecondaryBuffer(0);	// FIXME: do this better...

	/* we need to go back to normal drawing setings after the upload */
	mgaCtx->new_state = ~0;
}


/*
 * mgaCreateTexObj
 * Allocate space for and load the mesa images into the texture memory block.
 * This will happen before drawing with a new texture, or drawing with a
 * texture after it was swapped out or teximaged again.
 */
static void mgaCreateTexObj(mgaContextPtr ctx,
			    struct gl_texture_object *tObj) 
{
	mgaTextureObjectPtr		t;
	int					i, ofs, size;
	PMemBlock				mem;
	struct gl_texture_image *image;
	int					LastLevel;
	int					s, s2;
	int					textureFormat;
	
	mgaMsg( 10,"mgaCreateTexObj( %p )\n", tObj );

	t = malloc( sizeof( *t ) );
	if ( !t ) {
		mgaError( "mgaCreateTexObj: Failed to malloc mgaTextureObject\n" );
		return;
	}
	memset( t, 0, sizeof( *t ) );
	
	image = tObj->Image[ 0 ];
	if ( !image ) {
		return;
	}

	if ( 0 ) {
		/* G400 texture format options */

	} else {
		/* G200 texture format options */
		
		switch( image->Format ) {
		case GL_RGB:
	 	case GL_LUMINANCE:
	 		if ( image->IntFormat != GL_RGB5 &&
			     ( image->IntFormat == GL_RGB8 || mgaglx.default32BitTextures ) ) {
	 			t->texelBytes = 4;
	 			textureFormat = TMC_tformat_tw32;
	 		} else {
				t->texelBytes = 2;
				textureFormat = TMC_tformat_tw16;
			}
			break;
		case GL_ALPHA:
	 	case GL_LUMINANCE_ALPHA:
		case GL_INTENSITY:
		case GL_RGBA:
	 		if ( image->IntFormat != GL_RGBA4 &&
			     ( image->IntFormat == GL_RGBA8 || mgaglx.default32BitTextures ) ) {
	 			t->texelBytes = 4;
	 			textureFormat = TMC_tformat_tw32;
	 		} else {
				t->texelBytes = 2;
				textureFormat = TMC_tformat_tw12;
			}
			break;
		case GL_COLOR_INDEX:
			textureFormat = TMC_tformat_tw8;
			t->texelBytes = 1;
			t->UsePalette = 1;
			break;
		default:
			mgaError( "mgaCreateTexObj: bad image->Format\n" );
			free( t );
			return;	
		}
	}
		
	/* we are going to upload all levels that are present, even if
	   later levels wouldn't be used by the current filtering mode.  This
	   allows the filtering mode to change without forcing another upload
	   of the images */
	LastLevel = MGA_TEX_MAXLEVELS-1;

	ofs = 0;
	for ( i = 0 ; i <= LastLevel ; i++ ) {
		int		levelWidth, levelHeight;
		
		t->offsets[i] = ofs;
		image = tObj->Image[ i ];
		if ( !image ) {
			LastLevel = i - 1;
			mgaMsg( 10, "  missing images after LastLevel: %i\n", LastLevel );
			break;
		}
		/* the G400 doesn't work with textures less than 8 units in size */
		levelWidth = image->Width;
		levelHeight = image->Height;
		if ( levelWidth < 8 ) {
			levelWidth = 8;
		}
		if ( levelHeight < 8 ) {
			levelHeight = 8;
		}
		size = levelWidth * levelHeight * t->texelBytes;
		size = ( size + 31 ) & ~31;	/* 32 byte aligned */
		ofs += size;
	}
	t->totalSize = ofs;
	
	/* allocate a buffer for all levels, swapping out stuff if needed */
        while ( ( mem = mmAllocMem( textureHeap, ofs, 6, 0 ) ) == 0 ) {
        	if ( mgaDestroyOldestTexObj() ) {
        		/* can't hold this texture at all */
			mgaMsg( 10,"mgaCreateTexObj: Couldn't allocate buffer\n" );
        		free( t );
        		return;
        	}
        }

	/* dump the heap contents if loglevel is high enough */
	if ( mgaglx.logLevel >= 15 ) {
		mmDumpMemInfo( textureHeap );
	}

	/* fill in our mga texture object */
	t->magic = MGA_TEXTURE_OBJECT_MAGIC;
	t->tObj = tObj;
	t->ctx = ctx;
	t->next = mgaglx.TexObjList;
	mgaglx.TexObjList = t;
	
	t->memBlock = mem;

	/* base image */	  
	image = tObj->Image[ 0 ];

	/* setup hardware register values */		
	t->texCtl = TMC_takey_1 | TMC_tamask_0 | textureFormat;

	if (image->WidthLog2 >= 3) {
		t->texCtl |= (image->WidthLog2 - 3) << TMC_tpitch_SHIFT;
	} else {
		t->texCtl |= TMC_tpitchlin_enable | 
			(image->Width << TMC_tpitchext_SHIFT);
	}


	t->texCtl2 = TMC_ckstransdis_enable;
	if ( ctx->gl_ctx->Light.Model.ColorControl == GL_SEPARATE_SPECULAR_COLOR ) {
		t->texCtl2 |= TMC_specen_enable;
	}
	
	t->texFilter = TF_minfilter_nrst | TF_magfilter_nrst |
		TF_filteralpha_enable | (0x10 << TF_fthres_SHIFT)
		| ( LastLevel << TF_mapnb_SHIFT );
	
	t->texOrg = textureHeapPhysical + t->memBlock->ofs;
  	/* turn on sysmem reads if we are using sysmem texturing */
   	if ( textureHeapPhysical ) {
		t->texOrg |= TO_texorgmap_sys | use_agp;
	}
	t->texOrg1 = textureHeapPhysical + t->memBlock->ofs + t->offsets[1];
	t->texOrg2 = textureHeapPhysical + t->memBlock->ofs + t->offsets[2];
	t->texOrg3 = textureHeapPhysical + t->memBlock->ofs + t->offsets[3];
	t->texOrg4 = textureHeapPhysical + t->memBlock->ofs + t->offsets[4];

	/* warp texture registers */
	if (MGA_IS_G200(MGAchipset)) {
		ofs = 28;
	} else {
   		ofs = 11;	/* 11 */
   	}

	s = image->Width;
	s2 = image->WidthLog2;
	t->texWidth = 
		MGA_FIELD(TW_twmask, s - 1) |
		MGA_FIELD(TW_rfw,    (10 - s2 - 8) & 63 ) |
		MGA_FIELD(TW_tw,     (s2 + ofs ) | 0x40 );

	
	s = image->Height;
	s2 = image->HeightLog2;
	t->texHeight = 
		MGA_FIELD(TH_thmask, s - 1) |
		MGA_FIELD(TH_rfh,    (10 - s2 - 8) & 63 ) | 
		MGA_FIELD(TH_th,     (s2 + ofs ) | 0x40 );

    
	/* set all the register values for filtering, border, etc */	
	mgaSetTexWrapping( t, tObj->WrapS, tObj->WrapT );
	mgaSetTexFilter( t, tObj->MinFilter, tObj->MagFilter );
	mgaSetTexBorderColor( t, tObj->BorderColor );

	/* create a texture palette if needed */
	if ( image->Format == GL_COLOR_INDEX ) {
		mgaUpdateTexturePalette( ctx->gl_ctx, tObj );
	}

  	tObj->DriverData = t;
  

	/* load each texture level */
	for ( i=0; i <= LastLevel ; i++ ) {
		image = tObj->Image[ i ];
		mgaUploadSubImage( t, i, 0, 0, image->Width, image->Height );
	}
	
	mgaMsg(10,"exiting mgaCreateTexObj\n");	
}

/*
============================================================================

PUBLIC MGA FUNCTIONS

============================================================================
*/

/*
 * mgaInitTextureSystem
 * Check for environment variable options
 */     
void mgaInitTextureSystem( void ) {
	if (glx_getint("mga_32bittextures") ) {
		mgaMsg( 1, "enabling mga_32bittextures\n" );
		mgaglx.default32BitTextures = 1;
	}
}
 


/*
 * mgaDestroyContextTextures
 */
void mgaDestroyContextTextures( mgaContextPtr ctx ) {
	mgaMsg( 1, "mgaDestroyContextTextures( %p )\n", ctx );

	/* FIXME: do we actually need to do anything here?  Shouldn't
	 * MESA call the normal delete function before freeing
	 * textures?  
	 */
#if 0
	while (ctx->TexObjList)
		mgaDestroyTexObj(ctx,ctx->TexObjList);
#endif
}


static void mgaTexRegs( mgaTextureObjectPtr t, mgaUI32 tex1_bit ) {

	DMALOCALS;

	if (tex1_bit)
		tex1_bit = TMC_map1_enable;
	
	MGADMAGETPTR(64);       
	DMAOUTREG(MGAREG_TEXCTL2, t->texCtl2 | tex1_bit );
	DMAOUTREG(MGAREG_TEXCTL, t->texCtl );
	DMAOUTREG(MGAREG_TEXFILTER, t->texFilter );
	DMAOUTREG(MGAREG_TEXBORDERCOL, t->texBorderCol );
	DMAOUTREG(MGAREG_TEXORG, t->texOrg );
	DMAOUTREG(MGAREG_TEXORG1, t->texOrg1 );
	DMAOUTREG(MGAREG_TEXORG2, t->texOrg2 );
	DMAOUTREG(MGAREG_TEXORG3, t->texOrg3 );
	DMAOUTREG(MGAREG_TEXORG4, t->texOrg4 );
		
	DMAOUTREG(MGAREG_TEXWIDTH, t->texWidth );
	DMAOUTREG(MGAREG_TEXHEIGHT, t->texHeight );

	/* Could move these out of here & end up with less if-spaghetti...
	 */
	if (MGA_IS_G200(MGAchipset)) {
		DMAOUTREG(0x2d00 + 24*4, t->texWidth );
		DMAOUTREG(0x2d00 + 34*4, t->texHeight );
	} else {
		DMAOUTREG(0x2d00 + 49*4, 0);
		DMAOUTREG(0x2d00 + 57*4, 0);
		DMAOUTREG(0x2d00 + 53*4, 0);
		DMAOUTREG(0x2d00 + 61*4, 0);

		if (tex1_bit) {
			DMAOUTREG(0x2d00 + 52*4, t->texWidth | 0x40 ); 
			DMAOUTREG(0x2d00 + 60*4, t->texHeight | 0x40 );
		} 
		else {
			if (!mgaCtx->multitex) {
				DMAOUTREG(0x2d00 + 52*4, 0x40 ); 
				DMAOUTREG(0x2d00 + 60*4, 0x40 );
			} 

			DMAOUTREG(0x2d00 + 54*4, t->texWidth | 0x40 );
			DMAOUTREG(0x2d00 + 62*4, t->texHeight | 0x40 );
		}
	}

	/* FIXME: we never modify these registers, they could
	   be one-time-set */
	DMAOUTREG( MGAREG_TEXTRANS, 0xffff );
	DMAOUTREG( MGAREG_TEXTRANSHIGH, 0xffff );

	/* If this is for tex1 only, need to return to broadcast mode
	 * for next time. 
	 */
	if (tex1_bit)
		DMAOUTREG(MGAREG_TEXCTL2, t->texCtl2 );
	
	DMAADVANCE();
}


void mgaUpdateTextureEnvG200( GLcontext *ctx )
{
	struct gl_texture_object *tObj = ctx->Texture.Unit[0].Current;
	mgaTextureObjectPtr t;

	if (!tObj || !tObj->DriverData)
		return;

	t = (mgaTextureObjectPtr)tObj->DriverData;

	switch (ctx->Texture.Unit[0].EnvMode) {
	case GL_REPLACE:
		t->texCtl &= ~TMC_tmodulate_enable;
		t->texCtl2 &= ~TMC_decaldis_enable;
		break;
	case GL_MODULATE:
		t->texCtl |= TMC_tmodulate_enable;
		t->texCtl2 &= ~TMC_decaldis_enable;
		break;
	case GL_DECAL:
		t->texCtl &= ~TMC_tmodulate_enable;
		t->texCtl2 &= ~TMC_decaldis_enable;
		break;
	case GL_BLEND:
		mgaCtx->SoftwareFallback |= FALLBACK_TEXTURE;
		break;
	default:
	}
}


void mgaUpdateTextureStage( GLcontext *ctx, int unit )
{
	mgaUI32 *reg = &mgaCtx->regTDUAL[unit];
	GLuint source = mgaCtx->tmu_source[unit];
	struct gl_texture_object *tObj = ctx->Texture.Unit[source].Current;

	mgaCtx->reg_dirty |= (1<<(MGA_SETUP_TDUAL0+unit));

	*reg = 0;
	if (unit == 1)
		*reg = mgaCtx->regTDUAL[0];
	
	if ( tObj != ctx->Texture.Unit[source].CurrentD[2] ) 
		return;
	
	if ( ((ctx->Enabled>>(source*4))&TEXTURE0_ANY) != TEXTURE0_2D ) 
		return;
	
	if (!tObj || !tObj->Complete) 
		return;

	switch (ctx->Texture.Unit[source].EnvMode) {
	case GL_REPLACE:
		*reg = (TD0_color_sel_arg1 |
			TD0_alpha_sel_arg1 );
		break;

	case GL_MODULATE:
		if (unit == 0)
			*reg = ( TD0_color_arg2_diffuse |
				 TD0_color_sel_mul | 
				 TD0_alpha_arg2_diffuse |
				 TD0_alpha_sel_arg1);
		else
			*reg = ( TD0_color_arg2_prevstage |
				 TD0_color_alpha_prevstage |
				 TD0_color_sel_mul | 
				 TD0_alpha_arg2_prevstage |
				 TD0_alpha_sel_arg1);
		break;
	case GL_DECAL:
		*reg = (TD0_color_arg2_fcol | 
			TD0_color_alpha_currtex |
			TD0_color_alpha2inv_enable |
			TD0_color_arg2mul_alpha2 |
			TD0_color_arg1mul_alpha1 |
			TD0_color_add_add |
			TD0_color_sel_add |
			TD0_alpha_arg2_fcol |
			TD0_alpha_sel_arg2 );
		break;
	case GL_BLEND:
		mgaCtx->SoftwareFallback |= FALLBACK_TEXTURE;
		break;
	default:
	}
}



void mgaUpdateTextureObject( GLcontext *ctx, int unit ) {
	mgaTextureObjectPtr t;
	struct gl_texture_object	*tObj;
	GLuint enabled;
	GLuint source = mgaCtx->tmu_source[unit];

	CHECK_CONTEXT( return; );

	mgaMsg(15,"mgaUpdateTextureState %d\n", unit);
	
	/* disable texturing until it is known to be good */
	mgaCtx->reg_dirty |= (1<<MGA_SETUP_DWGCTL);

	mgaCtx->regDWGCTL = (( mgaCtx->regDWGCTL & DC_opcod_MASK ) | 
			     DC_opcod_trap);

	enabled = (ctx->Texture.Enabled>>(source*4))&TEXTURE0_ANY;
	if (enabled != TEXTURE0_2D) {
		if (enabled)
			mgaCtx->SoftwareFallback |= FALLBACK_TEXTURE;
		return;
	}

	tObj = ctx->Texture.Unit[source].Current;

	if ( !tObj || tObj != ctx->Texture.Unit[source].CurrentD[2] ) 
		return;
	
	/* if the texture object doesn't exist at all (never used or
	   swapped out), create it now, uploading all texture images */

	if ( !tObj->DriverData ) {
		/* clear the current pointer so that texture object can be
		   swapped out if necessary to make room */
		mgaglx.currentTextureObject[source] = NULL;
		mgaCreateTexObj( mgaCtx, tObj );

		if ( !tObj->DriverData ) {
			mgaMsg( 5, "mgaUpdateTextureState: create failed\n" );
			mgaCtx->SoftwareFallback |= FALLBACK_TEXTURE;
			return;		/* can't create a texture object */
		}
	}

	/* we definately have a valid texture now */
	mgaCtx->regDWGCTL = (( mgaCtx->regDWGCTL & DC_opcod_MASK ) | 
			     DC_opcod_texture_trap);

	t = (mgaTextureObjectPtr)tObj->DriverData;

	/* make the texture current if necessary */
	mgaglx.currentTextureObject[unit] = t;


	t->texCtl2 &= ~TMC_dualtex_enable;
	if (ctx->Texture.Enabled == (TEXTURE0_2D|TEXTURE1_2D))
		t->texCtl2 |= TMC_dualtex_enable;

	t->texCtl2 &= ~TMC_specen_enable;
	if (ctx->Light.Model.ColorControl == GL_SEPARATE_SPECULAR_COLOR)
		t->texCtl2 |= TMC_specen_enable;

	
	/* note the use on this frame to prevent texture thrashing */
	if ( t->age != mgaglx.swapBuffersCount ) {
		mgaglx.c_textureUtilization += t->totalSize;
		t->age = mgaglx.swapBuffersCount;
	}

	/* perform the actual register writes */
	if ( 1 ) 
		mgaTexRegs( t, unit );		       
	
	/* update texture palette if needed */
	if ( t->UsePalette ) {
		if (ctx->Texture.SharedPalette) {
			if ( mgaglx.currentTexturePalette != 
			     mgaCtx->GlobalPalette ) {
				mgaLoadTexturePalette( mgaCtx->GlobalPalette );
			}
		} else {
			if ( mgaglx.currentTexturePalette != t->Palette ) {
	    			mgaLoadTexturePalette( t->Palette );
			}
    		}
	}		      
}






/* The G400 is now programmed quite differently wrt texture environment.
 */
void mgaUpdateTextureState( GLcontext *ctx )
{
	mgaCtx->SoftwareFallback &= ~FALLBACK_TEXTURE;

	if (MGA_IS_G400( MGAchipset )) {
		mgaUpdateTextureObject( ctx, 0 );		
		mgaUpdateTextureStage( ctx, 0 );

		if (mgaCtx->multitex) {
			mgaUpdateTextureObject( ctx, 1 );	
			mgaUpdateTextureStage( ctx, 1 );
		} else {
			mgaCtx->regTDUAL[1] = mgaCtx->regTDUAL[0];
		}

		if (mgaCtx->SoftwareFallback) {
			fprintf(stderr, "Software fallback!!!\n");
			FatalError("aborting");
		}
	} else {
		mgaUpdateTextureObject( ctx, 0 );
		mgaUpdateTextureEnvG200( ctx );
	}
}



/*
============================================================================

Driver functions called directly from mesa

============================================================================
*/

/*
 * mgaTexEnv
 */      
void mgaTexEnv( GLcontext *ctx, GLenum pname, const GLfloat *param ) {
	if (!VALID_MGA_CONTEXT(mgaCtx)) {
		return;
	}

	mgaMsg( 10, "mgaTexEnv( %i )\n", pname );

	if (pname == GL_TEXTURE_ENV_MODE) {
		/* force the texture state to be updated */
		mgaglx.currentTextureObject[0] = NULL;  
		mgaCtx->new_state |= MGA_NEW_TEXTURE;
	}
}

/*
 * mgaTexImage
 */ 
void mgaTexImage( GLcontext *ctx, GLenum target,
		  struct gl_texture_object *tObj, GLint level,
		  GLint internalFormat,
		  const struct gl_texture_image *image ) {
	mgaTextureObjectPtr t;

	CHECK_CONTEXT( return; );
  
	mgaMsg( 10,"mgaTexImage( %p, level %i )\n", tObj, level );
  
 	/* just free the mga texture if it exists, it will be recreated at
	mgaUpdateTextureState time. */  
	t = (mgaTextureObjectPtr) tObj->DriverData;
	if ( t ) {
		/* if this is the current object, it will force an update */
 	 	mgaDestroyTexObj( mgaCtx, t );
		mgaCtx->new_state |= MGA_NEW_TEXTURE;
	}
}

/*
 * mgaTexSubImage
 */      
void mgaTexSubImage( GLcontext *ctx, GLenum target,
		     struct gl_texture_object *tObj, GLint level,
		     GLint xoffset, GLint yoffset,
		     GLsizei width, GLsizei height,
		     GLint internalFormat,
		     const struct gl_texture_image *image ) {
	mgaTextureObjectPtr t;

	mgaMsg(10,"mgaTexSubImage() Size: %d,%d of %d,%d; Level %d\n",
		width, height, image->Width,image->Height, level);

	CHECK_CONTEXT( return; );

	t = (mgaTextureObjectPtr) tObj->DriverData;

	/* if we don't have a hardware texture, it will be automatically
	created with current state before it is used, so we don't have
	to do anything now */
	if ( !t || target != GL_TEXTURE_2D ) {
		return;
	}

	/* the texture currently exists, so directly update it */
	mgaUploadSubImage( t, level, xoffset, yoffset, width, height );
}

/*
 * mgaTexParameter
 * This just changes variables and flags for a state update, which
 * will happen at the next mgaUpdateTextureState
 */
void mgaTexParameter( GLcontext *ctx, GLenum target,
		      struct gl_texture_object *tObj,
		      GLenum pname, const GLfloat *params ) {
	mgaTextureObjectPtr t;

	CHECK_CONTEXT( return; );

	mgaMsg( 10, "mgaTexParameter( %p, %i )\n", tObj, pname );

	t = (mgaTextureObjectPtr) tObj->DriverData;

	/* if we don't have a hardware texture, it will be automatically
	created with current state before it is used, so we don't have
	to do anything now */
	if ( !t || target != GL_TEXTURE_2D ) {
		return;
	}

	switch (pname) {
	case GL_TEXTURE_MIN_FILTER:
	case GL_TEXTURE_MAG_FILTER:
		mgaSetTexFilter( t, tObj->MinFilter, tObj->MagFilter );
		break;

	case GL_TEXTURE_WRAP_S:
	case GL_TEXTURE_WRAP_T:
		mgaSetTexWrapping(t,tObj->WrapS,tObj->WrapT);
		break;
  
	case GL_TEXTURE_BORDER_COLOR:
		mgaSetTexBorderColor(t,tObj->BorderColor);
		break;

	default:
		return;
	}
	/* force the texture state to be updated */
	mgaglx.currentTextureObject[0] = NULL;  
	mgaCtx->new_state |= MGA_NEW_TEXTURE;
}

/*
 * mgaBindTexture
 */      
void mgaBindTexture( GLcontext *ctx, GLenum target,
		     struct gl_texture_object *tObj ) {
	CHECK_CONTEXT( return; );

	mgaMsg( 10, "mgaBindTexture( %p )\n", tObj );
		
	/* force the texture state to be updated 
	 */
	mgaglx.currentTextureObject[ctx->Texture.CurrentUnit] = 0;  
	mgaCtx->new_state |= MGA_NEW_TEXTURE;
}

/*
 * mgaDeleteTexture
 */      
void mgaDeleteTexture( GLcontext *ctx, struct gl_texture_object *tObj ) {
	CHECK_CONTEXT( return; );

	mgaMsg( 10, "mgaDeleteTexture( %p )\n", tObj );
	
	/* delete out driver data */
	if ( tObj->DriverData ) {
		mgaDestroyTexObj( mgaCtx, (mgaTextureObjectPtr)(tObj->DriverData) );
		mgaCtx->new_state |= MGA_NEW_TEXTURE;
	}
}

/*
 * mgaUpdateTexturePalette
 */      
void mgaUpdateTexturePalette( GLcontext *ctx, struct gl_texture_object *tObj ) {
	mgaTextureObjectPtr t;
	mgaUI16 *dst;
	mgaUI8  *src;
	int i, size, next;
	GLenum format;

	CHECK_CONTEXT( return; );
  
	mgaMsg( 10, "mgaUpdateTexturePalette( %p )\n", tObj );

	if ( tObj ) {
		/* private palette for a single texture */
		mgaMsg(8,"mgaUpdateTexturePalette(): size %d\n", 
		       tObj->PaletteSize);
		t = (mgaTextureObjectPtr) tObj->DriverData;
		if ( !t ) {
			return;
		}
		size  = tObj->PaletteSize;
		format= tObj->PaletteFormat;
		dst   = t->Palette;
		src   = (mgaUI8 *) tObj->Palette;
	} else {
	  	/* shared palette */
		size  = ctx->Texture.PaletteSize;
		format = ctx->Texture.PaletteFormat;
		dst   = mgaCtx->GlobalPalette;
		src   = (mgaUI8 *) ctx->Texture.Palette;
		mgaMsg(8, "mgaUpdateTexturePalette(): "
		       "size %d intFormat %x format %x\n",
		       size, ctx->Texture.PaletteIntFormat, 
		       ctx->Texture.PaletteFormat);
	}
	if ( size > 256 ) {
		mgaError("mgaUpdateTexturePalette(): palette > 256 entries!\n");
		return;
	}

	/* force the hardware texture palette to be updated before drawing again
	if this is the current palette */
	if ( mgaglx.currentTexturePalette == dst ) {
		mgaglx.currentTexturePalette = NULL;
		mgaCtx->new_state |= MGA_NEW_TEXTURE;
	}
  
  	/* update our pre-combined version of the palette */
	switch( format ) { 
	case GL_RGB: next = 3; break;
	case GL_RGBA:next = 4; break;
	default : 
		mgaError("mgaUpdateTexturePalette(): yet unsupported palette format %x\n");
		next = 4;
	}
	
	for (i=0; i < size; i++) {
		*dst = MGAPACKCOLOR565(src[RCOMP],src[GCOMP],src[BCOMP]);
		dst++;
		src += next;
	} 
}


/*
 * mgaIsTextureResident
 */      
GLboolean mgaIsTextureResident( GLcontext *ctx, struct gl_texture_object *t ) {
   mgaTextureObjectPtr mt = (mgaTextureObjectPtr)t->DriverData;
   return mt != NULL;
}

