/*
 * GLX Hardware Device Driver for Matrox G200/G400
 * Copyright (C) 1999 Wittawat Yamwong
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR 
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE 
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 *
 *    Wittawat Yamwong <Wittawat.Yamwong@stud.uni-hannover.de>
 */

/* TODO: remove most of these includes that we don't need... */

#include <stdlib.h>

#include "context.h"
#include "depth.h"
#include "macros.h"
#include "texstate.h"
#include "triangle.h"
#include "vb.h"
#include "types.h"

#include "xsmesaP.h"
#include "glx_log.h"
#include "mesaglx/context.h"
#include "mesaglx/matrix.h"
#include "mesaglx/types.h"

#define GC XXGC
#include "gcstruct.h"
#include "pixmapstr.h"
#include "servermd.h" /* PixmapBytePad */
#include "scrnintstr.h"
#include "regionstr.h"
#include "windowstr.h"
#undef GC

#include "vga.h"
#include "vgaPCI.h"
#include "mga.h"
#include "xaa/xf86xaa.h"

#include "mm.h"
#include "mgabuf.h"
#include "mgadd.h"
#include "g200_mac.h"
#include "mgalib.h"
#include "mgalog.h"
#include "mgadirect.h"
#include "mgapipeline.h"
#include "mgarender.h"
#include "mgatris.h"

extern XSMesaContext XSMesa;


/*
 * mgaBackToFront24
 * Special code to blit from a 32 bit back buffer to a 24 bit depth buffer
 */
static int mgaBackToFront24(DrawablePtr drawable, mgaBufferPtr buf) {
	RegionPtr		prgnClip;
	BoxPtr		pbox;
	int 			i,nbox;
	int 			xorg, yorg, pitch;
	DMALOCALS;

	prgnClip = &((WindowPtr)drawable)->clipList;
	pbox = REGION_RECTS(prgnClip);
	nbox = REGION_NUM_RECTS(prgnClip);
	if( !nbox ) {
		/* window is completely covered */
		return Success;
	}

	xorg = drawable->x;
	yorg = drawable->y;
	pitch = buf->Pitch;

	/* add the blit commands to the dma buffer */
	/* FIXME: this may have overflow issues */
	MGADMAGETPTR( 8000 );

	/* we are going to treat the buffers as 8bpp and copy three pixel
	wide strips to move the 24 bit rgb values.  */
	MGADMA_DSTORG(mgaFrontBuffer->Setup[MGA_SETUP_DSTORG]);
	MGADMA_MACCESS(0);	/* 8bpp */
	MGADMA_PITCH( mgaFrontBuffer->Pitch * 3 );

	MGADMA_SRCORG(buf->Setup[MGA_SETUP_DSTORG]);
	DMAOUTREG(MGAREG_AR5,pitch * 4);

	MGADMA_DWGCTL(DC_opcod_bitblt | DC_atype_rpl | DC_linear_xy |
	     DC_solid_disable | DC_arzero_disable | DC_sgnzero_enable |
	     DC_shftzero_enable | (0xC << DC_bop_SHIFT) | DC_bltmod_bfcol |
	     DC_pattern_disable | DC_transc_disable | DC_clipdis_enable); 
	     
	/* xy bitblt each visible rectangle */
	for (i=nbox; i > 0; i--) {
		int	c;
		int x = pbox->x1 - xorg;
		int y = pbox->y1 - yorg;
		int	src;
		int	dest;
		
		dest = pbox->x1 * 3;
		src = ( x + y*pitch ) * 4;
		for ( c = pbox->x1 ; c < pbox->x2 ; c++ ) {
			DMAOUTREG(MGAREG_AR0,src + 2);
			DMAOUTREG(MGAREG_AR3,src);
			MGADMA_FXBNDRY( dest, dest+2 );
			MGADMA_YDSTLEN_EXEC(pbox->y1, pbox->y2 - pbox->y1 );
			src += 4;
			dest += 3;
		}
		pbox++;
	}
  
	MGADMA_SRCORG(0);
 	DMAADVANCE();

   return Success;   
}

/*
 * mgaBackToFront
 * Blit the visible rectangles from the back buffer to the screen
 * Can't really do this from a direct context - don't have the clip
 * info, and getting it would be a lot slower than just sending a
 * request to the server to do the blit there.
 */
int mgaBackToFront(DrawablePtr drawable, mgaBufferPtr buf) {
	RegionPtr		prgnClip;
	BoxPtr		pbox;
	int 			i,nbox;
	int 			xorg, yorg, pitch;
	DMALOCALS;

	if (!VALID_MGA_BUFFER(buf)) {
		mgaError("BackToFront(): invalid back buffer\n");
		return BadMatch;
	}
	if ( !VALID_MGA_BUFFER(mgaFrontBuffer) ) {
		mgaError("BackToFront(): invalid mgaFrontBuffer\n");
		return BadMatch;
	}
	if ( drawable->width != buf->Width 
		|| drawable->height != buf->Height 
	 	|| drawable->type != DRAWABLE_WINDOW ) {
		mgaError("BackToFront(): bad drawable\n");
		return BadMatch;
	}

	/* 24 bit case is a separate hack */
	if ( vgaBitsPerPixel == 24 ) { 
		return mgaBackToFront24( drawable, buf );
	}

	prgnClip = &((WindowPtr)drawable)->clipList;
	pbox = REGION_RECTS(prgnClip);
	nbox = REGION_NUM_RECTS(prgnClip);
	if( !nbox ) {
		/* window is completely covered */
		return Success;
	}

	xorg = drawable->x;
	yorg = drawable->y;
	pitch = buf->Pitch;
	
	
	if (MESA_VERBOSE&VERBOSE_DRIVER)
	   fprintf(stderr, "needEnter3D in mgaBackToFront\n");

	/* add the blit commands to the dma buffer */
	MGADMAGETPTR( 10 + 10 * nbox );

	MGADMA_DSTORG(mgaFrontBuffer->Setup[MGA_SETUP_DSTORG]);
	MGADMA_MACCESS(mgaFrontBuffer->Setup[MGA_SETUP_MACCESS]);
	MGADMA_PITCH(mgaFrontBuffer->Setup[MGA_SETUP_PITCH]);
	MGADMA_SRCORG(buf->Setup[MGA_SETUP_DSTORG]);
	DMAOUTREG(MGAREG_AR5,pitch);
	MGADMA_DWGCTL(DC_opcod_bitblt | DC_atype_rpl | DC_linear_xy |
	     DC_solid_disable | DC_arzero_disable | DC_sgnzero_enable |
	     DC_shftzero_enable | (0xC << DC_bop_SHIFT) | DC_bltmod_bfcol |
	     DC_pattern_disable | DC_transc_disable | DC_clipdis_enable ); 
	     
	/* xy bitblt each visible rectangle */
	for (i=nbox; i > 0; i--) {
		int x = pbox->x1 - xorg;
		int y = pbox->y1 - yorg;
		int w = pbox->x2 - pbox->x1;
		int h = pbox->y2 - pbox->y1;
		int start = x + y*pitch;
		DMAOUTREG(MGAREG_AR0,start + w - 1);
		DMAOUTREG(MGAREG_AR3,start);
		
		MGADMA_FXBNDRY(pbox->x1,pbox->x2-1);
		MGADMA_YDSTLEN_EXEC(pbox->y1,h);
		pbox++;
	}
  
	MGADMA_SRCORG(0);
 	DMAADVANCE();

	return Success;   
}


/*
 * ClearBox
 * Add hardware commands to draw a filled box for the
 * debugging display.
 */
static void ClearBox( int x, int y, int w, int h, int r, int g, int b ) {
	DMALOCALS;
	mgaUI32	cmd;

	MGADMAGETPTR( 32 );

	cmd = DC_opcod_trap | DC_arzero_enable | DC_sgnzero_enable | 
	  DC_shftzero_enable | (0xC << DC_bop_SHIFT)  | 
	  DC_atype_rstr | DC_solid_enable;

	MGADMA_FCOL(mgaPackColor(r,g,b,255));
	MGADMA_YDSTLEN(y,h);
	MGADMA_FXBNDRY(x,x+w);
	MGADMA_DWGCTL_EXEC(cmd);

	DMAADVANCE();
}


/*
 * performanceBoxes
 * Draw some small boxesin the corner of the buffer
 * based on some performance information
 */
void mgaPerformanceBoxes( int is_direct ) {
	int		w, x, t;
	
	if ( !mgaglx.boxes || !mgaDB ) {
		return;
	}
  
  	{
	  	/* Make sure we draw into our window: */
		DMALOCALS;
		MGADMAGETPTR( 4 );
	  	MGADMA_DSTORG(mgaDB->Setup[MGA_SETUP_DSTORG]);
		MGADMA_MACCESS(mgaDB->Setup[MGA_SETUP_MACCESS]);
		MGADMA_PITCH(mgaDB->Setup[MGA_SETUP_PITCH]);
		MGADMA_PLNWT(mgaDB->Setup[MGA_SETUP_PLNWT]);
	  	DMAADVANCE();
  	}

       	/* draw a box to show we are active, so if it is't seen
	it means that it is completely software based rendering  */
	/* draw a purple box if we are direct rendering */
	if ( is_direct ) {			/* purple = direct (client dma) rendering */
		ClearBox( 4, 4, 8, 8, 255, 0, 255 );
	} else if ( mgaglx.dmaDriver ) {	/* white = server dma rendering */
		ClearBox( 4, 4, 8, 8, 255, 255, 255 );
	} else {				/* grey = server PDMA */
		ClearBox( 4, 4, 8, 8, 128, 128, 128 );
	}

	/* Direct client doesn't set these values correctly */
#if 0
	/* draw a line over the first box if we are using a system memory back buffer */
	if ( mgaDB->SystemMemory ) {
		ClearBox( 4, 2, 8, 1, 255, 255, 255 );
	}
	 
	/* draw a line over the first box if we are using a system memory depth buffer */
	if ( mgaDB->ZBuffer && mgaDB->ZBuffer->SystemMemory ) {
		ClearBox( 4, 14, 8, 1, 255, 255, 255 );
	}
#endif
	 
	/* draw a red box if we had to wait for drawing to complete 
	(software render or texture swap) */
	if ( mgaglx.c_drawWaits ) {
		ClearBox( 16, 4, 8, 8, 255, 0, 0 );
		mgaglx.c_drawWaits = 0;
	}
	
	/* draw a blue box if the register protection signal was hit */
	if ( mgaglx.c_signals ) {
		ClearBox( 28, 4, 8, 8, 0, 0, 255 );
		mgaglx.c_signals = 0;
	}
	
	/* draw a yellow box if textures were swapped */
	if ( mgaglx.c_textureSwaps ) {
		ClearBox( 40, 4, 8, 8, 255, 255, 0 );
		mgaglx.c_textureSwaps = 0;
	}
	
	/* draw a green box if we had to wait for dma to complete (full utilization) 
	on the previous frame */
	if ( !mgaglx.hardwareWentIdle ) {
		ClearBox( 64, 4, 8, 8, 0, 255, 0 );
	}
	mgaglx.hardwareWentIdle = 0;
	
	/* show buffer utilization */
	if ( mgaglx.c_dmaFlush > 1 ) {
		/* draw a solid bar if we flushed more than one buffer */
		ClearBox( 4, 16, 252, 4, 255, 32, 32 );
	} else {
		/* draw bars to represent the utilization of primary and secondary buffers */
		ClearBox( 4, 16, 252, 4, 32, 32, 32 );
		t = dma_buffer->maxPrimaryDwords + dma_buffer->maxSecondaryDwords;
		w = 252 * dma_buffer->primaryDwords / t;
		if ( w < 1 ) {
			w = 1;
		}
		ClearBox( 4, 16, w, 4, 196, 128, 128 );
		x = 252 * dma_buffer->maxPrimaryDwords / t;
		w = 252 * dma_buffer->secondaryDwords / t;
		if ( w < 1 ) {
			w = 1;
		}
		ClearBox( 4+x, 16, w, 4, 196, 128, 128 );
	}
	mgaglx.c_dmaFlush = 0;
}


/*
 * Copy the back buffer to the front buffer.  If there's no back buffer
 * this is a no-op.  Only called in indirect contexts.
 */
void mgaServerSwapBuffers( XSMesaBuffer b ) {
#ifdef PROFILE
	GLdouble t0 = gl_time();
#endif
	mgaBufferPtr buf;

	mgaglx.swapBuffersCount++;

	/* make sure mesa gives us everything */
	/* 3.1 leaves unsent verts without this, but there is probably
	a better internal call to get them sent... */
	if (mgaCtx && mgaCtx->gl_ctx)
	   glFlush();

	if ( !b->db_state ) {
		/* not double buffered, so do nothing */
	} else {
		ValidateGC(b->frontbuffer, b->cleargc);
		if ( b->backimage ) {
			if ( (buf = (mgaBufferPtr)b->backimage->devPriv) != NULL && buf->Drawable ) {
  			 	/* flush the last WARP triangle series */
				mgaWarpFinishSerie();

				/* diagnostic drawing tools */
				mgaPerformanceBoxes( 0 );

				/* hardware accelerated back to front blit */	
				mgaBackToFront( (DrawablePtr)b->frontbuffer,buf );

				/* make sure all dma is going to get executed */
				/* if everything has gone well, this will be the only flush each frame */
				mgaDmaFlush();
			} else {
				/* Use backimage's dimension (not buffer's) */
				(*b->cleargc->ops->PutImage)((DrawablePtr)b->frontbuffer,
							 b->cleargc,
							 b->frontbuffer->depth,
							 0, 0,
							 b->backimage->width, 
							 b->backimage->height,
							 0, ZPixmap, b->backimage->data);
			}
		} else {
			/* Copy pixmap to window on server */
			(*b->cleargc->ops->CopyArea)((DrawablePtr)b->backpixmap,
					       b->frontbuffer,
					       b->cleargc,
					       0, 0, b->width, b->height,
					       0, 0);
		}
	}
  
	/* report performance counters */
	mgaMsg(9, "swapBuffers: c_triangles:%i  c_lines:%i c_points:%i c_setup:%i c_textures:%i\n", 
		mgaglx.c_triangles, mgaglx.c_lines, mgaglx.c_points, mgaglx.c_setupPointers,
		mgaglx.c_textureSwaps );
	mgaglx.c_triangles = 0;
	mgaglx.c_lines = 0;
	mgaglx.c_points = 0;
	mgaglx.c_setupPointers = 0;
	mgaMsg(9, "---------------------------------------------------------\n" );

#ifdef PROFILE
  XSMesa->gl_ctx->SwapCount++;
  XSMesa->gl_ctx->SwapTime += gl_time() - t0;
#endif
}
