#!/usr/bin/perl -w
#
##############################################################################
#          Generic parser for easy hardware register specification
# V1.0
##############################################################################
# Author: simon pogarcic, sim@suse.de
#
# Permission is hereby granted, free of charge, to any person obtaining a
# copy of this software and associated documentation files (the "Software"),
# to deal in the Software without restriction, including without limitation
# the rights to use, copy, modify, merge, publish, distribute, sublicense,
# and/or sell copies of the Software, and to permit persons to whom the
# Software is furnished to do so, subject to the following conditions:
# 
# The above copyright notice and this permission notice shall be included
# in all copies or substantial portions of the Software.
# 
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
# OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
# SIMON POGARCIC, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM, 
# DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR 
# THERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE 
# OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
#
##############################################################################
#
#          PUT YOUR DEFINITIONS WHICH DOESN'T FIT FOR PARSER HERE
#   They will appear at the start of automaticly generated register file
#
##############################################################################
$code_start = <<'EOF';
/* author: stephen crowley, crow@debian.org */

/*
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * STEPHEN CROWLEY, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR 
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE 
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */

#ifndef _MGAREGS_H_
#define _MGAREGS_H_

EOF

$code_end = <<'EOF';

#endif 	/* _MGAREGS_H_ */

EOF

$text = <<'EOF';
###############################################################################

# ==========================================================================
# === REGISTER =============== OFFSET ======= ACCESS = CARD ================
# ==========================================================================

@REGION0

@BCOMM Power Graphic Mode Memory Space Registers

MGA_EXEC        0x0100

# G200 only
AGP_PLL		0x1e4c	rw	
@VALUES AGP_PLL_
	agp2xpllen_ 	[0]
@ENDVALUES

# G400 only
CFG_OR		0x1e4c  rw
@VALUES CFG_OR_
	comp_or_	[3]
	compfreq_	[4-7]
	comporup_	[8-11]
	compordn_	[12-15]
	e2pq_		[16]
	e2pqbypcsn_	[17]
	e2pqbypd_	[18]
	e2pbypclk_	[19]
	e2pbyp_		[20]
	rate_cap_or_	[21-23]
	rq_or_		[24-28]
@ENDVALUES

ALPHACTRL	0x2c7c	wo	
@VALUES AC_
	src_		[0-3]
		+zero		0
		+one		1
		+dst_color 	2
	        +om_dst_color   3
	        +src_alpha	4
		+om_src_alpha   5
		+dst_alpha	6
		+om_dst_alpha   7
		+src_alpha_sat  8
	dst_		[4-7]
		+zero		0
		+one		1
		+src_color	2
		+om_src_color	3
		+src_alpha	4
		+om_src_alpha	5
		+dst_alpha	6
		+om_dst_alpha	7
	amode_		[8-9]
		+FCOL		0
		+alpha_channel 	1
		+video_alpha	2
		+RSVD		3
	astipple_	[11]
	aten_		[12]
	atmode_		[13-15]
		+noacmp		0
		+ae		2
	   	+ane		3
		+alt		4
		+alte		5
		+agt		6
		+agte		7
	atref_		[16-23]
	alphasel_	[24-25]
		+fromtex	0
		+diffused	1
		+modulated	2
		+trans		3
@ENDVALUES		

ALPHASTART	0x2c70	wo
ALPHAXINC	0x2c74	wo
ALPHAYINC       0x2c78  wo
	
AR0		0x1c60	wo
@VALUES AR0_
	ar0_		[0-17]
@ENDVALUES

AR1             0x1c64  wo
@VALUES AR1_
	ar1_		[0-23]
@ENDVALUES

AR2             0x1c68  wo
@VALUES AR2_
	ar2_		[0-17]
@ENDVALUES

AR3             0x1c6c  wo
@VALUES AR3_
	ar3_		[0-23]
	spage_		[24-26]
@ENDVALUES

AR4             0x1c70  wo
@VALUES AR4_
	ar4_		[0-17]
@ENDVALUES

AR5             0x1c74  wo
@VALUES AR5_
	ar5_		[0-17]
@ENDVALUES

AR6             0x1c78  wo
@VALUES AR6_
	ar6_		[0-17]
@ENDVALUES

BCOL		0x1c20	wo

BESA1CORG	0x3d10	wo
BESA1ORG	0x3d00	wo
BESA2CORG	0x3d14	wo
BESA2ORG	0x3d04	wo
BESB1CORG	0x3d18	wo
BESB1ORG	0x3d08	wo	
BESB2CORG	0x3d1c	wo
BESB2ORG	0x3d0c	wo

BESCTL 		0x3d20	rw
@VALUES	BC_
	besen_		[0]
	besv1srcstp_	[6]{even/odd}
	besv2srcstp_	[8]
	beshfen_	[10]
	besvfen_	[11]
	beshfixc_	[12]{weight/coeff}
	bescups_	[16]
	bes420pl_	[17]{422/420}
	besdith_	[18]
	beshmir_	[19]
	besbwen_	[20]{color/bw}
	besblank_	[21]
	besfselm_	[24]{soft/hard}
	besfsel_	[25-26]
		+a1	0
		+a2	1
		+b1	2
		+b2	3
@ENDVALUES

BESGLOBCTL	0x3dc0	rw
@VALUES BGC_
	beshzoom_	[0]
	beshzoomf_	[1]
	bescorder_	[3]{even/odd}
	besreghup_	[4]
	besvcnt_	[16-27]
@ENDVALUES

BESHCOORD	0x3d28	wo
@VALUES BHC_
	besright_	[0-10]
	besleft_	[16-26]
@ENDVALUES

BESHISCAL	0x3d30	wo
@VALUES BHISF_
	beshiscal_	[2-20]
@ENDVALUES

BESHSRCEND	0x3d3c	wo
@VALUES BHSE_
	beshsrcend_	[2-25]
@ENDVALUES

BESHSRCLST	0x3d50	wo
@VALUES BHSL_
	beshsrclst_	[16-25]
@ENDVALUES

BESHSRCST	0x3d38	wo
@VALUES BHSS_
	beshsrcst_	[2-25]
@ENDVALUES

BESPITCH	0x3d24	wo
@VALUES BP_
	bespitch_	[0-11]
@ENDVALUES

BESSTATUS	0x3dc4	ro
@VALUES BS_
	besstat_	[0-1]
		+a1	0
		+a2	1
		+b1	2
		+b2	3
@ENDVALUES

BESV1SRCLST	0x3d54	wo
@VALUES BSF_
	besv1srclast_	[0-9]
@ENDVALUES

BESV2SRCLST	0x3d58	wo
@VALUES BSF_
	besv2srclst_	[0-9]
@ENDVALUES

BESV1WGHT	0x3d48	wo
@VALUES BSF_
	besv1wght_	[2-15]
	besv1wghts_	[16]
@ENDVALUES

BESV2WGHT	0x3d4c	wo
@VALUES BSF_
	besv2wght_	[2-15]
	besv2wghts_	[16]
@ENDVALUES

BESVCOORD	0x3d2c	wo
@VALUES BVC_
	besbot_		[0-10]
	bestop_		[16-26]
@ENDVALUES
       
BESVISCAL	0x3d34	wo
@VALUES BVISF_
	besviscal_	[2-20]
@ENDVALUES

# --- Video Junk, we dont need it ---
CODECADDR	0x3e44	wo
CODECCTL	0x3e40	wo
CODECHARDPTR	0x3e4c	ro
CODECHOSTPTR	0x3e48	wo
CODECLCODE	0x3e50	ro
# -----------------------------------

CXBNDRY		0x1c80	wo
@VALUES CXB_
	cxleft_		[0-11]
	cxright_	[16-27]
@ENDVALUES

CXLEFT		0x1ca0	wo
CXRIGHT		0x1ca4	wo

DMAMAP30	0x1e30	rw
DMAMAP74	0x1e34	rw
DMAMAPB8	0x1e38	rw
DMAMAPFC	0x1e3c	rw
DMAPAD		0x1c54	wo

DR0_Z32LSB	0x2c50	wo
DR0_Z32MSB	0x2c54	wo	
DR2_Z32LSB	0x2c60	wo
DR2_Z32MSB	0x2c64	wo
DR3_Z32LSB	0x2c68	wo
DR3_Z32MSB	0x2c6c	wo

DR0		0x1cc0	wo
DR2		0x1cc8	wo
DR3		0x1ccc	wo
DR4		0x1cd0	wo
DR6		0x1cd8	wo
DR7		0x1cdc	wo
DR8		0x1ce0	wo
DR10		0x1ce8	wo
DR11		0x1cec	wo
DR12		0x1cf0	wo
DR14		0x1cf8	wo	
DR15		0x1cfc	wo

DSTORG		0x2cb8	wo
@VALUES DO_
	dstmap_		[0]{fb/sys}
	dstacc_		[1]{pci/agp}
	dstorg_		[3-31]
@ENDVALUES

# special case?
DWG_INDIR_WT	0x1e80	wo

DWGCTL		0x1c00	wo
@VALUES DC_
	opcod_		[0-3]
		+line_open	0
		+autoline_open	1
		+line_close	2
		+autoline_close	3
		+trap		4
		+texture_trap	6
		+bitblt		8
		+iload		9
	atype_		[4-6]
		+rpl		0
		+rstr		1
		+zi		3
		+blk		4
		+i		7
	linear_		[7]{xy/linear}
	zmode_		[8-10]
		+nozcmp		0
		+ze		2
		+zne		3
		+zlt		4
		+zlte		5
		+zgt		6
		+zgte		7
	solid_		[11]
	arzero_		[12]
	sgnzero_	[13]
	shftzero_	[14]
	bop_		[16-19]
	trans_		[20-23]
	bltmod_		[25-28]
		+bmonolef	0
		+bmonowf	4
		+bplan		1
		+bfcol		2
		+bu32bgr	3
		+bu32rgb	7
		+bu24bgr	11
		+bu24rgb	15
	pattern_	[29]
	transc_		[30]
	clipdis_	[31]
@ENDVALUES

DWGSYNC		0x2c4c	rw
@VALUES DS_
	dwgsyncaddr_	[2-31]
@ENDVALUES

FCOL		0x1c24	wo

FIFOSTATUS	0x1e10 	ro
@VALUES FS_
	fifocount_	[0-6]
	bfull_		[8]
	bempty_		[9]
@ENDVALUES

FOGCOL		0x1cf4	wo
FOGSTART	0x1cc4	wo
FOGXINC		0x1cd4	wo
FOGYINC		0x1ce4	wo

FXBNDRY		0x1c84	wo
@VALUES XA_
	fxleft_		[0-15]
	fxright_	[16-31]
@ENDVALUES

FXLEFT		0x1ca8	wo
FXRIGHT 	0x1cac	wo

ICLEAR		0x1e18	wo
@VALUES IC_
	softrapiclr_	[0]
	pickiclr_	[2]
	vlineiclr_	[5]
	wiclr_		[7]
	wciclr_		[8]	
@ENDVALUES

IEN		0x1e1c	rw
@VALUES IE_
	softrapien_	[0]
	pickien_	[2]
	vlineien_	[5]
	extien_		[6]
	wien_		[7]
	wcien_		[8]
@ENDVALUES

LEN		0x1c5c	wo

MACCESS		0x1c04	wo
@VALUES MA_
	pwidth_		[0-1]
		+8	0
		+16	1
		+32	2
		+24	3
	zwidth_         [3-4]
		+16     0
		+32     1
		+15     2
		+24     3
	memreset_	[15]
	fogen_		[26]
	tlutload_	[29]
	nodither_	[30]
	dit555_		[31]
@ENDVALUES
 
MCTLWTST	0x1c08 	wo
@VALUES MCWS_
	casltncy_	[0-2]
	rrddelay_	[4-5]
        rcddelay_	[7-8]
	rasmin_		[10-12]
	rpdelay_	[14-15]
	wrdelay_	[18-19]
	rddelay_	[21]
	smrdelay_	[23-24]
	bwcdelay_	[26-27]
	bpldelay_	[29-31]
@ENDVALUES

MEMRDBK		0x1e44	rw
@VALUES MRB_
	mclkbrd0_	[0-3]
	mclkbrd1_	[5-8]
	strmfctl_	[22-23]
	mrsopcod_	[25-28]
@ENDVALUES

OPMODE		0x1e54	rw
@VALUES OM_
	dmamod_		[2-3]
		+general	0
		+blit		1
		+vector		2
		+vertex		3
	dmadatasiz_	[8-9]
		+8		0
		+16		1
		+32		2
	dirdatasiz_	[16-17]
                +8              0
		+16             1
		+32             2
@ENDVALUES

PAT0		0x1c10	wo
PAT1		0x1c14	wo

PITCH		0x1c8c	wo
@VALUES P_
	iy_		[0-12]
	ylin_		[15]
@ENDVALUES

PLNWT		0x1c1c	wo

PRIMADDRESS	0x1e58	rw
@VALUES PDCA_
	primod_		[0-1]
		+general 	0
		+blit		1
		+vector		2
		+vertex		3
	primaddress_	[2-31]	
@ENDVALUES

PRIMEND 	0x1e5c	rw
@VALUES PDEA_
	primnostart_	[0]
	pagpxfer_	[1]
	primend_	[2-31]
@ENDVALUES

PRIMPTR		0x1e50	rw
@VALUES PLS_
	primptren0_	[0]
	primptren1_	[1]
	primptr_	[3-31]
@ENDVALUES

RST		0x1e40	rw
@VALUES R_
	softreset_	[0]
	softextrst_	[1]
@ENDVALUES

SECADDRESS	0x2c40	rw
@VALUES SDCA_
	secmod_		[0-1]
	       +general		0
	       +blit		1
	       +vector		2
	       +vertex		3
	secaddress_	[2-31]
@ENDVALUES

SECEND		0x2c44	rw
@VALUES SDEA_
	sagpxfer_	[1]
	secend_		[2-31]
@ENDVALUES

SETUPADDRESS	0x2cd0	rw
@VALUES SETDCA_
	setupmod_	[0-1]
		+vertlist	0
	setupaddress_	[2-31]
@ENDVALUES

SETUPEND	0x2cd4	rw
@VALUES SETDEA_
	setupagpxfer_	[1]
	setupend_	[2-31]
@ENDVALUES

SGN		0x1c58	wo
@VALUES S_
	sdydxl_		[0]{y/x}
	scanleft_	[0]
	sdxl_		[1]{pos/neg}
	sdy_		[2]*
	sdxr_		[5]*
	brkleft_	[8]
	errorinit_	[31]
@ENDVALUES

SHIFT		0x1c50	wo
@VALUES FSC_
	x_off_		[0-3]
	funcnt_		[0-6]
	y_off_		[4-6]
	funoff_		[16-21]
	stylelen_	[16-21]
@ENDVALUES

SOFTRAP 	0x2c48	rw
@VALUES STH_
	softraphand_	[2-31]
@ENDVALUES

SPECBSTART	0x2c98	rw
SPECBXINC	0x2c9c	rw
SPECBYINC	0x2ca0	rw
SPECGSTART	0x2c8c	rw
SPECGXINC	0x2c90	rw
SPECGYINC	0x2c94	rw
SPECRSTART 	0x2c80	rw
SPECRXINC	0x2c84	rw
SPECRYINC	0x2c88	rw

SRC0		0x1c30	wo
SRC1		0x1c34	wo
SRC2		0x1c38	wo
SRC3		0x1c3c	wo

SRCORG		0x2cb4	wo
@VALUES SO_
	srcmap_		[0]{fb/sys}
	srcacc_		[1]{pci/agp}
	srcorg_		[3-31]	
@ENDVALUES

STATUS		0x1e14	rw
@VALUES STAT_
	softrapen_	[0]
	pickpen_        [2]
	vsyncsts_	[3]
	vsyncpen_	[4]
	vlinepen_	[5]
	extpen_		[6]
	wpen_		[7]
	wcpen_		[8]
	dwgengsts_	[16]
	endprdmasts_	[17]
	wbusy_		[18]
	swflag_		[28-31]
@ENDVALUES

STENCIL		0x2cc8	wo
@VALUES S_
	sref_           [0-7]
	smsk_           [8-15]
	swtmsk_         [16-23]
@ENDVALUES

STENCILCTL	0x2ccc	wo
@VALUES SC_
	smode_		[0-2]
		+salways	0
		+snever		1
		+se		2
		+sne		3
		+slt		4
		+slte		5
		+sgt		6
		+sgte		7
	sfailop_	[3-5]
		+keep		0
		+zero		1
		+replace	2
		+incrsat	3
		+decrsat	4
		+invert		5
		+incr		6
		+decr		7
	szfailop_	[6-8]
		+keep		0
		+zero		1
		+replace	2
		+incrsat	3
		+decrsat	4
		+invert		5
		+incr		6
		+decr		7
	szpassop_	[9-11]
		+keep		0
		+zero		1
		+replace	2
		+incrsat	3
		+decrsat	4
		+invert		5
		+incr		6
		+decr		7
@ENDVALUES


TDUALSTAGE0	0x2cf8	wo
@VALUES TD0_
	color_arg2_	[0-1]
                +diffuse        0
                +specular       1
                +fcol           2
                +prevstage      3
	color_alpha_	[2-4]
                +diffuse        0
                +fcol           1
                +currtex        2
                +prevtex        3
                +prevstage      4
	color_arg1_replicatealpha_	[5]
	color_arg1_inv_ 	[6]
	color_arg2_replicatealpha_	[7]
	color_arg2_inv_ 	[8]
	color_alpha1inv_	[9]
	color_alpha2inv_	[10]
	color_arg1mul_ 	[11]
                +disable      0
                +alpha1       1
	color_arg2mul_ 	[12]
                +disable      0
                +alpha2       1
	color_arg1add_ 	[13]
                +disable      0
                +mulout       1
	color_arg2add_ 	[14]
                +disable      0
                +mulout       1
	color_modbright_	[15-16]
                +disable      0
                +2x           1
                +4x           2
	color_add_     	[17]
                +sub          0
                +add          1
	color_add2x_   	[18]
	color_addbias_ 	[19]
	color_blend_   	[20]
	color_sel_ 	[21-22]
		+arg1		0
		+arg2		1
		+add		2
		+mul		3	
        alpha_arg1_inv_ 	[23]
        alpha_arg2_ 	[24-25]
                +diffuse        0
                +fcol           1
                +prevtex        2
                +prevstage      3
        alpha_arg2_inv_ 	[26]
        alpha_add_ 	[27]
        alpha_addbias_ 	[28]
        alpha_add2x_ 	[29]
        alpha_modbright_	[28-29]
                +disable        0
                +2x             1
                +4x             2
	alpha_sel_ 	[30-31]
		+arg1		0
		+arg2		1
		+add		2
		+mul		3
@ENDVALUES	       


TDUALSTAGE1	0x2cfc	wo
@VALUES TD1_
	color_arg2_	[0-1]
                +diffuse        0
                +specular       1
                +fcol           2
                +prevstage      3
	color_alpha_	[2-4]
                +diffuse        0
                +fcol           1
                +tex0           2
                +prevtex        3
                +prevstage      4
	color_arg1_replicatealpha_	[5]
	color_arg1_inv_ 	[6]
	color_arg2_replicatealpha_	[7]
	color_arg2_inv_ 	[8]
	color_alpha1inv_	[9]
	color_alpha2inv_	[10]
	color_arg1mul_ 	[11]
                +disable      0
                +alpha1       1
	color_arg2mul_ 	[12]
                +disable      0
                +alpha2       1
	color_arg1add_ 	[13]
                +disable      0
                +mulout       1
	color_arg2add_ 	[14]
                +disable      0
                +mulout       1
	color_modbright_	[15-16]
                +disable      0
                +2x           1
                +4x           2
	color_add_     	[17]
                +sub          0
                +add          1
	color_add2x_   	[18]
	color_addbias_ 	[19]
	color_blend_   	[20]
	color_sel_ 	[21-22]
		+arg1		0
		+arg2		1
		+add		2
		+mul		3	
        alpha_arg1_inv_ 	[23]
        alpha_arg2_ 	[24-25]
                +diffuse        0
                +fcol           1
                +prevtex        2
                +prevstage      3
        alpha_arg2_inv_ 	[26]
        alpha_add_ 	[27]
        alpha_addbias_ 	[28]
        alpha_add2x_ 	[29]
        alpha_modbright_	[28-29]
                +disable        0
                +2x             1
                +4x             2
	alpha_sel_ 	[30-31]
		+arg1		0
		+arg2		1
		+add		2
		+mul		3
@ENDVALUES	       

TEST0		0x1e48	rw
@VALUES TST_
	ramtsten_	[0]
	ramtstdone_	[1]
	wramtstpass_	[2]
	tcachetstpass_	[3]
	tluttstpass_	[4]
	luttstpass_	[5]
	besramtstpass_	[6]
	ringen_		[8]
	apllbyp_	[9]
	hiten_		[10]
	tmode_		[11-13]
	tclksel_	[14-16]
	ringcnten_	[17]
	ringcnt_	[18-29]
	ringcntclksl_	[30]
	biosboot_	[31]
@ENDVALUES

TEXBORDERCOL	0x2c5c	wo

TEXCTL		0x2c30	wo
@VALUES TMC_
	tformat_	[0-3]
		+tw4	0
		+tw8	1
		+tw15	2
		+tw16	3
		+tw12	4
		+tw32	6
		+tw422	10
	tpitchlin_	[8]
	tpitchext_	[9-19]
	tpitch_		[16-18]
	owalpha_	[22]
	azeroextend_	[23]
	decalckey_	[24]
	takey_		[25]{0/1}
	tamask_		[26]{0/1}
	clampv_		[27]
	clampu_		[28]
	tmodulate_	[29]
	strans_		[30]
	itrans_		[31]
@ENDVALUES

TEXCTL2		0x2c3c	wo
@VALUES TMC_
	decalblend_	[0]
	idecal_		[1]
	decaldis_	[2]
	ckstransdis_	[4]
	borderen_	[5]
	specen_		[6]
	dualtex_	[7]
	tablefog_	[8]
	bumpmap_	[9]
	map1_		[31]
@ENDVALUES

TEXFILTER	0x2c58	wo
@VALUES TF_
	minfilter_	[0-3]
		+nrst	0
		+bilin	2
		+cnst	3
		+mm1s	8
		+mm2s	9
		+mm4s	10
		+mm8s	12
	magfilter_	[4-7]
		+nrst	0
		+bilin	2
		+cnst	3
	avgstride_	[19]
	filteralpha_	[20]
	fthres_		[21-28]
	mapnb_		[29-31]
@ENDVALUES

TEXHEIGHT	0x2c2c	wo
@VALUES TH_
	th_		[0-5]
	rfh_		[9-14]
	thmask_		[18-28]
@ENDVALUES

TEXORG		0x2c24	wo
@VALUES TO_
	texorgmap_	[0]{fb/sys}
	texorgacc_	[1]{pci/agp}
	texorg_		[5-31]
@ENDVALUES

TEXORG1		0x2ca4	wo
TEXORG2		0x2ca8	wo
TEXORG3		0x2cac	wo
TEXORG4		0x2cb0	wo

TEXTRANS	0x2c34	wo
@VALUES TT_
	tckey_		[0-15]
	tkmask_		[16-31]
@ENDVALUES

TEXTRANSHIGH	0x2c38	wo
@VALUES TT_
        tckeyh_         [0-15]
        tkmaskh_	[16-31]
@ENDVALUES
		
TEXWIDTH	0x2c28	wo
@VALUES TW_
	tw_		[0-5]
	rfw_		[9-14]
	twmask_		[18-28]
@ENDVALUES

TMR0		0x2c00	wo
TMR1		0x2c04	wo
TMR2		0x2c08	wo
TMR3		0x2c0c	wo
TMR4		0x2c10	wo
TMR5		0x2c14	wo
TMR6		0x2c18	wo
TMR7		0x2c1c	wo
TMR8		0x2c20	wo

VBIADDR0	0x3e08	wo
VBIADDR1	0x3e0c	wo

VCOUNT		0x1e20	ro

# --- We dont need this, its for video input ---
#VICLEAR	0x3e34	wo
#VIEN		0x3e38	rw
#VINADDR0	0x3e10	wo
#VINADDR0	0x3e14	wo
#VINCTL		0x3e1c	wo
#VINCTL0	0x3e00	wo
#VINCTL1	0x3e04	wo
#VINNEXTWIN	0x3e18	wo
#VSTATUS	0x3e30	ro
# ---------------------------------------------------

WACCEPTSEQ      0x1dd4  wo
@VALUES WAS_
        seqdst0_        [0-5]
        seqdst1_        [6-11]
        seqdst2_        [12-17]
        seqdst3_        [18-23]
        seqlen_         [24-25]
        wfirsttag_      [26]
        wsametag_       [27]
        seqoff_         [28]
@ENDVALUES


WCODEADDR	0x1e6c	ro
@VALUES WMA_
	wcodeaddr_	[8-31]
@ENDVALUES

WFLAG		0x1dc4	wo
@VALUES WF_
	walustsflag_	[0-7]
	walucfgflag_	[8-15]
	wprgflag_	[16-31]
@ENDVALUES

WFLAG1          0x1de0  wo
@VALUES WF1_
        walustsflag1_   [0-7]
        walucfgflag1_   [8-15]
        wprgflag1_      [16-31]
@ENDVALUES

WFLAGNB		0x1e64	rw
WFLAGNB1        0x1e08  rw

WGETMSB		0x1dc8	wo
@VALUES WGV_
	wgetmsbmin_	[0-4]
	wgetmsbmax_	[8-12]
        wbrklefttop_    [16]
        wfastcrop_      [17]
        wcentersnap_    [18]
        wbrkrighttop_   [19]
@ENDVALUES

WIADDR		0x1dc0	wo
@VALUES WIA_
	wmode_		[0-1]
		+suspend	0
		+resume		1
		+jump		2
		+start		3
	wagp_		[2]{pci/agp}
	wiaddr_		[3-31]
@ENDVALUES

WIADDR2         0x1dd8  wo
@VALUES WIA2_
        wmode_          [0-1]
                +suspend        0
                +resume         1
                +jump           2
                +start          3
        wagp_           [2]{pci/agp}
        wiaddr_         [3-31]
@ENDVALUES

WIADDRNB	0x1e60	rw
WIADDRNB1       0x1e04  rw
WIADDRNB2       0x1e00  rw

WIMEMADDR	0x1e68	wo
@VALUES WIMA_
        wimemaddr_      [0-7]
@ENDVALUES

WIMEMDATA	0x2000	rw
WIMEMDATA1      0x2100  rw

WMISC		0x1e70	rw
@VALUES WM_
	wucodecache_	[0]
	wmaster_	[1]
	wcacheflush_	[3]
@ENDVALUES

WR              0x2d00  wo

WVRTXSZ		0x1dcc	wo
@VALUES WVS_
        wvrtxsz_        [0-5]
        primsz_         [8-13]
@ENDVALUES



XDST		0x1cb0	wo

XYEND		0x1c44	wo
@VALUES XYEA_
       	x_end_		[0-15]
	y_end_		[16-31]
@ENDVALUES

XYSTRT 		0x1c40	wo
@VALUES XYSA_
	x_start_	[0-15]
	y_start_	[16-31]
@ENDVALUES

YBOT		0x1c9c	wo

YDST		0x1c90	wo
@VALUES YA_
	ydst_		[0-22]
	sellin_		[29-31]
@ENDVALUES

YDSTLEN		0x1c88	wo
@VALUES YDL_
	length_		[0-15]
	yval_		[16-31]
@ENDVALUES

YDSTORG		0x1c94	wo
YTOP		0x1c98	wo

ZORG		0x1c0c	wo
@VALUES ZO_
	zorgmap_	[0]{fb/sys}
	zorgacc_	[1]{pci/agp}
	zorg_		[2-31]
@ENDVALUES


@END

###############################################################################
EOF

sub myprint;

$curr = `pwd`;
chomp $curr;

open OUT, ">$curr/mgaregs.h";

myprint $code_start;
myprint "/*************** (START) AUTOMATICLY GENERATED REGISTER FILE *****************/";
myprint "\n/*\n * Generated on ", `date`, " */\n\n";

for $parse (0,1) {

    $mode=0;

    my @lines=split(/\n/, $text);

    MAIN_LOOP:
    for (my $line_index=0; $line_index < $#lines; $line_index++) {
	$line = $lines[$line_index];

	$line =~ s/^\s*//;
	$line =~ s/\s*$//;
	$line =~ s/#.*//;
	$line || next MAIN_LOOP;

	if ($line =~ /\@BCOMM\s+(.*)/) {
		$comm = $1;
		$parse || myprint "\n/*\n * ", $comm, "\n */\n\n";
		next;
	}

	if ($line =~ /\@LCOMM\s+(.*)/) {
		$comm = $1;
		$parse || myprint "\n/* ", $comm, " */\n\n";
		next;
	}

	if ($line =~ /\@PCI/) {
		$name = "PCI Config Region";
		$parse || do {  myprint "\n"; $mode = 2; };
		next;
	}

	if ($line =~ /\@REGION0/) {
		$name = "Region 0 Registers";
		$parse || do {  myprint "\n"; $mode = 3; };
		next;
	}

	if ($line =~ /\@VALUES\s+(\w+)/) {
		$vpref = $1;
		$lastmode = $mode;
		$mode = 0;
		$parse || do { myprint "\n"; $mode = 4; };
		($strunset, $strset) = ("disable", "enable");
		$single_bit = 0;
		next;
	}

	if ($line =~ /\@ENDVALUES/) {
		$name = "";
		if( $single_bit ) {
		    val_formats("$vpref$lastval$l_strunset", "0x0", "");
		    val_formats("$vpref$lastval$l_strset", $bvset, "");
		}
		$parse || myprint "\n";
		$mode = $lastmode;
		next;
	}

	if ($line =~ /\@END/) {
		$name = "";
		$parse || myprint "\n\n\n";
		$mode = 0;
		next;
	}

	PCI_MODE: { $mode == 2 || last PCI_MODE;
	    ($reg, $offset, $access) = split ' ', $line;
	    reg_simplef($reg, $offset);
	    next MAIN_LOOP;
	}

	REGION0_MODE: { $mode == 3 || last REGION0_MODE;
	    ($reg, $offset, $access) = split ' ', $line;
	    reg_simplef($reg, $offset);
	    next MAIN_LOOP;
	}

	VALUES_MODE: { $mode == 4 || last VALUES_MODE;
	    if( $line =~ /(\w+)\s+\[\s*([0-9- ]+)\s*\]/ ) {
	        ($val, $bit) = ($1, $2);
		if( $line !~ /\]\s*\*/ ) {
		    ($strunset, $strset) = ("disable", "enable");
		}
		if( $line =~ /\]\s*\{\s*(\w+)\s*\/(\w+)\s*\}/ ) {
		    ($strunset, $strset) = ($1, $2);
		}
		if( $single_bit ) {
		    val_formats("$vpref$lastval$l_strunset", "0x0", "");
		    val_formats("$vpref$lastval$l_strset", $bvset, "");
		}

		if( $bit =~ /(\d+)\s*-\s*(\d+)/ ) {
		    ($bfrom, $bto) = ($1, $2);
		    $single_bit = 0;
		}
		elsif ( $bit =~ /(\d+)/ ) {
		    $bfrom = $bto = $1;
		    ($lastval,$l_strunset,$l_strset)=($val,$strunset,$strset);
		    $single_bit = 1;
		}
		else {
		    print("/* BIT RANGE SYNTAX ERROR ENCOUNTERD! */\n");
		    myprint("/* BIT RANGE SYNTAX ERROR ENCOUNTERD! */\n");
		}

		($bvmask, $bvset) = get_bits_value($bfrom,$bto);
		$comment = "/* bits $bfrom-$bto */";
		if($bfrom == $bto) { $comment = "/* bit $bfrom */"; }
		val_formats("$vpref${val}MASK", $bvmask, $comment);
		if (($bto - $bfrom > 1 || $bto eq '') && $lines[$line_index+1] !~ /\s*\+(\w+)\s+(\d+)/) {
			val_formats("$vpref${val}SHIFT", $bfrom, "");
		}
	    }
	    elsif($line =~ /\s*\+(\w+)\s+(\d+)/) {
		($vsuf, $rv) = ($1, $2);
		$single_bit = 0;
		$srv = shift_value_nrbits($rv, $bfrom);
		$comment = "/* val $rv, shift $bfrom */";
		val_formats("$vpref$val$vsuf", $srv, $comment);
	    }
	    else {
		print("/* VALUE LINE SYNTAX ERROR ENCOUNTERD! */\n");
		myprint("/* VALUE LINE SYNTAX ERROR ENCOUNTERD! */\n");
	    }
	    next MAIN_LOOP;
	}
    }
}

myprint "/**************** (END) AUTOMATICLY GENERATED REGISTER FILE ******************/\n";
myprint $code_end;

close OUT;



sub shift_value_nrbits
{
	my ($val, $nrbits) = @_;
	my ($ret);

	$ret = sprintf "0x%x", ($val << $nrbits);

	return $ret;
}



sub get_bits_value
{
	my ($start, $end) = @_;
	my ($cnt, $bmask, $bset);

	$bset = 0;
	for $cnt ($start..$end) {
	    $bset |= (1 << $cnt);
	}
	$bmask = sprintf "0x%x", ~$bset;
	$bset = sprintf "0x%x", $bset;

	return ($bmask,$bset);
}



sub val_formats
{
	my ($reg, $rval, $comm) = @_;
	my ($cs, $len, $rtab, $vtab);
	
	$len = length($reg);
	$rtab =" "."\t"x(5 - int(($len + 13)/8) );

	$len = length($rval);
	$vtab =" "."\t"x(2 - int(($len + 1)/8) );

	myprint "    #define ", $reg, $rtab, $rval, $vtab, "$comm\n";
}



sub reg_simplef
{
	my ($reg, $offset) = @_;
	my ($len, $tab);
	
	$len = length($reg);
	$tab =" "."\t"x(5 - int(($len + 9)/8) );

	myprint "#define MGAREG_", $reg, $tab, $offset;
	myprint "\n";
}




sub myprint { foreach $txt (@_) { print OUT $txt; } }
