/*
 * GLX Hardware Device Driver for Matrox G200/G400
 * Copyright (C) 1999 Wittawat Yamwong
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR 
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE 
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 *
 *    Wittawat Yamwong <Wittawat.Yamwong@stud.uni-hannover.de>
 */

/* $Id: mgaglx.c,v 1.21 1999/12/12 20:37:13 keithw Exp $ */

#include <stdlib.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "context.h"
#include "depth.h"
#include "macros.h"
#include "texstate.h"
#include "triangle.h"
#include "vb.h"
#include "types.h"

#include "xsmesaP.h"
#include "glx_log.h"
#include "glx_config.h"
#include "mesaglx/context.h"
#include "mesaglx/matrix.h"
#include "mesaglx/types.h"

#define GC XXGC
#include "gcstruct.h"
#include "pixmapstr.h"
#include "servermd.h" /* PixmapBytePad */
#include "scrnintstr.h"
#include "regionstr.h"
#include "windowstr.h"
#undef GC

#include "vga.h"
#include "vgaPCI.h"
#include "mga.h"
#include "xaa/xf86xaa.h"

#include "mm.h"
#include "mgabuf.h"
#include "mgadd.h"
#include "g200_mac.h"
#include "mgalib.h"
#include "mgalog.h"
#include "mgadirect.h"
#include "mgapipeline.h"
#include "mgarender.h"
#include "mgatris.h"
#include "mgadepth.h"

extern XSMesaContext XSMesa;

/* our collected globals */
mgaGlx_t	mgaglx;

void mgaServerSwapBuffers( XSMesaBuffer b );
void (*mgaGLXSwapBuffers)(XSMesaBuffer b) = mgaServerSwapBuffers;


/* are these for the multi-hardware glx support? */
GLboolean (*hwInitGLX)() = mgaInitGLX;
GLboolean (*hwInitVisuals)(VisualPtr *, DepthPtr *, int *, int *, 
                           int *, VisualID *, unsigned long, int) = NULL;



mgaBufferPtr mgaDB = NULL;
mgaBufferPtr mgaFrontBuffer = NULL;
mgaContextPtr mgaCtx = NULL;

int	usec( void ) {
	struct timeval tv;
	struct timezone tz;
	
	gettimeofday( &tv, &tz );
	
	return (tv.tv_sec & 2047) * 1000000 + tv.tv_usec;
}


/*
 * mgaDumpDB
 */
void mgaDumpDB(mgaBufferPtr buf)
{
/*  int i;  */

  mmDumpMemInfo( cardHeap );
  mgaMsg(1,"Dump DB:\n");
  mgaMsg(1,"  %s %s\n",(buf->Drawable) ? "Drawable" : "-",
         (buf->HasZORG) ? "HasZORG" : "-");
  /*
  for (i=0; i < sizeof(buf->Setup)/sizeof(mgaUI32); i++)
    mgaMsg(1,"  %d: 0x%08x\n",i,buf->Setup[i]);
  */
  mgaMsg(1,"  w,h,p = %d,%d,%d\n",buf->Width,buf->Height,buf->Pitch);
  mgaMsg(1,"End Dump DB\n");
}

/*
 * mgaGLXCreateDepthBuffer
 */
void mgaGLXCreateDepthBuffer(GLcontext* ctx)
{
  XSMesaContext xsmesa = (XSMesaContext) ctx->DriverCtx;
  mgaBufferPtr buf,zb=0;

  mgaMsg( 1, "mgaGLXCreateDepthBuffer\n" );
	
  /* deallocate current depth buffer if present */
  if (ctx->Buffer->Depth) {
      free(ctx->Buffer->Depth);
      ctx->Buffer->Depth = NULL;
    }
    
  /* allocate new depth buffer, but don't initialize it */
  /* FIXME: we shouldn't need to do this if we got a hardware buffer */
  ctx->Buffer->Depth = (GLdepth *) malloc(ctx->Buffer->Width
					  * ctx->Buffer->Height
					  * sizeof(GLdepth));
  if (!ctx->Buffer->Depth) {
      /* out of memory */
      ctx->Depth.Test = GL_FALSE;
      mgaError("Couldn't allocate depth buffer\n" );
    }
  
  if (xsmesa->xsm_buffer->db_state != BACK_XIMAGE ) {
    	return;
  }
  
  buf = (mgaBufferPtr)xsmesa->xsm_buffer->backimage->devPriv;
  if ( !buf ) {
        mgaMsg(5, "Where is my backbuffer?\n");
  	return;		/* should never happen */
  }
#if DEPTH_BITS==16
    zb = mgaAttachZBuffer(buf,NULL,MGA_ZBUFFER16);
#elif DEPTH_BITS==32
    zb = mgaAttachZBuffer(buf,NULL,MGA_ZBUFFER32);
#endif

  if (!zb) {
      /* We need a depth buffer, but cannot allocate it. So, disable hardware
	 acceleration. */
      buf->Drawable = 0;
      mgaError("mgaGLXCreateDepthBuffer(): mgaAttachZBuffer() failed!\n");
  }
  
  mgaDDInitDepthFuncs( ctx );	    
  
  if ( mgaglx.logLevel >= 1 )  {
    	mgaDumpDB(buf);
  }
}


/*
 * mgaGLXDestroyImage
 * Free all back/depth buffers for this image
 */
void mgaGLXDestroyImage(GLXImage* image)
{	 
  mgaMsg(10, "mgaGLXDestroyImage\n");
  if (image->devPriv) {
    if (image->devPriv == mgaDB) { 
      mgaDB = NULL;
      /* unset all driver callbacks immidiatly! */
      if (mgaCtx) mgaDisableHW(mgaCtx->gl_ctx);
    }
    mgaDestroyBuffer((mgaBufferPtr)image->devPriv);
     if ( mgaglx.logLevel >= 10 )  {
       mmDumpMemInfo( cardHeap );
     }
  } else if (image->data)
    free(image->data);
  xfree(image);
}

/*
 * mgaGLXCreateImage
 * This is the back buffer for rendering into
 */
GLXImage* mgaGLXCreateImage(WindowPtr pwindow, int depth, int width, int height, GLXImage* old_image)
{
  int Attrib, is_mgaDB;
  mgaBufferPtr buf;
  GLXImage* image;
  mgaMsg(10, "mgaGLXCreateImage\n");
  
  if (old_image)
  {
    is_mgaDB = (old_image->devPriv == mgaDB);
    mgaGLXDestroyImage(old_image);
  }
  else
    is_mgaDB = 0;

  image=(GLXImage*)xalloc(sizeof(GLXImage));
  if (!image)
    return (NULL);
  image->pwin = pwindow;
  image->width=width;
  image->height=height;
  image->data=0;
  image->devPriv=NULL;

  Attrib = -1;
  switch (depth)
    {
    case 8:
      Attrib = MGA_PF_INDEX;
     image->bits_per_pixel=8;
      break;
    case 15:
      Attrib = MGA_PF_555;
     image->bits_per_pixel=16;
      break;
    case 16:
      Attrib = MGA_PF_565;
     image->bits_per_pixel=16;
      break;
    case 24:	/* we can't render into a 24 bit image, so use a 32 bit one */
    case 32:
      Attrib = MGA_PF_8888;
     image->bits_per_pixel=32;
      break;
    default:
      mgaError("Unknown width in GLXCreateImage\n");
      break;
    }
  if (Attrib != -1)
    buf = mgaCreateBuffer(Attrib | MGA_COLORBUFFER /* | MGA_FORCE_SYSMEM */,0,width,height,0);
  else
    buf = NULL;
  image->devPriv = (void *) buf;
  if (!image->devPriv) {
    image->bytes_per_line=PixmapBytePad(width, depth);
    image->data = (char *) malloc( image->height * image->bytes_per_line );
    if (!image->data)
      {
	mgaError("mgaGLXCreateImage: malloc failed.");
	xfree(image);
	image = NULL;
      }
  } else {
    image->bytes_per_line=buf->Pitch * buf->BytesPerPixel;
    /* How can I draw XImage with any line pad? */
    /* e.g. bytes_per_line != PixmapBytePad */
    /* My work around: adjust width instead */
    image->width = buf->Pitch;
    image->data = buf->BufAddr;
    if ( mgaglx.logLevel >= 1 )  {
       mmDumpMemInfo( cardHeap );
     }
     if (is_mgaDB) mgaDB = buf;
  }

  
  if (is_mgaDB) 
     fprintf(stderr, "is_mgaDB");

  if (is_mgaDB && mgaCtx)
     mga_setup_DD_pointers( mgaCtx->gl_ctx );
     

  return (image);
}



/*
 * DoMakeCurrent
 * Changing either the context or the buffer will force this.
 */
static int DoMakeCurrent(XSMesaContext c, XSMesaBuffer b) {
	mgaContextPtr ctx, oldctx = mgaCtx;
	mgaBufferPtr  buf, oldbuf = mgaDB;
  
	mgaMsg( 10, "DoMakeCurrent( %p, %p )\n", c, b );

	mgaCtx = NULL;
	mgaDB = NULL;
	
	if ( c ) {
		ctx = c->hw_ctx;
		if (!VALID_MGA_CONTEXT(ctx)) {
			return -1;
		}
	} else {
		ctx = NULL;
	}
	
	if ( b && b->buffer == XIMAGE ) {
		buf = (mgaBufferPtr) b->backimage->devPriv;
		if (!buf || buf->magic != mgaBufferMagic) {
			return -1;
		}
		if (!buf->Drawable) {
			buf = NULL;
		}
	} else {
		buf = NULL;
	}

	if ( !ctx && buf ) {
		return -1;
	}
	if (!ctx) {
		return 0;
	}
	
	/* Need to push context information into buf->Setup values...
	 */
	ctx->new_state = ~0;
	ctx->DB = buf;

	/* valid conditions to render with */
	mgaCtx = ctx;
	mgaDB = ctx->DB;
	mgaMsg( 10, "DoMakeCurrent end\n");

	return 0;
}


/*
 * mgaGLXMakeCurrent
 * Note that GLX calls this every batch of commands,
 * so it will show up several times a frame even if
 * the client application isn't issuing makeCurrent calls.
 */ 
GLboolean mgaGLXMakeCurrent( XSMesaContext c ) {
	if (c == XSMesa) {
		return (GL_TRUE);
	}
	
	/* mesa 3.1 will sometimes leave unflushed vertexes */
	if ( XSMesa ) {
		glFlush();
	}

 	mgaMsg( 10, "mgaGLXMakeCurrent( %p )\n", c );
	XSMesa = c;                                                                 
	if (c) {
		DoMakeCurrent(c,c->xsm_buffer);
		gl_make_current(c->gl_ctx, 
		  c->xsm_buffer ? c->xsm_buffer->gl_buffer : NULL);
	} else {
	  	/* detach */
		DoMakeCurrent(NULL,NULL);
		gl_make_current(NULL, NULL);
	}
	return(GL_TRUE); 
}

/*
 * Bind buffer b to context c and make c the current rendering context.
 */
GLboolean mgaGLXBindBuffer( XSMesaContext c, XSMesaBuffer b ) {
	mgaMsg( 10, "mgaGLXBindBuffer( %p, %p )\n", c, b );
	DoMakeCurrent(c,b);
  {
    GLboolean r =XSMesaBindBuffer(c,b);
    mgaMsg( 10, "XSMesaBindBuffer= %d\n", r);
    return r;
  }
  
}



mgaContextPtr mgaCreateContext(GLcontext *ctx)
{
  mgaContextPtr c;

  c = (mgaContextPtr) calloc(1,sizeof(mgaContext));
  if (!c)
    return NULL;

  c->magic = mgaContextMagic;
  c->gl_ctx = ctx;
  c->regDWGCTL = ( DC_clipdis_disable | (0xC << DC_bop_SHIFT) |
		   DC_shftzero_enable | DC_zmode_nozcmp | 
		   DC_atype_zi );

  c->regALPHACTRL = AC_src_one | AC_dst_zero | AC_amode_FCOL | 
    AC_astipple_disable | AC_aten_disable | AC_atmode_noacmp |
    AC_alphasel_fromtex;

  c->renderindex = -1;		/* impossible value */
  c->new_state = ~0;
  c->reg_dirty = ~0;

  c->regFOGCOLOR = MGAPACKCOLOR888((mgaUI8)(ctx->Fog.Color[0]*255.0F), 
				   (mgaUI8)(ctx->Fog.Color[1]*255.0F), 
				   (mgaUI8)(ctx->Fog.Color[2]*255.0F));


  /* XXX Fix me: callback not registered when main VB is created.
   */
  if (ctx->VB) 
     mgaDDRegisterVB( ctx->VB );
  else
     fprintf(stderr, "**** Didn't create vb driver data\n");

  if (ctx->NrPipelineStages)
     ctx->NrPipelineStages = mgaDDRegisterPipelineStages(ctx->PipelineStage,
							 ctx->PipelineStage,
							 ctx->NrPipelineStages);

  if (!glx_getint("mga_no_fast_path")) {  	
     ctx->Driver.BuildPrecalcPipeline = mgaDDBuildPrecalcPipeline;
  } else {
     mgaMsg( 1, "enabling mga_no_fast_path\n" );
  }

  /* Ask mesa to clip fog coordinates for us.
   */
  ctx->TriangleCaps |= DD_CLIP_FOG_COORD;

  mgaDDRenderInit();
  mgaDDTrifuncInit();
  mgaDDFastPathInit();
  mgaDDSetupInit();
  mgaDDExtensionsInit( ctx );

  mgaMsg(2,"mgaCreateContext(): successful.\n");
  return c;
}


/*
 * Create a new XSMesaContext.
 * Input:  v - XSMesaVisual
 *         share_list - another XSMesaContext with which to share display
 *                      lists or NULL if no sharing is wanted.
 * Return:  an XSMesaContext or NULL if error.
 */
XSMesaContext mgaGLXCreateContext( XSMesaVisual v,  
                                   XSMesaContext share_list )
{
	XSMesaContext c;
	GLcontext *ctx;

    mgaMsg (0, "### Creating new xsmesa context for MGA ...\n" );

    c = (XSMesaContext) calloc( 1, sizeof(struct xsmesa_context) );
    if (!c)
    {
        return(NULL);
    }

    ctx = c->gl_ctx = gl_create_context(v->gl_visual,    
				  share_list ? share_list->gl_ctx : NULL,
				  (void *) c, GL_TRUE /* direct rendering */);
    if (!ctx)
    {
        free(c);
        return(NULL);
    }

    c->xsm_visual = v;
    c->xsm_buffer = NULL;       /* set later by XSMesaMakeCurrent */
    c->pixelformat = v->dithered_pf;    /* Dithering is enabled by default */
  
    if (v->gl_visual->DBflag)
    {
      	/* only create HW context when we have a double buffered visual */
	    c->hw_ctx = (void *)mgaCreateContext(c->gl_ctx);
    
	    if (!c->hw_ctx) {
	      mgaError("Cannot create hardware specific context.\n");
	      return (NULL);
	    }
    }
  
    c->gl_ctx->Driver.UpdateState = mga_setup_DD_pointers;

    mgaMsg( 1, "mgaGLXCreateContext succeeded: %p\n", c );

    return(c);
}


/*
 * mgaGLXDestroyContext
 */
void mgaGLXDestroyContext( XSMesaContext c ) {
	mgaContextPtr ctx;
	
	mgaMsg( 1, "mgaGLXDestroyContext( %p )\n", c );
	
	/* make sure all drawing is completed */
	mgaWaitDrawingEngine();

	ctx = (mgaContextPtr)c->hw_ctx;

	/* if this is our current context, make sure we clear it */
	if ( ctx == mgaCtx ) {
		mgaCtx = NULL;
	}
	if ( c == XSMesa ) {
		XSMesa = 0;
	}

	if ( ctx ) {
		if (ctx->magic != mgaContextMagic) {
			mgaError("mgaDestroyContext(): ctx->magic != mgaContextMagic\n");
		} else {
			/* FIXME: is this needed?  does mesa destroy them all for us? */
			mgaDestroyContextTextures( ctx );
      
			ctx->magic = 0;
			free(ctx);
			mgaMsg( 1,"mgaDestroyContext(): successfully destroyed.\n" );
		}
	}
			
	XSMesaDestroyContext(c);
}
				       
