/* -*- mode: C; c-basic-offset:8 -*- */

/*
 * GLX Hardware Device Driver for Matrox G200/G400
 * Copyright (C) 1999 Jeff Hartmann
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * JEFF HARTMANN, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR 
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE 
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 *
 * original by Jeff Hartmann <slicer@ionet.net>
 * 6/16/99: rewrite by John Carmack <johnc@idsoftware.com>
 */

/*

This file is only entered at startup.  After mgaGlxInit completes,
nothing here will be executed again.

*/

#include <stdlib.h>
#include <errno.h>
#include <unistd.h>
#include <sys/mman.h>
#include <stdio.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <signal.h>
#include <unistd.h>

#include "context.h"
#include "depth.h"
#include "macros.h"
#include "texstate.h"
#include "triangle.h"
#include "vb.h"
#include "types.h"

#include "xsmesaP.h"
#include "glx_log.h"
#include "glx_config.h"

#include "mesaglx/context.h"
#include "mesaglx/matrix.h"
#include "mesaglx/types.h"

#define GC XXGC
#include "gcstruct.h"
#include "pixmapstr.h"
#include "servermd.h" /* PixmapBytePad */
#include "scrnintstr.h"
#include "regionstr.h"
#include "windowstr.h"
#undef GC

#include "vga.h"
#include "vgaPCI.h"
#include "mga.h"		       
#include "xaa/xf86xaa.h"

#include "mm.h"
#include "mgabuf.h"
#include "mgadd.h"
#include "g200_mac.h"
#include "mgalib.h"
#include "mgalog.h"
#include "mgadirect.h"
#include "mgawarp.h"
#include "mgaglx.h"

#ifndef NO_MTRR

#define MTRR_NEED_STRINGS
#include <errno.h>
#include <asm/mtrr.h>
#include <sys/ioctl.h>

static int mtrr;

#endif


#ifndef NO_AGPGART

/* AGP kernel module interface */
#include <linux/agpgart.h>
#include <sys/ioctl.h>

#endif


#if defined(USE_X86_ASM)
#include "X86/common_x86asm.h"
#endif


/*
  Configuration values from config file:

mga_dma=0	: virtual buffer, pseudo dma
mga_dma=1	: physical buffer, pseudo dma
mga_dma=2	: physical buffer, real dma
mga_dma=3	: physical buffer, real dma, async

mga_dmaadr=92	: put the physical buffer at offset 92 * 0x100000 
			: use GLX_MGA_DMAADR=AGP to use agp memory
mga_dmasize=4	: use 4 * 0x100000 bytes of memory for the buffers

mga_cmdsize=4	: use 4 * 0x100000 bytes for commands, the rest for textures

mga_cardcmds=1	: put command buffers on card memory ( DOESN'T WORK! )
mga_systemtexture = 1	: put textures in main pci/agp memory instead of on card

*/

memHeap_t	*cardHeap;
mgaUI32		cardPhysical;
unsigned char	*cardVirtual;

memHeap_t	*textureHeap;
mgaUI32		textureHeapPhysical;	/* 0 if we aren't using system memory texturing */
unsigned char	*textureHeapVirtual;	/* correct for either local or PCI heaps */

memHeap_t	*sysmemHeap;		/* physical memory, if available */
mgaUI32		sysmemBytes;		/* size of memory block */
mgaUI32		sysmemPhysical;		/* 0 if we don't have a physical mapping */
unsigned char	*sysmemVirtual;

mgaUI32	*pseudoDmaVirtual;		/* our local mapping of pdma window */

int use_agp; /* will be set to "PDEA_pagpxfer_enable" if enabled or 0 if not */

#ifndef NO_AGPGART
int    gartfd;
void   *gartbuf;
struct gart_info gartinf;
#endif

/* private vars */
mgaDma_buffer	*dmaBuffers[2];


static	mgaUI32	bufferBytes;			/* size of buffer */
static	mgaUI32	bufferPhysical;		/* 0 if we don't have a physical mapping */
static	unsigned char	*bufferVirtual;

void mgaDmaResetBuffer( void );
void MgaSetSyncBusy( void );


/*
 * MapPseudoDmaWindow
 */
static void MapPseudoDmaWindow( void ) {
	mgaUI32	pseudoDmaPhysical;	

	pseudoDmaPhysical =  (pcibusRead(MGAPciTag, 0x00000018)) & 0xff800000;
	pseudoDmaVirtual = 
		 xf86MapVidMem(vga256InfoRec.scrnIndex, EXTENDED_REGION,
			   (pointer) ((unsigned long) pseudoDmaPhysical),
			   0x800000);
	mgaMsg( 1, "pseudoDmaPhysical : %p\n", pseudoDmaPhysical );
	mgaMsg( 1, "pseudoDmaVirtual : %p\n", pseudoDmaVirtual );
}

static int IsPowerOfTwo( int val ) {
	int		i;
	
	for ( i = 0 ; i < 32 ; i++ ) {
		if ( val == ( 1 << i ) ) {
			return 1;
		}
	}
	return 0;
}


static void MgaCloseMTRR( void ) {
   #ifndef NO_MTRR
	close(mtrr);
   #endif
}

static void MgaOpenMTRR( void ) {
#ifndef NO_MTRR
	if ((mtrr = open("/proc/mtrr", O_WRONLY, 0)) == -1) {
		if (errno == ENOENT) {
			  mgaError("/proc/mtrr not found: MTRR not enabled\n");
		}  else {
			  mgaError("Error opening /proc/mtrr: %s\n",strerror(errno));
			  mgaError("MTRR not enabled\n");
		}
		return;
	}
	atexit( MgaCloseMTRR );
#endif
}


int CoverRangeWithMTRR( int base, int range, int type ) {
	int		count;
	int		size;
	
	count = 0;
#ifndef NO_MTRR
	if ( mtrr == -1 ) {
		return 0;
	}

	mgaMsg( 1,"CoverRangeWithMTRR( 0x%x, 0x%x, %i )\n", base, range, type );
	
	while ( range ) {
		/* see how big of an mtrr we can make */

		/* size must be a power of 2 */
		for ( size = 2048 ; ; ) {
			size *= 2;

			/* the base must be a multiple of the size */
			if ( base != size * ( base / size ) ) {
				size >>= 1;
				break;
			}

			if ( size > range ) {
				size >>= 1;
				break;
			}
		}

		/* set it if we aren't just checking the number */
		if ( type != -1 ) {
			struct mtrr_sentry sentry;
			
			sentry.base = base;
			sentry.size = size;
			sentry.type = type;

			mgaMsg(1,"MTRR fragment added: addr=0x%x size=0x%x type=%i\n",
				   sentry.base, sentry.size, sentry.type );
			if ( ioctl(mtrr, MTRRIOC_SET_ENTRY, &sentry) == -1 ) {
				mgaError("Error doing ioctl(2) on /proc/mtrr: %s\n", strerror(errno));
			}
		}

		base += size;
		range -= size;
		count++;
	}

	mgaMsg( 1,"------\n" );
#endif

	return count;
}

/*
 * SetWriteCombining
 *
 * Checks for unsetting an existing MTRR if needed
 */
static void SetWriteCombining( long physical, int bytes ) {
#ifndef NO_MTRR
	struct mtrr_sentry sentry;
	struct mtrr_gentry gentry;
	int	i;
	
	if ( mtrr == -1 ) {
		return;
	}
	/* remove any MTRR that cover the range */
	for ( i = 0 ; i < 128 ; i++ ) {
		gentry.regnum = i;
		if (ioctl(mtrr, MTRRIOC_GET_ENTRY, &gentry) == -1) {
			break;
		}
		mgaMsg(1,"MTRR reg %i: addr=0x%x size=0x%x type=%i\n", i, 
			gentry.base, gentry.size, gentry.type );
		if ( gentry.base >= physical + bytes ) {
			continue;
		}
		if ( gentry.base + gentry.size <= physical ) {
			continue;
		}
		
		/* we must delete this entry */
		sentry.base = gentry.base;
		sentry.size = gentry.size;
		if (ioctl(mtrr, MTRRIOC_DEL_ENTRY, &sentry) == -1) {
			mgaError("Error doing MTRRIOC_DEL_ENTRY on /proc/mtrr: %s\n", strerror(errno));
		} else {
			mgaMsg(1,"MTRRIOC_DEL_ENTRY succeeded\n" );
		}

		/* recreate fragments around the new region if necessary */
		if ( gentry.base < physical ) {
			CoverRangeWithMTRR( gentry.base, physical - sentry.base, gentry.type );
		}
		if ( gentry.base + gentry.size > physical + bytes ) {
			CoverRangeWithMTRR( physical + bytes, gentry.base + gentry.size - sentry.base, gentry.type );
		}
		
		/* because we deleted an entry, we need to check this index again */
		i--;
	}

	/* set this range to write combining */
	sentry.base = physical;
	sentry.size = bytes;
	sentry.type = 1; /* write-combining */

	if ( ioctl(mtrr, MTRRIOC_SET_ENTRY, &sentry) == -1 ) {
		mgaError("Error doing ioctl(2) on /proc/mtrr: %s\n", strerror(errno));
		mgaError("MTRR not enabled\n");
	} else {
		mgaMsg(1,"MTRR enabled: write-combining, addr=0x%x size=0x%x\n",
			   sentry.base, sentry.size );
	}
	
#endif
}


static void MemoryBenchmark( void *buffer, int dwords ) {
	int		i;
	int		start, end;
	int		mb;
	int 		*base;
	
	base = (int *)buffer;
	
	start = usec();
	for ( i = 0 ; i < dwords ; i += 8 ) {
		base[i] = 
		base[i+1] =
		base[i+2] = 
		base[i+3] = 
		base[i+4] = 
		base[i+5] = 
		base[i+6] = 
		base[i+7] = 0x15151515;		/* dmapad nops */
	}
	end = usec();

	mb = ( (float)dwords / 0x40000 ) * 1000000 / (end - start);
		
	mgaMsg( 1, "MemoryBenchmark: %i mb/s\n", mb );

	/* make the last command a DWGSYNC for DmaBenchmark */
	dwords -= dwords % 5;
	base[dwords-5] = 0x15151593;
}

static void DmaBenchmark( unsigned int physical, int dwords ) {
	int		start, end;
	int		mb;
	float		fsec;
	mgaUI32	dmaEnd;

	MgaSetSyncBusy();

	/* only test a full quad of registers */
	dwords -= dwords % 5;
	
	dmaEnd = physical + dwords*4;                                                
  	start = usec();

	OUTREG( MGAREG_PRIMADDRESS, physical );
	OUTREG( MGAREG_PRIMEND, dmaEnd | use_agp);

	mgaWaitForDmaCompletion();
	
	end = usec();

	fsec = ( end - start ) / 1000000.0;

	mb = ( (float)dwords * 4 / 0x100000 ) / fsec;

	mgaMsg( 1, "DmaBenchmark 0x%x bytes, %5.3f sec: %i mb/s\n", dwords*4, fsec, mb );
}

/*
 * VisualDmaTest
 * This will scroll a half meg image on the screen using dma
 */
static void VisualDmaTest( void ) {
	int	*dest, *destPtr;
	int	cmd;
	int	i, j, k, r, g, b;
	DMALOCALS;
	int	*test;
	
	test = malloc( 16*1024*1024 );
	
	/* start the primary commands */
	for ( j = 0 ; j < 1280-512 ; j += 5 ) {
	
	dest = mgaAllocSecondaryBuffer( 512*256 );
	mgaglx.warp_serieStart = mgaAllocSecondaryBuffer(0);	// FIXME: do this better...

	/* fill in the secondary buffer */
	destPtr = dest;
        for ( i = 0 ; i < 512 ; i++ ) {
		for ( k = 0 ; k < 256 ; k++ ) {
			int	pix;
			
			r = i * 255 / 512;
			g = k * 255 / 256;
			b = 128;
			pix = MGAPACKCOLOR565(r,g,b) |  
				( MGAPACKCOLOR565(r,g,b) << 16 );
			*destPtr++ = pix;
		}
	}

	
	/* XY destination, linear iload */
	cmd = DC_opcod_iload | 		/* image load */
	 	DC_atype_rpl |			/* raster replace mode */
		DC_linear_linear | 		/* linear source */
		DC_bltmod_bfcol |		/* source data is pre-formatted color */
		(0xC << DC_bop_SHIFT)  | 	/* use source bit op */
	 	DC_sgnzero_enable |		/* normal scanning direction */
	 	DC_shftzero_enable |	/* required for iload */
	 	DC_clipdis_enable;		/* don't use the clip rect */

	for ( i = 0 ; i < 512 ; i+=20 ) {
		MGADMAGETPTR( 20 );

		MGADMA_YDSTLEN( i, 512 );	/* top to bottom */
		MGADMA_FXBNDRY( j, 511+j );	/* full width */

		DMAOUTREG( MGAREG_AR0, 512 * 512 - 1);	/* source pixel count */
		DMAOUTREG( MGAREG_AR3, 0 );			/* required */
	
		/* pad if needed so the exec is at the end of a quad */
		DMAOUTREG( MGAREG_DMAPAD, 0 );
		DMAOUTREG( MGAREG_DMAPAD, 0 );
		DMAOUTREG( MGAREG_DMAPAD, 0 );
		
		MGADMA_DWGCTL_EXEC(cmd);
	
		DMAADVANCE();

		/* send the secondary data */	
		mgaSecondaryDma( TT_BLIT, dest, 512*256 );
	}
	
	mgaDmaFlush();
	
		/* pound on memory some */
		for ( i = 0 ; i < 4*1024*1024 ; i++ ) {
			test[i] = test[(i+2*1024*1024)&(4*1024*1024-1)];
		}
	}

	free( test );
}

static void AllocateCardDmaBuffer( void ) {
	PMemBlock	block;
	
	bufferBytes = 0x100000;
	block = mmAllocMem( cardHeap, bufferBytes, 8, 0 );
	if ( !block ) {
		mgaMsg( 1, "failed to allocate 0x%x bytes from cardHeap for command buffers.\n"
			, bufferBytes );
		return;
	}
	mgaMsg( 1, "allocated 0x%x bytes from cardHeap for command buffers.\n"
		, bufferBytes );
	bufferVirtual = cardVirtual + mmOffset( block );
	bufferPhysical = cardPhysical + mmOffset( block );
}

static void AllocatePhysicalDmaBuffer( void ) {
	PMemBlock	block;
	
	/* determine total size of buffer */
	bufferBytes = mgaglx.cmdSize;
	if ( !bufferBytes ) {
		mgaMsg(1,"defaulting to GLX_MGA_CMDSIZE = 4\n" );
		bufferBytes = 4;
	} else {
		mgaMsg(1,"using GLX_MGA_CMDSIZE = %i\n", bufferBytes );
	}
	bufferBytes *= 0x100000;
	
	block = mmAllocMem( sysmemHeap, bufferBytes, 8, 0 );
	if ( !block ) {
		mgaMsg( 1, "failed to allocate 0x%x bytes from sysmemHeap for command buffers.\n"
			, bufferBytes );
		return;
	}
	mgaMsg( 1, "allocated 0x%x bytes from sysmemHeap for command buffers.\n"
		, bufferBytes );
	bufferVirtual = sysmemVirtual + mmOffset( block );
	bufferPhysical = sysmemPhysical + mmOffset( block );
}	

static void AllocateVirtualDmaBuffer( void ) {
	/* determine total size of buffer */
	bufferBytes = mgaglx.cmdSize;
	if ( !bufferBytes ) {
		mgaMsg(1,"defaulting to mga_cmdsize = 4\n" );
		bufferBytes = 4;
	} else {
		mgaMsg(1,"using mga_cmdsize = %i\n", bufferBytes );
	}
	bufferBytes *= 0x100000;
	bufferVirtual = malloc( bufferBytes + 0x1000);
	/* align it to page size, might help on something used as much as this */
	bufferVirtual = (pointer)(((unsigned long) bufferVirtual & ~0xFFF) + 0x1000);
	mgaMsg( 1, "allocated 0x%x bytes from virtual memory for command buffers.\n"
		, bufferBytes );
}

static void OptimizeDMA( void ) {
	int		option;

	/* turn on enhmemacc */
	option = pcibusRead( MGAPciTag, 0x40 );
	pcibusWrite(MGAPciTag, 0x00000040, option | (1<<22) );
}


static int IsSDRAM( void ) {
  	int		option;

	option = pcibusRead( MGAPciTag, 0x40 );
	if ( option & ( 1<<14 ) ) {
		mgaMsg(1,"    SGRAM features enabled\n" );
	  	return 0;
	}
  	return 1;
}

  

/*
 * AllocateCommandBuffers
 * The dma command buffers can be either virtual or in the sysmemHeap
 * I never got card allocated buffers functioning reliably...
 * Some fraction of the buffer will be used for primary dma, and the rest
 * will be for secondary dma.
 */
#define	PRIMARY_FRACTION	8
#define	OVERFLOW_DWORDS		40	/* a warp secondary dma, X server registers, dwgsync, etc */
static void AllocateCommandBuffers( void ) {
	/* try to allocate the command buffers in either sysmem or cardmem */
	if ( mgaglx.dmaDriver > 0 ) {
		if ( mgaglx.cardCmds ) {
			AllocateCardDmaBuffer();
		} else if ( sysmemHeap ) {
			AllocatePhysicalDmaBuffer();
		}
	}
		
	/* if we didn't get real memory, get a virtual buffer and use PDMA */
	if ( !bufferPhysical ) {
		mgaglx.dmaDriver = 0;
		AllocateVirtualDmaBuffer();
	}
	
	if (__glx_is_server) {
		/* benchmark the writing speed to the command buffer */
		MemoryBenchmark( bufferVirtual, bufferBytes / 4 );
		MemoryBenchmark( bufferVirtual, bufferBytes / 4 );

		/* benchmark the read speed of the card's dma */
		if ( mgaglx.dmaDriver >= 2 ) {
			DmaBenchmark( (unsigned int)bufferPhysical, bufferBytes / 4 );
	 		DmaBenchmark( (unsigned int)bufferPhysical, bufferBytes / 4 );
	 		DmaBenchmark( (unsigned int)bufferPhysical, bufferBytes / 4 );
	 		DmaBenchmark( (unsigned int)bufferPhysical, bufferBytes / 4 );
	 		DmaBenchmark( (unsigned int)bufferPhysical, bufferBytes / 4 );
	 		if ( !use_agp ) {
				OptimizeDMA();
		 		DmaBenchmark( (unsigned int)bufferPhysical, bufferBytes / 4 );
		 		DmaBenchmark( (unsigned int)bufferPhysical, bufferBytes / 4 );
		 		DmaBenchmark( (unsigned int)bufferPhysical, bufferBytes / 4 );
		 		DmaBenchmark( (unsigned int)bufferPhysical, bufferBytes / 4 );
		 		DmaBenchmark( (unsigned int)bufferPhysical, bufferBytes / 4 );
		 	}
		}
	}

	/* always leave enough room for a X server setup and DWGSYNC after overflow checks */

	/* setup the two buffers that will be ping-ponged */
	dmaBuffers[0] = malloc(sizeof(mgaDma_buffer));
	memset(dmaBuffers[0], '\0', sizeof(mgaDma_buffer));
	dmaBuffers[0]->virtualAddress = (mgaUI32 *)bufferVirtual;
	dmaBuffers[0]->physicalAddress = bufferPhysical;
	dmaBuffers[0]->maxPrimaryDwords = ( bufferBytes >> 3 ) / PRIMARY_FRACTION;
	dmaBuffers[0]->primaryOverflowPoint = dmaBuffers[0]->maxPrimaryDwords - OVERFLOW_DWORDS;
	dmaBuffers[0]->maxSecondaryDwords = ( bufferBytes >> 3 ) - dmaBuffers[0]->maxPrimaryDwords;
	
	dmaBuffers[1] = malloc(sizeof(mgaDma_buffer));
	memset(dmaBuffers[1], '\0', sizeof(mgaDma_buffer));
	dmaBuffers[1]->virtualAddress = (mgaUI32 *)bufferVirtual + bufferBytes/8;
	dmaBuffers[1]->physicalAddress = bufferPhysical + bufferBytes/2;
	dmaBuffers[1]->maxPrimaryDwords = ( bufferBytes >> 3 ) / PRIMARY_FRACTION;
	dmaBuffers[1]->primaryOverflowPoint = dmaBuffers[1]->maxPrimaryDwords - OVERFLOW_DWORDS;
	dmaBuffers[1]->maxSecondaryDwords = ( bufferBytes >> 3 ) - dmaBuffers[0]->maxPrimaryDwords;

	mgaMsg( 1, "dmaBuffers[]->maxPrimaryDwords = %i\n", dmaBuffers[0]->maxPrimaryDwords );
	mgaMsg( 1, "dmaBuffers[]->maxSecondaryDwords = %i\n", dmaBuffers[0]->maxSecondaryDwords );
	
	mgaDmaResetBuffer();
}


/*
 * AllocateGARTMemory
 */
static int AllocateGARTMemory(size_t size)
{
#ifndef NO_AGPGART
   int i, j, k, m, pages = (size + 4095) / 4096;
   struct gart_entry entry;
   
   gartfd = open("/dev/gart", O_RDWR);
   if (gartfd == -1)
     {
	mgaMsg(1, "unable to open /dev/gart: %s\n", sys_errlist[errno]);
	return -1;
     }
   
   if (ioctl(gartfd, GARTIOCINFO, &gartinf) != 0)
     {
	mgaMsg(1, "error doing ioctl(GARTIOCINFO): %s\n", sys_errlist[errno]);
	mgaMsg(1, "first attempt\n");
	close(gartfd);
	return -1;
     }
   
   gartbuf = mmap(NULL, gartinf.size * 0x100000, PROT_READ | PROT_WRITE, MAP_SHARED, gartfd, 0);
   if (gartbuf == (void *)0xffffffff)
     {
	mgaMsg(1, "mmap() on /dev/gart failed: %s\n", sys_errlist[errno]);
	close(gartfd);
	return -1;
     }
   
   if (__glx_is_server)
     {
        if(ioctl(gartfd, GARTIOCINIT) != 0)
          {
             mgaMsg(1, "Error initializing AGP point to point connection\n");
             close(gartfd);
             return -1;
          }
        /* Call information function a second time for the agp mode */
	
        if (ioctl(gartfd, GARTIOCINFO, &gartinf) != 0)
          {
             mgaMsg(1, "error doing ioctl(GARTIOCINFO): %s\n", sys_errlist[errno]);
             mgaMsg(1, "second attempt\n");
	     
             close(gartfd);
             return -1;
          }
        if(gartinf.mode == AGP_MODE2X && !MGA_IS_G400(MGAchipset))
          {
             mgaMsg(1, "enabling agp 2x pll encoding\n");
             OUTREG(MGAREG_AGP_PLL, AGP_PLL_agp2xpllen_enable);
          }
	if (MGA_IS_G400(MGAchipset))
	  {
	     i = INREG(MGAREG_CFG_OR);
	     if (i & CFG_OR_comp_or_enable)
	       { mgaMsg(1, "compensation override\n") }
	     else
	       mgaMsg(1, "internal compensation logic will be used\n");
	     
	     j = MGA_GET_FIELD(CFG_OR_compfreq, i);
	     	     
	     mgaMsg(1, "CFG_OR : 0x%x\n", i);
	     switch (j)
	       {
		case 0:
		  mgaMsg(1, "AGP Compensation Frequency: (%i) 16 msec\n", j);
		  break;
		case 1:
		  mgaMsg(1, "AGP Compensation Frequency: (%i) 31 msec\n", j);
		  break;
		case 2:
		  mgaMsg(1, "AGP Compensation Frequency: (%i) 63 msec\n", j);
		  break;
		case 3:
		  mgaMsg(1, "AGP Compensation Frequency: (%i) 0.125 sec\n", j);
		  break;
		case 4:
		  mgaMsg(1, "AGP Compensation Frequency: (%i) 0.25 sec\n", j);
		  break;
		case 5:
		  mgaMsg(1, "AGP Compensation Frequency: (%i) 0.5 sec\n", j);
		  break;
		case 6:
		  mgaMsg(1, "AGP Compensation Frequency: (%i) 1 sec\n", j);
		  break;
		case 7:
		  mgaMsg(1, "AGP Compensation Frequency: (%i) 2 sec\n", j);
		  break;
		default:
		  mgaMsg(1, "AGP Compensation Frequency: (%i) Unknown!\n", j);
	       }
	     mgaMsg(1, "AGP Compensation Value: high=%i low=%i\n", MGA_GET_FIELD(CFG_OR_comporup, i), MGA_GET_FIELD(CFG_OR_compordn, i));
	  }
     }
   else
     {
	return 0;
     }
   
   i = pages / 1024;
   j = pages % 1024;
   for(k = 0; k < i; k++)
     {
        entry.ofs = k * 1024;
        entry.size = 1024;
        if(ioctl(gartfd, GARTIOCINSERT, &entry))
          {
             /* free previous pages */
	     
             for(m = 0; m < k; m++)
               {
		  entry.ofs = m * 1024;
                  entry.size = 1024;
                  ioctl(gartfd, GARTIOCREMOVE, &entry);
               }
	     
             mgaMsg(1, "GART: allocation of %i pages failed\n", pages);
          }
     }
   if(j != 0)
     {
        entry.ofs = (i + 1) * 1024;
        entry.size = j;
        if(ioctl(gartfd, GARTIOCINSERT, &entry))
          {
             /* free previous pages */
             for(m = 0; m < i; m++)
               {
                  entry.ofs = m * 1024;
                  entry.size = 1024;
                  ioctl(gartfd, GARTIOCREMOVE, &entry);
               }
	     
             mgaMsg(1, "GART: allocation of %i pages failed\n", pages);
          }
     }
#endif /* NO_AGPGART */
   return 0;
}


/*
 * AllocateSystemMemory
 * Looks at environment variables to determine if a block
 * of physical memory has been left for graphics after the
 * memory available to the kernel.
 * System memory can be used for dma command buffers or
 * textures.
 */
static void AllocateSystemMemory( void ) {
 	int		fd;
#ifndef NO_AGPGART   
  	char		*adr;
#endif  	
    	  sysmemPhysical = 0;
    	  sysmemVirtual = 0;
    	  sysmemHeap = 0;	

	if ( !mgaglx.dmaDriver ) {
		return;
	}
	
  /* determine total requested size of buffer */
  sysmemBytes = mgaglx.dmaSize;
  if ( !sysmemBytes ) {
    mgaMsg(1,"mga_dmasize not set, skipping physical allocation\n" );
    return;
  }
  sysmemBytes *= 0x100000;

#ifndef NO_AGPGART
	/* try AGP memory */
	adr = glx_getvar( "mga_dmaadr" );
	if ( adr && !strcasecmp( adr, "agp" ) ) {
 	   if (!AllocateGARTMemory(sysmemBytes))
 	   {
		sysmemPhysical = gartinf.physical;
		sysmemVirtual = (unsigned char *)gartbuf;
		sysmemHeap = mmInit( sysmemBytes );
      					 
		mgaMsg( 1, "AGP Aperture: %p\n", sysmemPhysical );
		mgaMsg( 1, "sysmemSize: %p\n", sysmemBytes );
      
   		use_agp = PDEA_pagpxfer_enable /* | PDEA_primnostart_enable */ ;
    		mgaMsg(1, "use_agp = %x\n", use_agp);
    		return;
    	  }
	mgaMsg(1, "AllocateGARTMemory failed.\n" );
	return;
	}
#endif	/* NO_AGPGART */
	
  /* mgaglx.dmaDriverADR should be set to a value >= the mem= kernel parm */
  sysmemPhysical = mgaglx.dmaAdr;
  if ( sysmemPhysical < 16 ) {
    mgaMsg( 1, "unlikely mga_dmaadr=%i, skipping physical allocation\n", bufferPhysical );
    return;
  }
  sysmemPhysical *= 0x100000;


  
  /* FIXME!!!: should check sysmemPhysical against /proc/meminfo */
  
  fd = open( "/dev/mem", O_RDWR );
  if ( fd < 0 ) {
    mgaMsg( 1, "failed to open /dev/mem\n" );
    return;
  }
  
  sysmemVirtual = (unsigned char *) mmap( NULL, sysmemBytes, PROT_READ | PROT_WRITE,
					  MAP_SHARED, fd, (off_t)sysmemPhysical );
  if ( sysmemVirtual == MAP_FAILED ) {
    mgaMsg( 1, "failed to mmap sysmem\n" );
    close( fd );
    return;
  }

  /* FIXME: should verify the memory exists with read / write test */
  
  /* set to write combining */  
  if (__glx_is_server) {
 	/* due to MTRR fragmentation issues, we can't do this for all
           memory ranges - except on the K6...
	*/
  	if ( IsPowerOfTwo( sysmemPhysical )  
	     || (gl_identify_x86_cpu_features() & GL_CPU_3Dnow) 
	     )
	{
  		mgaMsg( 1, "Setting write combining on system heap.\n" );
  		SetWriteCombining( sysmemPhysical, sysmemBytes );
  	} else {
  		mgaMsg( 1, "Can't set write combining on system heap, not power of two.\n" );
  	}
  }
  
  /* create a heap */
  sysmemHeap = mmInit( sysmemBytes );
  
  mgaMsg( 1, "sysmemPhysical: %p\n", sysmemPhysical );
  mgaMsg( 1, "sysmemVirtual: %p\n", sysmemVirtual );
  mgaMsg( 1, "sysmemSize: %p\n", sysmemBytes );
  
}

/*
 * ChooseTextureHeap
 * Determine if textures should be stored in the cardHeap or the
 * sysmemHeap.
 */
static void ChooseTextureHeap( void ) {
	/* share textures on the card memory until proven otherwise */
	textureHeap = cardHeap;
      	textureHeapVirtual = (unsigned char *)vgaLinearBase;
      	textureHeapPhysical = 0;

	/* if we don't have a system memory heap, textures MUST be on the card */
	if ( !sysmemHeap ) {
		mgaMsg( 1, "No sysmemHeap, textures must be stored on card\n" );
		return;
	}
	
	/* see if we should use the system heap for textures or just dma commands */
	if ( !mgaglx.systemTexture ) {
		mgaMsg( 1, "mga_systemtexture not set, textures will be stored on card\n" );
		return;
	}

	/* make sure there is some memory left for textures */
	if ( sysmemBytes < bufferBytes + 1024*1024 ) {
		mgaMsg( 1, "sysmemBytes < bufferBytes + 1meg, textures will be stored on card\n" );
		return;
	}
	
	textureHeap = sysmemHeap;
      	textureHeapVirtual = sysmemVirtual;
      	textureHeapPhysical = sysmemPhysical;

	mgaMsg( 1, "Texturing from sysmemHeap\n" );
}


/*
 * mgaDmaInit
 *
*/
void mgaDmaInit(void) {
  	mgaUI32 devctrl;

        /* Server init - queries environment variables.  The client
	 * gets these values from the sever and initializes them in
	 * mgadirect.c 
	 */
        if (__glx_is_server) {
		mgaglx.dmaDriver = glx_getint("mga_dma");
		mgaglx.dmaSize = glx_getint("mga_dmasize");
		mgaglx.dmaAdr = glx_getint("mga_dmaadr");
		mgaglx.cmdSize = glx_getint("mga_cmdsize");
		mgaglx.cardCmds = glx_getint("mga_cardcmds");
		mgaglx.systemTexture = glx_getint("mga_systemtexture");
	}

  
  	devctrl = pcibusRead( MGAPciTag, 0x04 );
    	mgaMsg( 1, "devctrl = %08x\n", devctrl);
 
        use_agp = 0;
    
	if (__glx_is_server) {
		/* prepare to set write combining */
		MgaOpenMTRR();
	
		/* set write combining on the framebuffer */
		SetWriteCombining( xf86AccelInfoRec.ServerInfoRec->physBase,
				   xf86AccelInfoRec.ServerInfoRec->physSize );
	}

	/* get some system memory and make it write combining if we can */
	AllocateSystemMemory();

	/* set up the matrox pdma memory window */	
	MapPseudoDmaWindow();
	
	/* read the command environment variable */
	mgaMsg(1,"mgaDmaInit: mga_dma = %i\n", mgaglx.dmaDriver );

	/* setup the two command buffers in the apropriate memory space */
	AllocateCommandBuffers();
	
	/* check for using a PCI texture heap */
	ChooseTextureHeap();
	
	/* prepare the first buffer for use */
	mgaDmaResetBuffer();
}

/*
 * CreateFrontBuffer
 * Called during initialization to set up parameters needed
 * for the swapbuffers blit.
 */
static mgaBufferPtr CreateFrontBuffer( void )
{
  int Attrib;
  mgaBufferPtr buf;
  
  switch( vga256InfoRec.depth )  {
  case 15:
    Attrib = MGA_PF_555;
    break;
  case 16:
    Attrib = MGA_PF_565;
    break;
  case 24:
    if (vgaBitsPerPixel == 24)
      Attrib = MGA_PF_888;	/* we can't render to this, but we can blit to it */
    else
      Attrib = MGA_PF_8888;
    break;
  default:
    mgaError("No support for %d bits per pixel.\n",vgaBitsPerPixel);
    return NULL;
  }
  buf = mgaCreatePrimaryBuffer(Attrib,
			       vga256InfoRec.virtualX,
			       vga256InfoRec.virtualY,
			       vga256InfoRec.displayWidth);

  /* enable hardware dithering in 16 bit modes */
  switch(Attrib & MGA_PF_MASK) {
  case MGA_PF_555:
  case MGA_PF_565:
    buf->Setup[MGA_SETUP_MACCESS] |= MA_dit555_enable | 
      MA_nodither_enable; /* xf86 default */
    break;
  }
  buf->Setup[MGA_SETUP_CXBNDRY] = 0x0fff0000;
  buf->Setup[MGA_SETUP_YTOP] = 0;
  buf->Setup[MGA_SETUP_YBOT] = 0x00ffffff;
  return buf;
}



/*
 * This function should only verify that the current hardware is supported.
 * It should do no setup. As we support various Matrox chipsets, perhaps it
 * should return an indicator of which chipset is present.
 */
GLboolean det_hwGfx() {  
    mgaMsg(1,"Detected 0x%x Chip ID\n", MGAchipset);
   /* is this the best way check for mga presence? */
   if(!MGA_IS_G200(MGAchipset) && !MGA_IS_G400(MGAchipset)){ 
      mgaError("MGAchipset not set, no mga hardware?\n");
      return GL_FALSE;
   }

   /* FIXME: implement support for other depths... */
   if(vga256InfoRec.depth != 15 &&
      vga256InfoRec.depth != 16 &&
      vga256InfoRec.depth != 24 ) {
      mgaError("Unsupported depth: %d, only 15,16, and 24 bpp are supported right now\n",
	       vga256InfoRec.depth);
      return GL_FALSE;
   }

   return GL_TRUE;
}

/*
 * mgaInitLogging
 *
 */
void mgaInitLogging( void ) {		
	char	*logName;

	/* open the logfile and set loglevel */
	logName = glx_getvar("mga_logfile");
	if ( __glx_is_server ) { 
		mgaOpenLog( logName );
	} else {
		/* direct rendering clients use a different file
		so they don't stomp on the server's log */
		if ( logName ) {
			char	newName[1024];
			
			strcpy( newName, logName );
			strcat( newName, "_direct" );
			mgaOpenLog( newName );
		}
	}
	if (glx_getvar("mga_loglevel")) {
		mgaSetLogLevel(glx_getint("mga_loglevel"));
	} else {
		mgaSetLogLevel(DBG_LEVEL_BASE);
	}
}


/*
 * mgaDumpRegisters
 */
void mgaDumpRegisters( void ) {
	int	i, r;
	
	mgaMsg(1, "Configuration registers:\n" );
	for ( i = 0 ; i < 256 ; i+=4 ) {
	  	r = pcibusRead( MGAPciTag, i );
	  	mgaMsg(1, "0x%2x : 0x%8x\n", i, r );
	}
	
	mgaMsg(1, "Drawing registers:\n" );
	for ( i = 0x1c00 ; i < 0x1dff ; i+= 4 ) {
		r = INREG( i );
	  	mgaMsg(1, "0x%2x : 0x%8x\n", i, r );
	}
	for ( i = 0x2180 ; i < 0x2dff ; i+= 4 ) {
		r = INREG( i );
	  	mgaMsg(1, "0x%2x : 0x%8x\n", i, r );
	}

}

/* 
 * Soft reset (old, not used anymore)
 * This will reset 3D engine. If you don't do this, 3D drawing may not work
 * correctly. (esp. line with depth)
 * I have to do soft reset every time I turn on my computer.
 * If you don't use 3d line, everything should be ok w/o reset.
 */
void mgaSoftReset( void ) {
  OUTREG(MGAREG_RST, R_softreset_enable);
  usleep(20); /* in spec. minimum 10us */
  OUTREG(MGAREG_RST, R_softreset_disable);
}


/*
 * mgaInitGLX
 * This is the initial entry point for the mga hardware driver,
 * called at X server module load time, or libGL direct rendering
 * init time.
 */
GLboolean mgaInitGLX( void ) {
	mgaInitLogging();

	mgaMsg(1,"virtual (x, y) (%d, %d)\n", vga256InfoRec.virtualX,
		 vga256InfoRec.virtualY);
	mgaMsg(1,"width: %d\n", vga256InfoRec.displayWidth);
	mgaMsg(1,"depth: %d\n", vga256InfoRec.depth);
	mgaMsg(1,"memBase: 0x%08x\n", vgaLinearBase);
	mgaMsg(1,"videoRam: 0x%08x\n", vga256InfoRec.videoRam);

	/* check to make sure that we are on an apropriate chip and not
	running in 8bpp mode */
	if ( !det_hwGfx() ) {
		return GL_FALSE;
	}

	/* start up our card memory manager */
	cardHeap = mmInit(vga256InfoRec.videoRam * 1024);
	if ( !cardHeap ) {
		mgaMsg( 1,"cardHeap creation failed, exiting!\n" );
		return GL_FALSE;	/* really shouldn't happen */
	}
	cardPhysical = (mgaUI32)xf86AccelInfoRec.ServerInfoRec->physBase;
	cardVirtual = (unsigned char *)vgaLinearBase;
	
	/* reserve memory used by the desktop screen and set up
	some hardware acceleration values needed to draw to it */
	mgaFrontBuffer = CreateFrontBuffer();
	if (!mgaFrontBuffer) {
		mgaError("Cannot create front buffer.\n");
		return GL_FALSE;	/* really shouldn't happen */
	}
	/* reserve memory X uses for pixmap acceleration */
	mmReserveMem( cardHeap, xf86AccelInfoRec.PixmapCacheMemoryStart,
	       xf86AccelInfoRec.PixmapCacheMemoryEnd -
	       xf86AccelInfoRec.PixmapCacheMemoryStart);

	/* reserve last 1KB if hardware cursor is active*/
	if (MGAdac.isHwCursor) {	
		mmReserveMem( cardHeap, (vga256InfoRec.videoRam-1)*1024,1024);
	}
	/* the remaining memory is available for back buffers, depth
	buffers, and textures */
	mmDumpMemInfo( cardHeap );

	/* init the dma system */  
	mgaDmaInit();

	/* init warp */
	if ( !mgaWarpInit() ) {
 		FatalError( "Warp initialization failed" );
	}

	/* FIXME: what other GLXProcs pointers should we change? */
	GLXProcs.CreateContext = mgaGLXCreateContext;
	GLXProcs.DestroyContext = mgaGLXDestroyContext;
	GLXProcs.CreateImage = mgaGLXCreateImage;
	GLXProcs.DestroyImage = mgaGLXDestroyImage;
	GLXProcs.CreateDepthBuffer = mgaGLXCreateDepthBuffer;
	GLXProcs.MakeCurrent = mgaGLXMakeCurrent;
	GLXProcs.BindBuffer = mgaGLXBindBuffer;
	GLXProcs.SwapBuffers = mgaGLXSwapBuffers;
	GLXProcs.VendorPrivate = mgaGLXVendorPrivate;
	GLXProcs.AllowDirect = mgaGLXAllowDirect;

	if (!__glx_is_server) {
	   GLXProcs.ValidateFrontBuffer = mgaClientGetGeometry;
	}

	/* these vars can be changed between invocations of direct clients */
	if (glx_getint("mga_nullprims") ) {
		mgaMsg( 1, "enabling mga_nullprims\n" );
		mgaglx.nullprims = 1;
	}
	if (glx_getint("mga_skipdma") ) {
		mgaMsg( 1, "enabling mga_skipdma\n" );
		mgaglx.skipDma = 1;
	}
	if (glx_getint("mga_boxes") ) {
		mgaMsg( 1, "enabling mga_boxes\n" );
		mgaglx.boxes = 1;
	}
	if (glx_getint("mga_nofallback") ) {
		mgaMsg( 1, "enabling mga_nofallback\n" );
		mgaglx.noFallback = 1;
	}
	if (glx_getint("mga_nosgram") || IsSDRAM() ) {
		mgaMsg( 1, "enabling mga_nosgram\n" );
		mgaglx.nosgram = 1;
	}
	
	/* test by blitting to screen */
	// VisualDmaTest();

	// mgaDumpRegisters();

	mgaError("mgaInitGLX completed\n");
	return GL_TRUE;
}


