/*
  GLX Hardware Device Driver for Matrox Millenium G200
  Copyright (C) 1999  Stephen Crowley (crow@debian.org)
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
  
 original by Jeff Hartmann <slicer@ionet.net>
  6/16/99: rewrite by John Carmack <johnc@idsoftware.com>
*/

#ifndef MGADMA_H
#define MGADMA_H

#include "mgacommon.h"
#include "mgawarp.h"
#include "mm.h"
#include "os.h"

/* the g200 general purpose dma stream operates in blocks of five
dwords that program four registers at a time.  One index dword
specifies the registers to program with the following four dwords of data.

ADR0 | (ADR1<<8 ) | (ADR2<<16) | (ADR3<<24)
DATA0
DATA1
DATA2
DATA3
....
*/

#define DWGREG0 0x1c00
#define DWGREG0_END 0x1dff
#define DWGREG1 0x2c00
#define DWGREg1_END 0x2dff

/* page 407 in the specs, 4-29 */
#define ISREG0(r) (r >= DWGREG0 && r<= DWGREG0_END)
#define ADRINDEX0(r) (mgaUI8)((r - DWGREG0) >> 2)
#define ADRINDEX1(r) (mgaUI8)(((r - DWGREG1) >> 2) | 0x80)
#define ADRINDEX(r) (ISREG0(r) ? ADRINDEX0(r) : ADRINDEX1(r)) 

#define MGA_ADDRGEN(i0, i1, i2, i3) \
 (ADRINDEX(i0) | \
 (ADRINDEX(i1) << 8) | \
 (ADRINDEX(i2) << 16) | \
 (ADRINDEX(i3) << 24))


/*
   Before we submit a dma command buffer, we write SYNC_DMA_BUSY to DWGSYNC.
   The last command in the dma buffer will clear DWGSYNC.
   The X server might also change the value in DWGSYNC, but the register
   locking guarantees that dma will complete first.  We will spin until
   timeout in waitDmaCompletion() if the X server ever happens to use
   our magic value.

   Any time DWGSYNC is read and it holds SYNC_DMA_BUSY, then dma is still
   active and we need to wait until it is cleared.
   
   If any other code needs to add a DWGSYNC to the command stream for any
   reason, this value must be used.
*/
#define SYNC_DMA_BUSY	0xea832534		/* just a random number */

/* the main initialization of the entire mga hardware driver */
GLboolean mgaInitGLX( void );

/* a flush command will guarantee that all data added to the dma buffer
is on its way to the card, and will eventually complete with no more
intervention.  If running with pseudo dma, this will be the same as a finish
call, but if async dma is active then the card will be executing the commands
while the cpu is doing other work.  A protected memory region keeps X server
interaction with the hardware registers safe. */
void mgaDmaFlush( void );

/* the overflow function is called when a block can't be allocated
in the current dma buffer.  It flushes the current buffer and
records some information */
void mgaDmaOverflow( int newDwords );
void mgaDmaSecondaryOverflow( int newDwords );

/* a finish command will guarantee that all dma commands have actually
been consumed by the card.  Note that there may still be a couple primitives
that have not yet been executed out of the internal FIFO, so this does not
guarantee that the drawing engine is idle. */
void mgaDmaFinish( void );

/* a mgaWaitDrawingEngine command will guarantee that the framebuffer
is safe to read or write for software rendering */
int mgaWaitDrawingEngine( void );

typedef enum {
	TT_GENERAL,
	TT_BLIT,
	TT_VECTOR,
	TT_VERTEX
} transferType_t;

/* secondary dma is for TT_BLIT and TT_VERTEX transfers, and can be properly
simulated with pseudo dma if needed.  The data pointer should be from
a previous mgaAllocSecondaryBuffer() */
void mgaSecondaryDma( transferType_t transferType, mgaUI32 *data, int dwords );

/* for routines that need to add a variable stream of register writes, use
the MGADMAGETPTR() / DMAOUTREG() / DMAADVANCE() interface.  Call
mgaDmaGetPtr() with a length in dwords that is guaranteed to be greater
than what you will be writing.  Using a large value, like 100, does not
waste anything.  Program all the registers you need with DMAOUTREG(),
or the MGADMA_* macros, which use DMAOUTREG().  When you have finished,
call DMAADVANCE(), which will add any necessary padding and commit the
data to the dma buffer 

DMALOCALS must be included at the top of all functions that use these
macros to declare temporary variables.

*/

typedef struct _dma_buffer {
	mgaUI32	physicalAddress;
	mgaUI32 	*virtualAddress;

	mgaUI32	primaryDwords;			/* amount currently allocated */
	mgaUI32	primaryOverflowPoint;	/* force a flush if primaryDwords would cross this */
	mgaUI32	maxPrimaryDwords;

	/* the secondary buffer will start at virtualAddress + maxPrimaryDwords */
	mgaUI32	secondaryDwords;			/* amount currently allocated */
	mgaUI32	maxSecondaryDwords;
} mgaDma_buffer;

/* cardHeap is the 8 / 16 / 32 megs of memory on the video card */
extern	memHeap_t	*cardHeap;
extern	mgaUI32		cardPhysical;
extern	unsigned char	*cardVirtual;

/* sysmemHeap is system memory we have been given after the area used by the kernel */
extern	memHeap_t	*sysmemHeap;
extern	mgaUI32		sysmemPhysical;			/* 0 if we don't have a physical mapping */
extern	unsigned char	*sysmemVirtual;

/* textureHeap will point to either cardHeap or sysmemHeap */
extern	memHeap_t	*textureHeap;
extern	mgaUI32		textureHeapPhysical;		/* 0 if we aren't using PCI texturing */
extern	unsigned char	*textureHeapVirtual;		/* correct for either local or PCI heaps */

extern	mgaUI32		*pseudoDmaVirtual;		/* our local mapping of pdma window */

extern	mgaDma_buffer	*dma_buffer;

/* These are only required by mgadirect.c:
 */
extern  mgaUI32 mgaActiveDmaBuffer;
extern  void    (*mgaDoDmaFlush)( int );
extern  mgaDma_buffer	*dmaBuffers[2];

extern int use_agp;

extern int mgaWaitForDmaCompletion( void );
extern void mgaDmaResetBuffer( void );


#define DMALOCALS int outcount; mgaUI32 *dma_ptr; unsigned char tempIndex[4]

/* this will trigger an overflow if there wouldn't be enough room left over
for the dma flush routine to send unsent vertexes to WARP and reset the
registers for X */
#define MGADMAGETPTR(length)						\
   if (0)								\
      fprintf(stderr, "MGADMAGETPTR %d in %s\n", length, __FUNCTION__);	\
   if( dma_buffer->primaryDwords + length > 				\
       dma_buffer->primaryOverflowPoint )				\
     mgaDmaOverflow(length);						\
   dma_ptr = (dma_buffer->virtualAddress + dma_buffer->primaryDwords);	\
   outcount = 0;

#define MGADMAGETPTR_NO_OVERFLOW(length)				\
   if (0)								\
      fprintf(stderr, "MGADMAGETPTR_NO_OVERFLOW %d in %s\n", 		\
	      length, __FUNCTION__);					\
   dma_ptr = (dma_buffer->virtualAddress + dma_buffer->primaryDwords);	\
   outcount = 0;

#define DMAOUTREG(reg ,val) do {			\
   if (0)						\
      fprintf(stderr, "OUTREG %x : %x\n", reg, val);	\
   tempIndex[outcount]=ADRINDEX(reg);			\
   dma_ptr[1+outcount]=val;				\
   if ( ++outcount == 4 ) {				\
      outcount = 0;					\
      dma_ptr[0]=*(int *)tempIndex;			\
      dma_ptr+=5;					\
      dma_buffer->primaryDwords += 5;			\
   }							\
}while(0)

#define DMAADVANCE() do {			\
   if ( outcount & 3 ) {			\
      while(outcount & 3) {			\
	 tempIndex[outcount++]=0x15;		\
      }						\
      dma_ptr[0]=*(int *)tempIndex;		\
      dma_buffer->primaryDwords += 5;		\
   }						\
}while(0)



/* if a secondary dma is being started, use this */
#define DMAADVANCECHOP() do {					\
if ( outcount & 3 ) { \
dma_ptr[0]=*(int *)tempIndex; \
dma_buffer->primaryDwords += outcount+1;	       		\
} \
}while(0)

#include <stdio.h>

/* mgaAllocSecondaryBuffer will allocate space in the secondary dma buffer,
flushing the buffers if there isn't enough room.  The pointer returned is a
normal virtual address, but it is in physical memory so it can be sent
to the dma engine.  This is used for WARP vertexes. */
static inline mgaUI32	*mgaAllocSecondaryBuffer( int dwords ) {
	mgaUI32	*buf;

	/* make sure there is room */	
	if ( dma_buffer->secondaryDwords + dwords > 
	     dma_buffer->maxSecondaryDwords ) {
		mgaDmaSecondaryOverflow( dwords );
	}
	
	buf = dma_buffer->virtualAddress + dma_buffer->maxPrimaryDwords 
		+ dma_buffer->secondaryDwords;
	dma_buffer->secondaryDwords += dwords;
	 
	return buf;
}

/* texture uploads need to atomically allocate space in both the primary
and secondary buffers.  If they were done separetley and the second allocation
forced an overflow, the first would be in the wrong buffer. The secondary
buffer is returned, and an immediately subsequent MGADMAGETPR() of primaryDwords
or less is guaranteed to not cause an overflow. */
mgaUI32 *mgaAllocPrimaryAndSecondaryBuffer( int primaryDwords, int secondaryDwords );

#endif
