/*
 * GLX Hardware Device Driver for Matrox Millenium G200
 * Copyright (C) 1999 Wittawat Yamwong
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * WITTAWAT YAMWONG, OR ANY OTHER CONTRIBUTORS BE LIABLE FOR ANY CLAIM, 
 * DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR 
 * OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE 
 * OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 *
 *    Wittawat Yamwong <Wittawat.Yamwong@stud.uni-hannover.de>
 */

/* $Id: mgabuf.c,v 1.7 1999/11/22 21:21:07 johnc Exp $ */

#include <stdlib.h>

#include "vga.h"
#include "mgalib.h"
#include "mgadma.h"


#define BASEADDR ((unsigned char *)vgaLinearBase)

static int mgaInitBuffer(mgaBufferPtr buf, int Attrib, int Size,
			 int Width, int Height, int Pitch)
{
  int ofs;
  int Format,Type;
  int PitchAlign,OrgAlign;
  mgaUI32 maccess;
  int	systemMemory;
 
  buf->magic = mgaBufferMagic;
  buf->Attrib = Attrib;
  Type = Attrib & MGA_TYPE_MASK;
  Format = Attrib & MGA_PF_MASK;

    systemMemory = 0;

  switch (Type) {

  case MGA_COLORBUFFER:
    /* 
     * input:
     * Attrib,Width,Height
     * if Pitch == 0, Pitch will be calcualted from Width
     */
    buf->Drawable = 1;
    buf->Setup[0] = MGA_SETUP_0;
    buf->Setup[5] = MGA_SETUP_5;

    switch (Format) {
    case MGA_PF_INDEX: 
      buf->BytesPerPixel = 1;
      maccess = MA_pwidth_8;
      break;
    case MGA_PF_565:
      buf->BytesPerPixel = 2; 
      maccess = MA_pwidth_16;
      break;
    case MGA_PF_555:
      buf->BytesPerPixel = 2; 
      maccess = MA_pwidth_16 | MA_dit555_enable;
      break;
    case MGA_PF_888:
      buf->BytesPerPixel = 3;
      maccess = MA_pwidth_24;
      break;
    case MGA_PF_8888:
      buf->BytesPerPixel = 4;
      maccess = MA_pwidth_32;
      break;
    default:
      return -1;
    }

    OrgAlign = 6;
    PitchAlign = (1 << OrgAlign);

    if ( Pitch == 0 ) {
    	Pitch = (Width+PitchAlign-1 ) & ~(PitchAlign - 1);
    }
    buf->Pitch = Pitch;
    buf->Size = Pitch*buf->BytesPerPixel*Height;
    
    /* align to a 4k block, which seems to avoid some depth buffer problems
    on g400 */
    buf->Size = ( buf->Size + 4095 ) & ~4095;
    
    // first try to allocate in card memory
    if ( Attrib & MGA_FORCE_SYSMEM || 
    ( buf->MemBlock = mmAllocMem( cardHeap, buf->Size, OrgAlign, 0 ) ) == 0 ) {
    	// now try to allocate in pci / agp memory
    	if ( !sysmemHeap ) {
		return -1;
	}
	buf->MemBlock = mmAllocMem( sysmemHeap, buf->Size, OrgAlign, 0 );
	if ( buf->MemBlock == 0 ) {
		return -1;
	}
	mgaMsg( 1, "allocating back buffer in system memory\n" );
	systemMemory = 1 | use_agp;
	buf->SystemMemory = 1;
    }
    
    ofs = buf->MemBlock->ofs;

    buf->Setup[MGA_SETUP_PITCH] = Pitch & ~P_iy_MASK;
	if ( systemMemory ) {
    		// FIXME: is this sysmemPhysical offset right for AGP? 
	    buf->Setup[MGA_SETUP_DSTORG] = ( ofs + sysmemPhysical ) | systemMemory;
	} else {
	    buf->Setup[MGA_SETUP_DSTORG] = ofs;
	}
    buf->Setup[MGA_SETUP_MACCESS] = maccess;
    buf->Setup[MGA_SETUP_CXBNDRY] = (Width-1) << CXB_cxright_SHIFT;
    buf->Setup[MGA_SETUP_YTOP] = 0;
    buf->Setup[MGA_SETUP_YBOT] = (Height-1)*Pitch;
    buf->Setup[MGA_SETUP_PLNWT] = ~0;
    buf->Setup[MGA_SETUP_ZORG] = 0;
    buf->SetupSize = MGA_SETUP_SIZE;

    buf->Width = Width;
    buf->Height = Height;
    buf->BufAddr = &BASEADDR[ofs];
    break;

  case MGA_ZBUFFER16:
  case MGA_ZBUFFER32:
    /* 
     * input: 
     * Attrib,Pitch,Height,DGAFrameBuffer
     */
    buf->SetupSize = 0;
    if (Type == MGA_ZBUFFER32)
      buf->BytesPerPixel = 4;
    else
      buf->BytesPerPixel = 2;
    buf->Size = Pitch * Height * buf->BytesPerPixel;

    /* align to a 4k block, which seems to avoid some depth buffer problems
    on g400 */
    buf->Size = ( buf->Size + 4095 ) & ~4095;

    /* try to allocate it in the second 4mb bank if possible */
    /* we may have to force it into system memory, because the mga doesn't 
    seem to like back buffer in system memory and depth buffer in card memory */
    if ( ( Attrib & MGA_FORCE_SYSMEM ) || 
    (buf->MemBlock = mmAllocMem( cardHeap, buf->Size,7, 0x400000 ) ) == 0) {
    		/* oh well, try for anywhere in the card */
	    if ( ( Attrib & MGA_FORCE_SYSMEM ) ||
	    (buf->MemBlock = mmAllocMem( cardHeap, buf->Size, 7, 0 )) == 0) {
	    	/* can't allocate it anywhere on the card, so try system memory */
		    if ( !sysmemHeap || 
		    	(buf->MemBlock = mmAllocMem( sysmemHeap, buf->Size, 7, 0 )) == 0) {
			      return -1;
		    }
		    mgaMsg( 1, "allocating depth buffer in system memory\n" );
		    systemMemory = 1 | use_agp;
		    buf->SystemMemory = 1;
	    }
       }
    ofs = buf->MemBlock->ofs;
	if ( systemMemory ) {
    		// FIXME: is this sysmemPhysical offset right for AGP? 
	    buf->Setup[MGA_SETUP_DSTORG] = buf->Setup[MGA_SETUP_ZORG] = ( ofs + sysmemPhysical ) | systemMemory;
	} else {
	    buf->Setup[MGA_SETUP_DSTORG] = buf->Setup[MGA_SETUP_ZORG] = ofs;
	}
    buf->BufAddr = &BASEADDR[ofs];
    buf->Height = Height;
    buf->Pitch = Pitch;
    break;

  case MGA_MISCBUFFER:
    /*
     * Input: Attrib, Size, 
     */
    buf->SetupSize = 0;
    buf->Size = Size;
    if ((Width & 7) != 0)
      return -1;
    if ((buf->MemBlock = mmAllocMem( cardHeap, Size, 6, 0 )) == 0)
	return -1;
    ofs = buf->MemBlock->ofs;
    buf->Setup[MGA_SETUP_DSTORG] = ofs;
    buf->BufAddr = &BASEADDR[ofs];
    break;

  default:
    return -1;
  }
  return 0;
}

mgaBufferPtr mgaCreateBuffer(int Attrib, int Size, int Width, int Height,
			     int Pitch)
{
  mgaBufferPtr buf;

  buf = (mgaBufferPtr)calloc(1,sizeof(mgaBuffer));
  if (!buf)
    return NULL;
  if (mgaInitBuffer(buf,Attrib,Size,Width,Height,Pitch) == -1) {
    free(buf);
    return NULL;
  }
  buf->next = mgaglx.bufferList;
  mgaglx.bufferList = buf;
  return buf;
}

mgaBufferPtr mgaCreatePrimaryBuffer(int Attrib, int Width, int Height,
				  int Pitch)
{
  mgaBufferPtr buf;

  Attrib = (Attrib & ~MGA_TYPE_MASK) | MGA_COLORBUFFER;
  buf = mgaCreateBuffer(Attrib,0,Width,Height,Pitch);
  if (!buf)
    return NULL;
  mmMarkReserved(buf->MemBlock);
  
  return buf;
}

static void FreeBuffer(mgaBufferPtr buf)
{
  mgaBufferPtr p,priv;
  
  p = mgaglx.bufferList;
  priv = 0;
  while (p && p != buf) {
    priv = p;
    p = p->next;
  }
  if (p){
      if (priv){
	  priv->next = buf->next;
      }else{
	  mgaglx.bufferList = buf->next;
      }
  }
  if (mmFreeMem(buf->MemBlock) == -1) {
    mgaError("Could not free buffer %08x\n",buf->MemBlock->ofs);
    mmDumpMemInfo( cardHeap );
  }
  buf->magic = 0;
  free(buf);
}

static void mgaDetachZBuffer(mgaBufferPtr parent)
{
  mgaBufferPtr zb;
  
  if (!VALID_MGA_BUFFER(parent) || !parent->HasZORG)
    return;
  zb = parent->ZBuffer;
  parent->HasZORG = 0;
  parent->SetupSize = MGA_SETUP_SIZE;
  
  if (VALID_MGA_BUFFER(zb) ) {
    FreeBuffer(zb);
  }
  parent->ZBuffer = NULL;
}


int mgaDestroyBuffer(mgaBufferPtr buf)
{
  mgaBufferPtr p;

  if (!buf)
    return 0;
  if (buf->magic != mgaBufferMagic)
    return -1;

  switch (buf->Attrib & MGA_TYPE_MASK) {
  case MGA_ZBUFFER16:
  case MGA_ZBUFFER32:
    p = mgaglx.bufferList;
    while (p) {
      if (p->HasZORG && p->ZBuffer == buf) {
	p->HasZORG = 0;
	p->ZBuffer = 0;
	p->SetupSize = MGA_SETUP_SIZE;
      }
      p = p->next;
    }
    break;

  default:
    mgaDetachZBuffer(buf);
  }
  FreeBuffer(buf);
  return 0;
}

mgaBufferPtr mgaAttachZBuffer(mgaBufferPtr parent, mgaBufferPtr zb,
			      int Type)
{
	int	flag;
	
  if (!parent->Drawable)
    return NULL;
    
	/* if the parent is in system memory, force the depth buffer into system memory */
	if ( parent->SystemMemory ) {
		flag = MGA_FORCE_SYSMEM;
	} else {
		flag = 0;
	}
    
	mgaDetachZBuffer(parent);
	
    if ((zb = mgaCreateBuffer(Type | flag,0,0,parent->Height,parent->Pitch)) == NULL)
      return NULL;
  if ( Type == MGA_ZBUFFER16 )
    MGA_SET_FIELD(parent->Setup[MGA_SETUP_MACCESS],MA_zwidth_MASK,MA_zwidth_16);
  else if ( Type == MGA_ZBUFFER32 )
    MGA_SET_FIELD(parent->Setup[MGA_SETUP_MACCESS],MA_zwidth_MASK,MA_zwidth_32);
  else
    return NULL;
  parent->HasZORG = 1;
  parent->Setup[MGA_SETUP_ZORG] = zb->Setup[MGA_SETUP_ZORG];
  parent->SetupSize = MGA_SETUP_SIZE;
  parent->ZBuffer = zb;
  return zb;
}

